#!/usr/bin/python

# Cressel Anderson
# Georgia Tech: CS8803OPT
# Karen Liu
# Nov. 14, 2008
# Solver for non-quadratic conjugate gradient method

# this implementation only requires the objective function and the
# initial point.

# run this file: python nqcg.py
# it will run a default optimization

import inexact_line_search_dsc as ilsdsc
import numpy as np
import copy,time

class nonquadratic_conjugate_gradient_solver( ):
    '''Implementation of the algorithm on pg 167-168 in Practical
    Optimization by Antoniou and Lu

    There are some parameters that can be tweaked in this
    implementation.  line_search_epsilon is the smallest update during
    line search that is allowed to continue. line_seach_delta is the
    update distance needed bythe dsc algorithm. line_search_K is the
    reduction in line_search_delta between successive iterations of
    the line_search. update_epsilon is the minimum magnitude of change
    permitted before the solver terminates. determinant_epsilon is a
    threshold to prevent degenerate cases resulting from the
    replacement of a search direction vector.'''

    def __init__( self, x_0,
                  objective_function, 
                  line_search_epsilon=0.01, 
                  line_search_delta=0.1,
                  line_search_K = 0.4,
                  update_epsilon = 0.001,
                  determinant_epsilon = 0.01
                  ):

        self.F = objective_function
        self.l_epsilon = line_search_epsilon
        self.l_delta = line_search_delta
        self.l_K = line_search_K
        self.u_epsilon = update_epsilon
        self.d_epsilon = determinant_epsilon

        self.n = x_0.shape[0]
        self.x = []
        self.fs = []
        self.x.append([x_0])
        self.fs.append([self.F(self.x[0][-1])])
        self.alphas = []

    def f( self, alpha ):
        return self.F( self.current_x + alpha*self.direction )

    def line_search( self ):
        #print 'dir = ',self.direction
        #print 'cur = ',self.current_x
        my_ils = ilsdsc.inexact_line_search_dsc( 
            alpha=0.01, 
            delta=self.l_delta, 
            f=self.f, 
            K=self.l_K,
            epsilon = self.l_epsilon  )
        return my_ils.run()[0]

    def step_1(self):
        self.D = [np.matrix(np.eye(self.n))]
        self.k = 0
        self.det = [1]

    def step_2(self):
        self.alphas.append([])
        for i in range(1,self.n+1,1):
            self.current_x = self.x[self.k][i-1]
            self.direction = self.D[self.k][:,i-1]
            self.alphas[self.k].append( self.line_search() )
            #print 'self.x[self.k][i-1]  = ',self.x[self.k][i-1] 
            #print 'self.alphas[self.k][i-1]  = ',self.alphas[self.k][i-1]
            #print 'self.direction = ',self.direction
            self.x[self.k].append( self.x[self.k][i-1] + self.alphas[self.k][i-1]*self.direction )
            self.fs[self.k].append( self.F( self.x[self.k][-1] ) )

        self.alpha_k_m = max(self.alphas[self.k])
        self.m = self.alphas[self.k].index(self.alpha_k_m)

    def step_3(self):
        new_direction = self.x[self.k][self.n] - self.x[self.k][0]

        self.direction = new_direction
        self.current_x = self.x[self.k][0]

        self.alphas[self.k].append( self.line_search() )
        #print 'self.x[self.k][0]  = ',self.x[self.k][0] 
        #print 'self.alphas[self.k][self.n]*self.direction  = ',self.alphas[self.k][self.n]*self.direction 
        self.x[self.k].append( self.x[self.k][0] + self.alphas[self.k][self.n]*self.direction )

        self.fs[self.k].append( self.F( self.x[self.k][-1] ) )
        self.lambda_k = np.linalg.norm( self.x[self.k][self.n] - self.x[self.k][0] )

    def step_4(self):
        return np.linalg.norm( self.alphas[self.k][self.n] * self.direction )

    def step_5(self):
        #print 'Determinant: ',self.det[-1]
        if (self.det[-1] > self.d_epsilon ):
            newD = copy.deepcopy( self.D[self.k] )
            newD[:,self.m] = self.direction
            for i in range( self.m ):
                newD[:,i] = self.D[self.k][:,i]
            for i in range( self.m+1,self.n,1 ):
                newD[:,i] = self.D[self.k][:,i]
            self.D.append( newD )
                
            self.det.append( self.det[-1] * self.alpha_k_m / self.lambda_k )

        else:
            self.det.append( self.det[-1] )
            newD = copy.deepcopy( self.D[self.k] )
            self.D.append( newD )

        self.x.append([self.x[self.k][self.n]])
        self.fs.append( [self.F( self.x[self.k][-1] )] )
        self.k += 1

    def run(self):
        self.step_1()
        for i in range(30):
            self.step_2()
            self.step_3()

            v = self.step_4()
            if v < self.u_epsilon:
                return (self.x[self.k][self.n],self.fs[self.k][self.n])

            self.step_5()
        return self.x[self.k][-1], self.fs[self.k][-1]

    
if __name__ == '__main__':
    import nqcg


    # Define your objective function
    def f( x ):
        x1 = x[0,0]**2
        x2 = x[1,0]**2 
        return 100*(x2 - x1**2 )**2 + (1-x1)**2

    # Initialization
    x_0=np.matrix([-2,2]).T

    # Instatiate the solver with the objective function and initial value
    my_solver = nqcg.nonquadratic_conjugate_gradient_solver(
        x_0,
        objective_function = f  )
    

    # Run Solver
    start = time.time()
    x,fval = my_solver.run()
    print 'x* = ',x
    print 'f(x*) = ',fval
    print 'time = ',time.time()-start
