import math as m
import numpy as np
from numpy import power as pr
import sys
import functools as ft

class InitError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return repr(self.value)

def l2(a):
    return np.linalg.norm(a)

def test_line_search():
    print '####################################################'
    print '# Testing Line Search                              #'
    print '####################################################'
    def f(x):
        return x * x
    print 'result is ', golden_section(f, -20, 20)

    def fo(x):
        return np.linalg.norm(x)
    xk = np.matrix([-1.0, -1.0]).T
    dk = np.matrix([ 1.0,  1.0]).T

    print 'testing alpha finder'
    print golden_line_search(fo, xk, dk)

def golden_line_search(f, x, d):
    def f_alpha(alpha):
        return f(x + alpha * d)
    return golden_section(f_alpha, 0.0, 1.0)

def golden_section(f, xL, xU, tol=.0001):
    K  = 1.618034

    #Init
    k   = 1
    Ik1 = xU - xL
    Ik2 = Ik1 / K

    xlk = xL
    xuk = xU

    xak = xuk - Ik2
    xbk = xlk + Ik2

    fak = f(xak)
    if fak == None:
        raise RuntimeError, 'golden_section: f is not returning a value!'

    fbk = f(xbk)
    Ik2 = Ik2 / K

    while True:
        #print 'golden_section: evals >> %.2f %.2f'% (fak, fbk)
        if fak >= fbk:
            #4.7 4.12
            xlk1 = xak
            xak1 = xbk
            xbk1 = xlk1 + Ik2
            xuk1 = xuk

            fak1 = fbk
            fbk1 = f(xbk1)
        else:
            #4.13 4.18
            xlk1 = xlk
            xuk1 = xbk
            xak1 = xuk1 - Ik2
            xbk1 = xak

            fak1 = f(xak1)
            fbk1 = fak
        #print 'golden_section:         %.2f %.2f %.2f %.2f'% (xlk1, xak1, xbk1, xuk1)
        
        #Get ready for next iteration
        Ik1 = Ik2
        Ik2 = Ik2 / K

        xlk = xlk1
        xak = xak1
        xbk = xbk1
        xuk = xuk1

        fak = fak1
        fbk = fbk1

        #Check for termination conditions
        if Ik1 < tol or xak1 > xbk1:
            if fak1 > fbk1:
                x_star = .5 * (xbk1 + xuk1)
            if fak1 == fbk1:
                x_star = .5 * (xak1 + xbk1)
            if fak1 < fbk1:
                x_star = .5 * (xlk1 + xak1)

            f_star = f(x_star)
            return {'x': x_star, 'fx': f_star}
        k   = k + 1

def is_zero(a, epsilon = .0001):
    return np.linalg.norm(a) < epsilon

def print_mat(m, before_decimal=4, after_decimal=4):
    for i in xrange(m.shape[0]):
        print '   [ ',
        for j in xrange(m.shape[1]):
            #if is_zero(m[i,j]):
            #    print '0 ',
            #else:
            format_string = '%'+ str(before_decimal) + '.' + str(after_decimal) + 'f '
            print format_string % m[i,j],
        print ']'

def pseudo_inverse(A):
    return A.T * np.linalg.inv(A*A.T)

def nullspace_vectors(A):
    u,s,vt = np.linalg.svd(A)
    return vt.T[:, len(s):]

def eigen_vectors(A):
    u,s,vt = np.linalg.svd(A)
    return vt.T[:, 0:len(s)]

##
# Restate problem without equality constraints
def rm_constraints(H, p, A, b):
    vr = nullspace_vectors(A)
    Ap = pseudo_inverse(A)
    H_new = vr.T * H * vr
    p_new = b.T*Ap.T*H*vr + p.T*vr + b.T*Ap.T*H.T*vr 
    #print 'H_new.shape', H_new.shape, 'p_new.shape', p_new.shape
    def retrieve_solution(r):
        return vr * r + Ap*b
    return H_new, p_new, retrieve_solution

def recondition(mat, epsilon=.001):
    u, s, vt = np.linalg.svd(mat)
    s[np.where(np.abs(s) <= epsilon)] = s[np.where(np.abs(s) <= epsilon)] + .1
    return u*np.diag(s)*vt

##
# Primal Dual Path Following Method for Convex QP
# Specifically, this solves the problem
#   minimize  f(x) = 1/2 x.T * H * x + x.T*p
#   s.t.      Ax = b
#             x >= 0
# 
# @param H nxn symmetric positive semidefinite matrix nxn matrix
# @param p nx1 vector
# @param A pxn full row rank constraints matrix
# @param b px1 vector
# @param initial_guess (x0 (nx1), l0 (px1), m0 (nx1)) initial solution vector, only requires that x0 > 0 and m0 > 0, guess does not have to be strictly feasible
#                       lambdas are for equality constraints
#                       meus are for inequality constraints, has to be positive
# @param dual_gap_tolerance how far away dual & primal solutions should be before optimization terminate
def pathQP(H, p, A, b, initial_guess=None, rho=None, dual_gap_tolerance = 0.0001, debug=False):
    n      = H.shape[0] # number of variables
    m_cons = A.shape[0] # number of constraints

    assert H.shape[0] == H.shape[1]
    assert H.shape[0] == p.shape[0]
    assert A.shape[1] == H.shape[0]
    assert A.shape[0] == b.shape[0]

    #Put in guesses for x, lambda, meu
    if initial_guess == None:
        initial_guess = (None, None, None)
    x_g, l_g, m_g = initial_guess
    if x_g == None:
        x_g =  np.matrix(np.ones((n,1)))
    if l_g == None:
        l_g = -np.matrix(np.ones((m_cons, 1)))
    if m_g == None:
        m_g =  np.matrix(np.ones((n,1)))
    initial_guess = (x_g, l_g, m_g)

    #Put in guess for rho
    if rho < m.sqrt(n) or rho == None:
        nrho = 3*np.sqrt(n)
        if debug:
            if rho == None:
                print 'pathQP: rho is undefined (', rho, ') using', nrho
            else:
                print 'pathQP: rho is too small (', rho, ') using', nrho
        rho = nrho

    #Check +semidef of H
    u, s, v = np.linalg.svd(H)
    if np.any(s < 0):
        print 'pathQP: H matrix is not positive semidefinite!!'
        
    #Check rank of A
    #A = recondition(A)
    u, s, vt = np.linalg.svd(A)
    v = vt.T
    rankA = np.sum(np.abs(s) > 0.001)
    original_A = None
    original_b = None
    if  rankA < A.shape[0] and (not is_zero(A) and not is_zero(b)):
        #reduce = False
        #if reduce:
        print 'pathQP: A matrix is not full row rank (expected %d, actual %d). Reducing number of constraints.' % (A.shape[0], rankA)
        print 'eigen values:', 
        print_mat(np.matrix(s))
        original_A = A
        original_b = b
        A  = np.matrix(np.diag(s[:rankA]) * v[:,:rankA].T)
        b  = (u[:,:rankA].T * b)
        x0, l0, m0 = initial_guess
        l0 = l0[0:rankA, :]
        initial_guess = (x0, l0, m0)
        #else:
        #    print 'pathQP: A matrix is not full row rank (expected %d, actual %d). Reconditioning' % (A.shape[0], rankA)
        #    s[np.where(np.abs(s) <= 0.001)] = s[np.where(np.abs(s) <= 0.001)] + .1
        #    A = u * np.diag(s) * vt


    x0, l0, m0 = initial_guess
    if debug:
        print 'pathQP: input dimensions - H', H.shape, 'p', p.shape, 'x', p.shape, 'A', A.shape, 'b', b.shape
        print 'H'
        print_mat(H)
        print 'p'
        print_mat(p.T)
        print 'A'
        print_mat(A)
        print 'b'
        print_mat(b.T)
        print 'x0'
        print_mat(x0.T)
        print 'm0'
        print_mat(m0.T)
        print 'l0'
        print_mat(l0.T)

    if not np.all(m0 > 0):
        print 'pathQP: m0 not all positive, correcting'
        m0[np.where(m0 <= 0)] = .01

    wk  = initial_guess
    k   = 1
    n   = H.shape[0]
    e   = np.matrix(np.ones((n,1)))
    if debug:
        print 'pathQP: initial guess x', wk[0].T, 'l', wk[1].T, 'm', wk[2].T
    tk1 = 1

    while True:
        xk, lk, mk  = wk
        if debug:
            print 'xk',
            print_mat(xk.T)
            print 'mk',
            print_mat(mk.T)
            print 'lk',
            print_mat(lk.T)

        duality_gap = (xk.T * mk)[0,0]

        if QP_KKT(H, p, A, b, wk, epsilon=.00001):
            if debug:
                print 'pathQP: satisfied KKT, exiting'
            break
        elif duality_gap <= dual_gap_tolerance:
            if debug:
                print 'pathQP: satisfied duality tolerance, exiting'
            break
        else:
            if debug:
                print '==========================================================================='
                print 'ITERATION', k
                print '==========================================================================='
            tk1   = duality_gap / (n+rho)

            #Find constraints violating residuals
            rd    = H*xk + p - A.T*lk - mk #Residual in Lagrangian Dual 
            rp    = b - A*xk               #Residual in constraints

            #print 'rd Residual in Lagrangian Dual ', rd.T
            #print 'rp Residual in constraints     ', rp.T

            #Set up required matrices
            M     = np.diag(mk.A1)
            X     = np.diag(xk.A1)
            num_var   = xk.shape[0]
            num_cons  = A.shape[0]
            row_1     = np.concatenate((-H, A.T, np.eye(num_var)), 1)
            row_2     = np.concatenate((A, np.zeros((num_cons, num_cons)), np.zeros((num_cons, num_var))), 1)
            row_3     = np.concatenate((M, np.zeros((num_var, num_cons)), X), 1)

            large_mat = np.concatenate((row_1, row_2, row_3))
            large_mat = recondition(large_mat)
            large_res = np.concatenate((rd, rp, tk1*e - X*mk))
            ldelta_w  = np.linalg.inv(large_mat) * large_res
            #ldelta_w  = pseudo_inverse(large_mat) * large_res
            dx        = ldelta_w[0:num_var,0]
            dl        = ldelta_w[num_var: num_var+num_cons, 0]
            dm        = ldelta_w[num_var+num_cons:, 0]

            
            #print 'Large Mat',
            #ul, sl, vlt = np.linalg.svd(large_mat)
            #print l2((large_mat * ldelta_w) - large_res)
            #print_mat(np.matrix(sl))

            #Primal Path test
            #big_test    = np.linalg.norm((row_3*ldelta_w) - (tk1*e - X*mk))
            primal_test = np.linalg.norm((M*dx + X*dm) - (tk1*e - X*mk))
            #if not is_zero(big_test) or not is_zero(little_test):
            if not is_zero(primal_test):
                if debug:
                    print 'Primal Path Condition NOT SATISFIED, deviated by', primal_test
                #print '                                    (lil mat) >>', little_test
                #return

            if debug:
                if is_zero(dx):
                    print 'DX not making progress'
                else:
                    print 'DX changed by %.4f' % np.linalg.norm(dx), dx.T

                if is_zero(dm):
                    print 'DM not making progress'
                else:
                    print 'DM changed by %.4f' % np.linalg.norm(dm), dm.T

                if is_zero(dl):
                    print 'DL not making progress'
                else:
                    print 'DL changed by %.4f' % np.linalg.norm(dl), dl.T

            #Find analytical alpha k for the next iteration
            dx_idx = np.where(dx < 0)
            ap_div = -xk[dx_idx] / dx[dx_idx]
            if (ap_div.shape[0] * ap_div.shape[1]) == 0:
                ap = 1.0
            else:
                ap     = np.min(ap_div)

            dm_idx = np.where(dm < 0)
            ad_div = -mk[dm_idx] / dm[dm_idx]
            if (ad_div.shape[0] * ad_div.shape[1]) == 0:
                ad = 1.0
            else:
                ad = np.min(ad_div)
            a_max  = min(ap, ad)
            ak     = (1 - 0.000001) * a_max
            if debug:
                print 'ak is ', ak

            #Update solution
            wk = (xk+ak*dx, lk+ak*dl, mk+ak*dm)
            if original_A != None:
                original_residual = original_A*wk[0]- original_b
                print 'original_residual\n', original_residual, l2(original_residual)
            if debug:
                print 'Duality GAP', duality_gap, is_zero(duality_gap, epsilon=dual_gap_tolerance)
            yield {'solution':wk, 'iteration': k, 'gap':duality_gap}
        k = k+1
            
##
# Tests KKT conditions for problem of form
#       min      x.T * H * x + x.T *p
#       wrt      Ax  = b
#                 x >= 0
def QP_KKT(H, p, A, b, wk, epsilon=.0001, debug=False):
    x     = wk[0]
    lambd = wk[1]
    mu    = wk[2]

    success = True
    #Primal Feasibility
    primal1      = A * x - b
    primal1_test = np.abs(primal1) > epsilon
    if np.sum(primal1_test) > 0:
        if debug:
            print "KKT 1: Ax - b = 0,\n     norm %.4f" % (np.linalg.norm(primal1)), 'residual', primal1.A1
            print "               >> FAILED by", np.sum(primal1_test), "values."
        success = False
    #else:
    #    print 'KKT 1: passed with', 
    #    print_mat(primal1.T)

    primal2_test = np.abs(x[np.where(x < 0)]) > epsilon
    if np.sum(primal2_test) > 0:
        if debug:
            print "KKT 2: x > 0,\n     %d violations" % (np.sum(primal2_test))
            print "               >> FAILED by", np.sum(primal2_test), 'values.'
        success = False

    #Dual Feasibility
    dual1 = A.T * lambd + mu - H*x - p
    dual1_test = np.abs(dual1) > epsilon
    if np.sum(dual1_test) > 0:
        if debug:
            print "KKT 3: A.T*lambda + Mu - Hx - p = 0,\n     norm %.4f" % (np.linalg.norm(dual1)), 'residual', dual1.A1
            print "               >> FAILED by", np.sum(dual1_test), 'values.'
        success = False

    dual2_test = mu < -epsilon 
    if np.sum(dual2_test) > 0:
        if debug:
            print "KKT 4: mu > 0,\n     %d violations" % (np.sum(dual2_test))
            print "               >> FAILED by", np.sum(dual2_test), 'values.'
        success = False

    #Complementary Slackness
    cs      = np.matrix(np.diag(x.A1)) *  mu
    cs_test = np.abs(cs) > epsilon
    if np.sum(cs_test) > 0:
        if debug:
            print "KKT 5: X*mu = 0,\n     norm %.4f" % (np.linalg.norm(cs)), 'residual', cs.A1
            print "               >> FAILED by", np.sum(cs_test), 'values.'
        success = False

    return success

##
# Tests KKT conditions for problem of form
#       min      x.T * H * x + x.T *p
#       wrt      Ax  = b
# NOTE: there is no x>=0 constraint in this 
#       case.
def SQP_QP_KKT(H, p, A, b, wk):
    x     = wk[0]
    lambd = wk[1]

    epsilon = 0.01
    success = True
    #Primal Feasibility
    primal1      = A * x - b
    primal1_test = np.abs(primal1) > epsilon
    if np.sum(primal1_test) > 0:
        print "SQP QP KKT 1: Ax - b = 0,\n     norm %.4f" % (np.linalg.norm(primal1)), 'residual', primal1.A1
        print "               >> FAILED by", np.sum(primal1_test), "values."
        success = False

    #Dual Feasibility
    dual1 = A.T*lambd - H*x - p
    dual1_test = np.abs(dual1) > epsilon
    if np.sum(dual1_test) > 0:
        print "SQP QP KKT 3: A.T*lambda - Hx - p = 0,\n     norm %.4f" % (np.linalg.norm(dual1)), 'residual', dual1.A1
        print "               >> FAILED by", np.sum(dual1_test), 'values.'
        success = False

    return success

##
# Converts inequality constraints into equality constraints
# Solves alternate QP problem of type 
#   minimize  f(x) = 1/2 x.T * H * x + x.T*p
#   s.t.      Ax >= b
#
# @param H nxn symmetric positive semidefinite matrix nxn matrix
# @param p nx1 vector
# @param A pxn full row rank constraints matrix
# @param b px1 vector
# @param initial_guess (x0 (nx1), l0 (px1), m0 (nx1)) initial solution vector, only requires that x0 > 0 and m0 > 0, guess does not have to be strictly feasible
# @param dual_gap_tolerance how far away dual & primal solutions should be before optimization terminate
def alternate_pathQP(H, p, A, b, initial_guess=None, rho=None, dual_gap_tolerance=.00001):
    num_constraints = A.shape[0]
    n               = H.shape[0]
    H_1   = np.concatenate(( H,                            -H,                           np.matrix(np.zeros((n, num_constraints)))), 1)
    H_2   = np.concatenate((-H,                             H,                           np.matrix(np.zeros((n, num_constraints)))), 1)
    H_3   = np.concatenate((np.matrix(np.zeros((num_constraints, n))),  
                            np.matrix(np.zeros((num_constraints, n))), 
                            np.matrix(np.zeros((num_constraints, num_constraints)))), 1)

    H_hat = np.concatenate((H_1, H_2, H_3))
    p_hat = np.concatenate((p, -p,  np.matrix(np.zeros((num_constraints, 1)))))
    A_hat = np.concatenate((A, -A, -np.matrix(np.eye(num_constraints))), 1)

    xp = np.matrix(np.ones(initial_guess[0].shape))
    xn = np.matrix(np.ones(initial_guess[0].shape))
    xp[np.where(initial_guess[0] > 0)] = initial_guess[0][np.where(initial_guess[0] > 0)]
    xn[np.where(initial_guess[0] < 0)] = initial_guess[0][np.where(initial_guess[0] < 0)]
    ni  = np.matrix(np.ones((num_constraints, 1)))

    if initial_guess != None:
        x0_hat = np.concatenate((xp, xn, ni))
        l0_hat = initial_guess[1]
        m0_hat = np.concatenate((initial_guess[2], np.matrix(np.ones((n + num_constraints, 1)))))
        initial_hat = (x0_hat, l0_hat, m0_hat)
    else:
        initial_hat = None

    last_solution = None
    for s in pathQP(H_hat, p_hat, A_hat, b, initial_hat, rho, dual_gap_tolerance):
        last_solution = s
        xplus  = s['solution'][0][0:n,  0]
        xminus = s['solution'][0][n:2*n, 0]
        x = xplus - xminus
        yield {'solution': x, 'iteration': s['iteration'], 'gap': s['gap']}
    #central_path_KKT(H_hat, p_hat, A_hat, b, last_solution['solution'])

##
# Converts inequality constraints into equality constraints
# Solves alternate QP problem of type 
#   minimize  f(x) = 1/2 x.T * H * x + x.T*p
#   s.t.      Ax = b
#
# @param H nxn symmetric positive semidefinite matrix nxn matrix
# @param p nx1 vector
# @param A pxn full row rank constraints matrix
# @param b px1 vector
# @param initial_guess (x0 (nx1), l0 (px1), m0 (nx1)) initial solution vector, only requires that x0 > 0 and m0 > 0, guess does not have to be strictly feasible
# @param dual_gap_tolerance how far away dual & primal solutions should be before optimization terminate
def sqp_pathQP(H, p, A, b, initial_guess=None, rho=None, dual_gap_tolerance=.00001):
    num_constraints = A.shape[0]
    n               = H.shape[0]
    H_1   = np.concatenate(( H, -H), 1)
    H_2   = np.concatenate((-H,  H), 1)

    H_hat = np.concatenate((H_1, H_2))
    p_hat = np.concatenate((p, -p))
    A_hat = np.concatenate((A, -A), 1)

    xp = np.matrix(np.ones(initial_guess[0].shape))
    xn = np.matrix(np.ones(initial_guess[0].shape))
    xp[np.where(initial_guess[0] > 0)] = initial_guess[0][np.where(initial_guess[0] > 0)]
    xn[np.where(initial_guess[0] < 0)] = initial_guess[0][np.where(initial_guess[0] < 0)]

    if initial_guess != None:
        x0_hat = np.concatenate((xp, xn))
        l0_hat = initial_guess[1]
        m0_hat = np.matrix(np.ones(x0_hat.shape))
        initial_hat = (x0_hat, l0_hat, m0_hat)
    else:
        initial_hat = None

    last_solution = None
    for s in pathQP(H_hat, p_hat, A_hat, b, initial_hat, rho, dual_gap_tolerance):
        last_solution = s
        xplus  = s['solution'][0][0:n,  0]
        xminus = s['solution'][0][n:2*n, 0]
        x = xplus - xminus
        yield {'solution': x, 'iteration': s['iteration'], 'gap': s['gap']}
    #central_path_KKT(H_hat, p_hat, A_hat, b, last_solution['solution'])

def gen_qp_cond(A, b, C, d, x, epsilon=.001, debug=False):
    success = True
    #Ax >= b
    cond1  = A*x - b
    cond1b = np.abs(cond1[np.where(cond1 < 0.0)])
    if not np.all(cond1b < epsilon):
        if debug:
            print 'gen_qp_cond: Ax - b >= 0 is NOT true', 
            print_mat(cond1.T)
        success = False
    else:
        if debug:
            print 'gen_qp_cond: Ax - b >= 0 is true', np.abs(cond1).T

    #Cx = d
    cond2 = C*x - d
    if not is_zero(cond2):
        if debug:
            print 'gen_qp_cond: Cx - d = 0 is NOT true', l2(cond2)
        success = False
    else:
        if debug:
            print 'gen_qp_cond: Cx - d = 0 is true', l2(cond2)
    return success
    

##
# Converts inequality constraints into equality constraints
# Solves alternate QP problem of type 
#   minimize  f(x) = 1/2 x.T * H * x + x.T*p
#   s.t.      Ax >= b                  INEQUALITY (MU)
#             Cx  = d                  EQUALITY   (LAMBDA)
#
# @param H nxn symmetric positive semidefinite matrix nxn matrix
# @param p nx1 vector
# @param A pxn full row rank inequality constraints matrix
# @param b px1 vector
# @param C qxn full row rank equality constraints matrix
# @param d qx1 vector
# @param initial_guess (x0 (nx1), l0 (px1), m0 (px1)) initial solution vector, only requires that x0 > 0 and m0 > 0, guess does not have to be strictly feasible
# @param dual_gap_tolerance how far away dual & primal solutions should be before optimization terminate
def general_alternate_pathQP(H, p, A, b, C, d, initial_guess=None, rho=None, dual_gap_tolerance=.00001):
    debug = False
    p_dim = A.shape[0]
    q_dim = C.shape[0]
    n_dim = H.shape[0]
    H_1   = np.concatenate(( H, -H, np.matrix(np.zeros((n_dim, p_dim)))), 1)
    H_2   = np.concatenate((-H,  H, np.matrix(np.zeros((n_dim, p_dim)))), 1)
    H_3   = np.concatenate((np.matrix(np.zeros((p_dim, n_dim))),  
                            np.matrix(np.zeros((p_dim, n_dim))), 
                            np.matrix(np.zeros((p_dim, p_dim)))), 1)

    H_hat = np.concatenate((H_1, H_2, H_3))
    p_hat = np.concatenate((p, -p,  np.matrix(np.zeros((p_dim, 1)))))
    azeros= np.matrix(np.zeros((q_dim, p_dim)))
    A_hat = np.concatenate((np.concatenate((A, -A, -np.matrix(np.eye(p_dim))), 1),
                            np.concatenate((C, -C, azeros), 1)), 0)
    b_hat = np.vstack((b, d))

    if initial_guess != None:
        x0, l0, m0 = initial_guess
        xp = np.matrix(np.ones((H.shape[0],1)))
        xn = np.matrix(np.ones((H.shape[0],1)))
        ni = np.matrix(np.ones((p, 1)))
        xp[np.where(x0 > 0)] = x0[np.where(initial_guess[0] > 0)]
        xn[np.where(x0 < 0)] = x0[np.where(initial_guess[0] < 0)]
        x0_hat = np.concatenate((xp, xn, ni))
        l0_hat = np.concatenate((l0, m0))
        initial_hat = (x0_hat, l0_hat, None)
    else:
        initial_hat = None

    if debug:
        print 'H\n', H, H.shape
        print 'p\n', p, p.shape
        print 'A\n', A, A.shape
        print 'b\n', b, b.shape
        print 'C\n', C, C.shape
        print 'd\n', d, d.shape
        print 'x0\n', initial_guess

        print 'H_hat\n', H_hat, H_hat.shape
        print 'p_hat\n', p_hat, p_hat.shape
        print 'A_hat\n', A_hat, A_hat.shape
        print 'b_hat\n', b_hat, b_hat.shape
        print 'x0\n', initial_hat

    last_solution = None
    for s in pathQP(H_hat, p_hat, A_hat, b_hat, initial_hat, rho, dual_gap_tolerance):
        last_solution = s
        xplus  = s['solution'][0][0:n_dim,  0]
        xminus = s['solution'][0][n_dim:2*n_dim, 0]
        x = xplus - xminus
        if debug:
            print '=========================================='
            print 'iteration', s['iteration']
            print '=========================================='
            gen_qp_cond(A, b, C, d, x)
            #print 'b!!'
            #print_mat(b_hat)
            #print 'C*x',
            #print_mat(C*x)
            #print 'd',
            #print_mat(d)
            #print 'C*x-d',
            #print_mat(C*x-d)
            #print 'C -C',
            #print_mat(np.concatenate((C, -C, azeros), 1) * s['solution'][0] - d)
            #print 'A_hat'
            #print_mat(A_hat)
            #print 'A'
            #print_mat(A)
            #print 'Ahat * x - b_hat'
            #print_mat(A_hat * s['solution'][0] - b_hat)
            #print 'xplus', 
            #print_mat(xplus.T)
            #print 'xminus', 
            #print_mat(xminus.T)
            #print 'x is', 
            #print_mat(x.T)
            #print 'solution', 
            #print_mat(s['solution'][0].T)
            #print 'xplus', xplus.T, 'xminus', xminus.T
        yield {'solution': x, 'iteration': s['iteration'], 'gap': s['gap']}

def get_last_element(f):
    a = None
    for s in f:
        a = s
    return a

def flip(f):
    def ff(x):
        return -f(x)
    return ff

def func_abs(f):
    def ff(x):
        return abs(f(x))
    return ff

def gk_(x):
    x1 = x[0,0]
    x2 = x[1,0]
    x3 = x[2,0]
    return np.matrix([-4*pr(x1,3) -2*x1*pr(x2,2) -2*x1*pr(x3,2),
                      -8*pr(x2,3) -2*pr(x1,2)*x2,
                      -4*pr(x3,3) -2*pr(x1,2)*x3]).T

##
# AK for sample SQP problem from book
def Ak_(x):
    x1 = x[0,0]
    x2 = x[1,0]
    x3 = x[2,0]
    return np.matrix([[4*pr(x1,3), 4*pr(x2,3), 4*pr(x3,3)],
                      [16*x1,      28*x2,      14*x3]])

##
# ak for sample SQP problem from book
def ak_(x):
    x1 = x[0,0]
    x2 = x[1,0]
    x3 = x[2,0]
    return np.matrix([pr(x1,4) + pr(x2, 4) + pr(x3, 4) -25,
                      8*pr(x1,2) + 14*pr(x2,2)+7*pr(x3,2)-56]).T

##
# Wk for sample SQP problem from book
def Wk_(x, l):
    x1 = x[0,0]
    x2 = x[1,0]
    x3 = x[2,0]
    l1 = l[0,0]
    l2 = l[1,0]
    return np.matrix([[ -12*pr(x1,2) -2*pr(x2, 2) -2*pr(x3,2) -12*l1*pr(x1, 2) -16*l2,   -4*x1*x2,                                         -4*x1*x3],
                      [ -4*x1*x2,                                                        -24*pr(x2,2) -2*pr(x1,2) -12*l1*pr(x2,2) -28*l2,   0],
                      [ -4*x1*x3,                                                         0,                                               -12*pr(x3, 2) -2*pr(x1, 2) -12*l1*pr(x3, 2) -14*l2]])

def sum_list(l):
    r = l[0]
    for a in l[1:]:
        r = r + a
    return r

##
# Sequential Quadratic Program with equality constraints solver.
# Solve problems of the form:
#                 min f(x)
#     s.t. ai(x) = 0 for i = 1, 2, ..., p
def sqp_eq(x0, l0, f,   jacobian_f, hessian_f, 
                   ais, jacobian_ais, hessian_ais, tolerance, k_limit=None):
    debug = False
    n   = x0.shape[0]  # Dimension of problem
    p_c = len(ais)     # Number of constraints

    k   = 1
    xk  = x0
    lk  = l0

    while True:
        constraint_hessians = sum_list([lk[idx,0] * hess_a(xk) for idx, hess_a in enumerate(hessian_ais)])
        hess_fxk = hessian_f(xk)
        Wk       = hess_fxk - constraint_hessians
        Ak       = np.concatenate(tuple([jacob_a(xk).T for jacob_a in jacobian_ais]))
        gk       = jacobian_f(xk)
        ak       = np.concatenate(tuple([np.matrix([ai(xk)]) for ai in ais]))

        C        = np.concatenate((np.concatenate((Wk, -Ak.T), 1), 
                                   np.concatenate((-Ak, np.matrix(np.zeros((p_c, p_c)))), 1)), 0)
        D        = np.concatenate((Ak.T*lk - gk, ak), 0)
        step     = np.linalg.inv(C) * D

        dx       = step[0:n]
        dl       = step[n:]
        if not is_zero(Ak * dx + ak):
            print 'sqp_eq: constraint 1 not satisfied!', 
            print_mat(Ak * dx + ak)

        if not is_zero(Wk * dx + gk - Ak.T*(lk + dl)):
            print 'sqp_eq: constraint 2 not satisfied!', 
            print Wk * dx + gk - Ak.T*(lk + dl)

        #Do a line search
        start     = xk
        direction = dx

        def lagrangian(multipliers, some_x):
            constraints  = np.sum([multipliers[idx,0] * ai(some_x) for idx, ai in enumerate(ais)])
            function     = f(some_x)
            return function - constraints

        ak        = .95 * golden_line_search(ft.partial(lagrangian,lk), start, direction)['x']
        lk1_alter = np.linalg.inv(Ak*Ak.T) * Ak * (Wk*dx + gk)
        lk1       = lk + dl
        if not is_zero(lk1_alter - lk1):
            print 'sqp_eq: there is disagreement in calculation of lk+1', 
            print_mat((lk1_alter - lk1).T)
            print_mat(lk1_alter)
            print_mat(lk1)

        xk      = xk + ak * dx
        lk      = lk1
        if debug:
            print 'dx', dx.T, 'dl', dl.T
            print 'lagrangian', lagrangian(lk, xk)

        yield {'solution': xk, 'iteration': k, 'change': ak*dx}
        if (np.linalg.norm(dx) <= tolerance):
            break
        else:
            k = k + 1
            if k_limit != None and k >= k_limit:
                return

###
## Sequential Quadratic Program with equality constraints solver
## Solve problems of the form:
##                 min f(x)
##     s.t. ai(x) = 0 for i = 1, 2, ..., p
#def sqp_eq2(x0, l0, f,   jacobian_f, hessian_f, 
#                    ais, jacobian_ais, hessian_ais, tolerance, k_limit=None):
#    n   = x0.shape[0]  # Dimension of problem
#    p_c = len(ais)     # Number of constraints
#
#    k   = 1
#    xk  = x0
#    lk  = l0
#
#    while True:
#        constraint_hessians = sum_list([lk[idx,0] * hess_a(xk) for idx, hess_a in enumerate(hessian_ais)])
#        hess_fxk = hessian_f(xk)
#        Wk = hess_fxk - constraint_hessians
#        Ak = np.concatenate(tuple([jacob_a(xk).T for jacob_a in jacobian_ais]))
#        gk = jacobian_f(xk)
#        ak = np.concatenate(tuple([np.matrix([ai(xk)]) for ai in ais]))
#
#        last_solution = None
#        for s in pathQP_eq(Wk, gk, Ak, -ak):
#            delta = s['solution'][0]
#            last_solution = s
#
#        delta_x = delta[0:n,:]
#        if True:
#            #Do a line search
#            start     = xk
#            direction = delta_x
#            def phi_p(x):
#                penalty = -np.sum([lk[idx,0] * ai(x) for idx, ai in enumerate(ais)])
#                value   = f(x)
#                return value + penalty
#            a1_star      = golden_line_search(phi_p,     start, direction)['x']
#            #print a1_star
#            #return
#            # a2_stars      = [golden_line_search(flip(ai), start, direction)['x'] for ai in ais]
#
#            #if len(a2_stars) > 0:
#            #    a2_star_min  = np.min(a2_stars)
#            #    print a1_star, a2_star_min
#            #    ak = .95 * np.min([a1_star, a2_star_min])
#            #else:
#            ak      = .95 * a1_star
#            delta_x = ak * delta_x
#            #print '              ===> DELTA_x', delta_x.T,
#            #print 'AK', ak
#
#        #delta_l = delta[n:,:] don't need delta of lagrange multiplier
#        lk1     = np.linalg.inv(Ak*Ak.T) * Ak * (Wk*delta_x + gk)
#        xk      = xk + delta_x
#        yield {'solution': xk, 'iteration': k, 'change': delta_x}
#        if (np.linalg.norm(delta_x) <= tolerance):
#            break
#        else:
#            k = k + 1
#            if k_limit != None and k >= k_limit:
#                return
#        #return

##
# Solves sequential Quadratic Program with inequality & equality constraints:
# Solve problems of the form:
#                 min f(x)
#     s.t. ai(x) = 0 for i = 1, 2, ..., p
#          cj(x) >= 0 for j = 1, 2, ..., q
# @param f function to optimize 
# @param jacobian_f function returning column matrix of derivatives of f
# @param ais list of functions ai's, each return answer as column vectors
# @param jacobian_ais column matrix of derivatives of ai's
# @param cis list of functions ci's, each return answer as column vectors
# @param jacobian_cjs column matrix of derivatives of cj's
def gen_sqp(x0,  f,       jacobian_f, 
                 ais,     jacobian_ais, #equality constraints
                 cis,     jacobian_cis, #inequality constraints
                 beta = 100.0,
                 tolerance=0.00001, k_limit=None,
                 debug=False):
    n          = x0.shape[0]  # Dimension of problem
    p_c        = None         # Number of equality constraints
    q_c        = None         # Number of inequality constraints

    print 'gen_sqp: starting at x0 =', x0.T
    k   = 1
    xk  = x0
    lk  = None #equality, p of these
    mk  = None #inequality, q of these
    Zk  = np.matrix(np.eye(xk.shape[0]))

    def calc_Aek(some_x):
        return np.concatenate(tuple([jacob_a(some_x).T for jacob_a in jacobian_ais]))

    def calc_Aik(some_x):
        return np.concatenate(tuple([jacob_c(some_x).T for jacob_c in jacobian_cis]))

    def calc_flist(some_x, funcs):
        l = []
        for some_f in funcs:
            r = some_f(xk)
            if r.__class__ == np.matrix:
                l.append(r)
            else:
                l.append(np.matrix([r]))
        return np.concatenate(tuple(l), 0)

    while True:
        gk  = jacobian_f(xk)
        Aek = calc_Aek(xk)
        Aik = calc_Aik(xk) 
        #print 'Aik', Aik
        q_c = Aik.shape[0]
        p_c = Aek.shape[0]
        #if debug:
        #    print 'gen_sqp: There are %d equality, and %d INequality constraints.' %(p_c, q_c)
        ak  = calc_flist(xk, ais)
        ck  = calc_flist(xk, cis)
        dx  = [s['solution'] for s in general_alternate_pathQP(H=Zk, p=gk, A=Aik, b=-ck, C=Aek, d=-ak)][-1]
        if not gen_qp_cond(A=Aik, b=-ck, C=Aek, d=-ak, x=dx, debug=False):
            gen_qp_cond(A=Aik, b=-ck, C=Aek, d=-ak, x=dx, debug=True)
            break

        active_cons    = (Aik * dx + ck).A[:,0]
        active_indices = np.where(np.abs(active_cons) < .001) #active INequality constraints
        #print 'active_indices\n', active_indices
        #print 'locations we want'
        #print np.where(np.abs(active_cons) < .001)
        #print 'selecting it from'
        #print Aik
        #print 'to make sure'
        #print active_cons
        #print 'rows', rows, 'columns', columns
        #print 'active_cons', active_cons
        Aaik = Aik[active_indices]
        #print 'Aaik = Aik[active_indices]'
        #print Aik[active_indices]
        #return 
        #print 'trying the inner way'
        #print Aik[np.where(np.abs(active_cons) < .001)]
        #Aaik= np.matrix([])
        #print Aik[np.where(np.abs(active_cons) < .001)]
        #print 'Aek',
        #print_mat(Aek)
        #print 'Aik',
        #print_mat(Aik)
        #print 'Aaik',
        #print (Aaik)
        Aak  = np.concatenate((Aek, Aaik))
        lm   = np.linalg.inv(Aak*Aak.T) * Aak * (Zk*dx + gk)
        lk1      = lm[:p_c]  #equalities, p of these
        mk1_hat  = lm[p_c:]  #active inequalities, not of these
        mk1      = np.matrix(np.zeros((q_c,1)))
        mk1[active_indices] = mk1_hat
        #print 'active index is', active_indices
        #print 'mk before!', mk.T
        #print 'mk after!', mk.T
        #print 'lm', lm.T
        #print 'lk1', lk1
        #return

        def merit_function(lambdas, mus, xs):
            # INequality constraints
            iconstraints =        np.sum([mus[idx,0] * ci(xs) for idx, ci in enumerate(cis)])
            # equality constraints
            econstraints = beta * np.sum([np.power(ai(xs), 2) for ai in ais]) 
            r = f(xs) + econstraints - iconstraints
            #print 'merit_function:', r, xs.T
            violations = [ai(xs) for ai in ais]
            #print '   violations', violations
            return r

        def lagrangian(lambdas, mus, xs):
            iconstraints = np.sum([mus[idx,0]     * ci(xs) for idx, ci in enumerate(cis)])
            econstraints = np.sum([lambdas[idx,0] * ai(xs) for idx, ai in enumerate(ais)])
            r = f(xs) - econstraints - iconstraints
            #print 'lagrangian:', r, xs.T
            return r

        a1 = golden_line_search(ft.partial(merit_function, lk1, mk1), xk, dx)['x']
        #print '===================='
        #a1 = golden_line_search(ft.partial(lagrangian, lk1, mk1), xk, dx)['x']

        #return

        #print 'a1 is', a1
        a2s       = []
        for idx, ci in enumerate(cis):
            #if mu(idx) is zero
            if np.abs(mk1[idx, 0]) < 0.0001:
                a2s.append(golden_line_search(flip(ci), xk, dx)['x'])
        #print 'posible alphas', a2s

        if len(a2s) > 0:
            #print 'a2s is', a2s
            a2_star = np.max(a2s)
            #print 'selected alpha star', a2_star
            #print 'a1 is', a1
            #print 'a2_star is', a2_star 
            ak      = min(a1, a2_star)
        else:
            #print 'defaulting to a1'
            ak      = a1

        ak  = .95 * ak
        #print 'ak is (after .95 reduction)', ak
        #print 'dx is', 
        #print_mat(dx.T)

        dx  = ak*dx
        xk1 = xk+dx

        if debug:
            print 'dx', dx.T, 'lagrangian', lagrangian(lk1, mk1, xk1)
        yield {'solution': xk1, 'iteration': k, 'change': dx}
        if np.linalg.norm(dx) <= tolerance:
            print '\ngen_sqp: Exiting as dx is less than tolerance!'
            break

        #Update Hessian matrix
        #print '>> (jacobian_f(xk1) - gk)     \n', (jacobian_f(xk1) - gk)
        #print '>> (calc_Aek(xk1) - Aek).T*lk1\n', (calc_Aek(xk1) - Aek).T*lk1
        #print '>> calc_Aik(xk1) - Aik        \n', calc_Aik(xk1) - Aik
        #print '>> mk1                        \n', mk1
        #print '>> (calc_Aik(xk1) - Aik).T*mk1\n', (calc_Aik(xk1) - Aik).T*mk1
        gammak = (jacobian_f(xk1) - gk) - (calc_Aek(xk1) - Aek).T*lk1 - (calc_Aik(xk1) - Aik).T*mk1
        if dx.T*gammak >= .2*dx.T*Zk*dx:
            theta = 1.0
        else:
            theta = (.8*dx.T*Zk*dx) / (dx.T*Zk*dx - dx.T*gammak)
            theta = theta[0,0]

        #print '\ntheta\n', theta
        #print 'gammak\n', gammak
        #print 'dx\n', dx
        #print 'Zk\n', Zk
        nk   = theta * gammak + (1-theta)*Zk*dx
        Zk   = Zk + ((nk*nk.T)/(dx.T*nk)) - ((Zk*dx*dx.T*Zk) / (dx.T*Zk*dx))
        lk   = lk1
        mk   = mk1
        xk   = xk1
        k    = k+1


###
## Solves sequential Quadratic Program with inequality & equality constraints:
## Solve problems of the form:
##                 min f(x)
##     s.t. ai(x) = 0 for i = 1, 2, ..., p
##          cj(x) = 0 for j = 1, 2, ..., q
#def gen_sqp(start,  f,   jacobian_f, hessian_f, 
#                   ais, jacobian_ais, hessian_ais, 
#                   cis, jacobian_cis, hessian_cis,
#                   tolerance, k_limit=None):
#    debug      = False
#    n          = x0.shape[0]  # Dimension of problem
#    p_c        = len(ais)     # Number of equality constraints
#    q_c        = len(cis)     # Number of inequality constraints
#    l0, x0, m0 = start
#
#    k   = 1
#    xk  = x0
#    lk  = l0
#    mk  = m0
#
#    while True:
#        eq_constraint_hessians = sum_list([lk[idx, 0] * hess_a(xk) for idx, hess_a in enumerate(hessian_ais)])
#        in_constraint_hessians = sum_list([mk[idx, 0] * hess_c(xk) for idx, hess_c in enumerate(hessian_cis)])
#        hess_fxk = hessian_f(xk)
#        Zk       = hess_fxk - eq_constraint_hessians - in_constraint_hessians
#        gk       = jacobian_f(xk)
#        Aek      = np.concatenate(tuple([jacob_a(xk).T for jacob_a in jacobian_ais]))
#        Aik      = np.concatenate(tuple([jacob_c(xk).T for jacob_c in jacobian_cis]))
#        ak       = np.concatenate(tuple([np.matrix([ai(xk)]) for ai in ais]))
#        ck       = np.concatenate(tuple([np.matrix([ci(xk)]) for ci in cis]))
#        
#        dx       = [s['solution'] for s in general_alternate_pathQP(Zk, gk, Aek, -ak, Aik, -ck)][-1]
#
#        active_cons = Aik * dx + ck
#        Aaik = Aik[np.where(np.abs(active_cons) < .001)[0], :]
#        Aak  = np.concatenate((Aek, Aaik))
#        lm   = np.linalg.inv(Aak*Aak.T) * Aak * (Zk*dx + gk)
#        lk1  = lm[:p]
#        mk1  = lm[p:]
#
#        def merit_function(lambdas, meus, xs):
#            iconstraints = np.sum([meus[idx,0] * ci(some_x) for idx, ci in enumerate(cis)])
#            econstraints = beta * np.sum([np.power(ai(some_x), 2) for ai in ais])
#            return f(xs) - econstraints - iconstraints
#
#        def lagrangian(lambdas, meus, xs):
#            iconstraints = np.sum([meus[idx,0] * ci(some_x) for idx, ci in enumerate(cis)])
#            econstraints = np.sum([lambdas[idx,0] * ai(some_x) for ai in ais])
#            return f(xs) - econstraints - iconstraints
#
#        a1        = golden_line_search(ft.partial(merit_function, lk1, mk1), xk, dx)['x']
#        a2s       = []
#        for idx, ci in enumerate(cis):
#            if np.abs(mk1[idx, 0]) < 0.0001:
#                a2s.append(golden_line_search(flip(ci), xk, dx)['x'])
#        if len(a2s) > 0:
#            a2_star = np.max(a2s)
#            ak      = min(a1, a2_star)
#        ak = .95 * ak
#
#        xk1  = xk+ak*dx
#        lk   = lk1
#        mk   = mk1
#
#        if debug:
#            print 'dx', dx.T, 'lagrangian', lagrangian(lk, mk, xk)
#
#        yield {'solution': xk, 'iteration': k, 'change': ak*dx}
#        if np.linalg.norm(dx) <= tolerance:
#            break
#        else:
#            k = k+1


##
# Example 13.1
def example13_1():
    print '####################################################'
    print '# Example 13.1                                     #'
    print '####################################################'
    H = np.matrix([[1.0, 0.0], [0.0, 1.0]])
    A = np.matrix([1, 0.0])
    b = np.matrix([0.0])
    p = np.matrix([0, 0.0]).T

    solution = None
    for r in pathQP(H, p, A, b):
        x, l, mu = r['solution']
        print r['iteration'], '        X =>', x.T
        print                 '              Lambda = >', l.T
        print                 '              Mu     = >', mu.T
        solution = r['solution'][0]
    print 'Is this solution correct?', is_zero(solution), l2(solution)
    return is_zero(solution)

##
# Example 13.3
def example13_3():
    print '####################################################'
    print '# Example 13.3                                     #'
    print '####################################################'
    H  = np.matrix([[4.0, 0.0, 0.0],
                    [0.0, 1.0,-1.0],
                    [0.0,-1.0, 1.0]])
    p  = np.matrix([-8, -6, -6.0]).T
    A  = np.matrix([1, 1, 1.0])
    b  = np.matrix([3.0])
    x0 = np.matrix([1, 1000, 300.0]).T
    l0 = np.matrix([-1.0]).T
    m0 = np.matrix([1, 1.0, 1.0]).T

    initial_guess = (x0, l0, m0)
    rho = 3 + 2 * m.sqrt(3)
    current_sol = None
    for r in pathQP(H, p, A, b, initial_guess, rho=rho):
        x, l, mu = r['solution']
        print r['iteration'], '        X =>', x.T
        print                 '              Lambda = >', l.T
        print                 '              Mu     = >', mu.T
        current_sol = r
    QP_KKT(H, p, A, b, current_sol['solution'], debug = True)
    correct_solution = np.matrix([0.50, 1.25,  1.25]).T
    print 'Is solution correct?', is_zero(correct_solution - current_sol['solution'][0])
    return is_zero(correct_solution - current_sol['solution'][0])

##
# Example 13.4
def example13_4(rho = 14.0 + 20.0*m.sqrt(4.0)):
    print '####################################################'
    print '# Example 13.4                                     #'
    print '####################################################'
    H = np.matrix([[ 1.0,  0.0, -1.0,  0.0],
                   [ 0.0,  1.0,  0.0, -1.0],
                   [-1.0,  0.0,  1.0,  0.0],
                   [ 0.0, -1.0,  0.0,  1.0]])

    p = np.matrix(np.zeros((4,1)))

    A = np.matrix([[ 1.0,  0.0,  0.0,  0.0],
                   [ 0.0,  1.0,  0.0,  0.0],
                   [-1.0, -2.0,  0.0,  0.0],
                   [ 0.0,  0.0,  0.0,  1.0],
                   [ 0.0,  0.0,  1.0,  1.0],
                   [ 0.0,  0.0, -1.0, -2.0]])

    b  = np.matrix([0.0, 0.0, -2.0, 2.0, 3.0, -6.0]).T
    x0 = np.matrix(np.ones((4,1))) * 14.0
    l0 = np.matrix(np.ones((6,1))) * 11.0
    m0 = np.matrix(np.ones((4,1))) * 110.0

    solution = None
    for s in alternate_pathQP(H, p, A, b, (x0, l0, m0)):
        print s['iteration'], 'xk', s['solution'].T, 'gap', s['gap']
        print '   ==> Ax - b', (A * s['solution'] - b).T
        solution = s['solution']
    correct_solution = np.matrix([0.40, 0.8, 1.0, 2.0]).T
    print 'Is this solution correct?', l2(solution - correct_solution) < .0001, l2(solution - correct_solution)
    return l2(solution - correct_solution) < .0001

def example15_1a():
    print '####################################################'
    print '# Example 15.1 a                                     #'
    print '####################################################'
    H = np.matrix ([[  241.5,  -18.0,  -36.0  ],
                    [  -18.0,  93.0,  0.0  ],
                    [  -36.0,  0.0,  240.0  ]])
    p = np.matrix([  -175.5,  -54.0,  -162.0  ]).T
    A = np.matrix([[  108.0,  13.5,  108.0  ],
                   [   48.0,  42.0,   42.0  ]])
    b = np.matrix([  -142.0625, -110.5]).T
    #x0 = pseudo_inverse(A) * b
    x0 = np.matrix([  1.0,  1.0,  1.0  ]).T
    l0 = np.matrix([  -1.0,  -1.0  ]).T

    for r in sqp_pathQP(H, p, A, b, (x0, l0)):
        x = r['solution']
        print r['iteration'], ' X =>', x.T

def example15_1b():
    print '####################################################'
    print '# Example 15.1b                                    #'
    print '####################################################'
    H = np.matrix ([[  241.5,  -18.0,  -36.0  ],
                    [  -18.0,  93.0,  0.0  ],
                    [  -36.0,  0.0,  240.0  ]])
    p = np.matrix([  -175.5,  -54.0,  -162.0  ]).T
    A = np.matrix([[  108.0,  13.5,  108.0  ],
                   [   48.0,  42.0,   42.0  ]])
    b = np.matrix([  -142.0625, -110.5]).T
    x0 = np.matrix([  1.0,  1.0,  1.0  ]).T
    m0 = np.matrix([  1.0,  1.0,  1.0  ]).T
    l0 = np.matrix([  -1.0,  -1.0  ]).T
    rho = 10.0

    H_new, p_new, sol_mapper = rm_constraints(H, p, A, b)
    dim_problem = H_new.shape[0]
    x0_new = np.matrix([  1.0 ]).T
    m0_new = np.matrix([  1.0 ]).T
    l0_new = np.matrix([ -1.0 ]).T

    for r in pathQP(H_new, p_new, np.matrix(np.zeros((1,dim_problem))), np.matrix(np.zeros((1,1))), (x0_new, l0_new, m0_new), rho=rho):
        x_sol = sol_mapper(r['solution'])
        obj   = x_sol.T*H*x_sol + p.T*x_sol
        cons  = A*x_sol - b
        print 'objective', obj, 'constraint', np.linalg.norm(cons), ' >>> ', cons.T
        current_sol = r

def example15_1():
    print '####################################################'
    print '# Example 15.1                                     #'
    print '####################################################'
    def f(x):
        x1 = x[0,0]
        x2 = x[1,0]
        x3 = x[2,0]
        return -np.power(x1, 4) -2.0 * np.power(x2, 4) -np.power(x3, 4) -np.power(x1, 2)*np.power(x2, 2) -np.power(x1, 2)*np.power(x3, 2)

    def jacobian_f(x):
        x1 = x[0,0]
        x2 = x[1,0]
        x3 = x[2,0]
        return np.matrix([-4*np.power(x1, 3) - 2*x1*np.power(x2, 2) - 2.0*x1*np.power(x3, 2),
                          -8*np.power(x2, 3) - 2*np.power(x1,2)*x2,
                          -4*np.power(x3, 3) - 2*np.power(x1,2)*x3]).T

    def hessian_f(x):
        x1 = x[0,0]
        x2 = x[1,0]
        x3 = x[2,0]
        Dx1x1  = -12*np.power(x1,2) -2*np.power(x2,2) -2*np.power(x3,2)
        Dx2x2  =  -2*np.power(x1,2) -24*np.power(x2,2)
        Dx3x3  =  -2*np.power(x1,2) -12*np.power(x3,2)
        Dx1x2  = -4*x1*x2
        Dx2x3  = 0.0
        Dx1x3  = -4*x1*x3
        return np.matrix([[Dx1x1, Dx1x2, Dx1x3],
                          [Dx1x2, Dx2x2, Dx2x3],
                          [Dx1x3, Dx2x3, Dx3x3]])

    def a1(x):
        x1 = x[0,0]
        x2 = x[1,0]
        x3 = x[2,0]
        return np.power(x1, 4) + np.power(x2, 4) + np.power(x3, 4) - 25.0

    def jacobian_a1(x):
        x1 = x[0,0]
        x2 = x[1,0]
        x3 = x[2,0]
        Dx1 = 4*np.power(x1,3)
        Dx2 = 4*np.power(x2,3)
        Dx3 = 4*np.power(x3,3)
        return np.matrix([Dx1, Dx2, Dx3]).T

    def hessian_a1(x):
        x1 = x[0,0]
        x2 = x[1,0]
        x3 = x[2,0]
        Dx1x1  = 12*np.power(x1, 2)
        Dx2x2  = 12*np.power(x2, 2)
        Dx3x3  = 12*np.power(x3, 2)
        Dx1x2  = 0.0
        Dx2x3  = 0.0
        Dx1x3  = 0.0
        return np.matrix([[Dx1x1, Dx1x2, Dx1x3],
                          [Dx1x2, Dx2x2, Dx2x3],
                          [Dx1x3, Dx2x3, Dx3x3]])

    def a2(x):
        x1 = x[0,0]
        x2 = x[1,0]
        x3 = x[2,0]
        return 8 * np.power(x1, 2) + 14 * np.power(x2, 2) + 7 * np.power(x3, 2) - 56.0

    def jacobian_a2(x):
        x1  = x[0,0]
        x2  = x[1,0]
        x3  = x[2,0]
        Dx1 = 16*x1
        Dx2 = 28*x2
        Dx3 = 14*x3
        return np.matrix([Dx1, Dx2, Dx3]).T

    def hessian_a2(x):
        x1     = x[0,0]
        x2     = x[1,0]
        x3     = x[2,0]
        Dx1x1  = 16
        Dx2x2  = 28
        Dx3x3  = 14 
        Dx1x2  = 0.0
        Dx2x3  = 0.0
        Dx1x3  = 0.0
        return np.matrix([[Dx1x1, Dx1x2, Dx1x3],
                          [Dx1x2, Dx2x2, Dx2x3],
                          [Dx1x3, Dx2x3, Dx3x3]])

    solution = None
    for s in sqp_eq(np.matrix([3,1.5,3]).T, np.matrix([-3.0, -3.0]).T, f, jacobian_f, hessian_f, 
            [a1, a2], [jacobian_a1, jacobian_a2], [hessian_a1, hessian_a2], .00001, 300):
        print '>> iteration', s['iteration'], s['solution'].T,
        print 'obj %.2f' % f(s['solution']), 
        print 'a1 %.2f' % a1(s['solution']), 
        print 'a2 %.2f' % a2(s['solution'])
        solution = s['solution']
    
    r = is_zero(solution - np.matrix([[ 1.87406549, -0.46581962,  1.88472043]]).T)
    print 'Is this solution correct?', r
    return r

##
# utility problem, used during testing of solvers...
def example15_4a():
    H_hat  = np.matrix([[ 1,  0, -1, -0,  0],
                        [ 0,  1, -0, -1,  0],
                        [-1, -0,  1,  0,  0],
                        [-0, -1,  0,  1,  0],
                        [ 0,  0,  0,  0,  0.0]])

    p_hat = np.matrix([[ 10],
                       [  1],
                       [-10],
                       [ -1],
                       [  0.0]]) 

    A_hat = np.matrix([[ 10,   8, -10,  -8,  -1],
                       [  1,   0,  -1,  -0,   0],
                       [  0,  -1,  -0,   1,   0.0]])

    b_hat = np.matrix([[-32],
                       [ -4],
                       [ -0.0]]) 
    x0 = np.matrix([[-2.],
                 [ 0.],
                 [ 2.],
                 [ 0.],
                 [-8.]])

    for s in pathQP(H_hat, p_hat, A_hat, b_hat, (None,None,None), debug=True):
        pass
        #print '>>', s['iteration'], s['solution']

def example15_4():
    #c = np.matrix([[ 1,  0],
    #               [ 0, -1.0]])
    #d = np.matrix([1, -4.0]).T

    def f(x):
        x1 = x[0,0]
        x2 = x[1,0]
        return (x1*x1) + x2

    def jac_f(x):
        x1 = x[0,0]
        x2 = x[1,0]
        return np.matrix([2*x1, 1]).T

    def a1(x):
        x1 = x[0,0]
        x2 = x[1,0]
        return x1*x1 + x2*x2 - 9

    def jac_a1(x):
        x1 = x[0,0]
        x2 = x[1,0]
        return np.matrix([2*x1, 2*x2]).T

    cc1 = np.matrix([ 1,  0])
    cc2 = np.matrix([-1,  0])
    cc3 = np.matrix([ 0,  1])
    cc4 = np.matrix([ 0, -1.0])
    d = np.matrix([1, -5, 2, -4.0]).T

    def c1(x):
        return cc1*x - d[0,0]

    def c2(x):
        return cc2*x - d[1,0]

    def c3(x):
        return cc3*x - d[2,0]

    def c4(x):
        return cc4*x - d[3,0]

    def jc1(x):
        return cc1.T

    def jc2(x):
        return cc2.T

    def jc3(x):
        return cc3.T

    def jc4(x):
        return cc4.T

    correct_solution   = np.matrix([1.0, 2.8284]).T
    starting_solutions = [np.matrix([5.0, 4.0]).T,
                          np.matrix([5.0, 2.0]).T,
                          np.matrix([1.0, 2.0]).T,
                          np.matrix([3.0, 2.0]).T,
                          np.matrix([4.0, 3.0]).T]

    trials = []
    num_correct    = 0
    for start in starting_solutions:
        final_solution = None
        for s in gen_sqp(start, 
                         f,    jac_f, 
                         [a1], [jac_a1], 
                         [c1,  c2,  c3,  c4],  
                         [jc1, jc2, jc3, jc4]):
            x = s['solution']
            final_solution = x
            print s['iteration'], x.T,
            print 'dist. %.2f' % l2(correct_solution - x),
            print 'f(x) => %.2f' % f(x)
            print '                                        ',
            print 'a1(x) => %.2f' % a1(x), 
            print 'c1(x) => %.2f' % c1(x), 
            print 'c2(x) => %.2f' % c2(x), 
            print 'c3(x) => %.2f' % c3(x), 
            print 'c4(x) => %.2f' % c4(x)
        dist = l2(final_solution - correct_solution)
        if dist < .001:
            num_correct = num_correct + 1.0
        trials.append((dist, dist < .001))

    print '==================================================='
    print '                 results'
    print '==================================================='
    for idx, t in enumerate(trials):
        print 'trial', idx+1, '. distance from correct solution:', t
    print 'total correct', (num_correct / len(trials)) * 100, '%'
    return num_correct == len(trials)




if __name__ == '__main__':
    #test_line_search()
    #example13_1()
    #example15_1a()
    #Make sure that the included solver solves all these problems
    assert example13_3()
    assert example13_4()
    assert example15_1()
    assert example15_4()









































































































#u = np.matrix([[.2717,   -.8003, -.5345],
#               [-.1365,  -.5818,  0.8018],
#               [0.9527,  0.1449,  0.2673]])
#
#s = np.matrix([[14.8798, 0],
#               [0, 1.6101]])
#
#v = np.matrix([[ .1463, -.3171,  .6331, -.6908],
#               [-.6951, -.6284, -.3161, -.1485],
#               [ .6402, -.32  , -.6322, -.2969],
#               [ .2926, -.6342,  .3156,  .6423]])











#  ##
#  # Primal Dual Path Following Method for Convex QP
#  # Specifically, this solves the problem
#  #   minimize  f(x) = 1/2 x.T * H * x + x.T*p
#  #   s.t.      Ax = b
#  #             x >= 0
#  # 
#  # @param H nxn symmetric positive semidefinite matrix nxn matrix
#  # @param p nx1 vector
#  # @param A pxn full row rank constraints matrix
#  # @param b px1 vector
#  # @param initial_guess (x0 (nx1), l0 (px1), m0 (nx1)) initial solution vector
#  # @param dual_gap_tolerance how far away dual & primal solutions should be before optimization terminate
#  def pathQP_strict(H, p, A, b, initial_guess, dual_gap_tolerance):
#      wk  = initial_guess
#      k   = 1
#      n   = H.shape[0]
#      rho = m.sqrt(n)
#      e   = np.matrix(np.ones((n,1)))
#  
#      while True:
#          xk, lk, mk = wk
#          if (xk.T * mk) <= dual_gap_tolerance:
#              break
#          else:
#              tk1   = xk.T*mk / (n+rho)
#  
#              #Set up required matrices
#              M     = np.diag(mk.A[:,0])
#              X     = np.diag(xk.A[:,0])
#              gamma = np.linalg.inv(M + X*H)
#              Y     = np.linalg.inv(A*gamma*X*A.T) * A
#              y     = gamma * (X*mk - tk1*e)
#  
#              #Find deltas
#              dl    = Y*y
#              dx    = gamma*X*A.T*dl - y
#              dm    = H*dx - A.T*dl
#  
#              #Find alpha for the next iteration
#              dx_idx = np.where(dx < 0)
#              ap    = np.min(-xk[dx_idx] / dx[dx_idx])
#              dm_idx = np.where(dm < 0)
#              ad    = np.min(-mk[dm_idx] / dm[dm_idx])
#              a_max = min(ap, ad)
#              ak    = (1 - 0.000001) * a_max
#  
#              #Update solution
#              wk = (xk+ak*dx, lk+ak*dl, mk+ak*dm)
#              yield wk, k
#          k = k + 1

















###
## Primal Dual Path Following Method for Convex QP
## Specifically, this solves the problem
##   minimize  f(x) = 1/2 x.T * H * x + x.T*p
##   s.t.      Ax = b
## 
## @param H nxn symmetric positive semidefinite matrix nxn matrix
## @param p nx1 vector
## @param A pxn full row rank constraints matrix
## @param b px1 vector
## @param initial_guess (x0 (nx1), l0 (px1), m0 (nx1)) initial solution vector, only requires that x0 > 0 and m0 > 0, guess does not have to be strictly feasible
## @param dual_gap_tolerance how far away dual & primal solutions should be before optimization terminate
#def pathQP_eq(H, p, A, b, initial_guess=None, dual_gap_tolerance = 0.0001):
#    debug  = True
#    n      = H.shape[0] # number of variables
#    m_cons = A.shape[0] # number of constraints
#
#    #Set initil guess if there is None
#    if initial_guess == None:
#        x_g =  np.matrix(np.ones((n,1)))
#        l_g = -np.matrix(np.ones((m_cons, 1)))
#        initial_guess = (x_g, l_g)
#
#    #Check +semidef of H
#    u, s, v = np.linalg.svd(H)
#    if np.any(s < 0):
#        print 'pathQP_eq: H matrix is not positive semidefinite!!'
#        
#    #Check rank of A
#    u, s, v = np.linalg.svd(A)
#    rankA = np.sum(np.abs(s) > 0.001)
#    if  rankA < A.shape[0] and (not is_zero(A) and not is_zero(b)):
#        print 'pathQP_eq: A matrix is not full row rank (expected %d, actual %d). Reducing number of constraints...' % (A.shape[0], rankA)
#        print 'eigen values:', s
#        A = np.matrix(np.diag(s[0:rankA]) * v[0:rankA,:])
#        b = (u.T * b)[0:rankA, :]
#        x0, l0 = initial_guess
#        l0 = l0[0:rankA, :]
#        initial_guess = (x0, l0)
#
#    x0, l0 = initial_guess
#    print 'pathQP_eq: input dimensions - H', H.shape, 'p', p.shape, 'x', p.shape, 'A', A.shape, 'b', b.shape
#    print 'H'
#    print_mat(H)
#    print 'p'
#    print_mat(p.T)
#    print 'A'
#    print_mat(A)
#    print 'b'
#    print_mat(b.T)
#    print 'x0'
#    print_mat(x0.T)
#    print 'l0'
#    print_mat(l0.T)
#
#    wk  = initial_guess
#    k   = 1
#    n   = H.shape[0]
#    e   = np.matrix(np.ones((n,1)))
#    print 'pathQP_eq: initial guess x', wk[0].T, 'l', wk[1].T
#
#    while True:
#        xk, lk  = wk
#        if debug:
#            print 'xk',
#            print_mat(xk.T)
#            print 'lk',
#            print_mat(lk.T)
#        duality_gap = xk.T * H * xk - b.T*lk + p.T*xk
#
#        if SQP_QP_KKT(H, p, A, b, wk):
#            print 'pathQP_eq: satisfied KKT, exiting'
#            break
#        elif abs(duality_gap) <= dual_gap_tolerance:
#            print 'pathQP_eq: satisfied duality gap criteria', duality_gap
#            break
#        else:
#            if debug:
#                print '==========================================================================='
#                print 'ITERATION', k
#                print '==========================================================================='
#            #Find constraints violating residuals
#            rd    = H*xk + p - A.T*lk #Residual in Lagrangian Dual 
#            rp    = b - A*xk          #Residual in constraints
#
#            #Set up required matrices
#            X         = np.diag(xk.A1)
#            num_var   = xk.shape[0]
#            num_cons  = A.shape[0]
#            row_1     = np.concatenate((-H, A.T), 1)
#            row_2     = np.concatenate((A, np.zeros((num_cons, num_cons))), 1)
#
#            large_mat = np.concatenate((row_1, row_2))
#            large_res = np.concatenate((rd, rp))
#            ldelta_w  = np.linalg.inv(large_mat) * large_res
#            dx        = ldelta_w[0:num_var,0]
#            dl        = ldelta_w[num_var: num_var+num_cons, 0]
#
#            if is_zero(dx):
#                print 'DX not making progress'
#            else:
#                print 'DX changed by %.4f' % np.linalg.norm(dx), dx.T
#
#            if is_zero(dl):
#                print 'DL not making progress'
#            else:
#                print 'DL changed by %.4f' % np.linalg.norm(dl), dl.T
#
#            #Find analytical alpha k for the next iteration
#            dx_idx = np.where(dx < 0)
#            ap_div = -xk[dx_idx] / dx[dx_idx]
#            if (ap_div.shape[0] * ap_div.shape[1]) == 0:
#                ap = 1.0
#            else:
#                ap     = np.min(ap_div)
#
#            a_max  = ap
#            ak     = (1 - 0.000001) * a_max
#            if debug:
#                print 'ak is ', ak
#
#            #Update solution
#            wk = (xk+ak*dx, lk+ak*dl)
#            yield {'solution':wk, 'iteration': k}
#        k = k+1

