from forensics.win32.gdi import scan_cstring,gdi_types
from forensics.win32.gdi import SB_FLAGS
from forensics.win32.gdi import RtlRunDecodeUnicodeString
from forensics.object2 import Object
from forensics.object import read_value, obj_size

class _RTL_ATOM_TABLE(Object):
    """Class representing an _RTL_ATOM_TABLE
    """
    hasMembers = True
    name = "_RTL_ATOM_TABLE"

    def __new__(typ, *args, **kwargs):
        obj = object.__new__(typ)
        return obj
    
    # Custom attributes
    def getBuckets(self):
        bucket_off = self.get_member_offset('Buckets')
        buckets = []
        for i in range(self.NumBuckets):
            bkt_ptr = read_value(self.vm, 'pointer', bucket_off+(i*4))
            bkt = Object('_RTL_ATOM_TABLE_ENTRY', bkt_ptr, self.vm, profile=self.profile)
            buckets.append(bkt)
        return buckets
    Buckets = property(fget=getBuckets)

class _RTL_ATOM_TABLE_ENTRY(Object):
    """Class representing an _RTL_ATOM_TABLE_ENTRY
    """
    hasMembers = True
    name = "_RTL_ATOM_TABLE_ENTRY"

    def __new__(typ, *args, **kwargs):
        obj = object.__new__(typ)
        return obj

    def getName(self):
        name_off = self.get_member_offset('Name')
        name = self.vm.read(name_off, self.NameLength*2) # Wide chars
        return name.decode('utf-16-le', 'backslashreplace')
    Name = property(fget=getName)

class _TOOLBAR_WINDOW32(Object):
    hasMembers = True
    name = "TOOLBAR_WINDOW32"
    
    def __new__(typ, *args, **kwargs):
        obj = object.__new__(typ)
        return obj

    def getStringPtrTable(self):
        strs = []
        ptr_off = self.get_member_offset('pStringPtrTable')
        ptr_val = read_value(self.vm, 'pointer', ptr_off)
        for i in range(self.nStringEntries):
            str_ptr = read_value(self.vm, 'pointer', ptr_val + i*4)
            strs.append(scan_cstring(self.vm, str_ptr))
        return strs
    pStringPtrTable = property(fget=getStringPtrTable)
    
    def getButtonList(self):
        btns = []
        btn_size = obj_size(gdi_types, '_BUTTON')
        ptr_off = self.get_member_offset('pButtonList')
        ptr_val = read_value(self.vm, 'pointer', ptr_off)
        for i in range(0, self.nButtons * btn_size, btn_size):
            b = Object("_BUTTON", ptr_val+i, self.vm, profile=self.profile)
            if b.is_valid():
                btns.append(b)
            else:
                btns.append(None)
        return btns
    pButtonList = property(fget=getButtonList)

class _RECT(Object):
    hasMembers = True
    name = "RECT"

    def __new__(typ, *args, **kwargs):
        obj = object.__new__(typ)
        return obj

    def __str__(self):
        return "(%d,%d) (%d,%d)" % (self.left, self.top, self.right, self.bottom)

    def get_tup(self):
        return (self.left, self.top, self.right, self.bottom)

class _TEXT_DESCRIPTOR(Object):
    hasMembers = True
    name = "TEXT_DESCRIPTOR"
    
    def __new__(typ, *args, **kwargs):
        obj = object.__new__(typ)
        return obj

    def __str__(self):
        return self.pBuff

    # Custom Attributes
    def getBuffer(self):
        buf_ptr = self.get_member_offset('pBuff')
        buf_address = read_value(self.vm, 'pointer', buf_ptr)
        buf = self.vm.read(buf_address, (self.nChars-1)*2)
        
        if buf is None: return None

        try:
            buf_sz = buf.decode('utf-16-le').encode('ascii')
        except UnicodeEncodeError:
            return buf.decode('utf-16-le')

        return buf_sz

    pBuff = property(fget=getBuffer)

class _LARGE_UNICODE_STRING(Object):
    hasMembers = True
    name = "LARGE_UNICODE_STRING"
    
    def __new__(typ, *args, **kwargs):
        obj = object.__new__(typ)
        return obj

    def __str__(self):
        return self.Buffer

    # Custom Attributes
    def getBuffer(self):
        buf_ptr = self.get_member_offset('Buffer')
        buf_address = read_value(self.vm, 'pointer', buf_ptr)
        buf = self.vm.read(buf_address, self.Length)
        
        if buf is None: return None

        try:
            buf_sz = buf.decode('utf-16-le').encode('ascii')
        except UnicodeEncodeError:
            return buf.decode('utf-16-le')

        return buf_sz

    Buffer = property(fget=getBuffer)

class _EDIT_BOX(Object):
    hasMembers = True
    name = "EDIT_BOX"

    def __new__(typ, *args, **kwargs):
        obj = object.__new__(typ)
        return obj

    def getBuf(self):
        # Double dereference
        ptr = read_value(self.vm, 'pointer', self.get_member_offset('hBuf'))
        if not self.vm.is_valid_address(ptr): return ""
        ptr = read_value(self.vm, 'pointer', ptr)
        if not self.vm.is_valid_address(ptr): return ""
        s = self.vm.read(ptr, self.nChars*self.bytesPerChar)
        if not s: return ""
        if self.isEncoded:
            s = RtlRunDecodeUnicodeString(self.encKey, s)
        if self.bytesPerChar == 2:
            s = s.decode('utf-16-le', 'backslashreplace')
        return s
    hBuf = property(fget=getBuf)

class _STATUS_BAR(Object):
    hasMembers = True
    name = "STATUS_BAR"
    
    def __new__(typ, *args, **kwargs):
        obj = object.__new__(typ)
        return obj

    def getParts(self):
        ptr = read_value(self.vm, 'pointer', self.get_member_offset('Parts'))
        part_size = obj_size(gdi_types, '_SB_PART')

        parts = []
        if not ptr or not self.vm.is_valid_address(ptr):
            return parts
        for i in range(self.numParts):
            parts.append( 
                Object('_SB_PART', ptr+(i*part_size), self.vm, profile=self.profile)
            )
        return parts
            
    Parts = property(fget=getParts)

class _SB_PART(Object):
    hasMembers = True
    name = "SB_PART"
    
    def __new__(typ, *args, **kwargs):
        obj = object.__new__(typ)
        return obj
    
    def getText(self):
        if not self.flags & SB_FLAGS['SB_TEXT_VALID']:
            return ""
        ptr = read_value(self.vm, 'pointer', self.get_member_offset('pText'))
        if not ptr or not self.vm.is_valid_address(ptr):
            return ""
        return scan_cstring(self.vm, ptr)
    pText = property(fget=getText)
