from forensics.object2 import Object
from forensics.object import read_value
from forensics.win32.tasks import process_list, process_dtb, create_addr_space
from forensics.win32.modules import modules_list, module_modulename, module_baseaddr

gdi_types = {
  '_RTL_ATOM_TABLE': [ 0x14, {
    'Signature': [ 0x0, ['unsigned long']],
    'Lock': [ 0x4, ['pointer', ['void']]],
    'HandleTable': [ 0x8, ['pointer', ['_HANDLE_TABLE']]],
    'NumBuckets': [ 0xC, ['unsigned long']],
    'Buckets': [ 0x10, ['array', 1, ['pointer', ['_RTL_ATOM_TABLE_ENTRY']]]],
} ],
  '_RTL_ATOM_TABLE_ENTRY': [0xC, {
    'HashLink': [ 0x0, ['pointer', ['_RTL_ATOM_TABLE_ENTRY']]],
    'HandleIndex': [ 0x4, ['unsigned short']],
    'Atom': [ 0x6, ['unsigned short']],
    'ReferenceCount': [ 0x8, ['unsigned short']],
    'Flags': [ 0xA, ['unsigned char']],
    'NameLength': [ 0xB, ['unsigned char']],
    'Name': [ 0xC, ['array', 1, ['unsigned short']]],
} ],
  '_W32THREAD' : [ 0x28, { # ?? Found in win32k.pdb -- aka THREADINFO
    'pEThread' : [ 0x0, ['pointer', ['_ETHREAD']]],
    'RefCount' : [ 0x4, ['unsigned long']],
    'ptlW32' : [ 0x8, ['pointer', ['_TL']]],
    'pgdiDcattr' : [ 0xc, ['pointer', ['void']]],
    'pgdiBrushAttr' : [ 0x10, ['pointer', ['void']]],
    'pUMPDObjs' : [ 0x14, ['pointer', ['void']]],
    'pUMPDHeap' : [ 0x18, ['pointer', ['void']]],
    'dwEngAcquireCount' : [ 0x1c, ['unsigned long']],
    'pSemTable' : [ 0x20, ['pointer', ['void']]],
    'pUMPDObj' : [ 0x24, ['pointer', ['void']]],
    'inputQ' : [ 0x30, ['pointer', ['_INPUTQ']]],
    'rpdesk' : [ 0x3C, ['pointer', ['_DESKTOP']]],
    'pDeskInfo' : [ 0x40, ['pointer', ['_DESKTOPINFO']]],
    'MsgQueue' : [ 0xD0, ['_MSG_QUEUE']],
} ],
  '_INPUTQ' : [ 0x2C, {
    'focusedWnd': [ 0x24, ['pointer', ['_WINDOW']]],
    'activeWnd': [ 0x28, ['pointer', ['_WINDOW']]], 
} ],
  '_DESKTOP' : [ 0x30, { # Also a named object
    'pNextDesktop' : [ 0xC, ['pointer', ['_DESKTOP']]],
    'pWinsta' : [ 0x10, ['pointer', ['_WINDOW_STATION']]],
    'activeWnd' : [ 0x34, ['pointer', ['_WINDOW']]],
} ],
  '_WINDOW_STATION' : [ 0x30, { # Also a named object
    'pNextWinsta' : [ 0x4, ['pointer', ['_WINDOW_STATION']]],
    'pDesktop' : [ 0x8, ['pointer', ['_DESKTOP']]],
} ],
  '_DESKTOPINFO' : [ 0x10, {
    'pDesktopWindow' : [ 0x8, ['pointer', ['_WINDOW']]],
} ],
  '_LARGE_UNICODE_STRING' : [ 0xC, {
    'Length' : [ 0x0, ['unsigned long']],
    'MaximumLength' : [ 0x4, ['unsigned long']],
    'Buffer' : [ 0x8, ['pointer', ['unsigned short']]],
} ],
  '_CLASSINFO' : [ 0x00, {
    'ClassAtom' : [ 0x4, ['unsigned short']],
    'SuperclassAtom' : [ 0x6, ['unsigned short']],
} ],
  '_WINDOW' : [ 0x90, {
    'handleId' : [ 0x0, ['unsigned long']],
    'Win32Thread' : [ 0x8, ['pointer', ['_W32THREAD']]],
    'rpDesk' : [ 0xC, ['pointer', ['_DESKTOP']]],
    'pSelf' : [ 0x10, ['pointer', ['_WINDOW']]],
    'dwStyleEx': [ 0x1c, ['unsigned long']],
    'dwStyle':  [ 0x20,  ['unsigned long']],
    'hInstance': [0x24, ['pointer', ['void']]],
    'pNextWindow' : [ 0x2c, ['pointer', ['_WINDOW']]],
    'pPrevWindow' : [ 0x30, ['pointer', ['_WINDOW']]],
    'pParentWindow' : [ 0x34, ['pointer', ['_WINDOW']]],
    'pChildWindow' : [ 0x38, ['pointer', ['_WINDOW']]],
    'rcWnd' : [ 0x40, ['_RECT']],
    'rcClient' : [ 0x50, ['_RECT']],
    'pWndProc' : [ 0x60, ['pointer', ['void']]],
    'pClassInfo' : [ 0x64, ['pointer', ['_CLASSINFO']]],
    'dwId': [0x78, ['unsigned long']],
    'WindowText' : [0x80, ['_LARGE_UNICODE_STRING']],
    'cbWndExtra' : [0x8C, ['unsigned long']],
    'dwUserData' : [0x98, ['unsigned long']],
    'WndExtra' : [0xA4, ['array', 1, 'unsigned long']], # cbWndExtra bytes long
} ],
  '_RECT' : [ 0x10, { 
    'left' : [ 0x0, ['long']],
    'top' : [ 0x4, ['long']],
    'right' : [ 0x8, ['long']],
    'bottom' : [ 0xc, ['long']],
} ],
  '_MSG_QUEUE' : [ 0xC, {
    'NumberOfMessages' : [ 0x8, ['unsigned long']],
} ],
  '_TL' : [ 0xc, {
    'next' : [ 0x0, ['pointer', ['_TL']]],
    'pobj' : [ 0x4, ['pointer', ['void']]],
} ],
# ComCtl32 types
  '_TOOLBAR_WINDOW32': [ 0x90, { # Size is a guess
    'pButtonList' : [ 0x30, ['pointer', ['array', 1, ['_BUTTON']]]], # nButtons long
    'nButtons' : [ 0x84, ['unsigned long']], # Number of buttons in this toolbar
    'pStringPtrTable' : [ 0x58, ['pointer', ['array', 1, ['pointer', ['unsigned short']]]]],
    'nStringEntries' : [ 0x5c, ['unsigned long']],
    'bRectsCalculated' : [ 0xee, ['unsigned char']], # & 1 if rects have been calculated already
} ],
  '_BUTTON': [ 0x1c, {
    'dwButtonId' : [ 0x4, ['unsigned long']],
    'fsState' : [ 0x8, ['unsigned char']], # One of the TBSTATE_* flags
    'fsStyle' : [ 0x9, ['unsigned char']], # One of the TBSTYLE_* flags
    'nStringIndex' : [ 0x10, ['unsigned long']], # index into string pointer table
    'dwLeft'  : [ 0x14, ['unsigned long']],
    'dwTop'   : [ 0x18, ['unsigned long']],
} ],
  '_EDIT_BOX': [ 0xF0, {
    'hBuf': [ 0x0, ['pointer', ['pointer', ['unsigned short']]]],
    'nChars': [0xC, ['unsigned long']],
    'flags': [0x62, ['unsigned char']],
    'bytesPerChar': [0x6A, ['unsigned short']],
    'bEncKey': [0xEC, ['unsigned char']],
    'isEncoded': [0xED, ['unsigned char']],
} ],
  '_STATUS_BAR': [ 0x84, {
    'numParts': [0x74, ['unsigned long']],
    'Parts': [0x80, ['pointer', ['void']]],
} ],
  '_SB_PART': [ 0x20, {
    'pText': [ 0x0, ['pointer', ['unsigned short']]],
    'flags': [ 0x4, ['unsigned long']],
} ],
# riched20.dll
  'CTxtWinHost': [ 0x30, {
    'hWnd' : [ 0x8, ['unsigned long']],
    'pTxtEdit' : [ 0x10, ['pointer', ['CTxtEdit']]],
} ],
  'CTxtEdit': [ 0xC0, {
    'dwFlags' : [ 0x60, ['unsigned long']],
    'pTextDesc' : [ 0x98, ['pointer', ['_TEXT_DESCRIPTOR']]],
    'textLen' : [ 0xA8, ['unsigned long']],
} ],
  '_TEXT_DESCRIPTOR': [ 0x10, {
    'nChars' : [ 0x0, ['unsigned long']],
    'pBuff' : [0x4, ['pointer', ['array', 1, ['unsigned short']]]],
} ],
}

class FakeAtom(object):
    def __init__(self, name):
        self.Name = name

DEFAULT_ATOMS = {
  0x8000: FakeAtom("PopupMenu"),
  0x8001: FakeAtom("Desktop"),
  0x8002: FakeAtom("Dialog"),
  0x8003: FakeAtom("WinSwitch"),
  0x8004: FakeAtom("IconTitle"),
  0x8006: FakeAtom("ToolTip"),
}

WINDOW_STYLES = dict(
  WS_OVERLAPPED = 0x00000000L,
  WS_POPUP = 0x80000000L,
  WS_CHILD = 0x40000000L,
  WS_MINIMIZE = 0x20000000L,
  WS_VISIBLE = 0x10000000L,
  WS_DISABLED = 0x08000000L,
  WS_CLIPSIBLINGS = 0x04000000L,
  WS_CLIPCHILDREN = 0x02000000L,
  WS_MAXIMIZE = 0x01000000L,
  WS_CAPTION = 0x00C00000L,
  WS_BORDER = 0x00800000L,
  WS_DLGFRAME = 0x00400000L,
  WS_VSCROLL = 0x00200000L,
  WS_HSCROLL = 0x00100000L,
  WS_SYSMENU = 0x00080000L,
  WS_THICKFRAME = 0x00040000L,
  WS_GROUP = 0x00020000L,
  WS_TABSTOP = 0x00010000L,
  WS_MINIMIZEBOX = 0x00020000L,
  WS_MAXIMIZEBOX = 0x00010000L,
)

WINDOW_STYLES_EX = dict(
  WS_EX_DLGMODALFRAME = 0x00000001L,
  WS_EX_NOPARENTNOTIFY = 0x00000004L,
  WS_EX_TOPMOST = 0x00000008L,
  WS_EX_ACCEPTFILES = 0x00000010L,
  WS_EX_TRANSPARENT = 0x00000020L,
  WS_EX_MDICHILD = 0x00000040L,
  WS_EX_TOOLWINDOW = 0x00000080L,
  WS_EX_WINDOWEDGE = 0x00000100L,
  WS_EX_CLIENTEDGE = 0x00000200L,
  WS_EX_CONTEXTHELP = 0x00000400L,
  WS_EX_RIGHT = 0x00001000L,
  WS_EX_LEFT = 0x00000000L,
  WS_EX_RTLREADING = 0x00002000L,
  WS_EX_LTRREADING = 0x00000000L,
  WS_EX_LEFTSCROLLBAR = 0x00004000L,
  WS_EX_RIGHTSCROLLBAR = 0x00000000L,
  WS_EX_CONTROLPARENT = 0x00010000L,
  WS_EX_STATICEDGE = 0x00020000L,
  WS_EX_APPWINDOW = 0x00040000L,
)

TBSTATES = dict(
  TBSTATE_CHECKED = 0x01,
  TBSTATE_PRESSED = 0x02,
  TBSTATE_ENABLED = 0x04,
  TBSTATE_HIDDEN = 0x08,
  TBSTATE_INDETERMINATE = 0x10,
  TBSTATE_WRAP = 0x20,
  TBSTATE_ELLIPSES = 0x40,
  TBSTATE_MARKED = 0x80,
)

TBSTYLES = dict(
  TBSTYLE_BUTTON = 0x00,
  TBSTYLE_SEP = 0x01,
  TBSTYLE_CHECK = 0x02,
  TBSTYLE_GROUP = 0x04,
  TBSTYLE_DROPDOWN = 0x08,
  TBSTYLE_AUTOSIZE = 0x10,
  TBSTYLE_NOPREFIX = 0x20,
)

SB_TEXT_FLAGS = dict( # Lowest 4 bits of SB_PART.flags
  SBT_NOBORDERS = 0x01,
  SBT_POPOUT = 0x02,
  SBT_RTLREADING = 0x04,
)

SB_FLAGS = dict(
  SB_TEXT_VALID = 0xF000,
)

def scan_cstring(addr_space, start, unicode=True):
    if not start: return ""
    nullchr = "\x00\x00" if unicode else "\x00"
    nchr = 2 if unicode else 1
    s = ""
    i = start
    c = addr_space.read(i, nchr)
    while c and c != nullchr:
        s += c
        i += nchr
        c = addr_space.read(i, nchr)
    if unicode:
        return s.decode('utf-16-le', 'backslashreplace')
    else:
        return s

def get_flags(flags, val):
    return [f for f in flags if val & flags[f]]

def find_win32k(addr_space, types, symtab):
    """Locate the base address of win32k.sys in memory"""

    mods = modules_list(addr_space, types, symtab)
    for m in mods:
        if module_modulename(addr_space, types, m) == "win32k.sys":
            return module_baseaddr(addr_space, types, m)

def find_space(addr_space, types, symtab, addr):
    """Finds an address space for a given VA"""

    # short circuit if given one works
    if addr_space.is_valid_address(addr): return addr_space

    pslist = process_list(addr_space, types, symtab)
    for p in pslist:
        dtb = process_dtb(addr_space, types, p)
        space = create_addr_space(addr_space, dtb)
        if space and space.is_valid_address(addr):
            return space
    return None

def traverse_windows(win, fun=lambda x,y: None,
                          do_node=lambda x: True, level=0):
    """Traverses windows in their Z order, bottom to top.

    fun will be called for each window, and passed the window
    and current recursion level as arguments.
    
    do_node is a function that decides whether to do any action on this
    window.
    """
    seen = set()
    wins = []
    cur = win
    while cur.is_valid() and cur.v() != 0:
        if cur in seen:
            print "Cycle detected after %d siblings" % len(wins)
            print [(str(w.rcWnd),w.pClassInfo.SuperclassAtom) for w in wins]
            print str(cur.rcWnd),cur.pClassInfo.SuperclassAtom
            break
        seen.add(cur)
        wins.append(cur)
        cur = cur.pNextWindow
    while wins:
        cur = wins.pop()
        if not do_node(cur): continue
        
        fun(cur, level)
        
        if cur.pChildWindow.is_valid() and cur.pChildWindow.v() != 0:
            traverse_windows(cur.pChildWindow,fun=fun,do_node=do_node,level=level+1)

def find_first_window(win, match):
    """Find the first window in this subtree that matches."""
    wins = []
    cur = win
    while cur.is_valid() and cur.v() != 0:
        if match(cur): return cur
        wins.append(cur)
        cur = cur.pNextWindow

    while wins:
        cur = wins.pop()
        if cur.pChildWindow.is_valid() and cur.pChildWindow.v() != 0:
            found = find_first_window(cur.pChildWindow, match)
            if found: return found

    return None

def find_sibling_windows(win, match, reverse=False):
    """Find all sibling windows that match"""
    cur = win
    while cur.is_valid() and cur.v() != 0:
        if match(cur): yield cur
        if reverse:
            cur = cur.pPrevWindow
        else:
            cur = cur.pNextWindow

def read_atoms(addr_space, atom_table_addr, profile, include_defaults=True):
    """Read an atom hash table.
    
    Read an _RTL_ATOM_TABLE and return a dictionary mapping
    the Atom ID to the atom object.
    """

    atoms = {}
    AtomTable = Object("_RTL_ATOM_TABLE", atom_table_addr, addr_space, profile=profile)
    for (i,bkt) in enumerate(AtomTable.Buckets):
        cur = bkt
        while cur.is_valid() and cur.v() != 0:
            atoms[cur.Atom] = cur
            cur = cur.HashLink
    if include_defaults: atoms.update(DEFAULT_ATOMS)
    return atoms

def get_mouse_pos(addr_space, gpsi):
    # RE from win32k!xxxGetCursorPos
    xoff = 0x89C
    yoff = 0x8A0

    x = read_value(addr_space, 'unsigned long', gpsi+xoff)
    y = read_value(addr_space, 'unsigned long', gpsi+yoff)
    return x,y

def get_desktop_window(thrd):
    w32thread = thrd.Tcb.Win32Thread
    if not w32thread.is_valid() or w32thread.offset < 0x80000000:
        #print "W32Thread is not valid"
        return
    if (not w32thread.pDeskInfo.is_valid() or
        w32thread.pDeskInfo.v() < 0x80000000):
        #print "DESKTOPINFO is not valid"
        return
    if (not w32thread.pDeskInfo.pDesktopWindow.is_valid() or
        w32thread.pDeskInfo.pDesktopWindow.v() < 0x80000000):
        #print "pDesktopWindow is not valid"
        return
    return w32thread.pDeskInfo.pDesktopWindow

def get_focus(win):
    """Returns the window that has keyboard focus.

    Note: win must be on the same desktop as currently
    focused window.
    """
    
    return win.rpDesk.activeWnd.Win32Thread.inputQ.focusedWnd

# Some easy callbacks
def is_visible(win):
    """Return true if the window is visible"""
    style = get_flags(WINDOW_STYLES, win.dwStyle)
    return "WS_VISIBLE" in style

# For use with edit boxes
def RtlRunDecodeUnicodeString(key, data):
    s = "".join([ chr(ord(data[i-1]) ^ ord(data[i]) ^ key)
                  for i in range(1,len(data)) ])
    s = chr(ord(data[0]) ^ (key | 0x43)) + s
    return s

def get_tb(win):
    """Gets the Toolbar object for a window of class ToolbarWindow32"""
    vm = win.Win32Thread.pEThread.ThreadsProcess.vm
    if win.cbWndExtra == 0:
        return
    tb_addr = read_value(vm, 'pointer', win.get_member_offset('WndExtra'))
    tb = Object("_TOOLBAR_WINDOW32", tb_addr, vm, profile=win.profile)
    return tb

def get_richedit_text(win):
    """Gets the text for a window of class RichEdit20W"""
    vm = win.Win32Thread.pEThread.ThreadsProcess.vm
    if win.cbWndExtra == 0:
        return
    red_addr = read_value(vm, 'pointer', win.get_member_offset('WndExtra'))
    red = Object("CTxtWinHost", red_addr, vm, profile=win.profile)
    if not (red.is_valid() and red.pTxtEdit.is_valid() and
            red.pTxtEdit.pTextDesc.is_valid()):
        return

    return red.pTxtEdit.pTextDesc.pBuff

def get_editbox_text(win):
    """Gets the text for a window of class Edit"""
    vm = win.Win32Thread.pEThread.ThreadsProcess.vm
    if win.cbWndExtra == 0:
        return
    ed_addr = read_value(vm, 'pointer', win.get_member_offset('WndExtra'))
    ed = Object("_EDIT_BOX", ed_addr, vm, profile=win.profile)
    return ed.hBuf

def get_sb_text(win, i):
    """Gets the text for a window of class msctls_statusbar32"""
    vm = win.Win32Thread.pEThread.ThreadsProcess.vm
    if win.cbWndExtra == 0:
        return
    
    sb_addr = read_value(vm, 'pointer', win.get_member_offset('WndExtra'))
    sb = Object("_STATUS_BAR", sb_addr, vm, profile=win.profile)
    if i >= sb.numParts:
        return
    return sb.Parts[i].pText

# Helper for get_tb
def get_parent_container(win,limit=4):
    i = 1
    parent = win
    while i < limit and not parent.dwStyleEx & WINDOW_STYLES_EX['WS_EX_CONTROLPARENT']:
        parent = parent.pParentWindow
        i += 1
    return parent

def get_tb_buttons(win):
    """return a list of (name, rect) pairs for all buttons on the toolbar"""
    tb = get_tb(win)
    if not tb: return []
    if not tb.is_valid(): return []

    clipwin = get_parent_container(win) or win
    clip_right = min(clipwin.rcWnd.right, win.rcWnd.right)
    
    button_infos = []
    buttons = tb.pButtonList
    if not buttons or not all(buttons): return []

    height = win.rcWnd.bottom - win.rcWnd.top
    widths = [ buttons[i+1].dwLeft - buttons[i].dwLeft for i in range(len(buttons)-1) ]
    widths.append(clip_right - (buttons[-1].dwLeft+win.rcWnd.left))

    for b,w in zip(buttons,widths):
        sep = bool(b.fsStyle & TBSTYLES["TBSTYLE_SEP"])
        if b.nStringIndex < tb.nStringEntries:
            name = "" if sep else tb.pStringPtrTable[b.nStringIndex]
        else:
            name = ""
        topleft = (win.rcWnd.left + b.dwLeft, win.rcWnd.top + b.dwTop)
        bottomright = (topleft[0] + w, topleft[1] + height)
        rect = (topleft[0], topleft[1], bottomright[0], bottomright[1])
        button_infos.append( (name, rect) )
    return button_infos

