# Volatility
# Copyright (C) 2008 Volatile Systems
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or (at
# your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details. 
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
#

"""
@author:       AAron Walters and Brendan Dolan-Gavitt
@license:      GNU General Public License 2.0 or later
@contact:      awalters@volatilesystems.com,bdolangavitt@wesleyan.edu
@organization: Volatile Systems
"""

import forensics.win32.gdi as gdi

from forensics.win32.gdi import get_flags
from forensics.object2 import *
from forensics.win32.tasks import process_list
from forensics.win32.lists import list_entry
from struct import unpack
from vutils import *

atoms = {}

def print_win(win, level=0):
    try:
        img_name = win.Win32Thread.pEThread.ThreadsProcess.ImageFileName
    except:
        img_name = None
    try:
        atid = win.pClassInfo.ClassAtom
        wndclass = atoms[atid].Name
    except KeyError:
        wndclass = "[unknown %#x]" % win.pClassInfo.ClassAtom
    try:
        atid = win.pClassInfo.SuperclassAtom
        wndclass_super = atoms[atid].Name
    except KeyError:
        wndclass_super = "[unknown %#x]" % win.pClassInfo.SuperclassAtom

    style = get_flags(gdi.WINDOW_STYLES, win.dwStyle)
    style_ex = get_flags(gdi.WINDOW_STYLES_EX, win.dwStyleEx)
    print "  " * level + hex(win.handleId), repr(win.WindowText.Buffer), "(%s)" % img_name, hex(win.offset),
    print win.rcWnd, "[%s] [%s]" % (",".join(style), ",".join(style_ex)),
    print "Class:", wndclass, "Super:", wndclass_super

def print_if_owned_by(exename):
    def check_exe(win, level):
        try:
            img_name = win.Win32Thread.pEThread.ThreadsProcess.ImageFileName
        except:
            img_name = None
        if img_name == exename:
            print_win(win, level)
    return check_exe

def get_threads(proc):
    return list_entry(proc.vm, types, proc.profile, 
                      proc.ThreadListHead.v(), "_ETHREAD",
                      fieldname="ThreadListEntry")

class window_list(forensics.commands.command):

    # Declare meta information associated with this plugin
    
    meta_info = forensics.commands.command.meta_info 
    meta_info['author'] = 'Brendan Dolan-Gavitt'
    meta_info['copyright'] = 'Copyright (c) 2007,2008 Brendan Dolan-Gavitt'
    meta_info['contact'] = 'bdolangavitt@wesleyan.edu'
    meta_info['license'] = 'GNU General Public License 2.0 or later'
    meta_info['url'] = 'http://moyix.blogspot.com/'
    meta_info['os'] = 'WIN_32_XP_SP2'
    meta_info['version'] = '1.0'

    def parser(self):
        forensics.commands.command.parser(self)

        self.op.add_option('-a', '--atom-table',
            help='address of atom table',
            type='int', dest='atom')
        
        # Atom table addresses: Xen  0x1a8b98
        #                       NIST 0x1a70f8

    def help(self):
        return  "Print window hierarchy"
    
    def find_top_window(self, process_address_space, types, symtab, theProfile):
        top_windows = []
        windows = set()
        pslist = process_list(process_address_space, types, symtab)
        for p in pslist:
            proc = Object("_EPROCESS", p, process_address_space, profile=theProfile)
            if not proc.vm: continue
            win = None
            for thrd in get_threads(proc):
                win = gdi.get_desktop_window(thrd)
                if win: break

            if win and win not in windows:
                top_windows.append( (win, thrd, proc) )
                windows.add(win)

        return top_windows

    def execute(self):
        from vtypes import xpsp2types
        
        xpsp2types['_ETHREAD'][1]['ThreadListEntry'] = [ 0x22c, ['_LIST_ENTRY']]
        xpsp2types['_KTHREAD'][1]['Win32Thread'] = [ 0x130, ['pointer', ['_W32THREAD']]]
        xpsp2types['_KTHREAD'][1]['ServiceTable'] = [ 0xe0, ['pointer', ['void']]]

        theProfile = Profile()
        theProfile.add_types(gdi.gdi_types)

        (addr_space, symtab, types) = load_and_identify_image(self.op,
            self.opts)

        # Load the atom table
        global atoms
        if self.opts.atom:
            AtomTableAddrOff = self.opts.atom
            win32k_base = gdi.find_win32k(addr_space, types, symtab)
            atom_table_off = win32k_base+AtomTableAddrOff
        else:
            from forensics.win32.findatomtable import find_atomtable
            atom_table_off = find_atomtable(addr_space, types, symtab, theProfile)

        addr_space = gdi.find_space(addr_space, types, symtab, atom_table_off)
        UserAtomTableHandle = read_value(addr_space, 'pointer', atom_table_off)
        atoms = gdi.read_atoms(addr_space, UserAtomTableHandle, theProfile)

        wins = self.find_top_window(addr_space,types,symtab,theProfile)
        for win, thrd, proc in wins:
            print "********** PID: %d Name: %s **********" % (proc.UniqueProcessId.v(), proc.ImageFileName)
            # Print them out to the console
            gdi.traverse_windows(win, fun=print_win)
            #gdi.traverse_windows(win, fun=print_if_owned_by('msimn.exe'))
            #gdi.traverse_windows(win, fun=print_tb)
