package laser.utilities;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;
import java.util.ArrayList;
import java.util.Vector;

import laser.utilities.*;

/**
 * Class BackgroundSubtractor - The BackgroundSubtractor subtracts the
 * background from the laser scans.  It works by placing points in 
 * buckets (size determined by user input), finding the bucket with
 * enough points to place it over a percentage threshold (value
 * selected through emperical results), and then replacing all values
 * in that bucket and beyond with "0".
 * 
 * Other Info: Each laser has 361 scan points.
 * 
 * Input:
 * "java BackgroundSubtractor <bin_size> <config_file> <output_dir>"
 * The config_file should be a file with the number of lasers at the 
 * top and the path to each laser file below that.  The laser files 
 * should be time and space registered laser scan files.  The output_dir
 * should already exist.  Files can be in it but may be overwritten.
 * 
 * Output:
 * The background subtracted files will be output to the directory
 * specified.  The file names will be "laserX_bgs.txt" where X is the
 * number of the laser in order of processing.  A config file will
 * also be created in the directory for input to the next step 
 * (Laser2BTF).  The config file will be named "config_bgs.cfg".
 *
 * TO DO:
 * 		- progress indicator
 * 		- 360 vs. 361 output scans
 * 		- get better results
 * 			- add sliding bins option
 */
public class LaserBackgroundSubtract {

	private BufferedReader fileReader;
	private BufferedWriter fileWriter;
	private static BufferedReader inputFileReader;
	private static BufferedWriter outputFileWriter;
	private int binSize;
	private int numberBins;
	private int[][] scanCounts;
	private int[] startBinToDelete = new int[361];
	private Vector scanData = new Vector();
	private int lineCount = 0;
	private double percentThreshold = .25;
	private static int numberOfLasers;
	private static Vector configData = new Vector();
	
	public LaserBackgroundSubtract(int inputBinSize) {
		binSize = inputBinSize;
		numberBins = (int) Math.ceil(80000/binSize);
		scanCounts = new int[362][numberBins];		
	} // end constructor
			
	private void calculateBins() {
		String currentLine;
		int lineDataAsInt;
		int binPlacement;

		// make sure it is empty before reading into scanData
		scanData.removeAllElements();

		try {
			while ((currentLine = inputFileReader.readLine()) != null) {
				lineCount++;
				String lineData[] =  currentLine.split(",");
				scanData.add(lineData);
				// using int i = 1 on purpose
				for (int i = 1; i < lineData.length; i++) {
					lineDataAsInt = Integer.valueOf(lineData[i].trim()).intValue();
					binPlacement = (int) Math.floor(lineDataAsInt / binSize);
					
					// added to account for what to do if the scan value is negative (skip increment)
					if (binPlacement >= 0) {
						this.scanCounts[i][binPlacement]++;						
					}
				} // end for lineData.length loop
			} // end while readLine != null
		} catch (IOException e) {
			System.out.println("Error reading scan points");
		}
	} // end calculateBins

	private void findFarthestPoints() {
		
		boolean foundBackgroundPoint = false;
		
		for (int i = 1; i < scanCounts.length; i++) {
			//System.out.println("Scan Position " + i);
			for (int j = 0; j < numberBins; j++) {
				if (foundBackgroundPoint == false) {
					if (scanCounts[i][j] != 0) {
						if (scanCounts[i][j] > (lineCount * percentThreshold)) {
							//System.out.println("    Bin Number " + j + "  Bin Count " + scanCounts[i][j]);
							foundBackgroundPoint = true;
							startBinToDelete[i-1] = j;
						} // over threshold
					} // not eq 0
				} // found = false
			} // for j
			foundBackgroundPoint = false; // reset for next bin
		}
	} // end findFarthestPoints
	
	private void writeOutPoints() {
		String[] lineOfData;
				
		try {
			for (int i = 0; i < scanData.size(); i++) {
				lineOfData = (String[]) scanData.get(i);
				for (int j = 0; j < lineOfData.length - 1; j++) {
					if (j == 0) {
						outputFileWriter.write(lineOfData[j]);
					}
					
					if (j > 0) {
						if (Integer.valueOf(lineOfData[j].trim()).intValue() < (startBinToDelete[j-1] * binSize)) {
							outputFileWriter.write(lineOfData[j]);						
						} else {
							outputFileWriter.write(" 0");
						}
					}
					
					if (j < lineOfData.length - 2) {
						outputFileWriter.write(",");
					}
				}
				outputFileWriter.newLine();
			}
		} catch (IOException e) {
			System.out.println("Error writing tracks to output files");
			exitProgram();
		}		
	} // end writeOutPoints
		
	public static void exitProgram() {
		System.out.println("Exiting program...");
		System.exit(1);
	} // end exitProgram
	
	public static void main(String[] args) throws Exception {
		BufferedWriter configFileWriter;
		String outFileName;
		String configFileOut;
		int laserId;
		
        if (args.length == 3) {
    	    (new File(args[2])).mkdir();

        	// create the new lbs object sending it the bin size specified by the user
    		LaserBackgroundSubtract lbs = new LaserBackgroundSubtract(Integer.valueOf(args[0]).intValue());

    	    File[] inputFiles = (new File(args[1])).listFiles();    	    
    	    numberOfLasers = inputFiles.length;

    		// initialize output config file into output directory
    		configFileOut = args[2] + File.separator + "config_bgs.cfg";
			File configOut = new File(configFileOut);
			configFileWriter = new BufferedWriter(new FileWriter(configOut));
			configFileWriter.write(Integer.toString(numberOfLasers));

    		// for each file in the input directory specified
    	    
    	    for(int i = 0; i < inputFiles.length; i++) {
    	        inputFileReader = new BufferedReader(new FileReader(inputFiles[i]));
    	        System.out.println(inputFiles[i].getName());
    	        
    	        configFileWriter.newLine();
    	        outFileName = args[2] + File.separator + inputFiles[i].getName();
    	        configFileWriter.write(outFileName);

    	        outputFileWriter = new BufferedWriter(new FileWriter(outFileName));
//    	        outputFileWriter.write(outFileName);

    	        lbs.calculateBins();
    	        lbs.findFarthestPoints();
    	        lbs.writeOutPoints();    	        
    	        
    	        outputFileWriter.close();
    	        inputFileReader.close();
    	    } // end of for each file
    	    
    	    configFileWriter.close();
    	            		
        		// get next input and output and send on
 //       		for (int i = 1; i < configData.size(); i++) {
//        			lbs.calculateBins();
//        			lbs.findFarthestPoints();
//        			lbs.writeOutPoints();
//        		}

        } else {
        	System.out.println("usage: java LaserBackgroundSubtract <bin_size> <input_dir> <output_dir>");
        }        
	} // end main
}
