package laser.utilities;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.*;
import java.awt.Color;
import java.awt.geom.*;

import laser.blobdetect.*;
import laser.tracker.Track;

/**
 * Class DataSet - Represents data sets
 * For now, only represents background subtracted btf as input and
 * 		tracked data as output (btf)
 * 
 * @author <A HREF="mailto:summer@cc.gatech.edu">Summer Adams</A>
 * @version Version 1.0, February 27, 2006
 */
public class DataSet implements LaserTrackerConstants {
	
	private String inputDir;
	private String outputDir;
	private Vector originalScanPoints = new Vector();
	private String curTime;
	private BufferedReader idReader;
	private BufferedReader idReader2;
	private BufferedReader timeReader;
	private BufferedReader xReader;
	private BufferedReader yReader;
	private BufferedWriter timeWriter;
	private BufferedWriter xWriter;
	private BufferedWriter yWriter;
	private BufferedWriter idWriter;
	private Vector unusedScanPoints = new Vector();
	private Vector newUnusedScanPoints = new Vector();
	private Vector currentPassScanPoints = new Vector();
	private Vector previousPassScanPoints = new Vector();
	private Vector blobs = new Vector();
	private Vector blobs2 = new Vector();
	private int uniqueBlobId = 0;
					
	public DataSet(String inputDir, String outputDir) {
		this.inputDir = inputDir;
		this.outputDir = outputDir;
	} // End of Constructor
		
	/**
	 * Creates the buffered readers and writers
	 */
	public void createReadersAndWriters() {
		try {
			File timeIn = new File(inputDir + File.separator + TIME_COLLAPSE_BTF);
			File xIn = new File(inputDir + File.separator + X_COLLAPSE_BTF);
			File yIn = new File(inputDir + File.separator + Y_COLLAPSE_BTF);
			
			timeReader = new BufferedReader(new FileReader(timeIn));
			xReader = new BufferedReader(new FileReader(xIn));
			yReader = new BufferedReader(new FileReader(yIn));
			
			File timeOut = new File(outputDir + File.separator + TIME_STAMP_BTF);
			File xOut = new File(outputDir + File.separator + X_SCALE_BTF);
			File yOut = new File(outputDir + File.separator + Y_SCALE_BTF);
			File idOut = new File(outputDir + File.separator + ID_BTF);
			
			timeWriter = new BufferedWriter(new FileWriter(timeOut));
			xWriter = new BufferedWriter(new FileWriter(xOut));
			yWriter = new BufferedWriter(new FileWriter(yOut));
			idWriter = new BufferedWriter(new FileWriter(idOut));		

		} catch (IOException e) {
			System.out.println("Error loading input files");
			exitProgram();
		}
	} // createReadersAndWriters

	/**
	 * Creates the buffered readers for drawing only
	 */
	public void createReaders() {
		try {
			File timeIn = new File(inputDir + File.separator + TIME_COLLAPSE_BTF);
			File xIn = new File(inputDir + File.separator + X_COLLAPSE_BTF);
			File yIn = new File(inputDir + File.separator + Y_COLLAPSE_BTF);
			
			timeReader = new BufferedReader(new FileReader(timeIn));
			xReader = new BufferedReader(new FileReader(xIn));
			yReader = new BufferedReader(new FileReader(yIn));
		} catch (IOException e) {
			System.out.println("Error loading input files");
			exitProgram();
		}
	} // createReadersAndWriters

	public void writeLineToBTF(){
		try {
			for (int i = 0; i < blobs.size(); i++) {
				Blob blobObject = (Blob) blobs.get(i);
				
				if (blobObject.getBlobScanPoints().size() > 7) {
					timeWriter.write(curTime);
					timeWriter.newLine();
				
					Double xPoint = new Double(blobObject.getXOfCenter());
					xWriter.write(xPoint.toString());
					xWriter.newLine();
				
					Double yPoint = new Double(blobObject.getYOfCenter());
					yWriter.write(yPoint.toString());
					yWriter.newLine();
								
					int id = blobObject.getBlobId();
					idWriter.write("" + id);
					idWriter.newLine();
				}
			}	
		} catch (IOException e) {
			System.out.println("Error writing tracks to output files");
			exitProgram();
		}
	}

	public void loadScanPoints(){
		unusedScanPoints.removeAllElements();
		originalScanPoints.removeAllElements();
		try {
			curTime = timeReader.readLine();
			if (curTime != null) {
				String xLine[] =  xReader.readLine().split(",");
				String yLine[] =  yReader.readLine().split(",");
					
				for (int i = 0; i < xLine.length; i++) {
					double xPoint = Double.parseDouble(xLine[i]);
					double yPoint = Double.parseDouble(yLine[i]);
					originalScanPoints.add(new Point2D.Double(xPoint, yPoint));
				}
			}			
			unusedScanPoints = originalScanPoints;
			currentPassScanPoints.removeAllElements();
			previousPassScanPoints.removeAllElements();
		} catch (IOException e) {
			System.out.println("Error loading scan points");
		}
	} // loadScanPoints
	
	public void collapseDataSet() {
		// Create the readers and writers to collapse the data set
		try {
			Vector collapsedData = new Vector();
			File timeIn = new File(inputDir + File.separator + TIME_STAMP_BTF);
			File xIn = new File(inputDir + File.separator + SCAN_X_IMAGE_BTF);
			File yIn = new File(inputDir + File.separator + SCAN_Y_IMAGE_BTF);
			File idIn = new File(inputDir + File.separator + ID_BTF);
			
			timeReader = new BufferedReader(new FileReader(timeIn));
			xReader = new BufferedReader(new FileReader(xIn));
			yReader = new BufferedReader(new FileReader(yIn));
			idReader = new BufferedReader(new FileReader(idIn));
			
			File timeOut = new File(inputDir + File.separator + TIME_COLLAPSE_BTF);
			File xOut = new File(inputDir + File.separator + X_COLLAPSE_BTF);
			File yOut = new File(inputDir + File.separator + Y_COLLAPSE_BTF);
			File idOut = new File(inputDir + File.separator + ID_COLLAPSE_BTF);
			
			timeWriter = new BufferedWriter(new FileWriter(timeOut));
			xWriter = new BufferedWriter(new FileWriter(xOut));
			yWriter = new BufferedWriter(new FileWriter(yOut));
			idWriter = new BufferedWriter(new FileWriter(idOut));		

			String timestamp = timeReader.readLine();
			String id = idReader.readLine();
			String xscale = xReader.readLine();
			String yscale = yReader.readLine();
			System.out.println("about to read in files");
			while (timestamp != null) {
				CollapsedData c = new CollapsedData(timestamp, id, "type",
						xscale, yscale, "xscan", "yscan");
				collapsedData.add(c);
				
				timestamp = timeReader.readLine();
				id = idReader.readLine();
				xscale = xReader.readLine();
				yscale = yReader.readLine();
			}
			
			System.out.println("Done Sorting frames by timestamp...");
			
			Object[] sortedData = collapsedData.toArray();
			Arrays.sort(sortedData);
			
			System.out.println("Combining the Data...");
			sortedData = removeDuplicates(sortedData);
			System.out.println("Finished combining the Data...");
			
			timeReader.close();
			idReader.close();
			xReader.close();
			yReader.close();
	
			System.out.println("Creating the collapsed files..");
			System.out.println("Total Frames: " + sortedData.length);
			System.out.println("Printing (.) every 500 frames");
			
			for (int i = 0; i < sortedData.length; i++) {
				if (i % 500 == 0) {
					System.out.print(".");
				}
				
				CollapsedData c = (CollapsedData) sortedData[i];
				
				xWriter.write(c.getXscale());
				xWriter.newLine();
				
				yWriter.write(c.getYscale());
				yWriter.newLine();
								
				timeWriter.write("" + c.getTimestamp());
				timeWriter.newLine();
				
				idWriter.write(c.getId());
				idWriter.newLine();				
			}
			
			xWriter.close();
			yWriter.close();
			timeWriter.close();
			idWriter.close();
			
			System.out.println("");
			System.out.println("Done creating *.btf files (collapsed btf files)");

		} catch (IOException e) {
			System.out.println("Error loading input files for collapse step");
			exitProgram();
		}
	}

	/* written by Summer to run at very end of tracking */
	public void compressIds() {
		// Create the readers and writers to collapse the data set
		try {
			Boolean foundTrackId;
			Vector idList = new Vector();
			File idIn = new File(outputDir + File.separator + ID_BTF);
			
			idReader = new BufferedReader(new FileReader(idIn));
			idReader2 = new BufferedReader(new FileReader(idIn));
			
			File idOut = new File(outputDir + File.separator + "compressId.btf");
			
			idWriter = new BufferedWriter(new FileWriter(idOut));		

			String id = idReader.readLine();
			System.out.println("compressing track ids ...");
			System.out.println("assigning new ids ...");
			while (id != null) {
				foundTrackId = false;

				// for all ids in List, see if this one is already in there
				for (int i = 1; i < idList.size(); i++) {
					//System.out.println("idList size " + idList.size());
					//System.out.println("id " + id + " idList.get " + idList.get(i));
					if (Integer.parseInt(id) == Integer.parseInt((String) idList.get(i))) {
						//System.out.println("found id in list " + id);
						foundTrackId = true;
					}
				}
				
				// if not in the list, add it
				if (!foundTrackId) {
					//System.out.println("didn't find id " + id);
					//System.out.println("idList size " + idList.size());
					idList.add(id);
				}
				
				// read in new line and repeat above until all unique id's have been put in list
				id = idReader.readLine();
			}
			idReader.close();
			
			System.out.println("idList size " + idList.size());
			System.out.println("done with reading lines in");
			
			int[] newIdList = new int[idList.size()];
			for (int i = 0; i < idList.size()-1; i++) {
				newIdList[i] = Integer.parseInt((String) idList.get(i+1));
			}
			
			System.out.println("done assigning ids");
			
			id = idReader2.readLine();
			System.out.println("writing new track ids ...");
			int newId;
			while (id != null) {
				for (int i = 0; i < newIdList.length; i++)
					if (newIdList[i] == Integer.parseInt(id)) {
						newId = i+1;
						idWriter.write("" + newId);
						idWriter.newLine();
					}
				
				id = idReader2.readLine();				
			}
			
			idReader2.close();
			idWriter.close();
			System.out.println("done writing new track ids");
			
			System.out.println("done compressing ids");
			
		} catch (IOException e) {
			System.out.println("Error loading input files for compress step");
			exitProgram();
		}
	}

	private Object[] removeDuplicates(Object[] array) {
		System.out.println("Printing (.) every 500 frames");
		Vector v = new Vector();
		CollapsedData prev = (CollapsedData) array[0];
		v.add(prev);
		for (int i = 1; i < array.length; i++) {
			if (i % 500 == 0) {
				System.out.print(".");
			}
			CollapsedData temp = (CollapsedData) array[i];
			if (!(prev.getTimestamp() == temp.getTimestamp())) {
				v.add(temp);
			} else {
				String combinedXScale = prev.getXscale() + ", " + temp.getXscale();
				String combinedYScale = prev.getYscale() + ", " + temp.getYscale();
				String combinedScanX =  prev.getScanx() + ", " + temp.getScanx();
				String combinedScanY = prev.getScany() + ", " + temp.getScany();
				CollapsedData t = (CollapsedData) v.get(v.size() - 1);
				t.setXscale(combinedXScale);
				t.setYscale(combinedYScale);
				t.setScanx(combinedScanX);
				t.setScany(combinedScanY);
			}
			prev = (CollapsedData) v.get(v.size() - 1);
		}
		System.out.println();
		return v.toArray();
	}

	public void closeReadersAndWriters() {
		try {
			timeReader.close();
			xReader.close();
			yReader.close();
			idReader.close();
			timeWriter.close();
			xWriter.close();
			yWriter.close();
			idWriter.close();		

		} catch (IOException e) {
			System.out.println("Error: Error closing files");
			exitProgram();
		}
	} // closeReadersAndWriters

	public void closeReaders() {
		try {
			timeReader.close();
			xReader.close();
			yReader.close();
		} catch (IOException e) {
			System.out.println("Error: Error closing files");
			exitProgram();
		}
	} // closeReaders

	public void removeCollapseFiles() {
		File timeOut = new File(inputDir + File.separator + TIME_COLLAPSE_BTF);
		File xOut = new File(inputDir + File.separator + X_COLLAPSE_BTF);
		File yOut = new File(inputDir + File.separator + Y_COLLAPSE_BTF);
		File idOut = new File(inputDir + File.separator + ID_COLLAPSE_BTF);
			
		timeOut.delete();
		xOut.delete();
		yOut.delete();
		idOut.delete();
	} // removeCollapseFiles
	
	public static void exitProgram() {
		System.out.println("Exiting program...");
		System.exit(1);
	} // exitProgram

	public Vector getUnusedScanPoints() {
		return unusedScanPoints;
	}

	public Vector getCurrentPassScanPoints() {
		return currentPassScanPoints;
	}

	public Vector getPreviousPassScanPoints() {
		return previousPassScanPoints;
	}

	public Vector getBlobs() {
		return blobs;
	}

	public void calculateNextBlob() {
		Point2D.Double point;
		Point2D.Double currentCenter;
		Point2D.Double oldCenter;
		Point2D.Double currentScanPoint;
		Point2D.Double newCenter;
		int randomIndex, i, blobSize;
		Random random = new Random();
		double xSum;
		double ySum;
		
		uniqueBlobId++;
		
		// randomly choose a point in the scan
		if (unusedScanPoints.size() > 0) {
			randomIndex = Math.abs(random.nextInt()) % unusedScanPoints.size();
			currentCenter = (Point2D.Double) unusedScanPoints.get(randomIndex);
			oldCenter = (Point2D.Double) unusedScanPoints.get(randomIndex);

			currentPassScanPoints.removeAllElements();
			newUnusedScanPoints.removeAllElements();
			currentPassScanPoints.add(currentCenter);
		
			// use do while to make sure it is executed at least once
			i = 0;
			do {
				xSum = 0;
				ySum = 0;

				// calculate the new center	
				// add up all the x, and all the y, divide by number of points
				for (int j = 0; j < currentPassScanPoints.size(); j++) {
					currentScanPoint = (Point2D.Double) currentPassScanPoints.get(j);
					xSum += currentScanPoint.x;
					ySum += currentScanPoint.y;
				}
			
				newCenter = new Point2D.Double();
				newCenter.x = xSum / currentPassScanPoints.size();
				newCenter.y = ySum / currentPassScanPoints.size();			
				currentCenter = newCenter;
			
				currentPassScanPoints.removeAllElements();
				newUnusedScanPoints.removeAllElements();
			
				// find all relevant points based on center
				for (int k = 0; k < unusedScanPoints.size(); k++) {
					point = (Point2D.Double) unusedScanPoints.get(k);
					if (point.distance(currentCenter) < 0.6) {
						currentPassScanPoints.add(point);
					} else {
						newUnusedScanPoints.add(point);					
					}
				}
			
				i++;
			} while (i < 20);
		
		// add points to the blob
			blobSize = blobs.size() + 1;
//			blobs.add(new Blob(currentCenter, oldCenter, currentPassScanPoints, blobSize));
//			if (currentPassScanPoints.size() > 2) {
				blobs.add(new Blob(currentCenter, oldCenter, currentPassScanPoints, uniqueBlobId));				
//			}

//			System.out.print("blob id: " + uniqueBlobId + " ");
			
			unusedScanPoints.removeAllElements();
			unusedScanPoints.addAll(newUnusedScanPoints);
		}
	}

	public void trackBlob() {
		// don't worry about saving points for now.  will write them out as we go later.
		// reset all vector scan point values.  done in loadscanpoints
		// for each blob, get center point
		Point2D.Double center;
		Point2D.Double newCenter;
		Point2D.Double point;
		Point2D.Double currentScanPoint;
		Blob blobObject;
		double xSum;
		double ySum;
		int blobId;

		newUnusedScanPoints.removeAllElements();

		//System.out.println("Tracking blobs.  blobs.size: " + blobs.size());
		
		for (int i = 0; i < blobs.size(); i++){
			blobObject = (Blob) blobs.get(i);
			center = (Point2D.Double) blobObject.getBlobCenter();
			blobId = (int) blobObject.getBlobId();
			
			// find points close enough, remove from unused
			for (int k = 0; k < unusedScanPoints.size(); k++) {
				point = (Point2D.Double) unusedScanPoints.get(k);
				if (point.distance(center) < 0.6) {
					currentPassScanPoints.add(point);
				} else {
					newUnusedScanPoints.add(point);					
				}
			}
			// calculate new center then add the blob
			xSum = 0;
			ySum = 0;

			// calculate the new center	
			// add up all the x, and all the y, divide by number of points
			for (int j = 0; j < currentPassScanPoints.size(); j++) {
				currentScanPoint = (Point2D.Double) currentPassScanPoints.get(j);
				xSum += currentScanPoint.x;
				ySum += currentScanPoint.y;
			}
		
			newCenter = new Point2D.Double();
			newCenter.x = xSum / currentPassScanPoints.size();
			newCenter.y = ySum / currentPassScanPoints.size();

			/* blob list seems to be growing very large.  try not adding that don't have
			   at least 3 points (or some other number) */
			/* maybe make this change in calculateNextBlob as well? */
			/* made change to initial blob create, removed change from here checking for min size */

			if (currentPassScanPoints.size() > 2) {
				blobs2.add(new Blob(newCenter, center, currentPassScanPoints, blobId));				
			}

			unusedScanPoints.removeAllElements();
			currentPassScanPoints.removeAllElements();
			unusedScanPoints.addAll(newUnusedScanPoints);
			newUnusedScanPoints.removeAllElements();
		}
		
		blobs.removeAllElements();
		blobs.addAll(blobs2);
		blobs2.removeAllElements();
		
		/* BEGIN - making mods to add in new blobs
		 * basically if after tracking all blobs, the unused scan points has a number
		 * of points left, check to see if a large enough group are together to add as 
		 * a new blob
		 * 
		 */
		
		while (unusedScanPoints.size() > 7) {
			//System.out.println("In new blob detect.  unusedSize: " + unusedScanPoints.size());
			calculateNextBlob();
		}
				
		/* END - making mods to add in new blobs */
		
	}

	public void createNewBlob() {
		Point2D.Double randomPoint;
		int randomIndex;
		Random random = new Random();
		
		//randomly choose a point in the scan
		randomIndex = Math.abs(random.nextInt()) % unusedScanPoints.size();
		randomPoint = (Point2D.Double) unusedScanPoints.get(randomIndex);
		
		unusedScanPoints.removeElementAt(randomIndex);
		currentPassScanPoints.add(randomPoint);
//		blobs.add(new Blob(randomPoint));
	}
	
	// Run through the entire data set to create all blobs; assuming that this is being called immediately
	public void completeAllBlobs() {
		// first pass
		while (unusedScanPoints.size() > 0) {
			calculateNextBlob();
		}
		
		loadScanPoints();
		
	 	System.out.println("Creating the blobs and writing to btf..");
		System.out.println("Printing (.) every 500 timestamp lines (complete blob set for timestamp) written");		
		int i = 0;
		
		while (curTime != null) {
			i++;
			if (i % 500 ==0) {
				System.out.print(".");
			}
			trackBlob();
			writeLineToBTF();
			loadScanPoints();
		}
		System.out.println("");
	}

}
