package laser.utilities;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.StringTokenizer;
import java.util.Vector;

/**
 * <PRE>
 * Class ConfigFileHandler - Handles the configuration file passed as
 * an argument to the different programs for the laser traker. 
 * 
 * The configuration file must be in the following format:
 * <number_of_lasers>
 * <name> <x> <y> <theta> <color1> <color2> <color3> <filename>
 * .....
 * <PRE>
 * 
 * @author <A HREF="mailto:2adonis5@cc.gatech.edu">Jorge</A>
 * @version Version 1.0, September 12, 2005
 */
public class ConfigFileHandler {

	/** The config file */
	private File configFile;
	
	/** 
	 * A vector of LaserData objects created from the information
	 * about the lasers in the cofig file 
	 */
	private Vector lasers;
	
	/** True if it is a stats config file */
	private boolean stats;
	
	/**
	 * Constructor. Passed parameter is the cofig file name.
	 * @param configFile
	 */
	public ConfigFileHandler(String configFileName, boolean stats) {
		this.configFile = new File(configFileName);
		this.stats = stats;
		lasers = getLasersInfo();
	}
	
	/**
	 * Constructor. Passed parameter is the cofig file name.
	 * @param configFile
	 */
	public ConfigFileHandler(String configFileName) {
		this.configFile = new File(configFileName);
		this.stats = false;
		lasers = getLasersInfo();
	}
	
	/**
	 * Exits the program.
	 */
	public void exitProgram() {
		System.exit(0);
	}
	
	/**
	 * Reads the first line of the config file. Which should specify the
	 * number of lasers.
	 * @param reader
	 * @return the number of lasers specified in the config file
	 */
	private int readNumberOfLasers(BufferedReader reader) {
		int numLasers = -1;
		String tmp = null;
		try {
			tmp = reader.readLine();
			while (tmp != null && tmp.startsWith("//")) {
				tmp = reader.readLine();
			}
		} catch (IOException e1) {
			System.out.println("IOException reading config file.");
            return -1;
		}
		if (tmp != null) {
        	try {
        		numLasers = Integer.parseInt(tmp);
        	} catch (NumberFormatException e) {
        		System.out.println(
        				"Malformatted config file. First line (number of " + 
						"lasers) is not an int.");
				return -1;
        	}
        } else {
        	System.out.println("The config file is (number of lasers part) " + 
        			           "is incorrectly formatted.");
        }
        return numLasers;
	}
	
	/**
	 * Gets the LaserData object of a laser from the config file
	 * @return
	 */
	private LaserData getLaserData(BufferedReader reader) {
		LaserData laserData = null;
		
		String tmp = null;
		try {
			tmp = reader.readLine();
			while (tmp != null && tmp.startsWith("//")) {
				tmp = reader.readLine();
			}
		} catch (IOException e1) {
			System.out.println("IOException reading config file.");
            return null;
		}
		if (tmp != null) {
			laserData = parseLaserData(tmp);
        } else {
        	System.out.println("The config file is incorrectly formatted.");
        }
        return laserData;
	}
	
	/**
	 * Parses the given line in the format:
	 * <name> <x> <y> <theta> <color1> <color2> <color3> <filename>
	 * and creates a LaserData object from that information
	 * @param line
	 * @return
	 */
	private LaserData parseLaserData(String line) {
		LaserData retval = null;
		String laserName = null;
		double xPosition = -1;
		double yPosition = -1;
		double theta = -1;
		int colorR = -1;
		int colorG = -1;
		int colorB = -1;
		String fileName = null;
		StringTokenizer tokenizer = new StringTokenizer(line, " ");
		int i = 0;
		try {
			while (tokenizer.hasMoreTokens()) {
				i = i + 1;
				String token = tokenizer.nextToken();
				switch (i) {
					case 1:
						laserName = token;
						break;
					case 2:
						if (stats) {
							fileName = token;
							yPosition = 0;
							theta = 0;
							colorR = 1;
							colorG = 1;
							colorB = 1;
							return new LaserData(laserName, xPosition, yPosition, theta,
									colorR, colorG, colorB, fileName);
						} else {
							xPosition = Double.parseDouble(token);
						}
						break;
					case 3:
						yPosition = Double.parseDouble(token);
						break;
					case 4:
						theta = Double.parseDouble(token);
						break;
					case 5:
						colorR = Integer.parseInt(token);
						break;
					case 6:
						colorG = Integer.parseInt(token);
						break;		
					case 7:
						colorB = Integer.parseInt(token);
						break;
					case 8:
						fileName = token;
				}
			}
		} catch (NumberFormatException e) {
			System.out.println(
				"Laser information in config file is malformatted.");
			return null;
		}
		if (laserName == null || xPosition == -1 || yPosition == -1 ||
				theta == -1 || colorR == -1 || colorG == -1 ||
				colorB  == -1 || fileName == null) {
			System.out.println(
				"Laser information in config file is malformatted.");
		} else {
			retval = new LaserData(laserName, xPosition, yPosition, theta,
					colorR, colorG, colorB, fileName);
		}
		return retval;
	}
	
	/**
	 * Reads the config file and then returns a Vector of LaserData
	 * objects created from the laser information in the config file.
	 * Also checks if the config file is written correctly,
	 * if not, it will exit the program and print an error.
	 * @param configFile
	 * @return
	 */
	private Vector getLasersInfo() {
		int numOfLasers = -1;
		Vector retval = null;
		if (!configFile.exists()) {
			System.out.println("The config file " + 
					configFile.getName() + " does not exist.");
			exitProgram();
		} else {
			try {
				BufferedReader tmpReader = 
                	new BufferedReader(new FileReader(configFile));
				numOfLasers = readNumberOfLasers(tmpReader);
				if (numOfLasers == -1) {
					exitProgram();
				} else {
					retval = new Vector();
					int index = 0;
					while (index < numOfLasers) {
						LaserData laserData = getLaserData(tmpReader);
						if (laserData == null) {
							exitProgram();
						} else {
							retval.add(laserData);
							index = index + 1;
						}
					}
					if (index == numOfLasers) {
						//This means that the specified number of lasers
						//in the configuration file matches the lines
						//of laser information
					} else {
						//If that is not the case, we need to tell the user
						//what the problem is and exit the program
						System.out.println(
								"The number of lasers specified in the " +
								"config file " + configFile + 
								" is incorrect. There is either " +
								"more laser information or less than that " +
								"number.");
						exitProgram();
								
					}
				}
            } catch(IOException e) {
                System.out.println("IOException cause in getLaserInfo.");
                exitProgram();
            }
        }
		return retval;
	}
	
	/**
	 * Creates a copy of the Vector of LaserData objects
	 * @return
	 */
	public Vector clon() {
		Vector retval = new Vector();
		for (int i = 0; i < lasers.size(); i++) {
			LaserData ld = (LaserData) lasers.get(i);
			LaserData clonLD = new LaserData(ld.getLaserName(),
					ld.getXPosition(), ld.getYPosition(), ld.getTheta(),
					ld.getColorR(), ld.getColorG(), ld.getColorB(),
					new String(ld.getFileName()));
			retval.add(clonLD);
		}
		return retval;
	}
	
	/**
	 * Retus LaserData objects of all the lasers information found
	 * in the configuration file
	 * @return
	 */
	public Vector getLasers() {
		return lasers;
	}
	
	/**
	 * Returns an array of fileNames of all the lasers.
	 * @return
	 */
	public String[] getLasersFileName() {
		String retval[] = new String[lasers.size()];
		for (int i = 0; i < lasers.size(); i++) {
			LaserData laserData = (LaserData) lasers.get(i);
			retval[i] = laserData.getFileName();
		}
		return retval;
	}
	
}
