package laser.utilities;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Arrays;
import java.util.Vector;

import laser.utilities.LaserTrackerConstants;

/**
 * <PRE>
 * Class Collapser - Collapses the .btf files to have only unique timestamps. 
 * Usage - java Collapser <input_dir> <output_dir>
 * <PRE>
 * 
 * @author <A HREF="mailto:2adonis5@cc.gatech.edu">Jorge</A>
 * @version Version 1.0, October 05, 2005
 */
public class Collapser implements LaserTrackerConstants {
	
	private static final String CURRENT_DIR = "." + File.separator;
	
	/** Input directory were the .btf files are */
	private String inputDir;
	
	/** Output directory to put the created .btf files */
	private String outputDir;
	
	/** The file raders */
	private BufferedReader timestampReader;
	private BufferedReader idReader;
	private BufferedReader typeReader;
	private BufferedReader xscaleReader;
	private BufferedReader yscaleReader;
	private BufferedReader scanxReader;
	private BufferedReader scanyReader;
	
	/** File writers */
	private BufferedWriter timestampWriter;
	private BufferedWriter idWriter;
	private BufferedWriter typeWriter;
	private BufferedWriter xScaleWriter;
	private BufferedWriter yScaleWriter;
	private BufferedWriter scanxXimageWriter;
	private BufferedWriter scanYimageWriter;
	
	/** The input files */
	private File timestampInputFile;
	private File typeInputFile;
	private File idInputFile;
	private File xScaleInputFile;
	private File yScaleInputFile;
	private File scanXInputFile;
	private File scanYInputFile;
	
	/**
	 * Constructor
	 * @param inputDir
	 * @param outpurDir
	 */
	public Collapser(String inputDir, String outputDir) {
		this.inputDir = inputDir;
		this.outputDir = outputDir;
		checkOutputDir();
		initReaders();
		initWriters();
	}
	
	/**
	 * Returns the line numbers of the lines that should go in the same line
	 * @return
	 */
	public int[] getLineNumbersFor(String timestamp) {
		Vector lineNumbers = new Vector();
		
		int i = 0;
		
		try {
			BufferedReader tmpReader = 
            	new BufferedReader(new FileReader(timestampInputFile));
			
			String temp = tmpReader.readLine();
			while (temp != null) {
				if (timestamp.equals(temp)) {
					lineNumbers.add(new Integer(i));
				}
				temp = tmpReader.readLine();
				i = i + 1;
			}
			tmpReader.close();
		} catch (IOException e) {
			System.out.println("Exception in getLineNumbersFor(" + timestamp + ")");
			exitProgram();
		}
		Object[] arr = lineNumbers.toArray();
		int[] retval = new int[lineNumbers.size()];
		for (int t = 0; t < lineNumbers.size(); t++) {
			retval[t] = ((Integer) lineNumbers.get(t)).intValue();
		}
		
		return retval;
	}
	
	/** Collapses the input .btf files to have only unique timestamps */
	public void collapse() {
		System.out.println("Sorting by timestamp...");
		try {
			Vector collapsedData = new Vector();
			String timestamp = timestampReader.readLine();
			String id = idReader.readLine();
			String type = typeReader.readLine();
			String xscale = xscaleReader.readLine();
			String yscale = yscaleReader.readLine();
			String xscan = scanxReader.readLine();
			String yscan = scanyReader.readLine();
			while (timestamp != null) {
				CollapsedData c = new CollapsedData(timestamp, id, type,
						xscale, yscale, xscan, yscan);
				collapsedData.add(c);
				
				timestamp = timestampReader.readLine();
				id = idReader.readLine();
				type = typeReader.readLine();
				xscale = xscaleReader.readLine();
				yscale = yscaleReader.readLine();
				xscan = scanxReader.readLine();
				yscan = scanyReader.readLine();
			}
			
			System.out.println("Done Sorting frames by timestamp...");
			
			Object[] sortedData = collapsedData.toArray();
			Arrays.sort(sortedData);
			
			System.out.println("Combining the Data...");
			sortedData = removeDuplicates(sortedData);
			System.out.println("Finished combining the Data...");
			
			timestampReader.close();
			idReader.close();
			typeReader.close();
			scanxReader.close();
			scanyReader.close();
			xscaleReader.close();
			yscaleReader.close();
	
			System.out.println("Creating the collapsed files..");
			System.out.println("Total Frames: " + sortedData.length);
			System.out.println("Printing (.) every 500 frames");
			
			for (int i = 0; i < sortedData.length; i++) {
				if (i % 500 == 0) {
					System.out.print(".");
				}
				
				CollapsedData c = (CollapsedData) sortedData[i];
				
				xScaleWriter.write(c.getXscale());
				xScaleWriter.newLine();
				
				yScaleWriter.write(c.getYscale());
				yScaleWriter.newLine();
				
				scanxXimageWriter.write(c.getScanx());
				scanxXimageWriter.newLine();
				
				scanYimageWriter.write(c.getScany());
				scanYimageWriter.newLine();
				
				timestampWriter.write("" + c.getTimestamp());
				timestampWriter.newLine();
				
				idWriter.write(c.getId());
				idWriter.newLine();
				
				typeWriter.write(c.getType());
				typeWriter.newLine();
				
			}
			
			closeFiles();
			System.out.println("");
			System.out.println("Done creating *.btf files (collapsed btf files)");
		} catch(IOException e) {
			System.out.println("IOException cause in collapase()");
			exitProgram();
        }
	}
	
	/**
	 * Closes the files.
	 */
	private void closeFiles() {
		try {
			timestampReader.close();
			timestampWriter.close();
			idWriter.close();
			typeWriter.close();
			xScaleWriter.close();
			yScaleWriter.close();
			scanxXimageWriter.close();
			scanYimageWriter.close();
		} catch (IOException e) {
			System.out.println("Error closing files...");
			exitProgram();
		}
	}
	
	/**
	 * Gets the collapsed line to write
	 * @param numberLines
	 * @param toReadFile
	 * @param idTypeTime
	 * @return
	 */
	public String getCollapsedLineToWrite(int[] numberLines, File toReadFile,
			boolean idTypeTime) {
		String lineToWrite = "";
		
		try {
			BufferedReader tmpReader = 
            	new BufferedReader(new FileReader(toReadFile));
			String temp = null;
			int i = 0;
			temp = tmpReader.readLine();
			int numberLinesIndex = 0; 
			while (temp != null) {
				if (temp != null && numberLinesIndex < numberLines.length) {
					if (idTypeTime) {
						if (i == numberLines[numberLinesIndex]) {
							lineToWrite = lineToWrite + temp;
							return lineToWrite;
						}
					} else {
						if (numberLinesIndex == (numberLines.length - 1)
								&& i == numberLines[numberLinesIndex]) {
							lineToWrite = lineToWrite + temp;
							numberLinesIndex++;
						} else if (i == numberLines[numberLinesIndex]) {
							lineToWrite = lineToWrite + temp + ", ";
							numberLinesIndex++;
						}
					}
				}
				temp = tmpReader.readLine();
				i++;
			}
			tmpReader.close();
		} catch(IOException e) {
			System.out.println("IOException cause in getCollapsedLineToWrite");
        }
		
		return lineToWrite;
	}
	
	/**
	 * Removes the duplicate timestamps in the passed array of CollapsedData
	 * by putting the data of the objects together
	 * @param array
	 * @return
	 */
	private Object[] removeDuplicates(Object[] array) {
		System.out.println("Printing (.) every 500 frames");
		Vector v = new Vector();
		CollapsedData prev = (CollapsedData) array[0];
		v.add(prev);
		for (int i = 1; i < array.length; i++) {
			if (i % 500 == 0) {
				System.out.print(".");
			}
			CollapsedData temp = (CollapsedData) array[i];
			if (!(prev.getTimestamp() == temp.getTimestamp())) {
				v.add(temp);
			} else {
				String combinedXScale = prev.getXscale() + ", " + temp.getXscale();
				String combinedYScale = prev.getYscale() + ", " + temp.getYscale();
				String combinedScanX =  prev.getScanx() + ", " + temp.getScanx();
				String combinedScanY = prev.getScany() + ", " + temp.getScany();
				CollapsedData t = (CollapsedData) v.get(v.size() - 1);
				t.setXscale(combinedXScale);
				t.setYscale(combinedYScale);
				t.setScanx(combinedScanX);
				t.setScany(combinedScanY);
			}
			prev = (CollapsedData) v.get(v.size() - 1);
		}
		System.out.println();
		return v.toArray();
	}
	
    /**
     * Checks if the output dir is valid
     * @return
     */
    private void checkOutputDir() {
    	if (outputDir == null || outputDir.equals("")) {
    		outputDir = CURRENT_DIR;
    	} else {
    		File output = new File(outputDir);
			if (!output.exists()) {
				System.out.println(
						"Specified output path in the argument does not " + 
						"exist. Your output will be in the current directory.");
				outputDir = CURRENT_DIR;
			}
		}
    }
	
    /**
     * Exits the program.
     */
	public static void exitProgram() {
		System.out.println("Exiting program...");
		System.exit(1);
	}
	
	/**
	 * Initializes the file writers
	 */
	private void initWriters() {
		try {
			File timestampFile = new File(outputDir + File.separator + TIME_STAMP_BTF
					+ COLLAPSED_EXTENSION);
			File typeFile = new File(outputDir + File.separator + TYPE_BTF
					+ COLLAPSED_EXTENSION);
			File idFile = new File(outputDir + File.separator + ID_BTF
					+ COLLAPSED_EXTENSION);
			File xScaleFile = new File(outputDir + File.separator + X_SCALE_BTF
					+ COLLAPSED_EXTENSION);
			File yScaleFile = new File(outputDir + File.separator + Y_SCALE_BTF
					+ COLLAPSED_EXTENSION);
			File scanXFile = new File(outputDir + File.separator + SCAN_X_IMAGE_BTF
					+ COLLAPSED_EXTENSION);
			File scanYFile = new File(outputDir + File.separator + SCAN_Y_IMAGE_BTF
					+ COLLAPSED_EXTENSION);
			
			timestampWriter = new BufferedWriter(new FileWriter(timestampFile));
			idWriter = new BufferedWriter(new FileWriter(idFile));
			typeWriter = new BufferedWriter(new FileWriter(typeFile));
			xScaleWriter = new BufferedWriter(new FileWriter(xScaleFile));
			yScaleWriter = new BufferedWriter(new FileWriter(yScaleFile));
			scanxXimageWriter = new BufferedWriter(new FileWriter(scanXFile));
			scanYimageWriter = new BufferedWriter(new FileWriter(scanYFile));
			
		} catch (IOException e) {
			System.out.println("Error initializing input .btf files");
			exitProgram();
		}
	}
	
	/**
	 * Initializes the file readers
	 */
	private void initReaders() {
		try {
			
			timestampInputFile = new File(inputDir + File.separator + TIME_STAMP_BTF);
			typeInputFile = new File(inputDir + File.separator + TYPE_BTF);
			idInputFile = new File(inputDir + File.separator + ID_BTF);
			xScaleInputFile = new File(inputDir + File.separator + X_SCALE_BTF);
			yScaleInputFile = new File(inputDir + File.separator + Y_SCALE_BTF);
			scanXInputFile = new File(inputDir + File.separator + SCAN_X_IMAGE_BTF);
			scanYInputFile = new File(inputDir + File.separator + SCAN_Y_IMAGE_BTF);
			
			boolean filesExist = true;
			filesExist = timestampInputFile.exists();
			filesExist = typeInputFile.exists();
			filesExist = idInputFile.exists();
			filesExist = xScaleInputFile.exists();
			filesExist = yScaleInputFile.exists();
			filesExist = scanXInputFile.exists();
			filesExist = scanYInputFile.exists();
			
			if (!filesExist) {
				System.out.println("One or more of the input .btf files does not exist");
				exitProgram();
			}
			
			timestampReader = new BufferedReader(new FileReader(timestampInputFile));
			idReader = new BufferedReader(new FileReader(idInputFile));
			typeReader = new BufferedReader(new FileReader(typeInputFile));
			xscaleReader = new BufferedReader(new FileReader(xScaleInputFile));
			yscaleReader = new BufferedReader(new FileReader(yScaleInputFile));
			scanxReader = new BufferedReader(new FileReader(scanXInputFile));
			scanyReader = new BufferedReader(new FileReader(scanYInputFile));
			
		} catch (IOException e) {
			System.out.println("Error initializing input .btf files");
			exitProgram();
		}
	}
	
	public static void main(String[] args){
		if (args.length == 2){
			System.out.println("Starting program...");
			Collapser collapser = new Collapser(args[0], args[1]);
			collapser.collapse();
		} else
			System.out.println("usage: java Collapser <input_dir> <output_dir>");
	}
	
}
