// Usage:  java Register.java <config_file> <out_file> <laser_line>

// Reads in <config_file> to find information about lasers (files & initial guess placement),
// Gets <laser_line> from each file, registers the lasers in space and outputs exact locations
// to a new config file (<out_file>)
// Note:  <laser_line> starts with line 0

import java.io.*;
import java.util.*;


public class Register {
  // Global Constant List
  public static final int SCANSPERLASER = 361;

  // Global Variable List
  public static int NUMBEROFLASERS = 4;
  public static int TOTALSCANS = NUMBEROFLASERS * SCANSPERLASER;
  public static double[][] bestGuess = new double[NUMBEROFLASERS][3];
  public static double[][] finalMap = new double[TOTALSCANS][2];
  public static double[][][] rawData = new double[NUMBEROFLASERS][SCANSPERLASER][2];
  public static double currentThresh = 5000.0;
  public static int pointsInMap = 0;


  public static void main(String args[]) throws Exception {
    // Variable List
    double[][] tempCartData = new double[TOTALSCANS][2];
    double[] tempChange = new double[3];
    double[] bestChange = new double[3];
    double oldError = Double.MAX_VALUE;
    double currentError = Double.MAX_VALUE / 1;
    double tempError = 0;
    double base = 100.0;
    int[] scansPerLaser = new int[NUMBEROFLASERS];

    // Read in laser data and initial guesses
    readData(args[0], Integer.valueOf(args[2]).intValue());

    // Find the number of 'hits' (non-8191) per laser
    for(int i = 0; i < NUMBEROFLASERS; i++)
      for(int j = 0; j < SCANSPERLASER; j++)
        if(rawData[i][j][0] != 8191)
          scansPerLaser[i]++;

    // Convert laser1 data into cartesian and store on final map
    for(int i = 0; i < SCANSPERLASER; i++)
      // Don't add 8191 points
      if(rawData[0][i][0] != 8191) {
        finalMap[pointsInMap] = polar2Cart(rawData[0][i], bestGuess[0]);
        pointsInMap++;
      }

    // Loop for each laser i past 1
    for(int i = 1; i < NUMBEROFLASERS; i++) {
      // Reset necessary variables
      oldError = Double.MAX_VALUE;
      currentError = Double.MAX_VALUE / 10;
      base = 100;

      // Loop until error minimized
      System.out.print("\nStarting Laser " + i);
      while(currentError < oldError) {
        System.out.print(".");
        oldError = currentError;

        for(double x = base; x >= 0.1; x/=10)
          // Try all changes
          for(int j = -1; j <= 1; j++)
            for(int k = -1; k <= 1; k++)
              for(int l = -1; l <= 1; l++) {
                // Set tempChange to current change
                tempChange[0] = bestGuess[i][0] + j * x * 1.0;
                tempChange[1] = bestGuess[i][1] + k * x * 1.0;
                tempChange[2] = bestGuess[i][2] + l * x / 1000.0;

                // Get error for current change
                tempError = getError(i, tempChange);

                // If error is less than currentError, save change as bestChange & update currentError
                if(tempError < currentError) {
                  bestChange[0] = tempChange[0];
                  bestChange[1] = tempChange[1];
                  bestChange[2] = tempChange[2];
                  currentError = tempError;
                }
              }

        // If we took a small step this time, then decrease size of max step.
        if(Math.abs(bestChange[0] - bestGuess[i][0]) + Math.abs(bestChange[1] - bestGuess[i][1]) +
           Math.abs(bestChange[2] - bestGuess[i][2]) * 1000 < base) {
          if(base == 100)
            System.out.print("\nGetting closer");
          else if(base == 10)
            System.out.print("\nAlmost there");
          else if(base == 1)
            System.out.print("\nJust about there");
          else
            System.out.print("\n");
          base /= 10;
        }

        // Update bestGuess
        bestGuess[i][0] = bestChange[0];
        bestGuess[i][1] = bestChange[1];
        bestGuess[i][2] = bestChange[2];
      }

      // Add points from laser i onto final map
      for(int j = 0; j < SCANSPERLASER; j++)
        // Don't add 8191 points
        if(rawData[i][j][0] != 8191) {
          finalMap[pointsInMap] = polar2Cart(rawData[i][j], bestGuess[i]);
          pointsInMap++;
        }
    }

    // Round to the nearest 0.1cm and 0.0001rad
    for(int i = 0; i < NUMBEROFLASERS; i++) {
      bestGuess[i][0] = Math.round(bestGuess[i][0] * 100.0) / 100.0;
      bestGuess[i][1] = Math.round(bestGuess[i][1] * 100.0) / 100.0;
      bestGuess[i][2] = Math.round(bestGuess[i][2] * 10000.0) / 10000.0;
    }

    // Output results
    outputData(args[0], args[1]);

    // Exit smoothly
    return;
  }


  // Reads in data
  // File should take the format of:
  //   First line is the (space/comma ("x, y") delimited) approximate location of first laser (x, y, theta)
  //   Second line is the scans for laser one (space/comma ("x, y") delimited) - extra fields removed
  //   Repeat for however many lasers there are
  // Note:  If the data files are not found in the location specified in the config file by following
  //        relative paths from where Java was run, they will be looked for by following relative paths
  //        from the location of the config file
  public static void readData(String filename, int laserLine) {
    try {
      // Variable List
      BufferedReader inputFile = new BufferedReader(new FileReader(new File(filename)));
      String line = new String();
      int pos = 0;
      String temp = new String();
      String path = (new File(filename)).getParent();
      ArrayList config = new ArrayList();
      String[] configData = new String[8];

      // Initialize configData
      for(int i = 0; i < configData.length; i++)
        configData[i] = new String();

      // ***** Read in config info *****
      // Get a line
      while((line = inputFile.readLine()) != null) {
        // Go character by character to eliminate comments
        for(int i = 0; i < line.length(); i++) {
          if(line.charAt(i) == '/' && i < line.length() - 1 && line.charAt(i+1) == '/')
            break;
          else
            temp += line.charAt(i);
        }

        // If the line was blank or only comments, continue
        if(temp.length() == 0)
          continue;

        // Add line to list of lines
        config.add(temp);

        // Reset variables
        temp = new String();
      }

      // Close config file
      inputFile.close();

      // ***** Parse config info *****
      // Get number of lasers (first word)
      NUMBEROFLASERS = (Integer.valueOf(((String) config.get(0)).split(" ")[0])).intValue();
      TOTALSCANS = NUMBEROFLASERS * SCANSPERLASER;

      // For each laser
      for(int i = 0; i < NUMBEROFLASERS; i++) {
        // Get laser information
        configData = ((String) config.get(i+1)).split(" ");

        // Parse coordinates of each laser
        bestGuess[i][0] = Double.valueOf(configData[1]).doubleValue();
        bestGuess[i][1] = Double.valueOf(configData[2]).doubleValue();
        bestGuess[i][2] = Double.valueOf(configData[3]).doubleValue();

        // Look and open for laser file
        if((new File(configData[7])).exists())
          inputFile = new BufferedReader(new FileReader(new File(configData[7])));
        else if((new File(path, configData[7])).exists())
          inputFile = new BufferedReader(new FileReader(new File(path, configData[7])));
        else {
          System.out.println("ERROR:  File not found");
          System.exit(2);
        }

        // Grab line laserLine
        for(int j = 0; j < laserLine; j++)
          if((line = inputFile.readLine()) == null) {
            System.out.println("ERROR:  Not enough lines in the laser file");
            System.exit(2);
          }

        // Read in & parse raw scan data
        for(int j = 0; j < SCANSPERLASER; j++) {
          rawData[i][j][0] = Integer.valueOf(line.split(", ")[j+1]).intValue();
          rawData[i][j][1] = (j / (SCANSPERLASER - 1.0) * Math.PI) - (Math.PI / 2);
        }

        // Close laser file
        inputFile.close();
      }
    }
    catch (Exception e) {System.out.println(e);};

    // Return smoothly
    return;
  }


  // Function to calculate error
  public static double getError(int whichLaser, double[] mutation) {
    // Variable List
    double distance = Double.MAX_VALUE;
    double tempDistance = 0.0;
    double error = 0.0;
    double[] pointOnMap = new double[2];

    // For each point of current laser
    for(int i = 0; i < SCANSPERLASER; i++) {
      // If point is not a hit (8191) ignore it
      if(rawData[whichLaser][i][0] == 8191)
        continue;

      // Reset variables
      distance = Double.MAX_VALUE;

      // Project point onto map
      pointOnMap = polar2Cart(rawData[whichLaser][i], mutation);

      // Find the closest point in final map
      for(int j = 0; j < pointsInMap; j++) {
        tempDistance = Math.sqrt(Math.pow(pointOnMap[0] - finalMap[j][0], 2) + Math.pow(pointOnMap[1] - finalMap[j][1], 2));
        if(tempDistance < distance)
          distance = tempDistance;
      }

      // If nearest point is closer than currentThresh, add distance^2 to totalError
      if(distance < currentThresh)
        error += Math.sqrt(distance);
    }

    // Return SQRT(distance)
    return(Math.sqrt(error));
  }


  // Function to convert polar to cartesian
  public static double[] polar2Cart(double[] point, double[] mutation) {
    // Variable List
    double[] coords = new double[2];

    coords[0] = point[0] * Math.cos(point[1] + mutation[2]) + mutation[0];
    coords[1] = point[0] * Math.sin(point[1] + mutation[2]) + mutation[1];
//System.out.println(point[0]);
//System.out.println(point[1]);
//System.out.println(coords[0]);
//System.out.println(coords[1]);
//System.exit(0);
    // Return x, y coordinates
    return(coords);
  }


  // Output new config file
  public static void outputData(String oldConfig, String newConfig) {
    try {
      // Variable List
      BufferedReader inputFile = new BufferedReader(new FileReader(new File(oldConfig)));
      BufferedWriter out = new BufferedWriter(new FileWriter(new File(newConfig)));
      int whichLine = 0;
      String[] data = new String[99];
      String line = new String();

      // Get a line
      while((line = inputFile.readLine()) != null) {
        // Reset variables
        for(int i = 0; i < data.length; i++)
          data[i] = new String();

        // If line is blank or a comment
        if(line.length() == 0 || (line.charAt(0) == '/' && line.charAt(1) == '/'))
          out.write(line + "\n");

        // Or if line is the first real line or after last real line, simply write out
        else if(whichLine == 0 || whichLine > NUMBEROFLASERS) {
          out.write(line + "\n");
          whichLine++;
        }

        // Otherwise, tokenize, replace 2nd, 3rd and 4th elements, and write out
        else {
          data = line.split(" ");
          data[1] = bestGuess[whichLine-1][0] + "";
          data[2] = bestGuess[whichLine-1][1] + "";
          data[3] = bestGuess[whichLine-1][2] + "";
          whichLine++;
          for(int i = 0; i < data.length; i++)
            out.write(data[i] + " ");
          out.write("\n");
        }
      }

      // Close files
      inputFile.close();
      out.close();
    }
    catch (Exception e) {System.out.println(e);};

    // Return smoothly
    return;
  }
}