//package laser.utilities;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;
import java.util.ArrayList;
import java.util.Vector;

//import laser.utilities.*;

/**
 * Class BackgroundSubtractor - The BackgroundSubtractor subtracts the
 * background from the laser scans.  It works by placing points in
 * buckets (size determined by user input), finding the bucket with
 * enough points to place it over a percentage threshold (value
 * selected through emperical results), and then replacing all values
 * in that bucket and beyond with "0".
 *
 * Other Info: Each laser has 361 scan points.
 *
 * Input:
 * "java BackgroundSubtractor <bin_size> <config_file> <output_dir>
 * The config_file should be a file with the number of lasers at the
 * top and the path to each laser file below that.  The laser files
 * should be time and space registered laser scan files.  The output_dir
 * should already exist.  Files can be in it but may be overwritten.
 *
 * Output:
 * The background subtracted files will be output to the directory
 * specified.  The file names will be "laserX_bgs.txt" where X is the
 * number of the laser in order of processing.  A config file will
 * also be created in the directory for input to the next step
 * (Laser2BTF).  The config file will be named "config_bgs.cfg".
 *
 * @author <A HREF="mailto:summer@cc.gatech.edu">Summer</A>
 *
 * TO DO:
 */
public class LaserBackgroundSubtract {

  private BufferedReader fileReader;
  private BufferedReader configFileReader;
  private BufferedWriter fileWriter;
  private static BufferedWriter configFileWriter;
  private int binSize;
  private int numberBins;
  private int[][] scanCounts;
  private int[] startBinToDelete = new int[361];
  private Vector scanData = new Vector();
  private int lineCount = 0;
  private double percentThreshold = .25;
  private static int numberOfLasers;
  private static Vector configData = new Vector();

  public LaserBackgroundSubtract(int inputBinSize) {
    binSize = inputBinSize;
    numberBins = (int) Math.ceil(80000/binSize);
    scanCounts = new int[362][numberBins];
  } // end constructor

  private void processConfigFile(String configFileIn, String configFileOut) {
    String currentLine;

    try {
      File configIn = new File(configFileIn);
      File configOut = new File(configFileOut);
      configFileReader = new BufferedReader(new FileReader(configIn));
      configFileWriter = new BufferedWriter(new FileWriter(configOut));

      while ((currentLine = configFileReader.readLine()) != null) {
        configData.add(currentLine);
      }

      configFileReader.close();
    } catch (IOException e) {
      System.out.println("Error processing config file");
      exitProgram();
    }

    numberOfLasers = Integer.parseInt((String) configData.get(0));
  } // end processConfigFile

  private void openReadersAndWriters(String inputFile, String outputFile) {
    try {
      File fileIn = new File(inputFile);
      File fileOut = new File(outputFile);

      fileReader = new BufferedReader(new FileReader(fileIn));
      fileWriter = new BufferedWriter(new FileWriter(fileOut));
    } catch (IOException e) {
      System.out.println("Error loading input files");
      exitProgram();
    }
  } // end openReadersAndWriters

  private void calculateBins() {
    String currentLine;
    int lineDataAsInt;
    int binPlacement;

    // make sure it is empty before reading into scanData
    scanData.removeAllElements();

    try {
      while ((currentLine = fileReader.readLine()) != null) {
        lineCount++;
        String lineData[] =  currentLine.split(",");
        scanData.add(lineData);
        // using int i = 1 on purpose
        for (int i = 1; i < lineData.length; i++) {
          lineDataAsInt = Integer.valueOf(lineData[i].trim()).intValue();
          binPlacement = (int) Math.floor(lineDataAsInt / binSize);
          this.scanCounts[i][binPlacement]++;
        }
      }
    } catch (IOException e) {
      System.out.println("Error reading scan points");
    }
  } // end calculateBins

  private void findFarthestPoints() {

    boolean foundBackgroundPoint = false;

    for (int i = 1; i < scanCounts.length; i++) {
      //System.out.println("Scan Position " + i);
      for (int j = 0; j < numberBins; j++) {
        if (foundBackgroundPoint == false) {
          if (scanCounts[i][j] != 0) {
            if (scanCounts[i][j] > (lineCount * percentThreshold)) {
              //System.out.println("    Bin Number " + j + "  Bin Count " + scanCounts[i][j]);
              foundBackgroundPoint = true;
              startBinToDelete[i-1] = j;
            } // over threshold
          } // not eq 0
        } // found = false
      } // for j
      foundBackgroundPoint = false; // reset for next bin
    }
  } // end findFarthestPoints

  private void writeOutPoints() {
    String[] lineOfData;

    try {
      for (int i = 0; i < scanData.size(); i++) {
        lineOfData = (String[]) scanData.get(i);
        for (int j = 0; j < lineOfData.length - 1; j++) {
          if (j == 0) {
            fileWriter.write(lineOfData[j]);
          }

          if (j > 0) {
            if (Integer.valueOf(lineOfData[j].trim()).intValue() < (startBinToDelete[j-1] * binSize)) {
              fileWriter.write(lineOfData[j]);
            } else {
              fileWriter.write(" 0");
            }
          }

          if (j < lineOfData.length - 2) {
            fileWriter.write(",");
          }
        }
        fileWriter.newLine();
      }
    } catch (IOException e) {
      System.out.println("Error writing tracks to output files");
      exitProgram();
    }
  } // end writeOutPoints

  private void writeOutConfigFile(String outputFile) {

    try {
      File configFileOut = new File(outputFile);
      configFileWriter = new BufferedWriter(new FileWriter(configFileOut));

      configFileWriter.write("test");

      configFileWriter.close();
    } catch (IOException e) {
      System.out.println("Error writing out config file");
      exitProgram();
    }
  } // end writeOutConfigFile

  private void closeReadersAndWriters() {
    try {
      fileReader.close();
      fileWriter.close();
    } catch (IOException e) {
      System.out.println("Error closing files");
      exitProgram();
    }
  } // end closeReadersAndWriters

  private void closeConfigOut() {
    String currentLine;

    try {
      configFileWriter.close();
    } catch (IOException e) {
      System.out.println("Error closing config file");
      exitProgram();
    }

    numberOfLasers = Integer.parseInt((String) configData.get(0));
  } // end processConfigFile

  public static void exitProgram() {
    System.out.println("Exiting program...");
    System.exit(1);
  } // end exitProgram

  public static void main(String[] args) {
    String inputFile;
    String outputFile;
    String configFileOut;
    int laserId;
String tempStr = new String();

// Create output dir
(new File(args[2])).mkdir();

        if (args.length == 3){
            LaserBackgroundSubtract lbs = new LaserBackgroundSubtract(Integer.valueOf(args[0]).intValue());
            configFileOut = args[2] + File.separator + "config_bgs.cfg";
            lbs.processConfigFile(args[1], configFileOut);
            try {
              configFileWriter.write(Integer.toString(numberOfLasers));
            } catch (IOException e) {
              System.out.println("Error writing config file");
              exitProgram();
            }

            // get next input and output and send on
            for (int i = 1; i < configData.size(); i++) {
              inputFile = (String) configData.get(i);
              laserId = i - 1;
              tempStr = (String) configData.get(i);
              outputFile = args[2] + File.separator + tempStr.substring(tempStr.lastIndexOf("\\"));

              lbs.openReadersAndWriters(inputFile, outputFile);
              lbs.calculateBins();
              lbs.findFarthestPoints();
              lbs.writeOutPoints();
              lbs.closeReadersAndWriters();

                try {
                  configFileWriter.newLine();
                  configFileWriter.write(outputFile);
                } catch (IOException e) {
                  System.out.println("Error writing config file");
                  exitProgram();
                }
            }

            lbs.closeConfigOut();
        } else {
          System.out.println("usage: java LaserBackgroundSubtract <bin_size> <config_file> <output_dir>");
        }
  } // end main
}
