// Usage:  java Laser2BTF.java <config_file> <output_dir>

// Takes in a laser config file (and the laser files referrenced therein) and
// Converts to the BTF format, outputing to the specified directory.

import java.io.*;
import java.util.*;


public class Laser2BTF {
  // Global Constant List
  public static final int SIGFIGS = 3;

  // Global Variable List
  public static int numberOfLasers = 0;
  public static int maxLines = 0;
  public static int scansPerLaser = 0;
  public static String configData[][] = new String[0][8];
  public static int linesPerLaser[] = new int[0];
  public static long timestamp[][] = new long[0][0];
  public static int laserData[][][] = new int[0][0][0];
  public static double cartData[][] = new double[0][2];


  public static void main(String args[]) throws Exception {
    // Variable List
    double laserAt[][] = new double[0][3];
    int start = 0;
    double xOffset = 0;
    double yOffset = 0;
    int count = 0;

    // Start output
    System.out.print("Phase 1 (read in data)");

    // Read in config and data files
    inputData(args[0]);

    // Newline on output
    System.out.print("\nPhase 2 (convert data)");

    // Set laser locations
    laserAt = new double[numberOfLasers][3];
    for(int i = 0; i < numberOfLasers; i++) {
      laserAt[i][0] = (Double.valueOf(configData[i][1])).doubleValue();
      laserAt[i][1] = (Double.valueOf(configData[i][2])).doubleValue();
      laserAt[i][2] = (Double.valueOf(configData[i][3])).doubleValue();
    }

    // Initialize cartData
    cartData = new double[scansPerLaser][2];


    // ***** Convert each laser point to cartessian & write out *****
    try {
      // Create output dir
      (new File(args[1])).mkdir();

      // Open data files
      BufferedWriter timestampOut = new BufferedWriter(new FileWriter(new File(args[1] + File.separator + "timestamp.btf")));
      BufferedWriter id = new BufferedWriter(new FileWriter(new File(args[1] + File.separator + "id.btf")));
      BufferedWriter type = new BufferedWriter(new FileWriter(new File(args[1] + File.separator + "type.btf")));
      BufferedWriter scanxscale = new BufferedWriter(new FileWriter(new File(args[1] + File.separator + "scanxscale.btf")));
      BufferedWriter scanyscale = new BufferedWriter(new FileWriter(new File(args[1] + File.separator + "scanyscale.btf")));

      // For each sweep
      for(int j = 0; j < linesPerLaser[0]; j++)
        // For each laser
        for(int i = 0; i < numberOfLasers; i++) {
          // Output every 1000 lines processed
          if((j + i * linesPerLaser[i] + 1) % 1000 == 0)
            System.out.print(".");

          // For each laser point
          for(int k = 0; k < scansPerLaser; k++)
            // Get the absolute x,y coordinates of that point (mutating by the location of the laser)
            cartData[k] = polar2Cart(laserData[i][j][k], (Math.PI / 2 ) - (k * Math.PI / 360.0), laserAt[i]);

          // *** Output that line in each file ***
          timestampOut.write(timestamp[i][j] + "\n");
          id.write(i + "\n");
          type.write("laser\n");

          start = 0;
          // First point should be non-8191 (ignore 8191)
          while(cartData[start][0] == 8191 && ++start < scansPerLaser);

          // If there are no good points, just write out 0.0
          if(start == scansPerLaser)
            scanxscale.write("0.0");
          else
            scanxscale.write(doubleToString(cartData[start][0]));

          // Write out the rest of the points on this line
          for(int k = start; k < scansPerLaser; k++)
            // Ignore any other 8191
            if(cartData[k][0] != 8191) {
              scanxscale.write(", " + doubleToString(cartData[k][0]));
              if(cartData[k][0] < xOffset)
                xOffset = cartData[k][0];
            }
          scanxscale.write("\n");

          start = 0;
          // First point should be non-8191 (ignore 8191)
          while(cartData[start][0] == 8191 && ++start < scansPerLaser);

          // If there are no good pionts, just write out 0.0
          if(start == scansPerLaser)
            scanyscale.write("0.0");
          else
            scanyscale.write(doubleToString(cartData[start][1]));

          // Write out the rest of the points on this line
          for(int k = start; k < scansPerLaser; k++)
            // Ignore any other 8191
            if(cartData[k][1] != 8191) {
              scanyscale.write(", " + doubleToString(cartData[k][1]));
              if(cartData[k][1] > yOffset)
                yOffset = cartData[k][1];
            }
          scanyscale.write("\n");

        }

      // Close data files
      timestampOut.close();
      id.close();
      type.close();
      scanxscale.close();
      scanyscale.close();
    }
    catch (Exception e) {System.out.println(e);};

    // Exit smoothly
    return;
  }


  public static void inputData(String configFilename) {
    try {
      // Variable List
      String temp = new String();
      String line = new String();
      BufferedReader inputFile = new BufferedReader(new FileReader(new File(configFilename)));
      ArrayList<String> config = new ArrayList<String>();
      ArrayList tempData[] = new ArrayList[0];
      String path = (new File(configFilename)).getParent();
      String tempStr[] = new String[0];

      // ***** Read in config info *****
      // Get a line
      while((line = inputFile.readLine()) != null) {
        // Go character by character to eliminate comments
        for(int i = 0; i < line.length(); i++) {
          if(line.charAt(i) == '/' && i < line.length() - 1 && line.charAt(i+1) == '/')
            break;
          else
            temp += line.charAt(i);
        }

        // If the line was blank or only comments, continue
        if(temp.length() == 0)
          continue;

        // Add line to list of lines
        config.add(temp);

        // Reset variables
        temp = new String();
      }

      // Close config file
      inputFile.close();

      // ***** Parse config info *****
      // Get number of lasers (first word)
      numberOfLasers = (Integer.valueOf(((String) config.get(0)).split(" ")[0])).intValue();

      // Set and initialize dependent variables
      linesPerLaser = new int[numberOfLasers];
      configData = new String[numberOfLasers][8];
      tempData = new ArrayList[numberOfLasers];
      for(int i = 0; i < tempData.length; i++)
        tempData[i] = new ArrayList();

      // For each laser
      for(int i = 0; i < numberOfLasers; i++)
        // Stick the word in the right slot of configData ([laser][word])
        configData[i] = ((String) config.get(i+1)).split(" ");

      // ***** Read in laser data *****
      // For each laser file
      for(int i = 0; i < numberOfLasers; i++) {
        // Open laser file
        inputFile = new BufferedReader(new FileReader(new File(path, configData[i][7])));

        // Read in & store laser data
        while((line = inputFile.readLine()) != null)
          tempData[i].add(line);

        // Close laser file
        inputFile.close();

        // Delete the first 5 scans from each laser
        for(int j = 0; j < 5; j++)
          tempData[i].remove(0);

        // Set linesPerLaser (and maxLines)
        linesPerLaser[i] = tempData[i].size();
        if(linesPerLaser[i] > maxLines)
          maxLines = linesPerLaser[i];
      }

    // ***** Parse laser data *****
    // Determine number of scans per laser
    scansPerLaser = ((String) tempData[0].get(0)).split(" ").length - 1;

    // Initialize timestamp & laserData
    timestamp = new long[numberOfLasers][maxLines];
    laserData = new int[numberOfLasers][maxLines][scansPerLaser];

    // ** Populate timestamp and laserData **
    // For each laser
    for(int i = 0; i < numberOfLasers; i++)
      // For each line
      for(int j = 0; j < tempData[i].size(); j++) {
        // Output every 1000 lines
        if((j + i * tempData[i].size() + 1) % 1000 == 0)
          System.out.print(".");
        tempStr = ((String) tempData[i].get(j)).split(", ");
        timestamp[i][j] = (Long.valueOf(tempStr[0])).longValue();
        for(int k = 0; k < tempStr.length - 2; k++)
          laserData[i][j][k] = (Integer.valueOf(tempStr[k+1])).intValue();
      }
    }
    catch (Exception e) {System.out.println(e);};

    // Exit smoothly
    return;
  }


  public static double[] polar2Cart(double r, double theta, double[] mutation) {
    // Variable List
    double[] coords = new double[2];

    // If we have an "8191" point or "0" point (or other "weird" points, or "-1")
    if(r > 8000 || r == -1 || r == 0) {
      coords[0] = coords[1] = 8191;
      return(coords);
    }

    // Convert polar to cartessian, taking laser location into account
    // And convert from cm to m
    coords[0] = (r * Math.cos(theta + mutation[2]) + mutation[0]) / 100.0;
    coords[1] = (r * Math.sin(theta + mutation[2]) + mutation[1]) / 100.0;

    // Exit smoothly
    return(coords);
  }


  public static String doubleToString(double number) {
    // Variable List
    String tempStr = new String();

    // Get integer portion
    tempStr = (new Integer((int) number)).toString() + ".";

    for(int i = 1; i <= SIGFIGS; i++)
      tempStr += getDigit(number, i);

    return(tempStr);
  }


  public static int getDigit(double number, int which) {
    // Variable List
    int tempInt = 0;
    int bigger = 0;

    // Make number positive
    if(number < 0)
      number *= -1;

    // Put which in the ones position
    for(int i = 0; i < which; i++)
      number *= 10;

    // Strip remaining decimals
    number = (int) number;

    // Get everything except position
    bigger = ((int) number / 10) * 10;

    // Return ones position
    return((int) number - bigger);
  }
}
