#include "config.h"
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_TIMES_H
#include <sys/times.h>
#endif
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_UIO_H
#include <sys/uio.h>
#endif
#include "assert.h"
#include "io.h"
#include "io_interface.h"
#include "io_internal.h"
#include "unix_defs.h"

#ifndef SELECT_DEFINED
extern int select ARGS((int width, fd_set *readfds, fd_set *writefds, 
			fd_set *exceptfds, struct timeval *timeout));
#endif

static int
unix_read_func(conn, buffer, length, errno_p, result_p)
void *conn;
void *buffer;
int length;
int *errno_p;
char **result_p;
{
    int left = length;
    int iget;
    int fd = (int) (long) conn;

    iget = read(fd, (char *) buffer, length);
    if (iget == 0) {
	*result_p = "End of file";
	*errno_p = 0;
	return 0;		/* end of file */
    } else if (iget == -1) {
	int lerrno = errno;
	*errno_p = lerrno;
	if ((lerrno != EWOULDBLOCK) &&
	    (lerrno != EAGAIN) &&
	    (lerrno != EINTR)) {
	    /* serious error */
	    return -1;
	} else {
	    *errno_p = 0;
	    iget = 0;
	}
    }
    left = length - iget;
    while (left > 0) {
	iget = read(fd, (char *) buffer + length - left, left);
	if (iget == 0) {
	    *result_p = "End of file";
	    *errno_p = 0;
	    return length - left;	/* end of file */
	} else if (iget == -1) {
	    int lerrno = errno;
	    *errno_p = errno;
	    if ((lerrno != EWOULDBLOCK) &&
		(lerrno != EAGAIN) &&
		(lerrno != EINTR)) {
		/* serious error */
		return (length - left);
	    } else {
		*errno_p = 0;
		iget = 0;
	    }
	}
	left -= iget;
    }
    return length;
}

static int
unix_readv_func(conn, iov, icount, errno_p, result_p)
void *conn;
struct iovec *iov;
int icount;
int *errno_p;
char **result_p;
{
    int orig_icount = icount;
    int fd = (int) (long)conn;
    int iget;

    while (icount > 0) {
	iget = readv(fd, (struct iovec *) iov, icount);
	if (iget == 0) {
	    *result_p = "End of file";
	    *errno_p = 0;
	    return 0;		/* end of file */
	} else if (iget == -1) {
	    *errno_p = errno;
	    if ((*errno_p != EWOULDBLOCK) &&
		(*errno_p != EAGAIN) &&
		(*errno_p != EINTR)) {
		/* serious error */
		return -1;
	    } else {
		*errno_p = 0;
		iget = 0;
	    }
	}
	while (iget > 0) {
	    if (iov[0].iov_len <= iget) {
		iget -= iov[0].iov_len;
		iov++;
		icount--;
	    } else {
		iov[0].iov_len -= iget;
		iov[0].iov_base = ((char*)iov[0].iov_base) + iget;
		iget = 0;
	    }
	}
    }
    return orig_icount;
}

static int
unix_write_func(conn, buffer, length, errno_p, result_p)
void *conn;
void *buffer;
int length;
int *errno_p;
char **result_p;
{
    int left = length;
    int iget = 0;
    int fd = (int) (long)conn;

    while (left > 0) {
	iget = write(fd, (char *) buffer + length - left, left);
	if (iget == -1) {
	    int lerrno = errno;
	    *errno_p = errno;
	    if ((lerrno != EWOULDBLOCK) &&
		(lerrno != EAGAIN) &&
		(lerrno != EINTR)) {
		/* serious error */
		return (length - left);
	    } else {
		*errno_p = 0;
		iget = 0;
	    }
	}
	left -= iget;
    }
    return length;
}

static int
unix_writev_func(conn, iov, iovcnt, errno_p, result_p)
void *conn;
struct iovec *iov;
int iovcnt;
int *errno_p;
char **result_p;
{
    int fd = (int) (long) conn;
    int left = 0;
    int iget = 0;
    int iovleft, i;

    iovleft = iovcnt;

    /* sum lengths */
    for (i=0; i<iovcnt; i++) left += iov[i].iov_len;

    while (left > 0) {
	iget = writev(fd, (struct iovec *) &iov[iovcnt - iovleft], iovleft);
	if (iget == -1) {
	    if ((errno != EWOULDBLOCK) && (errno != EAGAIN)) {
		/* serious error */
		*errno_p = errno;
		return (iovcnt - iovleft);
	    } else {
		iget = 0;
	    }
	}
	if (iget == left)
	    return iovcnt;

	left -= iget;
	while (iget > 0) {
	    iget -= iov[iovcnt - iovleft].iov_len;
	    iovleft--;
	}

	if (iget < 0) {
	    /*
	     * Only part of the last block was written.  Modify IO 
	     * vector to indicate the remaining block to be written.
	     */
	    /* restore iovleft and iget to cover remaining block*/
	    iovleft++;
	    iget += iov[iovcnt - iovleft].iov_len;

	    /* adjust count down and base up by number of bytes written */
	    iov[iovcnt-iovleft].iov_len -= iget;
	    iov[iovcnt-iovleft].iov_base = 
		(char*)(iov[iovcnt-iovleft].iov_base) + iget;
	}
    }
    return iovcnt;
}

static int
unix_close_func(conn)
void *conn;
{
     return close((int) (long) conn);
}

static void *
unix_file_open_func(path, flag_str, input, output)
const char *path;
const char *flag_str;
int *input;
int *output;
{
    int flags;
    int file_id;
    long tmp_flags = (long)flag_str;
    *input = *output = 0;

    tmp_flags &= ~(O_TRUNC);
    tmp_flags &= ~(O_CREAT);
    if ((O_RDONLY == tmp_flags) ||
	(O_WRONLY == tmp_flags)) {
	 /* must be old style call */
	 flags = (long)flag_str;
	 *input = (O_RDONLY == (long)flag_str);
	 *output = (O_WRONLY & (long)flag_str);
    } else {
	 if (strcmp(flag_str, "r") == 0) {
	      flags = O_RDONLY;
	      *input = TRUE;
	 } else if (strcmp(flag_str, "w") == 0) {
	      flags = O_WRONLY | O_CREAT | O_TRUNC;
	      *output = TRUE;
	 } else {
	      fprintf(stderr, "Open flags value not understood for file \"%s\"\n",
		      path);
	      return NULL;
	 }
    }
    file_id = open(path, flags, 0777);
    if (file_id == -1) {
	 return NULL;
    } else {
	 return (void*) file_id;
    }
}
    
static int
unix_poll_func(conn)
void *conn;
{
    int fd = (int) (long) conn;
    struct timeval time;
    fd_set read_fds;
    int ret_val;

    time.tv_sec = time.tv_usec = 0;
    FD_ZERO(&read_fds);
    FD_SET(fd, &read_fds);
    ret_val = select(FD_SETSIZE, &read_fds, NULL, NULL, &time);
    return (ret_val > 0);
}

IOinterface_func os_file_read_func = unix_read_func;
IOinterface_func os_file_write_func = unix_write_func;
IOinterface_funcv os_file_readv_func = unix_readv_func;
IOinterface_funcv os_file_writev_func = unix_writev_func;

IOinterface_func os_read_func = unix_read_func;
IOinterface_func os_write_func = unix_write_func;
IOinterface_funcv os_readv_func = unix_readv_func;
IOinterface_funcv os_writev_func = unix_writev_func;
#ifndef IOV_MAX
#define IOV_MAX 16
#endif
int os_max_iov = IOV_MAX;
IOinterface_close os_close_func = unix_close_func;
IOinterface_poll  os_poll_func = unix_poll_func;
IOinterface_open os_file_open_func = unix_file_open_func;
IOinterface_func os_server_read_func = unix_read_func;
IOinterface_func os_server_write_func = unix_write_func;
IOinterface_init os_sockets_init_func = NULL;
