#include "config.h"
#include <fcntl.h>
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_SYS_UIO_H
#include <sys/uio.h>
#endif
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>

#include "tcl.h"
#include "io.h"
#include "io_tcl.h"
#include "io_interface.h"
#include "io_internal.h"
#include "unix_defs.h"

extern void PbioDoEventBind ARGS((PbioFilePtr file_rec, Tcl_Interp * interp,
				  int argc, char **argv));
int
PbioFileCommand(clientData, interp, argc, argv)
ClientData clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    PbioFilePtr file_rec = (PbioFilePtr) clientData;
    IOFile file = file_rec->file;
    IOFormat format;
    char buf[1024];
    char *buffer = buf;
    int buf_size = sizeof(buf);

    if (argc <= 1) {
	interp->result = "Too few arguments";
	return TCL_ERROR;
    }
    if (strcmp(argv[1], "read") == 0) {
	int result;
	Tcl_DString *str_result;

	/* got a read command */
	if ((buf_size = next_IOrecord_length(file)) > sizeof(buf)) {
	    buffer = malloc(next_IOrecord_length(file));
	}
	if ((result = read_raw_IOfile(file, buffer, buf_size, &format)) != 1) {
	    char *msg = Tcl_PosixError(interp);
	    Tcl_AppendResult(interp, "couldn't open file: ", file, " : ",
			     msg, (char *) NULL);
	    if (buf != buffer) {
		free(buf);
	    }
	    return TCL_ERROR;
	}
	str_result = get_IOrecord_tcl(file, format, buffer);
	Tcl_DStringResult(interp, str_result);
	free(str_result);
	if (buf != buffer) {
	    free(buf);
	}
    } else if (strcmp(argv[1], "write") == 0) {
	IOFormat format;
	IOFieldList field_list;
	char *buffer;
	char **listArgv;
	int listArgc, i;
	char ***fieldArgvList;
	/* got a write command */
	if (argc != 4) {
	    interp->result = "Wrong number arguments: PBIO write";
	    return TCL_ERROR;
	}
	/* got a write command */
	format = get_IOformat_by_name(file, argv[2]);
	if (format == NULL) {
	    Tcl_AppendResult(interp, "PBIO write format not found: ", argv[2],
			     (char *) NULL);
	    return TCL_ERROR;
	}
	field_list = field_list_of_IOformat(format);
	buffer = (char *) malloc(struct_size_IOfield(file, field_list));

	if (Tcl_SplitList(interp, argv[3], &listArgc, &listArgv) != TCL_OK) {
	    return TCL_ERROR;
	}
	fieldArgvList = (char ***) malloc(sizeof(char **) * listArgc);
	for (i = 0; i < listArgc; i++) {
	    IOFieldPtr field_ptr;
	    char **fieldArgv;
	    int fieldArgc;
	    if (Tcl_SplitList(interp, listArgv[i], &fieldArgc, &fieldArgv) !=
		TCL_OK) {
		return TCL_ERROR;
	    }
	    /* may be used to store fieldArgv in list for later freeing */
	    fieldArgvList[i] = NULL;

	    if (fieldArgc < 2) {
		Tcl_AppendResult(interp, "Too few format field elements ",
				 listArgv[i], (char *) NULL);
		free(listArgv);
		return TCL_ERROR;
	    }
	    field_ptr = get_IOfieldPtrFromList(field_list, fieldArgv[0]);
	    if (field_ptr == NULL) {
		Tcl_AppendResult(interp, "PBIO write field not found (or is embedded type which are not supported) : ",
				 fieldArgv[0], (char *) NULL);
		free(fieldArgv);
		free(listArgv);
		return TCL_ERROR;
	    } else {
		IOdata_type dtype = field_ptr->data_type;
		int itmp;
		long ltmp;
		double dtmp;
		switch (dtype) {
		case integer_type:
		case unsigned_type:
		    if (Tcl_GetInt(interp, fieldArgv[1], &itmp) != TCL_OK) {
			Tcl_AppendResult(interp, "Incorrect PBIO write field : ",
					 fieldArgv[0], (char *) NULL);
			return TCL_ERROR;
		    }
		    put_IOint(field_ptr, itmp, buffer);
		    break;
		case enumeration_type:
		    if (Tcl_GetInt(interp, fieldArgv[1], &itmp) != TCL_OK) {
			Tcl_AppendResult(interp, "Incorrect PBIO write field : ",
					 fieldArgv[0], (char *) NULL);
			return TCL_ERROR;
		    }
		    put_IOenum(field_ptr, itmp, buffer);
		    break;
		case float_type:
		    if (Tcl_GetDouble(interp, fieldArgv[1], &dtmp) != TCL_OK) {
			Tcl_AppendResult(interp, "Incorrect PBIO write field : ",
					 fieldArgv[0], (char *) NULL);
			return TCL_ERROR;
		    }
		    put_IOint(field_ptr, dtmp, buffer);
		    break;
		case char_type:
		    put_IOchar(field_ptr, fieldArgv[1][0], buffer);
		    break;
		case string_type:
		    /* hack zone! */
		    field_ptr->data_type = integer_type;
		    assert(sizeof(char *) == sizeof(long));
		    ltmp = (long) fieldArgv[1];
		    put_IOlong(field_ptr, ltmp, buffer);
		    /* store the fieldArgv (where the string is) for
		     * freeing later, after the write_IOfile */
		    fieldArgvList[i] = fieldArgv;
		    break;
		case unknown_type:
		    Tcl_AppendResult(interp, "Tcl/PBIO can't handle field type",
				     fieldArgv[2], (char *) NULL);
		    free(listArgv);
		    free(fieldArgv);
		    return TCL_ERROR;
		}
	    }
	    if (fieldArgvList[i] == NULL) {
		/* we didn't store the fieldArgv, free it */
		free(fieldArgv);
	    }
	}
	free(listArgv);

	if (write_IOfile(file, format, buffer) != 1) {
	    char *msg = Tcl_PosixError(interp);
	    Tcl_AppendResult(interp, "couldn't write file: ", file, " : ",
			     msg, (char *) NULL);
	    return TCL_ERROR;
	}
	for (i = 0; i < listArgc; i++) {
	    if (fieldArgvList[i] != NULL)
		free(fieldArgvList[i]);
	}
	free(buffer);
    } else if (strcmp(argv[1], "read_format") == 0) {
	IOFormat format;
	/* got a read format command */
	if ((format = read_format_IOfile(file)) == NULL) {
	    char *msg = Tcl_PosixError(interp);
	    Tcl_AppendResult(interp, "PBIO Read failed", msg, (char *) NULL);
	    return TCL_ERROR;
	}
	interp->result = name_of_IOformat(format);
	/* don't set free_proc...  Can't free this directly.  Close the
	 * file */

    } else if ((strcmp(argv[1], "write_get_format_info") == 0) ||
	       (strcmp(argv[1], "read_get_format_info") == 0) ||
	       (strcmp(argv[1], "get_format_info") == 0)) {
	IOFormat format;
	if (argc != 3) {
	    interp->result = "Wrong number arguments: get_format_info";
	    return TCL_ERROR;
	}
	format = get_IOformat_by_name(file, argv[2]);
	if (format == NULL) {
	    interp->result = "Format not found : get_format_info";
	    return TCL_ERROR;
	} else {
	    Tcl_DString *format_info = (Tcl_DString *) NULL;
	    format_info = get_IOformat_tcl(file, format);
	    Tcl_DStringResult(interp, format_info);
	    free(format_info);
	}
    } else if (strcmp(argv[1], "write_format") == 0) {
	IOFormat format;
	IOFieldList field_list;
	char **listArgv;
	int listArgc, i;

	/* got a write format command */
	if (argc != 4) {
	    interp->result = "Wrong number arguments: write_format";
	    return TCL_ERROR;
	}
	if (Tcl_SplitList(interp, argv[3], &listArgc, &listArgv) != TCL_OK) {
	    return TCL_ERROR;
	}
	field_list = (IOFieldList) malloc((listArgc + 1) * sizeof(IOField));
	for (i = 0; i < listArgc; i++) {
	    char **fieldArgv;
	    int fieldArgc;
	    if (Tcl_SplitList(interp, listArgv[i], &fieldArgc, &fieldArgv) !=
		TCL_OK) {
		return TCL_ERROR;
	    }
	    if (fieldArgc < 2) {
		Tcl_AppendResult(interp, "Too few format field elements ",
				 listArgv[i], (char *) NULL);
		free(field_list);
		free(listArgv);
		return TCL_ERROR;
	    }
	    field_list[i].field_name = strdup(fieldArgv[0]);
	    field_list[i].field_type = strdup(fieldArgv[1]);
	    field_list[i].field_offset = 0;
	    if (fieldArgc >= 3) {
		if (Tcl_GetInt(interp, fieldArgv[2],
			       &field_list[i].field_size) != TCL_OK) {
		    free(field_list);
		    free(fieldArgv);
		    free(listArgv);
		    return TCL_ERROR;
		}
	    } else {
		IOdata_type dtype = str_to_data_type(fieldArgv[1]);
		switch (dtype) {
		case integer_type:
		    field_list[i].field_size = sizeof(long);
		    break;
		case unsigned_type:
		    field_list[i].field_size = sizeof(unsigned long);
		    break;
		case float_type:
		    field_list[i].field_size = sizeof(double);
		    break;
		case char_type:
		    field_list[i].field_size = sizeof(char);
		    break;
		case string_type:
		    field_list[i].field_size = sizeof(char *);
		    break;
		case enumeration_type:
		    field_list[i].field_size = sizeof(long);
		    break;
		case unknown_type:
		    Tcl_AppendResult(interp, "Field size must be specified for",
				     fieldArgv[2], (char *) NULL);
		    free(field_list);
		    free(listArgv);
		    free(fieldArgv);
		    return TCL_ERROR;
		}
	    }
	    free(fieldArgv);
	}
	free(listArgv);

	field_list[listArgc].field_name = NULL;
	field_list[listArgc].field_type = NULL;
	field_list[listArgc].field_size = 0;
	field_list[listArgc].field_offset = 0;

	/* give them offsets */
	force_align_field_list(field_list, sizeof(char*));

	if ((format = register_IOrecord_format(argv[2], field_list, file))
	    == NULL) {
	    char *msg = Tcl_PosixError(interp);
	    Tcl_AppendResult(interp, "PBIO Read failed", msg, (char *) NULL);
	    free(field_list);
	    free(listArgv);
	    return TCL_ERROR;
	}
	free(field_list);
    } else if (strcmp(argv[1], "read_comment") == 0) {
	char *result;
	if ((result = read_comment_IOfile(file)) == NULL) {
	    char *msg = Tcl_PosixError(interp);
	    Tcl_AppendResult(interp, "PBIO Read failed", msg, (char *) NULL);
	    return TCL_ERROR;
	}
	interp->result = result;
    } else if (strcmp(argv[1], "write_comment") == 0) {
	if (argc != 3) {
	    interp->result = "Wrong number of arguments: write_comment";
	    return TCL_ERROR;
	}
	if (write_comment_IOfile(file, argv[2]) != 1) {
	    char *msg = Tcl_PosixError(interp);
	    Tcl_AppendResult(interp, "PBIO Write Comment failed", msg,
			     (char *) NULL);
	    return TCL_ERROR;
	}
    } else if (strcmp(argv[1], "next_record_type") == 0) {
	IORecordType result;
	result = next_IOrecord_type(file);
	switch (result) {
	case IOerror:
	    interp->result = "error";
	    break;
	case IOend:
	    interp->result = "end";
	    break;
	case IOdata:
	    interp->result = "data";
	    break;
	case IOformat:
	    interp->result = "format";
	    break;
	case IOcomment:
	    interp->result = "comment";
	    break;
	default:
	    interp->result = "Unknown result from file ";
	    return TCL_ERROR;
	    /* NOTREACHED */
	    break;
	}
    } else if (strcmp(argv[1], "next_record_format") == 0) {
	IOFormat format = next_IOrecord_format(file);
	if (format == NULL) {
	    interp->result = "Next record format failed";
	    return TCL_ERROR;
	} else {
	    interp->result = name_of_IOformat(format);
	}
    } else if (strcmp(argv[1], "close") == 0) {
	close_IOfile(file);
    } else if (strcmp(argv[1], "bind") == 0) {
	if (argc != 4) {
	    interp->result = "Wrong # arguments";
	    return TCL_ERROR;
	}
	if (!Tcl_CommandComplete(argv[3])) {
	    interp->result = "Script must be a complete Tcl command";
	    return TCL_ERROR;
	}
	PbioDoEventBind(file_rec, interp, argc, argv);
    } else {
	sprintf(interp->result, "Unknown command %s", argv[1]);
	return TCL_ERROR;
    }
    return TCL_OK;
}

void 
PbioFree(clientData)
ClientData clientData;
{
    PbioFilePtr file_rec = (PbioFilePtr) clientData;
    free_IOfile(file_rec->file);
    free(file_rec);
}

int
tcl_pbio_open(clientData, interp, argc, argv)
ClientData clientData;
Tcl_Interp *interp;
int argc;
char *argv[];
{
    int fd;
    char junk;
    IOFile file;
    PbioFilePtr file_rec;
    if (argc != 4) {
	interp->result = "Wrong # args";
	return TCL_ERROR;
    }
    /* try to read the argument as a number.... */
    if (sscanf(argv[2], "%d%c", &fd, &junk) != 1) {
	/* Failed.  It's a filename, presumably */
	file = open_IOfile(argv[2], argv[3]);
    } else {
	if (Tcl_GetInt(interp, argv[2], &fd) != TCL_OK) {
	    return TCL_ERROR;
	}
	file = open_IOfd(fd, argv[3]);
    }
    if (file == NULL) {
	char *msg = Tcl_PosixError(interp);
	Tcl_AppendResult(interp, "PBIO Read failed", msg, (char *) NULL);
	return TCL_ERROR;
    }
    file_rec = TclPbioInitFileRec(file, interp);
    Tcl_CreateCommand(interp, argv[1], PbioFileCommand, (ClientData) file_rec,
		      PbioFree);
    return TCL_OK;
}


PbioFilePtr 
TclPbioInitFileRec(file, interp)
IOFile file;
Tcl_Interp *interp;
{
    PbioFilePtr file_rec;

    file_rec = (PbioFilePtr) malloc(sizeof(struct PbioFileRec));
    file_rec->file = file;
    file_rec->interp = interp;
    file_rec->bound = 0;
    file_rec->end_script = 0;
    file_rec->error_script = 0;
    file_rec->comment_script = 0;
    file_rec->format_script = 0;
    file_rec->data_script = 0;
    Tcl_InitHashTable(&file_rec->dataBindingTable, TCL_STRING_KEYS);
    return file_rec;
}

int
Pbio_Init(interp)
Tcl_Interp *interp;
{

    Tcl_CreateCommand(interp, "PbioFile", tcl_pbio_open, NULL, NULL);
    return TCL_OK;
}


static void
get_IOfield_tcl(buffer, iofile, ioformat, field, data, string_base)
Tcl_DString *buffer;
IOFile iofile;
IOFormat ioformat;
int field;
void *data;
void *string_base;
{
    IOVarInfoList iovar = &ioformat->var_list[field];
    IOFieldList iofield = &ioformat->field_list[field];
    int field_offset = iofield->field_offset;
    int field_size = iofield->field_size;
    char *field_type = iofield->field_type;
    char *field_name = iofield->field_name;
    char *left_paren = NULL;
    char str[1024];		/* temp holding string */

    Tcl_DStringAppend(buffer, "{\"", -1);
    Tcl_DStringAppend(buffer, field_name, -1);
    Tcl_DStringAppend(buffer, "\" ", -1);
    if ((left_paren = strchr(field_type, '[')) == NULL) {
	sdump_value(str, field_type, field_size, field_offset,
		    iofile, data, string_base, ioformat->byte_reversal);
	Tcl_DStringAppend(buffer, str, -1);
    } else if (strchr(left_paren + 1, '[') == NULL) {
	/* single dimen array */
	int dimension = 0;
	char sub_type[64];
	int sub_field_size, offset = iofield->field_offset;
	Tcl_DStringAppend(buffer, "{ ", -1);
	*left_paren = 0;
	strcpy(sub_type, field_type);
	if (sscanf(left_paren + 1, "%d]", &dimension) != 1) {
	    if (iovar->var_array != TRUE) {
		*left_paren = '[';
		fprintf(stderr, "Couldn't parse array size in \"%s\"\n",
			field_type);
		return;
	    } else {
		IOgetFieldStruct descr;
		descr.offset = iofield->field_offset;
		descr.size = ioformat->pointer_size;
		descr.data_type = integer_type;
		descr.byte_swap = ioformat->byte_reversal;

		dimension = get_IOlong(iovar->control_field, data);
		offset = get_IOlong(&descr, data);
	    }
	}
	*left_paren = '[';
	sub_field_size = iofield->field_size;
	for (; dimension > 0; dimension--) {
	    if (sdump_value(str, sub_type, sub_field_size, offset, iofile,
			    data, string_base, ioformat->byte_reversal) == 0) {
		IOFormat subformat;
		char *typ = base_data_type(field_type);
		if ((subformat = get_IOformat_by_name(iofile, typ)) != NULL) {
		    int index;
		    Tcl_DStringAppend(buffer, "{ ", -1);
		    for (index = 0; index < subformat->field_count; index++) {
			get_IOfield_tcl(buffer, iofile, subformat, 
					index, (void *)((char *)data + offset),
					string_base);
		    }
		    Tcl_DStringAppend(buffer, "} ", -1);
		} else {
		    printf("Unknown type");
		}
		free(typ);
	    }
	    offset += sub_field_size;
	    Tcl_DStringAppend(buffer, str, -1);
	    Tcl_DStringAppend(buffer, " ", -1);
	}
	Tcl_DStringAppend(buffer, "} ", -1);
    } else {
	/* double dimen array */
	int dimension1 = 0;
	int dimension2 = 0;
	char sub_type[64];
	int sub_field_size, offset = iofield->field_offset;
	Tcl_DStringAppend(buffer, "{ ", -1);
	*left_paren = 0;
	strcpy(sub_type, field_type);
	if (sscanf(left_paren + 1, "%d][%d]", &dimension1, &dimension2) != 2) {
	    *left_paren = '[';
	    fprintf(stderr, "Couldn't parse array size in \"%s\"\n",
		    field_type);
	    return;
	}
	*left_paren = '[';
	sub_field_size = iofield->field_size;
	for (; dimension2 > 0; dimension2--) {
	    int i = 0;
	    Tcl_DStringAppend(buffer, "{ ", -1);
	    for (; i < dimension1; i++) {
		if (sdump_value(str, sub_type, sub_field_size, offset, iofile,
			data, string_base, ioformat->byte_reversal) == 0) {
		    IOFormat subformat;
		    char *typ = base_data_type(field_type);
		    if ((subformat = get_IOformat_by_name(iofile, typ)) != NULL) {
			int index;
			printf("{ ");
			for (index = 0; index < ioformat->field_count; index++) {
			    get_IOfield_tcl(buffer, iofile, subformat, 
					    index,(void*)((char *)data+offset),
					    string_base);
			}
			printf("} ");
		    } else {
			printf("Unknown type");
		    }
		    free(typ);
		}
		offset += sub_field_size;
		Tcl_DStringAppend(buffer, str, -1);
		Tcl_DStringAppend(buffer, " ", -1);
	    }
	    Tcl_DStringAppend(buffer, "} ", -1);
	}
	Tcl_DStringAppend(buffer, "} ", -1);
    }
    Tcl_DStringAppend(buffer, "} ", -1);
    return;
}

extern Tcl_DString *
get_IOrecord_tcl(iofile, ioformat, data)
IOFile iofile;
IOFormat ioformat;
void *data;
{
    int index;
    Tcl_DString *dstr = (Tcl_DString *) malloc(sizeof(Tcl_DString));

    Tcl_DStringInit(dstr);

    for (index = 0; index < ioformat->field_count; index++) {
	get_IOfield_tcl(dstr, iofile, ioformat, index, data, data);
    }
    return dstr;
}

extern Tcl_DString *
get_IOformat_tcl(iofile, ioformat)
IOFile iofile;
IOFormat ioformat;
{
    int index;
    Tcl_DString *dstr = (Tcl_DString *) malloc(sizeof(Tcl_DString));

    Tcl_DStringInit(dstr);

    for (index = 0; index < ioformat->field_count; index++) {
	char buffer[1024];
	sprintf(buffer, "\"%s\" \"%s\" %d %d",
		ioformat->field_list[index].field_name,
		ioformat->field_list[index].field_type,
		ioformat->field_list[index].field_size,
		ioformat->field_list[index].field_offset);

	Tcl_DStringAppendElement(dstr, buffer);
    }
    return dstr;
}
