#include "config.h"
#include <fcntl.h>
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_SYS_UIO_H
#include <sys/uio.h>
#endif
#include <unistd.h>
#include <stdlib.h>

#include "tk.h"
#include "io.h"
#include "io_tcl.h"
#include "io_internal.h"
#include "io_interface.h"
#include "unix_defs.h"

static void PbioDoEventBind ARGS((PbioFilePtr file_rec, Tcl_Interp *interp,
				  int argc, char **argv));
static void ExpandPbioPercents ARGS((char *before, IOFile file, char *comment, 
				     char *buffer, IOFormat format,
				     Tcl_DString *dsPtr));
static void DeleteBinding ARGS((PbioFilePtr file_rec, IORecordType record_type,
				char *format_name));

void
HandlePbio(ClientData clientData, int mask)
{
    PbioFilePtr file_rec = (PbioFilePtr) clientData;
    IOFile file = file_rec->file;
    IOFormat format;
    char buf[1024];
    char *buffer = buf;
    int result;
    Tcl_DString scripts, savedResult;
    Tcl_Interp *interp = file_rec->interp;
    Tcl_HashEntry *entryPtr;
    char *comment = (char *)NULL;
    int code;
    char *p, *end;
    IORecordType record_type;
    char *format_name;

    Tcl_DStringInit(&scripts);
    Tcl_DStringInit(&savedResult);
    Tcl_DStringGetResult(interp, &savedResult);

    switch(record_type = next_IOrecord_type(file)) {
    case IOerror:
	if (file_rec->error_script != NULL) {
	    Tcl_DStringAppend(&scripts, file_rec->error_script, -1);
	    Tcl_DStringAppend(&scripts, "", 1);  /* append NULL */
	} else {
	    Tcl_AddErrorInfo(interp, 
			     "Read Error occurred in PBIO handler for which there was no specified script.");
	    Tk_BackgroundError(interp);
	}
	Tk_DeleteFileHandler(file_id_IOfile(file_rec->file));
	break;
    case IOend:
	if (file_rec->error_script != NULL) {
	    Tcl_DStringAppend(&scripts, file_rec->error_script, -1);
	    Tcl_DStringAppend(&scripts, "", 1);  /* append NULL */
	} else {
	    Tcl_AddErrorInfo(interp, 
			     "Read Error occurred in PBIO handler for which there was no specified script.");
	    Tk_BackgroundError(interp);
	}
	Tk_DeleteFileHandler(file_id_IOfile(file_rec->file));
	break;
    case IOdata:
	if (next_IOconversion_length(file) > sizeof(buf)) {
	    buffer = malloc(next_IOconversion_length(file));
	}
	if ((result = read_raw_IOfile(file, buffer, 1024, &format)) != 1) {
	    Tcl_DString error;
	    char *msg = Tcl_PosixError(file_rec->interp);
	    Tcl_DStringInit(&error);
	    Tcl_DStringAppend(&error, "Read Error occurred in PBIO handler for which there was no specified script.", -1);
	    Tcl_DStringAppend(&error, msg, -1);
	    Tcl_AddErrorInfo(interp, Tcl_DStringValue(&error));
	    Tcl_DStringFree(&error);
	    if (buf != buffer) {
		free(buf);
	    }
	    Tk_BackgroundError(interp);
	    break;
	}
	format_name = name_of_IOformat(format);
	entryPtr = Tcl_FindHashEntry(&file_rec->dataBindingTable, format_name);
	if (entryPtr == NULL) {
	    /* no specific script */
	    if (file_rec->data_script != NULL) {
		ExpandPbioPercents(file_rec->data_script, file, comment, 
				   buffer, format, &scripts);
		Tcl_DStringAppend(&scripts, "", 1); /* add a NULL separator */
	    }
	} else {
	    PbioBindingPtr binding =(PbioBindingPtr)Tcl_GetHashValue(entryPtr);
	    ExpandPbioPercents(binding->script, file, comment, 
			       buffer, format, &scripts);
	    Tcl_DStringAppend(&scripts, "", 1); /* add a NULL separator */
	}
	break;
    case IOformat:
	/* got a  format */
	if ((format = read_format_IOfile(file)) == NULL) {
	    char *msg = Tcl_PosixError(file_rec->interp);
	    Tcl_AppendResult(interp, "PBIO Read format failed", msg, (char *) NULL);
	    Tk_BackgroundError(interp);
	    break;
	}
	format_name = name_of_IOformat(format);
	if (file_rec->format_script != NULL) {
	    ExpandPbioPercents(file_rec->format_script, file, comment, buffer, 
			       format, &scripts);
	    Tcl_DStringAppend(&scripts, "", 1); /* add a NULL separator */
	}
	break;
    case IOcomment:
	if ((comment = read_comment_IOfile(file)) == NULL) {
	    char *msg = Tcl_PosixError(file_rec->interp);
	    Tcl_AppendResult(interp, "PBIO Read comment failed", msg, (char *) NULL);
	    Tk_BackgroundError(interp);
	    break;
	}
	if (file_rec->comment_script != NULL) {
	    ExpandPbioPercents(file_rec->comment_script, file, comment, 
			       buffer, format, &scripts);
	    Tcl_DStringAppend(&scripts, "", 1); /* add a NULL separator */
	}
	break;
    }
    if (buf != buffer) {
	free(buf);
    }

    /*  [[[ Modeled after code in tkBind.c in tk4.0   -- GSE ]]]
     * Now go back through and evaluate the script for each object,
     * in order, dealing with "break" and "continue" exceptions
     * appropriately.
     *
     * There are two tricks here:
     * 1. Bindings can be invoked from in the middle of Tcl commands,
     *    where interp->result is significant (for example, a widget
     *    might be deleted because of an error in creating it, so the
     *    result contains an error message that is eventually going to
     *    be returned by the creating command).  To preserve the result,
     *    we save it in a dynamic string.
     * 2. The binding's action can potentially delete the binding,
     *    so file_rec may not point to anything valid once the action
     *    completes.  Thus we have to save file_rec->interp in a
     *    local variable in order to restore the result.
     */

    interp = file_rec->interp;
    Tcl_DStringInit(&savedResult);
    Tcl_DStringGetResult(interp, &savedResult);
    p = Tcl_DStringValue(&scripts);
    end = p + Tcl_DStringLength(&scripts);
    while (p != end) {
	Tcl_AllowExceptions(interp);
	code = Tcl_GlobalEval(interp, p);
	if (code != TCL_OK) {
	    if (code == TCL_CONTINUE) {
		/*
		 * Do nothing:  just go on to the next script.
		 */
	    } else if (code == TCL_BREAK) {
		break;
	    } else {
		DeleteBinding(file_rec, record_type, format_name);
		Tcl_AddErrorInfo(interp, "\n    (command bound to PBIO event -- binding deleted)");
		Tk_BackgroundError(interp);
		break;
	    }
	}

	/*
	 * Skip over the current script and its terminating null character.
	 */

	while (*p != 0) {
	    p++;
	}
	p++;
    }
    Tcl_DStringResult(interp, &savedResult);
    Tcl_DStringFree(&scripts);
}

int
PbioFileCommand(ClientData clientData, Tcl_Interp *interp, int argc, 
	      char *argv[])
{
    PbioFilePtr file_rec = (PbioFilePtr) clientData;
    IOFile file = file_rec->file;
    IOFormat format;
    char buf[1024];
    char *buffer = buf;

    if (argc <= 1) {
	interp->result = "Too few arguments";
	return TCL_ERROR;
    }
    if (strcmp(argv[1], "read") == 0) {
	int result;
	Tcl_DString *str_result;

	/* got a read command */
	if (next_IOconversion_length(file) > sizeof(buf)) {
	    buffer = malloc(next_IOconversion_length(file));
	}
	if ((result = read_raw_IOfile(file, buffer, 1024, &format)) != 1) {
	    char *msg = Tcl_PosixError(interp);
	    Tcl_AppendResult(interp, "couldn't open file: ", file, " : ",
			     msg, (char *) NULL);
	    if (buf != buffer) {
		free(buf);
	    }
	    return TCL_ERROR;
	}
	str_result = get_IOrecord_tcl(file, format, buffer);
	Tcl_DStringResult(interp, str_result);
	free(str_result);
	if (buf != buffer) {
	    free(buf);
	}
    } else if (strcmp(argv[1], "read_format") == 0) {
	IOFormat format;
	/* got a read format command */
	if ((format = read_format_IOfile(file)) == NULL) {
	    char *msg = Tcl_PosixError(interp);
	    Tcl_AppendResult(interp, "PBIO Read failed", msg, (char *) NULL);
	    return TCL_ERROR;
	}
	interp->result = name_of_IOformat(format);
	/* don't set free_proc...  Can't free this directly.  Close the file */

    } else if (strcmp(argv[1], "read_comment") == 0) {
	char *result;
	if ((result = read_comment_IOfile(file)) == NULL) {
	    char *msg = Tcl_PosixError(interp);
	    Tcl_AppendResult(interp, "PBIO Read failed", msg, (char *) NULL);
	    return TCL_ERROR;
	}
	interp->result = result;
    } else if (strcmp(argv[1], "next_record_type") == 0) {
	IORecordType result;
	result = next_IOrecord_type(file);
	switch (result) {
	case IOerror: 
	    interp->result = "error";
	    break;
	case IOend:
	    interp->result = "end";
	    break;
	case IOdata:
	    interp->result = "data";
	    break;
	case IOformat:
	    interp->result = "format";
	    break;
	case IOcomment:
	    interp->result = "comment";
	    break;
	default:
	    interp->result = "Unknown result from file ";
	    return TCL_ERROR;
	    break;
	}
    } else if (strcmp(argv[1], "close") == 0) {
	close_IOfile(file);
    } else if (strcmp(argv[1], "bind") == 0) {
	if (argc != 4) {
	    interp->result = "Wrong # arguments";
	    return TCL_ERROR;
	}
	if (!Tcl_CommandComplete(argv[3])) {
	    interp->result = "Script must be a complete Tcl command";
	    return TCL_ERROR;
	}
	PbioDoEventBind(file_rec, interp, argc, argv);
    } else {
	sprintf(interp->result, "Unknown command %s", argv[1]);
	return TCL_ERROR;
    }
    return TCL_OK;
}

static void PbioDoEventBind(file_rec, interp, argc, argv)
PbioFilePtr file_rec;
Tcl_Interp *interp;
int argc;
char **argv;
{
    /* <obj> bind <event> <script> */
    if (strcmp(argv[2], "<end>") == 0) {
	if (file_rec->end_script != NULL) free(file_rec->end_script);
	file_rec->end_script = strdup(argv[3]);
    } else if (strcmp(argv[2], "<error>") == 0) {
	if (file_rec->error_script != NULL) free(file_rec->error_script);
	file_rec->error_script = strdup(argv[3]);
    } else if (strcmp(argv[2], "<comment>") == 0) {
	if (file_rec->comment_script != NULL) free(file_rec->comment_script);
	file_rec->comment_script = strdup(argv[3]);
    } else if (strcmp(argv[2], "<format>") == 0) {
	if (file_rec->format_script != NULL) free(file_rec->format_script);
	file_rec->format_script = strdup(argv[3]);
    } else if (strcmp(argv[2], "<data>") == 0) {
	/* General data script */
	if (file_rec->data_script != NULL) free(file_rec->data_script);
	file_rec->data_script = strdup(argv[3]);
    } else {
	/* assume that it's a format name... */
	int new;
	Tcl_HashEntry *entryPtr;
	PbioBindingPtr binding;
	entryPtr = Tcl_CreateHashEntry(&file_rec->dataBindingTable, argv[2],
				       &new);
	if (new) {
	    /* got a new binding... */
	    binding = (PbioBindingPtr) malloc(sizeof(struct PbioBindingRec));
	    binding->script = strdup(argv[3]);
	    Tcl_SetHashValue(entryPtr, binding);
	} else {
	    /* binding already existed.  Just change script */
	    free(binding->script);
	    binding->script = strdup(argv[3]);
	}
    }
    if (!file_rec->bound) {
	Tk_CreateFileHandler(file_id_IOfile(file_rec->file),
			     TK_READABLE, HandlePbio, 
			     (ClientData) file_rec);
    }
}

static void 
DeleteBinding(file_rec, record_type, format_name)
PbioFilePtr file_rec;
IORecordType record_type;
char *format_name;
{
    switch(record_type) {
    case IOend:
	if (file_rec->end_script != NULL) free(file_rec->end_script);
	file_rec->end_script = NULL;
	break;
    case IOerror:
	if (file_rec->error_script != NULL) free(file_rec->error_script);
	file_rec->error_script = NULL;
	break;
    case IOcomment:
	if (file_rec->comment_script != NULL) free(file_rec->comment_script);
	file_rec->comment_script = NULL;
	break;
    case IOformat:
	if (file_rec->format_script != NULL) free(file_rec->format_script);
	file_rec->format_script = NULL;
	break;
    case IOdata:
	{
	    /* assume that it's a format name... */
	    Tcl_HashEntry *entryPtr;
	    entryPtr = Tcl_FindHashEntry(&file_rec->dataBindingTable,
					 format_name);
	    if (entryPtr == NULL) {
		/* must have been general handler */
		if (file_rec->data_script != NULL) free(file_rec->data_script);
		file_rec->data_script = NULL;
	    } else {
		PbioBindingPtr binding = Tcl_GetHashValue(entryPtr);
		free(binding->script);
		free(binding);
		Tcl_DeleteHashEntry(entryPtr);
	    }
	}
    }
}
void PbioFree(ClientData clientData) 
{
    PbioFilePtr file_rec = (PbioFilePtr) clientData;
    free_IOfile(file_rec->file);
    free(file_rec);
}

int
tcl_pbio_open(ClientData clientData, Tcl_Interp *interp, int argc, 
	      char *argv[])
{
    int fd, flags;
    char junk;
    IOFile file;
    PbioFilePtr file_rec;
    if (argc != 4) {
	interp->result = "Wrong # args";
	return TCL_ERROR;
    }
    if (strcmp(argv[3], "r") == 0) {
	flags = O_RDONLY;
    } else if (strcmp(argv[3], "w") == 0) {
	flags = O_WRONLY;
    } else {
	interp->result = "misunderstood flags argument";
	return TCL_ERROR;
    }
    /* try to read the argument as a number.... */
    if (sscanf(argv[2], "%d%c", &fd, &junk) != 1) {
	/* Failed.  It's a filename, presumably */
	file = open_IOfile(argv[2], flags);
    } else {
	if (Tcl_GetInt(interp, argv[2], &fd) != TCL_OK) {
	    return TCL_ERROR;
	}
	file = open_IOfd(fd, flags);
    }
    if (file == NULL) {
	char *msg = Tcl_PosixError(interp);
	Tcl_AppendResult(interp, "PBIO Read failed", msg, (char *) NULL);
	return TCL_ERROR;
    }
    file_rec = TclPbioInitFileRec(file, interp);
    Tcl_CreateCommand(interp, argv[1], PbioFileCommand, (ClientData)file_rec, 
		      PbioFree);
    return TCL_OK;
}


PbioFilePtr TclPbioInitFileRec(file, interp)
IOFile file;
Tcl_Interp *interp;
{
    PbioFilePtr file_rec;

    file_rec = (PbioFilePtr) malloc(sizeof(struct PbioFileRec));
    file_rec->file = file;
    file_rec->interp = interp;
    file_rec->bound = 0;
    file_rec->end_script = 0;
    file_rec->error_script = 0;
    file_rec->comment_script = 0;
    file_rec->format_script = 0;
    file_rec->data_script = 0;
    Tcl_InitHashTable(&file_rec->dataBindingTable, TCL_STRING_KEYS);
    return file_rec;
}

int
Pbio_Init(interp)
Tcl_Interp *interp;
{
    
    Tcl_CreateCommand(interp, "PbioFile", tcl_pbio_open, NULL, NULL);
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * ExpandPbioPercents --
 *
 *	Given a command and an event, produce a new command
 *	by replacing % constructs in the original command
 *	with information from the PBIO event.
 *
 * Results:
 *	The new expanded command is appended to the dynamic string
 *	given by dsPtr.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
ExpandPbioPercents(before, file, comment, buffer, format, dPtr)
    char *before;	/* Command containing percent
				 * expressions to be replaced. */
    IOFile file;		/* PBIO file data was read from */
    char *comment;		/* comment read from file if any */
    char *buffer;		/* buffer containing raw read data */
    IOFormat format;		/* format read ( or of data ); */
    Tcl_DString *dPtr;		/* Dynamic string in which to append
				 * new command. */
{
    int spaceNeeded, cvtFlags;	/* Used to substitute string as proper Tcl
				 * list element. */
    int number, length;
    char *format_name = (char *)NULL;
    Tcl_DString *record_data = (Tcl_DString*) NULL;
    Tcl_DString *format_info = (Tcl_DString*) NULL;

#define NUM_SIZE 40
    char *string;
    char numStorage[NUM_SIZE+1];

    if (format != NULL) format_name = name_of_IOformat(format);

    while (1) {
	/*
	 * Find everything up to the next % character and append it
	 * to the result string.
	 */

	for (string = before; (*string != 0) && (*string != '%'); string++) {
	    /* Empty loop body. */
	}
	if (string != before) {
	    Tcl_DStringAppend(dPtr, before, string-before);
	    before = string;
	}
	if (*before == 0) {
	    break;
	}

	/*
	 * There's a percent sequence here.  Process it.
	 */

	number = 0;
	string = "??";
	switch (before[1]) {
	    case 'C':
		string = comment;
		goto doString;
	    case 'N':
		string = format_name;
		goto doString;
	    case 'D':
		if ((record_data == NULL) && (buffer != NULL)) {
		    record_data = get_IOrecord_tcl(file, format, buffer);
		}
		string = Tcl_DStringValue(record_data);
		goto doString;
	    case 'F':
		if (format != NULL) {
		    format_info = get_IOformat_tcl(file, format);
		}
		string = Tcl_DStringValue(format_info);
		goto doString;
	    default:
		numStorage[0] = before[1];
		numStorage[1] = '\0';
		string = numStorage;
		goto doString;
	}

	sprintf(numStorage, "%d", number);
	string = numStorage;

	doString:
	spaceNeeded = Tcl_ScanElement(string, &cvtFlags);
	length = Tcl_DStringLength(dPtr);
	Tcl_DStringSetLength(dPtr, length + spaceNeeded);
	spaceNeeded = Tcl_ConvertElement(string,
		Tcl_DStringValue(dPtr) + length,
		cvtFlags | TCL_DONT_USE_BRACES);
	Tcl_DStringSetLength(dPtr, length + spaceNeeded);
	before += 2;
    }
    if (format_info != NULL) {
	Tcl_DStringFree(format_info);
	free(format_info);
    }
    if (record_data != NULL) {
	Tcl_DStringFree(record_data);
	free(record_data);
    }
}

static void
get_IOfield_tcl(buffer, iofile, iofield, data)
Tcl_DString *buffer;
IOFile iofile;
IOFieldList iofield;
void *data;
{
    int   field_offset = iofield->field_offset;
    int   field_size = iofield->field_size;
    char *field_type = iofield->field_type;
    char *field_name = iofield->field_name;
    int  byte_swap = iofile->byte_reversal;
    char *left_paren = NULL;
    char str[1024];  /* temp holding string */

    Tcl_DStringAppend(buffer, "{\"", -1);
    Tcl_DStringAppend(buffer, field_name, -1);
    Tcl_DStringAppend(buffer, "\" ", -1);
    if ((left_paren = strchr(field_type, '[')) == NULL) {
	sdump_value(str, field_type, field_size, field_offset, 
		    byte_swap, data);
	Tcl_DStringAppend(buffer, str, -1);
    } else if (strchr(left_paren+1, '[') == NULL ) {
	/* single dimen array */
	int dimension = 0;
	char sub_type[64];
	int sub_field_size, offset = iofield->field_offset;
	Tcl_DStringAppend(buffer, "{ ", -1);
	*left_paren = 0;
	strcpy(sub_type, field_type);
	if (sscanf(left_paren+1, "%d]", &dimension) != 1) {
	    *left_paren = '[';
	    printf("unknown array type %s", field_type);
	    return;
	}
	*left_paren = '[';
	if (iofield->field_size % dimension != 0) {
	    printf("array size not evenly divisible by dimension");
	    return;
	}
	sub_field_size = iofield->field_size / dimension;
	for(; dimension>0; dimension--) {
	    sdump_value(str, sub_type, sub_field_size, offset, byte_swap, data);
	    offset += sub_field_size;
	    Tcl_DStringAppend(buffer, str, -1);
	    Tcl_DStringAppend(buffer, " ", -1);
	}
	Tcl_DStringAppend(buffer, "} ", -1);
    } else {
	/* double dimen array */
	int dimension1 = 0;
	int dimension2 = 0;
	char sub_type[64];
	int sub_field_size, offset = iofield->field_offset;
	Tcl_DStringAppend(buffer, "{ ", -1);
	*left_paren = 0;
	strcpy(sub_type, field_type);
	if (sscanf(left_paren+1, "%d][%d]", &dimension1, &dimension2) != 2) {
	    *left_paren = '[';
	    printf("unknown array type %s", field_type);
	    return;
	}
	*left_paren = '[';
	if ((iofield->field_size % (dimension1*dimension2)) != 0) {
	    printf("array size not evenly divisible by dimensions");
	    return;
	}
	sub_field_size = iofield->field_size / (dimension1 * dimension2);
	for(; dimension2>0; dimension2--) {
	    int i = 0;
	    Tcl_DStringAppend(buffer, "{ ", -1);
	    for(; i < dimension1; i++) {
		sdump_value(str, sub_type, sub_field_size, offset, byte_swap, data);
		offset += sub_field_size;
		Tcl_DStringAppend(buffer, str, -1);
		Tcl_DStringAppend(buffer, " ", -1);
	    }
	    Tcl_DStringAppend(buffer, "} ", -1);
	}
	Tcl_DStringAppend(buffer, "} ", -1);
    }
    Tcl_DStringAppend(buffer, "} ", -1);
    return;
}

extern Tcl_DString*
get_IOrecord_tcl(iofile, ioformat, data)
IOFile iofile;
IOFormat ioformat;
void *data;
{
    int index;
    Tcl_DString *dstr = (Tcl_DString *) malloc(sizeof(Tcl_DString));

    Tcl_DStringInit(dstr);

    for (index = 0; index < ioformat->field_count; index++) {
	get_IOfield_tcl(dstr, iofile, &(ioformat->field_list[index]), data);
    }
    return dstr;
}

extern Tcl_DString*
get_IOformat_tcl(iofile, ioformat)
IOFile iofile;
IOFormat ioformat;
{
    int index;
    Tcl_DString *dstr = (Tcl_DString *) malloc(sizeof(Tcl_DString));

    Tcl_DStringInit(dstr);

    for (index = 0; index < ioformat->field_count; index++) {
	char buffer[1024];
	sprintf(buffer, "\"%s\" \"%s\" %d %d", 
	       ioformat->field_list[index].field_name,
	       ioformat->field_list[index].field_type,
	       ioformat->field_list[index].field_size,
	       ioformat->field_list[index].field_offset);
	
	Tcl_DStringAppendElement(dstr, buffer);
    }
    return dstr;
}
