#include "config.h"
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <sys/types.h>
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_WINDOWS_H
#include <winsock.h>
#endif
#include <ctype.h>
#ifdef HAVE_SYS_UIO_H
#include <sys/uio.h>
#endif
#include <stdlib.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#include "assert.h"
#include "io.h"
#include "io_interface.h"
#include "io_internal.h"
#include "unix_defs.h"

int IOdumpVerbose = 0;

static int write_IOreformat_cue ARGS((IOFile iofile, int format_id));
static int old_server_register_format ARGS((IOFile iofile, IOFormat ioformat));
static int server_register_format ARGS((IOFile iofile, IOFormat ioformat));
static int self_server_register_format ARGS((IOFile iofile,
					     IOFormat ioformat));

static void byte_swap ARGS((char *data, int size));
static IOFormat server_get_format ARGS((IOContext iocontext, void *buffer));
static IOFormat old_server_get_format ARGS((IOContext iocontext,
					    void *buffer));
static IOFormat self_server_get_format ARGS((IOContext iocontext,
					     void *buffer,
					     void *app_context));
static void unstringify_field_type ARGS((const char *type, char *buffer, 
					 int size));
static char *stringify_field_type ARGS((const char *type, 
					IOFormat base_format,
					char *buffer, int size));
static int get_host_port_format_ID ARGS((void *format_ID));
static int get_host_IP_format_ID ARGS((void *format_ID));
static int is_server_context ARGS((IOFile iofile));

/* 
 * this reference to establish_server_connection is a function pointer 
 * rather than an extern so that the server_acts.o file is not loaded 
 * unless create_IOcontext() (which is also in that file) appears in 
 * the user program.  This avoids requiring the user to always link 
 * with -lnsl, -lsocket, etc when he might not otherwise need to.
 */
int (*establish_server_connection_ptr) ARGS((IOFile iofile, int do_fallback));
int pbio_fixed_format_ids = -1;

void
IOfree_conversion(conv)
IOConversionPtr conv;
{
    int i;
    for (i = 0; i < conv->conv_count; i++) {
	if (conv->conversions[i].subconversion) {
	    IOfree_conversion(conv->conversions[i].subconversion);
	}
	if (conv->conversions[i].control_field) {
	    io_free(conv->conversions[i].control_field);
	}
    }
    if (conv->native_field_list) {
	for (i = 0; conv->native_field_list[i].field_name != NULL; i++) {
	    io_free((char*)conv->native_field_list[i].field_name);
	    io_free((char*)conv->native_field_list[i].field_type);
	}
	io_free(conv->native_field_list);
    }
    io_free(conv);
}

static void
free_IOformat_body(body)
IOFormatBodyPtr body;
{
    int i;
    body->ref_count--;
    if (body->ref_count != 0)
	return;
    io_free(body->format_name);
    for (i = 0; i < body->field_count; i++) {
	io_free((char*)body->field_list[i].field_name);
	io_free((char*)body->field_list[i].field_type);
	if (body->var_list != NULL) {
	    if (body->var_list[i].control_field != NULL) {
		free(body->var_list[i].control_field);
	    }
	}
    }
    io_free(body->field_list);
    io_free(body->var_list);
    if (body->server_format_rep != NULL)
	free(body->server_format_rep);
    if (body->server_ID.value != NULL)
	free(body->server_ID.value);
    if (body->opt_info != NULL) {
	IOOptInfo *o = body->opt_info;
/*	while (o->info_type != -1) {
	    free(o->info_block);
	    o++;
	    }*/
	free(o);
    }
    if ((body->xml_out != NULL) && (body->xml_out != (void*)-1)) {
	free_XML_output_info(body->xml_out);
    }
    io_free(body);
}

extern void
free_IOformat(format)
IOFormat format;
{
    free_IOformat_body(format->body);
    if (format->conversion)
	IOfree_conversion(format->conversion);
    if (format->field_subformats)
	io_free(format->field_subformats);
    io_free(format);
}

IOFormat
get_IOformat_by_name_IOcontext(iocontext, name)
IOContext iocontext;
const char *name;
{
    IOFile iofile = (IOFile) iocontext;
    if ((iofile->is_context) && (strrchr(name, ':') != NULL)) {
	char buffer[32];	/* server_ID size or larger */
	unstringify_field_type(name, buffer, sizeof(buffer));
	return get_format_IOcontext((IOContext) iofile, buffer);
    } else {
	return get_IOformat_by_name(iofile, name);
    }
}

IOFormat
get_IOformat_by_name(iofile, name)
IOFile iofile;
const char *name;
{
    int i;
    if (iofile->status == OpenBeginRead) {
	/* if we're at the very begining of the file, read all formats to */
	/* preserve old behavior */
	(void) next_IOrecord_format(iofile);
	iofile->status = OpenForRead;
    }
    if ((iofile->is_context) && (strrchr(name, ':') != NULL)) {
	return get_IOformat_by_name_IOcontext((IOContext)iofile, name);
    }
    for (i = iofile->reg_format_count - 1; i >= 0; i--) {
	if ((iofile->format_list[i] != NULL) &&		/* entry null if
							 * reregistering */
	 (strcmp(name, name_of_IOformat(iofile->format_list[i])) == 0)) {
	    if (iofile->is_context && !iofile->format_list[i]->native_format) {
		/* in contexts, never match non-native formats by name */
		continue;
	    }
	    return iofile->format_list[i];
	}
    }
    return NULL;
}

extern void
add_opt_info_IOformat(format, typ, len, block)
IOFormat format;
int typ;
int len;
void *block;
{
    int count = 0;
    if (format->body->opt_info == NULL) {
	format->body->opt_info = malloc(2*sizeof(IOOptInfo));
    } else {
	while(format->body->opt_info[count].info_type != -1) count++;
	format->body->opt_info = realloc(format->body->opt_info,
					 (count + 2) *sizeof(IOOptInfo));
    }
    format->body->opt_info[count].info_type = typ;
    format->body->opt_info[count].info_len = len;
    format->body->opt_info[count].info_block = (char*)block;
    format->body->opt_info[count + 1].info_type = -1;
}

static int format_server_verbose = -1;

static IOFormat
copy_ioformat_to_iofile(format, iofile)
IOFormat format;
IOFile iofile;
{
    IOFormat new_format = io_malloc(sizeof(IOFormatStruct));
    int i;

    for (i = 0; i < iofile->reg_format_count; i++) {
	if (iofile->format_list[i]->body == format->body) {
	    if (format_server_verbose == 1) {
		printf("Copy, format %lx exists in context %lx as format %lx\n",
		       (long)format, (long)iofile, (long)iofile->format_list[i]);
	    }
	    return iofile->format_list[i];
	}
    }
    if (format_server_verbose == 1) {
	printf("Copy, entering format %lx into context %lx as new format %lx\n",
	       (long) format, (long) iofile, (long) new_format);
    }
    new_format->iofile = iofile;
    new_format->native_format = format->native_format;
    new_format->conversion = NULL;
    new_format->body = format->body;
    new_format->warned_about_null_conversion = 0;
    format->body->ref_count++;
    new_format->field_subformats =
	io_malloc(sizeof(IOFormat) * format->body->field_count);

    for (i = 0; i < new_format->body->field_count; i++) {
	if (format->field_subformats[i] != NULL) {
	    new_format->field_subformats[i] =
		copy_ioformat_to_iofile(format->field_subformats[i], iofile);
	} else {
	    new_format->field_subformats[i] = NULL;
	}
    }
    if (iofile->reg_format_count == iofile->format_list_size) {
	expand_IOFile(iofile);
    }
    new_format->format_id = iofile->reg_format_count++;
    iofile->format_list[new_format->format_id] = new_format;

    return new_format;
}

static void
replicate_in_masters(format, iofile)
IOFormat format;
IOFile iofile;
{
    if (iofile->master_context == NULL)
	return;
    copy_ioformat_to_iofile(format, (IOFile)iofile->master_context);
    replicate_in_masters(format, (IOFile)iofile->master_context);
}

unsigned char ID_length[] = {8, 10};

IOFormat
get_format_IOcontext(iocontext, buffer)
IOContext iocontext;
void *buffer;
{
    return get_format_app_IOcontext(iocontext, buffer, NULL);
}

IOFormat
get_local_format_IOcontext(iocontext, buffer)
IOContext iocontext;
void *buffer;
{
    IOFile iofile = (IOFile) iocontext;
    int i;
    if (format_server_verbose == -1) {
	if (getenv("FORMAT_SERVER_VERBOSE") == NULL) {
	    format_server_verbose = 0;
	} else {
	    format_server_verbose = 1;
	}
    }
    if (format_server_verbose == 1) {
	printf("Get Format searching in context %lx for format ", 
	       (long)iocontext);
	print_server_ID(buffer);
	printf("\n");
    }
    for (i = iofile->reg_format_count - 1; i >= 0; i--) {
	if (memcmp(buffer, iofile->format_list[i]->body->server_ID.value,
		   iofile->format_list[i]->body->server_ID.length) == 0) {
	    return iofile->format_list[i];
	}
    }
    if (iofile->master_context != NULL) {
	return get_local_format_IOcontext(iofile->master_context, buffer);
    }
    return NULL;
}

IOFormat
get_format_app_IOcontext(iocontext, buffer, app_context)
IOContext iocontext;
void *buffer;
void *app_context;
{
    IOFile iofile = (IOFile) iocontext;
    IOFormat new_format;
    int i;

    
    if (format_server_verbose == -1) {
	if (getenv("FORMAT_SERVER_VERBOSE") == NULL) {
	    format_server_verbose = 0;
	} else {
	    format_server_verbose = 1;
	}
    }
    for (i = iofile->reg_format_count - 1; i >= 0; i--) {
	if (memcmp(buffer, iofile->format_list[i]->body->server_ID.value,
		   iofile->format_list[i]->body->server_ID.length) == 0) {
	    return iofile->format_list[i];
	}
    }
    if (iofile->master_context == NULL) {
	if (is_server_context(iofile)) {
	    new_format = self_server_get_format(iocontext, buffer, app_context);
	} else if (pbio_fixed_format_ids == 1) {
	    new_format = old_server_get_format(iocontext, buffer);
	} else {
	    new_format = server_get_format(iocontext, buffer);
	}
	if (format_server_verbose == 1) {
	    printf("Read format from format server  %lx\n",
		   (long)new_format);
	    if (new_format != NULL) {
		dump_IOFormat(new_format);
	    } else {
		printf("Format unknown\n");
	    }
	}
    } else {
	/* we're a subcontext, get the format from the master */
	IOFormat master_format =
	    get_format_app_IOcontext(((IOFile) iocontext)->master_context, 
				     buffer, app_context);

	if (master_format == NULL) {
	    new_format = NULL;
	} else {
	    /* copy it and enter it into our list */
	    new_format = copy_ioformat_to_iofile(master_format, (IOFile) iocontext);
	}
    }
    return new_format;
}

char **
get_subformat_names(field_list)
IOFieldList field_list;
{
    int name_count = 0;
    int field = 0;
    char **name_list = malloc(sizeof(char *));
    while (field_list[field].field_name != NULL) {
	char *base_type = base_data_type(field_list[field].field_type);
	if (str_to_data_type(base_type) == unknown_type) {
	    name_list = realloc(name_list, sizeof(char *) * (name_count + 1));
	    name_list[name_count++] = base_type;
	} else {
	    free(base_type);
	}
	field++;
    }
    name_list = realloc(name_list, sizeof(char *) * (name_count + 1));
    name_list[name_count] = NULL;
    if (name_count == 0) {
	free(name_list);
	name_list = NULL;
    }
    return name_list;
}

static
void
get_subformats_context(ioformat, format_list_p, format_count_p)
IOFormat ioformat;
IOFormat **format_list_p;
int *format_count_p;
{
    int field;
    for (field = 0; field < ioformat->body->field_count; field++) {
	IOFormat subformat = ioformat->field_subformats[field];
	if (subformat != NULL) {
	    int i;
	    get_subformats_context(subformat, format_list_p, format_count_p);
	    *format_list_p = realloc(*format_list_p,
			       sizeof(IOFormat) * (*format_count_p + 1));
	    for (i=0; i< *format_count_p; i++) {
		if ((*format_list_p)[i] == subformat) {
		    /* don't add it if it's already there */
		    subformat = NULL;
		}
	    }
	    if (subformat != NULL) {
		(*format_list_p)[(*format_count_p)++] = subformat;
	    }
	}
    }
}

IOFormat *
get_subformats_IOcontext(iocontext, buffer)
IOContext iocontext;
void *buffer;
{
    IOFormat ioformat = get_format_IOcontext(iocontext, buffer);
    if (ioformat == NULL) {
	int i;
	int id_size = ((IOFile) iocontext)->format_list[0]->body->server_ID.length;
	char hex_buffer[64];
	for (i = 0; i < id_size; i++) {
	    sprintf(&hex_buffer[2 * i], "%02x", ((unsigned char *) buffer)[i]);
	}
	printf("format \"%s\" not registered with format server.\n",
	       hex_buffer);
	return NULL;
    }
    return get_subformats_IOformat(ioformat);
}

IOFormat *
get_subformats_IOformat(ioformat)
IOFormat ioformat;
{
    int format_count = 0;
    IOFormat *format_list = malloc(sizeof(IOFormat));
    get_subformats_context(ioformat, &format_list, &format_count);
    format_list = realloc(format_list, sizeof(IOFormat) * (format_count + 2));
    format_list[format_count] = ioformat;
    format_list[format_count + 1] = NULL;
    return format_list;
}

IOFormat
get_IOformat_by_index(iofile, index)
IOFile iofile;
int index;
{
    if (index < iofile->reg_format_count) {
	return iofile->format_list[index];
    } else {
	return NULL;
    }
}


IOFormat
new_IOFormat()
{
    IOFormat ioformat;
    ioformat = (IOFormat) io_malloc(sizeof(IOFormatStruct));
    ioformat->conversion = NULL;
    ioformat->native_format = 1;
    ioformat->field_subformats = NULL;
    ioformat->warned_about_null_conversion = 0;
    ioformat->body = io_malloc(sizeof(IOFormatBody));
    ioformat->body->ref_count = 1;
    ioformat->body->format_name = NULL;
    ioformat->body->byte_reversal = 0;
    ioformat->body->pointer_size = 0;
    ioformat->body->IOversion = -1;
    ioformat->body->field_list = NULL;
    ioformat->body->var_list = NULL;
    ioformat->body->server_format_rep = NULL;
    ioformat->body->server_ID.length = 0;
    ioformat->body->server_ID.value = NULL;
    ioformat->body->opt_info = NULL;
    ioformat->body->xml_out = NULL;
    return (ioformat);
}

extern
void
expand_IOFile(iofile)
IOFile iofile;
{
    int new_count = iofile->format_list_size + 10;
    int new_size = sizeof(IOFormat) * (new_count);
    int i;

    if (iofile->format_list != NULL) {
	iofile->format_list = (IOFormat *)
	    io_realloc((void *) iofile->format_list, (unsigned) new_size);
    } else {
	iofile->format_list = (IOFormat *) io_malloc((unsigned) new_size);
    }
    iofile->format_list_size = new_count;
    for (i = iofile->reg_format_count; i < iofile->format_list_size; i++) {
	iofile->format_list[i] = NULL;
    }
}

extern void
generate_var_list(ioformat)
IOFormat ioformat;
{
    IOVarInfoList new_var_list;
    IOFieldList field_list = ioformat->body->field_list;
    int field_count = ioformat->body->field_count;
    int field;

    if (ioformat->body->var_list)
	free(ioformat->body->var_list);
    if (ioformat->field_subformats)
	free(ioformat->field_subformats);
    new_var_list = (IOVarInfoList)
	io_malloc((size_t) sizeof(IOVarInfoStruct) * field_count);
    ioformat->field_subformats = io_malloc(sizeof(IOFormat) * field_count);
    ioformat->body->var_list = new_var_list;
    for (field = 0; field < field_count; field++) {
	int control_int = -1;
	new_var_list[field].string = FALSE;
	new_var_list[field].var_array = FALSE;
	new_var_list[field].byte_vector = FALSE;
	new_var_list[field].control_field = NULL;
	ioformat->field_subformats[field] = NULL;
	new_var_list[field].dimen1 = new_var_list[field].dimen2 = 0;
	new_var_list[field].data_type =
	    array_str_to_data_type(field_list[field].field_type,
				   &new_var_list[field].dimen1,
				   &new_var_list[field].dimen2);
	if (new_var_list[field].data_type == string_type) {
	    ioformat->body->variant = TRUE;
	    new_var_list[field].string = TRUE;
	}
	if (new_var_list[field].data_type == unknown_type) {
	    char *base_type = base_data_type(field_list[field].field_type);
	    IOFormat subformat = get_IOformat_by_name(ioformat->iofile,
						      base_type);
	    new_var_list[field].byte_vector =
		(strcmp(base_type, "IOEncodeElem") == 0);
	    free(base_type);
	    if (subformat != NULL) {
		ioformat->body->variant |= subformat->body->variant;
	    }
	    ioformat->field_subformats[field] = subformat;
	}
	if (new_var_list[field].dimen1 == -1) {
	    control_int = get_var_array_control(field_list[field].field_type,
						field_list);
	    if (control_int != -1) {
		/* got a variant array */
		ioformat->body->variant = 1;
		new_var_list[field].var_array = TRUE;
		new_var_list[field].control_field =
		    get_IOfieldPtrFromList(field_list,
				     field_list[control_int].field_name);
		new_var_list[field].control_field->byte_swap =
		    ioformat->body->byte_reversal;
	    }
	}
    }
}

IOFormat
register_IOcontext_format(format_name, field_list, iocontext)
const char *format_name;
IOFieldList field_list;
IOContext iocontext;
{
    return register_IOrecord_format(format_name, field_list,
				    (IOFile) iocontext);
}

IOFormat
register_opt_format(format_name, field_list, optinfo, iocontext)
const char *format_name;
IOFieldList field_list;
IOOptInfo *optinfo;
IOContext iocontext;
{
    return register_general_IOrecord_format(format_name, field_list,
					    (IOFile) iocontext, sizeof(char*),
					    optinfo);
}

IOFormat
register_IOrecord_format(format_name, field_list, iofile)
const char *format_name;
IOFieldList field_list;
IOFile iofile;
{
    return register_general_IOrecord_format(format_name, field_list, iofile,
					    sizeof(char *), NULL);
}

#ifdef WORDS_BIGENDIAN
#define OUR_BYTE_ORDER 1
#define OTHER_BYTE_ORDER 0
#else
#define OUR_BYTE_ORDER 0
#define OTHER_BYTE_ORDER 1
#endif

static format_rep
build_server_format_rep(ioformat)
IOFormat ioformat;
{
    int rep_size = (sizeof(struct _field_wire_format) *
		     (ioformat->body->field_count));
    int i;
    struct _format_wire_format *rep;
    char *string_base;
    int cur_offset;
    struct _field_wire_format *fields;

    rep_size += strlen(ioformat->body->format_name) + 1;
    for (i = 0; i < ioformat->body->field_count; i++) {
	rep_size += strlen(ioformat->body->field_list[i].field_name) + 1;
	rep_size += strlen(ioformat->body->field_list[i].field_type) + 1;
	if (ioformat->field_subformats[i] != NULL) {
	    if (strchr(ioformat->body->field_list[i].field_type, ':') == NULL) {
		/* add format server ID representation - 16 plus two seps */
		IOFormat subformat = ioformat->field_subformats[i];
		switch (version_of_format_ID(subformat->body->server_ID.value)) {
		case 0:
		    rep_size += 18;
		    break;
		default:
		    rep_size += (2 * subformat->body->server_ID.length + 1);
		    break;
		}
	    }
	}
    }

    if (ioformat->body->opt_info == NULL) {
	/* can use version 0 wire format */
	rep_size += sizeof(struct _format_wire_format_0);

	rep = malloc(rep_size);
	cur_offset = (sizeof(struct _format_wire_format_0) + 
		      (sizeof(struct _field_wire_format) * 
		       (ioformat->body->field_count)));
	rep->f.f0.server_rep_version = 0;
	rep->f.f0.header_size = sizeof(struct _format_wire_format_0);
    } else {
	int opt_info_count = 0;

	/* must use version 1 wire format */
	rep_size += sizeof(struct _format_wire_format_1);

	while (ioformat->body->opt_info[opt_info_count].info_type != 0) {
	    rep_size += ioformat->body->opt_info[opt_info_count].info_len;
	    opt_info_count++;
	}
	rep_size += (opt_info_count + 1) * sizeof(struct _opt_info_wire_format);

	rep = malloc(rep_size);
	cur_offset = (sizeof(struct _format_wire_format_1) + 
		      (sizeof(struct _field_wire_format) * 
		       (ioformat->body->field_count)));
	rep->f.f0.server_rep_version = 1;
	rep->f.f0.header_size = sizeof(struct _format_wire_format_1);

	/* fill in some version 1 specific fields */
	rep->f.f1.column_major_arrays = 0;
	rep->f.f1.unused_in_format_1 = 0;  /* so it's not random */
	rep->f.f1.opt_info_offset = 0; /* will be set later */
    }
    string_base = (char *) rep;

    rep->f.f0.name_offset = cur_offset;
    strcpy(string_base + cur_offset, ioformat->body->format_name);
    cur_offset += strlen(ioformat->body->format_name) + 1;

    rep->f.f0.field_count = ioformat->body->field_count;
    rep->f.f0.record_length = ioformat->body->record_length;
    rep->f.f0.record_byte_order = ioformat->body->byte_reversal ? 
	OTHER_BYTE_ORDER : OUR_BYTE_ORDER;
    rep->f.f0.pointer_size = ioformat->body->pointer_size;
    rep->f.f0.floating_point_rep = 0;

    fields = (struct _field_wire_format *) ((char*) rep + 
					    rep->f.f0.header_size);
    for (i = 0; i < ioformat->body->field_count; i++) {
	fields[i].field_size = ioformat->body->field_list[i].field_size;
	fields[i].field_offset = ioformat->body->field_list[i].field_offset;
	fields[i].field_name_offset = cur_offset;
	strcpy(string_base + cur_offset, 
	       ioformat->body->field_list[i].field_name);
	cur_offset += strlen(ioformat->body->field_list[i].field_name) + 1;
	fields[i].field_type_offset = cur_offset;
	if (ioformat->field_subformats[i] == NULL) {
	    strcpy(string_base + cur_offset,
		   ioformat->body->field_list[i].field_type);
	    cur_offset += strlen(ioformat->body->field_list[i].field_type) +1;
	} else {
	    stringify_field_type(ioformat->body->field_list[i].field_type,
				 ioformat->field_subformats[i],
				 string_base + cur_offset, 1024);
	    cur_offset += (strlen(string_base + cur_offset) + 1);
	}
    }
    if (rep->f.f0.server_rep_version == 1) {
	struct _opt_info_wire_format *info_base;
	struct _opt_info_wire_format tmp_base;
	int opt_info_count = 0, i;
	while (ioformat->body->opt_info[opt_info_count].info_type != 0) {
	    opt_info_count++;
	}
	/* fill in opt info fields */
	rep->f.f1.opt_info_offset = cur_offset;
	info_base = (struct _opt_info_wire_format *)(cur_offset +string_base);
	cur_offset += (opt_info_count + 1) *
	    sizeof(struct _opt_info_wire_format);

	tmp_base.info_type = 0;
	tmp_base.info_len = 0;
	tmp_base.info_offset = 0;
	memcpy(&info_base[opt_info_count], &tmp_base, sizeof(tmp_base));

	for (i=0; i < opt_info_count; i++) {
	    tmp_base.info_type = ioformat->body->opt_info[i].info_type;
	    tmp_base.info_len = ioformat->body->opt_info[i].info_len;
	    tmp_base.info_offset = cur_offset;
	    memcpy(&info_base[i], &tmp_base, sizeof(tmp_base));
	    memcpy(string_base + cur_offset, 
		   ioformat->body->opt_info[i].info_block,
		   ioformat->body->opt_info[i].info_len);
	    cur_offset += tmp_base.info_len;
	}
    }

    assert(cur_offset == rep_size);
    rep->f.f0.format_rep_length = rep_size;
    return (format_rep) rep;
}

/* 
 * search_compatible_formats looks in contexts to see if a format (for 
 * which we are seeking a format_ID) has already been registered or has
 * been retrieved from the format server.  This is done by comparing their 
 * server_format_rep fields.
 */
static IOFormat
search_compatible_formats(iocontext, ioformat)
IOContext iocontext;
IOFormat ioformat;
{
    int i, search_rep_length;
    IOFile iofile = (IOFile) iocontext;

    if (ioformat->body->server_format_rep == NULL) {
	return NULL;
    }
    search_rep_length = ioformat->body->server_format_rep->format_rep_length;

    /* search locally first */
    for (i = 0; i < iofile->reg_format_count; i++) {
	IOFormat tmp = iofile->format_list[i];
	int rep_length;

	if (tmp == ioformat)
	    continue;
	if (tmp->body->server_format_rep == NULL) {
	    continue;
	}
	if (strcmp(ioformat->body->format_name,
		   iofile->format_list[i]->body->format_name) != 0)
	    continue;
	rep_length = tmp->body->server_format_rep->format_rep_length;
	if (search_rep_length != rep_length) {
	    if (format_server_verbose == 1) {
		printf("Format %s found in context %lx, but server reps have different lengths, %d and %d\n",
		       ioformat->body->format_name, (long) iocontext,
		       search_rep_length, rep_length);
	    }
	    continue;
	}
	if (memcmp(ioformat->body->server_format_rep, tmp->body->server_format_rep,
		   search_rep_length) == 0) {
	    return tmp;
	} else {
	    if (format_server_verbose == 1) {
		printf("Format %s found in context %lx, but server reps are different\n",
		       ioformat->body->format_name, (long) iocontext);
	    }
	}
    }
    if (iofile->master_context != NULL) {
	return search_compatible_formats(iofile->master_context, ioformat);
    }
    return NULL;
}

static int
is_server_context(IOFile iofile)
{
    if (iofile->master_context != NULL) {
	return is_server_context((IOFile) iofile->master_context);
    } else {
	return (iofile->server_callback != NULL);
    }
}

static int
get_server_port(IOContext iocontext)
{
    IOFile iofile = (IOFile) iocontext;
    if (iofile->master_context != NULL) {
	return get_server_port(iofile->master_context);
    } else {
	return iofile->server_get_port_callback(iofile->server_client_data);
    }
}

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

IOFormat
register_general_IOrecord_format(format_name, field_list, iofile,
				 pointer_size, opt_info)
const char *format_name;
IOFieldList field_list;
IOFile iofile;
int pointer_size;
IOOptInfo *opt_info;
{
    IOFormat ioformat;
    IOFieldList new_field_list;
    int id, new_format_id = -1, field;
    int field_count = count_IOfield(field_list);

    if ((iofile->status != OpenNoHeader) && (iofile->status != OpenHeader)) {
	iofile->result = "Attempt to register IO format in inappropriate IOfile state";
	return NULL;
    }
    for (id = 0; id < iofile->reg_format_count; id++) {
	if (strcmp(format_name, iofile->format_list[id]->body->format_name) == 0) {
	    if (iofile->is_context &&
		!iofile->format_list[id]->native_format) {
		/* 
		 * Never replace a non-native format in a context.  
		 * They aren't matched by name anyway.
		 */
		continue;
	    }
	    free_IOformat(iofile->format_list[id]);
	    new_format_id = id;
	}
    }

    ioformat = new_IOFormat();
    ioformat->body->format_name = io_strdup(format_name);
    ioformat->body->pointer_size = pointer_size;
    ioformat->body->IOversion = iofile->IOversion;
    if (new_format_id == -1) {
	if (iofile->reg_format_count == iofile->format_list_size) {
	    expand_IOFile(iofile);
	}
	ioformat->format_id = iofile->reg_format_count++;
    } else {
	ioformat->format_id = new_format_id;
	if ((iofile->status == OpenHeader) && (!iofile->is_context)) {
	    /* 
	     * already open and headers are away.  Cue the reader that
	     * we're reregistering a format.
	     */
	    if (!write_IOreformat_cue(iofile, new_format_id))
		return NULL;
	}
    }
    iofile->format_list[ioformat->format_id] = ioformat;
    ioformat->iofile = iofile;


    new_field_list = (IOFieldList) io_malloc((size_t) sizeof(IOField) *
					     (field_count + 1));
    ioformat->body->record_length = 0;
    ioformat->body->variant = FALSE;

    for (field = 0; field < field_count; field++) {
	int field_size = 0;
	if (strchr(field_list[field].field_type, '[') == NULL) {
	    /* not an array */
	    field_size = field_list[field].field_size;
	} else {
	    if (get_var_array_control(field_list[field].field_type,
				      field_list) != -1) {
		/* variant array, real_field_size is
		 * iofile->body->pointer_size */
		field_size = ioformat->body->pointer_size;
		if (field_size <= 0) {
		    fprintf(stderr, "Pointer Size is not positive! BAD! pointer size = %d\n",
			    ioformat->body->pointer_size);
		    return NULL;
		}
	    } else {
		int dimen1 = 1, dimen2 = 1;
		IOdata_type base_type;
		if (!validate_data_type(field_list[field].field_type)) {
		    printf("Field type validation failed for field \"%s\", \n\t\t\t\tfield type \"%s\"\n  Format registration rejected.\n",
			   field_list[field].field_name,
			   field_list[field].field_type);
		    return NULL;
		}
		base_type = array_str_to_data_type(field_list[field].field_type,
						   &dimen1, &dimen2);
		if ((dimen1 == -1) || (dimen2 == -1)) {
		    fprintf(stderr, "Array dimension in type spec \"%s\" not recognized.\n",
			    field_list[field].field_type);
		    fprintf(stderr, "Dimension must be a field name (for dynamic arrays) or a positive integer.\n");
		    fprintf(stderr, "To use a #define'd value for the dimension, use the IOArrayDecl() macro.\n");
		    return NULL;
		}
		if ((base_type != unknown_type) &&
		    (field_list[field].field_size > 16)) {
		    fprintf(stderr, "Field size for field %s in format %s is large, check to see if it is valid.\n",
			    field_list[field].field_name, ioformat->body->format_name);
		}
		field_size = field_list[field].field_size * dimen1 * dimen2;
		if (field_size <= 0) {
		    fprintf(stderr, "Field Size is not positive!  field type \"%s\" base= %d, dimen1=%d, dimen2=%d\n",
			    field_list[field].field_type,
			    field_list[field].field_size, dimen1, dimen2);
		    return NULL;
		}
	    }
	}

	ioformat->body->record_length = Max(ioformat->body->record_length,
					 field_list[field].field_offset +
					    field_size);
	new_field_list[field].field_name = io_strdup(field_list[field].field_name);
	new_field_list[field].field_type = io_strdup(field_list[field].field_type);
	new_field_list[field].field_size = field_list[field].field_size;
	new_field_list[field].field_offset = field_list[field].field_offset;
    }
    new_field_list[field_count].field_name = NULL;
    new_field_list[field_count].field_type = NULL;
    new_field_list[field_count].field_size = 0;
    new_field_list[field_count].field_offset = 0;

    if (ioformat->body->record_length == 0) {
	(void) fprintf(stderr, "Format %s has zero max length.\n",
		       format_name);
	(void) fprintf(stderr, "Format rejected.\n ");
	io_free(new_field_list);
	return NULL;
    }
    ioformat->body->field_count = field_count;
    ioformat->body->field_list = new_field_list;
    qsort(ioformat->body->field_list, field_count, sizeof(IOField),
	  field_offset_compar);
    generate_var_list(ioformat);

    if (opt_info != NULL) {
	int opt_info_count = 0;
	while (opt_info[opt_info_count].info_type != 0) opt_info_count++;
	ioformat->body->opt_info = malloc(sizeof(opt_info[0]) * 
					  (opt_info_count + 1));
	memcpy(ioformat->body->opt_info, opt_info, sizeof(opt_info[0]) * 
	       (opt_info_count + 1));
    } else {
	ioformat->body->opt_info = NULL;
    }


    for (field = 0; field < field_count; field++) {
	if (ioformat->body->var_list[field].string == TRUE) {
	    ioformat->body->variant = TRUE;
	} else {
	    char *base_type =
	    base_data_type(ioformat->body->field_list[field].field_type);
	    IOFormat subformat;
	    if (str_to_data_type(base_type) == unknown_type) {
		subformat = get_IOformat_by_name(iofile, base_type);
		if (ioformat->body->var_list[field].var_array == TRUE) {
		    /* got a variant array */
		    ioformat->body->variant = 1;
		} else {
		    if (subformat != NULL) {
			ioformat->body->variant |= subformat->body->variant;
		    }
		}
		if (subformat == NULL) {
		    (void) fprintf(stderr, "Field \"%s\" base type \"%s\" is not a simple type or registered format name.\n",
			    ioformat->body->field_list[field].field_name,
				   base_type);
		    (void) fprintf(stderr, "Format rejected.\n ");
		    free_IOformat(ioformat);
		    return NULL;
		} else {
		    if (ioformat->body->field_list[field].field_size <
			subformat->body->record_length) {
			(void) fprintf(stderr, "Field \"%s\" base type \"%s\" has size %d\n",
				       ioformat->body->field_list[field].field_name,
				       base_type, 
				       ioformat->body->field_list[field].field_size);
			(void) fprintf(stderr, "  this is smaller than record size for \"%s\" (%d bytes)\n",
				       base_type, 
				       subformat->body->record_length);
			free_IOformat(ioformat);
			return NULL;
		    }
		    ioformat->field_subformats[field] = subformat;
		}
	    }
	    free(base_type);
	}
    }
    for (field = 1; field < field_count; field++) {
	int last_field_end = new_field_list[field - 1].field_offset +
	new_field_list[field - 1].field_size;
	if (ioformat->body->var_list[field - 1].var_array) {
	    last_field_end = new_field_list[field - 1].field_offset +
		ioformat->body->pointer_size;
	}
	if (new_field_list[field].field_offset < last_field_end) {
	    (void) fprintf(stderr, "Fields \"%s\" and \"%s\" in format \"%s\" overlap!  Format rejected.\n",
			   new_field_list[field - 1].field_name,
			   new_field_list[field].field_name,
			   format_name);
	    io_free(new_field_list);
	    return NULL;
	}
    }
    if (iofile->is_context) {
	IOFormat cache_format;
	if (format_server_verbose == -1) {
	    if (getenv("FORMAT_SERVER_VERBOSE") == NULL) {
		format_server_verbose = 0;
	    } else {
		format_server_verbose = 1;
	    }
	}
	ioformat->body->server_format_rep = build_server_format_rep(ioformat);
	cache_format = search_compatible_formats((IOContext) iofile,
						 ioformat);
	if (cache_format == NULL) {
	    /* register format with server */
	    if (is_server_context(iofile)) {
		self_server_register_format(iofile, ioformat);
	    } else if (pbio_fixed_format_ids == 1) {
		old_server_register_format(iofile, ioformat);
	    } else {
		server_register_format(iofile, ioformat);
	    }
	    if (format_server_verbose == 1) {
		printf("Registered format with format server - %lx  %d in context %lx\n",
		    (long) ioformat, ioformat->format_id, (long) iofile);
		dump_IOFormat(ioformat);
	    }
	} else {
	    if (cache_format->iofile == iofile) {
		printf("Cache hit in same IOFILE!\n");
	    }
	    free_IOformat_body(ioformat->body);
	    ioformat->body = cache_format->body;
	    ioformat->body->ref_count++;

	    if (format_server_verbose == 1) {
		printf("Cache hit on format registration %lx \"%s\" ", 
		       (long)ioformat, ioformat->body->format_name);
		print_format_ID(ioformat);
		printf("\n");
	    }
	}
    } else if (iofile->status == OpenHeader) {
	/* already open and headers are away.  Write the format now */
	if (!write_IOformat(iofile, ioformat))
	    return NULL;
    }
    replicate_in_masters(ioformat, iofile);
    return ioformat;
}

INT4 PBIO_self_server_IP_addr = 0;

static int
self_server_register_format(iofile, ioformat)
IOFile iofile;
IOFormat ioformat;
{
    static int pid = 0;
    INT4 IP_addr = PBIO_self_server_IP_addr;
    IOFormat new_format = ioformat;

    if (format_server_verbose == -1) {
	if (getenv("FORMAT_SERVER_VERBOSE") == NULL) {
	    format_server_verbose = 0;
	} else {
	    format_server_verbose = 1;
	}
    }
    /* we're a format server ourselves, assign an ID */
    if (iofile->master_context != NULL) {
	return self_server_register_format((IOFile) iofile->master_context,
					   ioformat);
    }
    if (pid == 0) {
	pid = (int) getpid();
    }
    if (ioformat->iofile != iofile) {
	/* registering in a master, copy it down */
	new_format = copy_ioformat_to_iofile(ioformat, iofile);
    }
    ioformat->body->server_ID.length = 10;
    ioformat->body->server_ID.value = malloc(10);
    ((version_1_format_ID *) ioformat->body->server_ID.value)->version = 1;
    ((version_1_format_ID *) ioformat->body->server_ID.value)->salt =
	(pid & 0xff);
    ((version_1_format_ID *) ioformat->body->server_ID.value)->port =
	get_server_port((IOContext) iofile);
    ((version_1_format_ID *) ioformat->body->server_ID.value)->IP_addr =
	IP_addr;
    ((version_1_format_ID *) ioformat->body->server_ID.value)->format_identifier = new_format->format_id;
    ioformat->body->server_format_rep = build_server_format_rep(ioformat);
    if (format_server_verbose == 1) {
	printf("Registering %s to locally-issued format ID ",
	       ioformat->body->format_name);
	print_format_ID(ioformat);
	printf("\n");
    }
    return 1;
}

extern
int
count_IOfield(list)
IOFieldList list;
{
    int i = 0;
    while (list[i].field_name != NULL) {
	i++;
    }
    return i;
}

extern
int
struct_size_IOfield(iofile, list)
IOFile iofile;
IOFieldList list;
{
    int i = 0;
    int struct_size = 0;
    while (list[i].field_name != NULL) {
	int field_size = 0;
	if (get_var_array_control(list[i].field_type, list) != -1) {
	    /* variant array, real_field_size is ioformat->pointer_size */
	    if ((iofile == NULL) || (iofile->pointer_size == 0)) {
		field_size = sizeof(char *);
	    } else {
		field_size = iofile->pointer_size;
	    }
	} else {
	    if (iofile && (iofile->IOversion > 3)) {
		int dimen1 = 1, dimen2 = 1;
		array_str_to_data_type(list[i].field_type, &dimen1, &dimen2);
		field_size = list[i].field_size * dimen1 * dimen2;
	    } else {
		field_size = list[i].field_size;
	    }
	}
	assert(field_size > 0);
	struct_size = Max(struct_size,
			  (list[i].field_offset + field_size));
	i++;
    }
    return struct_size;
}

extern
int
struct_size_field_list(list, pointer_size)
IOFieldList list;
int pointer_size;
{
    int i = 0;
    int struct_size = 0;
    while (list[i].field_name != NULL) {
	int field_size = 0;
	if (get_var_array_control(list[i].field_type, list) != -1) {
	    /* variant array, real_field_size is ioformat->pointer_size */
	    field_size = pointer_size;
	} else {
	    int dimen1 = 1, dimen2 = 1;
	    array_str_to_data_type(list[i].field_type, &dimen1, &dimen2);
	    field_size = list[i].field_size * dimen1 * dimen2;
	}
	assert(field_size > 0);
	struct_size = Max(struct_size,
			  (list[i].field_offset + field_size));
	i++;
    }
    return struct_size;
}

extern
IOFieldList
field_list_of_IOformat(format)
IOFormat format;
{
    return format->body->field_list;
}

extern
int
compare_field_lists(list1, list2)
IOFieldList list1, list2;
{
    int i = 0;
    do {
	if ((strcmp(list1[i].field_name, list2[i].field_name) != 0) ||
	    (list1[i].field_size != list2[i].field_size) ||
	    (list1[i].field_offset != list2[i].field_offset) ||
	    !IO_field_type_eq(list1[i].field_type, list2[i].field_type)) {
	    return 1;
	}
	i++;
    } while ((list1[i].field_name != NULL) || (list2[i].field_name != NULL));

    return 0;
}

/* 
 * Max field list returns the "maximum" of two field lists. 
 * Basically, it prefers list1.  If all fields in list2 are in list 1, and 
 * if no fields in list2 are bigger than the corresponding field in
 * list1, the maximum returned is list1.  If there is a field in list2 
 * that is bigger than the corresponding field in list1, we'll still
 * use the ordering and even try to use the offsets in list1.  If
 * there's actually a field in list2 that isn't in list1, then most
 * bets are off as to exactly what will happen.  Basically the
 * resulting list will contain all fields, but we'll use the offsets
 * from their original lists to establish an ordering and then bump
 * the offsets up as necessary to ensure no overlap.  The result is 
 * likely a jumble that isn't necessarily terribly close to what was in 
 * either list...  Bummer for performance perhaps, but being smarter is
 * hard. 
 */
extern
IOFieldList
max_field_lists(list1, list2)
IOFieldList list1, list2;
{
    IOFieldList max_field_list = NULL;
    IOFieldList tlist2;
    int field_count1 = count_IOfield(list1);
    int field_count2 = count_IOfield(list2);
    int max_field_count = 0;
    int i, j;

    /* make a copy of list2 that we can scratch on */
    tlist2 = (IOFieldList) io_malloc((size_t) sizeof(IOField) * field_count2);
    memcpy(tlist2, list2, sizeof(IOField) * field_count2);

    max_field_list = (IOFieldList) io_malloc((size_t) sizeof(IOField) *
				      (field_count1 + field_count2 + 1));

    for (i = 0; i < field_count1; i++) {
	/* for each field in list1 */
	for (j = 0; j < field_count2; j++) {
	    if (tlist2[j].field_name &&
	      (strcmp(list1[i].field_name, tlist2[j].field_name) == 0)) {
		break;
	    }
	}
	if (j == field_count2) {
	    /* field exists only in 1, use it as is */
	    max_field_list[max_field_count].field_name =
		io_strdup(list1[i].field_name);
	    max_field_list[max_field_count].field_type =
		io_strdup(list1[i].field_type);
	    max_field_list[max_field_count].field_size =
		list1[i].field_size;
	    max_field_list[max_field_count].field_offset =
		list1[i].field_offset;
	    max_field_count++;
	} else {
	    /* field list in both.  Check types and take max size */
	    if (!IO_field_type_eq(list1[i].field_type, tlist2[j].field_type)) {
		/* Yikes!  Types are bad!  Serious problem */
		io_free(tlist2);
		io_free(max_field_list);
		return NULL;
	    }
	    max_field_list[max_field_count].field_name =
		io_strdup(list1[i].field_name);
	    max_field_list[max_field_count].field_type =
		io_strdup(list1[i].field_type);
	    max_field_list[max_field_count].field_size =
		Max(list1[i].field_size, tlist2[j].field_size);
	    /* tentative offset */
	    max_field_list[max_field_count].field_offset =
		list1[i].field_offset;
	    max_field_count++;
	    /* mark field in list 2 as used */
	    tlist2[j].field_name = NULL;
	}
    }
    for (j = 0; j < field_count2; j++) {
	if (tlist2[j].field_name != NULL) {
	    /* didn't mark this field as used.  Must only be in list2 */
	    max_field_list[max_field_count].field_name =
		io_strdup(tlist2[j].field_name);
	    max_field_list[max_field_count].field_type =
		io_strdup(tlist2[j].field_type);
	    max_field_list[max_field_count].field_size =
		tlist2[j].field_size;
	    max_field_list[max_field_count].field_offset =
		tlist2[j].field_offset;
	    max_field_count++;
	}
    }
    io_free(tlist2);

    max_field_list[max_field_count].field_name = NULL;
    max_field_list[max_field_count].field_type = NULL;
    max_field_list[max_field_count].field_size = 0;
    max_field_list[max_field_count].field_offset = 0;

    /* sort the max list by offset now */
    qsort(max_field_list, max_field_count, sizeof(IOField),
	  field_offset_compar);
    for (i = 1; i < max_field_count; i++) {
	int last_field_end = max_field_list[i - 1].field_offset +
	max_field_list[i - 1].field_size;
	if (max_field_list[i].field_offset < last_field_end) {
	    /* damn, got some overlap.  Push this field up to fix. */
	    max_field_list[i].field_offset = last_field_end;
	}
    }
    return max_field_list;
}

extern int
min_align_size(size)
int size;
{
    int align_size = 8;		/* conservative on current machines */
    switch (size) {
    case 7:
    case 6:
    case 5:
    case 4:
	align_size = 4;
	break;
    case 3:
	align_size = 2;
	break;
    case 2:
    case 1:
	align_size = size;
	break;
    }
    return align_size;
}

extern void
force_align_field_list(field_list, pointer_size)
IOFieldList field_list;
int pointer_size;
{
    int i;
    int last_field_size = 0, cur_field_size;
    for (i = 0; field_list[i].field_name != NULL; i++) {
	int align_size;
	int align_field_size;
	if (get_var_array_control(field_list[i].field_type, field_list)
	    != -1) {
	    /* variant array, real_field_size is pointer_size */
	    align_field_size = pointer_size;
	    cur_field_size = pointer_size;
	} else {
	    int dimen1 = 1, dimen2 = 1;
	    IOdata_type base_type;
	    base_type = array_str_to_data_type(field_list[i].field_type,
					       &dimen1, &dimen2);
	    if ((base_type != unknown_type) &&
		(field_list[i].field_size > 16)) {
		fprintf(stderr, "Field size for field %s is large, check to see if it is valid.\n",
			field_list[i].field_name);
	    }
	    align_field_size = field_list[i].field_size;
	    cur_field_size = field_list[i].field_size * dimen1 * dimen2;
	}
	align_size = min_align_size(align_field_size);
	if (i > 0) {
	    int last_field_end = field_list[i - 1].field_offset +
	    last_field_size;
	    if (field_list[i].field_offset < last_field_end) {
		/* damn, got some overlap.  Push this field up to fix. */
		field_list[i].field_offset = last_field_end;
	    }
	}
	if ((field_list[i].field_offset % align_size) != 0) {
	    /* field is mis-aligned, move it up...  */
	    int offset = field_list[i].field_offset + align_size;
	    /* adjust down to the boundary */
	    offset -= (field_list[i].field_offset % align_size);
	    field_list[i].field_offset = offset;
	}
	last_field_size = cur_field_size;
    }
}

static void
assign_local_sizes_field_list(IOFieldList fl, IOContext c)
{
    while (fl->field_name != NULL) {
	IOdata_type base_type;
	int dimen1, dimen2;
	int element_size = fl->field_size;
	base_type = array_str_to_data_type(fl->field_type, &dimen1, &dimen2);
	if (dimen1 == -1) {
	    dimen1 = dimen2 = 1;
	} else {
	    element_size = fl->field_size / (dimen1 * dimen2);
	}
	switch(base_type) {
	case integer_type:
	case unsigned_type:
	case boolean_type:
	case enumeration_type:
	    if ((element_size == 0) || (element_size > sizeof(long))) {
		element_size = sizeof(long);
	    }
	    break;
	case float_type:
	    if ((element_size == 0) || (element_size > SIZEOF_DOUBLE)) {
		element_size = sizeof(double);
	    }
	    break;
	case char_type:
	    if (element_size == 0) {
		element_size = 1;
	    }
	    break;
	case string_type:
	    element_size = sizeof(char*);
	    break;
	case unknown_type:
	{
	    char *base = base_data_type(fl->field_type);
	    IOFormat subformat = get_IOformat_by_name((IOFile)c, base);
	    if (subformat != NULL) {
		element_size = subformat->body->record_length;
	    } else {
		fprintf(stderr, "Field type \"%s\" unknown in local sizes context\n",
			base);
	    }
	    free(base);
	    break;
	}
	}
	fl->field_size = element_size * (dimen1 * dimen2);
	fl->field_offset = 0;
	fl++;
    }
}

extern
IOFieldList
localize_field_list(IOFieldList list, IOContext c)
{
    assign_local_sizes_field_list(list, c);
    force_align_field_list(list, sizeof(char*));
    return list;
}
    
extern
IOFieldList
get_local_field_list(IOFormat ioformat)
{
    int i = 0;
    IOFieldList fl = copy_field_list(field_list_of_IOformat(ioformat));
    while(fl[i].field_name != NULL) {
	char *colon = strchr(fl[i].field_type, ':');
	char *bracket = strchr(fl[i].field_type, '[');

	if (colon != NULL) {
	    IOConversionPtr subconv = 
		ioformat->field_subformats[i]->conversion;
	    if (subconv != NULL) {
		fl[i].field_size += subconv->base_size_delta;
	    }
	    *colon = 0;
	}
	if (bracket != NULL) strcpy(colon, bracket);
	i++;
    }
    force_align_field_list(fl, sizeof(char*));
    return fl;
}

extern
IOFieldList
copy_field_list(list)
IOFieldList list;
{
    int field_count = count_IOfield(list);
    IOFieldList new_field_list;
    int field;

    new_field_list = (IOFieldList) io_malloc((size_t) sizeof(IOField) *
					     (field_count + 1));
    for (field = 0; field < field_count; field++) {
	new_field_list[field].field_name = io_strdup(list[field].field_name);
	new_field_list[field].field_type = io_strdup(list[field].field_type);
	new_field_list[field].field_size = list[field].field_size;
	new_field_list[field].field_offset = list[field].field_offset;
    }
    new_field_list[field_count].field_name = NULL;
    new_field_list[field_count].field_type = NULL;
    new_field_list[field_count].field_offset = 0;
    new_field_list[field_count].field_size = 0;
    return new_field_list;
}

extern
void
free_field_list(list)
IOFieldList list;
{
    int i = 0;
    while (list[i].field_name != NULL) {
	io_free((char*)list[i].field_name);
	io_free((char*)list[i].field_type);
	i++;
    }
    io_free(list);
}

static int
field_name_compar(a, b)
const void *a;
const void *b;
{
    IOFieldList ia = (IOFieldList) a;
    IOFieldList ib = (IOFieldList) b;
    return (strcmp(ia->field_name, ib->field_name));
}

IOformat_order
IOformat_cmp(format1, format2)
IOFormat format1;
IOFormat format2;
{
    IOformat_order tmp_result = Format_Equal;
    IOFieldList field_list1 =
	copy_field_list(field_list_of_IOformat(format1));
    IOFieldList field_list2 =
	copy_field_list(field_list_of_IOformat(format2));
    IOFormat *subformats1 = NULL, *subformats2 = NULL, *tmp_subformats = NULL;
    int field_count1, field_count2;
    int i, j, limit;

    /* count fields */
    for (field_count1 = 0; field_list1[field_count1].field_name != NULL;
	 field_count1++);

    /* count fields */
    for (field_count2 = 0; field_list2[field_count2].field_name != NULL;
	 field_count2++);

    qsort(field_list1, field_count1, sizeof(IOField), field_name_compar);
    qsort(field_list2, field_count2, sizeof(IOField), field_name_compar);

    limit = field_count1;
    if (field_count2 < limit)
	limit = field_count2;
    i = j = 0;
    while ((i < field_count1) && (j < field_count2)) {
	int name_cmp;
	if ((name_cmp = strcmp(field_list1[i].field_name,
			       field_list2[j].field_name)) == 0) {
	    /* fields have same name */
	    if (!IO_field_type_eq(field_list1[i].field_type,
				  field_list2[j].field_type)) {
		return Format_Incompatible;
	    }
	} else if (name_cmp < 0) {
	    /* name_cmp>0  a field in field_list1 that doesn't appear in 2 */
	    if (tmp_result == Format_Less) {
		/* we previously found the opposite */
		tmp_result = Format_Incompatible;
		goto free_lists;
	    } else {
		/* tentatively decide that list1 is a superset of 2 */
		tmp_result = Format_Greater;
	    }
	    /* skip fields in 1 until we find the next field in 2 */
	    while (strcmp(field_list1[i].field_name,
			  field_list2[j].field_name) < 0) {
		i++;
		if (i == field_count1) {
		    /* didn't find one */
		    tmp_result = Format_Incompatible;
		    goto free_lists;
		}
	    }
	    i--;
	    j--;
	} else {
	    /* name_cmp>0  a field in field_list2 that doesn't appear in 1 */
	    if (tmp_result == Format_Greater) {
		tmp_result = Format_Incompatible;
		goto free_lists;
	    } else {
		tmp_result = Format_Less;
	    }
	    /* skip fields in field_list2 until we find matching */
	    while (strcmp(field_list1[i].field_name,
			  field_list2[j].field_name) > 0) {
		j++;
		if (j == field_count2) {
		    /* didn't find one */
		    tmp_result = Format_Incompatible;
		    goto free_lists;
		}
	    }
	    i--;
	    j--;
	}
	i++;
	j++;
    }
    if (i < field_count1) {
	if (tmp_result == Format_Less) {
	    tmp_result = Format_Incompatible;
	    goto free_lists;
	} else {
	    tmp_result = Format_Greater;
	}
    } else if (j < field_count2) {
	if (tmp_result == Format_Greater) {
	    tmp_result = Format_Incompatible;
	    goto free_lists;
	} else {
	    tmp_result = Format_Less;
	}
    }
    /* go through subformats */
    tmp_subformats = subformats1 = get_subformats_IOformat(format1);
    subformats2 = get_subformats_IOformat(format2);

    while (*subformats1 != NULL) {
	char *sub1_name = name_of_IOformat(*subformats1);
	int i = 0;
	if (*subformats1 == format1) {
	    /* self appears in subformat list, skip it */
	    subformats1++;
	    continue;
	}
	while (subformats2[i] != NULL) {
	    if (strcmp(sub1_name, name_of_IOformat(subformats2[i])) == 0) {
		/* same name, compare */
		IOformat_order sub_result;
		sub_result = IOformat_cmp(*subformats1,
					  subformats2[i]);
		if (sub_result == Format_Incompatible) {
		    tmp_result = Format_Incompatible;
		    goto free_lists;
		}
		if (sub_result != Format_Equal) {
		    /* subformats are not equal! */
		    if (tmp_result == Format_Equal) {
			/* we thought they were equal before */
			tmp_result = sub_result;
		    } else if (tmp_result != sub_result) {
			tmp_result = Format_Incompatible;
			goto free_lists;
		    }
		}
		break;
	    }
	    i++;
	}
	subformats1++;
    }

 free_lists:
    free_field_list(field_list1);
    free_field_list(field_list2);
    if (tmp_subformats) free(tmp_subformats);
    if (subformats2) free(subformats2);
    return tmp_result;
}

IOFormat
reregister_IOformat(other_format, iofile)
IOFormat other_format;
IOFile iofile;
{
    IOFormat ioformat;
    IOFieldList new_field_list, field_list = other_format->body->field_list;
    int id, field, new_format_id = -1;
    char *format_name = other_format->body->format_name;

    if ((iofile->status != OpenNoHeader) && (iofile->status != OpenHeader)) {
	iofile->result = "Attempt to register IO format in inappropriate IOfile state";
	return NULL;
    }
    for (id = 0; id < iofile->reg_format_count; id++) {
	if (strcmp(format_name, iofile->format_list[id]->body->format_name) == 0) {
	    io_free(iofile->format_list[id]->body->field_list);
	    io_free(iofile->format_list[id]);
	    new_format_id = id;
	}
    }

    ioformat = new_IOFormat();
    ioformat->body->format_name = format_name;
    ioformat->body->pointer_size = sizeof(char *);
    ioformat->body->IOversion = iofile->IOversion;
    if (new_format_id == -1) {
	if (iofile->reg_format_count == iofile->format_list_size) {
	    expand_IOFile(iofile);
	}
	ioformat->format_id = iofile->reg_format_count++;
    } else {
	ioformat->format_id = new_format_id;
	if (iofile->status == OpenHeader) {
	    /* 
	     * already open and headers are away.  Cue the reader that
	     * we're reregistering a format. 
	     */
	    if (!write_IOreformat_cue(iofile, new_format_id))
		return NULL;
	}
    }
    iofile->format_list[ioformat->format_id] = ioformat;
    ioformat->iofile = iofile;


    new_field_list = (IOFieldList) io_malloc((size_t) sizeof(IOField) *
				  (other_format->body->field_count + 1));
    ioformat->body->record_length = other_format->body->record_length;
    ioformat->body->variant = other_format->body->variant;
    for (field = 0; field < other_format->body->field_count; field++) {
	new_field_list[field].field_name = io_strdup(field_list[field].field_name);
	new_field_list[field].field_type = io_strdup(field_list[field].field_type);
	new_field_list[field].field_size = field_list[field].field_size;
	new_field_list[field].field_offset = field_list[field].field_offset;
    }
    new_field_list[other_format->body->field_count].field_name = NULL;
    new_field_list[other_format->body->field_count].field_type = NULL;
    new_field_list[other_format->body->field_count].field_size = 0;
    new_field_list[other_format->body->field_count].field_offset = 0;

    ioformat->body->field_count = other_format->body->field_count;
    ioformat->body->field_list = new_field_list;
    qsort(ioformat->body->field_list, ioformat->body->field_count, sizeof(IOField),
	  field_offset_compar);
    if (iofile->status == OpenHeader) {
	/* already open and headers are away.  Write the format now */
	if (!write_IOformat(iofile, ioformat))
	    return NULL;
    }
    return ioformat;
}

extern
IOFormat
duplicate_IOrecord_format(src, iofile)
IOFormat src;
IOFile iofile;
{
    IOFormat ioformat;
    IOFieldList new_field_list;
    int id, field;

    if ((iofile->status != OpenNoHeader) && (iofile->status != OpenHeader)) {
	iofile->result = "Attempt to register IO format in inappropriate IOfile state";
	return NULL;
    }
    for (id = 0; id < iofile->reg_format_count; id++) {
	if (strcmp(src->body->format_name,
		   iofile->format_list[id]->body->format_name) == 0) {
	    iofile->result = "Attempt to register duplicate format name";
	    return NULL;
	}
    }
    ioformat = new_IOFormat();
    ioformat->format_id = 0;
    ioformat->body->format_name = io_strdup(src->body->format_name);
    ioformat->body->pointer_size = src->body->pointer_size;
    ioformat->body->IOversion = src->body->IOversion;
    ioformat->native_format = src->native_format;
    if (iofile->reg_format_count == iofile->format_list_size) {
	expand_IOFile(iofile);
    }
    ioformat->format_id = iofile->reg_format_count++;
    iofile->format_list[ioformat->format_id] = ioformat;
    ioformat->iofile = iofile;

    new_field_list = (IOFieldList)
	io_malloc((size_t) sizeof(IOField) * (src->body->field_count + 1));
    ioformat->body->record_length = 0;
    ioformat->body->variant = src->body->variant;;
    for (field = 0; field < src->body->field_count; field++) {
	ioformat->body->record_length = Max(ioformat->body->record_length,
			      src->body->field_list[field].field_offset +
				src->body->field_list[field].field_size);
	new_field_list[field].field_name =
	    io_strdup(src->body->field_list[field].field_name);
	new_field_list[field].field_type =
	    io_strdup(src->body->field_list[field].field_type);
	new_field_list[field].field_size = src->body->field_list[field].field_size;
	new_field_list[field].field_offset =
	    src->body->field_list[field].field_offset;
    }
    new_field_list[src->body->field_count].field_name = NULL;
    new_field_list[src->body->field_count].field_type = NULL;
    new_field_list[src->body->field_count].field_size = 0;
    new_field_list[src->body->field_count].field_offset = 0;
    ioformat->body->field_count = src->body->field_count;
    ioformat->body->field_list = new_field_list;
    return ioformat;
}

extern
int
duplicate_IOfile_format(srcfile, iofile)
IOFile srcfile, iofile;
{
    int i;
    IOFormat ioformat;
    for (i = 0; i < srcfile->reg_format_count; i++) {
	ioformat = duplicate_IOrecord_format(srcfile->format_list[i],
					     iofile);
	if (ioformat == NULL)
	    return -1;
    }
    iofile->byte_reversal = srcfile->byte_reversal;
    iofile->architecture = io_strdup(srcfile->architecture);
    return 0;
}

extern
IOFormat
next_IOrecord_format(iofile)
IOFile iofile;
{
    if ((iofile->status != OpenForRead) && (iofile->status != OpenBeginRead))
	return NULL;

    if (iofile->read_ahead == FALSE) {
	(void) next_IOrecord_type(iofile);
    }
    while (iofile->next_record_type != IOdata) {
	switch (iofile->next_record_type) {
	case IOcomment:
	    (void) read_comment_IOfile(iofile);
	    (void) next_IOrecord_type(iofile);
	    break;
	case IOformat:
	    (void) read_format_IOfile(iofile);
	    (void) next_IOrecord_type(iofile);
	    break;
	default:
	    return NULL;
	}
    }
    return iofile->next_record_format;
}

extern
int
index_of_IOformat(format)
IOFormat format;
{
    return format->format_id;
}

extern
char *
name_of_IOformat(format)
IOFormat format;
{
    return format->body->format_name;
}

extern
int
pointer_size_of_IOformat(format)
IOFormat format;
{
    return format->body->pointer_size;
}

extern
IOFile
iofile_of_IOformat(format)
IOFormat format;
{
    return format->iofile;
}


static int
write_IOreformat_cue(iofile, format_id)
IOFile iofile;
int format_id;
{
    FILE_INT file_format_id = format_id;
    FILE_INT code = REREGISTERED_FORMAT_FOLLOWS;

    if (!put_AtomicInt(iofile, &code))
	return 0;
    if (!put_AtomicInt(iofile, &file_format_id))
	return 0;
    return 1;
}


#define WRITEV_FIELD_COUNT_LIMIT 10
#define WRITE_BLOCK_SIZE (6 + WRITEV_FIELD_COUNT_LIMIT * 4)
#define  WRITE_VEC_SIZE (2 + 3 * WRITEV_FIELD_COUNT_LIMIT)
extern int
write_IOformat(iofile, ioformat)
IOFile iofile;
IOFormat ioformat;
{
    FILE_INT write_block[WRITE_BLOCK_SIZE];
    internal_iovec write_vec[WRITE_VEC_SIZE];
    FILE_INT name_length, field_count, record_length, format_id, pointer_size;
    int field;
    FILE_INT code = FORMAT_FOLLOWS;
    int rec_len_index = 4;

    int write_vec_index = 0;
    int write_block_index = 0;

    name_length = strlen(ioformat->body->format_name);
    format_id = ioformat->format_id;
    field_count = ioformat->body->field_count;
    record_length = ioformat->body->record_length;
    pointer_size = ioformat->body->pointer_size;
    write_block[0] = code;
    write_block[1] = name_length;
    write_block[2] = format_id;
    write_block[3] = field_count;
    if (iofile->IOversion >= 5) {
	write_block[4] = pointer_size;
	rec_len_index = 5;
    }
    write_block[rec_len_index] = record_length;

    write_block_index = rec_len_index + 1;

    write_vec[0].iov_base = &write_block[0];
    write_vec[0].iov_len = write_block_index * sizeof(FILE_INT);
    write_vec[0].iov_offset = 0;
    write_vec[1].iov_base = ioformat->body->format_name;
    write_vec[1].iov_len = name_length + 1;
    write_vec[1].iov_offset = 0;

    write_vec_index = 2;
    field = 0;
    while (field < ioformat->body->field_count) {
	while ((write_vec_index + 3 < WRITE_VEC_SIZE) &&
	       (field < ioformat->body->field_count)) {
	    IOField *iofield = &(ioformat->body->field_list[field]);
	    /* field_name_len */
	    int field_name_len = strlen(iofield->field_name);
	    /* field_type_len */
	    int field_type_len = strlen(iofield->field_type);
	    write_block[write_block_index] = field_name_len;
	    write_block[write_block_index + 1] = field_type_len;
	    /* field_size */
	    write_block[write_block_index + 2] = iofield->field_size;
	    /* field_offset */
	    write_block[write_block_index + 3] = iofield->field_offset;

	    write_vec[write_vec_index].iov_base =
		&write_block[write_block_index];
	    write_block_index += 4;

	    write_vec[write_vec_index].iov_len = 4 * sizeof(FILE_INT);
	    write_vec[write_vec_index].iov_offset = 0;
	    write_vec[write_vec_index + 1].iov_base = 
		(void*)iofield->field_name;
	    write_vec[write_vec_index + 1].iov_len = field_name_len + 1;
	    write_vec[write_vec_index + 1].iov_offset = 0;
	    write_vec[write_vec_index + 2].iov_base = 
		(void*)iofield->field_type;
	    write_vec[write_vec_index + 2].iov_len = field_type_len + 1;
	    write_vec[write_vec_index + 2].iov_offset = 0;
	    write_vec_index += 3;
	    field++;
	}
	if (AtomicWriteV(iofile, &write_vec[0], write_vec_index) !=
	    write_vec_index) {
	    return 0;
	}
	write_vec_index = 0;
	write_block_index = 0;
    }
    return 1;
}

IOFormat
read_format_IOfile(iofile)
IOFile iofile;
{
    FILE_INT name_length;
    FILE_INT field_count;
    FILE_INT record_length;
    FILE_INT format_id;
    IOFormat ioformat = new_IOFormat();
    IOVarInfoList new_var_list;
    int field;

    iofile->status = OpenForRead;
    ioformat->body->byte_reversal = iofile->byte_reversal;
    ioformat->body->IOversion = iofile->IOversion;
    ioformat->body->pointer_size = iofile->pointer_size;
    ioformat->native_format = 0;
    if ((iofile->IOversion > 1) && (next_IOrecord_type(iofile) != IOformat)) {
	iofile->result = "Next record type is not a format";
	return NULL;
    }
    reset_read_ahead(iofile);
    if (!get_AtomicInt(iofile, &name_length))
	return NULL;
    if (iofile->IOversion >= 3) {
	if (!get_AtomicInt(iofile, &format_id))
	    return NULL;
    } else {
	format_id = iofile->reg_format_count;
    }
    if (!get_AtomicInt(iofile, &field_count))
	return NULL;
    if (iofile->IOversion >= 5) {
	FILE_INT pointer_size;
	if (!get_AtomicInt(iofile, &pointer_size))
	    return NULL;
	ioformat->body->pointer_size = pointer_size;
    }
    if (!get_AtomicInt(iofile, &record_length))
	return NULL;

    ioformat->body->format_name = io_malloc(name_length + 1);
    ioformat->body->field_count = field_count;
    ioformat->body->variant = FALSE;
    ioformat->body->record_length = record_length;
    ioformat->conversion = NULL;
    AtomicRead(iofile->file_id, ioformat->body->format_name, name_length + 1, iofile);
    ioformat->body->field_list = (IOFieldList) io_malloc(sizeof(IOField) *
				      (ioformat->body->field_count + 1));
    new_var_list = (IOVarInfoList)
	io_malloc((size_t) sizeof(IOVarInfoStruct) * field_count);
    ioformat->body->var_list = new_var_list;

    for (field = 0; field < ioformat->body->field_count; field++) {
	FILE_INT values[4];
	FILE_INT field_name_len;
	FILE_INT field_type_len;
	IOField *iofield = &(ioformat->body->field_list[field]);

	if (AtomicRead(iofile->file_id, &values[0], 4 * sizeof(FILE_INT),
		       iofile) != 4 * sizeof(FILE_INT))
	    return NULL;
	if (iofile->byte_reversal) {
	    int i;
	    for (i = 0; i < 4; i++) {
		byte_swap((char *) &values[i], 4);
	    }
	}
	field_name_len = values[0];
	field_type_len = values[1];
	iofield->field_size = values[2];
	iofield->field_offset = values[3];
	iofield->field_name = io_malloc(field_name_len + 1);
	iofield->field_type = io_malloc(field_type_len + 1);
	if (AtomicRead(iofile->file_id, (void*)iofield->field_name,
		     field_name_len + 1, iofile) != field_name_len + 1) {
	    return NULL;
	}
	if (AtomicRead(iofile->file_id, (void*)iofield->field_type,
		     field_type_len + 1, iofile) != field_type_len + 1) {
	    return NULL;
	}
    }
    ioformat->body->field_list[ioformat->body->field_count].field_name = NULL;
    ioformat->body->field_list[ioformat->body->field_count].field_type = NULL;
    ioformat->body->field_list[ioformat->body->field_count].field_offset = 0;
    ioformat->body->field_list[ioformat->body->field_count].field_size = 0;

    if (format_id >= iofile->format_list_size) {
	expand_IOFile(iofile);
    }
    if (format_id > iofile->reg_format_count) {
	iofile->result = "Internal error. skipped format ids.";
	return NULL;
    }
    if (format_id == iofile->reg_format_count) {
	/* new format came in */
	iofile->reg_format_count++;
    }
    ioformat->format_id = format_id;

    if (iofile->format_list[ioformat->format_id] != NULL) {
	free_IOformat(iofile->format_list[ioformat->format_id]);
    }
    iofile->format_list[ioformat->format_id] = ioformat;
    ioformat->iofile = iofile;
    generate_var_list(ioformat);
    return ioformat;
}

extern void
dump_IOFormat(ioformat)
IOFormat ioformat;
{
    int index, i;
    printf("\tFormat %d \"%s\"; size = %d; Field_Count = %d; Endian = %d;\n\t\t  Pointer size = %d; File Version = %d; ",
	   ioformat->format_id, ioformat->body->format_name, ioformat->body->record_length,
	   ioformat->body->field_count, ioformat->body->byte_reversal,
	   ioformat->body->pointer_size, ioformat->body->IOversion);
    printf("Server ID = ");
    for (i = 0; i < ioformat->body->server_ID.length; i++) {
	printf("%02x", ((unsigned char *) ioformat->body->server_ID.value)[i]);
    }
    printf("\n");
    for (index = 0; index < ioformat->body->field_count; index++) {
	printf("\t\t%s \"%s\"; size = %d; offset = %d\n",
	       ioformat->body->field_list[index].field_name,
	       ioformat->body->field_list[index].field_type,
	       ioformat->body->field_list[index].field_size,
	       ioformat->body->field_list[index].field_offset);
    }
    dump_IOConversion(ioformat->conversion);
    if (ioformat->body->opt_info == NULL) {
	printf("\tNo Optional Format Info\n");
    } else {
	int i = 0;
	while (ioformat->body->opt_info[i].info_type != 0) {
	    int typ = ioformat->body->opt_info[i].info_type;
	    int j, text = 1, len = ioformat->body->opt_info[i].info_len;
	    printf("\t Opt Info \"%c%c%c%c\", size %d, block begins:\n\t\t",
		   (typ >> 24) & 0xff, (typ >> 16) & 0xff, (typ >> 8) & 0xff,
		   typ & 0xff, len);
	    for(j=0; ((j < 10) && (j < len)); j++) {
		if (!isprint((int)((char*)ioformat->body->opt_info[i].info_block)[j])) text = 0;
	    }
	    if (text == 1) {
		printf("\"");
		for(j=0;((j < 50) && (j < len)); j++) {
		    printf("%c", ((char*)ioformat->body->opt_info[i].info_block)[j]);
		}
		printf("\"\n");
	    } else {
		for(j=0;((j < 20) && (j < len)); j++) {
		    printf("%02x ", ((unsigned char*)ioformat->body->opt_info[i].info_block)[j]);
		}
		printf("\n");
	    }
	    i++;
	}
    }
}

extern void
dump_IOFormat_as_XML(ioformat)
IOFormat ioformat;
{
    int index, i;
    printf("<IOFormat>\n");
    printf("<formatID>%d</formatID>\n", ioformat->format_id);
    printf("<formatName>%s</formatName>\n", ioformat->body->format_name);
    printf("<recordLength>%d</recordLength>\n", ioformat->body->record_length);
    printf("<fieldCount>%d</fieldCount>\n", ioformat->body->field_count);
    printf("<byteReversal>%d</byteReversal>\n", ioformat->body->byte_reversal);
    printf("<pointerSize>%d</pointerSize>\n", ioformat->body->pointer_size);
    printf("<IOversion>%d</IOversion>\n", ioformat->body->IOversion);
    printf("<serverID>");
    for (i = 0; i < ioformat->body->server_ID.length; i++) {
	printf("%02x", ((unsigned char *) ioformat->body->server_ID.value)[i]);
    }
    printf("</serverID>\n");
    for (index = 0; index < ioformat->body->field_count; index++) {
	printf("<IOField>\n");
	printf("<fieldName>%s</fieldName>\n<fieldType>%s</fieldType>\n<fieldSize>%d</fieldSize>\n<fieldOffset>%d</fieldOffset>\n",
	       ioformat->body->field_list[index].field_name,
	       ioformat->body->field_list[index].field_type,
	       ioformat->body->field_list[index].field_size,
	       ioformat->body->field_list[index].field_offset);
    }
    dump_IOConversion_as_XML(ioformat->conversion);

}

static int
dump_IOfield ARGS((IOFile iofile, IOFormat ioformat, int index, void *data,
		   void *string_base, int encode, int verbose));

extern void
dump_IOfield_as_XML ARGS((IOFile iofile, IOFormat ioformat, int index, void *data,
			  void *string_base, int encode, int verbose));

extern void
dump_raw_IOrecord(iofile, ioformat, data)
IOFile iofile;
IOFormat ioformat;
void *data;
{
    int index;
    if (IOdumpVerbose)
	printf("Record type %s :", ioformat->body->format_name);
    for (index = 0; index < ioformat->body->field_count; index++) {
	(void) dump_IOfield(iofile, ioformat, index, data, data, 1,
			    IOdumpVerbose);
    }
    printf("\n");
}

extern int
dump_limited_unencoded_IOrecord(iofile, ioformat, data, character_limit)
IOFile iofile;
IOFormat ioformat;
void *data;
int character_limit;
{
    int index;
    int char_count = 0;
    for (index = 0; index < ioformat->body->field_count; index++) {
	if ((character_limit > 0) && (char_count > character_limit))
	    break;
	char_count += dump_IOfield(iofile, ioformat, index, data, data, 0,
				   1);
    }
    printf("\n");
    return ((character_limit > 0) && (char_count > character_limit));
}

extern int
dump_limited_encoded_IOrecord(iofile, ioformat, data, character_limit)
IOFile iofile;
IOFormat ioformat;
void *data;
int character_limit;
{
    int index;
    int char_count = 0;
    for (index = 0; index < ioformat->body->field_count; index++) {
	if ((character_limit > 0) && (char_count > character_limit))
	    break;
	char_count += dump_IOfield(iofile, ioformat, index, data, data, 1,
				   1);
    }
    printf("\n");
    return ((character_limit > 0) && (char_count > character_limit));
}

extern void
dump_unencoded_IOrecord(iofile, ioformat, data)
IOFile iofile;
IOFormat ioformat;
void *data;
{
    int index;
    if (IOdumpVerbose)
	printf("Record type %s :", ioformat->body->format_name);
    for (index = 0; index < ioformat->body->field_count; index++) {
	(void) dump_IOfield(iofile, ioformat, index, data, data, 0, 1);
    }
    printf("\n");
}

extern void
dump_unencoded_IOrecord_as_XML(iofile, ioformat, data)
IOFile iofile;
IOFormat ioformat;
void *data;
{
    int index;
    if (IOhas_XML_info(ioformat)) {
	dump_XML_record(ioformat, data, 0);
	return;
    }
    printf("<%s>\n", ioformat->body->format_name);
    for (index = 0; index < ioformat->body->field_count; index++) {
	dump_IOfield_as_XML(iofile, ioformat, index, data, data, 0, 1);
    }
    printf("</%s>\n", ioformat->body->format_name);
}


/* big enough for most single values */
#define MAX_VALUE_SIZE 512

extern int
sdump_value(str, field_type, field_size, field_offset, iofile, data,
	    string_base, byte_reversal, encode)
char *str;
const char *field_type;
int field_size;
int field_offset;
IOFile iofile;
void *data;
void *string_base;
int byte_reversal;
int encode;
{
    IOgetFieldStruct descr;
    descr.offset = field_offset;
    descr.size = field_size;
    descr.data_type = str_to_data_type(field_type);
    descr.byte_swap = byte_reversal;

    if (descr.data_type == integer_type) {
	if (field_size <= sizeof(long)) {
	    long tmp = get_IOlong(&descr, data);
	    sprintf(str, "%ld", tmp);
	} else if (field_size == 2 * sizeof(long) && field_size == 8) {
	    unsigned long low_long;
	    long high_long;
	    get_IOlong8(&descr, data, &low_long, &high_long);
	    if (high_long == 0) {
		sprintf(str, "%ld", low_long);
	    } else {
		sprintf(str, "0x%lx%08lx", high_long, low_long);
	    }
	} else if (field_size > sizeof(long)) {
	    sprintf(str, "+long int+");
	} else {
	    sprintf(str, "+int size %d+", field_size);
	}
    } else if (descr.data_type == unsigned_type) {
	if (field_size <= sizeof(unsigned long)) {
	    unsigned long tmp = get_IOulong(&descr, data);
	    sprintf(str, "%lu", tmp);
	} else if (field_size == 2 * sizeof(long) && field_size == 8) {
	    unsigned long low_long, high_long;
	    get_IOulong8(&descr, data, &low_long, &high_long);
	    if (high_long == 0) {
		sprintf(str, "%lu", low_long);
	    } else {
		sprintf(str, "0x%lx%08lx", high_long, low_long);
	    }
	} else if (field_size > sizeof(long)) {
	    sprintf(str, "+ulong int+");
	} else {
	    sprintf(str, "+uint size %u+", field_size);
	}
    } else if (descr.data_type == enumeration_type) {
	sprintf(str, "%u", *(int *) ((char *) data + field_offset));
    } else if (descr.data_type == boolean_type) {
	if (*(int *) ((char *) data + field_offset) == 0) {
	    strcpy(str, "false");
	} else {
	    strcpy(str, "true");
	}
    } else if (descr.data_type == float_type) {
	if (field_size == sizeof(float)) {
	    float tmp = get_IOfloat(&descr, data);
	    sprintf(str, "%g", tmp);
	} else if (field_size == sizeof(double)) {
	    double tmp = get_IOdouble(&descr, data);
	    sprintf(str, "%g", tmp);
#if SIZEOF_LONG_DOUBLE != 0 && SIZEOF_LONG_DOUBLE != SIZEOF_DOUBLE
	} else if (field_size == sizeof(long double)) {
	    long double tmp;
	    memcpy(&tmp, (float *) ((char *) data + field_offset),
		   sizeof(double));
	    sprintf(str, "%Lg", tmp);
#endif
	} else {
	    if (field_size < sizeof(float)) {
		sprintf(str, "+tiny float+");
	    } else if (field_size > sizeof(double)) {
		sprintf(str, "+big float+");
	    } else {
		sprintf(str, "+float size %u+", field_size);
	    }
	}
    } else if (descr.data_type == char_type) {
	sprintf(str, "%c", *(char *) ((char *) data + field_offset));
    } else if (descr.data_type == string_type) {
	char *tmp_str = (char *) get_IOaddr(&descr, data, string_base, encode);
	if (tmp_str == 0) {
	    sprintf(str, "\"\"");
	} else {
	    if (strlen(tmp_str) + 3 < MAX_VALUE_SIZE) {
		sprintf(str, "\"%s\"", tmp_str);
	    } else {
		str[0] = '"';
		strncpy(&str[1], tmp_str, MAX_VALUE_SIZE - 10);
		str[MAX_VALUE_SIZE - 10] = 0;
		strcat(str, " ... \"");
	    }
	}
    } else if (strcmp(field_type, "R3vector") == 0) {
	sprintf(str, "(%g, %g, %g)", *(double *) ((char *) data + field_offset),
	     *(double *) ((char *) data + field_offset + sizeof(double)),
	*(double *) ((char *) data + field_offset + 2 * sizeof(double)));
    } else {
	return 0;
    }
    return 1;
}

static int
dump_value(field_type, field_size, field_offset, iofile, data, string_base,
	   byte_reversal, encode, verbose)
char *field_type;
int field_size;
int field_offset;
IOFile iofile;
void *data;
void *string_base;
int byte_reversal;
int encode;
int verbose;
{
    char buf[MAX_VALUE_SIZE];	/* surely big enough for a single value? */
    IOFormat ioformat;
    int char_count = 0;
    if (sdump_value(buf, field_type, field_size, field_offset, iofile, data,
		    string_base, byte_reversal, encode)) {
	printf("%s", buf);
	char_count = strlen(buf);
    } else {
	char *typ = base_data_type(field_type);
	if ((ioformat = get_IOformat_by_name_IOcontext((IOContext) iofile, typ)) != NULL) {
	    int index;
	    printf("{ ");
	    for (index = 0; index < ioformat->body->field_count; index++) {
		char_count +=
		    dump_IOfield(iofile, ioformat, index,
				 (void *) ((char *) data + field_offset),
				 string_base, encode, verbose);
	    }
	    printf("} ");
	    char_count += 4;
	} else {
	    printf("Unknown type");
	}
	free(typ);
    }
    return char_count;
}

static int
dump_IOfield(iofile, ioformat, field, data, string_base, encode, verbose)
IOFile iofile;
IOFormat ioformat;
int field;
void *data;
void *string_base;
int encode;
int verbose;
{
    IOFieldList iofield = &ioformat->body->field_list[field];
    IOVarInfoList iovar = &ioformat->body->var_list[field];
    int field_offset = iofield->field_offset;
    int field_size = iofield->field_size;
    const char *field_type = iofield->field_type;
    const char *field_name = iofield->field_name;
    int char_count = 0;

    char *left_paren = NULL;
    if (verbose) {
	printf("%s = ", field_name);
	char_count += strlen(field_name) + 3;
    }
    if ((left_paren = strchr(field_type, '[')) == NULL) {
	char_count +=
	    dump_value(field_type, field_size, field_offset, iofile, data,
	    string_base, ioformat->body->byte_reversal, encode, verbose);
    } else if (strchr(left_paren + 1, '[') == NULL) {
	/* single dimen array */
	int dimension = 0;
	char sub_type[64];
	int sub_field_size;
	int offset = iofield->field_offset;
	printf("{ ");
	*left_paren = 0;
	strcpy(sub_type, field_type);
	*left_paren = '[';
	if (sscanf(left_paren + 1, "%d]", &dimension) != 1) {
	    if (iovar->var_array != TRUE) {
		fprintf(stderr, "Couldn't parse array size in \"%s\"\n",
			field_type);
		return char_count;
	    } else {
		IOgetFieldStruct descr;
		long tmp_offset;
		descr.offset = iofield->field_offset;
		descr.size = ioformat->body->pointer_size;
		descr.data_type = integer_type;
		descr.byte_swap = ioformat->body->byte_reversal;
		dimension = get_IOlong(iovar->control_field, data);
		tmp_offset = get_IOlong(&descr, data);
		if (encode) {
		    data = (char *) string_base + tmp_offset;
		} else {
		    data = (void *) tmp_offset;
		}
		offset = 0;
	    }
	    sub_field_size = iofield->field_size;
	} else {
	    /* normal internal array */
	    if (ioformat->body->IOversion <= 3) {
		sub_field_size = iofield->field_size / dimension;
	    } else {
		sub_field_size = iofield->field_size;
	    }
	}
	for (; dimension > 0; dimension--) {
	    char_count +=
		dump_value(sub_type, sub_field_size, offset, iofile, data,
		      string_base, ioformat->body->byte_reversal, encode,
			   verbose);
	    offset += sub_field_size;
	    if (dimension != 1)
		printf(", ");
	}
	printf("}");
    } else {
	/* double dimen array */
	int dimension1 = 0;
	int dimension2 = 0;
	char sub_type[64];
	int sub_field_size, offset = iofield->field_offset;
	printf("{ ");
	*left_paren = 0;
	strcpy(sub_type, field_type);
	if (sscanf(left_paren + 1, "%d][%d]", &dimension1, &dimension2) != 2) {
	    *left_paren = '[';
	    fprintf(stderr, "Couldn't parse array size in \"%s\"\n",
		    field_type);
	    return char_count;
	}
	*left_paren = '[';
	if (ioformat->body->IOversion <= 3) {
	    sub_field_size = iofield->field_size / (dimension1 * dimension2);
	} else {
	    sub_field_size = iofield->field_size;
	}
	for (; dimension2 > 0; dimension2--) {
	    int i = 0;
	    printf("\n\t{ ");
	    for (; i < dimension1; i++) {
		char_count +=
		    dump_value(sub_type, sub_field_size, offset, iofile,
			data, string_base, ioformat->body->byte_reversal,
			       encode, verbose);
		offset += sub_field_size;
		if (i != dimension1 - 1)
		    printf(", ");
	    }
	    printf("}");
	}
	printf("}");
    }
    if (verbose)
	printf("; ");
    else
	printf(" ");

    return char_count;
}


extern int
sdump_value_as_XML(str, field_type, field_size, field_offset, iofile, data,
		   string_base, byte_reversal, encode)
char *str;
char *field_type;
int field_size;
int field_offset;
IOFile iofile;
void *data;
void *string_base;
int byte_reversal;
int encode;
{
    IOgetFieldStruct descr;
    descr.offset = field_offset;
    descr.size = field_size;
    descr.data_type = str_to_data_type(field_type);
    descr.byte_swap = byte_reversal;

    if (descr.data_type == integer_type) {
	if (field_size <= sizeof(long)) {
	    long tmp = get_IOlong(&descr, data);
	    sprintf(str, "%ld", tmp);
	} else if (field_size == 2 * sizeof(long) && field_size == 8) {
	    unsigned long low_long;
	    long high_long;
	    get_IOlong8(&descr, data, &low_long, &high_long);
	    if (high_long == 0) {
		sprintf(str, "%ld", low_long);
	    } else {
		sprintf(str, "0x%lx%08lx", high_long, low_long);
	    }
	} else if (field_size > sizeof(long)) {
	    sprintf(str, "<scalar type=\"long\" />");
	} else {
	    sprintf(str, "<scalar type=\"int\" size=\"%d\" />", field_size);
	}
    } else if (descr.data_type == unsigned_type) {
	if (field_size <= sizeof(unsigned long)) {
	    unsigned long tmp = get_IOulong(&descr, data);
	    sprintf(str, "%lu", tmp);
	} else if (field_size == 2 * sizeof(long) && field_size == 8) {
	    unsigned long low_long, high_long;
	    get_IOulong8(&descr, data, &low_long, &high_long);
	    if (high_long == 0) {
		sprintf(str, "%lu", low_long);
	    } else {
		sprintf(str, "0x%lx%08lx", high_long, low_long);
	    }
	} else if (field_size > sizeof(long)) {
	    sprintf(str, "<scalar type=\"unsignedLong\" />");
	} else {
	    sprintf(str, "<scalar type=\"unsignedInt\" size=\"%d\" />", field_size);
	}
    } else if (descr.data_type == enumeration_type) {
	sprintf(str, "%u", *(int *) ((char *) data + field_offset));
    } else if (descr.data_type == boolean_type) {
	if (*(int *) ((char *) data + field_offset) == 0) {
	    strcpy(str, "false");
	} else {
	    strcpy(str, "true");
	}
    } else if (descr.data_type == float_type) {
	if (field_size == sizeof(float)) {
	    float tmp = get_IOfloat(&descr, data);
	    sprintf(str, "%g", tmp);
	} else if (field_size == sizeof(double)) {
	    double tmp = get_IOdouble(&descr, data);
	    sprintf(str, "%g", tmp);
#if SIZEOF_LONG_DOUBLE != 0 && SIZEOF_LONG_DOUBLE != SIZEOF_DOUBLE
	} else if (field_size == sizeof(long double)) {
	    long double tmp;
	    memcpy(&tmp, (float *) ((char *) data + field_offset),
		   sizeof(double));
	    sprintf(str, "%Lg", tmp);
#endif
	} else {
	    if (field_size < sizeof(float)) {
		sprintf(str, "<scalar type=\"small-float\" />");
	    } else if (field_size > sizeof(double)) {
		sprintf(str, "<scalar type=\"big-float\" />");
	    } else {
		sprintf(str, "<scalar type=\"float\" size=\"%u\" />", field_size);
	    }
	}
    } else if (descr.data_type == char_type) {
	sprintf(str, "%c", *(char *) ((char *) data + field_offset));
    } else if (descr.data_type == string_type) {
	char *tmp_str = (char *) get_IOaddr(&descr, data, string_base, encode);
	if (tmp_str != 0) {
	    if (strlen(tmp_str) + 3 < MAX_VALUE_SIZE) {
		sprintf(str, "%s", tmp_str);
	    } else {
		sprintf(str, "%s", tmp_str);
		/* 
		 * str[0] = '"'; strncpy(&str[1], tmp_str,
		 * MAX_VALUE_SIZE-10); str[MAX_VALUE_SIZE-10] = 0;
		 * strcat(str, " ... \""); */
	    }
	}
    } else {
	return 0;
    }
    return 1;
}

static void
byte_swap(data, size)
char *data;
int size;
{
    int i;
    assert((size % 2) == 0);
    for (i = 0; i < size / 2; i++) {
	char tmp = data[i];
	data[i] = data[size - i - 1];
	data[size - i - 1] = tmp;
    }
}

/* 
 * ** Code below this point relates to operations to and from the format
 * server  */

static int
put_serverAtomicInt(fd, file_int_ptr, iofile)
void *fd;
FILE_INT *file_int_ptr;
IOFile iofile;
{
#if SIZEOF_INT == 4
    int tmp_value = *file_int_ptr;
    int junk_errno;
    char *junk_result_str;
    if (os_server_write_func(fd, &tmp_value, 4, &junk_errno, &junk_result_str) != 4)
	return 0;
#else
    Baaad shit;
#endif
    return 1;
}

static int
get_serverAtomicInt(fd, file_int_ptr, byte_reversal)
void *fd;
FILE_INT *file_int_ptr;
int byte_reversal;
{
#if SIZEOF_INT == 4
    int tmp_value;
    int junk_errno;
    char *junk_result_str;
    if (os_server_read_func(fd, &tmp_value, 4, &junk_errno, &junk_result_str) != 4)
	return 0;
#else
    Baaad shit;
#endif
    if (byte_reversal)
	byte_swap((char *) &tmp_value, 4);
    *file_int_ptr = tmp_value;
    return 1;
}

static int server_read_failure = 0;

extern int
serverAtomicRead(fd, buffer, length)
void *fd;
void *buffer;
int length;
{
    char *junk_result_str = NULL;
    int junk_errno;
    int ret = os_server_read_func(fd, buffer, length, &junk_errno,
				  &junk_result_str);
    if (ret != length) {
	if (format_server_verbose == 1) {
	    printf("server read error, return is %d, length %d, errno %d\n",
		   ret, length, junk_errno);
	    if (junk_result_str != NULL) {
		printf("result_string is %s\n", junk_result_str);
	    }
	}
	server_read_failure++;
    }
    return ret;
}

static int
serverAtomicWrite(fd, buffer, length)
void *fd;
void *buffer;
int length;
{
    char *junk_result_str;
    int junk_errno;
    return os_server_write_func(fd, buffer, length, &junk_errno,
				&junk_result_str);
}


static void
provisional_use_warning(fd)
int fd;
{
    static int warned = 0;

    if (warned)
	return;
    warned++;

    fprintf(stderr, "The contacted format_server daemon allows only temporary use.\n");
    fprintf(stderr, " See http://www.cc.gatech.edu/systems/projects/MOSS/servers.html for more info.\n");
}

static void
dump_server_error(char *string, IOFile context)
{
}

static int
old_server_register_format(iofile, ioformat)
IOFile iofile;
IOFormat ioformat;
{
    int tries = 0;
    char format_registration_char = 'F';
    char return_char = 0;

    if (iofile->master_context != NULL) {
	return old_server_register_format((IOFile) iofile->master_context,
					  ioformat);
    }
  retry:
    if (tries++ > 2) {
	dump_server_error("Failed to contact format server\n", iofile);
    }
    if (establish_server_connection_ptr == NULL) {
	char *addr_str;
	char var_str[60];

	sprintf(var_str, "Server_conn_func_addr_%lx", (long) getpid());
	if ((addr_str = getenv(var_str)) == NULL) {
	    printf("Something really outrageous has happened, please tell Greg\n");
	    exit(1);
	} else {
	    sscanf(addr_str, "%lx", (long *) &establish_server_connection_ptr);
	}
    }
    if (establish_server_connection_ptr(iofile, 1) == 0) {
	dump_server_error("Failed to contact format server\n", iofile);
	return 0;
    }
    if (serverAtomicWrite(iofile->server_fd, &format_registration_char, 1) != 1)
	goto retry;

    if (!write_format_to_fd(iofile->server_fd, ioformat))
	goto retry;

    if (serverAtomicRead(iofile->server_fd, &return_char, 1) != 1) {
	dump_server_error("Read failed for format server.  Out of domain?\n",
			  iofile);
    }
    if (return_char == 'P') {
	provisional_use_warning((int) (long) iofile->server_fd);
	serverAtomicRead(iofile->server_fd, &return_char, 1);
    }
    assert(return_char == 'I');
    ioformat->body->server_ID.length = 8;
    ioformat->body->server_ID.value = malloc(ioformat->body->server_ID.length);
    if (serverAtomicRead(iofile->server_fd, (void *) ioformat->body->server_ID.value,
			 ioformat->body->server_ID.length)
	!= ioformat->body->server_ID.length) {
	goto retry;
    }
    if (ioformat->iofile != iofile) {
	/* registering in a master, copy it down */
	
	copy_ioformat_to_iofile(ioformat, iofile);
    }
    return 1;
}

static int
server_register_format(iofile, ioformat)
IOFile iofile;
IOFormat ioformat;
{
    int tries = 0;

    if (iofile->master_context != NULL) {
	return server_register_format((IOFile) iofile->master_context,
				      ioformat);
    }
  retry:
    if (tries++ > 2) {
	dump_server_error("Failed to contact format server\n", iofile);
    }
    if (establish_server_connection_ptr == NULL) {
	char *addr_str;
	char var_str[60];

	sprintf(var_str, "Server_conn_func_addr_%lx", (long) getpid());
	if ((addr_str = getenv(var_str)) == NULL) {
	    printf("Something really outrageous has happened, please tell Greg\n");
	    exit(1);
	} else {
	    sscanf(addr_str, "%lx", (long *) &establish_server_connection_ptr);
	}
    }
    if (establish_server_connection_ptr(iofile, 1) == 0) {
	dump_server_error("Failed to contact format server\n", iofile);
	return 0;
    } {
	struct {
	    char reg[2];
	    unsigned short len;
	    } tmp = {{'f', 1}, 0 };	/* format reg, version 1 */
	int ret;
	int errno;
	char *errstr;
	char ret_info[2];

	format_rep rep = ioformat->body->server_format_rep;

	tmp.len = htons(rep->format_rep_length);
	ret = os_server_write_func(iofile->server_fd, &tmp, sizeof(tmp),
				   &errno, &errstr);
	if (ret != sizeof(tmp))
	    goto retry;
	ret = os_server_write_func(iofile->server_fd, 
				   (char*) rep + sizeof(tmp.len),
				   rep->format_rep_length - sizeof(tmp.len),
				   &errno, &errstr);
	if (ret != rep->format_rep_length - sizeof(tmp.len))
	    goto retry;

	if (serverAtomicRead(iofile->server_fd, &ret_info[0], 2) != 2) {
	    dump_server_error("Read failed for format server.  Out of domain?\n",
			      iofile);
	    return 0;
	}
	if (ret_info[0] == 'P') {
	    provisional_use_warning((int) (long) iofile->server_fd);
	} else {
	    if (ret_info[0] != 'I') {
		printf("Bad character from format server %c\n", ret_info[0]);
		return 0;
	    }
	}
	ioformat->body->server_ID.length = ret_info[1];
	ioformat->body->server_ID.value = malloc(ioformat->body->server_ID.length);
	if (serverAtomicRead(iofile->server_fd, (void *) ioformat->body->server_ID.value,
			     ioformat->body->server_ID.length)
	    != ioformat->body->server_ID.length) {
	    goto retry;
	}
    }

    if (ioformat->iofile != iofile) {
	/* registering in a master, copy it down */
	copy_ioformat_to_iofile(ioformat, iofile);
    }
    return 1;
}

static const char xchars[] = "0123456789abcdef";
#define nibble2hex(val) (xchars[val & 0x0f])

static char *
stringify_field_type(type, base_format, buffer, size)
const char *type;
IOFormat base_format;
char *buffer;
int size;
{
    char *index_start;
    unsigned char *server_ID;

    if (base_format == NULL) {
	strcpy(buffer, type);
	return (char*)type;
    }
    if (strchr(type, ':') != NULL) {
	strcpy(buffer, type);
	return (char*)type;		/* already stringified */
    }
    server_ID = (unsigned char *) base_format->body->server_ID.value;

    index_start = strchr(type, '[');
    assert(strlen(type) + 2 * base_format->body->server_ID.length + 2 <= (unsigned int) size);
    if (base_format->body->server_ID.length == 8) {
	if (index_start == NULL) {
	    sprintf(buffer, "%s:%02x%02x%02x%02x:%02x%02x%02x%02x", type,
		  server_ID[0], server_ID[1], server_ID[2], server_ID[3],
		 server_ID[4], server_ID[5], server_ID[6], server_ID[7]);
	} else {
	    *index_start = 0;
	    sprintf(buffer, "%s:%02x%02x%02x%02x:%02x%02x%02x%02x[%s", type,
		  server_ID[0], server_ID[1], server_ID[2], server_ID[3],
		  server_ID[4], server_ID[5], server_ID[6], server_ID[7],
		    (index_start + 1));
	    *index_start = '[';
	}
    } else {
	int i;
	char *end;
	strcpy(buffer, type);
	index_start = strchr(buffer, '[');
	if (index_start != NULL)
	    *index_start = 0;
	strcat(buffer, ":");
	end = buffer + strlen(buffer);
	for (i = 0; i < base_format->body->server_ID.length; i++) {
	    end[2 * i] = nibble2hex(base_format->body->server_ID.value[i] >> 4);
	    end[2 * i + 1] = nibble2hex(base_format->body->server_ID.value[i]);
	}
	end[2 * i] = 0;
	if (index_start != NULL) {
	    index_start = strchr(type, '[');
	    strcat(buffer, index_start);
	}
    }
    return buffer;
}

extern int
global_name_eq(format1, format2)
IOFormat format1;
IOFormat format2;
{
    if (format1->body->server_ID.length != format2->body->server_ID.length)
	return 0;
    return !memcmp(format1->body->server_ID.value, format2->body->server_ID.value,
		   format1->body->server_ID.length);
}

extern
char *
global_name_of_IOformat(format)
IOFormat format;
{
    int size = strlen(format->body->format_name) + 3 +
    2 * format->body->server_ID.length;
    char *buffer = malloc(size);
    return stringify_field_type(format->body->format_name, format, buffer, size);
}

static
unsigned char
hex2byte(char c)
{
    if (isdigit((int) c))
	return (unsigned char) (c - '0');
    else if (islower((int) c))
	return (unsigned char) (10 + c - 'a');
    else
	return (unsigned char) (10 + c - 'A');
}

static void
unstringify_field_type(type, buffer, size)
const char *type;
char *buffer;
int size;
{
    char *last_colon = strrchr(type, ':');
    char *second_last_colon;
    char *server_ID = (char *) buffer;
    int int_server_ID[8];
    int i;

    assert(size > 8);
    *last_colon = 0;
    second_last_colon = strrchr(type, ':');
    *last_colon = ':';
    if (second_last_colon != NULL) {
	assert(sscanf(second_last_colon, ":%02x%02x%02x%02x:%02x%02x%02x%02x",
		      &int_server_ID[0], &int_server_ID[1],
		      &int_server_ID[2], &int_server_ID[3],
		      &int_server_ID[4], &int_server_ID[5],
		      &int_server_ID[6], &int_server_ID[7]) == 8);
	for (i = 0; i < 8; i++) {
	    server_ID[i] = int_server_ID[i];
	}
    } else {
	/* must not be version 0 format ID */
	unsigned char byte;
	int count = 0;
	char *tmp = last_colon + 1;
	while ((*tmp != '[') && (*tmp != 0)) {
	    byte = (unsigned char) (hex2byte(tmp[0]) << 4);
	    byte |= hex2byte(tmp[1]);
	    server_ID[count] = byte;
	    count++;
	    tmp += 2;
	}
    }
}


extern int
write_format_to_fd(fd, ioformat)
void *fd;
IOFormat ioformat;
{
    FILE_INT name_length, field_count, record_length;
    int field;
    int result = 1;
    int record_byte_order;
    int pointer_size;
    int IOversion;

    if ((!ioformat) || (!ioformat->body->format_name)) {
	int zero = 0;
	result = put_serverAtomicInt(fd, &zero, NULL);
	return !result;
    }
    name_length = strlen(ioformat->body->format_name);
    field_count = ioformat->body->field_count;
    record_length = ioformat->body->record_length;
    pointer_size = ioformat->body->pointer_size;
    IOversion = ioformat->body->IOversion;
    if (ioformat->body->byte_reversal) {
	record_byte_order = OTHER_BYTE_ORDER;
    } else {
	record_byte_order = OUR_BYTE_ORDER;
    }
    result &= put_serverAtomicInt(fd, &name_length, NULL);
    result &= put_serverAtomicInt(fd, &field_count, NULL);
    result &= put_serverAtomicInt(fd, &record_length, NULL);
    result &= put_serverAtomicInt(fd, &record_byte_order, NULL);
    result &= put_serverAtomicInt(fd, &pointer_size, NULL);
    result &= put_serverAtomicInt(fd, &IOversion, NULL);
    if (!result)
	return 0;
    if (serverAtomicWrite(fd, ioformat->body->format_name, name_length + 1) !=
	name_length + 1)
	return 0;

    for (field = 0; field < ioformat->body->field_count; field++) {
	FILE_INT field_name_len, field_type_len, field_size, field_offset;
	IOField *iofield = &(ioformat->body->field_list[field]);
	char field_type_buffer[512];
	char *ftype;
	IOFormat subformat = NULL;
	if (ioformat->body->var_list != NULL) {
	    subformat = ioformat->field_subformats[field];
	}
	memset(field_type_buffer, 0, sizeof(field_type_buffer));
	ftype = stringify_field_type(iofield->field_type, subformat,
				     field_type_buffer,
				     (int) sizeof(field_type_buffer));
	field_name_len = strlen(iofield->field_name);
	field_type_len = strlen(ftype);
	field_size = iofield->field_size;
	field_offset = iofield->field_offset;
	result &= put_serverAtomicInt(fd, &field_name_len, NULL);
	result &= put_serverAtomicInt(fd, &field_type_len, NULL);
	result &= put_serverAtomicInt(fd, &field_size, NULL);
	result &= put_serverAtomicInt(fd, &field_offset, NULL);
	if (!result)
	    return 0;
	if (serverAtomicWrite(fd, iofield->field_name, field_name_len + 1) !=
	    field_name_len + 1)
	    return 0;
	if (serverAtomicWrite(fd, ftype, field_type_len + 1) !=
	    field_type_len + 1)
	    return 0;
    }
    return 1;
}

extern IOFormat
expand_ioformat_from_rep(rep)
format_rep rep;
{
    IOFormat ioformat = new_IOFormat();
    struct _field_wire_format *fields;
    int field;
    UINT2 tmp;
    INT4 tmp2;
    int byte_reversal = ((rep->record_byte_order & 0x1) != OUR_BYTE_ORDER);

    ioformat->body->server_format_rep = rep;
    tmp = rep->name_offset;
    if (byte_reversal) byte_swap((char*)&tmp, 2);
    ioformat->body->format_name = io_malloc(strlen((char *) rep + tmp) + 1);
    strcpy(ioformat->body->format_name, (char *) rep + tmp);
    tmp = rep->field_count;
    if (byte_reversal) byte_swap((char*)&tmp, 2);
    ioformat->body->field_count = tmp;
    ioformat->body->variant = FALSE;
    tmp2 = rep->record_length;
    if (byte_reversal) byte_swap((char*)&tmp2, 4);
    ioformat->body->record_length = tmp2;
    ioformat->native_format = 0;
    ioformat->conversion = NULL;
    ioformat->body->byte_reversal = ((rep->record_byte_order & 0x1) != OUR_BYTE_ORDER);
    ioformat->body->pointer_size = rep->pointer_size;
    ioformat->body->IOversion = CURRENT_IO_VERSION;
    ioformat->body->field_list = (IOFieldList) io_malloc(sizeof(IOField) *
				      (ioformat->body->field_count + 1));
    ioformat->body->var_list = NULL;

    if (rep->server_rep_version == 0) {
	fields = (struct _field_wire_format *) 
	    ((char*)rep + sizeof(struct _format_wire_format_0));
    } else {
	fields = (struct _field_wire_format *) 
	    ((char*) rep + rep->header_size);
    }
    for (field = 0; field < ioformat->body->field_count; field++) {
	IOField *iofield = &(ioformat->body->field_list[field]);
	struct _field_wire_format *wire = &fields[field];
	tmp = wire->field_name_offset;
	if (byte_reversal) byte_swap((char*)&tmp, 2);
	iofield->field_name = io_malloc(strlen((char *) rep + tmp) + 1);
	strcpy((char*)iofield->field_name, (char *) rep + tmp);
	tmp = wire->field_type_offset;
	if (byte_reversal) byte_swap((char*)&tmp, 2);
	iofield->field_type = io_malloc(strlen((char *) rep + tmp) + 1);
	strcpy((char*)iofield->field_type, (char *) rep + tmp);
	iofield->field_size = wire->field_size;
	if (byte_reversal) byte_swap((char*)&iofield->field_size, 4);
	iofield->field_offset = wire->field_offset;
	if (byte_reversal) byte_swap((char*)&iofield->field_offset, 4);
    }
    ioformat->body->field_list[ioformat->body->field_count].field_size = 0;
    ioformat->body->field_list[ioformat->body->field_count].field_offset = 0;
    ioformat->body->field_list[ioformat->body->field_count].field_name = NULL;
    ioformat->body->field_list[ioformat->body->field_count].field_type = NULL;
    if (rep->server_rep_version == 1) {
	struct _format_wire_format_1 *rep1 = 
	    (struct _format_wire_format_1 *)rep;
	struct _opt_info_wire_format tmp_info;
	int offset, info_count = 0;

	tmp = rep1->opt_info_offset;
	if (byte_reversal) byte_swap((char*)&tmp, 2);

	if (tmp != 0) {
	    offset = tmp;
	    ioformat->body->opt_info = malloc(sizeof(IOOptInfo));
	    do {
		memcpy(&tmp_info, offset + (char*) rep, sizeof(tmp_info));
		if (tmp_info.info_type != 0) {
		    ioformat->body->opt_info = 
			realloc(ioformat->body->opt_info,
				sizeof(IOOptInfo) * (info_count + 2));
		    tmp2 = tmp_info.info_type;
		    if (byte_reversal) byte_swap((char*)&tmp2, 4);
		    ioformat->body->opt_info[info_count].info_type = tmp2;
			
		    tmp2 = tmp_info.info_len;
		    if (byte_reversal) byte_swap((char*)&tmp2, 4);
		    ioformat->body->opt_info[info_count].info_len = tmp2;

		    tmp2 = tmp_info.info_offset;
		    if (byte_reversal) byte_swap((char*)&tmp2, 4);
		    ioformat->body->opt_info[info_count].info_block = 
			(char*)rep + tmp2;
		    info_count++;
		    offset += sizeof(tmp_info);
		}
	    } while (tmp_info.info_type != 0);
	    ioformat->body->opt_info[info_count].info_type = 0;
	    ioformat->body->opt_info[info_count].info_len = 0;
	    ioformat->body->opt_info[info_count].info_block = 0;
	}
    }
    return ioformat;
}

static IOFormat
old_read_format_from_fd(fd, byte_reversal)
void *fd;
int byte_reversal;
{
    FILE_INT name_length;
    FILE_INT field_count;
    FILE_INT record_length;
    IOFormat ioformat = new_IOFormat();
    IOVarInfoList new_var_list;
    int field;
    int record_byte_order;
    int pointer_size;
    int IOversion;

    get_serverAtomicInt(fd, &name_length, byte_reversal);
    if (name_length == 0) {
	free(ioformat);
	return NULL;
    }
    get_serverAtomicInt(fd, &field_count, byte_reversal);
    get_serverAtomicInt(fd, &record_length, byte_reversal);
    get_serverAtomicInt(fd, &record_byte_order, byte_reversal);
    get_serverAtomicInt(fd, &pointer_size, byte_reversal);
    get_serverAtomicInt(fd, &IOversion, byte_reversal);

    ioformat->body->format_name = io_malloc(name_length + 1);
    ioformat->body->field_count = field_count;
    ioformat->body->variant = FALSE;
    ioformat->body->record_length = record_length;
    ioformat->native_format = 0;
    ioformat->conversion = NULL;
    ioformat->body->byte_reversal = (record_byte_order != OUR_BYTE_ORDER);
    ioformat->body->pointer_size = pointer_size;
    ioformat->body->IOversion = IOversion;
    serverAtomicRead(fd, ioformat->body->format_name, name_length + 1);
    ioformat->body->field_list = (IOFieldList) io_malloc(sizeof(IOField) *
				      (ioformat->body->field_count + 1));
    new_var_list = (IOVarInfoList)
	io_malloc((size_t) sizeof(IOVarInfoStruct) * field_count);
    ioformat->body->var_list = new_var_list;
    ioformat->field_subformats = 
	(IOFormat *) io_malloc((size_t) sizeof(IOFormat) * field_count);

    for (field = 0; field < ioformat->body->field_count; field++) {
	FILE_INT field_name_len;
	FILE_INT field_type_len;
	FILE_INT field_size;
	FILE_INT field_offset;
	IOField *iofield = &(ioformat->body->field_list[field]);
	ioformat->body->var_list[field].control_field = NULL;
	ioformat->field_subformats[field] = NULL;
	get_serverAtomicInt(fd, &field_name_len, byte_reversal);
	get_serverAtomicInt(fd, &field_type_len, byte_reversal);
	get_serverAtomicInt(fd, &field_size, byte_reversal);
	get_serverAtomicInt(fd, &field_offset, byte_reversal);

	iofield->field_size = field_size;
	iofield->field_offset = field_offset;
	iofield->field_name = io_malloc(field_name_len + 1);
	iofield->field_type = io_malloc(field_type_len + 1);
	serverAtomicRead(fd, (void*)iofield->field_name, field_name_len + 1);
	serverAtomicRead(fd, (void*)iofield->field_type, field_type_len + 1);
	new_var_list[field].data_type =
	    array_str_to_data_type(iofield->field_type,
				   &new_var_list[field].dimen1,
				   &new_var_list[field].dimen2);

    }
    ioformat->body->field_list[field_count].field_size = 0;
    ioformat->body->field_list[field_count].field_offset = 0;
    ioformat->body->field_list[field_count].field_name = NULL;
    ioformat->body->field_list[field_count].field_type = NULL;
    ioformat->body->server_format_rep = build_server_format_rep(ioformat);
    return ioformat;
}

static IOFormat
read_format_from_fd(fd, byte_reversal)
void *fd;
int byte_reversal;
{
    FILE_INT name_length = 0;
    FILE_INT field_count;
    FILE_INT record_length;
    IOFormat ioformat = new_IOFormat();
    IOVarInfoList new_var_list;
    int field;
    int record_byte_order;
    int pointer_size;
    int IOversion;

    if (format_server_verbose == -1) {
	if (getenv("FORMAT_SERVER_VERBOSE") == NULL) {
	    format_server_verbose = 0;
	} else {
	    format_server_verbose = 1;
	}
    }
    if (get_serverAtomicInt(fd, &name_length, byte_reversal) != 1)
	goto fail;
    if (name_length == 0)
	goto fail;
    if (get_serverAtomicInt(fd, &field_count, byte_reversal) != 1)
	goto fail;
    if (get_serverAtomicInt(fd, &record_length, byte_reversal) != 1)
	goto fail;
    if (get_serverAtomicInt(fd, &record_byte_order, byte_reversal) != 1)
	goto fail;
    if (get_serverAtomicInt(fd, &pointer_size, byte_reversal) != 1)
	goto fail;
    if (get_serverAtomicInt(fd, &IOversion, byte_reversal) != 1)
	goto fail;

    ioformat->body->format_name = io_malloc(name_length + 1);
    ioformat->body->field_count = field_count;
    ioformat->body->variant = FALSE;
    ioformat->body->record_length = record_length;
    ioformat->native_format = 0;
    ioformat->conversion = NULL;
    ioformat->body->byte_reversal = (record_byte_order != OUR_BYTE_ORDER);
    ioformat->body->pointer_size = pointer_size;
    ioformat->body->IOversion = IOversion;
    if (serverAtomicRead(fd, ioformat->body->format_name, name_length + 1) != name_length + 1)
	goto fail;
    ioformat->body->field_list = (IOFieldList) io_malloc(sizeof(IOField) *
				      (ioformat->body->field_count + 1));
    new_var_list = (IOVarInfoList)
	io_malloc((size_t) sizeof(IOVarInfoStruct) * field_count);
    ioformat->body->var_list = new_var_list;
    ioformat->field_subformats = io_malloc(sizeof(IOFormat) * field_count);

    for (field = 0; field < ioformat->body->field_count; field++) {
	FILE_INT field_name_len;
	FILE_INT field_type_len;
	FILE_INT field_size;
	FILE_INT field_offset;
	IOField *iofield = &(ioformat->body->field_list[field]);
	ioformat->body->var_list[field].control_field = NULL;
	ioformat->field_subformats[field] = NULL;
	if (get_serverAtomicInt(fd, &field_name_len, byte_reversal) != 1)
	    goto fail;
	if (get_serverAtomicInt(fd, &field_type_len, byte_reversal) != 1)
	    goto fail;
	if (get_serverAtomicInt(fd, &field_size, byte_reversal) != 1)
	    goto fail;
	if (get_serverAtomicInt(fd, &field_offset, byte_reversal) != 1)
	    goto fail;

	iofield->field_size = field_size;
	iofield->field_offset = field_offset;
	iofield->field_name = io_malloc(field_name_len + 1);
	iofield->field_type = io_malloc(field_type_len + 1);
	if (serverAtomicRead(fd, (void*)iofield->field_name, 
			     field_name_len + 1) != field_name_len + 1)
	    goto fail;
	if (serverAtomicRead(fd, (void*)iofield->field_type, 
			     field_type_len + 1) != field_type_len + 1)
	    goto fail;
	new_var_list[field].data_type =
	    array_str_to_data_type(iofield->field_type,
				   &new_var_list[field].dimen1,
				   &new_var_list[field].dimen2);

    }
    ioformat->body->field_list[field_count].field_size = 0;
    ioformat->body->field_list[field_count].field_offset = 0;
    ioformat->body->field_list[field_count].field_name = NULL;
    ioformat->body->field_list[field_count].field_type = NULL;

    ioformat->body->server_format_rep = build_server_format_rep(ioformat);
    return ioformat;
  fail:
    free(ioformat);
    return NULL;
}

/* 
 * server_read_format() is used by the format server to read a format 
 * being registered by a client.
 */
extern IOFormat
server_read_format(fs, fsc)
format_server fs;
FSClient fsc;
{
    return old_read_format_from_fd(fsc->fd, fsc->byte_reversal);
}

static IOFormat
old_server_get_format(iocontext, buffer)
IOContext iocontext;
void *buffer;
{
    IOFile iofile = (IOFile) iocontext;
    IOFormat ioformat;
    IOFieldList field_list;
    char format_get_char = 'G';
    char return_char = 0;
    int id_size = 8;
    int field;
    int retry_count = 0;

  retry:
    if (retry_count > 3)
	return NULL;

    if (establish_server_connection_ptr == NULL) {
	char *addr_str;
	char var_str[60];

	sprintf(var_str, "Server_conn_func_addr_%lx", (long) getpid());
	if ((addr_str = getenv(var_str)) == NULL) {
	    printf("Something really outrageous has happened, please tell Greg\n");
	    exit(1);
	} else {
	    sscanf(addr_str, "%lx", (long *) &establish_server_connection_ptr);
	}
    }
    if (establish_server_connection_ptr(iofile, 1) == 0) {
	printf("Failed to contact format server\n");
	exit(1);
    }
    if (serverAtomicWrite(iofile->server_fd, &format_get_char, 1) != 1) {
	retry_count++;
	goto retry;
    }
    if (serverAtomicWrite(iofile->server_fd, buffer, id_size) != id_size) {
	retry_count++;
	goto retry;
    }
    if (serverAtomicRead(iofile->server_fd, &return_char, 1) != 1) {
	if (format_server_verbose == 1) {
	    printf("Retrying because of failed read\n");
	}
	retry_count++;
	goto retry;
    }
    if (return_char == 'P') {
	provisional_use_warning((int) (long) iofile->server_fd);
	if (serverAtomicRead(iofile->server_fd, &return_char, 1) != 1) {
	    if (format_server_verbose == 1) {
		printf("Retrying because of failed read\n");
	    }
	    retry_count++;
	    goto retry;
	}
    }
    assert(return_char == 'F');
    ioformat = read_format_from_fd(iofile->server_fd,
				   iofile->server_byte_reversal);

    if (iofile->reg_format_count == iofile->format_list_size) {
	expand_IOFile(iofile);
    }
    if (ioformat == NULL)
	return NULL;

    ioformat->format_id = iofile->reg_format_count++;
    iofile->format_list[ioformat->format_id] = ioformat;
    ioformat->iofile = iofile;
    ioformat->body->record_length = 0;
    ioformat->body->variant = FALSE;
    ioformat->body->server_ID.length = 8;
    ioformat->body->server_ID.value = malloc(8);
    memcpy(ioformat->body->server_ID.value, buffer, id_size);
    field_list = ioformat->body->field_list;
    for (field = 0; field < ioformat->body->field_count; field++) {
	int field_size = 0;
	if (get_var_array_control(field_list[field].field_type, field_list)
	    != -1) {
	    /* variant array, real_field_size is ioformat->pointer_size */
	    field_size = ioformat->body->pointer_size;
	} else {
	    int dimen1 = 1, dimen2 = 1;
	    IOdata_type base_type;
	    base_type = array_str_to_data_type(field_list[field].field_type,
					       &dimen1, &dimen2);
	    if ((base_type != unknown_type) &&
		(field_list[field].field_size > 16)) {
		fprintf(stderr, "Field size for field %s in format %s is large, check to see if it is valid.\n",
			field_list[field].field_name, ioformat->body->format_name);
	    }
	    field_size = field_list[field].field_size * dimen1 * dimen2;
	}
	assert(field_size > 0);
	ioformat->body->record_length = Max(ioformat->body->record_length,
					 field_list[field].field_offset +
					    field_size);
    }
    generate_var_list(ioformat);
    for (field = 0; field < ioformat->body->field_count; field++) {
	if (ioformat->body->var_list[field].string == TRUE) {
	    ioformat->body->variant = TRUE;
	} else {
	    char *base_type =
	    base_data_type(field_list[field].field_type);
	    IOFormat subformat = NULL;

	    /* if field is of another record format, fill that in */
	    if (str_to_data_type(base_type) == unknown_type) {
		subformat = get_IOformat_by_name_IOcontext((IOContext) iofile,
							   base_type);
		ioformat->field_subformats[field] = subformat;
	    }
	    if (ioformat->body->var_list[field].var_array == TRUE) {
		/* got a variant array */
		ioformat->body->variant = 1;
	    } else {
		/* if field is variant by its subformat being variant */
		if (subformat != NULL) {
		    ioformat->body->variant |= subformat->body->variant;
		}
	    }
	    free(base_type);
	}
    }
    ioformat->body->server_format_rep = build_server_format_rep(ioformat);
    return ioformat;
}

static void
add_format_to_iofile(iofile, ioformat, id_size, id_buffer)
IOFile iofile;
IOFormat ioformat;
int id_size;
void *id_buffer;
{
    IOFieldList field_list;
    int field;

    if (format_server_verbose) {
	printf("Entering format %s into context %lx ", 
	       ioformat->body->format_name,
	       (long)iofile);
	print_server_ID(id_buffer);
    }
    ioformat->iofile = iofile;
    ioformat->body->record_length = 0;
    ioformat->body->variant = FALSE;
    ioformat->body->server_ID.length = id_size;
    ioformat->body->server_ID.value = malloc(id_size);
    memcpy(ioformat->body->server_ID.value, id_buffer, id_size);
    field_list = ioformat->body->field_list;
    for (field = 0; field < ioformat->body->field_count; field++) {
	int field_size = 0;
	if (get_var_array_control(field_list[field].field_type, field_list)
	    != -1) {
	    /* variant array, real_field_size is ioformat->pointer_size */
	    field_size = ioformat->body->pointer_size;
	} else {
	    int dimen1 = 1, dimen2 = 1;
	    IOdata_type base_type;
	    base_type = array_str_to_data_type(field_list[field].field_type,
					       &dimen1, &dimen2);
	    if ((base_type != unknown_type) &&
		(field_list[field].field_size > 16)) {
		fprintf(stderr, "Field size for field %s in format %s is large, check to see if it is valid.\n",
			field_list[field].field_name, ioformat->body->format_name);
	    }
	    field_size = field_list[field].field_size * dimen1 * dimen2;
	}
	ioformat->body->record_length = Max(ioformat->body->record_length,
					 field_list[field].field_offset +
					    field_size);
    }
    generate_var_list(ioformat);
    for (field = 0; field < ioformat->body->field_count; field++) {
	if (ioformat->body->var_list[field].string == TRUE) {
	    ioformat->body->variant = TRUE;
	} else {
	    char *base_type =
	    base_data_type(field_list[field].field_type);
	    IOFormat subformat = NULL;

	    /* if field is of another record format, fill that in */
	    if (str_to_data_type(base_type) == unknown_type) {
		subformat = get_IOformat_by_name_IOcontext((IOContext) iofile,
							   base_type);
		ioformat->field_subformats[field] = subformat;
	    }
	    if (ioformat->body->var_list[field].var_array == TRUE) {
		/* got a variant array */
		ioformat->body->variant = 1;
	    } else {
		/* if field is variant by its subformat being variant */
		if (subformat != NULL) {
		    ioformat->body->variant |= subformat->body->variant;
		}
	    }
	    free(base_type);
	}
    }
    if (iofile->reg_format_count == iofile->format_list_size) {
	expand_IOFile(iofile);
    }
    ioformat->format_id = iofile->reg_format_count++;
    iofile->format_list[ioformat->format_id] = ioformat;
}

static IOFormat
self_server_get_format(iocontext, buffer, app_context)
IOContext iocontext;
void *buffer;
void *app_context;
{
    IOFile iofile = (IOFile) iocontext;
    IOFormat ioformat;
    int host_IP;
    int host_port;
    int format_length;
    char *format_rep_return;

    if (format_server_verbose == -1) {
	if (getenv("FORMAT_SERVER_VERBOSE") == NULL) {
	    format_server_verbose = 0;
	} else {
	    format_server_verbose = 1;
	}
    }
    if (iofile->master_context != NULL) {
	return self_server_get_format(iofile->master_context, buffer,
				      app_context);
    }
    host_IP = get_host_IP_format_ID(buffer);
    host_port = get_host_port_format_ID(buffer);
    format_length = ID_length[version_of_format_ID(buffer)];
    format_rep_return = iofile->server_callback(buffer, format_length, 
						host_IP,
						host_port, app_context,
						iofile->server_client_data);
    if (format_rep_return == NULL) {
	if (format_server_verbose == 1)
	    printf("Format server callback returned NULL\n");
	if ((ioformat = get_local_format_IOcontext(iocontext, buffer))) {
	    if (format_server_verbose == 1)
		printf("But we found it anyway....\n");
	    return ioformat;
	} else {
	    if (format_server_verbose == 1)
		printf("And we failed anyway....\n");
	    return NULL;
	}
    }
    ioformat = expand_ioformat_from_rep((format_rep)format_rep_return);
    if (ioformat == NULL)
	return NULL;
    add_format_to_iofile(iofile, ioformat, format_length, buffer);
    return ioformat;
}

static IOFormat
server_get_format(iocontext, buffer)
IOContext iocontext;
void *buffer;
{
    IOFile iofile = (IOFile) iocontext;
    IOFormat ioformat;
    int id_size = 8;
    int retry_count = 0;

  retry:
    if (retry_count > 3)
	return NULL;
    if (establish_server_connection_ptr == NULL) {
	char *addr_str;
	char var_str[60];

	sprintf(var_str, "Server_conn_func_addr_%lx", (long) getpid());
	if ((addr_str = getenv(var_str)) == NULL) {
	    printf("Something really outrageous has happened, please tell Greg\n");
	    exit(1);
	} else {
	    sscanf(addr_str, "%lx", (long *) &establish_server_connection_ptr);
	}
    }
    if (establish_server_connection_ptr(iofile, 1) == 0) {
	printf("Failed to contact format server\n");
	exit(1);
    } {
	char get[2] =
	{'g', 8};		/* format get, size */
	char block_version;
	UINT2 length;
	char return_char = 0;
	format_rep rep;

	if (version_of_format_ID(buffer) < sizeof(ID_length)) {
	    id_size = get[1] = ID_length[version_of_format_ID(buffer)];
	}
	if (serverAtomicWrite(iofile->server_fd, &get[0], 2) != 2) {
	    perror("write to Format server failed");
	    return NULL;
	}
	if (serverAtomicWrite(iofile->server_fd, buffer, id_size) != id_size) {
	    perror("write to Format server failed");
	    return NULL;
	}
	if (serverAtomicRead(iofile->server_fd, &return_char, 1) != 1) {
	    if (format_server_verbose == 1) {
		printf("Retrying because of failed read\n");
	    }
	    retry_count++;
	    goto retry;
	}
	if (return_char == 'P') {
	    provisional_use_warning((int) (long) iofile->server_fd);
	    if (serverAtomicRead(iofile->server_fd, &return_char, 1) != 1) {
		if (format_server_verbose == 1) {
		    printf("Retrying because of failed read\n");
		}
		retry_count++;
		goto retry;
	    }
	}
	if (return_char != 'f') {
	    if (format_server_verbose == 1) {
		printf("Retrying because of failed read\n");
	    }
	    retry_count++;
	    goto retry;
	}
	if (serverAtomicRead(iofile->server_fd, &block_version, 1) != 1) {
	    if (format_server_verbose == 1) {
		printf("Retrying because of failed read\n");
	    }
	    retry_count++;
	    goto retry;
	}
	if (block_version != 1) {
	    if (format_server_verbose == 1) {
		fprintf(stderr, "Unknown version \"%d\"in block registration\n", block_version);
	    }
	    return NULL;
	}
	if (serverAtomicRead(iofile->server_fd, &length, sizeof(length)) !=
	    sizeof(length)) {
	    if (format_server_verbose == 1) {
		printf("Retrying because of failed read\n");
	    }
	    retry_count++;
	    goto retry;
	}
	length = ntohs(length);
	if (length == 0) {
	    ioformat = NULL;
	} else {
	    rep = io_malloc(length);
	    rep->format_rep_length = length;
	    if (serverAtomicRead(iofile->server_fd, ((char *) rep) + sizeof(length),
		 length - sizeof(length)) != (length - sizeof(length))) {
		if (format_server_verbose == 1) {
		    printf("Retrying because of failed read\n");
		}
		retry_count++;
		goto retry;
	    }
	    ioformat = expand_ioformat_from_rep(rep);
	}
    }
    if (ioformat == NULL)
	return NULL;
    add_format_to_iofile(iofile, ioformat, id_size, buffer);
    return ioformat;
}

extern void
server_get_server_ID(fd, server_ID)
void *fd;
void *server_ID;
{
    int id_size = 8;

    serverAtomicRead(fd, server_ID, id_size);
}

extern void
print_server_ID(ID)
unsigned char *ID;
{
    int id_size = 8;
    int i;
    switch (version_of_format_ID(ID)) {
    case 0:
	for (i = 0; i < id_size; i++) {
	    printf("%2x", ID[i]);
	}
	break;
    case 1:{
	version_1_format_ID id1;
	memcpy(&id1, ID, sizeof(10));
	printf("<ID ver=%d, salt %d, port %d, IP_addr %x, formatID %d>\n",
	       id1.version, id1.salt, ntohs(id1.port),
	       ntohl(id1.IP_addr), ntohs(id1.format_identifier));
	    break;
	}
    default:
	printf("<Unknown format version %d\n",
	       *((unsigned char *) ID));
	break;
    }
}

extern void
print_format_ID(ioformat)
IOFormat ioformat;
{
    print_server_ID( (unsigned char *) ioformat->body->server_ID.value);
}

static int
get_host_IP_format_ID(format_ID)
void *format_ID;
{
    switch (version_of_format_ID(format_ID)) {
    case 0:
	return 0;
	break;
    case 1:{
	    version_1_format_ID *id1 = (version_1_format_ID *) format_ID;
	    int tmp;
	    memcpy(&tmp, &id1->IP_addr, 4);
	    return ntohl(tmp);
	    break;
	}
    default:
	printf("<Unknown format version %d\n",
	       *((unsigned char *) format_ID));
	break;
    }
    return 0;
}

static int
get_host_port_format_ID(format_ID)
void *format_ID;
{
    switch (version_of_format_ID(format_ID)) {
    case 0:
	return 0;
	break;
    case 1:{
	    version_1_format_ID *id1 = (version_1_format_ID *) format_ID;
	    short port;
	    memcpy(&port, &id1->port, 2);
	    return ntohs(port);
	    break;
	}
    default:
	printf("<Unknown format version %d\n",
	       *((unsigned char *) format_ID));
	break;
    }
    /* not reached */
    return 0;
}

/* write header information to the format server */
extern void
server_write_header(iofile)
IOFile iofile;
{
    FILE_INT magic = MAGIC_NUMBER + 2;
    FILE_INT float_format = Format_IEEE;
    FILE_INT server_pid;
    put_serverAtomicInt(iofile->server_fd, &magic, iofile);
    put_serverAtomicInt(iofile->server_fd, &float_format, iofile);

    get_serverAtomicInt(iofile->server_fd, &magic, 0);
    get_serverAtomicInt(iofile->server_fd, &server_pid, 0);
    if ((iofile->server_pid != 0) && (iofile->server_pid != server_pid)) {
	fprintf(stderr, "Format server restarted.  Dying\n");
	exit(0);
    } else {
	iofile->server_pid = server_pid;
    }
    if (magic != MAGIC_NUMBER) {
	if (magic == REVERSE_MAGIC_NUMBER) {
	    iofile->server_byte_reversal = 1;
	} else {
	    /* close client */
	    return;
	}
    }
}

extern void
server_write_char(fsc, chr)
FSClient fsc;
char *chr;
{
    serverAtomicWrite(fsc->fd, chr, 1);
}

extern void
server_read_header(fsc)
FSClient fsc;
{
    FILE_INT magic;
    FILE_INT float_format;
    FILE_INT pid = getpid();

    fsc->byte_reversal = 0;
    get_serverAtomicInt(fsc->fd, &magic, fsc->byte_reversal);

    if (magic != MAGIC_NUMBER) {
	if (magic == REVERSE_MAGIC_NUMBER) {
	    fsc->byte_reversal = 1;
	} else {
	    if (magic == (MAGIC_NUMBER + 1)) {
		fsc->version = 1;
	    } else if (magic == (REVERSE_MAGIC_NUMBER + 0x1000000)) {
		fsc->byte_reversal = 1;
		fsc->version = 1;
	    } else {
		if (magic == (MAGIC_NUMBER + 2)) {
		    fsc->version = 2;
		} else if (magic == (REVERSE_MAGIC_NUMBER + 0x2000000)) {
		    fsc->byte_reversal = 1;
		    fsc->version = 2;
		} else {
		    /* close client */
		    return;
		}
	    }
	}
    }
    get_serverAtomicInt(fsc->fd, &float_format, fsc->byte_reversal);
    magic = MAGIC_NUMBER;
    put_serverAtomicInt(fsc->fd, &magic, NULL);
    if (fsc->version >= 2)
	put_serverAtomicInt(fsc->fd, &pid, NULL);
    if (float_format != Format_IEEE) {
	assert(FALSE);
    }
}

extern int
version_of_format_ID(void *server_ID)
{
    /* format ID is at least 64 bits */
    char *char_ID = (char *) server_ID;
    if ((char_ID[4] == 0) && (char_ID[5] == 0) && 
	(char_ID[6] == 0) && (char_ID[7] == 0)) {
	/* version 0 format_IDs have no version info, but second int is
	 * zero */
	return 0;
    } else {
	/* first byte is version ID */
	return *((char *) server_ID);
    }
}

extern char *
get_server_rep_IOformat(ioformat, rep_length)
IOFormat ioformat;
int *rep_length;
{
    if (ioformat->body->server_format_rep == NULL) {
	ioformat->body->server_format_rep = 
	    build_server_format_rep(ioformat);
    }
    *rep_length = ioformat->body->server_format_rep->format_rep_length;
    return (char*)ioformat->body->server_format_rep;
}

extern char *
get_server_ID_IOformat(ioformat, id_length)
IOFormat ioformat;
int *id_length;
{
    *id_length = ioformat->body->server_ID.length;
    return ioformat->body->server_ID.value;
}


extern IOFormat
load_external_format_IOcontext(iocontext, server_id, id_size, server_rep)
IOContext iocontext;
char *server_id;
int id_size;
char *server_rep;
{
    IOFormat ioformat = get_local_format_IOcontext(iocontext, server_id);

    if (ioformat != NULL) {
	if (format_server_verbose == 1) {
	    printf("Load external format already exists  - ");
	    print_server_ID((void*)server_id);
	}
	/* format is already here */
	return ioformat;
    }
    ioformat = expand_ioformat_from_rep((format_rep)server_rep);

    if (ioformat == NULL) {
	if (format_server_verbose == 1) {
	    printf("Couldn't expand external format  - ");
	    print_server_ID((void*)server_id);
	}
	return NULL;
    }
    add_format_to_iofile((IOFile)iocontext, ioformat, id_size, server_id);
    return ioformat;
}
