#include "config.h"
#include "assert.h"
#include <stdio.h>
#ifdef STDC_HEADERS
#include <stdlib.h>
#endif
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <sys/types.h>
#include <ctype.h>
#ifdef HAVE_SYS_UIO_H
#include <sys/uio.h>
#endif
#include "io.h"
#include "io_interface.h"
#include "io_internal.h"
#include "unix_defs.h"
#include "assert.h"
extern char *getenv ARGS((const char *name));

#ifdef HAVE_VCODE_H
typedef void *drisc_ctx;
#define __NDEBUG__
#include "vcode.h"
#include "drisc_vcode.h"
#include "io_gen.h"
#define VCALL6V(subr, argstr, arg1, arg2, arg3, arg4, arg5, arg6) v_scallv((v_vptr)subr, argstr, arg1, arg2, arg3, arg4, arg5, arg6)
#define VCALL5V(subr, argstr, arg1, arg2, arg3, arg4, arg5) v_scallv((v_vptr)subr, argstr, arg1, arg2, arg3, arg4, arg5)
#define VCALL4V(subr, argstr, arg1, arg2, arg3, arg4) v_scallv((v_vptr)subr, argstr, arg1, arg2, arg3, arg4)
#define VCALL3V(subr, argstr, arg1, arg2, arg3) v_scallv((v_vptr)subr, argstr, arg1, arg2, arg3)
#define VCALL2V(subr, argstr, arg1, arg2) v_scallv((v_vptr)subr, argstr, arg1, arg2)
#define TYPE_ALIGN(t) vcode_type_align[t]
#endif

#ifdef HAVE_DRISC_H
#include "drisc.h"
#include "io_gen.h"
#define static_ctx c 
#define VCALL6V(subr, argstr, arg1, arg2, arg3, arg4, arg5, arg6) dr_scallv(c, (void*)subr, argstr, arg1, arg2, arg3, arg4, arg5, arg6)
#define VCALL5V(subr, argstr, arg1, arg2, arg3, arg4, arg5) dr_scallv(c, (void*)subr, argstr, arg1, arg2, arg3, arg4, arg5)
#define VCALL4V(subr, argstr, arg1, arg2, arg3, arg4) dr_scallv(c, (void*)subr, argstr, arg1, arg2, arg3, arg4)
#define VCALL3V(subr, argstr, arg1, arg2, arg3) dr_scallv(c, (void*)subr, argstr, arg1, arg2, arg3)
#define VCALL2V(subr, argstr, arg1, arg2) dr_scallv(c, (void*)subr, argstr, arg1, arg2)
#define TYPE_ALIGN(t) dr_type_align(t)
#define _vrr(x) x
#endif

static MAX_INTEGER_TYPE get_big_int ARGS((IOFieldPtr iofield, void *data));
static MAX_FLOAT_TYPE get_big_float ARGS((IOFieldPtr iofield, void *data));
static MAX_UNSIGNED_TYPE get_big_unsigned ARGS((IOFieldPtr iofield, void *data));
static void byte_swap ARGS((char *data, int size));

static int get_double_warn = 0;
static int get_long_warn = 0;

static void
convert_addr_field(iofile, src_spec, src, dest_size, dest, string_offset_size,
		   string_base_p, size_delta, converted_strings, src_offset_p,
		   dest_p, required_alignment)
IOFile iofile;
IOFieldPtr src_spec;
void *src;
int dest_size;
void *dest;
int string_offset_size;
char **string_base_p;
int size_delta;
int converted_strings;
int *src_offset_p;
void **dest_p;
int required_alignment;
{
    int align_tmp;
    if ((dest_size == sizeof(char *)) && (*string_base_p != NULL)) {
	IOgetFieldStruct tmp_src_field;
	MAX_INTEGER_TYPE tmp_int;
	char **dest_field = (char **) dest;
	tmp_src_field = *src_spec;

	tmp_src_field.data_type = integer_type;

	tmp_int = get_big_int(&tmp_src_field, src);

	*src_offset_p = tmp_int;
	if (tmp_int != 0) {
	    /* handle possibly different string base */
	    tmp_int -= (long) string_offset_size;
	    *dest_field = *string_base_p + tmp_int;
	    if ((align_tmp = (((unsigned long)*dest_field) % required_alignment)) != 0) {
		*dest_field += (required_alignment - align_tmp);
		*string_base_p  += (required_alignment - align_tmp);
	    }
	} else {
	    *dest_field = NULL;
	}
	*dest_p = *dest_field;
    } else if ((dest_size > sizeof(char *)) && (*string_base_p != NULL)) {
	/* native field is bigger than char*, store it */
	IOgetFieldStruct tmp_src_field;
	MAX_UNSIGNED_TYPE tmp_int;
	tmp_src_field = *src_spec;

	tmp_src_field.data_type = integer_type;

	tmp_int = get_big_unsigned(&tmp_src_field, src);

	*src_offset_p = tmp_int;

	if (tmp_int != 0) {
	    /* handle possibly different string base */
	    tmp_int -= (long) string_offset_size;
	    *dest_p = tmp_int + *string_base_p;
	    tmp_int = tmp_int + (MAX_UNSIGNED_TYPE) (unsigned long) *string_base_p;
	    if ((align_tmp = (((unsigned long)*dest_p) % required_alignment)) != 0) {
		*dest_p = (char*)*dest_p + (required_alignment - align_tmp);
		tmp_int += (required_alignment - align_tmp);
		*string_base_p  += (required_alignment - align_tmp);
	    }
	} else {
	    *dest_p = NULL;
	}
	tmp_src_field.offset = 0;
	tmp_src_field.size = sizeof(MAX_UNSIGNED_TYPE);
	tmp_src_field.byte_swap = FALSE;
	pbio_internal_convert_field(iofile, &tmp_src_field, &tmp_int,
				    unsigned_type, dest_size,
				    dest, 0, *string_base_p, 0, FALSE);
    } else {
	/* not a native field struct.  Keep the pointer as an offset. */
	IOgetFieldStruct tmp_src_field;
	MAX_INTEGER_TYPE tmp_int;
	tmp_src_field = *src_spec;

	tmp_src_field.data_type = integer_type;
	tmp_int = get_big_int(&tmp_src_field, src);

	*src_offset_p = tmp_int;
	if (tmp_int != 0) {
	    if (!converted_strings) {
		/* record size might change, adjust offset */
		tmp_int += size_delta;
	    }
	    *dest_p = tmp_int + *string_base_p;
	    if ((align_tmp = (((unsigned long)*dest_p) % required_alignment)) != 0) {
		*dest_p =  (char*)*dest_p + (required_alignment - align_tmp);
		tmp_int += (required_alignment - align_tmp);
		*string_base_p  += (required_alignment - align_tmp);
	    }
	} else {
	    *dest_p = NULL;
	}
	tmp_src_field.offset = 0;
	tmp_src_field.size = sizeof(MAX_INTEGER_TYPE);
	tmp_src_field.byte_swap = FALSE;
	pbio_internal_convert_field(iofile, &tmp_src_field, &tmp_int,
				    integer_type, dest_size,
				    dest, 0, *string_base_p, 0, FALSE);
    }
}

extern void
pbio_internal_convert_field(iofile, src_spec, src, dest_type, dest_size, dest,
	  string_offset_size, string_base, size_delta, converted_strings)
IOFile iofile;
IOFieldPtr src_spec;
void *src;
IOdata_type dest_type;
int dest_size;
void *dest;
int string_offset_size;
char *string_base;
int size_delta;
int converted_strings;
{
    /* quick check to see if it's just a copy... */
    if ((dest_type != string_type) && (dest_type == src_spec->data_type) &&
	(dest_size == src_spec->size)) {
	if (src_spec->byte_swap) {
	    int i;
	    char *destc = (char *) dest;
	    char *srcc = (char *) src + src_spec->offset;
	    for (i = 0; i < (dest_size >> 1); i++) {
		char tmp = srcc[dest_size - i - 1];
		destc[dest_size - i - 1] = srcc[i];
		destc[i] = tmp;
	    }
	    if ((dest_size & 0x1) != 0) {
		destc[dest_size >> 1] = srcc[dest_size >> 1];
	    }
	} else {
	    char *srcc = (char *) src + src_spec->offset;
	    memcpy(dest, srcc, dest_size);
	}
	return;
    }
    switch (dest_type) {
    case integer_type:
	{
	    MAX_INTEGER_TYPE tmp = get_big_int(src_spec, src);
	    if (dest_size == sizeof(char)) {
		char *dest_field = (char *) dest;
		*dest_field = (char) tmp;
	    } else if (dest_size == sizeof(short)) {
		short *dest_field = (short *) dest;
		*dest_field = (short) tmp;
	    } else if (dest_size == sizeof(int)) {
		int *dest_field = (int *) dest;
		*dest_field = tmp;
	    } else if (dest_size == sizeof(long)) {
		long *dest_field = (long *) dest;
		*dest_field = tmp;
#if SIZEOF_LONG_LONG != 0
	    } else if (dest_size == sizeof(long long)) {
		long long lltmp = tmp;
		memcpy(dest, &lltmp, sizeof(long long));
#endif
	    } else {
		iofile->result = "size problems in conversion";
	    }
	    break;
	}
    case char_type:
	{
	    char tmp = get_IOchar(src_spec, src);
	    char *dest_field = (char *) dest;
	    *dest_field = tmp;
	    break;
	}
    case boolean_type:
    case enumeration_type:
	{
	    int tmp = get_IOenum(src_spec, src);
	    int *dest_field = (int *) dest;
	    *dest_field = tmp;
	    break;
	}
    case unsigned_type:
	{
	    MAX_UNSIGNED_TYPE tmp = get_big_unsigned(src_spec, src);
	    if (dest_size == sizeof(unsigned char)) {
		unsigned char *dest_field = (unsigned char *) dest;
		*dest_field = (unsigned char) tmp;
	    } else if (dest_size == sizeof(unsigned short)) {
		unsigned short *dest_field = (unsigned short *) dest;
		*dest_field = (unsigned short) tmp;
	    } else if (dest_size == sizeof(int)) {
		unsigned int *dest_field = (unsigned int *) dest;
		*dest_field = tmp;
	    } else if (dest_size == sizeof(long)) {
		unsigned long *dest_field = (unsigned long *) dest;
		*dest_field = tmp;
#if SIZEOF_LONG_LONG != 0
	    } else if (dest_size == sizeof(long long)) {
		unsigned long long *dest_field = (unsigned long long *) dest;
		*dest_field = tmp;
#endif
	    } else {
		iofile->result = "size problems in conversion";
	    }
	    break;
	}
    case float_type:
	{
	    MAX_FLOAT_TYPE tmp = get_big_float(src_spec, src);
	    if (dest_size == sizeof(float)) {
		float *dest_field = (float *) dest;
		*dest_field = (float) tmp;
	    } else if (dest_size == sizeof(double)) {
		double *dest_field = (double *) dest;
		*dest_field = tmp;
#if SIZEOF_LONG_DOUBLE != 0
	    } else if (dest_size == sizeof(long double)) {
		long double *dest_field = (long double *) dest;
		*dest_field = tmp;
#endif
	    } else {
		iofile->result = "size problems in conversion";
	    }
	    break;
	}
    case string_type:
	{
	    void *junk;
	    int junk_int;
	    /* use Junk values because we don't care about address output */
	    convert_addr_field(iofile, src_spec, src, dest_size, dest,
			       string_offset_size, &string_base, size_delta,
			       converted_strings, &junk_int, &junk, 1);
	    break;
	}
    default:
	assert(FALSE);
    }
}

static void
do_var_part_conv ARGS((IOConversionPtr conv, IOconvFieldStruct * conv_field,
		     void *src_area, void *final_area, int control_value,
		       void *src_string_base, void **final_string_base));

static int debug_code_generation = -1;

void
IOconvert_record(conv, src, dest, final_string_base, src_string_base)
IOConversionPtr conv;
void *src;
void *dest;
void *final_string_base;
void *src_string_base;
{
    int i;
    long *control_value = NULL;

    if (src_string_base == NULL) {
	src_string_base = final_string_base;
    }
    if (conv->conv_func) {
	if (debug_code_generation) {
	    int i;
	    int *tmp = (int *) (((char *) src_string_base) -
				(((long) src_string_base) % 4));
	    printf("record contents :\n");
	    for (i = 0; i < 30; i += 4) {
		printf("%lx: %8x %8x %8x %8x\n", (long) ((char *) src) + (i * 4),
		       ((int *) src)[i], ((int *) src)[i + 1],
		       ((int *) src)[i + 2], ((int *) src)[i + 3]);
	    }
	    if (src_string_base != NULL) {
		printf("string contents :\n");
		for (i = 0; i < 30; i += 4) {
		    printf("%lx: %8x %8x %8x %8x\n", (long) ((char *) tmp) + (i * 4),
			   ((int *) tmp)[i],
			   ((int *) tmp)[i + 1],
			   ((int *) tmp)[i + 2],
			   ((int *) tmp)[i + 3]);
		}
	    }
	}
	conv->conv_func(src, dest, final_string_base, src_string_base);
	return;
    }
    for (i = 0; i < conv->conv_count; i++) {
	if (conv->conversions[i].control_field != NULL) {
	    if (control_value == NULL) {
		int j;
		control_value = (long *) io_malloc(sizeof(long) * conv->conv_count);
		for (j = 0; j < conv->conv_count; j++)
		    control_value[j] = 0;
	    }
	    control_value[i] = (long)
		get_big_int(conv->conversions[i].control_field, src);
	}
    }
    for (i = 0; i < conv->conv_count; i++) {
	IOFieldPtr src_spec = &conv->conversions[i].src_field;
	int dimen1 = conv->conversions[i].array_dimen1;
	int dimen2 = conv->conversions[i].array_dimen2;
	int byte_swap = conv->conversions[i].src_field.byte_swap;

	if (IOhas_error(conv->iofile))
	    return;
	if (conv->conversions[i].src_field.size == 1) byte_swap = 0;
	if (!byte_swap &&
	    (src_spec->size == conv->conversions[i].dest_size) &&
	    (conv->conversions[i].subconversion == NULL) &&
	    (conv->conversions[i].control_field == NULL) &&
	    ((src_spec->data_type != string_type) || (final_string_base == NULL))) {
	    /* data movement is all that is required */
	    void *dest_field = (void *) (conv->conversions[i].dest_offset +
					 (char *) dest);
	    if ((dimen1 == 1) && (dimen2 == 1)) {
		memcpy(dest_field, (char *) src + src_spec->offset,
		       src_spec->size);
	    } else {
		memcpy(dest_field, (char *) src + src_spec->offset,
		       src_spec->size * dimen1 * dimen2);
	    }
	} else {
	    int j;
	    void *dest_field = (void *) (conv->conversions[i].dest_offset +
					 (char *) dest);
	    IOgetFieldStruct tmp_spec;
	    IOdata_type dest_type = src_spec->data_type;
	    int dest_size = conv->conversions[i].dest_size;
	    int iter_count;
	    tmp_spec = *src_spec;

	    if (conv->conversions[i].control_field != NULL) {
		/* really a variant array, adjust address like a string */
		tmp_spec.data_type = string_type;
		tmp_spec.size = conv->ioformat->body->pointer_size;
		dest_type = string_type;
		dest_size = conv->target_pointer_size;
	    }
	    tmp_spec.offset = src_spec->offset - tmp_spec.size;
	    iter_count = 1;
	    if ((dimen1 != 1) || (dimen2 != 1)) {
		iter_count = dimen1 * dimen2;
	    }
	    for (j = 0; j < iter_count; j++) {
		tmp_spec.offset += tmp_spec.size;
		if (dest_type != unknown_type) {
		    /* simple (native) field or variant array */
		    if (dest_type != string_type) {
			pbio_internal_convert_field(conv->iofile, &tmp_spec, src,
						    dest_type, dest_size,
				    dest_field, conv->string_offset_size,
						    final_string_base,
						    conv->base_size_delta,
						conv->converted_strings);
		    } else {
			int src_offset;
			void *src_base;
			void *dst_base;
			/* string or variant array */
			int req_align;
			if (conv->conversions[i].control_field != NULL) {
			    req_align = min_align_size(conv->conversions[i].dest_size);
			} else {
			    req_align = 1;
			}
			convert_addr_field(conv->iofile, &tmp_spec, src,
					   dest_size, dest_field,
					   conv->string_offset_size,
					   &final_string_base,
					   conv->base_size_delta,
					   conv->converted_strings,
					   &src_offset, &dst_base,
					   req_align);
			src_base = (char *) src_string_base + src_offset
			    - conv->string_offset_size;
			if ((conv->conversion_type == copy_strings) ||
			  (conv->conversions[i].control_field != NULL)) {
			    int control = 0;
			    if (control_value != NULL)
				control = control_value[i];
			    do_var_part_conv(conv, &conv->conversions[i],
					     src_base, dst_base,
					     control, src_string_base,
					     &final_string_base);
			}
		    }
		} else {
		    IOConversionPtr subconv =
		    conv->conversions[i].subconversion;
		    void *subsrc = (void *) ((char *) src + tmp_spec.offset);
		    IOconvert_record(subconv, subsrc, dest_field,
				     final_string_base, src_string_base);
		}
		dest_field = (void *) ((char *) dest_field +
				       conv->conversions[i].dest_size);
	    }
	}
    }
    if (control_value != NULL) {
	io_free(control_value);
    }
}

static void
do_var_part_conv(conv, conv_field, src_area, final_area,
		 control_value, src_string_base, final_string_base)
IOConversionPtr conv;
IOconvFieldStruct *conv_field;
void *src_area;
void *final_area;
int control_value;
void *src_string_base;
void **final_string_base;
{
    int copy_length = 0;
    IOFieldPtr src_spec = &(conv_field->src_field);

    /* handle copying or conversion */
    if (conv_field->control_field == NULL) {
	/* simple string */
	if ((src_area != NULL) && (final_area != NULL)) {
	    int copy_length = strlen(src_area) + 1;
	    memcpy(final_area, src_area, copy_length);
	}
    } else {
	/* variant array */
	int byte_swap = conv_field->src_field.byte_swap;
	if (src_spec->size == 1) byte_swap = 0;
	if (!byte_swap &&
	    (src_spec->size == conv_field->dest_size) &&
	    (conv_field->subconversion == NULL) &&
	    (src_spec->data_type != string_type)) {

	    /* data movement is all that is required */
	    if (conv->conversion_type == copy_strings) {
		copy_length = conv_field->dest_size *
		    control_value;
		memcpy(final_area, src_area, copy_length);
	    } else {
		/* not even that is necessary */
	    }
	} else {
	    /* must do conversions */
	    int length = control_value;
	    int j;
	    void *dest_field = final_area;
	    IOgetFieldStruct tmp_spec;
	    IOdata_type dest_type = src_spec->data_type;
	    int dest_size = conv_field->dest_size;

	    tmp_spec = *src_spec;
	    *final_string_base = (char *) *final_string_base +
		(length * (dest_size - src_spec->size));
	    for (j = 0; j < length; j++) {
		tmp_spec.offset = j * src_spec->size;
		if (dest_type != unknown_type) {
		    /* simple (native) field in variant array */
		    if (src_spec->data_type != string_type) {
			pbio_internal_convert_field(conv->iofile, &tmp_spec,
						    src_area, dest_type, 
						    dest_size, dest_field,
						    conv->string_offset_size,
						    *final_string_base,
						    conv->base_size_delta,
						    conv->converted_strings);
		    } else {
			/* variant string array */
			void *dest_base;
			void *src_base;
			int src_offset;
			convert_addr_field(conv->iofile, &tmp_spec, src_area,
					   dest_size, dest_field,
					   conv->string_offset_size, 
					   final_string_base, 
					   conv->base_size_delta,
					   conv->converted_strings,
					   &src_offset, &dest_base, 1);
			src_base = (char *) src_string_base + src_offset
			    - conv->string_offset_size;
			if (conv->conversion_type == copy_strings) {
			    int copy_length = strlen(src_base) + 1;
			    memcpy(dest_base, src_base, copy_length);
			}
		    }
		        
		} else {
		    IOConversionPtr subconv =
		    conv_field->subconversion;
		    void *subsrc = (char *) src_area + j * src_spec->size;
		    IOconvert_record(subconv, subsrc, dest_field,
				     *final_string_base,
				     src_string_base);
		}
		dest_field = (void *) ((char *) dest_field +
				       conv_field->dest_size);
	    }
	}
    }
}

static MAX_INTEGER_TYPE
get_big_int(iofield, data)
IOFieldPtr iofield;
void *data;
{
    if (iofield->data_type == integer_type) {
	if (iofield->size == sizeof(char)) {
	    char tmp;
	    memcpy(&tmp, (char *) data + iofield->offset, sizeof(char));
	    return (long) tmp;
	} else if (iofield->size == sizeof(short)) {
	    short tmp;
	    memcpy(&tmp, (char *) data + iofield->offset, sizeof(short));
	    if (iofield->byte_swap)
		byte_swap((char *) &tmp, sizeof(short));
	    return (long) tmp;
	} else if (iofield->size == sizeof(int)) {
	    int tmp;
	    memcpy(&tmp, (char *) data + iofield->offset, sizeof(int));
	    if (iofield->byte_swap)
		byte_swap((char *) &tmp, sizeof(int));
	    return (long) tmp;
	} else if (iofield->size == sizeof(long)) {
	    long tmp;
	    memcpy(&tmp, (char *) data + iofield->offset, sizeof(long));
	    if (iofield->byte_swap)
		byte_swap((char *) &tmp, sizeof(long));
	    return tmp;
	} else if (iofield->size == 2 * sizeof(long)) {
	    long tmp;
	    int low_bytes_offset = iofield->offset;
#ifdef WORDS_BIGENDIAN
	    if (!iofield->byte_swap) {
		low_bytes_offset += sizeof(long);
	    }
#else
	    if (iofield->byte_swap) {
		low_bytes_offset += sizeof(long);
	    }
#endif
	    memcpy(&tmp, (char *) data + low_bytes_offset, sizeof(long));
	    if (iofield->byte_swap)
		byte_swap((char *) &tmp, sizeof(long));
	    return tmp;
	} else {
	    if (!IO_shut_up && !get_long_warn) {
		fprintf(stderr, "Get Long failed!  Size problems.  File int size is %d.\n", iofield->size);
		get_long_warn++;
	    }
	    return -1;
	}
    } else if (iofield->data_type == unsigned_type) {
	MAX_UNSIGNED_TYPE tmp = get_big_unsigned(iofield, data);
	return (MAX_UNSIGNED_TYPE) tmp;
    } else if (iofield->data_type == float_type) {
	MAX_FLOAT_TYPE tmp = get_big_float(iofield, data);
#ifndef METICULOUS_FLOATS_AND_LONGS
	return (MAX_INTEGER_TYPE) (long) (double) tmp;
#else
	return (MAX_INTEGER_TYPE) tmp;
#endif
    } else {
	fprintf(stderr, "Get IOlong failed on invalid data type!\n");
	exit(1);
    }
    /* NOTREACHED */
    return 0;
}

static MAX_UNSIGNED_TYPE
get_big_unsigned(iofield, data)
IOFieldPtr iofield;
void *data;
{
    if ((iofield->data_type == unsigned_type) || 
	(iofield->data_type == enumeration_type) || 
	(iofield->data_type == boolean_type)) {
	if (iofield->size == sizeof(char)) {
	    unsigned char tmp;
	    memcpy(&tmp, (char *) data + iofield->offset, sizeof(char));
	    return (MAX_UNSIGNED_TYPE) tmp;
	} else if (iofield->size == sizeof(short)) {
	    unsigned short tmp;
	    memcpy(&tmp, (char *) data + iofield->offset, sizeof(short));
	    if (iofield->byte_swap)
		byte_swap((char *) &tmp, sizeof(short));
	    return (MAX_UNSIGNED_TYPE) tmp;
	} else if (iofield->size == sizeof(int)) {
	    unsigned int tmp;
	    memcpy(&tmp, (char *) data + iofield->offset, sizeof(int));
	    if (iofield->byte_swap)
		byte_swap((char *) &tmp, sizeof(int));
	    return (MAX_UNSIGNED_TYPE) tmp;
	} else if (iofield->size == sizeof(long)) {
	    unsigned long tmp;
	    memcpy(&tmp, (char *) data + iofield->offset, sizeof(long));
	    if (iofield->byte_swap)
		byte_swap((char *) &tmp, sizeof(long));
	    return tmp;
	} else if (iofield->size == 2 * sizeof(long)) {
	    unsigned long tmp;
	    int low_bytes_offset = iofield->offset;
#ifdef WORDS_BIGENDIAN
	    if (!iofield->byte_swap) {
		low_bytes_offset += sizeof(long);
	    }
#else
	    if (iofield->byte_swap) {
		low_bytes_offset += sizeof(long);
	    }
#endif
	    memcpy(&tmp, (char *) data + low_bytes_offset, sizeof(long));
	    if (iofield->byte_swap)
		byte_swap((char *) &tmp, sizeof(long));
	    return tmp;
	} else {
	    if (!IO_shut_up && !get_long_warn) {
		fprintf(stderr, "Get Long failed!  Size problems.  File int size is %d.\n", iofield->size);
		get_long_warn++;
	    }
	    return 0;
	}
    } else if (iofield->data_type == integer_type) {
	MAX_INTEGER_TYPE tmp = get_big_int(iofield, data);
	return (MAX_UNSIGNED_TYPE) tmp;
    } else if (iofield->data_type == float_type) {
	MAX_FLOAT_TYPE tmp = get_big_float(iofield, data);
#ifndef METICULOUS_FLOATS_AND_LONGS
	return (MAX_UNSIGNED_TYPE) (long) (double) tmp;
#else
	return (MAX_UNSIGNED_TYPE) tmp;
#endif
    } else {
	fprintf(stderr, "Get IOulong failed on invalid data type!\n");
	exit(1);
    }
    /* NOTREACHED */
    return 0;
}

static MAX_FLOAT_TYPE
get_big_float(iofield, data)
IOFieldPtr iofield;
void *data;
{
    if (iofield->data_type == float_type) {
	if (iofield->size == sizeof(float)) {
	    float tmp;
	    MAX_FLOAT_TYPE tmp2;
	    memcpy(&tmp, ((char *) data + iofield->offset), sizeof(float));
	    if (iofield->byte_swap)
		byte_swap((char *) &tmp, sizeof(float));
	    tmp2 = tmp;
	    return tmp2;
	} else if (iofield->size == sizeof(double)) {
	    double tmp;
	    MAX_FLOAT_TYPE tmp2;
	    memcpy(&tmp, ((char *) data + iofield->offset), sizeof(double));
	    if (iofield->byte_swap)
		byte_swap((char *) &tmp, sizeof(double));
	    tmp2 = tmp;
	    return tmp2;
#if SIZEOF_LONG_DOUBLE != 0
	} else if (iofield->size == sizeof(long double)) {
	    long double tmp;
	    MAX_FLOAT_TYPE tmp2;
	    memcpy(&tmp, ((char *) data + iofield->offset),
		   sizeof(long double));
	    if (iofield->byte_swap)
		byte_swap((char *) &tmp,
			  sizeof(long double));
	    tmp2 = tmp;
	    return tmp2;
#endif
	} else {
	    if (!IO_shut_up && !get_double_warn) {
		fprintf(stderr, "Get Double failed!  Size problems.  File double size is %d.\n", iofield->size);
		get_double_warn++;
	    }
	    return 0.0;
	}
    } else if (iofield->data_type == integer_type) {
	MAX_INTEGER_TYPE tmp = get_big_int(iofield, data);
#ifndef METICULOUS_FLOATS_AND_LONGS
	/* 
	 * A concession to inter-compiler interoperability...  We
	 * shouldn't need the (double)(long) casts here.  If we don't
	 * use them, AND we're on a machine which doesn't support "long 
	 * long" and "long double" in native code, gcc generates calls 
	 * to libgcc and the resulting library can't be linked with
	 * anything but GCC.  This can be a problem for installed
	 * libraries.  Using the casts avoids that particular problem,
	 * at a cost of data loss in the case of someone converting a
	 * "long long" that doesn't fit in a long to a floating point
	 * value. 
	 */
	return (MAX_FLOAT_TYPE) (double) (long) tmp;
#else
	return (MAX_FLOAT_TYPE) tmp;
#endif
    } else if (iofield->data_type == unsigned_type) {
	MAX_UNSIGNED_TYPE tmp = get_big_unsigned(iofield, data);
#ifndef METICULOUS_FLOATS_AND_LONGS
	return (MAX_FLOAT_TYPE) (double) (long) tmp;
#else
	return (MAX_FLOAT_TYPE) tmp;
#endif
    } else {
	fprintf(stderr, "Get Double failed on invalid data type!\n");
	exit(1);
    }
    /* NOTREACHED */
    return 0;
}

static void
byte_swap(data, size)
char *data;
int size;
{
    int i;
    assert((size % 2) == 0);
    for (i = 0; i < size / 2; i++) {
	char tmp = data[i];
	data[i] = data[size - i - 1];
	data[size - i - 1] = tmp;
    }
}

#if !defined(HAVE_VCODE_H) && !defined(HAVE_DRISC_H)
extern
 conv_routine
generate_conversion(conv, alignment)
IOConversionPtr conv;
int alignment;
{
    return NULL;
}
#else

#define max(x,y) (x<y?y:x)

static int
drisc_type(field)
IOgetFieldStruct field;
{
    switch(field.data_type) {
    case integer_type:
    case unsigned_type:
	switch(field.size) {
	case 1:
	    return DR_C;
	case 2:
	    return DR_S;
	case 4:
	    return DR_I;
	case 8:
	    return DR_L;
	}
	return DR_I;
    case float_type:
	if (field.size == SIZEOF_DOUBLE) {
	    return DR_D;
	} else if (field.size == SIZEOF_FLOAT) {
	    return DR_F;
	} else {
	    return DR_I;
	}
    case char_type:
	return DR_C;
    case string_type:
	return DR_P;
    case boolean_type:
    case enumeration_type:
	return DR_I;
    default:
	return DR_I;
    }
}

static int
subfield_required_align(c, conv, i, offset)
drisc_ctx c;
IOConversionPtr conv;
int i;
int offset;
{
    int field_required_align;
    int field_align, access_align;
    static int structure_align = -1;
    if (structure_align == -1) {
	int j;
	for (j=DR_C; j< DR_D; j++) {
	    structure_align = max(structure_align, TYPE_ALIGN(j));
	}
    }
    if (conv->conversions[i].control_field != NULL) {
	field_required_align = TYPE_ALIGN(DR_P);
    } else if (conv->conversions[i].subconversion == NULL) {
	int drisc_data_type = drisc_type(conv->conversions[i].src_field);
	field_required_align = TYPE_ALIGN(drisc_data_type);
    } else {
	field_required_align = structure_align;
    }
    field_align = offset % field_required_align;
    if (field_align == 0) {
	access_align = field_required_align;
    } else {
	int access_align1 = min_align_size(field_align);
	int access_align2 =
	    min_align_size(field_required_align - field_align);
	access_align = max(access_align1, access_align2);
    }
    return access_align;
}

static int
conv_required_alignment(c, conv)
drisc_ctx c;
IOConversionPtr conv;
{
    int i;
    int required_align = 0;
    if (conv->conv_count == 0) return 0;
    for (i = 0; i < conv->conv_count; i++) {
	int subfield_requires = 
	    subfield_required_align(c, conv, i, 
				    conv->conversions[i].src_field.offset);
	required_align = max(subfield_requires, required_align);
    }
    assert(required_align != 0);
    return required_align;
}

static
 conv_routine
 generate_conversion_code ARGS((drisc_ctx c,
				IOConversionPtr conv, dr_reg_t * args,
				int assume_align, int register_args,
				int extra_src_offset, 
				int extra_dest_offset));

static int pbio_conversion_generation = -1;
static int generation_verbose = -1;
#ifdef HAVE_VCODE_H
extern void v_dump();
void (*dr_dump_routine)ARGS((void *routine)) = (void(*)ARGS((void *)))v_dump;
#endif


#define gen_fatal(str) do {fprintf(stderr, "%s\n", str); exit(0);} while (0)

extern
 conv_routine
generate_conversion(conv, base_alignment)
IOConversionPtr conv;
int base_alignment;
{
    void *insns;
    drisc_ctx c = NULL;
    void (*conversion_routine)();
    dr_reg_t args[6];
    dr_reg_t tmp_regs[10];
    char *format_name = conv->ioformat->body->format_name;
    int count = 0, register_args = 1;

    if (pbio_conversion_generation == -1) {
	char *gen_string = getenv("PBIO_CONVERSION_GENERATION");
	pbio_conversion_generation = 0;
	if (gen_string != NULL) {
	    if (sscanf(gen_string, "%d", &pbio_conversion_generation) != 1) {
		if (*gen_string == 0) {
		    /* empty string, just turn on generation */
		    pbio_conversion_generation = 1;
		} else {
		    printf("Unable to parse PBIO_CONVERSION_GENERATION environment variable \"%s\".\n", gen_string);
		}
	    }
	}
	debug_code_generation =
	    (getenv("PBIO_CONVERSION_DEBUG") != NULL);
	generation_verbose =
	    (getenv("PBIO_CONVERSION_VERBOSE") != NULL);
    }
    if (!pbio_conversion_generation)
	return NULL;
    if (generation_verbose) {
	printf("For format %s ===================\n", format_name);
	dump_IOConversion(conv);
    }
    insns = io_malloc(2048);
    {
	/* 
	 *  Determine whether or not some arguments should be left on the
	 *  stack.  The issue is that some architectures (x86) don't have
	 *  enough registers to follow the usual vcode convention of
	 *  immediately moving all arguments into registers for the
	 *  convenience of the generated subroutine.  If we let it do that,
	 *  we don't have enough temporary registers for use here.  So, we
	 *  first find out how many we have by doing getreg() until it fails
	 *  (or a max of 10 times).  If we don't have at least 8 registers
	 *  (four for args and four for temporary use), then we only want 
	 *  the first two arguments of the conversion code (the source and
	 *  destination address) to be kept in registers.  Trick vcode into
	 *  doing this by making all but two of the registers unavailable.
	 *  
	 *  IF WE DETERMINE WE HAVE ENOUGH REGISTERS, 'register_args' is set
	 *  to TRUE and the 'args' array contains the usual register
	 *  numbers.  If we don't have enough registers, 'register_args' is
	 *  false and args[2] and args[3] contain the stack offsets of the
	 *  third and fourth arguments.
	 */
#ifdef HAVE_VCODE_H
	v_init();
#else
	c = dr_init();
#endif
	count = 0;
	for (; count < sizeof(tmp_regs)/sizeof(tmp_regs[0]); count++) 
	    tmp_regs[count] = -1;
	count = 0;
	for (; count < sizeof(tmp_regs)/sizeof(tmp_regs[0]); count++) {
	    if (dr_getreg(c, &tmp_regs[count], DR_I, DR_VAR) == 0) {
		break;
	    }
	}
	if (count <= 8) {
	    int i;
	    register_args = 0;
	    for (i= 2; i < count; i++) {  /* Make all but 2 unavail*/
		dr_mk_unavail(c, DR_I, tmp_regs[i]);
	    }
	}
    }
    if (register_args) {
	/* Normal, lots of registers, case */

#ifdef HAVE_VCODE_H
	v_lambda("convert", "%p%p%p%p", args, V_NLEAF, insns, 2048);
#else
	dr_proc_params(c, "convert", "%p%p%p%p");
	args[0] = dr_param(c, 0);
	args[1] = dr_param(c, 1);
	args[2] = dr_param(c, 2);
	args[3] = dr_param(c, 3);
#endif

    } else {
	/* very few registers case */

	drisc_parameter_type dr_params[4];	/* drisc param info */
	int i;
	for (i=0; i < 4; i++) {
	    dr_param_alloc(c, i, DR_P, (dr_reg_t*)&args[i]);
	    dr_param_struct_alloc(c, i, DR_P, &(dr_params[i]));
	}
#ifdef HAVE_VCODE_H
	v_clambda("convert", V_NLEAF, insns, 2048);
#else
	dr_proc(c, "convert");
#endif

	/* store argument stack offsets in args[2] and args[3] */
	args[2] = dr_params[2].offset;
	args[3] = dr_params[3].offset;
	for (count=2; count<sizeof(tmp_regs)/sizeof(tmp_regs[0]); count++) {
	    if (tmp_regs[count] != -1) {
		/* make the other registers available again */
		dr_mk_avail(c, DR_I, tmp_regs[count]);
	    }
	}
    }
    if (debug_code_generation) {
	if (register_args) {
	    VCALL6V( printf, "%P%P%p%p%p%p",
		     "convert for %s called with src= %lx, dest %lx, final_string =%lx, src_string =%lx\n",
		     format_name, args[0], args[1], args[2], args[3]);
	} else {
#ifdef HAVE_DRISC_H	    
	    dr_reg_t v_at;
	    if (dr_getreg(c, &v_at, DR_I, DR_TEMP) == 0) {
		gen_fatal("Out of regs 1\n");
	    }
#endif
	    VCALL4V(printf, "%P%P%p%p",
		     "convert for %s called with src= %lx, dest %lx\n",
		     format_name, args[0], args[1]);
	    dr_ldpi(c, v_at, dr_lp(c), args[2]);
	    VCALL2V(printf, "%P%p",
		     "               src_string_base %lx\n",
		     v_at);
	    dr_ldpi(c, v_at, dr_lp(c), args[3]);
	    VCALL2V(printf, "%P%p",
		     "               final_string_base %lx\n",
		     v_at);
#ifdef HAVE_DRISC_H	    
	    dr_putreg(c, v_at, DR_I);
#endif
	}
    }
    conv->required_alignment = conv_required_alignment(c, conv);
    if (register_args) {
	dr_reg_t tmp;
	int mask;
	int zero_target = dr_genlabel(c);
	if (dr_getreg(c, &tmp, DR_I, DR_VAR) == 0) {
	    printf("out of regs for mod\n");
	}
	switch(conv->required_alignment) {
	case 2:
	    mask = 0x1;
	    break;
	case 4:
	    mask = 0x3;
	    break;
	case 8:
	    mask = 0x7;
	    break;
	default:
	    mask = 0;
	}
	if (mask != 0) {
	    dr_anduli(c, tmp, args[0], mask);
	    dr_beqli(c, tmp, 0, zero_target);
	    VCALL4V(printf, "%P%P%p%d",
		    "convert for %s called with bad align src= %lx, align is %d\n",
		    format_name, args[0], conv->required_alignment);
	    dr_label(c, zero_target);
	}
    }
    generate_conversion_code(c, conv, args, base_alignment, register_args, 0, 0);
    conversion_routine = (void(*)())dr_end(c);
    if (generation_verbose) {
#ifdef HAVE_VCODE_H
	if (dr_dump_routine != NULL) {
	    dr_dump_routine((void *) conversion_routine);
	} else {
	    printf("Call vdump_init() to enable dumping\n");
	}
#else
	dr_dump(c);
#endif
    }
    return (conv_routine) conversion_routine;
}
/* #define REG_DEBUG(x) printf x ;*/
#define REG_DEBUG(x)

static void
gen_var_part_conv ARGS((drisc_ctx c, IOConversionPtr conv, int i, int control_base,
			int assume_align, dr_reg_t src_addr, dr_reg_t dest_addr,
			dr_reg_t src_string_base, dr_reg_t final_string_base,
			int register_args));


static void
gen_simple_field_conv(c, tmp_spec, assume_align, src_addr, src_offset, 
		      dest_size, dest_type, dest_addr, dest_offset,
		      string_offset_size, final_string_base, base_size_delta,
		      converted_strings)
drisc_ctx c;
IOgetFieldStruct tmp_spec;
int assume_align;
dr_reg_t src_addr;
int src_offset;
int dest_size;
IOdata_type dest_type;
dr_reg_t dest_addr;
int dest_offset;
int string_offset_size;
dr_reg_t final_string_base;
int base_size_delta;
int converted_strings;
{
    /* simple conversion */
    iogen_oprnd src_oprnd;
    int src_drisc_type;
    int dst_drisc_type;
    int src_required_align;
    int dst_required_align;
    int src_is_aligned = 1;
    int dst_is_aligned = 1;
    src_drisc_type = drisc_type(tmp_spec);
    dst_drisc_type = drisc_type(tmp_spec);
    src_required_align = TYPE_ALIGN(src_drisc_type);
    dst_required_align = TYPE_ALIGN(dst_drisc_type);
    if ((assume_align < src_required_align) || 
	((src_offset % src_required_align) != 0)) {
	src_is_aligned = 0;
    }
    if ((assume_align < dst_required_align) || 
	((dest_offset % dst_required_align) != 0)) {
	dst_is_aligned = 0;
    }

    assert(dest_type != string_type);

    src_oprnd = gen_operand(src_addr, src_offset, tmp_spec.size,
			    tmp_spec.data_type,
			    src_is_aligned,
			    tmp_spec.byte_swap);

    if (src_oprnd.address) {
	/* load the data if gen_byte_swap didn't */
	gen_load(c, &src_oprnd);
    }
    if (src_oprnd.data_type != dest_type) {
	iogen_oprnd tmp_oprnd;
	tmp_oprnd = gen_type_conversion(c, src_oprnd, dest_type);
	free_oprnd(c, src_oprnd);
	src_oprnd = tmp_oprnd;
    }
    if (src_oprnd.size != dest_size) {
	iogen_oprnd tmp_oprnd;
	tmp_oprnd = gen_size_conversion(c, src_oprnd, dest_size);
	free_oprnd(c, src_oprnd);
	src_oprnd = tmp_oprnd;
    }
    gen_store(c, src_oprnd, dest_addr, dest_offset,
	      dest_size, dest_type, dst_is_aligned);
    free_oprnd(c, src_oprnd);
}

static void
gen_addr_field_conv(c, tmp_spec, assume_align, src_addr, src_offset,
		    dest_size, dest_addr, dest_offset, string_offset_size,
  final_string_base, src_string_base, base_size_delta, converted_strings,
		    string_src_reg, string_dest_reg, register_args)
drisc_ctx c;
IOgetFieldStruct tmp_spec;
int assume_align;
dr_reg_t src_addr;
int src_offset;
int dest_size;
dr_reg_t dest_addr;
int dest_offset;
int string_offset_size;
dr_reg_t final_string_base;
dr_reg_t src_string_base;
int base_size_delta;
int converted_strings;
dr_reg_t *string_src_reg;
dr_reg_t *string_dest_reg;
int register_args;
{
    iogen_oprnd src_oprnd;
    int src_drisc_type;
    int dst_drisc_type;
    int src_required_align;
    int dst_required_align;
    int src_is_aligned = 1;
    int dst_is_aligned = 1;
    int null_target = dr_genlabel(c);


    src_drisc_type = drisc_type(tmp_spec);
    dst_drisc_type = DR_P;
    src_required_align = TYPE_ALIGN(src_drisc_type);
    dst_required_align = TYPE_ALIGN(dst_drisc_type);
    if ((assume_align < src_required_align) || 
	((src_offset % src_required_align) != 0)) {
	src_is_aligned = 0;
    }
    if ((assume_align < dst_required_align) || 
	((dest_offset % dst_required_align) != 0)) {
	dst_is_aligned = 0;
    }


    /* handle addresses */
    src_oprnd = gen_fetch(c, src_addr, src_offset, tmp_spec.size,
			  integer_type, src_is_aligned,
			  tmp_spec.byte_swap);

    /* src_oprnd now holds the offset value */
    if (dest_size >= sizeof(char *)) {

	if (src_oprnd.size != dest_size) {
	    /* make it the right size to operate on */
	    iogen_oprnd tmp_oprnd;
	    tmp_oprnd = gen_size_conversion(c, src_oprnd, sizeof(long));
	    free_oprnd(c, src_oprnd);
	    src_oprnd = tmp_oprnd;
	}
	/* generate : if it's zero, leave it zero  branch away */
	dr_beqli(c, src_oprnd.vc_reg, 0, null_target);

	/* else, sub the string_offset_size */
	dr_subli(c, src_oprnd.vc_reg, src_oprnd.vc_reg,
		string_offset_size);
 
	/* Moving to here to more effiecntly use registers */   
	if (!dr_getreg(c, string_src_reg, DR_P, DR_VAR))
	  gen_fatal("gen field convert out of registers C\n");
	if (!dr_getreg(c, string_dest_reg, DR_P, DR_VAR))
	  gen_fatal("gen field convert out of registers D\n");
	REG_DEBUG(("Getting reg %d for string src reg\n", *string_src_reg));
	REG_DEBUG(("Getting reg %d for string dest reg\n", *string_dest_reg));

	/* calculate the address of this in the source */
	if (register_args) {
	    dr_addl(c, *string_src_reg, src_oprnd.vc_reg, src_string_base);
	} else {
	    dr_ldpi(c, *string_src_reg, dr_lp(c), src_string_base);
	    dr_addl(c, *string_src_reg, src_oprnd.vc_reg, *string_src_reg);
	}
	    
	/* and the address in the destination */
	if (register_args) {
	    dr_addl(c, src_oprnd.vc_reg, src_oprnd.vc_reg, final_string_base);
	} else {
	    dr_ldpi(c, *string_dest_reg, dr_lp(c), final_string_base);
	    dr_addl(c, src_oprnd.vc_reg, src_oprnd.vc_reg, *string_dest_reg);
	}
	dr_label(c, null_target);

	dr_movp(c, *string_dest_reg, src_oprnd.vc_reg);

	if (dest_size > sizeof(char *)) {
	    iogen_oprnd tmp_oprnd;
	    tmp_oprnd = gen_size_conversion(c, src_oprnd,
					    dest_size);
	    free_oprnd(c, src_oprnd);
	    src_oprnd = tmp_oprnd;
	}
	/* store the final address in the field */
	gen_store(c, src_oprnd, dest_addr, dest_offset, dest_size,
		  integer_type, dst_is_aligned);
	free_oprnd(c, src_oprnd);
	REG_DEBUG(("Regs %d and %d are src and dest \n",
		   _vrr(*string_src_reg), _vrr(*string_dest_reg)));
    } else {

        assert(FALSE);
	/* Removed 4/20/00 per Greg Eisenhauer - 
	   this is special case handling that may not work anymore anyway -- 
	   if you want it to work againlater, add the register allocation for 
	   string_src_reg and string_dest_reg */

	if (src_oprnd.size != dest_size) {
	    iogen_oprnd tmp_oprnd;
	    tmp_oprnd = gen_size_conversion(c, src_oprnd, dest_size);
	    free_oprnd(c, src_oprnd);
	    src_oprnd = tmp_oprnd;
	}
	/* dest field too small to store pointer in */
	/* store as integer offset */
	gen_store(c, src_oprnd, dest_addr, 0, dest_size,
		  integer_type, dst_is_aligned);

	/* generate : if it's zero, leave it zero  branch away */
	dr_beqli(c, src_oprnd.vc_reg, 0, null_target);

	/* else, sub the string_offset_size */
	dr_subli(c, src_oprnd.vc_reg, src_oprnd.vc_reg,
		string_offset_size);

	/* calculate the address of this in the source */
	dr_addl(c, *string_src_reg, src_oprnd.vc_reg, src_string_base);

	/* and the address in the destination */
	dr_addl(c, src_oprnd.vc_reg, src_oprnd.vc_reg,
	       final_string_base);

	dr_label(c, null_target);

	dr_movp(c, *string_dest_reg, src_oprnd.vc_reg);

	gen_store(c, src_oprnd, dest_addr, dest_offset, dest_size,
		  integer_type, dst_is_aligned);
	free_oprnd(c, src_oprnd);
    }
}

static void
gen_subfield_conv(c, conv, i, tmp_spec, dest_type, assume_align, src_addr,
	     src_offset, dest_size, dest_addr, dest_offset, control_base,
		  final_string_base, src_string_base, register_args)
drisc_ctx c;
IOConversionPtr conv;
int i;
IOgetFieldStruct tmp_spec;
IOdata_type dest_type;
int assume_align;
dr_reg_t src_addr;
int src_offset;
int dest_size;
dr_reg_t dest_addr;
int dest_offset;
int control_base;
dr_reg_t final_string_base;
dr_reg_t src_string_base;
int register_args;
{
    if (dest_type != unknown_type) {
	/* simple (native) field or variant array */
	if (dest_type != string_type) {
	    gen_simple_field_conv(c, tmp_spec, assume_align,
				  src_addr, src_offset,
				  dest_size, dest_type,
				  dest_addr, dest_offset,
				  conv->string_offset_size,
				  final_string_base,
				  conv->base_size_delta,
				  conv->converted_strings);
	} else {
	    dr_reg_t string_src_reg;
	    dr_reg_t string_dest_reg;
	    gen_addr_field_conv(c, tmp_spec, assume_align,
				src_addr, src_offset,
				dest_size, dest_addr,
				dest_offset,
				conv->string_offset_size,
				final_string_base,
				src_string_base,
				conv->base_size_delta,
				conv->converted_strings,
				&string_src_reg,
				&string_dest_reg,
				register_args);
	    if ((conv->conversion_type == copy_strings) ||
		(conv->conversions[i].control_field != NULL)) {
		if (register_args) {
		    /* many register case */
		    gen_var_part_conv(c, conv, i, control_base,
				      assume_align,
				      string_dest_reg,
				      string_src_reg,
				      src_string_base,
				      final_string_base,
				      register_args);
		    REG_DEBUG(("Putting reg %d for string src reg\n", string_src_reg));
		    REG_DEBUG(("Putting reg %d for string dest reg\n", string_dest_reg));
		    dr_putreg(c, string_src_reg, DR_P);
		    dr_putreg(c, string_dest_reg, DR_P);
		} else {
		    /*         limited register case  :
		     * save old src and dest values so we can reuse those
		     * registers for the conversion of the variable part
		     */
		    int src_storage = dr_local(c, DR_P);
		    int dest_storage = dr_local(c, DR_P);
		    dr_stpi(c, src_addr, dr_lp(c), src_storage);
		    dr_stpi(c, dest_addr, dr_lp(c), dest_storage);
		    dr_movp(c, src_addr, string_src_reg);
		    dr_movp(c, dest_addr, string_dest_reg);
		    dr_putreg(c, string_src_reg, DR_P);
		    dr_putreg(c, string_dest_reg, DR_P);
		    REG_DEBUG(("Putting reg %d for string src reg\n", string_src_reg));
		    REG_DEBUG(("Putting reg %d for string dest reg\n", string_src_reg));
		    gen_var_part_conv(c, conv, i, control_base,
				      assume_align,
				      dest_addr,
				      src_addr,
				      src_string_base,
				      final_string_base,
				      register_args);
		    /* restore values of src_addr and dest_addr */
		    dr_ldpi(c, src_addr, dr_lp(c), src_storage);
		    dr_ldpi(c, dest_addr, dr_lp(c), dest_storage);
		}
	    } else {
		REG_DEBUG(("Putting reg %d for string src reg\n", string_src_reg));
		REG_DEBUG(("Putting reg %d for string dest reg\n", string_dest_reg));
		dr_putreg(c, string_src_reg, DR_P);
		dr_putreg(c, string_dest_reg, DR_P);
	    }
	}
    } else {
	/* subconversion */
	IOConversionPtr subconv =
	    conv->conversions[i].subconversion;
	if ((assume_align == 8) &&
	    ((src_offset % subconv->required_alignment) == 0)) {
	    if (register_args) {
		/* many register case */
		dr_reg_t new_src, new_dest;
		if (!dr_getreg(c, &new_src, DR_P, DR_TEMP) ||
		    !dr_getreg(c, &new_dest, DR_P, DR_TEMP))
		    gen_fatal("temp vals in subcall\n");
		REG_DEBUG(("Getting %d and %d for new src & dest\n", 
			   new_src, new_dest));
		dr_addpi(c, new_src, src_addr, src_offset);
		dr_addpi(c, new_dest, dest_addr, dest_offset);
		VCALL4V(subconv->conv_func, "%p%p%p%p", new_src,
			 new_dest, final_string_base, src_string_base);
		REG_DEBUG(("Putting %d and %d for new src & dest\n", 
			   new_src, new_dest));
		dr_putreg(c, new_src, DR_P);
		dr_putreg(c, new_dest, DR_P);
	    } else {
		int src_storage = dr_local(c, DR_P);
		int dest_storage = dr_local(c, DR_P);
		dr_reg_t reg_src_string_base, reg_final_string_base;

		/* save values of src_addr and dest_addr */
		dr_stpi(c, src_addr, dr_lp(c), src_storage);
		dr_stpi(c, dest_addr, dr_lp(c), dest_storage);

		if (!dr_getreg(c, &reg_src_string_base, DR_P, DR_TEMP) ||
		    !dr_getreg(c, &reg_final_string_base, DR_P, DR_TEMP))
		    gen_fatal("temp string vals in subcall\n");
		REG_DEBUG(("Getting %d and %d for reg src base & reg dest base\n", reg_src_string_base, reg_final_string_base));
		dr_addpi(c, src_addr, src_addr, src_offset);
		dr_addpi(c, dest_addr, dest_addr, dest_offset);
		dr_ldpi(c, reg_final_string_base, dr_lp(c), final_string_base);
		dr_ldpi(c, reg_src_string_base, dr_lp(c), src_string_base);
		VCALL4V(subconv->conv_func, "%p%p%p%p", src_addr,
			 dest_addr, reg_final_string_base, reg_src_string_base);
		REG_DEBUG(("Putting %d and %d for reg src base & reg dest base\n", reg_src_string_base, reg_final_string_base));
		dr_putreg(c, reg_src_string_base, DR_P);
		dr_putreg(c, reg_final_string_base, DR_P);
		/* restore values of src_addr and dest_addr */
		dr_ldpi(c, src_addr, dr_lp(c), src_storage);
		dr_ldpi(c, dest_addr, dr_lp(c), dest_storage);
	    }
	} else {
	    /* misaligned substructure, can't call subconversion */
	    dr_reg_t args[4];
	    args[0] = src_addr;
	    args[1] = dest_addr;
	    args[2] = final_string_base;
	    args[3] = src_string_base;
	    generate_conversion_code(c, subconv, args, assume_align, 
				     register_args, src_offset, dest_offset);
	}
    }
}

extern
 conv_routine
generate_conversion_code(c, conv, args, assume_align, register_args, 
			 extra_src_offset, extra_dest_offset)
drisc_ctx c;
IOConversionPtr conv;
dr_reg_t *args;
int assume_align;
int register_args;
int extra_src_offset;
int extra_dest_offset;
{
    int i;
    dr_reg_t src_addr = args[0];
    dr_reg_t dest_addr = args[1];
    dr_reg_t final_string_base = args[2];
    dr_reg_t src_string_base = args[3];
    int control_base = -1;

    for (i = 0; i < conv->conv_count; i++) {
	iogen_oprnd src_oprnd;
	if (conv->conversions[i].control_field != NULL) {
	    IOFieldPtr control_field = conv->conversions[i].control_field;
	    int src_is_aligned = 1;
	    int src_drisc_type = drisc_type(*control_field);
	    int src_required_align = TYPE_ALIGN(src_drisc_type);
	    if ((assume_align < src_required_align) || 
		(((control_field->offset +extra_src_offset) % src_required_align) == 0)) {
		src_is_aligned = 0;
	    }
	    if (control_base == -1) {
		control_base = dr_localb(c, sizeof(int) * conv->conv_count);
	    }
	    src_oprnd = gen_fetch(c, src_addr, 
				  control_field->offset + extra_src_offset,
				  control_field->size,
				  control_field->data_type,
				  src_is_aligned, control_field->byte_swap);
	    if (src_oprnd.size != sizeof(int)) {
		iogen_oprnd tmp_oprnd;
		tmp_oprnd = gen_size_conversion(c, src_oprnd, sizeof(int));
		free_oprnd(c, src_oprnd);
		src_oprnd = tmp_oprnd;
	    }
	    gen_store(c, src_oprnd, dr_lp(c), control_base + i * sizeof(int),
		      sizeof(int), integer_type, TRUE /* aligned */ );
	    free_oprnd(c, src_oprnd);
	}
    }
    for (i = 0; i < conv->conv_count; i++) {
	IOFieldPtr src_spec = &conv->conversions[i].src_field;
	int dimen1 = conv->conversions[i].array_dimen1;
	int dimen2 = conv->conversions[i].array_dimen2;
	int byte_swap = conv->conversions[i].src_field.byte_swap;
#ifdef HAVE_VCODE_H
	if (generation_verbose) {
	    printf("Conversion %d in format %s starts at %lx\n",
		   i, conv->ioformat->body->format_name, (long)v_ip);
	}
#endif
	if (conv->conversions[i].src_field.size == 1) byte_swap = 0;
	if (!byte_swap &&
	    (src_spec->size == conv->conversions[i].dest_size) &&
	    (conv->conversions[i].subconversion == NULL) &&
	    (conv->conversions[i].control_field == NULL) &&
	    ((src_spec->data_type != string_type))) {
	    /* data movement is all that is required */
	    int total_size = src_spec->size * dimen1 * dimen2;

	    if (total_size <= sizeof(long)) {
		iogen_oprnd src_oprnd;
		int src_is_aligned = (assume_align >= sizeof(long)) &
		    (((src_spec->offset +extra_src_offset) % 8) == 0);
		int dst_is_aligned = (assume_align >= sizeof(long)) &
		    (((conv->conversions[i].dest_offset + extra_dest_offset) % 8) == 0);
		src_oprnd = gen_fetch(c, src_addr, 
				      src_spec->offset + extra_src_offset,
				      total_size, src_spec->data_type,
				      src_is_aligned, 0);
		gen_store(c, src_oprnd, dest_addr,
			  conv->conversions[i].dest_offset+extra_dest_offset,
			  conv->conversions[i].dest_size,
			  src_spec->data_type, dst_is_aligned);
		free_oprnd(c, src_oprnd);
	    } else {
		dr_reg_t size_reg;
		if (!dr_getreg(c, &size_reg, DR_I, DR_TEMP))
		    gen_fatal("gen field convert out of registers AA\n");
		REG_DEBUG(("Getting %d for memcpy\n", size_reg));
		dr_seti(c, size_reg, total_size);
		gen_memcpy(c, src_addr, src_spec->offset + extra_src_offset, 
			   dest_addr,
			   conv->conversions[i].dest_offset + extra_dest_offset,
			   size_reg);
		REG_DEBUG(("Putting %d for memcpy\n", size_reg));
		dr_putreg(c, size_reg, DR_I);
	    }
	} else {
	    dr_reg_t loop_var;
	    int loop;

	    int dest_offset = conv->conversions[i].dest_offset + extra_dest_offset;
	    int src_offset = src_spec->offset + extra_src_offset;
	    IOdata_type dest_type = src_spec->data_type;
	    int dest_size = conv->conversions[i].dest_size;
	    IOgetFieldStruct tmp_spec;
	    tmp_spec = *src_spec;
	    tmp_spec.offset = 0;

	    if (conv->conversions[i].control_field != NULL) {
		/* really a variant array, adjust address like a string */
		tmp_spec.data_type = string_type;
		tmp_spec.size = conv->ioformat->body->pointer_size;
		dest_type = string_type;
		dest_size = conv->target_pointer_size;
	    }
	    if (dimen1 * dimen2 == 1) {
		gen_subfield_conv(c, conv, i, tmp_spec, dest_type, 
				  assume_align, src_addr, src_offset,
				  dest_size, dest_addr, dest_offset,
				  control_base,
				  final_string_base, src_string_base,
				  register_args);
	    } else {
		int src_storage = dr_local(c, DR_P);
		int dest_storage = dr_local(c, DR_P);
		int loop_storage = 0;
		int src_align_offset = src_offset %
		    subfield_required_align(c, conv, i, 0);
		/* save values of src_addr and dest_addr */
		dr_stpi(c, src_addr, dr_lp(c), src_storage);
		dr_stpi(c, dest_addr, dr_lp(c), dest_storage);

		if (((conv->conversions[i].control_field != NULL) ||
		    (conv->conversions[i].subconversion == NULL)) &&
		    !debug_code_generation) {
		    if (!dr_getreg(c, &loop_var, DR_I, DR_TEMP))
			gen_fatal("gen field convert out of registers BB \n");
		} else {
		    /* may call a subconversion in here, use VARs */
		    if (!dr_getreg(c, &loop_var, DR_I, DR_VAR))
			gen_fatal("gen field convert out of registers CC\n");
		}
		REG_DEBUG(("Getting %d as loop_var\n", _vrr(loop_var)));
		dr_addpi(c, src_addr, src_addr, src_offset - src_align_offset);
		dr_addpi(c, dest_addr, dest_addr, dest_offset);

		if (conv->conversions[i].control_field != NULL) {
		    /* 
		     * really a variant array, adjust address like a string 
		     */
		    tmp_spec.data_type = string_type;
		    tmp_spec.size = conv->ioformat->body->pointer_size;
		}
		/* gen conversion loop */
		loop = dr_genlabel(c);
		dr_seti(c, loop_var, dimen1 * dimen2);
		dr_label(c, loop);
		if (!register_args) {
		    /* store away loop var and free the reg */
		    loop_storage = dr_local(c, DR_I);
		    dr_stii(c, loop_var, dr_lp(c), loop_storage);
		    REG_DEBUG(("Putting %d as loop_var\n", _vrr(loop_var)));
		    dr_putreg(c, loop_var, DR_I);
		}
		gen_subfield_conv(c, conv, i, tmp_spec, dest_type, assume_align,
				  src_addr, src_align_offset,
				  dest_size, dest_addr, 0,
				  control_base,
				  final_string_base, src_string_base,
				  register_args);

		/* generate end of loop */
		if (!register_args) {
		    /* store away loop var and free the reg */
		    dr_getreg(c, &loop_var, DR_I, DR_TEMP);
		    REG_DEBUG(("Getting %d as loop_var\n", _vrr(loop_var)));
		    dr_ldii(c, loop_var, dr_lp(c), loop_storage);
		}
		dr_subli(c, loop_var, loop_var, 1);
		dr_addpi(c, src_addr, src_addr, tmp_spec.size);
		dr_addpi(c, dest_addr, dest_addr,
			conv->conversions[i].dest_size);
		if (debug_code_generation) {
		    VCALL4V(printf, "%P%p%p%p",
			     "loopvar = %x, src %x, dest %x\n", loop_var,
			     src_addr, dest_addr);
		}
		dr_bgtli(c, loop_var, 0, loop);
		dr_putreg(c, loop_var, DR_I);
		REG_DEBUG(("Putting %d as loop_var\n", _vrr(loop_var)));

		/* restore values of src_addr and dest_addr */
		dr_ldpi(c, src_addr, dr_lp(c), src_storage);
		dr_ldpi(c, dest_addr, dr_lp(c), dest_storage);
	    }
	}
    }
    return NULL;
}

static void
gen_var_part_conv(c, conv, i, control_base, assume_align, dest_addr,
		  src_addr, src_string_base, final_string_base, 
		  register_args)
drisc_ctx c;
IOConversionPtr conv;
int i;
int control_base;
int assume_align;
dr_reg_t src_addr, dest_addr;
dr_reg_t src_string_base, final_string_base;
int register_args;
{
    /* 
     * This section handles converting or moving the body 
     * of a dynamic array or string (the actual pointer was handled 
     * above)
     */
/** handle copying or conversion */
    if (debug_code_generation) {
	if (register_args) {
	    VCALL6V(printf, "%P%I%p%p%p%p",
		     "varconvpart conversion %d src %lx, dest %lx, src_string_base %lx, final_string_base %lx\n",
		     i, src_addr, dest_addr, 
		     src_string_base, final_string_base);
	} else {
#ifdef HAVE_DRISC_H	    
	    dr_reg_t v_at;
	    if (dr_getreg(c, &v_at, DR_I, DR_TEMP) == 0) {
		gen_fatal("Out of regs 1\n");
	    }
#endif
	    VCALL5V(printf, "%P%P%I%p%p",
		     "varconvpart conversion \"%s\"%d src %lx, dest %lx\n",
		     conv->ioformat->body->field_list[i].field_name,
		     i, src_addr, dest_addr);
	    dr_ldpi(c, v_at, dr_lp(c), src_string_base);
	    VCALL2V(printf, "%P%p",
		     "                       src_string_base %lx\n",
		     v_at);
	    dr_ldpi(c, v_at, dr_lp(c), final_string_base);
	    VCALL2V(printf, "%P%p",
		     "                       final_string_base %lx\n",
		     v_at);
#ifdef HAVE_DRISC_H	    
	    dr_putreg(c, v_at, DR_I);
#endif
	}
    }
    if (conv->conversions[i].control_field == NULL) {
	/* generate a call to strcpy to do the move */
	int end = dr_genlabel(c);
	dr_beqpi(c, src_addr, 0, end);
	VCALL2V(strcpy, "%p%p", dest_addr, src_addr);
	dr_label(c, end);
    } else {
	IOFieldPtr src_spec = &conv->conversions[i].src_field;
	/* variant array */
	int byte_swap = conv->conversions[i].src_field.byte_swap;
	if (conv->conversions[i].src_field.size == 1) byte_swap = 0;
	if (!byte_swap &&
	    (src_spec->size == conv->conversions[i].dest_size) &&
	    (conv->conversions[i].subconversion == NULL) &&
	    (src_spec->data_type != string_type)) {

	    /* data movement is all that is required */
	    if (conv->conversion_type == copy_strings) {
		dr_reg_t size_reg;
		if (!dr_getreg(c, &size_reg, DR_I, DR_TEMP))
		    gen_fatal("gen var convert size out of registers BB \n");
		REG_DEBUG(("Getting %d as size_reg\n", _vrr(size_reg)));
		/* load control (array size) value */
		dr_ldii(c, size_reg, dr_lp(c), control_base + i * sizeof(int));
		/* scale it by destination size */
		dr_mulii(c, size_reg, (dr_reg_t)size_reg,
			conv->conversions[i].dest_size);
		/* generate a memcpy */
		gen_memcpy(c, src_addr, 0, dest_addr, 0, size_reg);
		dr_putreg(c, size_reg, DR_I);
		REG_DEBUG(("Putting %d as size_reg\n", _vrr(size_reg)));
	    } else {
		/* 
		 * not even that is necessary, variant part is
		 * already where it needs to go 
		 */
	    }
	} else {
	    /* must do conversions one by one */
	    int src_storage = dr_local(c, DR_P);
	    int dest_storage = dr_local(c, DR_P);
	    int loop_storage = dr_local(c, DR_I);
	    dr_reg_t loop_var;
	    int dest_size = conv->conversions[i].dest_size;
	    int loop = dr_genlabel(c);
	    int end = dr_genlabel(c);
	    IOdata_type dest_type = src_spec->data_type;

	    /* save values of src_addr and dest_addr */
	    dr_stpi(c, src_addr, dr_lp(c), src_storage);
	    dr_stpi(c, dest_addr, dr_lp(c), dest_storage);

	    if (((conv->conversions[i].control_field != NULL) ||
		 (conv->conversions[i].subconversion == NULL)) &&
		!debug_code_generation) {
		if (!dr_getreg(c, &loop_var, DR_I, DR_VAR))
		    gen_fatal("gen var convert loop out of registers CC\n");
	    } else {
		/* may call a subconversion in here, use VARs */
		if (!dr_getreg(c, &loop_var, DR_I, DR_VAR))
		    gen_fatal("gen var convert loop out of registers DD\n");
	    }
	    REG_DEBUG(("Getting %d as var_loop_var\n", _vrr(loop_var)));
	    dr_ldii(c, loop_var, dr_lp(c), control_base + i * sizeof(int));

	    if ((dest_size - src_spec->size) != 0) {
		dr_reg_t tmp;
		if (!dr_getreg(c, &tmp, DR_I, DR_TEMP)) {
		    gen_fatal("in var loop EE");
		}
		REG_DEBUG(("Getting %d as tmp str adjust\n", tmp));
		dr_mulii(c, tmp, loop_var, (dest_size - src_spec->size));

		/* adjust string base !!!!  non-local variation */
		/* this is because new variant part may change size */
		if (register_args) {
		    dr_addi(c, final_string_base, final_string_base, tmp);
		} else {
#ifdef HAVE_DRISC_H	    
		    dr_reg_t v_at;
		    if (dr_getreg(c, &v_at, DR_I, DR_TEMP) == 0) {
			gen_fatal("Out of regs 1\n");
		    }
#endif
		    dr_ldpi(c, v_at, dr_lp(c), final_string_base);
		    if (debug_code_generation) {
			VCALL2V(printf, "%P%p",
				 "before adjustment    final_string_base %lx\n",
				 v_at);
		    }
		    dr_ldpi(c, v_at, dr_lp(c), final_string_base);
		    dr_addi(c, v_at, v_at, tmp);
		    dr_stpi(c, v_at, dr_lp(c), final_string_base);
		    dr_ldpi(c, v_at, dr_lp(c), final_string_base);
		    if (debug_code_generation) {
			VCALL2V(printf, "%P%p",
				 "after adjustment    final_string_base %lx\n",
				 v_at);
		    }
#ifdef HAVE_DRISC_H	    
		    dr_putreg(c, v_at, DR_I);
#endif
		}
		dr_putreg(c, tmp, DR_I);
		REG_DEBUG(("Putting %d as tmp str adjust\n", tmp));
	    }
	    dr_label(c, loop);
	    dr_beqli(c, loop_var, 0, end);
	    if (debug_code_generation) {
		VCALL4V(printf, "%P%i%p%p",
			 "varloopvar = %x, src %x, dest %x\n",
			 loop_var, src_addr, dest_addr);
	    }
	    if (!register_args) {
		/* store away loop var and free the reg */
		dr_stii(c, loop_var, dr_lp(c), loop_storage);
		dr_putreg(c, loop_var, DR_I);
		REG_DEBUG(("Putting %d as loop_var\n", loop_var));
	    }
	    dr_savep(c, src_addr);
	    dr_savep(c, dest_addr);
	    dr_savep(c, loop_var);
	    if (dest_type != string_type) {
		int subelement_align = 1;
		if (conv->conversions[i].subconversion != NULL) {
		    int required_align = conv_required_alignment(c, conv->conversions[i].subconversion);
		    int src_structure_size =
			conv->conversions[i].src_field.size;
		    if ((src_structure_size % required_align) != 0) {
			subelement_align = 0;
		    }
		}
		gen_subfield_conv(c, conv, i, *src_spec, dest_type,
				  subelement_align, src_addr, 0, dest_size,
				  dest_addr, 0,
				  control_base, final_string_base,
				  src_string_base, register_args);
	    } else {
		IOConversionStruct tmp_conv = *conv;
		tmp_conv.conv_count = 0;
		tmp_conv.conversions[0] = conv->conversions[i];
		tmp_conv.conversions[0].control_field = NULL;
		gen_subfield_conv(c, &tmp_conv, 0, *src_spec, dest_type,
				  assume_align, src_addr, 0, dest_size,
				  dest_addr, 0,
				  control_base, final_string_base,
				  src_string_base, register_args);
	    }
	    dr_restorep(c, loop_var);
	    dr_restorep(c, dest_addr);
	    dr_restorep(c, src_addr);
	    if (!register_args) {
		/* store away loop var and free the reg */
		dr_getreg(c, &loop_var, DR_I, DR_TEMP);
		REG_DEBUG(("Getting %d as lop_var FF\n", loop_var));
		dr_ldii(c, loop_var, dr_lp(c), loop_storage);
	    }
	    dr_subli(c, loop_var, loop_var, 1);
	    dr_addpi(c, src_addr, src_addr, src_spec->size);
	    dr_addpi(c, dest_addr, dest_addr,
		    conv->conversions[i].dest_size);
	    dr_jv(c, loop);
	    dr_putreg(c, loop_var, DR_I);
	    REG_DEBUG(("Putting %d as lop_var\n", loop_var));
	    dr_label(c, end);
	    /* restore values of src_addr and dest_addr */
	    dr_ldpi(c, src_addr, dr_lp(c), src_storage);
	    dr_ldpi(c, dest_addr, dr_lp(c), dest_storage);
	}
    }
}

#endif
