#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_WINDOWS_H
#include <windows.h>
#define FD_SETSIZE 1024
#include <winsock.h>
#define getpid()	_getpid()
#define __ANSI_CPP__
#else
#include <unistd.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_TIMES_H
#include <sys/times.h>
#endif
#include <sys/socket.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#ifdef HAVE_SYS_UIO_H
#include <sys/uio.h>
#endif
#ifdef HAVE_SYS_UN_H
#include <sys/un.h>
#endif
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#include <signal.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#include "io.h"
#include "io_interface.h"
#include "io_internal.h"
#include "unix_defs.h"
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#include <string.h>
#include <assert.h>

typedef struct format_list {
    IOFormat	format;
    time_t	last_reference;
    struct format_list *next;
} format_list;

typedef struct _format_server {
    int port;			/* server port as announced to world */
    char *hostname;		/* server's hostname */
    void *conn_sock_inet;	/* public sock for IP socket connection */
    char *usock_name;		/* file name for unix socket connection */
    int portCount;		/* number of current ports */
    FSClient *ports;	/* array of ports index by socket fd */
    time_t *timestamp;		/* array of timestamps */
    fd_set fdset;		/* bitmap of those conns */
    char *data_buffer;		/* buffer for data */
    int buffer_size;
    format_list	*lists[2][1];    /* first dimension byte order, second float format */
} _fsserver;

static FSClient
format_server_accept_conn_sock ARGS((format_server fs, void *conn_sock));
static format_server format_server_create();
static void format_server_handle_data ARGS((format_server fs, FSClient fsc));
static int format_server_listen ARGS((format_server fs, int port_num));
static int format_server_poll_and_handle ARGS((format_server fs));
static void timeout_old_conns();
static void timeout_oldest_conn();

#ifndef HAVE_WINDOWS_H
#ifndef SELECT_DEFINED
extern int select ARGS((int width, fd_set *readfds, fd_set *writefds, 
			fd_set *exceptfds, struct timeval *timeout));
#endif
#ifndef BIND_DEFINED
extern int bind ARGS((int s, struct sockaddr *name, int namelen));
#endif
#ifndef LISTEN_DEFINED
extern int listen ARGS((int s, int backlog));
#endif
#ifndef ACCEPT_DEFINED
extern int accept ARGS((int s, struct sockaddr *addr, int *addrlen));
#endif
extern int socket ARGS((int domain, int type, int protocol));
#ifndef CONNECT_DEFINED
extern int connect ARGS((int s, struct sockaddr *name, int namelen));
#endif
#ifndef SETSOCKOPT_DEFINED
extern int setsockopt ARGS((int s, int level, int optname, const char *optval, int optlen));
#endif
#ifndef GETSOCKNAME_DEFINED
extern int getsockname ARGS((int s, struct sockaddr *name, int *namelen));
#endif
#ifndef GETHOSTNAME_DEFINED
extern int gethostname ARGS((char *name, int namelen));
#endif
#endif
#if defined(HAVE_GETDOMAINNAME) && !defined(GETDOMAINNAME_DEFINED)
extern int getdomainname ARGS((char *name, int namelen));
#endif
#if !defined(GETPEERNAME_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int getpeername ARGS((int, struct sockaddr *, int *));
#endif
#ifdef STDC_HEADERS
#include <stdarg.h>
#else
#include <varargs.h>
#endif
extern time_t time();
extern pid_t getpid();

static int no_fork = 0;
static int quiet = 0;

static void dump_stats_to_log();

static FILE* log = (void*) 0;

extern void
LOG(char *format, ...)
{
    va_list ap;
    time_t now;
    struct stat stat_buf;

    if ((log == (void*)-1) ||  (log == (void*)0)){
	int restart = 0;
	if (log == (void*)0) {	
	    restart++;
	    if (stat("/users/c/chaos/bin/format_server_report", &stat_buf) == 0){
		system("/users/c/chaos/bin/format_server_report restart");
	    }
	}
	log = fopen("/tmp/format_server_log", "a");
	if (log == NULL) {
	    log = (void*)-1;
	    return;
	}
	if (restart) {
	    fprintf(log, "\n\nRestarting format server \n");
	}
    }
#ifdef STDC_HEADERS
    va_start(ap, format);
#else
    va_start(ap);
#endif
    time(&now);
    vfprintf(log, format, ap);
    va_end(ap);
    fprintf(log, "\n");
    fflush(log);
    stat("/tmp/format_server_log", &stat_buf);
    if (stat_buf.st_size > 102400) {
	dump_stats_to_log();
	fflush(log);
	fclose(log);
	if (stat("/users/c/chaos/bin/format_server_report", &stat_buf) == 0){
	    system("/users/c/chaos/bin/format_server_report log");
	}
	unlink("/tmp/format_server_log");
	log = (void*)-1;
    }
}

static void 
die_with_error(sig)
int sig;
{
    fprintf(stderr, "Format server not responding.  Timeout.\n");
    exit(1);
}

static format_server server_fs;

int
main(argc, argv)
int argc; char **argv;
{
    int i;
    format_server fs;
    IOFile test_file;
    char testing_char = 'T';

    for (i=1; i<argc; i++) {
	if (strcmp(argv[i], "-no_fork") == 0) {
	    no_fork++;
	} else if (strcmp(argv[i], "-quiet") == 0) {
	    quiet++;
	} else {
	    fprintf(stderr, "Unknown argument \"%s\"\n", argv[i]);
	    exit(1);
	}
    }
    
    /* test to see if format server is running */
    test_file = new_IOFile();
#ifndef HAVE_WINDOWS_H
    signal(SIGALRM, die_with_error);
    alarm(30);
#endif
    if (!quiet) {
	putenv(strdup("FORMAT_SERVER_VERBOSE=1"));
    }
    establish_server_connection(test_file, /* nofallback */ 0);
    if (AtomicWrite(test_file->server_fd, &testing_char, 1, test_file) 
	== 1) {
	/* already running */
	if (!quiet) {
	    printf("Use FORMAT_SERVER_HOST environment variable to change format server location\n");
	    printf("Format server already running.  Exiting\n");
	}
	exit(0);
#ifdef HAVE_FORK
    } else {
	/* make really, really certain there's no format_server running */
	FILE *format_server_pid_file = fopen("/tmp/format_server_pid", "r");
	alarm(0);
	if (format_server_pid_file != NULL) {
	    long server_pid;
	    if (fscanf(format_server_pid_file, "%lx", &server_pid) == 1) {
		if (kill(server_pid, 0) == 0 ) {
		    fprintf(stderr, "Format Server %lx not responding, but still running\n",
			   server_pid);
		    exit(0);
		}
	    }
	    fclose(format_server_pid_file);
	}

	if (!no_fork) {
	    if (fork() != 0) {
		/* I'm the parent, return now */
		exit(0);
	    }
	}
	format_server_pid_file = fopen("/tmp/format_server_pid", "w");
	if (format_server_pid_file) {
	    fprintf(format_server_pid_file, "%lx\n", (long)getpid());
	    fclose(format_server_pid_file);
	}
	if (!no_fork) {
	    if (!quiet) {
		printf("Forking server to background\n");
	    }
	    if (fork() != 0) {
		/* I'm the parent, return now */
		exit(0);
	    }
	} else {
	    if (!quiet) {
		printf("Running...\n");
	    }
	}
#endif
    }

    {
#ifdef RLIMIT_NOFILE
	struct rlimit lim;
	if (getrlimit(RLIMIT_NOFILE, &lim) != 0) {
	    perror("Getrlimit");
	}
	lim.rlim_cur = lim.rlim_max;
	if (setrlimit(RLIMIT_NOFILE, &lim) != 0) {
	    perror("Setrlimit");
	}
	if (getrlimit(RLIMIT_CORE, &lim) != 0) {
	    perror("Setrlimit");
	}
	lim.rlim_cur = lim.rlim_max;
	if (setrlimit(RLIMIT_CORE, &lim) != 0) {
	    perror("Setrlimit");
	}
	chdir("/tmp");
#endif
    }
    server_fs = fs = format_server_create();
    if (format_server_listen(fs, FS_PORT) != -1) {
	while (1) {
	    format_server_poll_and_handle(fs);
	}
    }
    return 0;
}

static void FSClient_close(FSClient fsc);
#define CONN_TIMEOUT_INTERVAL 3600

static int
format_server_poll_and_handle(format_server fs)
{
    int i, res;
    fd_set rd_set = fs->fdset;
    struct timeval timeout;

#if !defined(__FD_SET) && !defined(HAVE_WINDOWS_H)
    LOG("  selecting, rd_set is %lx,%lx,%lx,%lx,%lx,%lx", rd_set.fds_bits[0],
	rd_set.fds_bits[1],rd_set.fds_bits[2],rd_set.fds_bits[3],
	rd_set.fds_bits[4],rd_set.fds_bits[5]);
#endif
    /* scan existing conn for incoming data & do something */
    timeout.tv_usec = 0;
    timeout.tv_sec = CONN_TIMEOUT_INTERVAL + 5;

    res = select(FD_SETSIZE, &rd_set, (fd_set *) NULL,
		 (fd_set *) NULL, &timeout);
    if (res == -1) {
	if (errno == EBADF) {
	    int i;
	    int found_one = 0;
	    for (i = 0; i< FD_SETSIZE; i++) {
		if (FD_ISSET(i, &rd_set)) {
		    fd_set test_set;
		    timeout.tv_usec = 0;
		    timeout.tv_sec = 0;
		    FD_ZERO(&test_set);
		    FD_SET(i, &test_set);
		    errno = 0;
		    select(FD_SETSIZE, &test_set, (fd_set*)NULL, 
			   (fd_set *) NULL, &timeout);
		    if (errno == EBADF) {
			fprintf(stderr, "Select failed, fd %d\n is bad.  Removing from select list.\n",
				i);
			LOG("REmoving bad fd %d", i);
			FSClient_close(fs->ports[i]);
			found_one++;
		    }
		}
	    }
	    if (found_one == 0) {
		fprintf(stderr, "Bad file descriptor in select.  Failed to localize.  Exiting.\n");
		exit(1);
	    }
	} else if (errno == EINTR) {
	    return 0;
	}
	fprintf(stderr, "select failed, errno %d\n", errno);
	return -1;
    } else if (res == 0) {
	timeout_old_conns();
	return 0;
    }

    if ((long)fs->conn_sock_inet >= 0 &&
	FD_ISSET((unsigned long)fs->conn_sock_inet, &rd_set)) {
	FD_CLR((unsigned long) fs->conn_sock_inet, &rd_set);
	format_server_accept_conn_sock(fs, fs->conn_sock_inet);
    }
    for (i = 0; i < FD_SETSIZE; i++) {	/* check each possible read fd */
	FSClient fsc = fs->ports[i];
	if (fsc == NULL || !FD_ISSET(i, &rd_set))
	    continue;
	fs->timestamp[i] = time(NULL);
	format_server_handle_data(fs, fsc);
	
    }
    timeout_old_conns();
    return res;
}

/*
typedef enum {
    write_record_state, write_format_state, not_blocked
} block_type;
extern unsigned alarm ARGS((unsigned));
static void record_blocking_state ARGS((block_type state,
					FSClient port));
static int block_count = 0;
static FSClient block_port = NULL;
static block_type block_state = not_blocked;

static void
clear_blocking_state()
{
    block_port = NULL;
    block_state = not_blocked;
    block_count++;
}

static void
record_blocking_state(state, port)
block_type state;
FSClient port;
{
    block_port = port;
    block_state = state;
}

static int last_block_count = 0;

static void
check_blocking_state(signal)
int signal;
{
    if ((block_count == last_block_count) && (block_state != not_blocked)
	&& (block_port != NULL)) {
	fprintf(stderr, "Warning!  No forward progress!!\n");
	fprintf(stderr, " Writing to port [%s, %d] \"%s\".\n",
		block_port->hostname, block_port->port,
		block_port->exchange_name);
    }
    last_block_count = block_count;
    alarm(1);
}


void
DEinit_block_check()
{
    struct sigaction sigact;
    sigact.sa_flags = 0;
    sigact.sa_handler = check_blocking_state;
    sigemptyset(&sigact.sa_mask);
    sigaddset(&sigact.sa_mask, SIGALRM);
    sigaction(SIGALRM, &sigact, NULL);
    alarm(1);
}
*/

static int
format_eq(form1, form2)
IOFormat form1;
IOFormat form2;
{
    int i;
    int all_zero = 1;
    int all_eq = 1;
    if (form1->body->server_ID.length != 0) {
	if (form1->body->server_ID.length != form2->body->server_ID.length) 
	    return 0;
	for (i=0 ; i< form1->body->server_ID.length; i++) {
	    if (form1->body->server_ID.value[i] != 0) {
		all_zero = 0;
	    }
	    if (form1->body->server_ID.value[i] != 
		form2->body->server_ID.value[i]) {
		all_eq = 0;
	    }
	}
    }
    if (all_eq && !all_zero) return 1;
    if ((form1->body->server_format_rep != 0) && (form2->body->server_format_rep != 0)) {
	/* both have server reps.  Equal if server reps are equal */
	if (form1->body->server_format_rep->format_rep_length !=
	    form2->body->server_format_rep->format_rep_length) return 0;
	return (memcmp(form1->body->server_format_rep, form2->body->server_format_rep,
		       form1->body->server_format_rep->format_rep_length) == 0);
    }	    
    if (form1->body->field_count != form2->body->field_count) return 0;
    if (strcmp(form1->body->format_name, form2->body->format_name) != 0) return 0;
    if (form1->body->record_length != form2->body->record_length) return 0;
    if (form1->body->pointer_size != form2->body->pointer_size) return 0;
    for (i=0; i< form1->body->field_count; i++) {
	if (strcmp(form1->body->field_list[i].field_name,
		   form2->body->field_list[i].field_name) != 0) return 0;
	if (strcmp(form1->body->field_list[i].field_type,
		   form2->body->field_list[i].field_type) != 0) return 0;
	if (form1->body->field_list[i].field_size != 
	    form2->body->field_list[i].field_size) return 0;
	if (form1->body->field_list[i].field_offset != 
	    form2->body->field_list[i].field_offset) return 0;
    }
    return 1;
}

static int server_format_count = 0;

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

#ifndef WORDS_BIGENDIAN
static void
byte_swap(data, size)
char *data;
int size;
{
    int i;
    assert((size % 2) == 0);
    for (i = 0; i < size / 2; i++) {
	char tmp = data[i];
	data[i] = data[size - i - 1];
	data[size - i - 1] = tmp;
    }
}
#endif

static IOFormat
find_format(fs, fsc, ioformat, new_format_mode, requested_id_version)
format_server fs;
FSClient fsc;
IOFormat ioformat;
int new_format_mode;
int requested_id_version;
{
    format_list *list = fs->lists[fsc->byte_reversal][0];
    format_list *last = NULL, *new = NULL;
    /* always, search list for this hosts byte order */
    while (list != NULL) {
	int server_ID_version = 
	    version_of_format_ID(list->format->body->server_ID.value);
	if (format_eq(ioformat, list->format) && 
	    (requested_id_version == server_ID_version)) {
	    free_IOformat(ioformat);
	    return list->format;
	}
	last = list;
	list = list->next;
    }
    if (new_format_mode) {
	static int pid = 0;
	static INT4 IP_addr = 0;
	static INT2 port = FS_PORT;
	char *tmp;
	if (pid == 0) {
	    struct hostent *host;
	    char buf[MAXHOSTNAMELEN];
	    pid = (int) getpid();
	    gethostname(buf, MAXHOSTNAMELEN);
	    host  = gethostbyname(buf);
	    IP_addr = htonl(*((int*) host->h_addr_list[0]));
	}
	/* didn't find it in the list, create a blank */
	new = (format_list*)io_malloc(sizeof(format_list));
	ioformat->format_id = server_format_count;
	new->format = ioformat;
	switch(requested_id_version) {
	case 0:
	    tmp = (char*)ioformat;
#ifndef WORDS_BIGENDIAN
	    byte_swap(tmp, sizeof(tmp));
#endif
	    tmp += (0x7 & pid);   /* add bottom 3 bits of pid to format id */
	    ioformat->body->server_ID.length = 8;
	    ioformat->body->server_ID.value = malloc(8);
	    memcpy(ioformat->body->server_ID.value, &tmp, sizeof(ioformat));
	    memset(ioformat->body->server_ID.value + sizeof(ioformat), 0, 
		   ioformat->body->server_ID.length - sizeof(ioformat));
	    if (sizeof(ioformat) == 8) {
		/* make sure the top byte is zero */
		*((char*)ioformat->body->server_ID.value) = 0;
	    }
	    break;
	case 1:
	    ioformat->body->server_ID.length = 10;
	    ioformat->body->server_ID.value = malloc(10);
	    ((version_1_format_ID *)ioformat->body->server_ID.value)->version = 1;
	    ((version_1_format_ID *)ioformat->body->server_ID.value)->salt = 
		(pid & 0xff);
	    ((version_1_format_ID *)ioformat->body->server_ID.value)->port = port;
	    ((version_1_format_ID *)ioformat->body->server_ID.value)->IP_addr = 
		IP_addr;
	    ((version_1_format_ID *)ioformat->body->server_ID.value)->format_identifier = ioformat->format_id;
	    break;
	}
	new->next = NULL;
	server_format_count++;
	if (last == NULL) {
	    fs->lists[fsc->byte_reversal][0] = new;
	} else {
	    last->next = new;
	}
	return ioformat;
    } else {
	/* search other list */
	int other_byte_order = fsc->byte_reversal ? 0 : 1;
	list = fs->lists[other_byte_order][0];
	while (list != NULL) {
	    if (format_eq(ioformat, list->format)) {
		free_IOformat(ioformat);
		return list->format;
	    }
	    last = list;
	    list = list->next;
	}
	return NULL;
    }
}

static void
server_write_id(fsc, ioformat)
FSClient fsc;
IOFormat ioformat;
{
    int junk_errno;
    char *junk_str;
    assert(ioformat->body->server_ID.length == 8);
    os_server_write_func(fsc->fd, ioformat->body->server_ID.value, 8, &junk_errno,
			 &junk_str);
}

/* 
 * Close an on-line connection.
 */
static void
FSClient_close(FSClient fsc)
{
    format_server fs = fsc->fs;
    int fd = (int) (long) fsc->fd;
    LOG("Closing client fd %d- %s", fd, fsc->hostname);
    if (fd != 0) {
	fs->ports[fd] = NULL;
	fs->timestamp[fd] = 0;
	FD_CLR((unsigned long)fd, &fs->fdset);
	os_close_func((void*)(long)fd);
	fs->portCount--;
    }
    io_free(fsc);
}

static int test_count = 0;
static int registration_count = 0;
static int get_count = 0;
static int select_handle_count = 0;

static void
format_server_handle_data(fs, fsc)
format_server fs;
FSClient fsc;
{
    IOFormat ioformat;
    char	next_action;
    LOG("    handle data, fd %d - %s", fsc->fd, fsc->hostname);
    select_handle_count++;
    if (serverAtomicRead(fsc->fd, &next_action, 1) != 1) {
	FSClient_close(fsc);
	return;
    }
    switch(next_action) {
    case 'T': 
	if (no_fork) {
	    printf("Testing\n");
	}
	LOG("    Testing ");
	test_count++;
	break;
    case 'F':
	if (no_fork) {
	    printf("Format Registration\n");
	}
	LOG("    Registration ");
	ioformat = server_read_format(fs, fsc);
	if (ioformat == NULL) {
	    FSClient_close(fsc);
	    return;
	}
	ioformat = find_format(fs, fsc, ioformat,
			       1 /* new format mode */, 0);
	if (no_fork) {
	    printf("Returning -> ");
	    print_format_ID(ioformat);
	    printf("\n");
	    dump_IOFormat(ioformat);
	}
	if (fsc->version > 0) server_write_char(fsc, "I");
	server_write_id(fsc, ioformat);
	registration_count++;
	break;
    case 'f': {
	char block_version;
	UINT2 length;

	format_rep rep;
	if (no_fork) {
	    printf("New style block format registration\n");
	}
	LOG("    Registration ");
	if (serverAtomicRead(fsc->fd, &block_version, 1) != 1) {
	    FSClient_close(fsc);
	    return;
	}
	if (block_version > 1) {
	    fprintf(stderr, "Unknown version in block registration\n");
	    FSClient_close(fsc);
	    return;
	}
	if (serverAtomicRead(fsc->fd, &length, sizeof(length)) !=
	    sizeof(length)) {
	    FSClient_close(fsc);
	    return;
	}
	length = ntohs(length);
	rep = io_malloc(length);
	rep->format_rep_length = length;
	if (serverAtomicRead(fsc->fd, ((char*)rep) + sizeof(length), 
			     length - sizeof(length)) !=
	    (length - sizeof(length))) {
	    FSClient_close(fsc);
	    return;
	}
	ioformat = expand_ioformat_from_rep(rep);
	ioformat = find_format(fs, fsc, ioformat,
			       1 /* new format mode */, block_version);
	if (no_fork) {
	    printf("Returning -> ");
	    print_format_ID(ioformat);
	    printf("\n");
	    dump_IOFormat(ioformat);
	}
	{
	    char ret[2];
	    if (fsc->provisional != 0) {
		ret[0] = 'P';
	    } else {
		ret[0] = 'I';
	    }
	    ret[1] = ioformat->body->server_ID.length;
	    if (os_server_write_func(fsc->fd, &ret[0], 2, NULL, NULL) != 2) {
		FSClient_close(fsc);
		return;
	    }
	    if (os_server_write_func(fsc->fd, ioformat->body->server_ID.value,
				     ioformat->body->server_ID.length, NULL, 
				     NULL) != ioformat->body->server_ID.length) {
		FSClient_close(fsc);
		return;
	    }
	}
	registration_count++;
	break;
    }
    case 'G':
	LOG("    Get format ");
	if (no_fork) printf("Get Format   -> ");
	ioformat = new_IOFormat();
	ioformat->body->field_list = NULL;
	ioformat->body->var_list = NULL;
	ioformat->body->field_count = 0;
	ioformat->body->format_name = NULL;
	ioformat->conversion = NULL;
	ioformat->body->server_format_rep = NULL;
	ioformat->body->server_ID.length = 8;
	ioformat->body->server_ID.value = malloc(8);
	server_get_server_ID(fsc->fd, ioformat->body->server_ID.value);
	if (no_fork) {
	    print_format_ID(ioformat);
	    printf("\n");
	}
	ioformat = find_format(fs, fsc, ioformat,
			     0 /* not new format mode */, 0);
	if (no_fork) {
	    printf("Returning -> ");
	    if (ioformat == NULL) {
		printf("	NULL\n");
	    } else {
		dump_IOFormat(ioformat);
	    }
	}
	if (fsc->provisional != 0) server_write_char(fsc, "P");
	if (fsc->version > 0) server_write_char(fsc, "F");
	write_format_to_fd(fsc->fd, ioformat);
	get_count++;
	break;
    case 'g': {
	char format_id_length;
	struct {
	    char reg[2];
	    unsigned short len;
	    } tmp = {{'f', 1}, 0 };	/* format reg, version 1 */
	int ret;
	int errno;
	char *errstr;
	LOG("    new style get format");
	if (no_fork) printf("new style Get Format   -> ");
	if (serverAtomicRead(fsc->fd, &format_id_length, 1) != 1) {
	    FSClient_close(fsc);
	    return;
	}
	ioformat = new_IOFormat();
	ioformat->body->field_list = NULL;
	ioformat->body->var_list = NULL;
	ioformat->body->field_count = 0;
	ioformat->body->format_name = NULL;
	ioformat->conversion = NULL;
	ioformat->body->server_format_rep = NULL;
	ioformat->body->server_ID.length = format_id_length;
	ioformat->body->server_ID.value = malloc(format_id_length);
	if (serverAtomicRead(fsc->fd, ioformat->body->server_ID.value,
			     format_id_length) != format_id_length) {
	    FSClient_close(fsc);
	    return;
	}
	if (no_fork) {
	    print_format_ID(ioformat);
	    printf("\n");
	}
	ioformat = find_format(fs, fsc, ioformat,
			     0 /* not new format mode */, 
			       version_of_format_ID(ioformat->body->server_ID.value));
	if (no_fork) {
	    printf("Returning -> ");
	    if (ioformat == NULL) {
		printf("	NULL\n");
	    } else {
		dump_IOFormat(ioformat);
	    }
	}
	
	if ((ioformat != NULL) && 
	    (ioformat->body->server_format_rep != NULL)){
	    tmp.len = htons(ioformat->body->server_format_rep->format_rep_length);
	} else {
	    tmp.len = 0;
	}
	ret = os_server_write_func(fsc->fd, &tmp, sizeof(tmp), 
				   &errno, &errstr);
	if (ret != sizeof(tmp)) {
	    FSClient_close(fsc);
	    break;
	}
	if ((ioformat != NULL) && 
	    (ioformat->body->server_format_rep != NULL)){
	    format_rep rep = ioformat->body->server_format_rep;
	    ret = os_server_write_func(fsc->fd, 
				       (char*)rep + sizeof(rep->format_rep_length),
				       rep->format_rep_length - sizeof(rep->format_rep_length),
				       &errno, &errstr);
	    if (ret != rep->format_rep_length - sizeof(rep->format_rep_length)) {
		FSClient_close(fsc);
		break;
	    }
	}
	get_count++;
	break;
    }
    case 'D':
	LOG("	Dump known format IDs");
	{
	    int junk_errno;
	    char *junk_str;
	    int out_count = 0, i;
	    int byte_order;
	    int tmp;
	    server_ID_type * out_list = malloc(sizeof(out_list[0]));
	    for (byte_order = 0; byte_order <= 1; byte_order++) {
		format_list *list = fs->lists[byte_order][0];
		while (list != NULL) {
		    out_list = realloc(out_list,  sizeof(out_list[0]) * 
				       (out_count + 1));
		    out_list[out_count] = list->format->body->server_ID;
		    out_count++;
		    list = list->next;
		}
	    }
	    tmp = htonl(out_count);
	    os_server_write_func(fsc->fd, &tmp, 4, &junk_errno,
				 &junk_str);
	    for (i=0 ; i < out_count; i++) {
		tmp = htonl(out_list[i].length);
		os_server_write_func(fsc->fd, &tmp, 4,
				     &junk_errno, &junk_str);
		os_server_write_func(fsc->fd, out_list[i].value, 
				     out_list[i].length,
				     &junk_errno, &junk_str);
	    }		
	}
	break;
    default:
	LOG("Unknown request from fd %d- %s", fsc->fd, fsc->hostname);
	printf("Unknown request %c, %x\n", next_action, (unsigned char) next_action);
	FSClient_close(fsc);
	break;
    }
}    


/* 
 * Create and initialize a connection data structure.
 */
static FSClient
FSClient_create(format_server fs)
{
    FSClient fsc = (FSClient) io_malloc(sizeof(*fsc));
    fsc->fd = NULL;
    fsc->hostname = NULL;
    fsc->port = 0;
    fsc->provisional = 0;
    fsc->fs = fs;
    fsc->version = 0;
    return fsc;
}

static void
get_qual_hostname(char *buf, int len)
{
    struct hostent *host;

    gethostname(buf, len);
    buf[len - 1] = '\0';
    if (memchr(buf, '.', strlen(buf)) == NULL) {
	/* no dots, probably not fully qualified */
	int end = strlen(buf);
	buf[end] = '.';
#ifdef HAVE_GETDOMAINNAME
	getdomainname((&buf[end]) + 1, len - strlen(buf));
	if (buf[end+1] == 0) {
	    char *tmp_name;
	    buf[end] = 0;
	    /* getdomainname was useless, hope that gethostbyname helps */
	    tmp_name = (gethostbyname(buf))->h_name;
	    strncpy(buf, tmp_name, len);
	}
#else
	{ 
	    char *tmp_name;
	    /* no getdomainname, hope that gethostbyname will help */
	    tmp_name = (gethostbyname(buf))->h_name;
	    strncpy(buf, tmp_name, len);
	}
#endif
	buf[len - 1] = '\0';
    }
    if (((host = gethostbyname(buf)) == NULL) ||
	(memchr(buf, '.', strlen(buf)) == NULL)){
	/* bloody hell, what do you have to do? */

	gethostname(buf, len);
	host  = gethostbyname(buf);
	host = gethostbyaddr(host->h_addr_list[0], host->h_length, host->h_addrtype);
	if (host != NULL) {
	  strncpy(buf, host->h_name, len);
	}
    }
    
    if (((host = gethostbyname(buf)) == NULL) ||
	(memchr(buf, '.', strlen(buf)) == NULL)){
	/* just use the bloody IP address */
	gethostname(buf, len);
	host  = gethostbyname(buf);
	if (host != NULL) {
	    char *tmp = inet_ntoa(*(struct in_addr*)(host->h_addr_list[0]));
	    strncpy(buf, tmp, len);
	} else {
	    strcpy(buf, "Unknown");
	}
    }
}

/* 
 * Create and initialize a data exchange.
 */
static format_server
format_server_create()
{
    int i, j;
    format_server fs = (format_server)io_malloc(sizeof(*fs));
    int max_fd = FD_SETSIZE;	/* returns process fd table size */
    if (fs == NULL)
	return NULL;

    /* initialize data structs */
    fs->port = -1;
    fs->hostname = NULL;
    fs->conn_sock_inet = (void *)-1;
    fs->usock_name = NULL;
    for(i = 0; i<2 ; i++) {
	for (j=0; j<1; j++) {
	    fs->lists[i][j] = NULL;
	}
    }

    fs->portCount = 0;
    fs->ports = (FSClient *) io_malloc(sizeof(FSClient) * max_fd);
    memset((char *) fs->ports, 0, sizeof(FSClient) * max_fd);

    fs->timestamp = (time_t *) io_malloc(sizeof(time_t) * max_fd);
    memset((char *) fs->timestamp, 0, sizeof(FSClient) * max_fd);

    FD_ZERO(&fs->fdset);
    fs->data_buffer = (char*) io_malloc(1);
    fs->buffer_size = 1;
    /* 
     * ignore SIGPIPE's  (these pop up when ports die.  we catch the 
     * failed writes) 
     */
#ifdef SIGPIPE
    signal(SIGPIPE, SIG_IGN);
#endif

    return fs;
}


/* 
 * Create an IP socket for connection from other data exchanges.
 */
static int
format_server_listen(format_server fs, int port_num)
{
    int length;
    char buf[256];
    struct sockaddr_in sock_addr;
    int sock_opt_val = 1;
    int conn_sock;

    if ((long)fs->conn_sock_inet >= 0) {
	fprintf(stderr, "INET connection socket already open %lx\n",
		(long)fs->conn_sock_inet);
	return -1;
    }
    conn_sock = socket(AF_INET, SOCK_STREAM, 0);
    if (conn_sock < 0) {
	
	fprintf(stderr, "Cannot open INET socket %d\n", conn_sock);
	return -1;
    }
    sock_addr.sin_family = AF_INET;
    sock_addr.sin_addr.s_addr = INADDR_ANY;
    sock_addr.sin_port = htons((unsigned short) port_num);
    if (setsockopt((long)conn_sock, SOL_SOCKET, SO_REUSEADDR, (char *) &sock_opt_val,
		   sizeof(sock_opt_val)) != 0) {
	fprintf(stderr, "Failed to set 1REUSEADDR on INET socket\n");
	return -1;
    }
    if (bind(conn_sock, (struct sockaddr *) &sock_addr,
	     sizeof sock_addr) < 0) {
	fprintf(stderr, "Cannot bind INET socket\n");
	return -1;
    }
    sock_opt_val = 1;
    if (setsockopt(conn_sock, SOL_SOCKET, SO_REUSEADDR, (char *) &sock_opt_val,
		   sizeof(sock_opt_val)) != 0) {
	perror("Failed to set 2REUSEADDR on INET socket");
	return -1;
    }
    length = sizeof sock_addr;
    if (getsockname(conn_sock, (struct sockaddr *) &sock_addr, &length) < 0) {
	fprintf(stderr, "Cannot get socket name\n");
	return -1;
    }
    fs->port = ntohs(sock_addr.sin_port);

    /* begin listening for conns and set the backlog */
    if (listen(conn_sock, FD_SETSIZE)) {
	fprintf(stderr, "listen failed\n");
	return -1;
    }
    /* record the server's hostname */
    get_qual_hostname(buf, sizeof(buf));
    fs->hostname = (char *) io_strdup(buf);

    fs->conn_sock_inet = (void *)conn_sock;
    if (conn_sock > FD_SETSIZE) {
	fprintf(stderr, "Internal Error, stupid WINSOCK large FD bug.\n");
	fprintf(stderr, "Increase FD_SETSIZE.  Item not added to fdset.\n");
    }
    FD_SET(conn_sock, &fs->fdset);
    return 0;
}

static void out_domain_rejection(int fd, FSClient fsc);

/* 
 * Accept socket connection from other data exchs.  If conn_sock is -1,
 * check if there is connection request at conn_sock_inet or
 * conn_sock_unix.
 */
extern FSClient
format_server_accept_conn_sock(format_server fs, void *conn_sock)
{
    int sock;
    FSClient fsc;
    int delay_value = 1;

    struct linger linger_val;

    linger_val.l_onoff = 1;
    linger_val.l_linger = 60;
    if (fs == NULL)
	return NULL;
    if ((long)conn_sock == -1) {
	fd_set fds;
	struct timeval timeout;
	FD_ZERO(&fds);
	LOG("minus 1 variation");
	if ((long)fs->conn_sock_inet >= 0) {
	    FD_SET((unsigned long)fs->conn_sock_inet, &fds);
	    if ((long)fs->conn_sock_inet > FD_SETSIZE) {
		fprintf(stderr, "Internal Error, stupid WINSOCK large FD bug.\n");
		fprintf(stderr, "Increase FD_SETSIZE.  Item not added to fdset.\n");
	    }
	}
	timeout.tv_sec = 0L;
	timeout.tv_usec = 0L;

	if (select(FD_SETSIZE, &fds, (fd_set *) NULL, (fd_set *) NULL,
		   &timeout) > 0) {
	    if (FD_ISSET((int)(long)fs->conn_sock_inet, &fds)) {
		format_server_accept_conn_sock(fs, fs->conn_sock_inet);
	    }
	}
	return NULL;
    }
    
    if ((sock = accept((unsigned int)(long)conn_sock, (struct sockaddr *) 0, (int *) 0)) < 0) {
	LOG("accept failed");

	timeout_oldest_conn();

	if ((sock = accept((unsigned int)(long)conn_sock, (struct sockaddr *) 0, (int *) 0)) < 0) {
	    LOG("accept failed twice");
	    fprintf(stderr, "Cannot accept socket connection\n");
	    return NULL;
	}
    }
    LOG("\nAccepting fd %d", sock);
    if (setsockopt(sock, SOL_SOCKET, SO_LINGER, (char *) &linger_val,
		   sizeof(struct linger)) != 0) {
	perror("set SO_LINGER");
	return NULL;
    }
    setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char *) &delay_value,
	       sizeof(delay_value));

    fsc = FSClient_create(fs);
    fsc->fd = (void*)sock;

    FD_SET(sock, &fs->fdset);
    if (sock > FD_SETSIZE) {
	fprintf(stderr, "Internal Error, stupid WINSOCK large FD bug.\n");
	fprintf(stderr, "Increase FD_SETSIZE.  Item not added to fdset.\n");
    }
    fs->ports[sock] = fsc;
    fs->timestamp[sock] = time(NULL);
    fs->portCount++;

    if (no_fork) {
	printf("FSxchange_accept_conn_sock successful\n");
    }
    if (fsc->hostname == NULL) fsc->hostname = "none";
    if (no_fork) {
	printf("fsc->fd = %d, port = %d, hostname = %s\n", 
	       (int)(long)fsc->fd, fsc->port, 
	       fsc->hostname);
    }
    server_read_header(fsc);
    out_domain_rejection(sock, fsc);
    return fsc;
}

typedef struct host_info {
    struct in_addr ip_addr;
    time_t intro_time;
} *host_info_p;

static host_info_p hostlist = NULL;
static int host_count = 0;

static time_t
get_time_for_host(ip_addr, hostname)
struct in_addr ip_addr;
char *hostname;
{
    int i;
    if (hostlist == NULL) {
	hostlist = malloc(sizeof(struct host_info));
    }
    for (i = 0; i< host_count; i++) {
	if (memcmp(&ip_addr, &hostlist[i].ip_addr, sizeof(struct in_addr)) == 0) {
	    return hostlist[i].intro_time;
	}
    }
    hostlist = realloc(hostlist, sizeof(struct host_info) * (host_count + 1));
    memcpy(&hostlist[host_count].ip_addr, &ip_addr, sizeof(struct in_addr));
    hostlist[host_count].intro_time = time(NULL);
    host_count++;
    return hostlist[host_count-1].intro_time;
}

static char *postfix_string = FORMAT_SERVICE_DOMAIN;
static char **postfix_list = NULL;

/* 3 days of free use of our daemons for every host */
#define GRACE_PERIOD_SEC 60 * 60 * 24 * 3

static void
out_domain_rejection(fd, fsc)
int fd;
FSClient fsc;
{
    struct sockaddr sock_addr;
    int sock_len = sizeof(sock_addr);
    time_t intro_time;

    if (postfix_list == NULL) {
	char *tmp = postfix_string;
	int i, postfix_count = 1;
	while ((tmp = strchr(tmp, ':')) != NULL) {
	    postfix_count++;
	    tmp++;
	}
	postfix_list = malloc(sizeof(char*) * (postfix_count + 1));
	postfix_list[0] = io_strdup(postfix_string);
	for (i=1; i<postfix_count; i++) {
	    postfix_list[i] = strchr(postfix_list[i-1], ':');
	    if (postfix_list[i] != NULL) {
		*postfix_list[i] = 0;  /* kill : */
		postfix_list[i]++;
	    }
	}
	postfix_list[postfix_count] = NULL;
    }
	    
    if (getpeername(fd, &sock_addr, &sock_len) == 0) {
	struct hostent *host;
	if (sock_addr.sa_family == AF_INET) {
	    char *hostname;
	    struct sockaddr_in *in_sock = (struct sockaddr_in *) &sock_addr;
	    host = gethostbyaddr((char *) &in_sock->sin_addr,
				 sizeof(struct in_addr), AF_INET);
	    if (host != NULL) {
		char **tmp_list = postfix_list;
		while (*tmp_list != NULL) {
		    char *postfix = *(tmp_list++);
		    char **alias;
		    int postlen = strlen(postfix);
		    int hostlen = strlen(host->h_name);
		    /* 
		     * Check to see if our FORMAT_SERVICE_DOMAIN is a postfix
		     * of the hostname.  If not, we'll reject the host as being
		     * outside of our service domain.
		     */
		    if (hostlen > postlen) {
			if (strcmp(&host->h_name[hostlen - postlen], postfix) == 0) {
			    /* good host */
			    fsc->hostname = io_strdup(host->h_name);
			    LOG("    Connection from %s", host->h_name);
			    return;
			}
		    } else {
			if (strcmp(host->h_name, "localhost") == 0) {
			    fsc->hostname = io_strdup(host->h_name);
			    LOG("    Connection from %s", host->h_name);
			    return;
			}
		    }
		    for (alias = host->h_aliases; *alias != 0; alias++) {
			if (hostlen > postlen) {
			    if (strcmp(alias[hostlen - postlen], postfix) == 0) {
				/* good host */
				fsc->hostname = io_strdup(host->h_name);
				LOG("    Connection from %s", host->h_name);
				return;
			    }
			}
		    }
		}
		hostname = host->h_name;
		fsc->hostname = io_strdup(hostname);
	    } else {
		/* no reverse DNS ??? */
		hostname = (char*)inet_ntoa(in_sock->sin_addr);
		fsc->hostname = io_strdup(hostname);
	    }

	    intro_time = get_time_for_host(in_sock->sin_addr, hostname);
	    if ((time(NULL) - intro_time) < GRACE_PERIOD_SEC) {
		if (fsc->version > 0) {
		    printf("Sending character \"P\"\n");
		    LOG("accepting provisional host, %s", hostname);
		    fsc->provisional = 1;
		} else {
		    LOG("rejecting old provisional host, %s", hostname);
		    close(fd);
		}
		return;
	    }

	    printf("Rejecting out-domain host %s, aliases: ", hostname);
	    if (host != NULL) {
		char **alias;
		for (alias = host->h_aliases; *alias != 0; alias++) {
		    printf(" %s", *alias);
		}
	    }
	    printf("\n");
	}
    }
    /* must not be a good host, close it down ... */
    LOG("Rejecting unknown host");
    close(fd);
}

static void
timeout_old_conns()
{
    static int last_timeout_time = 0;
    time_t now = time(NULL);
    int i;

    if (last_timeout_time == 0) {
	last_timeout_time = time(NULL);
	return;
    }
    if (now - last_timeout_time < CONN_TIMEOUT_INTERVAL) {
	return;
    }
    last_timeout_time = now;
    for (i=0; i<FD_SETSIZE; i++) {
	if (server_fs->ports[i] != NULL) {
	    if (now - server_fs->timestamp[i] > CONN_TIMEOUT_INTERVAL) {
		LOG("Timeout -> fd %d", i);
		FSClient_close(server_fs->ports[i]);
	    }
	}
    }
}

static void
timeout_oldest_conn()
{
    int oldest_time = 0;
    int oldest_index = 0;
    int i;

    for (i=0; i<FD_SETSIZE; i++) {
	if (server_fs->ports[i] != NULL) {
	    if (oldest_time == 0) {
		oldest_time = server_fs->timestamp[i];
		oldest_index = i;
	    } else if (oldest_time > server_fs->timestamp[i]) {
		oldest_time = server_fs->timestamp[i];
		oldest_index = i;
	    }
	}
    }
    if (oldest_time != 0) {
	LOG("Closing oldest -> fd %d", i);
	FSClient_close(server_fs->ports[oldest_index]);
    }
}

static
void dump_stats_to_log()
{
    int i, fsc_count = 0;
    time_t now = time(NULL);
    fprintf(log, "\n\n");
    fprintf(log, "Currently registered formats  -- %d\n", server_format_count);
    fprintf(log, "Data handle count -- %d\n", select_handle_count);
    fprintf(log, "Test count -- %d\n", test_count);
    fprintf(log, "Registration count -- %d\n", registration_count);
    fprintf(log, "Format Get count -- %d\n", get_count);
    fprintf(log, "\n format_server portCount = %d\n", server_fs->portCount);
    for (i=0; i<FD_SETSIZE; i++) {
	if (server_fs->ports[i] != NULL) {
	    fsc_count++;
	    fprintf(log, "	fd %d - last use %s                                  age %ld\n", i, 
		    ctime(&server_fs->timestamp[i]), 
		    (long)now - server_fs->timestamp[i]);
	}
    }
    fprintf(log, " format_server client count = %d\n", fsc_count);
}
