/*
 *  This file contains functions to assist in using the PBIO IOContext
 *  routines for type encoding.
 */
#include "config.h"

#include <malloc.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>

#include <gen_thread.h>

#include "io.h"
#include "DE.h"
#include "otl.h"
#include "unix_defs.h"

extern DExchange otl_private_de;
extern thr_mutex_t otl_private_de_lock;
extern thr_mutex_t otl_private_type_lock;
extern IOContext otl_private_type_context;

static IOField any_fields[] = {
    { "length", "integer", sizeof(int), IOOffset(CORBA_any*, _type) },
    { "value", "char[length]", sizeof(char), IOOffset(CORBA_any*, _value) },
   { NULL, NULL, 0, 0}
};

extern void
init_type_context()
{
    assert(register_IOcontext_format("any", any_fields, 
				     otl_private_type_context) != NULL);
}

extern void
otl_set_pbio_type(format, type_string, fields)
IOFormat *format;
char *type_string;
IOFieldList fields;
{
    /* 
     *  (*format) should have been checked for equality to NULL before 
     *  this is called (as an optimization), but check again after we 
     *  lock the IOcontext value, just to be sure.  Make sure we do the 
     *  back assignment while holding the lock so that the check and assign
     *  are within the critical section.
     */
    thr_mutex_lock(otl_private_type_lock);
    if (*format == NULL) {
	if ((*format = get_IOformat_by_name((IOFile)otl_private_type_context,
					    type_string)) == NULL) {
	    *format = register_IOcontext_format(type_string, fields, 
						otl_private_type_context);
	}
    } 
    thr_mutex_unlock(otl_private_type_lock);
}

extern int
otl_check_conversion(encoded_block)
void *encoded_block;
{
    IOFormat ioformat;
    
    thr_mutex_lock(otl_private_type_lock);
    ioformat = get_format_IOcontext(otl_private_type_context,
					     encoded_block);
    thr_mutex_unlock(otl_private_type_lock);
    if (ioformat == NULL) {
	fprintf(stderr, "Unknown IOFormat in encoded block\n");
	return 0;
    }
    return has_conversion_IOformat(ioformat);
}

extern void
otl_set_pbio_types(encoded_buffer, format_list)
void *encoded_buffer;
otl_list_of_formats *format_list;
{
    IOFormat *ioformats;
    IOFormat *format;

    thr_mutex_lock(otl_private_type_lock);
    if ((ioformats = get_subformats_IOcontext(otl_private_type_context,
					   (void *) encoded_buffer)) == NULL) {
	fprintf(stderr, "IOFormat list is null\n");
	return;
    }
    format = ioformats;
    while (*format != NULL) {
	int i = 0;
	if (!has_conversion_IOformat(*format)) {
	    int native_struct_size;
	    while(format_list[i].format_name != NULL) {
		if (strcmp(format_list[i].format_name,
			   name_of_IOformat(*format)) == 0) {
		    break;
		}
		i++;
	    }
	    if (format_list[i].field_list != NULL) {
		native_struct_size = 
		    struct_size_IOfield((IOFile)otl_private_type_context, 
					format_list[i].field_list);
		set_conversion_IOcontext(otl_private_type_context, *format,
					 format_list[i].field_list,
					 native_struct_size);
	    } else if (strcmp(name_of_IOformat(*format), "any") == 0) {
		    set_conversion_IOcontext(otl_private_type_context, 
					     *format, any_fields,
					     sizeof(CORBA_any));
	    } else {
		fprintf(stderr, "Probable compiler error, format \"%s\" not found in otl_set_pbio_types list.\n", 
			name_of_IOformat(*format));
	    }
	}
	format++;
    }
    free(ioformats);
    thr_mutex_unlock(otl_private_type_lock);
}

extern void*
otl_pbio_type_encode(format, data, len_return)
IOFormat format;
void *data;
int *len_return;
{
    IOEncodeVector vector;
    int i, buffer_len = 0;
    char *return_buffer;

    vector = otl_pbio_type_encodev(format, data, NULL);

    i = 0;
    while (vector[i].iov_len != 0) {
	buffer_len += vector[i].iov_len;
	i++;
    }
    return_buffer = malloc(buffer_len);

    i = 0;
    buffer_len = 0;
    while (vector[i].iov_len != 0) {
	memcpy(&return_buffer[buffer_len], vector[i].iov_base,
	       vector[i].iov_len);
	buffer_len += vector[i].iov_len;
	i++;
    }
    free(vector);
    *len_return = buffer_len;

    return (void*) return_buffer;
}

extern IOEncodeVector
otl_pbio_type_encodev(format, data, free_block_p)
IOFormat format;
void *data;
void **free_block_p;
{
    IOEncodeVector vector;

    thr_mutex_lock(otl_private_type_lock);
    vector = encode_IOcontext_release_vector(otl_private_type_context,
					format, data);
    *free_block_p = (void*)vector;
    thr_mutex_unlock(otl_private_type_lock);
    return vector;
}

extern void*
otl_pbio_type_encode_to_buffer(format, data, buffer, len_return)
IOFormat format;
void *data;
void **buffer;
int *len_return;
{
    IOEncodeVector vector;
    int i, buffer_len = 0;
    char *return_buffer = *buffer;

    thr_mutex_lock(otl_private_type_lock);
    vector = encode_IOcontext_to_vector(otl_private_type_context,
					format, data);
    i = 0;
    while (vector[i].iov_len != 0) {
	buffer_len += vector[i].iov_len;
	i++;
    }
    if (buffer_len > *len_return) {
	if (*buffer == NULL) {
	    return_buffer = *buffer = malloc(buffer_len);
	} else {
	    *buffer = realloc(*buffer, buffer_len);
	    return_buffer = *buffer;
	}
    }

    i = 0;
    buffer_len = 0;
    while (vector[i].iov_len != 0) {
	memcpy(&return_buffer[buffer_len], vector[i].iov_base,
	       vector[i].iov_len);
	buffer_len += vector[i].iov_len;
	i++;
    }
    free(vector);
    *len_return = buffer_len;

    thr_mutex_unlock(otl_private_type_lock);
    return (void*) return_buffer;
}
    
extern void*
otl_pbio_type_decode(data, data_len)
void *data;
int data_len;
{
    int buffer_len;
    void *buffer;

    thr_mutex_lock(otl_private_type_lock);


    buffer_len = this_IOrecord_length(otl_private_type_context, data, 
				      data_len);

    buffer = malloc(buffer_len);

    decode_to_buffer_IOcontext(otl_private_type_context, data, buffer);

    thr_mutex_unlock(otl_private_type_lock);
    return buffer;
}

extern void
otl_pbio_decode(data, data_len, dest)
void *data;
int data_len;
void *dest;
{
    thr_mutex_lock(otl_private_type_lock);

    decode_IOcontext(otl_private_type_context, data, dest);

    thr_mutex_unlock(otl_private_type_lock);
}

