
#include "config.h"

#include <malloc.h>
#include <ctype.h>
#include <memory.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>

#include <gen_thread.h>

#include "atom.h"
#include "attr.h"
#include "io.h"
#include "DE.h"
#include "otl.h"
#include "comm_group.h"
#include "otl_formats.h"
#include "otl_obj.h"
#include "unix_defs.h"

extern DExchange otl_private_de;
extern DEPort otl_get_dep ARGS((char *name, int port));
extern int OTL_verbose;
extern int otl_debug;

static void
add_list_to_xmit_obj(xmit_obj, list, count, start_index)
xmit_object_ref xmit_obj;
attr_list list;
int count;
int start_index;
{
    int i;
    for (i = 0; i < count; i++) {
	int j = i + start_index;
	attr_p tmp = get_attr(list, i);
	xmit_obj->attrs[j].attr_name = tmp->attr_id;
	xmit_obj->attrs[j].attr_type = tmp->val_type;
	switch(tmp->val_type) {
	case Attr_Opaque: {
	    attr_opaque_p o = (attr_opaque_p) tmp->value;
	    xmit_obj->attrs[j].attr_atom_val = (long) 0;
	    if (o != NULL) {
		xmit_obj->attrs[j].opaque_len = o->length;
		xmit_obj->attrs[j].opaque_buffer = o->buffer;
	    } else {
		xmit_obj->attrs[j].opaque_len = 0;
		xmit_obj->attrs[j].opaque_buffer = NULL;
	    }
	    break;
	}
	case Attr_String: {
	    xmit_obj->attrs[j].attr_atom_val = (long) 0;
	    xmit_obj->attrs[j].opaque_len = strlen((char *) tmp->value) + 1;
	    xmit_obj->attrs[j].opaque_buffer = tmp->value;
	    break;
	}
	default: {
	    xmit_obj->attrs[j].attr_atom_val = (long) tmp->value;
	    xmit_obj->attrs[j].opaque_len = 0;
	    xmit_obj->attrs[j].opaque_buffer = NULL;
	}
	}
    }
}

extern void
init_xmit_obj(xmit_obj, initial_count)
xmit_object_ref xmit_obj;
int initial_count;
{
    xmit_obj->attr_count = initial_count;
    xmit_obj->attrs = (xmit_attr_ref) malloc(sizeof(xmit_attr) * xmit_obj->attr_count);
}

extern void
free_xmit_obj(xmit_obj)
xmit_object_ref xmit_obj;
{
    free(xmit_obj->attrs);
}

extern void
fill_xmit_obj(xmit_obj, object_name)
xmit_object_ref xmit_obj;
attr_list object_name;
{
    int initial_count = 0;

    if (object_name != NULL) {
	initial_count = attr_count(object_name);
    }
    init_xmit_obj(xmit_obj, initial_count);
    if (object_name != NULL) {
	add_list_to_xmit_obj(xmit_obj, object_name, initial_count, 0);
    }
}

extern attr_list
attr_list_from_xmit(xmit_obj)
xmit_object_ref xmit_obj;
{
    attr_list tmp = create_attr_list();
    int i;


    for (i = 0; i < xmit_obj->attr_count; i++) {
	switch(xmit_obj->attrs[i].attr_type) {
	case Attr_String: 
	    add_attr(tmp, xmit_obj->attrs[i].attr_name, Attr_String,
		     strdup(xmit_obj->attrs[i].opaque_buffer));
	    break;
	case Attr_Opaque: {
	    attr_opaque_p o = NULL;
	    if (xmit_obj->attrs[i].opaque_buffer != NULL) {
		o = malloc(sizeof(attr_opaque));
		o->length = xmit_obj->attrs[i].opaque_len;
		o->buffer = malloc(o->length);
		memcpy(o->buffer, xmit_obj->attrs[i].opaque_buffer, 
		       o->length);
	    }
	    add_attr(tmp, xmit_obj->attrs[i].attr_name, Attr_Opaque, 
		     (attr_value) o);
	    break;
	}
	default:
	    add_attr(tmp, xmit_obj->attrs[i].attr_name,
		     xmit_obj->attrs[i].attr_type,
		     (attr_value) xmit_obj->attrs[i].attr_atom_val);

	    break;
	}
    }
    return tmp;
}

void
add_to_attr_list_from_xmit_obj(list, xmit_obj)
attr_list list;
xmit_object_ref xmit_obj;
{
    int i;

    for (i = 0; i < xmit_obj->attr_count; i++) {
	switch(xmit_obj->attrs[i].attr_type) {
	case Attr_String: 
	    set_attr(list, xmit_obj->attrs[i].attr_name, Attr_String,
		     strdup(xmit_obj->attrs[i].opaque_buffer));
	    break;
	case Attr_Opaque: {
	    attr_opaque_p o = NULL;
	    if (xmit_obj->attrs[i].opaque_buffer != NULL) {
		o = malloc(sizeof(attr_opaque));
		o->length = xmit_obj->attrs[i].opaque_len;
		o->buffer = malloc(o->length);
		memcpy(o->buffer, xmit_obj->attrs[i].opaque_buffer,
		       o->length);
	    }
	    set_attr(list, xmit_obj->attrs[i].attr_name, Attr_Opaque, 
		     (attr_value) o);
	    break;
	}
	default:
	    set_attr(list, xmit_obj->attrs[i].attr_name,
		     xmit_obj->attrs[i].attr_type,
		     (attr_value) xmit_obj->attrs[i].attr_atom_val);

	}
    }
}

static attr_value
get_is_local_object(list)
attr_list list;
{
    static atom_t home_host_atom = 0;
    static atom_t home_port_atom = 0;


    char *home_host;
    int home_port;

    if (home_host_atom == 0) {
	home_host_atom = attr_atom_from_string("OTL:HOME_HOST");
	home_port_atom = attr_atom_from_string("OTL:HOME_PORT");
    }
    if (!query_attr(list, home_host_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) & home_host)) {
	/* 
	 * Not Yet Implemented, but code should do:
	 * OTL invoke to the value of the OTL:NAME_SERVER
	 * attribute, followed by an otl_wait() and
	 * otl_release_handle().  Upon completion, there
	 * should be a home_host value in
	 * obj->name_trans_cache. 
	 */
	if (!query_attr(list, home_host_atom, /* type pointer */ NULL,
			/* value pointer */ (attr_value *) & home_host)) {
	    return 0;	/* Serious error */
	}
    }
    if (!query_attr(list, home_port_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) & home_port)) {
	return 0;  /* if no otl home port, must not be local */
    }
    return (attr_value) ((strcmp(home_host, DExchange_host_name(otl_private_de)) == 0) && (home_port == DExchange_inet_port(otl_private_de)));
}

static attr_value
get_direct_call(list)
attr_list list;
{
    /*
     * To do a direct call to an object, it must be local,
     * the method must *not* be threaded, and we must be using the
     * same interface as the object was created with (I.E. the object 
     * is not in a subclass).
     */
    static atom_t otl_object_interface_atom = 0;
    static atom_t otl_invoke_interface_atom = 0;
    static atom_t is_local_atom = 0;

    atom_t obj_interface;
    atom_t invoke_interface;
    int is_local;

    if (otl_object_interface_atom == 0) {
	otl_object_interface_atom = attr_atom_from_string("OTL:OBJECT_INTERFACE");
	otl_invoke_interface_atom = attr_atom_from_string("OTL:INVOKE_INTERFACE");
	is_local_atom = attr_atom_from_string("OTL:IS_LOCAL");
    }
    if (!query_attr(list, otl_object_interface_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) & obj_interface)) {
	return 0;
    }
    if (!query_attr(list, otl_invoke_interface_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) & invoke_interface)) {
	return 0;
    }
    if (obj_interface != invoke_interface) {
	return 0;
    }
    /* if ( threaded ) return 0 */

    /*
     * if OTL:IS_LOCAL attribute is present, it should be in the name cache.
     * Quickly check that and return the value if it is set.
     */
    if (query_attr(list, is_local_atom, /* type pointer */ NULL,
		   /* value pointer */ (attr_value*)&is_local)) {
	return (attr_value) is_local;
    } else {
	return get_is_local_object(list);
    }
}

/* 
 * otl_is_local_object()
 * This otl utility routine exists to determine object
 * locality.  This determination may require several steps,
 * including performing name translation if that is
 * required.  Assuming that no name translation is required,
 * either the obj->object_name or the obj->name_trans_cache
 * will contain a OTL:HOME_HOST attribute which will be
 * compared to the current host.  If they are equivalent,
 * the attribute OTL:IS_LOCAL will be added to
 * obj->name_trans_cache.  (This attribute is checked for at
 * the beginning of the routine as a shortcut to this
 * process.)  If the OTL:HOME_HOST is not present, the
 * OTL:NAME_SERVER attribute must hold a value that can be
 * invoked to translate the name and set the OTL:HOME_HOST
 * attribute. 
 */
extern int
otl_is_local_object(obj_name, name_cache)
attr_list obj_name;
attr_list name_cache;
{
    static atom_t is_local_atom = 0;
    attr_list this_invoke;
    int is_local = 0;

    if (is_local_atom == 0) {
	is_local_atom = attr_atom_from_string("OTL:IS_LOCAL");
    }
    /*
     * if OTL:IS_LOCAL attribute is present, it should be in the name cache.
     * Quickly check that and return the value if it is set.
     */
    if (name_cache && 
	query_attr(name_cache, is_local_atom, /* type pointer */ NULL,
		   /* value pointer */ (attr_value*)&is_local)) {
	return is_local ;
    } else {
	/* 
	 * no cached value for OTL:IS_LOCAL.  determine and set one 
	 * for next time.  Must join the two lists because
	 */
	this_invoke = attr_join_lists(obj_name, name_cache);

	otl_get_attr(is_local_atom, Attr_Int4, (attr_value *) & is_local, 
	 	     this_invoke, (attr_value(*)())get_is_local_object,
		     name_cache);
	free_attr_list(this_invoke);
	return is_local;
    }
}

/* 
 * otl_is_local_object()
 * This otl utility routine exists to determine object
 * locality.  This determination may require several steps,
 * including performing name translation if that is
 * required.  Assuming that no name translation is required,
 * either the obj->object_name or the obj->name_trans_cache
 * will contain a OTL:HOME_HOST attribute which will be
 * compared to the current host.  If they are equivalent,
 * the attribute OTL:IS_LOCAL will be added to
 * obj->name_trans_cache.  (This attribute is checked for at
 * the beginning of the routine as a shortcut to this
 * process.)  If the OTL:HOME_HOST is not present, the
 * OTL:NAME_SERVER attribute must hold a value that can be
 * invoked to translate the name and set the OTL:HOME_HOST
 * attribute. 
 */
extern int
otl_direct_call(obj_name, method_cache)
attr_list obj_name;
attr_list method_cache;
{
    static atom_t direct_call_atom = 0;
    attr_list this_invoke;
    int direct_call = 0;

    if (direct_call_atom == 0) {
	direct_call_atom = attr_atom_from_string("OTL:DIRECT_CALL");
    }
    /*
     * if OTL:DIRECT_CALL attribute is present, it should be in the method cache.
     * Quickly check that and return the value if it is set.
     */
    if (method_cache && 
	query_attr(method_cache, direct_call_atom, /* type pointer */ NULL,
		   /* value pointer */ (attr_value*)&direct_call)) {
	return direct_call;
    } else {
	/* 
	 * no cached value for OTL:DIRECT_CALL.  determine and set one 
	 * for next time.  Must join the two lists because
	 */
	this_invoke = attr_join_lists(obj_name, method_cache);

	otl_get_attr(direct_call_atom, Attr_Int4, (attr_value *) &direct_call, 
	 	     this_invoke, (attr_value(*)())get_direct_call,
		     method_cache);
	free_attr_list(this_invoke);
	return direct_call;
    }
}

extern int
otl_protocol_invoke(obj_name)
attr_list obj_name;
{
    static atom_t otl_protocol_atom = 0;
    static atom_t otl_protocol = 0;
    atom_t this_protocol;
    if (otl_protocol_atom == 0) {
	otl_protocol_atom = attr_atom_from_string("OTL:PROTOCOL_TAG");
	otl_protocol = attr_atom_from_string("OTL:OTL_PROTOCOL");
    }
    query_attr(obj_name, otl_protocol_atom, /* type pointer */ NULL,
		   /* value pointer */ (attr_value*)&this_protocol);
    return (this_protocol == otl_protocol);
}

extern attr_value
get_dep_func(list)
attr_list list;
{
    static atom_t otl_dep = 0;
    static atom_t home_host_atom = 0;
    static atom_t home_port_atom = 0;
    char *home_host_name;
    int port;
    DEPort dep;

    if (otl_dep == 0) {
	otl_dep = attr_atom_from_string("OTL:DEP");
	home_host_atom = attr_atom_from_string("OTL:HOME_HOST");
	home_port_atom = attr_atom_from_string("OTL:HOME_PORT");
    }
    if (!query_attr(list, home_host_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) & home_host_name)) {
	/* 
	 * OTL invoke to the value of the OTL:NAME_SERVER
	 * attribute, followed by an otl_wait() and
	 * otl_release_handle().  Upon completion, there
	 * should be a home_host value in
	 * obj->name_trans_cache. 
	 */
	if (!query_attr(list, home_host_atom, /* type pointer */ NULL,
			/* value pointer */ (attr_value *) & home_host_name)) {
	    return 0;	/* Serious error */
	}
    }
    if (!query_attr(list, home_port_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) & port)) {
    }
    dep = otl_get_dep(home_host_name, port);
    return (attr_value) dep;
}

extern
int get_iiop_connection ARGS((DExchange de, char *host, int port));

extern void
remote_iiop_call_terminating();

extern
void send_request_msg ARGS((COBS_sequence *key, int method_id, 
			    otl_class_info obj_info,
			    void *param_struct, int fd, int my_request_id,
			    int response_expected, int version,
			    otl_handle handle));

static otl_handle
internal_otl_invoke(obj_name, name_cache, method_cache, handle, class_info)
attr_list obj_name;
attr_list name_cache;
attr_list method_cache;
otl_handle handle;
otl_class_info class_info;
{
    static atom_t otl_dep = 0;
    static atom_t otl_protocol_atom = 0;
    static atom_t otl_protocol = 0;
    static atom_t iiop_protocol = 0;
    static atom_t no_return_atom = 0;
    static int invoke_format_id = 0;
    atom_t this_protocol = otl_protocol;
    attr_list this_invoke;
    int is_local = 0;
    int no_return = 0;

    if (otl_dep == 0) {
	otl_dep = attr_atom_from_string("OTL:DEP");
	otl_protocol_atom = attr_atom_from_string("OTL:PROTOCOL_TAG");
	this_protocol = otl_protocol = 
	    attr_atom_from_string("OTL:OTL_PROTOCOL");
	iiop_protocol = attr_atom_from_string("OTL:IIOP_PROTOCOL");
	no_return_atom = attr_atom_from_string("OTL:NO_RETURN");
	invoke_format_id = DEget_format_id(otl_private_de, "invoke message");
    }
    this_invoke = attr_join_lists(obj_name, name_cache);
    if (otl_debug) {
	printf("OTL_INVOKE:  Attrs ->");
	dump_attr_list(this_invoke);
    }
    is_local = otl_is_local_object(this_invoke, name_cache);
    handle->cache = method_cache;
    add_ref_attr_list(method_cache);
    handle->wait_count = 0;
    /* one reference is returned to user, the other is held 
       pending call termination. */
    handle->ref_count = 2;
    query_attr(this_invoke, no_return_atom, NULL, (attr_value *) &no_return);
    query_attr(this_invoke, otl_protocol_atom, NULL, (attr_value *) 
	       &this_protocol);
    if (is_local) {
	otl_handle_invocation(this_invoke, handle);
    } else if (this_protocol == otl_protocol) {
	DEPort dep;
	invoke_msg msg;

	otl_get_attr(otl_dep, Attr_Int4, (attr_value *) & dep, this_invoke, 
		     get_dep_func, name_cache);

	if (dep == NULL) {
	    CORBA_exception_set(&handle->ev, CORBA_SYSTEM_EXCEPTION,
				"CORBA_COMM_FAILURE", NULL);
	    otl_do_call_termination(handle);
	    return handle;
	}
	fill_xmit_obj(&msg.target, this_invoke);
	fill_xmit_obj(&msg.method, method_cache);
	msg.handle = handle;
	msg.pvector = handle->pvector;
	msg.pveclen = handle->pveclen;
	handle->de_condition = DECondition_get(otl_private_de, dep);
	if (DEport_write_data(dep, invoke_format_id, &msg) != 1) {
	    CORBA_exception_set(&handle->ev, CORBA_SYSTEM_EXCEPTION,
				"CORBA_COMM_FAILURE", NULL);
	    otl_do_call_termination(handle);
	    return handle;
	}
	free_xmit_obj(&msg.target);
	free_xmit_obj(&msg.method);
    } else if (this_protocol == iiop_protocol) {
	static atom_t otl_method_id_atom;
	COBS_sequence iiop_obj_key;
	int method_id, connection_fd;
	char *host;
	int port;
	int version;
	attr_opaque_p op;
	static atom_t home_host_atom = 0;
	static atom_t iiop_port_atom = 0;
	static atom_t iiop_version_atom = 0;
	static atom_t iiop_object_key_atom = 0;

	if (home_host_atom == 0) {
	    home_host_atom = attr_atom_from_string("OTL:HOME_HOST");
	    iiop_port_atom = attr_atom_from_string("OTL:IIOP_PORT");
	    iiop_version_atom = attr_atom_from_string("OTL:IIOP_VERSION");
	    iiop_object_key_atom = attr_atom_from_string("OTL:IIOP_OBJECT_KEY");
	}
	if (otl_method_id_atom == 0) {
	    otl_method_id_atom =
	      attr_atom_from_string("OTL:METHOD_ID");
	}
	query_attr(method_cache, otl_method_id_atom, NULL,
		   (attr_value*)&method_id);
	if(otl_debug) {
	    printf("calling IIOP method \"%s\"\n", class_info->methods[method_id].name);
	}
	query_attr(this_invoke, home_host_atom, NULL, (attr_value *) 
		   &host);
	query_attr(this_invoke, iiop_port_atom, NULL, (attr_value *) 
		   &port);
	version = 0;
	query_attr(this_invoke, iiop_version_atom, NULL, (attr_value *) 
		   &version);
	query_attr(this_invoke, iiop_object_key_atom, NULL,
			     (attr_value*)&op);
	connection_fd = get_iiop_connection(otl_private_de, host, port);

	iiop_obj_key._length = iiop_obj_key._maximum = op->length;
	iiop_obj_key._buffer = op->buffer;

	handle->de_condition = DECondition_get(otl_private_de, NULL);
	handle->termination_handler = remote_iiop_call_terminating;
	send_request_msg(&iiop_obj_key, method_id, class_info, 
			 handle->local_pb, connection_fd, 
			 handle->de_condition,
			 1 /*response_expected*/, version, handle);

    }
    free_attr_list(this_invoke);
    if (no_return) {
	handle->terminated = 1;
    }
    return handle;
}

extern otl_handle
otl_invokev(obj_name, name_cache, method_cache, class_info, pvector, local_pb, pb_size)
attr_list obj_name;
attr_list name_cache;
attr_list method_cache;
otl_class_info class_info;
IOEncodeVector pvector;
void *local_pb;
int pb_size;
{
    otl_handle handle;
    int pveclen=0;
    handle = get_otl_handle();
    handle->pvector = pvector;
    if (pvector != NULL) {
	while (pvector[pveclen].iov_len != 0) pveclen++;
    } else {
	pveclen = 0;
    }
    handle->pveclen = pveclen;
    handle->local_pb = local_pb;
    handle->local_pb_size = pb_size;
    return internal_otl_invoke(obj_name, name_cache, method_cache, handle,
			       class_info);
}

extern otl_handle
otl_invoke(obj_name, name_cache, method_cache, pblock, plen)
attr_list obj_name;
attr_list name_cache;
attr_list method_cache;
char *pblock;
int plen;
{
    otl_handle handle;
    struct _io_encode_vec pvector[2];
    handle = get_otl_handle();
    handle->pvector = pvector;
    handle->pveclen = 1;
    pvector[0].iov_base = pblock;
    pvector[0].iov_len = plen;
    pvector[1].iov_base = NULL;
    pvector[1].iov_len = 0;
    return internal_otl_invoke(obj_name, name_cache, method_cache, handle,
			       NULL);
}

extern void
otl_do_call_termination(handle)
otl_handle handle;
{
    int cache_attr_count = 0;

    thr_mutex_lock(handle->handle_lock);
    handle->terminated++;
    if (otl_debug) {
	printf(" Call termination.  Cached attrs: ");
	dump_attr_list(handle->cache);
    }
    thr_mutex_unlock(handle->handle_lock);
    if (handle->termination_handler) {
	handle->termination_handler(handle, handle->termination_data);
    }
    if (handle->dep != NULL) {
	/* remote invocation finishing */
	invoke_return_msg msg;
	static int invoke_return_format_id = 0;
	if (invoke_return_format_id == 0) {
	    invoke_return_format_id = 
	      DEget_format_id(otl_private_de, "invoke return message");
	}

	msg.handle = handle->orig_handle;
	cache_attr_count = attr_count(handle->cache);
	init_xmit_obj(&msg.cache, cache_attr_count);
	if (handle->cache == NULL) {
	    handle->cache = create_attr_list();
	}
	add_list_to_xmit_obj(&msg.cache, handle->cache, 
			     cache_attr_count, 0);
	msg.rveclen = 0;
	msg.rvector = handle->rvector;

	if (handle->rvector != NULL) {
	    int rveclen = 0;
	    while (msg.rvector[rveclen].iov_base != NULL) rveclen++;
	    msg.rveclen = rveclen;
	}
	DEport_write_data(handle->dep, 
			   invoke_return_format_id, &msg);
	if (otl_debug) {
	    printf(" Call termination: wrote data to %s,handle %lx\n", 
		   DEport_name(handle->dep), (long)handle->orig_handle);
	}
	free_xmit_obj(&msg.cache);
	if (handle->free_block) {
	    free(handle->free_block);
	}
    } else {
	if (handle->free_block) {
	    free(handle->free_block);
	}
	if (handle->de_condition != 0) {
	    /* invocation that was sent remotely finishing */
	    DECondition_signal(otl_private_de, handle->de_condition);
	} else {
	    /* local invocation finishing */
	    thr_condition_signal(handle->wait_condition);
	}
    }
    otl_free_handle(handle);
}

extern void
otl_wait(handle)
otl_handle handle;
{
    thr_mutex_lock(handle->handle_lock);
    if (!handle->terminated) {
	handle->wait_count++;
	if (handle->de_condition == 0) {
	    thr_condition_wait(handle->wait_condition, handle->handle_lock);
	} else {
	    int ret;
	    thr_mutex_unlock(handle->handle_lock);
	    ret = DECondition_wait(otl_private_de, handle->de_condition);
	    thr_mutex_lock(handle->handle_lock);
	    if (ret != 1) {
		CORBA_exception_set(&handle->ev, CORBA_SYSTEM_EXCEPTION,
				    "CORBA_COMM_FAILURE", NULL);
		handle->de_condition = 0;
		otl_do_call_termination(handle);
	    }
	}
	handle->wait_count--;
    }
    thr_mutex_unlock(handle->handle_lock);
}

extern void
otl_free_handle(handle)
otl_handle handle;
{
    handle->ref_count--;
    if (handle->ref_count <= 0) {
	thr_mutex_free(handle->handle_lock);
	thr_condition_free(handle->wait_condition);
	free_attr_list(handle->cache);
	free_attr_list(handle->this_invoke_attrs);
	if (handle->msg_block) {
	    DEreturn_buffer(otl_private_de, handle->msg_block);
	}
	COBS_free_Environment_object(&handle->ev);
	free(handle);
    }
}

extern attr_list
otl_get_cache(handle)
otl_handle handle;
{
    return handle->cache;
}

extern attr_list
otl_get_name(handle)
otl_handle handle;
{
    return handle->this_invoke_attrs;
}

extern void
otl_get_return_block(handle, block_p, block_len_p)
otl_handle handle;
void** block_p;
int *block_len_p;
{
    *block_p = handle->return_block;
    *block_len_p = handle->return_block_len;
}

extern void
otl_set_return_vector(handle, stack_internal_return, rvector, free_block)
otl_handle handle;
void *stack_internal_return;
IOEncodeVector rvector;
void *free_block;
{
    handle->rvector = rvector;
    handle->stack_internal_return = stack_internal_return;
    handle->free_block = free_block;
}

static void
do_threaded_call(handle)
otl_handle handle;
{
    /* 
     *  the app code may call do_call_termination.  Increment the handle 
     *  ref count to make sure it would be around to check its terminated
     *  status.  The extra otl_free_handle at the end kills it if necessary.
     */
    handle->ref_count++;
    if (handle->dispatch_addr) {
	handle->dispatch_addr(handle->self, handle->pblock, handle->plen,
			      handle->pvector, handle);
    }
    if (!handle->terminated) {
	otl_do_call_termination(handle);
    }
    otl_free_handle(handle);
}    

static attr_value
otl_get_dispatch_addr(list)
attr_list list;
{
    static atom_t otl_dispatch_style_atom = 0;
    attr_value_type dispatch_type = Attr_Undefined;
    attr_value dispatch_style;

    if (otl_dispatch_style_atom == 0) {
	otl_dispatch_style_atom = attr_atom_from_string("OTL:DISPATCH_STYLE");
    }
    query_attr(list, otl_dispatch_style_atom, &dispatch_type, 
	       (attr_value *) &dispatch_style);
    if ((dispatch_type == Attr_Int4) || (dispatch_type == Attr_Int8)) {
	((void (*)())(long)dispatch_style)(list);
    } else if (dispatch_type == Attr_Atom) {
	static atom_t otl_vf_table_style_atom;
	static atom_t otl_vf_table_atom;
	static atom_t otl_method_id_atom;
	if (otl_vf_table_style_atom == 0) {
	    otl_vf_table_style_atom = 
	      attr_atom_from_string("OTL:DISPATCH_BY_VIRTUAL_FUNCTION");
	    otl_vf_table_atom =
	      attr_atom_from_string("OTL:VIRTUAL_FUNCTION_TABLE_BASE");
	    otl_method_id_atom =
	      attr_atom_from_string("OTL:METHOD_ID");
	}
	if ((atom_t)(long)dispatch_style == otl_vf_table_style_atom) {
	    void (**vf_table) ();
	    int method_id;
	    query_attr(list, otl_vf_table_atom, NULL, (attr_value*)&vf_table);
	    query_attr(list, otl_method_id_atom, NULL,
		       (attr_value*)&method_id);
	    if(otl_debug) {
		printf("method id is %d\n", method_id);
	    }
	    return (attr_value) (long)vf_table[method_id];
	}
    }
    return (attr_value) 0;
}

void
otl_handle_invocation(this_invoke_attrs, handle)
attr_list this_invoke_attrs;
otl_handle handle;
{
    static atom_t otl_dispatch_addr = 0;
    static atom_t otl_obj_self = 0;
    static atom_t otl_is_threaded = 0;
    static atom_t no_return_atom = 0;
    void (*dispatch_addr) ();
    void *self_ptr = NULL;
    attr_list cache_list = handle->cache;
    attr_list all_attrs;
    int is_threaded = 0;

    if (otl_dispatch_addr == 0) {
	otl_is_threaded = attr_atom_from_string("OTL:IS_THREADED");
	otl_dispatch_addr = attr_atom_from_string("OTL:DISPATCH_ADDRESS");
	otl_obj_self = attr_atom_from_string("OTL:OBJECT_SELF");
	no_return_atom = attr_atom_from_string("OTL:NO_RETURN");
    }
/* otl_get_attr(otl_dispatch_addr, &dispatch_addr, obj,
 * get_disp_addr_func); */
    all_attrs = attr_join_lists(this_invoke_attrs, cache_list);
    query_attr(all_attrs, otl_is_threaded, NULL,
	       (attr_value *) &is_threaded);

    otl_get_attr(otl_dispatch_addr, Attr_Int4, (attr_value *) & dispatch_addr, 
		 all_attrs, otl_get_dispatch_addr, cache_list);
    otl_get_attr(otl_obj_self, Attr_Int4, (attr_value *) & self_ptr, 
		 all_attrs, NULL, cache_list);

    free_attr_list(all_attrs);

    add_ref_attr_list(this_invoke_attrs);
    handle->this_invoke_attrs = this_invoke_attrs;

    if(otl_debug) {
	printf("Dispatch address is %lx\n", (long) dispatch_addr);
    }
    if (is_threaded) {
	thr_thread_t thrd;
	handle->dispatch_addr = dispatch_addr;
	handle->self = self_ptr;
	if (handle->msg_block == NULL) {
	    /* 
	     * if the parameter buffer is not saved from a message,
	     * we also need to copy the params if this is a 
	     * local call but they don't want to wait for it.
	     */
	    int no_return = 0;
	    query_attr(this_invoke_attrs, no_return_atom, NULL, 
		       (attr_value *) &no_return);
	    if (no_return) {
		if (handle->plen != 0) {
		    void *new_pblock = malloc(handle->plen);
		    memcpy(new_pblock, handle->pblock, handle->plen);
		    handle->pblock = new_pblock;
		    handle->free_block = new_pblock;
		} else if (handle->pveclen != 0) {
		    /* must have a pvector.  we won't need it */
		    void *tmp_pb = malloc(handle->local_pb_size);
		    assert(handle->local_pb != NULL);
		    memcpy(tmp_pb, handle->local_pb, handle->local_pb_size);
		    handle->local_pb = tmp_pb;
		    handle->free_block = tmp_pb;
		}
	    }
	}
	thrd = thr_fork((void_arg_func)do_threaded_call, handle);
	if (thrd == NULL) {
	    printf("Fork failed!\n");
	} else {
	    thr_thread_detach(thrd);
	}
    } else {
	if (dispatch_addr) {
	    dispatch_addr(self_ptr, handle->pblock, handle->plen, NULL, 
			  handle);
	}
	if (!handle->terminated) {
	    otl_do_call_termination(handle);
	}
    }
}

void
otl_free_obj_ref(obj)
object_ref obj;
{
    free_attr_list(obj->obj_name);
    free_attr_list(obj->name_trans_cache);
    free_attr_list(obj->method_specific_cache);
    free(obj);
}


extern void
otl_get_attr(attr_name, typ, value_p, list, set_function, cache_list)
atom_t attr_name;
attr_value_type typ;
attr_value *value_p;
attr_list list;
attr_value(*set_function) ();
attr_list cache_list;
{
    attr_value_type val_type = typ;
    if (query_attr(list, attr_name, &val_type, (attr_value *) value_p)) {
	return;
    }
    if (set_function) {
	*value_p = set_function(list);
    }
    if (cache_list) {
	add_attr(cache_list, attr_name, val_type, *value_p);
    }
}

extern otl_handle
get_otl_handle()
{
    otl_handle handle = (otl_handle)malloc(sizeof(otl_handle_structure));
    handle->cache = NULL;
    handle->dep = NULL;
    handle->dispatch_addr = NULL;
    handle->self = NULL;
    handle->pblock = NULL;
    handle->plen = 0;
    handle->msg_block = NULL;
    handle->free_block = NULL;
    handle->pvector = NULL;
    handle->pveclen = 0;
    handle->local_pb = NULL;
    handle->local_pb_size = 0;
    handle->wait_count = 0;
    handle->terminated = 0;
    handle->orig_handle = 0;
    handle->this_invoke_attrs = NULL;
    handle->ref_count = 0;
    COBS_init_Environment_object(&handle->ev);
    handle->return_block = NULL;
    handle->return_block_len = 0;
    handle->rvector = NULL;
    handle->rvector_len = 0;
    handle->stack_internal_return = NULL;
    assert((handle->handle_lock = thr_mutex_alloc()) != NULL);
    assert((handle->wait_condition = thr_condition_alloc()) != NULL);
    handle->de_condition = 0;
    handle->termination_handler = NULL;
    handle->termination_data = NULL;
    return handle;
}


static char*
strdqcat(str, size_p, str2)
char *str;
int *size_p;
char *str2;
{
    int str2_len = strlen(str2);
    str = realloc(str, *size_p + str2_len + 1);
    strcpy(str + *size_p, str2);
    (*size_p) += str2_len;
    return str;
}

static char*
strdcat(str, size_p, str2)
char *str;
int *size_p;
char *str2;
{
    return strdqcat(str, size_p, str2);
}

static const char xchars [] = "0123456789abcdef";

#define nibble2hex(val) (xchars[val & 0x0f])

static char *
add_list_to_string(str, size_p, list)
char *str;
int *size_p;
attr_list list;
{
    int i;
    for (i = 0; i < attr_count(list); i++) {
	attr_p tmp = get_attr(list, i);
	char str_tmp[128];
	memset(str_tmp, 0, sizeof(str_tmp));
	if (tmp->val_type == Attr_List) continue;
	sprintf(str_tmp, "{%d", tmp->attr_id);
	str = strdcat(str, size_p, str_tmp);
	switch(tmp->val_type) {
	case Attr_Undefined:
	    str = strdcat(str, size_p, ",U,");
	    break;
	case Attr_Int4:
	    sprintf(str_tmp, ",4,%d", (int) (long)tmp->value);
	    str = strdcat(str, size_p, str_tmp);
	    break;
	case Attr_Int8:
	    sprintf(str_tmp, ",8,%ld", (long) tmp->value);
	    str = strdcat(str, size_p, str_tmp);
	    break;
	case Attr_String:
	    str = strdcat(str, size_p, ",S,");
	    str = strdcat(str, size_p, (char*)tmp->value);
	    break;
	case Attr_Opaque: {
	    attr_opaque_p o = (attr_opaque_p) tmp->value;
	    if (o == NULL) {
		str = strdcat(str, size_p, ",O0,");
	    } else {
		char tmp3[3];
		int i;
		sprintf(str_tmp, ",O%d,", o->length);
		str = strdcat(str, size_p, str_tmp);
		tmp3[2] = 0;
		for (i=0; i<o->length; i++) {
		    tmp3[0] = nibble2hex(((char*)o->buffer)[i] >> 4);
		    tmp3[1] = nibble2hex(((char*)o->buffer)[i]);
		    str = strdcat(str, size_p, tmp3);
		}
	    }
	    break;
	}
	case Attr_Atom:
	    sprintf(str_tmp, ",A,%d", (int) (long)tmp->value);
	    str = strdcat(str, size_p, str_tmp);
	    break;
	case Attr_List:
	default:
	    assert(0);
	}
	str = strdqcat(str, size_p, "},");
    }
    return str;
}

char *
obj_ref_to_string(obj)
object_ref obj;
{
    char *str = NULL;
    int size = 0;

    if (obj != NULL) {
	str = malloc(1);
	str[0] = 0;
	str = add_list_to_string(str, &size, obj->obj_name);
	str = add_list_to_string(str, &size, obj->name_trans_cache);
    }
    return str;
}

extern int otl_private_iiop_port;

char *
obj_ref_to_iiop(obj)
object_ref obj;
{
    static atom_t home_host_atom = 0;
    static atom_t iiop_port_atom = 0;
    static atom_t iiop_version_atom = 0;
    static atom_t otl_self_atom = 0;
    char *home_host_name;
    int port = -1;
    char real_port[20];
    char *ret;
    int len;
    int version = 0;
    void **state;

    if (home_host_atom == 0) {
	home_host_atom = attr_atom_from_string("OTL:HOME_HOST");
	iiop_port_atom = attr_atom_from_string("OTL:IIOP_PORT");
	iiop_version_atom = attr_atom_from_string("OTL:IIOP_VERSION");
        otl_self_atom = attr_atom_from_string("OTL:OBJECT_SELF");
    }
    if (!query_attr(obj->obj_name, home_host_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) & home_host_name)) {
    }
    if (!query_attr(obj->obj_name, iiop_port_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) & port)) {
    }
    if (!query_attr(obj->obj_name, iiop_version_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) & version)) {
    }
    if (!query_attr(obj->obj_name, otl_self_atom, NULL, (void**)&state)) {
    }
    len = strlen("iiop:1.0//") + 1;
    len += strlen(home_host_name) +1;
    sprintf(real_port, "%d", port);
    len += strlen(real_port)+1+16;
    ret = malloc(len);
    sprintf(ret, "iiop:1.%d//%s:%s/%016lx", version, home_host_name,
	    real_port, (unsigned long)state);
    return ret;
}

object_ref
iiop_to_obj_ref(obj_str)
char *obj_str;
{
    static atom_t home_host_atom = 0;
    static atom_t iiop_port_atom = 0;
    static atom_t iiop_version_atom = 0;
    static atom_t iiop_object_key_atom = 0;
    static atom_t iiop_protocol;
    static atom_t otl_protocol_atom;
    attr_opaque_p op;
    char *home_host_name;
    char *colon;
    object_ref obj;
    int port = -1;
    int version;
    int name_len;
    char obj_key[17];

    if (home_host_atom == 0) {
	home_host_atom = attr_atom_from_string("OTL:HOME_HOST");
	iiop_port_atom = attr_atom_from_string("OTL:IIOP_PORT");
	iiop_version_atom = attr_atom_from_string("OTL:IIOP_VERSION");
        iiop_object_key_atom = attr_atom_from_string("OTL:IIOP_OBJECT_KEY");
	iiop_protocol = attr_atom_from_string("OTL:IIOP_PROTOCOL");
	otl_protocol_atom = attr_atom_from_string("OTL:PROTOCOL_TAG");
    }

    obj = (object_ref) malloc(sizeof(object_ref_struct));
    obj->obj_name = create_attr_list();
    obj->name_trans_cache = create_attr_list();
    obj->method_specific_cache = create_attr_list();

    obj_str = obj_str + 5; /* skip iiop: */
    if (strncmp(obj_str, "1.", 2) == 0) obj_str += 2;
    switch (*obj_str) {
    case '0':
	version = 0;
	obj_str++;
	break;
    case '1':
	version = 1;
	obj_str++;
	break;
    default:
	printf("Bad IIOP version in string \"%s\"\n", obj_str);
	return NULL;
    }
    if (*obj_str == '/') obj_str++;
    if (*obj_str == '/') obj_str++;
    colon = strchr(obj_str, ':');
    name_len = colon - obj_str;
    home_host_name = malloc(name_len + 1);
    strncpy(home_host_name, obj_str, name_len);
    home_host_name[name_len] = 0;
    colon++;
    sscanf(colon, "%d/%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c%c", &port, 
	   &obj_key[0], &obj_key[1], &obj_key[2], &obj_key[3],
	   &obj_key[4], &obj_key[5], &obj_key[6], &obj_key[7],
	   &obj_key[8], &obj_key[9], &obj_key[10], &obj_key[11],
	   &obj_key[12], &obj_key[13], &obj_key[14], &obj_key[15]);
    obj_key[16] = 0;
    set_attr(obj->obj_name, home_host_atom, Attr_String,
	     /* value pointer */ (attr_value ) home_host_name);
    set_attr(obj->obj_name, iiop_port_atom, Attr_Int4,
		    /* value pointer */ (attr_value ) port);
    set_attr(obj->obj_name, iiop_version_atom, Attr_Int4,
		    /* value pointer */ (attr_value ) version);
    op = malloc(sizeof(attr_opaque));
    op->length = 17;
    op->buffer = malloc(17);
    memcpy(op->buffer, &obj_key[0], 17);
    set_attr(obj->obj_name, iiop_object_key_atom, Attr_Opaque, 
	     (attr_value)op);
    set_attr(obj->obj_name, otl_protocol_atom, Attr_Atom, (attr_value ) 
	     iiop_protocol);

    return obj;
}

int
has_otl_profile(object_ref obj)
{
    static atom_t home_port_atom = 0;
    int port;
    if (home_port_atom == 0) {
	home_port_atom = attr_atom_from_string("OTL:HOME_PORT");
    }
    if (!query_attr(obj->obj_name, home_port_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) &port)) {
	return 0;
    }
    return 1;
}

int
has_iiop_profile(object_ref obj)
{
    static atom_t iiop_object_key_atom = 0;
    int port;
    if (iiop_object_key_atom == 0) {
	iiop_object_key_atom = attr_atom_from_string("OTL:IIOP_OBJECT_KEY");
    }
    if (!query_attr(obj->obj_name, iiop_object_key_atom, 
		    /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) &port)) {
	return 0;
    }
    return 1;
}

void
get_obj_iiop_info(obj, host_p, port_p, opaque_p, version_p)
object_ref obj;
char **host_p;
short *port_p;
void **opaque_p;
int *version_p;
{
    static atom_t home_host_atom = 0;
    static atom_t iiop_port_atom = 0;
    static atom_t iiop_version_atom = 0;
    static atom_t otl_self_atom = 0;
    int port;
    void **state;
    char *host_name;
    int version = 0;

    if (home_host_atom == 0) {
	home_host_atom = attr_atom_from_string("OTL:HOME_HOST");
	iiop_port_atom = attr_atom_from_string("OTL:IIOP_PORT");
	iiop_version_atom = attr_atom_from_string("OTL:IIOP_VERSION");
        otl_self_atom = attr_atom_from_string("OTL:OBJECT_SELF");
    }
    if (!query_attr(obj->obj_name, home_host_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) &host_name)) {
	fprintf(stderr, "No home host attribute\n");
    }
    if (!query_attr(obj->obj_name, iiop_port_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) &port)) {
	fprintf(stderr, "No iiop port attribute\n");
    }
    if (!query_attr(obj->obj_name, iiop_version_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) &version)) {
    }
    if (!query_attr(obj->obj_name, otl_self_atom, NULL, (void**)&state)) {
	fprintf(stderr, "No self attribute\n");
    }
    *host_p = host_name;
    *opaque_p = state;
    *port_p = port;
    *version_p = version;
}

void
get_iiop_profile_info(obj, host_p, port_p, version_p, opaque_p, opaque_len_p)
object_ref obj;
char **host_p;
short *port_p;
int *version_p;
void **opaque_p;
int *opaque_len_p;
{
    static atom_t home_host_atom = 0;
    static atom_t iiop_port_atom = 0;
    static atom_t iiop_version_atom = 0;
    static atom_t iiop_object_key_atom = 0;
    int port;
    int version = 0;
    char *host_name;
    attr_opaque_p obj_key;

    if (home_host_atom == 0) {
	home_host_atom = attr_atom_from_string("OTL:HOME_HOST");
	iiop_port_atom = attr_atom_from_string("OTL:IIOP_PORT");
	iiop_version_atom = attr_atom_from_string("OTL:IIOP_VERSION");
        iiop_object_key_atom = attr_atom_from_string("OTL:IIOP_OBJECT_KEY");
    }
    if (!query_attr(obj->obj_name, home_host_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) &host_name)) {
	fprintf(stderr, "No home host attribute\n");
    }
    if (!query_attr(obj->obj_name, iiop_port_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) &port)) {
	fprintf(stderr, "No iiop port attribute\n");
    }
    query_attr(obj->obj_name, iiop_version_atom, /* type pointer */ NULL,
		    /* value pointer */ (attr_value *) &version);

    if (!query_attr(obj->obj_name, iiop_object_key_atom, NULL, 
		    (attr_value*)&obj_key)) {
	fprintf(stderr, "No object key attribute\n");
    }
    *host_p = host_name;
    *version_p = version;
    *opaque_p = obj_key->buffer;
    *opaque_len_p = obj_key->length;
}

static
unsigned char
hex2byte (char c)
{
    if (isdigit ((int)c))
	return (unsigned char) (c - '0');
    else if (islower ((int)c))
	return (unsigned char) (10 + c - 'a');
    else
	return (unsigned char) (10 + c - 'A');
}

static int
add_list_from_string(str, list)
char *str;
attr_list list;
{
    while (1) {
	char *first_comma = strchr(str, ',');
	char *value = first_comma ? strchr(first_comma+1, ',') : NULL;
	char *end = NULL;
	int attr_id, length;
	attr_value_type val_type = Attr_Undefined;
	attr_value val;
	
	if (*str == 0) return 1;   /* success */
	if (value == NULL) return 0;
	if (sscanf(str, "{%d,", &attr_id) != 1) return 0;
	switch(*(first_comma+1)) {
	case 'U':
	    val_type = Attr_Undefined;
	    end = value + 2;
	    break;
	case '4':
	    val_type = Attr_Int4;
	    if (sscanf(value, ",%d", (int*)&val) != 1) return 0;
	    end = strchr(value+1, ',') + 1;
	    if (end == (char*)1) end = value + strlen(value);
	    break;
	case '8':
	    val_type = Attr_Int8;
	    if (sscanf(value, ",%ld", (long*)&val) != 1) return 0;
	    end = strchr(value+1, ',') + 1;
	    if (end == (char*)1) end = value + strlen(value);
	    break;
	case 'A':
	    val_type = Attr_Atom;
	    if (sscanf(value, ",%d", (int*)&val) != 1) return 0;
	    end = strchr(value+1, ',') + 1;
	    if (end == (char*)1) end = value + strlen(value);
	    break;
	case 'S':
	    val_type = Attr_String;
	    end = strchr(value+1, '}');
	    value += 1; /* skip "," */
	    length = end - value + 1;
	    val = malloc(length);
	    strncpy(val, value, length);
	    ((char*)val)[length-1] = 0; /* terminate string */
	    end += 2;
	    break;
	case 'O': {
	    int len;
	    if (sscanf(first_comma + 2, "%d", &len) != 1) return 0;
	    while (*value != ',') value++; /* skip to start of buffer */
	    value++;
	    val_type = Attr_Opaque;
	    if (len == 0) {
		val = (attr_value) NULL;
	    } else {
		attr_opaque_p o = malloc(sizeof(attr_opaque));
		int i;
		val = (attr_value)o;
		o->length = len;
		o->buffer = malloc(len);
		for (i=0 ; i < len; i++) {
		    unsigned char	byte;
		    byte = (unsigned char) (hex2byte (value[0]) << 4);
		    byte |= hex2byte (value [1]);
		    ((char*)o->buffer)[i] = byte;
		    value += 2;
		}
	    }
	    end = value + 2;
	    break;
	}    
	default:
	    assert(0);
	}
	add_attr(list, attr_id, val_type, val);
	str = end;
    }
}

extern object_ref
COBS_string_to_object(char *name);
    
object_ref
obj_ref_from_string(str)
char *str;
{
    object_ref obj = NULL;
    if (str == NULL) return NULL;
    if (str[0] == 0) return NULL;

    if (str[0] == '{') {
	/* must be OTL-style attribute list */
	obj = (object_ref) malloc(sizeof(object_ref_struct));
	obj->obj_name = create_attr_list();
	obj->name_trans_cache = create_attr_list();
	obj->method_specific_cache = create_attr_list();
	add_list_from_string(str, obj->obj_name);
    } else if (((str[0] == 'I') || (str[0] == 'i')) &&
	       ((str[1] == 'O') || (str[1] == 'o')) &&
	       ((str[2] == 'R') || (str[2] == 'r')) &&
	       (str[3] == ':')) {
	obj = COBS_string_to_object(str);
    } else if (((str[0] == 'I') || (str[0] == 'i')) &&
	       ((str[1] == 'I') || (str[1] == 'i')) &&
	       ((str[2] == 'O') || (str[2] == 'o')) &&
	       ((str[3] == 'P') || (str[3] == 'p')) &&
	       (str[4] == ':')) {
	return iiop_to_obj_ref(str);
    } else {
	printf("Can't determine reference type from string \"%s\"\n", str);
    }
    return obj;
}

extern void 
otl_copy_local_pb(p, handle)
void **p;
otl_handle handle;
{
    *p = handle->local_pb;
}

CORBA_Environment*
otl_ev_handle(handle)
otl_handle handle;
{
    return &handle->ev;
}

void
otl_get_exception(handle, ev)
otl_handle handle;
CORBA_Environment *ev;
{
    ev->_major = handle->ev._major;
    handle->ev._major = CORBA_NO_EXCEPTION;
    ev->_minor = handle->ev._minor;
    handle->ev._minor = NULL;
    ev->exception_data = handle->ev.exception_data;
    handle->ev.exception_data = NULL;
}

typedef struct _Sensor_Type_ObjectExists {
    char*           object_class;
    char*           object_reference;
    char*           object_channel;
} ObjectExistsEvent, *ObjectExistsPtr;

/*** sensor ObjectExists ***/
static IOField ObjectExistsFields[] = {
    {"object_class", "string", sizeof(char*),
        IOOffset(ObjectExistsPtr, object_class)},
    {"object_reference", "string", sizeof(char*),
        IOOffset(ObjectExistsPtr, object_reference)},
    {"object_channel", "string", sizeof(char*),
        IOOffset(ObjectExistsPtr, object_channel)},
    {NULL, NULL, 0, 0}
};

static ObjectExistsEvent *obj_list = NULL;
static int obj_count = 0;
static IOFormat exists_format = NULL;
extern ECSourceHandle master_handle;

extern void
otl_mon_register_object(class, obj_ref, chan_id)
char *class;
char *obj_ref;
char *chan_id;
{
    void *free_block;
    IOEncodeVector encoded_statev;

    if (obj_list == NULL) {
	obj_list = malloc(sizeof(obj_list[0]));
	otl_set_pbio_type(&exists_format, "Object Exists", ObjectExistsFields);
    } else {
	obj_list = realloc(obj_list, sizeof(obj_list[0]) * (obj_count + 1));
    }
    obj_list[obj_count].object_class = class;
    obj_list[obj_count].object_reference = strdup(obj_ref);
    obj_list[obj_count].object_channel = strdup(chan_id);
    encoded_statev = otl_pbio_type_encodev(exists_format, &obj_list[obj_count],
					   &free_block);
    obj_count++;
    ECsubmit_eventV(master_handle, encoded_statev);
    /* freeing the free_block also gets encoded_statev */
    free(free_block);
}

static int moss_debug_flag = -1;

extern void
otl_mon_subscribe_handler(int subscribe, int subscribe_count,
			  void *client_data)
{
    int i;
    void *free_block;
    IOEncodeVector encoded_statev;

    if (moss_debug_flag) {
	printf("Got a new subscriber...  sending him %d things\n", obj_count);
    }
    for(i=0; i < obj_count; i++) {
	if (moss_debug_flag) {
	    printf("Before encodev\n");
	}
	encoded_statev = otl_pbio_type_encodev(exists_format, &obj_list[i],
					       &free_block);
	if (moss_debug_flag) {
	    printf("Before submit %lx\n", (long)master_handle);
	}
	ECsubmit_eventV(master_handle, encoded_statev);
	/* freeing the free_block also gets encoded_statev */
	free(free_block);
    }
}

void
dump_obj_ref(object_ref o)
{
    printf("Obj_name ");
    dump_attr_list(o->obj_name);
    printf("Name_cache ");
    dump_attr_list(o->name_trans_cache);
    printf("Method_cache ");
    dump_attr_list(o->method_specific_cache);
}

typedef CORBA_Object (*create_func_t)();
typedef void (*update_func_t)(void *update_event, int length, void *client_data);

typedef struct {
    char *class;
    create_func_t create_func;
    update_func_t update_func;
} class_rec, *class_rec_ptr;

static int num_classes = 0;
static class_rec_ptr class_list = NULL;

void
MOSS_add_class(class, create_func_p, update_func_p)
char *class;
void *create_func_p;
void *update_func_p;
{
    create_func_t create_func = (create_func_t)(long)create_func_p;
    update_func_t update_func = (update_func_t)(long)update_func_p;
    if (class_list == NULL) {
	class_list = malloc(sizeof(class_rec));
    } else {
	class_list = realloc(class_list, sizeof(class_rec) * (num_classes +1));
    }
    class_list[num_classes].class = class;
    class_list[num_classes].create_func = create_func;
    class_list[num_classes].update_func = update_func;
    num_classes++;
}

static DExchange moss_private_de = NULL;

static
void 
handle_channel(event, length, client_data)
void *event;
int length;
void *client_data;
{
    ObjectExistsEvent *exists_event;
    int i;
    if (!otl_check_conversion(event)) {
	otl_list_of_formats format_list[] = {
	    {"Object Exists", ObjectExistsFields},
	    {NULL, NULL}};
	otl_set_pbio_types(event, format_list);
    }
    exists_event = otl_pbio_type_decode(event, length);
    if (moss_debug_flag) {
	printf("Got Object Exists notification for object of class %s\n",
	       exists_event->object_class);
	printf("   Obj ref is %s, channel is %s\n", 
	       exists_event->object_reference, exists_event->object_channel);
    }
    for (i=0; i < num_classes; i++) {
	if (strcmp(exists_event->object_class, class_list[i].class) == 0) {
	    CORBA_Object o = class_list[i].create_func();
	    static int otl_remote_object_atom = -1;
	    EChannel chan;
	    ECSinkHandle handle;

	    if (moss_debug_flag) {
		printf("     Creating mirror object\n");
	    }
	    chan = EChannel_open(moss_private_de,exists_event->object_channel);
	    handle = ECsink_subscribe(chan, class_list[i].update_func, o);
	    assert(handle != NULL);
		
	    if (otl_remote_object_atom == -1) {
		otl_remote_object_atom =
		    attr_atom_from_string("OTL:REMOTE_OBJECT_STR");
	    }
	    add_attr(o->obj_name, otl_remote_object_atom, Attr_String,
		     (attr_value) strdup(exists_event->object_reference));
	    return;
	}
    }
}

void
MOSS_Init(char *group_name, DExchange de)
{

    if (moss_private_de == NULL) {
	comm_group_return groups;
	EChannel chan;

	moss_private_de = de;

	if (moss_debug_flag == -1) {
	    char *deb = getenv("MOSS_DEBUG");
	    moss_debug_flag = 0;
	    if (deb) {
		if (sscanf(deb, "%d", &moss_debug_flag) != 1) {
		    /* enable at lowest level */
		    moss_debug_flag = 1;
		}
	    }
	}

	init_cobs_DE(NULL, NULL, de);

	groups = matching_comm_groups(group_name, "cobs");

	chan = DEchannel_initiate_first(moss_private_de, groups);

	if (chan == NULL) {
	    fprintf(stderr, "MOSS_Init failed.  No channels for group \"group_name\"\n");
	    exit(0);
	}
	ECsink_subscribe(chan, handle_channel, NULL);
    }
}
