/*
 * threads.c
 */


#include "config.h"
#include	<signal.h>

#include <stdio.h>
#include <sys/types.h>
#ifdef HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif

#include	"results.h"
#include	"general.h"
#include	"lock.h"
#include	"queue.h"
#include	"sync.h"
#include	"internal.h"
#ifdef WITH_MONITORING
#include	"monitor.h"
#endif
#include 	"mon.sensorextern.h"
#include	"hwlib.h"
#include	"threads.h"
#include	"memalloc.h"
#include	"arch_init.h"

exported int    next_thread_type;

extern  shared	struct  local_info  *processor;
extern  private	locked_queue_t      local_thread_queue[MAX_PROC];
extern  cthread_t current_thread_array[MAX_PROC];
extern 		int 	            next_thread_type;

extern  int             monitor_process ARGS((int lm_node));
extern  int		steer_process ARGS((int node));

extern	shared	int	*num_of_procs;

extern	atom_int	*total_threads;
extern	atom_int	*total_active;

extern	private	sched_info_t	local_scheduler[MAX_PROC];
extern	short *processes;

struct desc {		      /* stuff for dotell(),tell(),cthread_publish() */
	any_t value;
	any_t *pointer;
};

typedef struct desc *desc_t;

/* cthreads functions used here */
extern void internal_cthread_exit ARGS((any_t res));
extern void internal_cthread_yield();
extern void app_cthread_exit_notification();
extern void child_exit();
extern void add_thread_to_lm_list ARGS((cthread_t thrd));
extern void dotell ARGS((desc_t desc));

static void thread_free();

void
start_thread(func, arg)
any_t		(*func)();
any_t	arg;
{
   any_t	tmp;
   cthread_t current_thread = current_thread_array[virtual_processor()];

   DBG2("Entered in Start thread, func is %lx arg is %lx\n",
         (long)func, (long)arg);
   tmp = (*func)(arg);
   DBG2("exit from thread %x with result %x\n",(int)current_thread,(int)tmp);
   internal_cthread_exit((any_t)tmp);
   /* cthread_exit((*func)(arg)); */
}

void
start_threadV(func, n, arg)
any_t	(* func)();
int	n;
any_t	arg;
{
   any_t		*args;

   DBG2("Entered in Start thread(V), func is %lx n is %ld\n",
         (long)func, (long)n);
   args = &arg;
   internal_cthread_exit((*func)(n,args));
}

void
internal_barrier()
{
   cthread_t current_thread = current_thread_array[virtual_processor()];

    current_thread->status |= S_ON_BARRIER;
    while (*total_threads >1 ) {
	internal_cthread_yield();
    }
    current_thread->status &= (~S_ON_BARRIER);
}

void
internal_cthread_exit(res)
any_t	res;
{
   int		i;
   cthread_t current_thread = current_thread_array[virtual_processor()];

   if ((current_thread->status & S_INTERNAL) == 0) {
       app_cthread_exit_notification();
       DM_Sensor_thread_exit();
   }

   if (current_thread->status & S_MAIN) {
      internal_barrier();
   }	
   atomadd(total_threads, -1);
   atomadd(total_active, -1);



   processor[virtual_processor()].load -= 1;
   atomadd(&(local_scheduler[virtual_processor()]->active_threads), -1);
   DBG3("Processor %d terminating, total_active is %d, num_procs is %d\n", 
	virtual_processor(), *total_active, *num_of_procs);
   if (*total_active == 0) { /* The last thread exited. Signal the	*/
#ifdef WITH_MONITORING
       *monitor_status = (short) MONITOR_SHOULD_EXIT;
#endif
       for (i=0; i< *num_of_procs; i++) { /* termination of the	*/
	   processor[i].terminate = TERMINATE;/* library.	*/
       }
       child_exit(0);
   }
   internal_mutex_lock(&current_thread->tlock);
   current_thread->status |= S_FINISHED;
   current_thread->return_value = res;
   internal_mutex_unlock(&current_thread->tlock);
   if(current_thread->status & S_DETACHED){
      thread_free();
      local_scheduler[virtual_processor()]->schedule();
   }
   internal_condition_signal(&current_thread->done, NULL);
   for(;;){
      internal_mutex_lock(&current_thread->tlock);
      if(current_thread->synced == 1){
         internal_mutex_unlock(&current_thread->tlock);
         thread_free();
         local_scheduler[virtual_processor()]->schedule();
      } else {	/* _rsp twice in succession will cause a crash */
	  internal_mutex_unlock(&current_thread->tlock);
      }
      if (processor[virtual_processor()].terminate == TERMINATE) {
	  child_exit(0);
      }
      internal_cthread_yield();
   }
}


cthread_t
cthread_thread_alloc(func, arg, node)
any_t		(*func)();
any_t	arg;
int		node;
{
    int		i,tmp;
    cthread_t	newthread;
    int		numproc;

#ifdef WITH_MONITORING
    if (monitoring_enabled && func == (any_t(*)())monitor_process)
	numproc = *num_of_procs+number_of_lms;
    else if (steering_enabled && func == (any_t(*)())steer_process)
	numproc = *num_of_procs+number_of_lms+1;
    else if (adv_steer_enabled && func == (any_t(*)())adv_steer_thread)
	numproc = *num_of_procs+number_of_lms+1;
    else if (func == (any_t(*)()) dotell)
	numproc = *num_of_procs+number_of_lms+1;
    else
#endif
	numproc = *num_of_procs;

    if (node == N_CURRENT) {
	node = virtual_processor();
	DBG1("Allocating Monitor on node %i\n",node);
    } else if(node == N_ANYWHERE || node >= numproc || node <= N_LESS){
	node = 0;
	tmp = processor[0].load;
	for (i = 1; i < numproc; i++) {
	    if( tmp >= processor[i].load){  /* the = is to reduce load on 0 */
		node = i;
		tmp = processor[i].load;
	    }
	}
	DBG1("In fork:: node = %d \n",node);
    }
	
    if ((newthread = (cthread_t)locked_dequeue((processor[node].scheduler)->free_list)) == 0){
	return NULL;
    }

    /*
     *	Normally the following will not be necessary.
     */

    newthread->name = 0;
    newthread->status = S_INITIALLY;
    newthread->thread_data = 0;
    newthread->home    =(short) node;
    newthread->on_node =(short) node;
    newthread->agent   = 0;		/* using own stack		*/
    newthread->synced = 0;
    newthread->already_started = 0;
    newthread->func = func;
    newthread->sensor_switch = TRUE;
    newthread->mon_info = NULL;
    newthread->arg = (long *) arg;
    newthread->fork_count++;
    atomadd(total_threads,1);
    return newthread;
}

void
cthread_thread_schedule(newthread)
cthread_t newthread;
{
    int node = newthread->on_node;
    int local_proc = virtual_processor();
    atomadd(total_active, 1);
    atomadd(&processor[node].load, 1);
    if (local_scheduler[local_proc]->sched_thread_init !=0)
      local_scheduler[local_proc]->sched_thread_init();
#ifdef WITH_MONITORING
    if (monitoring_enabled
	&& (newthread->func!=(any_t(*)())steer_process))
      newthread->mon_info = monitor_init_thread((void*)newthread,
						newthread->mon_info);
#endif
    (processor[node].scheduler)->sched_put_thread(node,newthread);
    atomadd(&((processor[node].scheduler)->active_threads), 1);
    DBG("out of fork stuff \n");
}

cthread_t
cthread_fork(func, arg, node)
any_t		(*func)();
any_t	arg;
int		node;
{
   cthread_t	newthread;

   DBG3("TRYING to fork a thread in node %d with func = %x, arg = %x\n", 
	node, (int)func, (int)arg);

   newthread = cthread_thread_alloc(func, arg, node);
   if (newthread != NULL) cthread_thread_schedule(newthread);
   return(newthread);
}


#ifdef Divine_intervention
void
cthread_sleep(time)
int		time;
{
   cthread_t current_thread = current_thread_array[virtual_processor()];
   internal_mutex_lock(&current_thread->tlock);
   current_thread->actat   = getrtc() + time;
   current_thread->status |= S_SLEEPING;
   internal_mutex_unlock(&current_thread->tlock);
   internal_cthread_yield();   
}

void
cthread_wakeup(thread)
cthread_t	thread;
{
   cthread_t current_thread = current_thread_array[virtual_processor()];
   internal_mutex_lock(&thread->tlock);
   current_thread->status &= ~S_SLEEPING;
   internal_mutex_unlock(&thread->tlock);
}
#endif

RESULT
internal_cthread_detach(t)
cthread_t	t;
{
   internal_mutex_lock(&t->tlock);
   if(t->status & (S_JOINING | S_FREED)){
      internal_mutex_unlock(&t->tlock);
      return(T_JOINING_OR_FREED);
   }
   t->status |= S_DETACHED;
   if (t->status & S_FINISHED)
      t->synced = 1;
   internal_mutex_unlock(&t->tlock);

   return(T_SUCCEED);
}

RESULT
internal_cthread_join(thr, resptr)
cthread_t	thr;
any_t		*resptr;
{
   any_t	tmp;
   cthread_t	t;
   cthread_t current_thread = current_thread_array[virtual_processor()];

   t = thr;
   if(t == current_thread)
      return(T_CANNOT_JOIN_MYSHELF);
   internal_mutex_lock(&t->tlock);
   if(t->status & (S_DETACHED | S_FREED | S_JOINING)){
      internal_mutex_unlock(&t->tlock);
      return(T_CANNOT_JOIN);
   }
   if(t->status & S_FINISHED){
      tmp = t->return_value;
      t->synced = 1;
      t->status |= S_JOINING;	/* To avoid future joinings 	*/
      internal_mutex_unlock(&t->tlock);
      if (resptr != NULL) *resptr = tmp;
      return(T_SUCCEED);
   }
   t->status |= S_JOINING;

   internal_condition_wait(&t->done, &t->tlock);

   tmp = t->return_value;
   t->synced = 1;
   internal_mutex_unlock(&t->tlock);
   if (resptr != NULL) *resptr = tmp;
   return(T_SUCCEED);
}

static void
thread_free()
{
   locked_queue_t	tfreelist;
   int proc = virtual_processor();
   cthread_t current_thread = current_thread_array[proc];

   DBG1("freeing Thread %lx\n",(long)current_thread);
   internal_mutex_lock(&current_thread->tlock);
   current_thread->status |= S_FREED;
   current_thread->already_started = 0;
   internal_mutex_unlock(&current_thread->tlock);

   tfreelist = (processor[current_thread->home /*on_node*/].scheduler)->free_list;

   cthread_spin_lock(&tfreelist->lqlock);
   enqueue(&tfreelist->lcqueue, (queue_item_t) current_thread);
   cthread_spin_unlock(&tfreelist->lqlock);
   DBG2("in thread_free, Now first is %x, tail is %x\n",
	(int)local_thread_queue[proc]->lcqueue.first,
	(int)local_thread_queue[proc]->lcqueue.tail);
}

cthread_t
cthread_self()
{
   cthread_t current_thread = current_thread_array[virtual_processor()];
   return(current_thread);
}

any_t
cthread_data(t)
cthread_t	t;
{
   return(t->thread_data);
}

void
cthread_set_data(t, data)
cthread_t	t;
any_t		data;
{
   t->thread_data = data;
}

void *
cthread_sched_info(t)
cthread_t	t;
{
   return(t->sched_info);
}

void
cthread_set_sched_info(t, info)
cthread_t	t;
void		*info;
{
   t->sched_info = info;
}

string_t
cthread_name(t)
cthread_t	t;
{
   return(t->name);
}

void
cthread_set_name(t, tname)
cthread_t	t;
string_t	tname;
{
    char *tmp_name;
/*   *NOTE*  This is a memory leak!  Cthread_set_name never frees memory!
     See the Todo file for notes on this....

    if (t->name != NULL) {
	memory_free((memory_t) t->name);
    }
*/
    if (tname != NULL) {
	RESULT res;
	res = memory_alloc((memory_t *) &tmp_name, strlen(tname) + 1,
			   virtual_processor());
	if (res != T_SUCCEED) {
	    t->name = NULL;
	    return;
	}

	strcpy(tmp_name, tname);
	tname = tmp_name;
    }
    t->name = tname;
    if (cthread_self() != NULL) {
	/* 
	 * cthread_self() is null if we're in setup.  Shouldn't generate a
	 * sensor then anyway.
	 */
	DM_Sensor_thread_set_name(t, tname);
    }
}

