#include <signal.h>
#include <malloc.h>

#include "general.h"
#include "sched_utils.h"
#include "config.h"
#include "init.h"
#include "arch_init.h"

struct sigaction sig_handlers[MAX_PROC][NO_OF_SIGNALS];
static struct sigaction newvec, oldvec;
static struct sigaction *old_sigpipe_hndlr, *old_sigalrm_hndlr;

/* unix externs */
extern void perror();

#ifdef SA_STACK
#define STACK_FLAG SA_STACK
#else
#ifdef SA_ONSTACK
#define STACK_FLAG SA_ONSTACK
#else
#define STACK_FLAG 0
#endif
#endif

#ifndef SA_SIGINFO
#define SA_SIGINFO 0
#endif



void
setup_child_sighandlers(ill_hndlr, seg_hndlr, bus_hndlr)
void (*ill_hndlr)();
void (*seg_hndlr)();
void (*bus_hndlr)();
{
#ifdef HAVE_SIGALTSTACK
   struct sigaltstack	sigst;
   sigst.ss_sp = malloc(10240);
   sigst.ss_size = 10240;
   sigst.ss_flags = 0;
   if (sigaltstack(&sigst, NULL) != 0) {
       perror("sigstack");
   }
#endif
   if (allow_core_dumps) return;

   DBG1("Installing child signal handlers in proc %d\n", virtual_processor());
   if (ill_hndlr) {
       newvec.sa_handler = ill_hndlr;
       sigemptyset(&newvec.sa_mask);
       sigaddset(&newvec.sa_mask, SIGILL );
       newvec.sa_flags = STACK_FLAG | SA_SIGINFO;
       DBG1("Installing SIGILLsignal handler in proc %d\n", 
	    virtual_processor());
       if (sigaction( SIGILL, &newvec, &oldvec ) != 0) perror("sigaction");
   }
   if (seg_hndlr) {
       newvec.sa_handler = seg_hndlr;
       sigemptyset(&newvec.sa_mask);
       sigaddset(&newvec.sa_mask, SIGSEGV );
       newvec.sa_flags = STACK_FLAG | SA_SIGINFO;
       DBG1("Installing SIGSEGV signal handler in proc %d\n", 
	    virtual_processor());
       if (sigaction( SIGSEGV, &newvec, &oldvec ) != 0) perror("sigaction");
   }
   if (bus_hndlr) {
       newvec.sa_handler = bus_hndlr;
       sigemptyset(&newvec.sa_mask);
       sigaddset(&newvec.sa_mask, SIGBUS );
       newvec.sa_flags = STACK_FLAG | SA_SIGINFO;
       DBG1("Installing SIGBUS signal handler in proc %d\n", 
	    virtual_processor());
       if (sigaction( SIGBUS, &newvec, &oldvec ) != 0) perror("sigaction");
   }
}

void  setup_term_sighandler(term_hndlr)
void (*term_hndlr)();
{
   if (term_hndlr) {
       newvec.sa_handler = term_hndlr;
       sigemptyset(&newvec.sa_mask);
       sigaddset(&newvec.sa_mask, SIGTERM );
       newvec.sa_flags = SA_SIGINFO;
       DBG1("Installing SIGTERM signal handler in proc %d\n", 
	    virtual_processor());
       if (sigaction( SIGTERM, &newvec, &oldvec ) != 0) perror("sigaction");
   }
 }

void  setup_chld_sighandler(chld_hndlr)
void (*chld_hndlr)();
{
   if (chld_hndlr) {
       newvec.sa_handler = chld_hndlr;
       sigemptyset(&newvec.sa_mask);
       sigaddset(&newvec.sa_mask, SIGCHLD );
       newvec.sa_flags = SA_SIGINFO;
       DBG1("Installing SIGCHLD signal handler in proc %d\n", 
	    virtual_processor());
       if (sigaction( SIGCHLD, &newvec, &oldvec ) != 0) perror("sigaction");
   }
 }

void
setup_scheduler_sighandlers()
{
    int sig_no;
    int node = virtual_processor();

    /* install the signal handlers */
    for (sig_no = 0; sig_no < NO_OF_SIGNALS; sig_no++) {
	if (sig_handlers[node][sig_no].sa_handler != 0) {
	    DBG1("Overwriting handler for signal %d\n", sig_no);
	    if (sigaction( sig_no, &sig_handlers[node][sig_no], &oldvec) == -1) {
		perror("Signal not installed");
	    }
	}
    }
}

void
init_sighandler_struct()
{
    int i;
    int j;
    for (i = 0; i < MAX_PROC; i++) {
	for (j = 0; j < NO_OF_SIGNALS; j++) {
	    sig_handlers[i][j].sa_flags = 0;
	}
    }
    for (j = 1; j < NO_OF_SIGNALS; j++) {
	sigaddset(&sig_handlers[0][j].sa_mask, j);
    }
    for (i = 1; i < MAX_PROC; i++) {
	for (j = 0; j < NO_OF_SIGNALS; j++) {
	    sig_handlers[i][j].sa_mask = sig_handlers[0][j].sa_mask;
	}
    }
}

void
fill_sighandler_struct(node, alert_hndlrs)
int node;
void (*alert_hndlrs[NO_OF_SIGNALS])();
{
    int j;
    for (j = 0; j < NO_OF_SIGNALS; j++) {
	sig_handlers[node][j].sa_handler = alert_hndlrs[j];
    }
}

void
setup_master_sighandlers(intr_hndlr, quit_hndlr)
void (*intr_hndlr)();
void (*quit_hndlr)();
{
    if (intr_hndlr) {
	newvec.sa_handler = intr_hndlr;
	sigemptyset(&newvec.sa_mask);
	sigaddset(&newvec.sa_mask, SIGINT );
	newvec.sa_flags = 0;
	sigaction( SIGINT, &newvec, &oldvec );

        /* add handlers for all the signals that might crash the
         * master and leave the children hanging around
         */
	sigemptyset(&newvec.sa_mask);

#ifdef SIGEMT
	sigaction( SIGEMT, &newvec, &oldvec );
#endif
	sigaction( SIGFPE, &newvec, &oldvec );
	sigaction( SIGHUP, &newvec, &oldvec );
	sigaction( SIGIOT, &newvec, &oldvec );
	sigaction( SIGPIPE, &newvec, &oldvec );
        if( oldvec.sa_handler != SIG_IGN && 
             oldvec.sa_handler != SIG_DFL) {
         	sigaction( SIGPIPE, &oldvec, (struct sigaction *)0);
	}
	sigaction( SIGPOLL, &newvec, &oldvec );
#ifdef SIGEMT
	sigaction( SIGEMT, &newvec, &oldvec );
#endif
	sigaction( SIGUSR1, &newvec, &oldvec );
	sigaction( SIGUSR2, &newvec, &oldvec );
    }
    if (quit_hndlr) {
	newvec.sa_handler = quit_hndlr;
	sigemptyset(&newvec.sa_mask);
	sigaddset(&newvec.sa_mask, SIGQUIT );
	newvec.sa_flags = 0;
	sigaction( SIGQUIT, &newvec, &oldvec );
    }
}



void
setup_sigpipe_handler(hndlr)
void (*hndlr)();
{
  if( hndlr != NULL) {
	newvec.sa_handler = hndlr;
	sigemptyset(&newvec.sa_mask);
	sigaddset(&newvec.sa_mask, SIGPIPE);
	newvec.sa_flags = STACK_FLAG;
	sigaction( SIGPIPE, &newvec, old_sigpipe_hndlr);
      }
  else 
  if( old_sigpipe_hndlr != NULL) {
	newvec.sa_handler = (void (*)())old_sigpipe_hndlr;
	sigemptyset(&newvec.sa_mask);
	sigaddset(&newvec.sa_mask, SIGPIPE);
	newvec.sa_flags = STACK_FLAG;
	sigaction( SIGPIPE, &newvec, NULL);
  }
}

void
setup_sigalrm_handler(hndlr)
void (*hndlr)();
{
  if( hndlr != NULL) {
	newvec.sa_handler = hndlr;
	sigemptyset(&newvec.sa_mask);
	sigaddset(&newvec.sa_mask, SIGALRM);
	newvec.sa_flags = STACK_FLAG;
	sigaction( SIGALRM, &newvec, old_sigalrm_hndlr);
      }
  else 
  if( old_sigalrm_hndlr != NULL) {
	newvec.sa_handler = (void (*)())old_sigalrm_hndlr;
	sigemptyset(&newvec.sa_mask);
	sigaddset(&newvec.sa_mask, SIGALRM);
	newvec.sa_flags = STACK_FLAG;
	sigaction( SIGALRM, &newvec, NULL);
  }
}
