/*  File lm.c
 *
 *  This file contains the source code for the local monitor process
 *  running with the cthreads package
 *
 */

#include "config.h"
#ifdef HAVE_WINDOWS_H
#include <windows.h>
#include <process.h>
#define getpid() _getpid()
#else
#include <unistd.h>
#endif
#include <sys/types.h>
#include <fcntl.h>
#include <stdio.h>
#ifdef HAVE_POLL_H
#include <poll.h>
#endif

#include        "results.h"
#include        "general.h"
#include        "lock.h"
#include        "queue.h"
#include        "sync.h"
#include        "internal.h"
#include	"memalloc.h"
#include	"init.h"
#include	"arch_init.h"

#include        "threads.h"
#include        "monitor.h"
#include	"monitor_ur.h"

extern private int current_lm;

extern void cthread_perror ARGS((char *msg, RESULT err));
extern void child_exit ARGS((int));
extern void internal_cthread_yield();

/* random Unix externals */
extern void relinquish_processor();

int
SendProbeData(type, proc, sensor, int_fld, float_fld, str_fld)
    short type, proc, sensor;
    long    int_fld;
    double  float_fld;
    char    *str_fld;
{
    ProbeData  prb_data;
    prb_data.type = type;
    prb_data.proc = proc;
    prb_data.sensor = sensor;
    prb_data.int_data = int_fld;
    prb_data.float_data = float_fld;
    if (str_fld) prb_data.str_data = str_fld;
    else prb_data.str_data = "an empty string";
    write_IOfile(steer_outfile, steer_resp_fmt, &prb_data);
    return 0;
}

extern int
handle_simple_steer_command(de, dep, format_id, steer_cmd, length)
DExchange de;
DEPort dep;
int format_id;
URSimpleCommandPtr steer_cmd;
int length;
{
    int steer_type = steer_cmd->type;

    switch (steer_type) {
    case UR_STEER_TEST:
	SendProbeData(steer_type, 32366, 22195, 122689,
		      9876.54321, "just a dumb test message");
	break;
    case UR_STEER_SIMPLE:
	break;
    case UR_STEER_FLOAT:
	break;
    case UR_STEER_NAMED:
	break;
	/* default: */
    }
    return 0;
}

int
steer_process(node)
int node;
{
    char *buf = allocate_and_copy(size_of_monitor_buffer);
    int i, num_alive_lms = number_of_lms;
    int lm_alive_flag[MAX_NUMBER_OF_LMS];
    SimpleTraceDataRecordPtr  simple_event = (SimpleTraceDataRecordPtr) buf;
    int size, monitor_exit_type;

    DBG1("Steering process starting %lx\n", (long)getpid());
    monitor_exit_type = monitor_get_sensor_type_value("monitor_exit");

    if (start_socket_connection(&steer_info->de, &steer_info->dep) == -1) {
	DBG1("steer: failed to connect to the central monitor ... steering off %lx\n", (long)getpid());
	steering_enabled = FALSE;
	internal_cthread_exit ((any_t) -1);
    }
    DBG1("Steering process after start_socket_connection  %lx \n", (long)getpid());
    init_steer_output_conn (steer_info);

    for (i = 0; i < number_of_lms; lm_alive_flag[i++] = TRUE);
    while (num_alive_lms > 0) {
	/* any steering commands from steering socket ? */
	if (monitor_output == SOCKET_OUTPUT) {
/*	  DBG1("******************** steering polling. %lx..\n", pollfds);*/
	    DExchange_poll_and_handle(steer_info->de, FALSE);
	}
	for (i = 0; i < number_of_lms; i++) {
	  DBG1("* %lx ******Waiting to retrieve...\n", (long)getpid());
	    if (lm_alive_flag[i] &&
		RetrieveBuffer(&lm_to_steer_bufs[i], (char*) buf,
			       (unsigned int*) &size)) {
		DBG("********************Steering process got a buffer\n");
		if (simple_event->type == monitor_exit_type) {
		    lm_alive_flag[i] = FALSE;
		    num_alive_lms--;
		}
	    }
	}
	internal_cthread_yield();
	relinquish_processor();
/*	DBG1("* %lx ***********Steering process relinquishing\n", getpid());*/
    }
    DBG("Steering process exiting\n");
    child_exit(0);
    return 0;  /* to make compiler happy */
}

int
register_sensor_for_steer(sensor_type)
int sensor_type;
{
    if (!monitoring_enabled) return -1;
    if (steer_sensor_switch && sensor_type == -1) {
	int i;
	for (i = 0; i < MAX_NUM_SENSORS; i++) steer_sensor_switch[i] = TRUE;
    } else if (sensor_type < 0 || sensor_type >= MAX_NUM_SENSORS
	       || steer_sensor_switch == NULL) {
	return -1;
    } else {
	steer_sensor_switch[sensor_type] = TRUE;
    }
    return 0;
}

int
unregister_sensor_for_steer(sensor_type)
int sensor_type;
{
    if (!monitoring_enabled) return -1;
    if (steer_sensor_switch && sensor_type == -1) {
	int i;
	for (i = 0; i < MAX_NUM_SENSORS; i++) steer_sensor_switch[i] = FALSE;
    } else if (sensor_type < 0 || sensor_type >= MAX_NUM_SENSORS
	       || steer_sensor_switch == NULL) {
	return -1;
    } else {
	steer_sensor_switch[sensor_type] = FALSE;
    }
    return 0;
}

/* at most 256 user-registered steering handlers, -1 means all */
/* int
register_user_steer_handler(indx, handler)
    int         indx;
    ActionProc  handler;
{
   if (!monitoring_enabled) return -1;
    if (indx == -1) {
	int  i;
	for (i = 0; i < USER_STEER_MAX; i++)
	    user_steer_handler_list[USER_STEER_BASE + i] = handler;
    } else if ((indx >= 0) && (indx < USER_STEER_MAX)) {
	user_steer_handler_list[indx] = handler;
    } else {
	return -1;
    }
    return 0;
}

int
unregister_user_steer_handler(indx)
    int         indx;
{
    if (!monitoring_enabled) return -1;
    if (indx == -1) {
	int  i;
	for (i = 0; i < USER_STEER_MAX; i++)
	    user_steer_handler_list[USER_STEER_BASE + i] = NULL;
    } else if ((indx >= 0) && (indx < USER_STEER_MAX)) {
	user_steer_handler_list[indx] = NULL;
    } else {
	return -1;
    }
    return 0;
} */

/* register format for steer */
int
register_steer_command_handler(fmt_name, field_list, data_handler)
char *fmt_name;
IOFieldList field_list;
CommandHandlerProc data_handler;
{
    int  i;
 /* enable it even if steering not active */
    if (!command_handlers_list) return -1; 
    for (i = 0; (i < USER_STEER_MAX) &&
	   (command_handlers_list[i].fmt_name != NULL); i++);
    if (i < USER_STEER_MAX) {
	int  str_len = strlen(fmt_name);
	memory_alloc((memory_t*) &command_handlers_list[i].fmt_name,
		     str_len+1, N_ANYWHERE);
	strcpy(command_handlers_list[i].fmt_name, fmt_name);
	command_handlers_list[i].field_list = field_list;
	command_handlers_list[i].data_handler = data_handler;
	return i;
    }
    return -1;
}

int
unregister_steer_command_handler(fmt_name)
char *fmt_name;
{
    int  fmt_index;
 /* enable it even if steering not active */
    if (!command_handlers_list) return -1; 

    fmt_index = get_command_handler_index(fmt_name);
    if (fmt_index >= 0) {
	memory_free((memory_t) command_handlers_list[fmt_index].fmt_name);
	command_handlers_list[fmt_index].fmt_name = NULL;
	command_handlers_list[fmt_index].field_list = NULL;
    }
    return fmt_index;
}




