/* sched_utils.c
 */
#include "config.h"
#ifdef HAVE_WINDOWS_H
#include <windows.h>
#endif
#include <memory.h>
#include <sys/types.h>
#ifdef sparc
#include <sys/mman.h>
#endif
#include <stdio.h>

#include "results.h"
#include "general.h"
#include "sched_utils.h"
#include "internal.h"
#include "init.h"
#ifdef WITH_MONITORING
#include "monitor.h"
#endif
#include "mon.sensorextern.h"
#include "hwlib.h"
#include "arch_init.h"
#include "sched_io.h"

#ifdef sparc
extern void perror();
extern int mprotect();
#endif

extern int getpagesize();

static void
  init_tcb ARGS((struct cthread * tcb,
		 long stack,
		 int node));

extern struct configuration config;
extern private struct sched		*local_scheduler[MAX_PROC];
extern shared struct local_info	*processor;
extern cthread_t current_thread_array[MAX_PROC];/* The currently executing thread*/
extern shared int               *num_of_procs;

extern void fill_sighandler_struct ARGS((int i, void(*hndlrs[])()));

shared struct sched_configuration	sched[MAX_PROC];
shared short 		scheduler_installed[MAX_PROC];


extern char *get_stack_memory ARGS((int size));
extern void child_exit ARGS((int status));

/*
 * INITIALIZATION FUNCTIONS:
 */

void (**cthread_in_schedule_function)();

/********************************************************/
/* Initializes a thread control block			*/
/********************************************************/

static void
init_tcb(tcb, stack, node)
cthread_t	tcb;
long 		stack;
int		node;
{
        LOCK_INIT(tcb->tlock);
        tcb->name = 0;
	tcb->return_value = 0;
	tcb->status = S_INITIALLY;
	tcb->thread_data = 0;
	tcb->on_node = node;
	tcb->home    = node;
	tcb->synced = 0;
	tcb->already_started = 0;
	internal_condition_init(&tcb->done);
	tcb->fork_count = -1;

	tcb->jbuf = (cthread_context)Get_Shared_Memory(CONTEXT_SIZE, node);

	/* 
	**  the following machine-specific code adjusts the initial SP 
	**  value to allow room for the first activation record.  The 
	**  nature of the adjustment is machine-specific and the numbers
	**  have possibly been derived experimentally (and so may need
	**  to be changed if something on the machine or OS changes).
	*/
#if defined(__ksr__)
	tcb->stack_top = (void *)((((long)stack)/128)*128);
#elif defined(__sgi)
	tcb->stack_top = (any_t)((((long)(stack))/8)*8 - 88);
#elif defined(_AIX)
	tcb->stack_top = (any_t)((((long)(stack))/8)*8 - 88);
#else
	tcb->stack_top = (void *)((stack/8)*8 - 104);
#endif
	DBG3("Thread %lx jbuf is %lx stack top is %lx\n", (long) tcb, 
	     (long) tcb->jbuf, (long)tcb->stack_top);
	*(long *)(tcb->stack_top) = 0;   /* Cause a page-in */
#ifdef WITH_MONITORING
	if (monitoring_enabled) {
	    tcb->mon_info = monitor_init_thread((void*)tcb, NULL);
	} else {
	    tcb->mon_info = NULL;
	}
#endif

  /* init the pending_io_block */
    memset( &tcb->io, 0, sizeof(struct pending_io_block));
    tcb->io.type = NO_PENDING_IO;

}

/********************************************************/
/* Creates and Initializes a no. of ready queues in     */
/* a node                                               */
/********************************************************/

int
allocate_and_init_readyqs(scb, node, no_of_qs)
sched_info_t scb;
int node;
int no_of_qs;
{
    int size,i;
    locked_queue_t	queues_array;

    scb->no_of_readyqs = no_of_qs;

    /* 
    ** kludge here on the removal of config.readyqs_per_proc
    ** bodhi, you'll need to clean this up, perhaps to dynamically 
    ** allocate more readyqs if needed.
    */	   
    if (no_of_qs > 1) {
	printf("ERROR: The library is not configured for");
	printf(" %d number of ready queues\n", no_of_qs);
	printf("Use cthread_configure to change the configuration\n");
	return(T_SCHED_TOO_MANY_READYQS);
    }
    size = (no_of_qs+1)*sizeof(locked_queue_t);
    scb->lqueue = (locked_queue_t *)Get_Shared_Memory(size, node);

    size = (no_of_qs+1)*sizeof(struct locked_queue);
    queues_array = (locked_queue_t) Get_Shared_Memory(size, node);

    for (i = 0; i<no_of_qs; i++) {
	scb->lqueue[i] = &queues_array[i];
	locked_queue_init(scb->lqueue[i]);
    }
    return(T_SUCCEED);
}

/********************************************************/
/* Creates and Initializes a free queue in a node	*/
/********************************************************/
int
allocate_and_init_freelist(scb, node, no_of_threads)
sched_info_t scb;
int node, no_of_threads;
{
	int size,i,priv_size;
        cthread_t tcb_array;
	char *stackbase;
        cth_queue_t qtmp;

        if (no_of_threads > config.threads_per_proc) {
                printf("ERROR: The library is not configured for");
                printf(" %d number of threads per processor\n", no_of_threads);
                printf("Use cthread_configure to change the configuration\n");
                return(T_SCHED_TOO_MANY_THREADS);

        }
	size =  no_of_threads * sizeof(struct cthread);
	tcb_array = (cthread_t)Get_Shared_Memory(size, node);
	if ((config.stack_size % getpagesize()) != 0) {
	    config.stack_size = (config.stack_size / getpagesize() + 1) 
				 * getpagesize();
	}
	DBG2("Usable stack size is %d (0x%x) bytes\n", config.stack_size,
	     config.stack_size);

	/* 
	** the extra getpagesize() below is for memory to be protected
	** using mprotect so that we can detect stack overflow
	*/
	priv_size = no_of_threads * (config.stack_size + getpagesize());

	/* round up to page size boundary, ARCH specific get routine */
	stackbase = (char *) get_stack_memory(priv_size + getpagesize());

	DBG2(" getting 0x%x bytes of memory at %lx\n",
	     priv_size + getpagesize(), (long)stackbase);
	if ((((long)stackbase) % getpagesize()) != 0) {
	    stackbase = (char *)((((long)stackbase) / getpagesize() + 1) * 
				 getpagesize());
	}
        scb->free_list = 
	     (locked_queue_t)Get_Shared_Memory(sizeof(struct locked_queue), 
					       node);
        locked_queue_init(scb->free_list);
        qtmp = (cth_queue_t)&scb->free_list->lcqueue;

        for (i=0; i< no_of_threads; i++){
	    enqueue(qtmp, (queue_item_t)&tcb_array[i]);

#ifdef sparc
	    DBG2("Protecting stack end.  Region %lx to %lx\n",
		 (long)stackbase, (long) ((char *)stackbase + getpagesize()));
	    if (mprotect((caddr_t)stackbase, getpagesize(), 
			 PROT_NONE) != 0){
		DBG3("stack %lx, config.stack_size %x, page %d\n", 
		      (long) stackbase, config.stack_size, getpagesize());
		perror("mprotect");
	    }
#endif
	    tcb_array[i].usable_stack_limit = (stackbase + getpagesize());
	    tcb_array[i].usable_stack_base = stackbase + getpagesize() +
	      config.stack_size - 1;

	    DBG3(" task %lx, stack_limit = %lx, stack_base = %lx\n",
		 (long)&tcb_array[i], (long)tcb_array[i].usable_stack_limit,
		 (long)tcb_array[i].usable_stack_base);
	    init_tcb(&tcb_array[i], (long)tcb_array[i].usable_stack_base, 
		     node);
	    stackbase = (void *)((char *)stackbase + getpagesize() +
				 config.stack_size);
        }
	return(T_SUCCEED);
}

/********************************************************/
/* Initializes a scheduler  control block		*/
/********************************************************/

int scb_size()
{
    return sizeof(struct sched) + io_info_size();
}

sched_info_t
allocate_and_init_scb(node, no_of_qs, no_of_threads )
int node, no_of_qs, no_of_threads;
{
        int i, size;
        sched_info_t scb;

       	size = scb_size();
	scb =  (sched_info_t)Get_Shared_Memory(size,node);
	
	scb->sched_global_init  = 0;
	scb->sched_vproc_init   = 0;
	scb->sched_thread_init  = 0;
	scb->sched_get_thread   = 0;
	scb->sched_put_thread   = 0;
	scb->sched_empty_readyq = 0;
	scb->schedule           = 0;
	scb->proc_idle          = 0;
        scb->sched_thread       = 0;
        scb->active_threads     = 0;
        scb->no_of_readyqs      = no_of_qs;
        scb->node               = node;
        LOCK_INIT(scb->sched_lock);

/*
 * This is essential since the cthread_init function checks for
 * null to install signal handlers 
 */
        for (i=0; i < NO_OF_SIGNALS; i++) {
		scb->alert_hndlrs[i] = 0;
        }
	allocate_and_init_readyqs(scb, node, no_of_qs);
        allocate_and_init_freelist(scb, node, no_of_threads);

        init_sched_io(scb);

        return(scb);
}

/********************************************************/
/* Initializes a scheduler  configuration  block	*/
/********************************************************/

void
init_sched_config_block(scb)
sched_configuration_t scb;
{
        int i;

        scb->sched_global_init  = 0;
        scb->sched_vproc_init   = 0;
        scb->sched_thread_init  = 0;
        scb->sched_get_thread   = 0;
        scb->sched_put_thread   = 0;
        scb->sched_empty_readyq = 0;
        scb->schedule           = 0;
        scb->proc_idle          = 0;
        scb->no_of_readyqs      = 0;
        scb->no_of_threads      = 0;
	for (i=0; i < NO_OF_SIGNALS; i++) {
                scb->alert_hndlrs[i] = 0;
        }
}

/********************************************************/
/* The following function is used as the default        */
/* function when the optional function scb->schedule is */
/* NULL              					*/
/********************************************************/

void 
default_schedule()
{
	cthread_t    next_thread;
	int current_proc = virtual_processor();

        check_pending_io();
        do {
  	    while(((local_scheduler[current_proc]->sched_empty_readyq)()) && 
                   (processor[current_proc].terminate != TERMINATE)) {
			if (local_scheduler[current_proc]->proc_idle != 0)
			local_scheduler[current_proc]->proc_idle();
		      }
	   next_thread = 
             (cthread_t)(local_scheduler[current_proc]->sched_get_thread)
	       (current_proc);
           if (next_thread == 0) {
              /* for sure processor[current_proc].terminate == TERMINATE */
                if(  local_scheduler[current_proc]->pending_io_count == 0) {
                  	child_exit(0);
		      }
	      }
        } while (next_thread == 0);
        current_thread_array[current_proc] = next_thread;
	sched_dispatch(next_thread);

}


static private cthread_t last_thread[MAX_PROC];
static int processor_idle[MAX_PROC];

void
sched_dispatch(next_thread)
cthread_t next_thread;
{
    int proc = virtual_processor();
    cthread_t current_thread = current_thread_array[proc];

    if  ((current_thread != NULL) && 
	 ((current_thread->status & S_ON_BARRIER) != S_ON_BARRIER) &&
	 ((current_thread->status & S_FINISHED) != S_FINISHED)) {
	last_thread[proc] = current_thread;
    }
    current_thread_array[proc] = current_thread = next_thread;

    if (processor_idle[proc]) {
	processor_idle[proc] = FALSE;
	DM_Sensor_processor_end_idle(proc);
    }

    if ((last_thread[proc] != next_thread) && 
	((current_thread->status & S_ON_BARRIER) != S_ON_BARRIER) &&
	((current_thread->status & S_FINISHED) != S_FINISHED)) {
       DM_Sensor_thread_off_runq(last_thread[proc]); 
   }

    DISPATCH(next_thread);
}


extern void relinquish_processor();

void 
sched_relinquish_processor()
{
    int proc = virtual_processor();
    if (!processor_idle[proc]) {
	cthread_t current_thread = current_thread_array[virtual_processor()];
	cthread_t save_thread = current_thread;

        processor_idle[proc] = TRUE;

	/* muck with current_thread for sensor restore at end */
	if (current_thread == NULL) {
	    if (last_thread[proc] != NULL) {
		current_thread_array[virtual_processor()] = last_thread[proc];
	    } else {
		current_thread_array[virtual_processor()] = *initial_thread;
	    }
	}
	DM_Sensor_processor_begin_idle(proc);
	current_thread_array[virtual_processor()] = save_thread;
    }
    relinquish_processor();
}

void
  install_scheduler(from_node, to_node, this_scheduler)
int from_node, to_node;
sched_configuration_t this_scheduler;
{
    int i;
    for (i = from_node; i<= to_node; i++) {
	sched[i].sched_global_init = this_scheduler->sched_global_init;
	sched[i].sched_vproc_init = this_scheduler->sched_vproc_init;
	sched[i].sched_thread_init = this_scheduler->sched_thread_init;
	sched[i].sched_get_thread = this_scheduler->sched_get_thread;
	sched[i].sched_put_thread = this_scheduler->sched_put_thread;
	sched[i].sched_empty_readyq = this_scheduler->sched_empty_readyq;
	sched[i].schedule = this_scheduler->schedule;

        if (this_scheduler->no_of_threads == 0) {
	    sched[i].no_of_threads = config.threads_per_proc;
        } else {
	    sched[i].no_of_threads = this_scheduler->no_of_threads;
	}
        if (this_scheduler->no_of_readyqs == 0) {
	    sched[i].no_of_readyqs = 1;
        } else {
	    sched[i].no_of_readyqs = this_scheduler->no_of_readyqs;
	}
	fill_sighandler_struct(i, this_scheduler->alert_hndlrs);
	scheduler_installed[i] = TRUE;
	processor_idle[i] = FALSE;
	last_thread[i] = NULL;
    }
}

extern void 
cthread_set_in_schedule_function(func)
void (*func)();
{
    extern short threads_started;
    if (threads_started) {
	int proc = virtual_processor();
	cthread_in_schedule_function[proc] = func;
    } else {
	int i;
	if (cthread_in_schedule_function == NULL) {
	    cthread_in_schedule_function = (void (**)())
	      allocate_and_share(sizeof(cthread_in_schedule_function[0]) * 
				 MAX_PROC, 0);
	}
	for (i = 0; i<MAX_PROC; i++) {
	    cthread_in_schedule_function[i] = func;
	}
    }
}

/*
 * QUEUE FUNCTIONS:
 */

/********************************************************/
/* FIFO queue handling functions 			*/
/********************************************************/




