#include        "config.h"
#include	<stdio.h>
#include	<signal.h>
#include	<string.h>
#include	<malloc.h>
#include	<errno.h>
#include        <memory.h>

#ifndef SEGV_PROT
/* for System V systems like Solaris */
#define SEGV_PROT SEGV_ACCERR
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <fcntl.h>
#include <sys/time.h>

#include        <sys/mman.h>
#include        <unistd.h>
#include        <sys/wait.h>
#include	<sys/m_wait.h>
#include 	<sys/ipc.h>
#include 	<sys/shm.h>

#include	"results.h"
#include	"general.h"
#include	"init.h"
#include	"lock.h"
#include	"queue.h"
#include	"sync.h"
#include	"internal.h"
#include	"memalloc.h"
#include	"cth_sched.h"
#include	"hwlib.h"
#include	"sched_utils.h"
#include	"idebug.h"
#include	"threads.h"
#include	"rs6000_init.h"
#include        "sched_io.h"
#ifdef WITH_MONITORING
#include "monitor.h"
static void rs6000_monitorinit ARGS((int procs));
#endif

extern shared short *monitor_status;
extern shared mutex_t logicalTimeStamp_lock;
extern shared unsigned long *logicalTimeStamp_counter;
extern shared LOCK *print_sema;

/* these symbols needed from init.c */
private int private_current_proc;
extern struct local_info *processor;
extern short threads_started;
extern int *num_of_procs;
extern atom_int *total_threads;
extern atom_int *total_active;
typedef long *process_table_t;
extern process_table_t processes;	/* An array of the processes    */
extern shared long dynamic_memory_per_processor;
extern struct configuration config;	/* configuration        */
#define		NO_SIZE		-1
#define		NO_NODE		-1
extern shared mem_free_list_t *memory_pool;

LOCK *ss_lock;
extern cthread_t current_thread_array[MAX_PROC];
extern shared struct sched *local_scheduler[MAX_PROC];

atom_int *sync_init;

/* externals not defined by sys/shm.h */
#ifndef _AIX
extern int shmdt ARGS((void *shmaddr));
extern void *shmat ARGS((int shmid, void *shmaddr, int shmflg));
extern int shmctl ARGS((int shmid, int cmd,...));
extern int shmget ARGS((key_t key, int size, int shmflg));
extern int wait3 ARGS((int *statusp, int options, struct rusage * rusage));
extern int atexit(void (*func) (void));
#endif
extern int setpgrp ARGS(());
extern int sigblock ARGS((int mask));
extern int sigsetmask ARGS((int mask));
extern int sigpause ARGS((int sigmask));

/* externals not defined by sys/ipc.h */
extern key_t ftok ARGS((const char *path, int id));

/* other random unix externals */
extern void perror ARGS((const char *s));
extern int fflush ARGS((FILE *));
extern int select();

/* defined is $OS_signal.c */
extern void setup_child_sighandlers();
extern void setup_scheduler_sighandlers();
extern void init_sighandler_struct();
extern void setup_master_sighandlers();
extern void setup_term_sighandler();
extern void setup_chld_sighandler();

extern int getpagesize();
extern int page_aligned_size ARGS((int));
extern int generic_shared_size ARGS((int));

typedef union wait wait_stat_type;
#define TERM_SIG(stat) WTERMSIG(stat)
#define RET_CODE(stat) WEXITSTATUS(stat)

extern RESULT init_convert ARGS((int procs));
extern int seg_hndlr();

static void abnormal ARGS((wait_stat_type stat, int wid));
void child_exit_handler ARGS((int wid, wait_stat_type stat));
void child_exit ARGS((int));
void clean_shut_down ARGS(());

extern void monitorinit ARGS((int procs));

void relinquish_processor();

static int use_shared_memory;
shared int real_num_of_procs;

u_char *shm_base, *monmem_base;
long shm_offset, monmem_offset, shm_limit;
key_t shm_key, monmem_key;
int shm_fn, shm_id, monmem_fn, monmem_id;

static volatile int total_procs_exit = 0;
static int total_procs_forked = 0;
static int parent_waiting_the_end = 0;


#ifdef HAVE_SIGINFO_H
int general_hndlr ARGS((int signo, siginfo_t siginfo, void *context));
int chld_hndlr ARGS((int signo, siginfo_t siginfo, void *context));
#else
int chld_hndlr ARGS((int signo, int code, void *scp, char *addr));
int general_hndlr ARGS((int signo, int code, void *scp, char *addr));
#endif

void 
master_on_exit()
{
    if (virtual_processor() == 0 &&
	total_procs_forked > 0 &&
	total_procs_exit > 0) {
	child_exit(0);
    }
}

int
current_processor()
{
    return private_current_proc;
}


#ifdef HAVE_SIGINFO_H
int
ill_hndlr(signo, siginfo, context)
int signo;
siginfo_t siginfo;
void *context;
#else
int
ill_hndlr(signo, code, scp, addr)
int signo, code;
void *scp;
char *addr;
#endif
{
    fflush(stdout);
    fflush(stderr);
    fprintf(stderr, "Illegal instruction signal.  Possible stack overflow on processor %d.\n", virtual_processor());
    fprintf(stderr, "Thread ");
    fprint_thread(stderr, current_thread_array[virtual_processor()]);
    fprintf(stderr, " was active.\nTerminating program.\n");

#ifdef HAVE_SIGINFO_H
    general_hndlr(signo, siginfo, context);
#else
    general_hndlr(signo, code, scp, addr);
#endif
    return 0;
}

#ifdef HAVE_SIGINFO_H
int
seg_hndlr(signo, siginfo, context)
int signo;
siginfo_t siginfo;
void *context;
#else
int
seg_hndlr(signo, code, scp, addr)
int signo, code;
void *scp;
char *addr;
#endif
{
#ifdef TEST2
    cthread_t current_thread = current_thread_array[virtual_processor()];

    fflush(stdout);
    fflush(stderr);
    if (signo == SIGSEGV) {
	fprintf(stderr, "Segmentation fault signal (11) on processor %d.\n",
		virtual_processor());
    } else if (signo == SIGBUS) {
	fprintf(stderr, "Bus error signal (10) on processor %d.\n",
		virtual_processor());
    } else {
	fprintf(stderr, "Signal %d on processor %d.\n", signo,
		virtual_processor());
    }

#ifdef HAVE_SIGINFO_H
    if (siginfo.si_code == SEGV_ACCERR) {
	fprintf(stderr, "Signal code suggests protection violation and a possible stack overflow.\n");
    }
    fprintf(stderr, "signo %d, code %d, stacktop %lx\n",
	    signo, siginfo.si_code, (long) current_thread->stack_top);

#else				/* ndef  HAVE_SIGINFO_H */

    fprintf(stderr, "signo %d, code %d, addr %lx, stacktop %lx\n",
	    signo, code, (long) addr, (long) current_thread->stack_top);
#endif				/* HAVE_SIGINFO_H */

    fprintf(stderr, "Thread ");
    fprint_thread(stderr, current_thread);
    fprintf(stderr, " was active.\nTerminating program.\n");
#endif

#ifdef HAVE_SIGINFO_H
    general_hndlr(signo, siginfo, context);
#else
    general_hndlr(signo, code, scp, addr);
#endif

    return 0;
}
#ifdef HAVE_SIGINFO_H
int
general_hndlr(signo, siginfo, context)
int signo;
siginfo_t siginfo;
void *context;
#else
int
general_hndlr(signo, code, scp, addr)
int signo, code;
void *scp;
char *addr;
#endif

{
    int oldmask;
    oldmask = sigblock(sigmask(SIGCHLD));

    if (virtual_processor() != 0) {
	exit(signo);
    } else {
	/* the parent shutdown everything */
	child_exit_handler(processes[0], signo);

	sigsetmask(oldmask);
    }
    return 0;
}

#ifdef HAVE_SIGINFO_H
int
chld_hndlr(signo, siginfo, context)
int signo;
siginfo_t siginfo;
void *context;
#else
int
chld_hndlr(signo, code, scp, addr)
int signo, code;
void *scp;
char *addr;
#endif
{
    wait_stat_type stat;
    int wid;

    /* the parent shutdown everything */
    while ((wid = wait3((void *) &stat, WNOHANG | WUNTRACED, NULL)) != 0) {
	if (wid != -1)
	    child_exit_handler(wid, stat);
    }
    return 0;
}


void
child_exit_handler(int wid, wait_stat_type stat)
{
    int i, force_parent_exit = 0;

    DBG3("Child %d terminated with termsig %d, retcode %d\n",
	 wid, TERM_SIG(stat), RET_CODE(stat));

    total_procs_exit--;
    if (TERM_SIG(stat) != 0 || RET_CODE(stat) != 0) {
	if (parent_waiting_the_end < 0) {
	    parent_waiting_the_end = 1;

	    /* the parent is already ended and waiting for the children *
	     *  signal all the remaining children to exit */
	    for (i = 1; i < total_procs_forked; i++) {
		if (processes[i] != wid) {
		    kill(processes[i], SIGKILL);
		}
	    }
	} else if (parent_waiting_the_end == 0) {
	    parent_waiting_the_end = 1;
	    force_parent_exit = 1;
	    if (wid != processes[0])
		total_procs_exit--;

	    for (i = 1; i < total_procs_forked; i++) {
		if (processes[i] != wid) {
		    kill(processes[i], SIGKILL);
		}
	    }
	} else {
	    /* the parent is already signaled to finish */
	}
	if (RET_CODE(stat) != 0) {
	    abnormal(stat, wid);
	}
    } else {
	DBG("Normal exit\n");
	if (wid == processes[0]) {
	    force_parent_exit = 1;
	}
    }

    if (total_procs_exit <= 0) {
	clean_shut_down();
	exit(0);
    }
    if (force_parent_exit != 0) {
	while (total_procs_exit > 0) {
	    sigpause(0);	/* wait for the children to shut down */
	}
	clean_shut_down();
	exit(0);
    }
}


void
child_exit(status)
int status;
{
    if (virtual_processor() == 0) {
	int i;
	wait_stat_type stat;
	stat.w_status = 0;
	if (parent_waiting_the_end == 0) {
	    /* the loop is mainly meant to shut down the steering and  *
	     * local monitor processes */

	    for (i = 1; i < total_procs_forked; i++) {
		processor[i].terminate = TERMINATE;
	    }
	    parent_waiting_the_end = -1;
	}
	child_exit_handler(processes[0], stat);
    }
    exit(status);
}
void
clean_shut_down()
{
    close_global_io();
    shmdt(shm_base);
    shmctl(shm_id, IPC_RMID, 0);
    close(shm_fn);
    unlink(".shm_mem");

    shmdt(monmem_base);
    shmctl(monmem_id, IPC_RMID, 0);
    close(monmem_fn);
    unlink(".mon_mem");
}


void
child_init(node, start_func, arg)
int node;
int (*start_func) ();
any_t arg;
{
    setup_child_sighandlers(ill_hndlr, seg_hndlr, seg_hndlr);
    setup_term_sighandler(general_hndlr);

    if (use_shared_memory) {
	shmdt(shm_base);
	shm_base = (u_char *) shmat(shm_id, 0, 0);
	if (shm_base == (u_char *) - 1) {
	    perror("Child Cannot attach shared memory");
	    child_exit(1);
	}
#ifdef WITH_MONITORING
	if (monitoring_enabled) {
	    monmem_base = (u_char *) shmat(monmem_id, 0, 0);
	    if (monmem_base == (u_char *) - 1) {
		perror("Cannot create shared memory");
		child_exit(1);
	    }
	}
#endif
    }
    current_thread_array[node] = 0;
    local_scheduler[node] = processor[node].scheduler;

    DBG3("In child %d, processor = %x, processes = %x\n",
	 node, (int) processor, (int) processes);
    DBG4("In child %d no_of_procs = %x *no_of_procs = %d mem_pool = %x\n",
	 node, (int) num_of_procs, *num_of_procs,
	 (int) memory_pool);

    setup_scheduler_sighandlers();

    if (local_scheduler[node]->sched_vproc_init != 0) {
	(local_scheduler[node]->sched_vproc_init) (node);
    }
    threads_started = 1;


/*******************************************************************/
/* sync_init has to be equal to node + 1 to make sure */
/* that all the child proceses wait till the child 0 is done       */
/* forking the init thread. This is necessary so that the monitor  */
/* thread doesn't get the chance to lock the list->lock before the */
/* init thread executes add_thread_to_lm_list                      */
/*******************************************************************/
    while (*sync_init != node + 1) {
	struct timeval delay_time;
	int delay_unit = (*num_of_procs - node);
	delay_time.tv_sec = 0;
	delay_time.tv_usec = delay_unit * 10000 + 1;	/* return
							 * millisecs */
	DBG3("Sync init = %d  current proc = %d delaying for %d\n", *sync_init,
	     node, delay_unit);
	select(32, NULL, NULL, NULL, &delay_time);
    }

    start_proc_initial_thread(start_func);

    atomadd(sync_init, -1);
    while (*sync_init != 0) {
	struct timeval delay_time;
	delay_time.tv_sec = 0;
	delay_time.tv_usec = (*sync_init) * 10000;	/* return
							 * millisecs */
	DBG2("Sync post init = %d  current proc = %d\n",
	     *sync_init, node);
	select(32, NULL, NULL, NULL, &delay_time);
    }

    DBG("Children initialized. Starting our queue\n");
    if ((start_func == NULL) && (node == 0)) {
	current_thread_array[node] = *initial_thread;
	(*initial_thread)->already_started = 1;
	swap_context_with_func((void (*)()) local_scheduler[node]->schedule,
			       NULL);
    } else {
	local_scheduler[node]->schedule();
    }
}				/* child init */

void
create_shared_arena(procs)
int procs;
{
    int size = generic_shared_size(procs);
    if (procs > 1) {
	use_shared_memory = 1;
	if ((shm_fn = open(".shm_mem", O_RDWR | O_CREAT | O_TRUNC, 0777)) == -1)
	    perror(".shm_mem");
	shm_key = ftok(".shm_mem", '1');

	shm_id = shmget(shm_key, size, IPC_CREAT | 0600);
	if (shm_id == -1) {
	    perror("SHared Memory GET failed!");
	    fprintf(stderr, "Possible cause:  Requested thread/processor/memory \
configuration requires\n\tmore shared memory than is \
available.  Reduce requirements.\n");
	    shmctl(shm_id, IPC_RMID, 0);
	    close(shm_fn);
	    unlink(".shm_mem");
	    exit(1);
	}
	shm_base = (u_char *) shmat(shm_id, 0, 0);
	if (shm_base == (u_char *) - 1) {
	    perror("Cannot create shared memory");
	    shmctl(shm_id, IPC_RMID, 0);
	    close(shm_fn);
	    unlink(".shm_mem");
	    exit(1);
	}
	shm_offset = 0;
	shm_limit = size;
    } else {
	use_shared_memory = 0;
	shm_base = (u_char *) malloc(size);
	shm_offset = 0;
	shm_limit = size;
    }
}

RESULT
doinit(start_func, procs)
int (*start_func) ();
int procs;
{
    int i, chid = 0, node;
    long *shared_shorts;
    wait_stat_type stat;
    int total_procs = procs;

#ifdef WITH_MONITORING
    if (monitoring_enabled) {
	total_procs += number_of_lms;
	if (steering_enabled)
	    total_procs++;
    }
#endif
    /* 
     * Initialize the signal handler data structure.
     */
    init_sighandler_struct();

    create_shared_arena(total_procs);
#ifdef WITH_MONITORING
    if (monitoring_enabled)
	rs6000_monitorinit(total_procs);
#endif

    shared_shorts = (long *) allocate_and_share(SHARED_SHORT_SPACE, -1);

    num_of_procs = &real_num_of_procs;
    *num_of_procs = procs;
    total_threads = (short *) (shared_shorts++);
    *total_threads = 0;
    total_active = (short *) (shared_shorts++);
    *total_active = 0;
    sync_init = (short *) (shared_shorts++);
    *sync_init = total_procs;
#ifdef WITH_MONITORING
    monitor_status = (short *) (shared_shorts++);
    *monitor_status = (short) MONITOR_NOT_INITIALIZED;
#endif
    ss_lock = (LOCK *) (shared_shorts++);
    LOCK_INIT(*ss_lock);

    print_sema = (LOCK *) (shared_shorts++);
    *print_sema = 0;

    init_shared_data(total_procs);

    DBG1("Before procinit total_procs %d \n", total_procs);
    for (node = 0; node < total_procs; node++) {
	procinit(node);
	DBG1("procinit for node %d done \n", node);
    }

    DBG("Per processor structures inited\n");

    atexit(master_on_exit);

    setup_master_sighandlers(general_hndlr, general_hndlr);
    if (init_global_io() < 0)
	exit(1);

    setup_chld_sighandler(chld_hndlr);


    total_procs_forked = total_procs_exit = total_procs;

    stat.w_status = SIGINT;
    processes[0] = getpid();
    processes[total_procs] = PROC_SENTINEL;
    private_current_proc = 0;
    if (total_procs > 1) {
	for (node = 1; node < total_procs; node++) {
	    if ((chid = fork()) == -1) {
		perror("fork failed");
		fprintf(stderr, "Can't support so many processes.  Reduce -num_procs.\n");
		total_procs_forked = total_procs_exit = node;
		goto kill_children;
	    }
	    if (chid == 0) {
		/* if(setpgrp() == -1) perror("setpgrp"); */
		private_current_proc = node;
		child_init(node, start_func, NULL);
		return T_SUCCEED;
	    } else {
		processes[node] = chid;
	    }
	}
	DBG("Process table:\n");
	for (i = 0; i < total_procs; i++) {
	    DBG2("%d         %ld\n", i, processes[i]);
	}
    }
    child_init(0, start_func, NULL);
    return T_SUCCEED;
  kill_children:		/* GOTO TARGET FROM JUST ABOVE ! */
    child_exit_handler(processes[0], stat);
    return T_SUCCEED;
}

static void
abnormal(stat, wid)
wait_stat_type stat;
int wid;
{
    process_table_t pptr;
    long int proc;

    fflush(stdout);
    fflush(stderr);

    for (proc = 0, pptr = processes; *pptr != PROC_SENTINEL && *pptr != wid;
	 pptr++, proc++);

    if (*pptr == PROC_SENTINEL) {
	printf("Received a termination of an unknown child (%d) termsig %d retcode %d\n",
	       wid, TERM_SIG(stat), RET_CODE(stat));
    } else {
	switch (TERM_SIG(stat)) {
	case SIGINT:
	    printf("Child %d died with INTERRUPT SIGNAL %d on logical proc %ld\n",
		   wid, SIGINT, proc);
	    break;
	case SIGQUIT:
	    printf("Child %d died with QUIT SIGNAL %d on logical proc %ld\n",
		   wid, SIGQUIT, proc);
	    break;
	case SIGBUS:
	    printf("Child %d died with BUS ERROR signal %d on logical proc %ld\n",
		   wid, SIGBUS, proc);
	    break;
	case SIGSEGV:
	    printf("Child %d died with segmentation violation signal %d on logical proc %ld\n",
		   wid, SIGSEGV, proc);
	    break;
	case SIGTERM:
	    printf("Child %d died from kill %d on logical proc %ld\n",
		   wid, SIGTERM, proc);
	    break;
	case SIGILL:
	    printf("Child %d died with illegal instruction signal %d on logical proc %ld\n",
		   wid, SIGILL, proc);
	    break;
	case SIGKILL:
	    if (processor[proc].terminate == 0) {
		printf("Child %d died with SIGKILL signal %d on logical proc %ld\n",
		       wid, SIGKILL, proc);
	    }
	    break;
	default:
	    printf("Child %d died with termination signal %d on logical proc %ld\n",
		   wid, TERM_SIG(stat), proc);
	}
    }
}


any_t
allocate_and_share(size, node)
unsigned size;
int node;
{
    u_char *address;
    address = (u_char *) ((int) shm_base + shm_offset);
    shm_offset = shm_offset + size;
    if (shm_offset >= shm_limit) {
	printf("WARNING: Shared Memory Exhausted offset %ld, limit %ld, size %d\n",
	       shm_offset, shm_limit, size);
    }
    return ((any_t) address);
}

any_t
allocate_and_copy(size)
unsigned size;
{
    char *address;

    if ((address = (char *) malloc(size)) == NULL) {
	perror("Internal allocate_and_copy");
	exit(1);
    }
    return ((any_t) address);
}

char *st_limit = 0;

char *
get_stack_memory(size)
int size;
{
    int page_size = getpagesize();
    char *tmp_ptr;

    /* KSR like treatment to the stack to fool the longjmp code --
     * Stole Greg's code from ksr_init.c -- Bodhi */
    /* the goal of this code is to get memory for the thread stacks from
     * the unused regions of *this* stack. (I.E. the stack of the  
     * initialization thread-of-control, be it kernel thread or process.)
     * One obviously must do this carefully.  The stack grows down from
     * high memory and is extended on demand by the OS.  To ensure that
     * the OS fills in all the stack we don't just subtrace 'size' from
     * the current value of the stack and return it.  Instead, we 'walk' 
     * down the memory to be used writing in a zero on every page to
     * force the OS to 'create' that memory.  (This might not be
     * necessary, but it's a precaution.)  We add an arbitrary 10240 to 
     * size so we'll have that much of a gap between the resulting block 
     * and the top of this thread's stack.  This gives us room to call
     * a few more routines on this thread without running into a thread
     * stack. */
    /* 
     *  UNLIKE the ksr, on the RS6000, this gets called by the master
     * process, not the child threads.  The stack_limit value is later 
     * inherited by each child process when they are forked.  Thus we
     * must ensure that st_limit is the minimum of all the st_limits. */
    char *stack_limit = (char *) &stack_limit;

    size += 10240;

    /* round down to nearest page size multiple */
    stack_limit = (char *) ((((long) stack_limit - size) / page_size) * page_size);

    /* the st_limit variable is defined in rs6000.c.  It needs to be
     * smaller (higher on the stack) than the maximum SP value that
     * will be used in a thread.  Thus we make it the stack_limit minus 
     * a page for good measure. */
    if ((st_limit == 0) || ((stack_limit - page_size) < st_limit)) {
	DBG("setting st_limit\n");
	st_limit = stack_limit - page_size;
	for (tmp_ptr = ((char *) &stack_limit) - 16;
	     tmp_ptr >= stack_limit;
	     tmp_ptr -= page_size) {
	    *tmp_ptr = 0;
	}
    }
    DBG1("Stack limit is %lx\n", (long) st_limit);
    return stack_limit;

}

extern RESULT
init_convert(procs)
int procs;
{
    int i, j;

    /* if the processor is online, enter in the conversion table */
    j = 0;
    for (i = 0; i < procs; i++) {
	processor[j++].logical_to_phys = i;
    }
    return (T_SUCCEED);
}				/* init_convert */

void
check_num_procs(procs)
int procs;
{
    int total_procs = procs;
#ifdef WITH_MONITORING
    if (monitoring_enabled) {
	total_procs += number_of_lms;	/* procs reserved for monitoring */
	if (steering_enabled)
	    total_procs++;
    }
#endif
    if (total_procs > MAX_PROC) {
	fprintf(stderr, "Requested number of processors exceeds Cthreads internal limits\n");
	exit(1);
    }
}

int
check_stack_size(size)
int size;
{
    size += getpagesize();	/* add one page for protection */

    /* round up to the nearest integral multiple of page size */
    if ((size % getpagesize()) != 0) {
	size = ((size / getpagesize()) + 1) * getpagesize();
    }
    return size;
}

void *
allocate_monmem(size, node)
int size;
int node;
{
    long n;
    u_char *address;

    address = (u_char *) monmem_base + monmem_offset;
    if ((n = (unsigned long) address % sizeof(long))) {
	address += sizeof(long) - n;
	monmem_offset += sizeof(long) - n;
    }
    monmem_offset = monmem_offset + size;

    return ((void *) address);
}

#ifdef WITH_MONITORING
static void
rs6000_monitorinit(procs)
int procs;
{
    int size, monbuf_size, bufsteer_size;

    DBG2("monitor init, procs is %d, %d\n", procs,
	 config.threads_per_proc);
    monbuf_size = sizeof(MonitorBuffer) + sizeof(struct cth_mutex) +
     size_of_monitor_buffer;
    bufsteer_size = size_of_lm_steer_buffer + sizeof(MonitorBuffer) +
	sizeof(struct cth_mutex);
    size =
	page_aligned_size(sizeof(LocalMonitorInfo) * number_of_lms +
			  MAX_NUM_SENSORS +
			  procs * config.threads_per_proc * monbuf_size +
			  sizeof(unsigned long) +
			  sizeof(struct cth_mutex) +
			  MAX_NUM_SENSORS +
			  number_of_lms * bufsteer_size +
			  MAX_NUM_SAMPLING_INST *
        (2 * sizeof(unsigned long) + 3 * sizeof(void *) + sizeof(char)) +
			  getpagesize());

    if ((monmem_fn = open(".mon_mem", O_RDWR | O_CREAT | O_TRUNC, 0777)) == -1)
	perror(".mon_mem");
    monmem_key = ftok(".mon_mem", '1');

    DBG1("getting shared monitor memory, size is %d\n", size);
    monmem_id = shmget(monmem_key, size, IPC_CREAT | 0600);
    if (monmem_id == -1) {
	perror("SHared Memory GET failed  Monmem!");
	shmctl(monmem_id, IPC_RMID, 0);
	close(monmem_fn);
	unlink(".mon_mem");
	exit(1);
    }
    monmem_base = (u_char *) shmat(monmem_id, 0, 0);
    if (monmem_base == (u_char *) - 1) {
	perror("Cannot create monitor shared memory  monmem");
	shmctl(monmem_id, IPC_RMID, 0);
	close(monmem_fn);
	unlink(".monmem_mem");
	exit(1);
    }
    monmem_offset = 0;

    monitorinit(procs);
}
#endif

void
relinquish_processor()
{
    struct timeval delay_time;
    delay_time.tv_sec = 0;
    delay_time.tv_usec = 10000;	/* return millisecs */
    select(32, NULL, NULL, NULL, &delay_time);
}


unsigned long start_time;

unsigned long
cthread_timestamp()
{
#ifdef WITH_MONITORING
    struct timeval tp0;
    unsigned long timestamp;

    extern int gettimeofday();

    if (logicalTimeStamp) {
	internal_mutex_lock(&logicalTimeStamp_lock->mlock);
	timestamp = (*logicalTimeStamp_counter)++;
	internal_mutex_unlock(&logicalTimeStamp_lock->mlock);
    } else {
	gettimeofday(&tp0, NULL);
	timestamp = (unsigned long) tp0.tv_sec * 1000000 + tp0.tv_usec;
	timestamp -= start_time;
    }
    return timestamp;
#else
    return 0;
#endif
}

void
get_start_timestamp_and_resolution(start_time_p, resolution_p)
unsigned long *start_time_p;
unsigned long *resolution_p;
{
#ifdef WITH_MONITORING
    struct timeval tp0;

    extern int gettimeofday();

    if (logicalTimeStamp) {
	*resolution_p = 1;
	start_time = 0;
    } else {
	gettimeofday(&tp0, NULL);
	*resolution_p = 1000000;
	start_time = (unsigned long) tp0.tv_sec * 1000000 + tp0.tv_usec;
    }
    *start_time_p = start_time;
#endif
}
