#include <stdio.h>
#include <malloc.h>
#include <fcntl.h>
#include <string.h>
#include "assert.h"
#include "general.h"
#include "results.h"
#include "init.h"
#include "lock.h"
#include "queue.h"
#include "sync.h"
#include "internal.h"
#include "idebug.h"

#ifndef SSCANF_DEFINED
extern int sscanf ARGS((const char *, const char *, ...));
#endif
#if defined(sparc)
extern void exit();
extern void perror();
extern char 	*strcpy();
extern size_t	strlen();
#endif

static void convert_file_to_argv();

#ifndef atoi
extern int atoi();
#endif
extern struct configuration config;   /* declared in init.c */
static int number_of_procs = -1;
static int recursive_call = 0;

#ifdef WITH_MONITORING
#include "monitor.h"
/* monitor settings */
static char *machineName = NULL;
static int port_number = 0;
static char *traceFileName = NULL;
static monitor_output_t  monitor_output_format = SOCKET_OUTPUT;
#endif

extern void EnableAdvSteering();

char *applicationName = NULL;
extern void print_version();

int
cthread_parse_args(argc, argv)
int argc;
char *argv[];
{
    int i;
    int return_argc = 0;
    
    if (recursive_call == 0) {
	applicationName = argv[0];
    }
    
    for(i=0; i<argc; i++) {
	
	/* configuration options */
	if (strcmp(argv[i], "-num_procs") == 0) {
	    if (i == argc -1) 
	      fprintf(stderr, "Too few arguments to -num_procs\n");
	    if (sscanf(argv[++i], "%d", &number_of_procs) != 1) {
		(void) fprintf(stderr, "Invalid -num_procs parameter >%s<\n", argv[i]);
	    }
	} else if (strcmp(argv[i], "-cthread_stack_size") == 0) {
	    if (i == argc -1) 
	      fprintf(stderr, "Too few arguments to -cthread_stack_size\n");
	    if (sscanf(argv[++i], "%d", &config.stack_size) != 1) {
		(void) fprintf(stderr, "Invalid -cthread_stack_size parameter >%s<\n", argv[i]);
	    }
	} else if (strcmp(argv[i], "-cthread_mem_exp") == 0) {
	    if (i == argc -1) 
	      fprintf(stderr, "Too few arguments to -cthread_mem_exp\n");
	    if (sscanf(argv[++i], "%d", &config.memory_exponent) != 1) {
		(void) fprintf(stderr, "Invalid -cthread_mem_exp parameter >%s<\n", argv[i]);
	    }
	    
	} else if (strcmp(argv[i], "-cthread_threads_per_proc") == 0) {
	    if (i == argc -1) 
	      fprintf(stderr, "Too few arguments to -threads_per_proc\n");
	    if (sscanf(argv[++i], "%d", &config.threads_per_proc) != 1) {
		(void) fprintf(stderr, "Invalid -cthread_threads_per_proc parameter >%s<\n", argv[i]);
	    }
	    /* cthread_debug */
	} else if (strcmp(argv[i], "-cthread_debug") == 0) {
	    if (i == argc -1) 
	      fprintf(stderr, "Too few arguments to -cthread_debug\n");
	    add_to_debug_string_table(argv[++i]);
	    
	} else if (strcmp(argv[i], "-cthread_allow_core") == 0) {
	    allow_core_dumps++;

#ifdef WITH_MONITORING
	    /*  Monitoring and steering options */
	} else if (strcmp(argv[i], "-cthread_steering") == 0) {
	    steering_enabled++;
	    if (!monitoring_enabled) {
		monitoring_enabled++;
		monitor_output_format = SOCKET_OUTPUT;
	    }

	} else if (strcmp(argv[i], "-cthread_adv_steer") == 0) {
	    extern int as_Enable();
	    as_Enable();

	} else if (strcmp(argv[i], "-cthread_default_monitoring") == 0) {
	    default_monitoring_enabled++;
	    if (!monitoring_enabled) {
		monitoring_enabled++;
		monitor_output_format = SOCKET_OUTPUT;
	    }

	} else if (strcmp(argv[i], "-cthread_distrib_monitoring") == 0) {
	    distrib_monitoring_enabled++;
	    if (!monitoring_enabled) {
		monitoring_enabled++;
		monitor_output_format = SOCKET_OUTPUT;
	    }

	} else if (strcmp(argv[i], "-cthread_dist_node_ID") == 0) {
              if (i > argc -1) {
                fprintf(stderr, "Too few arguments to -cthread_dist_node_ID\n");
                return -1;
            } else {
                dist_node_ID = atoi(argv[++i]);
                if (dist_node_ID < 0) {
                    fprintf(stderr, "Invalid node ID value >%s<\n", argv[i]);
                    return -1;
                }
          }
	} else if (strncmp(argv[i], "-cthread_monitor_lms", 20) == 0) {
	    if (i == argc-1) {
		fprintf(stderr, "Too few arguments to -cthread_monitor_lms\n");
	    } else {
		if (sscanf(argv[++i], "%d", &number_of_lms) != 1)
		fprintf(stderr, "Invalid number of local monitors >%s<\n",
			argv[i]);
	    }

	} else if (strcmp(argv[i], "-cthread_monitor_socket") == 0) {
	    if (i >= argc - 2) {
		fprintf(stderr, "Too few arguments to -cthread_monitor_socket\n");
		monitor_output_format = NO_OUTPUT;
	    } else {
		machineName = strdup(argv[++i]);
		if (sscanf(argv[++i], "%d", &port_number) != 1){
		    (void) fprintf(stderr, "Invalid sensor socket number >%s<\n", argv[i]);
		}
		monitoring_enabled++;
		monitor_output_format = SOCKET_OUTPUT;
	    }
	    
	} else if (strcmp(argv[i], "-cthread_monitor_file") == 0 ||
		   strncmp(argv[i], "-cthread_monitor_file_binary", 23) == 0) {
	    if (i == argc -1) {
		fprintf(stderr, "Too few arguments to -cthread_monitor_file or -cthread_monitor_binary\n");
		monitor_output_format = NO_OUTPUT;
	    } else {
		traceFileName = strdup(argv[++i]);
		monitoring_enabled++;
		monitor_output_format = BINARY_FILE_OUTPUT;
	    }
	    
	} else if (strncmp(argv[i], "-cthread_monitor_buffer_size", 20) == 0) {
	    if (i == argc -1) {
		fprintf(stderr, "Too few arguments to -cthread_monitor_buffer_size\n");

	    } else {
		if (sscanf(argv[++i], "%d", &size_of_monitor_buffer) != 1)
		(void) fprintf(stderr, "Invalid monitor buffer size >%s<\n", argv[i]);
	    }
	    
	} else if (strcmp(argv[i], "-cthread_logical_ts") == 0) {
	    logicalTimeStamp++;
#endif
	    
	    /* more arguments in a file   -arg_file */
	} else if (strcmp(argv[i], "-cthread_version") == 0) {
	    print_version();
	} else if (strcmp(argv[i], "-arg_file") == 0) {
	    int new_argc;
	    char **new_argv;
	    convert_file_to_argv(argv[++i], &new_argc, &new_argv);
	    if (new_argc == 0) {
		fprintf(stderr, 
			"Warning no arguments read from %s\n", argv[i]);
	    } else {
		recursive_call++;
		cthread_parse_args(new_argc, new_argv);
		recursive_call--;
	    }
	    
	} else {
	    if (recursive_call) {
		fprintf(stderr,
			"Warning unknown option >%s< ignored.\n", argv[i]);
	    } else {
		argv[return_argc++] = argv[i];
	    }
	}
    }
    
    return(return_argc);
}

static void
convert_file_to_argv(name, argcp, argvp)
char *name;
int *argcp;
char ***argvp;
{
    int argc=0;
    char **argv = (char **) malloc(sizeof(char *));
    char line[2048];
    FILE *file = fopen(name, "r");
    if (file == NULL) {
	perror("opening argument file");
	*argcp = 0;
	return;
    }
    while(fgets(line, sizeof(line), file)) {
	char arg[1024];
	int begin = 0;
	int used;
	while(sscanf(&line[begin], "%s%n", arg, &used) == 1) {
	    argv = (char **) realloc(argv, (argc +1) * sizeof(char *));
	    argv[argc] = strdup(arg);
	    argc++;
	    begin += used;
	}
    }
    *argcp = argc;
    *argvp = argv;
}

extern shared long dynamic_memory_per_processor;

/* functions declared in machine specific *_init.c files */
extern void check_num_procs ARGS((int procs));
extern int check_stack_size ARGS((int procs));

void
verify_arg_consistency(procs_p)
int *procs_p;
{
    if (number_of_procs != -1) {
	/* procs specified on command line override cthread_init */
	*procs_p = number_of_procs;
    }

    if (*procs_p <= 0) {
	fprintf(stderr, "Number of processors must be greater than zero\n");
    }
    config.stack_size = check_stack_size(config.stack_size);
    dynamic_memory_per_processor = 1<<config.memory_exponent;
#ifdef WITH_MONITORING
    if (monitoring_enabled) {
	if (number_of_lms == 0) {
	    /* by default, use one local monitor for every
	       DEFAULT_NUM_PROCS_PER_LM processors */
	    number_of_lms = (*procs_p - 1) / DEFAULT_NUM_PROCS_PER_LM + 1;
	}
	if (number_of_lms > MAX_NUMBER_OF_LMS) {
	    number_of_lms = MAX_NUMBER_OF_LMS;
	} else if (number_of_lms < 0) {
	    number_of_lms = 1;
	}

	CheckMonitoringSettings(machineName, port_number, traceFileName,
				  &monitor_output_format, applicationName);
	cm_machine_name = machineName;
	cm_port_number = port_number;
	trace_file_name = traceFileName;
    }
#endif

    check_num_procs(*procs_p);  /* machine specific */
}

