#include "assert.h"
#include "config.h"

#include <stdio.h>
#include <sys/types.h>
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#include <string.h>
#include <memory.h>

#include	"results.h"
#include	"general.h"
#include	"lock.h"
#include	"queue.h"
#include	"sync.h"
#include	"internal.h"
#include	"arch_init.h"

#include	"monitor.h"
#include	"io.h"
#include	"DE.h"
#include	"comm_group.h"
#include	"monitor.h"
#include	"monitor_ur.h"
#include	"mon.sensors.h"

#include	"mon_globals.h"

extern int 	Number_Of_User_Specified_Events;

extern void* allocate_monmem ARGS((int size, int node));
static void TurnOnSamplingSensor ARGS((int  ss_num));
static int register_sampling_sensor ARGS((unsigned long samp_interval,
					  SamplingSensorProc ss_proc));

char		variant_event_flag[MAX_NUM_SENSORS];
IOField     	*arrayof_event_flds[MAX_NUM_SENSORS];

exported shared char            *cm_machine_name = NULL;
exported shared int             cm_port_number = -1;
exported shared char		*app_list_default = "*";
exported shared char		*type_list_default = "reorder,exchange,*";
exported shared char            *trace_file_name = NULL;
exported shared int             number_of_lms = 0;
exported shared LocalMonitorInfoPtr lm_info[MAX_NUMBER_OF_LMS];
exported shared mutex_t         lm_info_lock = NULL;
exported shared SteerServerInfoPtr  steer_info;
exported shared int		size_of_monitor_buffer = 2048;
exported shared int		size_of_lm_steer_buffer = 4096;
exported shared int		monitoring_enabled = 0;
exported shared int		default_monitoring_enabled = 0;
exported shared int             distrib_monitoring_enabled = 0;
exported shared int             dist_node_ID= 0;
exported shared int		steering_enabled = 0;

exported shared int		adv_steer_enabled = 0;

exported shared monitor_output_t monitor_output = 0;
exported shared LMStateType	loc_mon_state = LM_NONE;
exported shared IOFormat	short_iofmt, int_iofmt, long_iofmt;
exported shared IOFormat	float_iofmt, double_iofmt, string_iofmt;
exported shared IOFile		steer_infile, steer_outfile;
exported shared IOFormat	steer_simplecom_fmt, steer_floatcom_fmt,
				steer_namedcom_fmt;
exported shared IOFormat	steer_resp_fmt;
exported shared int		logicalTimeStamp = 0;
exported shared mutex_t		logicalTimeStamp_lock = NULL;
exported shared unsigned long	*logicalTimeStamp_counter = NULL;

exported shared short		*monitor_status = NULL;
exported shared char		*steer_sensor_switch = NULL;
exported shared ActionProc	*monitor_action_list = NULL;
exported shared ActionProc	*user_steer_handler_list = NULL;
exported shared CommandHandlersInfo *command_handlers_list = NULL;
exported shared MonitorBufferPtr lm_to_steer_bufs = NULL;

exported shared mutex_t		sampling_sensor_lock = NULL;
exported shared SamplingSensorProc *sampling_sensor_list = NULL;
exported shared int		*sampling_sensor_num = NULL;
exported shared char		*sampling_sensor_switch = NULL;
exported shared unsigned long   *sampling_sensor_last_ts = NULL;
exported shared unsigned long   *sampling_sensor_interval = NULL;
exported shared void		**listof_ss_whole_fields = NULL;
exported shared void		**listof_ss_sampled_fields = NULL;


typedef struct {
    int struct_version;
    int sensor_initialized;
    int sensor_enabled;
    char *sensor_name;
    IOFieldList sensor_field_list;
    int sensor_type_value;
} sensor_descriptor, *sensor_descriptor_ptr;

static int sensor_count = 0;
static sensor_descriptor_ptr sensor_list[MAX_NUM_SENSORS];
static struct cth_mutex sensor_list_lock;

extern shared int *num_of_procs;
extern int sensor_set_switch();

static int cthread_sensor_switch ARGS((sensor_descriptor_ptr sensor));
static int find_sensor ARGS((char *name));

unsigned long  cthread_ts_resolution = 1000000;	/* resolution */

static IOField ur_simple_command_flds[] = {
    {"type", "integer", sizeof(short), IOOffset(URSimpleCommandPtr,type)},
    {"proc", "integer", sizeof(short), IOOffset(URSimpleCommandPtr, proc)},
    {"sensor", "integer", sizeof(short), IOOffset(URSimpleCommandPtr,sensor)},
    {"int_field", "integer", sizeof(int),
     IOOffset(URSimpleCommandPtr, int_field)},
    {"aux_int", "integer", sizeof(int),
     IOOffset(URSimpleCommandPtr, aux_int)},
    {NULL, NULL, 0, 0},
};

static IOField ur_float_command_flds[] = {
    {"type", "integer", sizeof(short), IOOffset(URFloatCommandPtr,type)},
    {"proc", "integer", sizeof(short), IOOffset(URFloatCommandPtr, proc)},
    {"sensor", "integer", sizeof(short), IOOffset(URFloatCommandPtr,sensor)},
    {"float_field", "float", sizeof(double),
     IOOffset(URFloatCommandPtr, float_field)},
    {"aux_float", "float", sizeof(double),
     IOOffset(URFloatCommandPtr, aux_float)},
    {NULL, NULL, 0, 0},
};

static IOField ur_named_command_flds[] = {
    {"type", "integer", sizeof(short), IOOffset(URNamedCommandPtr, type)},
    {"proc", "integer", sizeof(short), IOOffset(URNamedCommandPtr, proc)},
    {"sensor", "integer", sizeof(short), IOOffset(URNamedCommandPtr,sensor)},
    {"file_name", "string", sizeof(char*),
     IOOffset(URNamedCommandPtr, file_name)},
    {"aux_name", "string", sizeof(char*),
     IOOffset(URNamedCommandPtr, aux_name)},
    {NULL, NULL, 0, 0},
};

extern cthread_t cthread_self();

static void
cthread_init_sensor_header(buffer, sensor)
SimpleTraceDataRecordPtr buffer;
sensor_descriptor_ptr sensor;
{
    buffer->timestamp = cthread_timestamp();
    buffer->type = sensor->sensor_type_value;
    buffer->perturbation = 0;
    buffer->distnodeID = dist_node_ID;
    buffer->thread = (unsigned long) cthread_self();
}

static void
cthread_update_sensor_header(buffer, sensor_num)
SimpleTraceDataRecordPtr buffer;
int sensor_num;
{
    unsigned long time = cthread_timestamp();
    buffer->perturbation += (int)(time - buffer->timestamp);
    buffer->timestamp = time;
}

static char*
cthread_sensor_memory_alloc(size)
int size;
{
    char *val;
    RESULT res;
    res = memory_alloc((memory_t*)&val, size, N_ANYWHERE);
    if (res != T_SUCCEED) {
	return NULL;
    } else {
	return val;
    }
}

static int
cthread_shared_memory_insert_buffer(sensor, size)
char *sensor;
int size;
{
    return InsertBuffer(GetMonitorBufferPtr(cthread_self()),
			sensor, size);
}

void
CheckMonitoringSettings(machineName, port_number, monFileName,
			       monitor_output_format, app_name)
char *machineName;
int port_number;
char *monFileName;
monitor_output_t  *monitor_output_format;
char *app_name;
{
    monitor_output = *monitor_output_format;
    return;
}

void
end_socket_connection(lm, abort_flag)
LocalMonitorInfoPtr lm;
int abort_flag;
{
    if( !abort_flag) {
       DExchange_shutdown_connection(lm->de, lm->dep, SHUTDOWN_CONNECTION, NULL);
     }
    DEport_close(lm->dep);
    DExchange_close(lm->de);
  }

int
start_socket_connection(de_p, dep_p)
DExchange *de_p;
DEPort *dep_p;
{
    DExchange	  de = DExchange_create();
    DEPort dep = NULL;


    DBG("start_socket_connection\n");
    *de_p = de;
    if (cm_machine_name != NULL) {
	if (cm_port_number == -1) {
	    dep = DExchange_initiate_conn(de, cm_machine_name, -1, 
					  TRUE /* block by default */);
	} else {
	    dep = DExchange_initiate_conn(de, cm_machine_name, cm_port_number, 
					  TRUE /* block by default */);
	}
    } else {
	if (group_server_present()) {
	    comm_group_return groups;
	    groups = matching_comm_groups(app_list_default, type_list_default);
	    dep = DExchange_initiate_first(de, groups,
					   TRUE /* block by default */);
	    free(groups);
	} else {
	    printf("Error:  No group server present and monitoring output unspecified\n");
	    return -1;
	}
    }
    if (dep == NULL) {
	return -1;
    }
    *dep_p = dep;

    /* open PBIO file descriptor for receiving information */
    DBG("socket_connection complete\n");
    if (dep == NULL) return -1;
    
    return 0;
}



static int
handle_URsimple_command(de, dep, format_id, ur, ur_length, lm_state)
DExchange de;
DEPort dep;
int format_id;
URSimpleCommand *ur;
int ur_length;
LMStateTypePtr lm_state;
{
    int i;
    switch (ur->type) {
    case UR_APPLICATION_QUIT:
	*lm_state = LM_QUIT;
	break;
    case UR_APPLICATION_PAUSE:
	*lm_state = LM_PAUSE;
	break;
    case UR_APPLICATION_CONTINUE:
	*lm_state = LM_RUN;
	break;
    case UR_APPLICATION_DISCARD:
	*lm_state = LM_DISCARD;
	break;
    case UR_TURNON_SENSOR:
	printf("UR_TURNON_SENSOR outdated.  Use by_name rather than by number\n");
	break;
    case UR_TURNOFF_SENSOR:
	printf("UR_TURNOFF_SENSOR outdated.  Use by_name rather than by number\n");
	break;
    case UR_TURNON_ALL_SENSORS:
	mutex_lock(&sensor_list_lock);
	for (i = 0; i < sensor_count; i++) {
	    sensor_list[i]->sensor_enabled = TRUE;
	}
	mutex_unlock(&sensor_list_lock);
	break;
    case UR_TURNOFF_ALL_SENSORS:
	mutex_lock(&sensor_list_lock);
	for (i = 0; i < sensor_count; i++) {
	    sensor_list[i]->sensor_enabled = FALSE;
	}
	mutex_unlock(&sensor_list_lock);
	break;
    default:
	DBG2("lm: unknown simple command <%d, %d>\n",
	     ur->type, ur->int_field);
    }
    return 0;
}

static int
handle_URfloat_command(de, dep, format_id, ur, ur_length)
DExchange de;
DEPort dep;
int format_id;
URFloatCommand *ur;
int ur_length;
{
    switch (ur->type) {
    default:
	DBG2("lm: unknown float command (%d, %f)\n",
	     ur->type, ur->float_field);
    }
    return 0;
}

static int
handle_URnamed_command(de, dep, format_id, ur, ur_length)
DExchange de;
DEPort dep;
int format_id;
URNamedCommand *ur;
int ur_length;
{
    switch (ur->type) {
    case UR_TURNON_NAMED_SENSOR:
	if (ur->file_name == NULL || strlen(ur->file_name) == 0) {
	    DBG("lm: NULL sensor name in TURN SENSOR ON command\n");
	} else {
	    int sensor_type_value;
	    mutex_lock(&sensor_list_lock);
	    sensor_type_value = find_sensor(ur->file_name);
	    sensor_list[sensor_type_value]->sensor_enabled = TRUE;
	    mutex_unlock(&sensor_list_lock);
	}
	break;
    case UR_TURNOFF_NAMED_SENSOR:
	if (ur->file_name == NULL || strlen(ur->file_name) == 0) {
	    DBG("lm: NULL sensor name in TURN SENSOR OFF command\n");
	} else {
	    int sensor_type_value;
	    mutex_lock(&sensor_list_lock);
	    sensor_type_value = find_sensor(ur->file_name);
	    sensor_list[sensor_type_value]->sensor_enabled = False;
	    mutex_unlock(&sensor_list_lock);
	}
	break;
    default: 
	DBG1("lm: unknown named command of format [%d]\n",
	     ur->type);
    }
    return 0;
}

extern void
init_lm_output_conn (lm)
LocalMonitorInfoPtr  lm;
{

    DBG("LM: opening output connection\n");
    /* open PBIO handle if BINARY_FILE_OUTPUT */
    if (monitor_output == BINARY_FILE_OUTPUT) {
	char buf[256];
	if (number_of_lms > 1) {
	    sprintf(buf, "%s.%d", trace_file_name, lm->lm_num + 1);
	} else {
	    sprintf(buf, "%s", trace_file_name);
	}
	lm->de = DExchange_create();
	lm->dep = DExchange_accept_conn_file(lm->de, buf, "w");

	if (lm->dep == NULL) {
	    fprintf(stderr, "open_IOfile failed for %s\n", buf);
	    monitor_output = NO_OUTPUT;
	    return;
	}
    }

    DBG("LM: registering formats\n");
    if( monitor_output == SOCKET_OUTPUT ||
	monitor_output == BINARY_FILE_OUTPUT) {

	DExchange_register_format(lm->de, "simple user request",
				  ur_simple_command_flds);
	DExchange_register_function(lm->de, "simple user request",
				     (DataHandlingFunc)handle_URsimple_command,
				     &loc_mon_state);
	DExchange_register_format(lm->de, "float user request",
				  ur_float_command_flds);
	DExchange_register_function(lm->de, "float user request",
				     (DataHandlingFunc)handle_URfloat_command,
				     &loc_mon_state);
	DExchange_register_format(lm->de, "named user request",
				  ur_named_command_flds);
	DExchange_register_function(lm->de, "named user request",
				     (DataHandlingFunc)handle_URnamed_command, 
				     &loc_mon_state);
    }
    DBG("LM: initial output complete\n");
}

extern int
handle_simple_steer_command ARGS((DExchange de, DEPort dep, int format_id,
				  URSimpleCommandPtr steer_cmd, int length, void*));

extern void
init_steer_output_conn(steer_info)
SteerServerInfoPtr  steer_info;
{

    DExchange_register_format(steer_info->de, "simple steer request",
			      ur_simple_command_flds);
    DExchange_register_function(steer_info->de, "simple steer request",
				(DataHandlingFunc)handle_simple_steer_command,
				NULL);
    DExchange_register_format(steer_info->de, "float steer request",
			      ur_float_command_flds);
    DExchange_register_function(steer_info->de, "float steer request",
				(DataHandlingFunc)handle_simple_steer_command,
				NULL);
    DExchange_register_format(steer_info->de, "named steer request",
			      ur_named_command_flds);
    DExchange_register_function(steer_info->de, "named steer request",
				(DataHandlingFunc)handle_simple_steer_command, 
				NULL);

}


static IOField SimpleTraceDataFields[] = {
    {"type", "integer", sizeof(int), IOOffset(SimpleTraceDataRecordPtr, type)},
    {"perturbation", "integer", sizeof(int),
       IOOffset(SimpleTraceDataRecordPtr, perturbation)},
    {"distnodeID", "integer", sizeof(int),
       IOOffset(SimpleTraceDataRecordPtr, distnodeID)},
    {"timestamp", "unsigned integer", sizeof(unsigned long),
       IOOffset(SimpleTraceDataRecordPtr, timestamp)},
    {"thread", "unsigned integer", sizeof(unsigned long),
       IOOffset(SimpleTraceDataRecordPtr, thread)},
    {NULL, NULL, 0, 0}
};

static IOField *
add_header_fields(field_list)
IOField *field_list;
{
    IOField *new_list;
    static int header_fields_size = 0;
    static int header_fields_count = 0;
    int field_list_size = 0;
    int j;
    if (header_fields_size == 0) {
	int i;
	for ( i= 0; SimpleTraceDataFields[i].field_name != NULL; i++ ) ;
	header_fields_size = i * sizeof(SimpleTraceDataFields[0]);
	header_fields_count = i;
    }
    for ( field_list_size= 0; field_list[field_list_size].field_name != NULL; 
	 field_list_size++ ) ;
    field_list_size = (field_list_size +1) * sizeof(field_list[0]);
    new_list = (IOField *) malloc (field_list_size + header_fields_size);
    memcpy(new_list, SimpleTraceDataFields, header_fields_size);
    for (j = 0; field_list[j].field_name != NULL; j++) {
	new_list[j+header_fields_count] = field_list[j];
	new_list[j+header_fields_count].field_offset += sensor_header_size;
    }
    new_list[j+header_fields_count] = field_list[j];  /* add null */
    return new_list;
}
    
static void
Initialize_Event_Fields()
{
    int  i;
    
    /*** initialize event information arrays ***/
    for (i = 0; i < MAX_NUM_SENSORS; i++) {
	arrayof_event_flds[i] = NULL;
    }

    if ( Number_Of_User_Specified_Events != 0 ) {
	 printf("\007\n");
	 printf("STOP!  You are using sensor code from an outdated version of the sensor\n");
	 printf("compiler!  Please see: \n");
	 printf("http://www.cc.gatech.edu/systems/projects/Cthreads/sensor_conversion.html\n");
	 printf("for information on conversion.\n");
	exit(1);
    }
}

static void
add_buffer_to_lm_list(mon_buf)
MonitorBufferPtr  mon_buf;
{
    int  i, m = 0;
    MonitorBufferPtr  ptr;

    /* check if the thread is already in one of the list */
    internal_mutex_lock(&lm_info_lock->mlock);
    for (i = 0; i < number_of_lms; i++) {
	if (lm_info[m]->monitor_load > lm_info[i]->monitor_load)
	  m = i;
	ptr = lm_info[i]->buffer_list;
	while (ptr != NULL) {
	    if (ptr == mon_buf) {
		internal_mutex_unlock(&lm_info_lock->mlock);
		return;   /* this thread is in the list already */
	    }
	    ptr = ptr->next;
	}
    }
    /* add thread to local monitor m */
    mon_buf->next = lm_info[m]->buffer_list;
    lm_info[m]->buffer_list = mon_buf;
    lm_info[m]->monitor_load++;
    internal_mutex_unlock(&lm_info_lock->mlock);
}

typedef struct thread_mon_info {
    MonitorBufferPtr mon_buf;
} *thread_mon_info_p;

void *
monitor_init_thread(thread_id, info)
void *thread_id;
void *info;
{
    thread_mon_info_p mon_info = (thread_mon_info_p) info;
    if (mon_info == NULL) {
	mon_info = allocate_monmem(sizeof(struct thread_mon_info), N_ANYWHERE);
	mon_info->mon_buf = InitBuffer(size_of_monitor_buffer, N_ANYWHERE);
	add_buffer_to_lm_list(mon_info->mon_buf);
    }
    DBG3("Monitor_init_thread %lx, mon_info %lx, buffer %lx\n", (long)thread_id,
	 (long)mon_info, (long)mon_info->mon_buf);
    return mon_info;
}

 
MonitorBufferPtr
GetMonitorBufferPtr(thread_id)
cthread_t thread_id;
{
    return ((thread_mon_info_p)thread_id->mon_info)->mon_buf;
}

void
monitor_delete_mutex(void *c, void* info){}

void
monitor_delete_condition(void *c, void* info){}

void 
monitorinit(procs)
int procs;
{
    extern int adv_steer_enabled;
    extern void InitializeAdvSteering();
    int i, j;
    unsigned long  start_ts;

    /* initialize sensor global variables */
    sensor_non_blocking_submit_event = cthread_shared_memory_insert_buffer;
    sensor_turn_on_sampling_sensor = TurnOnSamplingSensor;
    sensor_register_sampling_sensor = register_sampling_sensor;
    sensor_test_switch = cthread_sensor_switch;
    sensor_init_header = cthread_init_sensor_header;
    sensor_update_header = cthread_update_sensor_header;
    sensor_memory_alloc = cthread_sensor_memory_alloc;
    sensor_header_size = sizeof(SimpleTraceDataRecord);
    sensor_header_size = 8*((sensor_header_size +7) / 8 ); /* round up */
    sensor_listof_ss_whole_fields = listof_ss_whole_fields;
    sensor_listof_ss_sampled_fields = listof_ss_sampled_fields;
    internal_mutex_init(&sensor_list_lock);

    for (i = 0; i < number_of_lms; i++) {
	lm_info[i] = (LocalMonitorInfoPtr)
	  allocate_monmem(sizeof(LocalMonitorInfo), N_CURRENT);
	lm_info[i]->buffer_list = NULL;
	lm_info[i]->lm_num = i;
	if (i == 0) {
	    lm_info[i]->monitor_load = 2;
	    /* LM#0, forward commands to me, I'll process them */
	    lm_info[i]->default_forward = TRUE;
	} else {
	    lm_info[i]->monitor_load = 0;
	    /* other LMs, don't forward any commands to me */
	    lm_info[i]->default_forward = FALSE;
	}
	for (j = 0; j < MAX_NUM_SENSORS; j++)
	  lm_info[i]->ioevent_format_ids[j] = -1;
	lm_info[i]->de = NULL;
	lm_info[i]->dep = NULL;
	lm_info[i]->signon_fmt = NULL;
	lm_info[i]->signoff_fmt = NULL;

	lm_info[i]->simplecom_fmt = NULL;
	lm_info[i]->floatcom_fmt = NULL;
	lm_info[i]->namedcom_fmt = NULL;
    }
    lm_info_lock = (mutex_t)
      allocate_monmem(sizeof(struct cth_mutex), N_CURRENT);
    internal_mutex_init(lm_info_lock);
    Initialize_Event_Fields();
    if (steering_enabled) {
	steer_sensor_switch =
	    (char *)allocate_monmem(MAX_NUM_SENSORS, N_CURRENT);
	for (i = 0; i < MAX_NUM_SENSORS; steer_sensor_switch[i++] = TRUE);
#if 0
	printf("lms count=%d, size=%d\n", number_of_lms, 
	       size_of_lm_steer_buffer);fflush(stdout);
#endif
	lm_to_steer_bufs = InitMultiBuffer(number_of_lms,
					   size_of_lm_steer_buffer, N_CURRENT);
	assert(lm_to_steer_bufs != NULL );

	/* allocate and initialize steering server data structure */
	steer_info = (SteerServerInfoPtr)
	    allocate_monmem(sizeof(SteerServerInfo), N_CURRENT);
	steer_info->server_num = -1;
	steer_info->default_forward = TRUE;
	steer_info->simplecom_fmt = NULL;
	steer_info->floatcom_fmt = NULL;
	steer_info->namedcom_fmt = NULL;
    }
    /*  initialize monitor action list and command handler */
    monitor_action_list =
      (ActionProc *) allocate_monmem(sizeof(ActionProc)*MAX_NUM_SENSORS,
				     N_CURRENT);
    for (i = 0; i < MAX_NUM_SENSORS; monitor_action_list[i++] = NULL);
    user_steer_handler_list = (ActionProc *)
	allocate_monmem(sizeof(ActionProc) * USER_STEER_MAX, N_CURRENT);
    for (i = 0; i < USER_STEER_MAX; user_steer_handler_list[i++] = NULL);
    command_handlers_list = (CommandHandlersInfoPtr)
	allocate_monmem(sizeof(CommandHandlersInfo)*USER_STEER_MAX, N_CURRENT);
    for (i = 0; i < USER_STEER_MAX; i++) {
	command_handlers_list[i].fmt_name = NULL;
	command_handlers_list[i].field_list = NULL;
	command_handlers_list[i].registered = 0;
	command_handlers_list[i].data_handler = NULL;
    }

    /* initialize data structure for sampling sensors */
    sampling_sensor_lock = (mutex_t)
	allocate_monmem(sizeof(struct cth_mutex), N_CURRENT);
    internal_mutex_init(sampling_sensor_lock);
    sampling_sensor_list = (SamplingSensorProc*)
	allocate_monmem(sizeof(SamplingSensorProc)*MAX_NUM_SAMPLING_INST, N_CURRENT);
    for (i = 0; i < MAX_NUM_SAMPLING_INST; sampling_sensor_list[i++] = NULL);
    sampling_sensor_num = (int*) allocate_monmem(sizeof(int), N_CURRENT);
    *sampling_sensor_num = 0;
    sampling_sensor_switch = (char*)
	allocate_monmem(MAX_NUM_SAMPLING_INST, N_CURRENT);
    for (i=0; i < MAX_NUM_SAMPLING_INST; sampling_sensor_switch[i++] = False);
    sampling_sensor_last_ts = (unsigned long*)
	allocate_monmem(sizeof(unsigned long)*MAX_NUM_SAMPLING_INST,N_CURRENT);
    for (i = 0; i < MAX_NUM_SAMPLING_INST; sampling_sensor_last_ts[i++] = 0);
    sampling_sensor_interval = (unsigned long*)
	allocate_monmem(sizeof(unsigned long)*MAX_NUM_SAMPLING_INST,N_CURRENT);
    for (i = 0; i < MAX_NUM_SAMPLING_INST; sampling_sensor_interval[i++] = 0);
    listof_ss_whole_fields = (void**)
	allocate_monmem(sizeof(void*)*MAX_NUM_SAMPLING_INST,N_CURRENT);
    listof_ss_sampled_fields = (void**)
	allocate_monmem(sizeof(void*)*MAX_NUM_SAMPLING_INST,N_CURRENT);

    /* turn off cthread_yield unless specifically requested*/
    sensor_set_switch("thread_yield", FALSE);
    /* remainder of default sensors set to default_monitoring_enabled */
    sensor_set_switch("thread_init", default_monitoring_enabled);
    sensor_set_switch("thread_fork", default_monitoring_enabled);
    sensor_set_switch("thread_exit", default_monitoring_enabled);
    sensor_set_switch("thread_begin_join", default_monitoring_enabled);
    sensor_set_switch("thread_end_join", default_monitoring_enabled);
    sensor_set_switch("thread_detach", default_monitoring_enabled);
    sensor_set_switch("thread_set_name", default_monitoring_enabled);
    sensor_set_switch("mutex_begin_lock", default_monitoring_enabled);
    sensor_set_switch("mutex_end_lock", default_monitoring_enabled);
    sensor_set_switch("mutex_unlock", default_monitoring_enabled);
    sensor_set_switch("mutex_alloc", default_monitoring_enabled);
    sensor_set_switch("mutex_init", default_monitoring_enabled);
    sensor_set_switch("mutex_free", default_monitoring_enabled);
    sensor_set_switch("mutex_clear", default_monitoring_enabled);
    sensor_set_switch("mutex_set_name", default_monitoring_enabled);
    sensor_set_switch("condition_alloc", default_monitoring_enabled);
    sensor_set_switch("condition_init", default_monitoring_enabled);
    sensor_set_switch("condition_free", default_monitoring_enabled);
    sensor_set_switch("condition_clear", default_monitoring_enabled);
    sensor_set_switch("condition_begin_wait", default_monitoring_enabled);
    sensor_set_switch("condition_end_wait", default_monitoring_enabled);
    sensor_set_switch("condition_broadcast", default_monitoring_enabled);
    sensor_set_switch("condition_signal", default_monitoring_enabled);
    sensor_set_switch("condition_set_name", default_monitoring_enabled);
    sensor_set_switch("program_exit", default_monitoring_enabled);
    sensor_set_switch("monitor_exit", default_monitoring_enabled);
    sensor_set_switch("thread_off_runq", default_monitoring_enabled);
    sensor_set_switch("processor_begin_idle", default_monitoring_enabled);
    sensor_set_switch("processor_end_idle", default_monitoring_enabled);

    logicalTimeStamp_lock =
      (mutex_t)allocate_monmem(sizeof(struct cth_mutex), N_CURRENT);
    internal_mutex_init(logicalTimeStamp_lock);
    logicalTimeStamp_counter =
      (unsigned long *)allocate_monmem(sizeof(unsigned long), N_CURRENT);
    *logicalTimeStamp_counter = 1;
    get_start_timestamp_and_resolution(&start_ts, &cthread_ts_resolution);

    if( adv_steer_enabled ) {
        extern int as_Initialize();
        as_Initialize();
    }
}

extern int
monitor_get_sensor_type_value(name)
char *name;
{
    int value;
    mutex_lock(&sensor_list_lock);
    value = find_sensor(name);
    mutex_unlock(&sensor_list_lock);
    return value;
}

extern char*
monitor_get_sensor_name(type_value)
int type_value;
{
    return sensor_list[type_value]->sensor_name;
}

static int
find_sensor(name)
char *name;
{
    int i = 0;
    while ( (i < sensor_count) && 
	    (strcmp(sensor_list[i]->sensor_name, name) != 0)) {
	i++;
    }
    if (i == sensor_count) {
	/* we didn't find it, create a temporary and initialize it */
	sensor_list[i] = malloc(sizeof(sensor_descriptor));
	sensor_list[i]->sensor_name = name;
	sensor_list[i]->sensor_initialized = 0;
	sensor_list[i]->sensor_enabled = 1;
	sensor_list[i]->struct_version = 1;
	sensor_list[i]->sensor_type_value = i;
	sensor_count++;
    }
    return i;
}
    

static void
init_sensor(sensor)
sensor_descriptor_ptr sensor;
{
    int sensor_type_value, j;

    mutex_lock(&sensor_list_lock);
    if (sensor->sensor_initialized) {
	/* checking inside the lock, someone got here first, just return */
	mutex_unlock(&sensor_list_lock);
	return;
    }
    sensor_type_value = find_sensor(sensor->sensor_name);
    DBG2("Initializing sensor %s, dynamic type value %d\n", 
	 sensor->sensor_name, sensor_type_value);

    /* copy values from temporary sensor descriptor*/
    sensor->sensor_enabled = sensor_list[sensor_type_value]->sensor_enabled;
    sensor->sensor_type_value = sensor_type_value;

   /* add the sensor to the list for the LMs to register later */
    free(sensor_list[sensor_type_value]);
    sensor_list[sensor_type_value] = sensor;

    variant_event_flag[sensor_type_value] = False;
    arrayof_event_flds[sensor_type_value] =
	add_header_fields(sensor->sensor_field_list);
    for (j=0; (arrayof_event_flds[sensor_type_value][j].field_name != NULL) 
	     && !variant_event_flag[sensor_type_value]; j++){
	if (!strcmp(arrayof_event_flds[sensor_type_value][j].field_type, 
		    "string"))
	    variant_event_flag[sensor_type_value] = True;
    }

    sensor->sensor_initialized = TRUE;
    mutex_unlock(&sensor_list_lock);
}    

void
lm_check_sensor_list(lm)
LocalMonitorInfoPtr lm;
{
    /* subroutine for the LMs to use to register any new sensors */
    int tmp_count, i;

    tmp_count = sensor_count;

    for (i = 0 ; i < tmp_count; i++) {
	sensor_descriptor_ptr sd = sensor_list[i];
	if ((sensor_list[i]->sensor_field_list != NULL) &&
	    (lm->ioevent_format_ids[i] == -1)) {
	    DExchange_register_format(lm->de, sd->sensor_name,
				      arrayof_event_flds[sd->sensor_type_value]);
	    lm->ioevent_format_ids[i] = 
		DEget_format_id(lm->de, sd->sensor_name);
	}
    }
}

static int
cthread_sensor_switch(sensor)
sensor_descriptor_ptr sensor;
{
    if (!monitoring_enabled) return FALSE;
    if (!sensor->sensor_initialized) {
	init_sensor(sensor);
    }
    return sensor->sensor_enabled;
}


int
sensor_set_switch(sensor, swtch)
char *sensor;
int swtch;
{
    int old_swtch = -1, sensor_number;
    
    mutex_lock(&sensor_list_lock);
    sensor_number = find_sensor(sensor);

    old_swtch = sensor_list[sensor_number]->sensor_enabled;
    sensor_list[sensor_number]->sensor_enabled = swtch;
    mutex_unlock(&sensor_list_lock);

    return old_swtch;
}

int
register_monitor_action(sensor_type, action)
int sensor_type;
ActionProc  action;
{
    if (sensor_type < 0 || sensor_type >= MAX_NUM_SENSORS
	|| monitor_action_list == NULL) {
	return -1;
    }
    monitor_action_list[sensor_type] = action;
    return 0;
}

static int
register_sampling_sensor(samp_interval, ss_proc)
    unsigned long	samp_interval;
    SamplingSensorProc  ss_proc;
{
    int  ss_num = -1;
    internal_mutex_lock(&sampling_sensor_lock->mlock);
    if (*sampling_sensor_num < MAX_NUM_SAMPLING_INST) {
	ss_num = *sampling_sensor_num;
	sampling_sensor_list[ss_num] = ss_proc;
	sampling_sensor_switch[ss_num] = False;
	sampling_sensor_interval[ss_num] = samp_interval * 
	    (cthread_ts_resolution / 1000);
	(*sampling_sensor_num)++;
    }
    internal_mutex_unlock(&sampling_sensor_lock->mlock);
    return ss_num;
}

static void
TurnOnSamplingSensor(ss_num)
    int  ss_num;
{
    internal_mutex_lock(&sampling_sensor_lock->mlock);
    sampling_sensor_switch[ss_num] = True;
    sampling_sensor_last_ts[ss_num] = cthread_timestamp();
    internal_mutex_unlock(&sampling_sensor_lock->mlock);
}

extern int
get_command_handler_index(fmt_name)
char *fmt_name;
{
    int  i = 0;
    while ((i < USER_STEER_MAX) &&
	   (command_handlers_list[i].fmt_name == NULL ||
	    strcmp(command_handlers_list[i].fmt_name, fmt_name)))
	i++;
    if (i >= USER_STEER_MAX) i = -1;
    return i;
}
