/* linux_init.c */

#include	<stdio.h>
#include	<signal.h>
#include	<string.h>
#include	<malloc.h>
#include	<errno.h>
#include        <memory.h>
#ifndef SEGV_PROT
/* for System V systems like Solaris */
#define SEGV_PROT SEGV_ACCERR
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/fcntl.h>
#include <sys/time.h>

#include        <sys/mman.h>
#include        <fcntl.h>
#include        <unistd.h>
#include        <sys/wait.h>
#include 	<sys/ipc.h>
#include 	<sys/shm.h>

#include	"results.h"
#include	"general.h"
#include	"init.h"
#include	"lock.h"
#include	"queue.h"
#include	"sync.h"
#include	"internal.h"
#include	"memalloc.h"
#include	"cth_sched.h"
#include	"hwlib.h"
#include	"sched_utils.h"
#include	"idebug.h"
#include	"threads.h"
#include	"linux_init.h"
#include        "sched_io.h"
#ifdef WITH_MONITORING
#include "monitor.h"
static void linux_monitorinit ARGS((int procs));
#endif

extern shared short             *monitor_status;
extern shared mutex_t           logicalTimeStamp_lock;
extern shared unsigned long     *logicalTimeStamp_counter;
extern shared LOCK 		*print_sema;
private int private_current_proc;

/* these symbols needed from init.c */
extern private locked_queue_t local_thread_queue[MAX_PROC];
extern struct local_info *processor;
extern short threads_started;
extern int *num_of_procs;
extern atom_int *total_threads;
extern atom_int *total_active;
typedef long *process_table_t;
extern process_table_t	processes;	/* An array of the processes 	*/
extern shared long		dynamic_memory_per_processor;
extern struct configuration	config;	/* configuration	*/
#define		NO_SIZE		-1
#define		NO_NODE		-1
extern shared mem_free_list_t *memory_pool;
extern int page_aligned_size ARGS((int));

LOCK    *ss_lock;
extern cthread_t	current_thread_array[MAX_PROC];
extern shared struct sched		*local_scheduler[MAX_PROC];

atom_int *sync_init;

/* externals not defined by sys/shm.h */
extern int shmget ARGS((key_t key, size_t size, int shmflg));

/* other random unix externals */
extern void perror ARGS((const char *s));
#ifndef memset
extern void 	*memset();
#endif
extern int fflush ARGS((FILE *));
extern int select();

/* defined is $OS_signal.c */
extern void setup_child_sighandlers();
extern void setup_scheduler_sighandlers();
extern void init_sighandler_struct();
extern void setup_master_sighandlers();
extern void setup_term_sighandler();
extern void setup_chld_sighandler();

#ifdef __svr4__
#define WAIT_STAT_TYPE int 
#define TERM_SIG(stat) (stat & 0xff)
#define RET_CODE(stat) ((stat & 0xff00) >> 8)
#else
#define WAIT_STAT_TYPE union wait
#define TERM_SIG(stat) stat.w_termsig
#define RET_CODE(stat) stat.w_retcode
#endif

static void abnormal ARGS((WAIT_STAT_TYPE stat, int wid));
extern RESULT init_convert ARGS((int procs));
extern int ill_hndlr();
extern int seg_hndlr();

extern void monitorinit ARGS((int procs));
void relinquish_processor();
void child_exit ARGS((int));
void clean_shut_down ARGS(());

static int use_shared_memory;
shared int real_num_of_procs;
static volatile int total_procs_exit = 0;
static int total_procs_forked = 0;
static int parent_waiting_the_end = 0;
static int parent_in_exit = 0;

u_char          *shm_base, *monmem_base;
long            shm_offset, monmem_offset;
key_t 		shm_key, monmem_key;
int 		shm_fn, shm_id, monmem_fn, monmem_id;

void child_exit_handler ARGS((int wid, WAIT_STAT_TYPE stat));
int chld_hndlr ARGS((int signo, int code, void *scp, char *addr));

void
master_on_exit(int status)
{
    DBG3("In master on exit, VP=%d, total= %d, exit=%d\n",
	 virtual_processor(), total_procs_forked, total_procs_exit);
    parent_in_exit = 1;
    if (virtual_processor() == 0 &&
	total_procs_forked > 0 &&
	total_procs_exit > 0) {
	child_exit(status);
    }
}

int
current_processor()
{
    return private_current_proc;
}

void
child_init(node,start_func,arg)
int		node;
int		(*start_func)();
any_t		arg;
{
    setup_child_sighandlers(ill_hndlr, seg_hndlr, seg_hndlr);

    if (use_shared_memory) {
	shmdt(shm_base);
	shm_base = (u_char *)shmat(shm_id,0,0);
	if (shm_base == (u_char *) -1) {
	    perror("Child Cannot attach shared memory");
	    exit(1);
	}
#ifdef WITH_MONITORING
	if (monitoring_enabled) {
	    monmem_base = (u_char *)shmat(monmem_id,0,0);
	    if (monmem_base == (u_char *) -1) {
		perror("Cannot create shared memory");
		exit(1);
	    }
	}
#endif
    }

    private_current_proc = node;
    current_thread_array[node] = NULL;
    local_scheduler[virtual_processor()] = processor[virtual_processor()].scheduler;

    DBG3("In child %d, processor = %x, processes = %x\n",
	 virtual_processor(), (int) processor, (int) processes);
    DBG4("In child %d no_of_procs = %x *no_of_procs = %d mem_pool = %x\n",
	 virtual_processor(), (int) num_of_procs, *num_of_procs, 
	 (int)memory_pool);

    setup_scheduler_sighandlers();

    if (local_scheduler[virtual_processor()]->sched_vproc_init != 0) {
	(local_scheduler[virtual_processor()]->sched_vproc_init)(node);
    }

    threads_started = 1;


/*******************************************************************/
/* sync_init has to be equal to virtual_processor() + 1 to make sure */
/* that all the child proceses wait till the child 0 is done       */
/* forking the init thread. This is necessary so that the monitor  */
/* thread doesn't get the chance to lock the list->lock before the */
/* init thread executes add_thread_to_lm_list                      */
/*******************************************************************/
    while (*sync_init != virtual_processor() + 1) {
	struct timeval delay_time;
	int delay_unit = (*num_of_procs - virtual_processor());
	delay_time.tv_sec = 0;
	delay_time.tv_usec = delay_unit * 10000;  /* return millisecs */
	DBG3("Sync init = %d  current proc = %d delaying for %d\n",*sync_init,
	     virtual_processor(), delay_unit);
	select(32, NULL, NULL, NULL, &delay_time);
    }

    start_proc_initial_thread(start_func);

    atomadd(sync_init, -1);
    while (*sync_init != 0) {
	struct timeval delay_time;
	delay_time.tv_sec = 0;
	delay_time.tv_usec = (*sync_init) * 10000;  /* return millisecs */
	DBG2("Sync post init = %d  current proc = %d\n",
	     *sync_init, virtual_processor());
	select(32, NULL, NULL, NULL, &delay_time);
    }

    DBG("Children initialized. Starting our queue\n");
    if ((start_func == NULL) && (node == 0)) {
	current_thread_array[node] = *initial_thread;
	(*initial_thread)->already_started = 1;
	swap_context_with_func((void (*)()) local_scheduler[node]->schedule,
			       NULL);
    } else {
      local_scheduler[virtual_processor()]->schedule();
    }
} /* child init */

void
fork_children_and_wait(start_func, procs)
int (*start_func) ();
int procs;
{
    int node, i, chid;
    WAIT_STAT_TYPE stat;

    setup_chld_sighandler(chld_hndlr);

    total_procs_forked = total_procs_exit = procs;

    stat = (WAIT_STAT_TYPE) SIGINT;
    processes[0] = getpid();
    processes[procs] = PROC_SENTINEL;
    private_current_proc = 0;
    if (procs > 1) {
	for (node = 1; node < procs; node++) {
	    if ((chid = fork()) == -1) {
		perror("fork failed");
		fprintf(stderr, "Can't support so many processes.  Reduce -num_procs.\n");
		total_procs_forked = total_procs_exit = node;
		goto kill_children;
	    }
	    if (chid == 0) {
		private_current_proc = node;
		child_init(node, start_func, NULL);
		printf("(AQ) Internal Error:Returned from child init !!!!!!\n");
		exit(1);
	    } else {
		processes[node] = chid;
	    }
	}
	DBG("Process table:\n");
	for (i = 0, processes; i < procs; i++) {
	    DBG2("%d         %ld\n", i, processes[i]);
	}
    }
    child_init(0, start_func, NULL);
    return;
  kill_children:		/* GOTO TARGET FROM JUST ABOVE ! */
    child_exit_handler(processes[0], stat);
    return;
}

int
quit_hndlr( signo, code, scp, addr )
int signo, code;
struct sigcontext *scp;
char *addr;
{
    int i;
    process_table_t      pptr;
        
/* Kill the processes */
    for (i=0, pptr = processes; i< *num_of_procs; i++) {
	kill(*pptr++, SIGKILL);
    }

   /* shut down the IO control */
    close_global_io();
/* Clean up shared memory */

    shmdt(shm_base);
    shmctl(shm_id, IPC_RMID, 0);
    close(shm_fn); unlink(".shm_mem");

#ifdef WITH_MONITORING
    if (monitoring_enabled) {
	shmdt(monmem_base);
	shmctl(monmem_id, IPC_RMID, 0);
	close(monmem_fn); unlink(".mon_mem");
    }
#endif
    exit(1);
}

int
ill_hndlr( signo, code, scp, addr )
int signo, code;
struct sigcontext *scp;
char *addr;
{
    fflush(stdout);
    fflush(stderr);
    fprintf(stderr, "Illegal instruction signal.  Possible stack overflow on processor %d.\n", virtual_processor());
    fprintf(stderr, "Thread "); 
    fprint_thread(stderr, current_thread_array[virtual_processor()]);
    fprintf(stderr, " was active.\nTerminating program.\n");

    exit(4);
}

int
seg_hndlr( signo, code, scp, addr )
int signo, code;
struct sigcontext *scp;
char *addr;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];

    fflush(stdout);
    fflush(stderr);
    if (signo == 11) {
	fprintf(stderr, "Segmentation fault signal (11) on processor %d.\n", 
		virtual_processor());
    } else {
	fprintf(stderr, "Signal %d on processor %d.\n", signo,
		virtual_processor());
    }
#if 0
    if (FC_CODE(code) == FC_OBJERR) code = FC_ERRNO(code);

    if (code == SEGV_PROT) {
	fprintf(stderr, "Signal code suggests protection violation and a possible stack overflow.\n");
    }
#endif
    fprintf(stderr, "signo %d, code %d, addr %lx, stacktop %lx\n",
	   signo, code, (long) addr, (long)current_thread->stack_top);
    fprintf(stderr, "Thread "); 
    fprint_thread(stderr, current_thread);
    fprintf(stderr, " was active.\nTerminating program.\n");

    exit(1);
}

int
intr_hndlr( signo, code, scp, addr )
int signo, code;
struct sigcontext *scp;
char *addr;
{
   /* shut down the IO control */
    close_global_io();

    shmdt(shm_base); 
    shmctl(shm_id, IPC_RMID, 0); 
    close(shm_fn); unlink(".shm_mem");
#ifdef WITH_MONITORING
    if (monitoring_enabled) {
	shmdt(monmem_base); 
	shmctl(monmem_id, IPC_RMID, 0); 
	close(monmem_fn); unlink(".mon_mem");
    }
#endif
    exit(1);
}

void
create_shared_arena(procs) 
int procs;
{
    int size = generic_shared_size(procs);

    if (procs > 1) {
	use_shared_memory = 1;
	if ((shm_fn = open(".shm_mem",O_RDWR|O_CREAT|O_TRUNC, 0777)) == -1)
	  perror(".shm_mem");
	shm_key = ftok(".shm_mem", '1');	

	shm_id = shmget(shm_key, size, IPC_CREAT|0600);
	if (shm_id == -1) {
	    perror("SHared Memory GET failed!");
	    fprintf(stderr, "Possible cause:  Requested thread/processor/memory \
configuration requires\n\tmore shared memory than is \
available.  Reduce requirements.\n");
	    shmctl(shm_id, IPC_RMID, 0);
	    close(shm_fn);
	    unlink(".shm_mem");
	    exit(1);
	}
	shm_base = (u_char *)shmat(shm_id,0,0);
	if (shm_base == (u_char *) -1) {
	    perror("Cannot create shared memory");
	    shmctl(shm_id, IPC_RMID, 0);
	    close(shm_fn);
	    unlink(".shm_mem");
	    exit(1);
	}
	shm_offset = 0;
    } else {
	use_shared_memory = 0;
	shm_base = (u_char *) malloc(size);
	shm_offset = 0;
    }
}

int
chld_hndlr(signo, code, scp, addr)
int signo, code;
void *scp;
char *addr;
{
    WAIT_STAT_TYPE stat;
    int wid;

    DBG("In chld_hndlr\n");
#if !defined(__svr4__) && !defined(__SVR4)
    while ((wid = wait3((int *) &stat, WNOHANG | WUNTRACED, NULL)) != 0) {
	DBG1("Wait3 returns %ld\n", (long) wid);
	if (wid == -1) {
	    DBG("Returning from chld_hndlr\n");
	    return 0;
	}
	child_exit_handler(wid, stat);
    }
#else
    while ((wid = waitpid((pid_t) - 1, (int *) &stat, WNOHANG | WUNTRACED)) != 0) {
	DBG2("Waitpid returns %ld, stat = %lx\n", (long) wid, (long) stat);
	if (wid == -1) {
	    DBG("Returning from chld_hndlr\n");
	    return 0;
	}
	child_exit_handler(wid, stat);
    }
#endif
    DBG("Returning from chld_hndlr\n");
    return 0;
}


void
child_exit_handler(int wid, WAIT_STAT_TYPE stat)
{
    int i, force_parent_exit = 0;

    DBG("In Child Exit Handler\n");

    total_procs_exit--;
    if (TERM_SIG(stat) != 0 || RET_CODE(stat) != 0) {
	DBG3("Child %d terminated with termsig %d, retcode %d\n",
	     wid, TERM_SIG(stat), RET_CODE(stat));

	if (parent_waiting_the_end < 0) {
	    parent_waiting_the_end = 1;

	    /* 
	     * the parent is already ended and waiting for the children
	     * signal all the remaining children to exit 
	     */
	    for (i = 1; i < total_procs_forked; i++) {
		if (processes[i] != wid) {
		    DBG1("Killing child %lx\n", processes[i]);
		    kill(processes[i], SIGKILL);
		}
	    }
	} else if (parent_waiting_the_end == 0) {
	    parent_waiting_the_end = 1;
	    force_parent_exit = 1;
	    if (wid != processes[0])
		total_procs_exit--;

	    for (i = 1; i < total_procs_forked; i++) {
		processor[i].terminate = TERMINATE;
	    }
	    sleep(1);
	    for (i = 1; i < total_procs_forked; i++) {
		if (processes[i] != wid) {
		    DBG1("Killing child %lx\n", processes[i]);
		    kill(processes[i], SIGKILL);
		}
	    }
	} else {
	    /* the parent is already signaled to finish */
	}
	abnormal(stat, wid);
    } else {
	DBG("Normal exit\n");
	if (wid == processes[0]) {
	    force_parent_exit = 1;
	}
    }

    if (total_procs_exit <= 0) {
	clean_shut_down();
	if (!parent_in_exit)
	    exit(0);
    }
    if (force_parent_exit != 0) {
	while (total_procs_exit > 0) {

#if !defined(__svr4__) && !defined(__SVR4)
	    sigpause(0);	/* wait for the children to shut down */
#else
	    /* SVR4 looses the SIGCHLD while held */
	    wid = waitpid((pid_t) - 1, (int *) &stat, WUNTRACED);
	    if (wid == -1) {
		if (errno == ECHILD) {
		    total_procs_exit = 0;
		} else if (errno != EINTR) {
		    perror("Waitpid error");
		}
	    } else {
		total_procs_exit--;
		abnormal(stat, wid);
	    }
#endif
	}
	clean_shut_down();
	if (!parent_in_exit)
	    exit(0);
    }
}

void
child_exit(status)
int status;
{
    DBG1("in nonLWP child_exit %d\n", virtual_processor());
    if (virtual_processor() == 0) {
	int i;
	if (parent_waiting_the_end == 0) {
	    /* 
	     * the loop is mainly meant to shut down the steering and 
	     * local monitor processes 
	     */

	    for (i = 1; i < total_procs_forked; i++) {
		processor[i].terminate = TERMINATE;
	    }
	}
	child_exit_handler(processes[0], (WAIT_STAT_TYPE) 0);
    }
    if (!parent_in_exit) {
	DBG("in nonLWP child_exit exitting\n");
	exit(0);
    }
}

void
clean_shut_down()
{
    close_global_io();
    shmdt(shm_base);
    shmctl(shm_id, IPC_RMID, 0);
    close(shm_fn);
    unlink(".shm_mem");

    shmdt(monmem_base);
    shmctl(monmem_id, IPC_RMID, 0);
    close(monmem_fn);
    unlink(".mon_mem");
}

void
doinit(start_func, procs)
int             (*start_func)();
int             procs;
{
   int			node;
   long			*shared_shorts;
   int			total_procs = procs;
   
#ifdef WITH_MONITORING
   if (monitoring_enabled) {
       total_procs += number_of_lms;
       if (steering_enabled) total_procs++;
   }
#endif
   /*
    * Initialize the signal handler data structure.
    */
   init_sighandler_struct();

   create_shared_arena(total_procs);
#ifdef WITH_MONITORING
   if (monitoring_enabled)
     linux_monitorinit(total_procs);
#endif

   shared_shorts = (long *)allocate_and_share(SHARED_SHORT_SPACE,-1);

   num_of_procs		= &real_num_of_procs;
   *num_of_procs	= procs;
   total_threads	= (short *)(shared_shorts++);
   *total_threads	= 0;
   total_active		= (short *)(shared_shorts++);
   *total_active	= 0;
   sync_init		= (short *)(shared_shorts++);
   *sync_init		= total_procs;
#ifdef WITH_MONITORING
   monitor_status       = (short *)(shared_shorts++);
   *monitor_status      = (short) MONITOR_NOT_INITIALIZED;
#endif
   ss_lock              = (LOCK *)(shared_shorts++);
   LOCK_INIT(*ss_lock);

   print_sema = (LOCK *)(shared_shorts++);
   *print_sema = 0;

   init_shared_data(total_procs);

   DBG1("Before procinit total_procs %d \n",  total_procs);
   for (node=0; node < total_procs; node++) {
       procinit(node); 
       DBG1("procinit for node %d done \n", node);
   }

   DBG("Per processor structures inited\n");
    /* prevent the master to exit without shutting down the children */
    atexit((void(*)(void))master_on_exit);
   
   setup_master_sighandlers(intr_hndlr, quit_hndlr);
   if( init_global_io() < 0) exit(1);

    fork_children_and_wait(start_func, total_procs);

    /* never returns */
}

any_t
allocate_and_share(size,node)
unsigned	size;
int		node;
{
    u_char *address;
    address = (u_char *)((int)shm_base + shm_offset);
    shm_offset = shm_offset + size;
    return((any_t) address);
}

any_t
allocate_and_copy(size)
unsigned	size;
{
    char *address;

    if ((address = (char *) malloc(size)) == NULL) {
	perror("Internal allocate_and_copy");
	exit(1);
    }

    return((any_t) address);
}

char *
get_stack_memory(size)
int size;
{
    return (char *) allocate_and_copy(size);
}

static void
abnormal(stat, wid)
WAIT_STAT_TYPE	stat;
int		wid;
{
   process_table_t	pptr;

   fflush(stdout);
   fflush(stderr);
   for (pptr = processes; *pptr != PROC_SENTINEL && *pptr != wid ; pptr++) ;
   if( *pptr == PROC_SENTINEL){
      printf("Received a termination of an unknown child (%d) termsig %d retcode %d\n",
            wid, TERM_SIG(stat), RET_CODE(stat));
   } else {
       if( RET_CODE(stat) == 4) {
	   printf("Child %d died with illegal instruction signal 4 on logical proc %ld\n",
		  wid,  ((long)pptr - (long)processes) / sizeof(processes[0]));
       } else {
	   printf("Child %d died with termination signal %d on logical proc %ld\n",
		  wid, TERM_SIG(stat),
		  ((long)pptr - (long)processes) / sizeof(processes[0]));
       }
   }
}

extern RESULT
init_convert(procs)
int procs;
{
     int       i, j;

     /* if the processor is online, enter in the conversion table */
     j = 0;
     for (i=0; i<procs; i++) {
               processor[j++].logical_to_phys = i;
     }
     return(T_SUCCEED);
} /* init_convert */

void
check_num_procs(procs)
int procs;
{
    int total_procs = procs;
#ifdef WITH_MONITORING
    if (monitoring_enabled) {
	total_procs += number_of_lms;   /* procs reserved for monitoring */
	if (steering_enabled) total_procs++;
    }
#endif
    if (total_procs > MAX_PROC) {
	fprintf(stderr, "Requested number of processors exceeds Cthreads internal limits\n");
	exit(1);
    }
}

int
check_stack_size(size)
int size;
{
    size += getpagesize();   /* add one page for protection */

    /* round up to the nearest integral multiple of page size */
    if ((size % getpagesize()) != 0) {
	size = ((size / getpagesize()) + 1) * getpagesize();
    }
    return size;
}

void *
allocate_monmem(size,node)
int	size;
int	node;
{
    long   n;
    u_char *address;

    address = (u_char *) monmem_base + monmem_offset;
    if ((n = (unsigned long) address % sizeof(long))) {
	address += sizeof(long) - n;
	monmem_offset += sizeof(long) - n;
    }
    monmem_offset = monmem_offset + size;

    return((void *) address);
}

#ifdef WITH_MONITORING
static void 
linux_monitorinit(procs)
int procs;
{
    int size, monbuf_size, bufsteer_size;

    DBG2("monitor init, procs is %d, %d\n", procs, 
	 config.threads_per_proc);
    monbuf_size = sizeof(MonitorBuffer) + sizeof(struct cth_mutex)+
	size_of_monitor_buffer;
    bufsteer_size = size_of_lm_steer_buffer + sizeof(MonitorBuffer) +
	sizeof(struct cth_mutex);
    size =
	page_aligned_size(sizeof(LocalMonitorInfo)*number_of_lms +
			  MAX_NUM_SENSORS + 
			  procs * config.threads_per_proc * monbuf_size +
			  sizeof(unsigned long) +
			  sizeof(struct cth_mutex) +
			  MAX_NUM_SENSORS +
			  number_of_lms * bufsteer_size +
			  MAX_NUM_SAMPLING_INST *
			  (2*sizeof(unsigned long)+3*sizeof(void*)+sizeof(char))+
			  getpagesize());

    if ( (monmem_fn = open(".mon_mem",O_RDWR|O_CREAT|O_TRUNC, 0777)) == -1)
    perror(".mon_mem");
    monmem_key = ftok(".mon_mem", '1');

    DBG1("getting shared monitor memory, size is %d\n", size);
    monmem_id = shmget(monmem_key, size, IPC_CREAT|0600);
    if (monmem_id == -1) {
	perror("SHared Memory GET failed  Monmem!");
	shmctl(monmem_id, IPC_RMID, 0);
	close(monmem_fn);
	unlink(".mon_mem");
	exit(1);
    }
    monmem_base = (u_char *)shmat(monmem_id,0,0);
    if (monmem_base == (u_char *) -1) {
	perror("Cannot create monitor shared memory  monmem");
	shmctl(monmem_id, IPC_RMID, 0);
	close(monmem_fn);
	unlink(".monmem_mem");
	exit(1);
    }
    monmem_offset = 0;

    monitorinit(procs);
}
#endif

void
relinquish_processor()
{
    struct timeval delay_time;
    delay_time.tv_sec = 0;
    delay_time.tv_usec = 10000;  /* return millisecs */
    select(32, NULL, NULL, NULL, &delay_time);
}

unsigned long start_time;

unsigned long
cthread_timestamp()
{
#ifdef WITH_MONITORING
    struct timeval tp0;
    unsigned long  timestamp;

    extern int gettimeofday();

    if (logicalTimeStamp) {
	internal_mutex_lock(&logicalTimeStamp_lock->mlock);
	timestamp = (*logicalTimeStamp_counter)++;
	internal_mutex_unlock(&logicalTimeStamp_lock->mlock);
    } else {
	gettimeofday(&tp0, NULL);
	timestamp = (unsigned long) tp0.tv_sec*1000000 + tp0.tv_usec;
	timestamp -= start_time;
    }
    return timestamp;
#else
    return 0;
#endif
}

void
get_start_timestamp_and_resolution(start_time_p, resolution_p)
unsigned long *start_time_p;
unsigned long *resolution_p;
{
#ifdef WITH_MONITORING
    struct timeval tp0;

    extern int gettimeofday();

    if (logicalTimeStamp) {
	*resolution_p = 1;
	start_time = 0;
    } else {
	gettimeofday(&tp0, NULL);
	*resolution_p = 1000000;
	start_time = (unsigned long) tp0.tv_sec*1000000 + tp0.tv_usec;
    }
    *start_time_p = start_time;
#endif
}
