#include	<stdio.h>
#include	<math.h>
#include	<sys/wait.h>
#include	"assert.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <sys/fcntl.h>
#include <errno.h>
#include <ksr/ksr_timers.h>

#include	"results.h"
#include	"general.h"
#include	"init.h"
#include	"lock.h"
#include	"queue.h"
#include	"sync.h"
#include	"internal.h"
#include	"memalloc.h"
#include	"cth_sched.h"
#include	"hwlib.h"
#include	"sched_utils.h"
#include	"idebug.h"
#include	"threads.h"
#include "monitor.h"
#include	"ksr_init.h"

extern void monitorinit ARGS((int procs));
static void ksr_monitorinit ARGS((int procs));
static void get_start_times_master ARGS((int total_procs, 
					      atom_int *sync_init, 
					      int current_proc));

/* these symbols needed from init.c */
struct sched_configuration       mon_sched_conf;
extern shared short             *monitor_status;
extern shared mutex_t           logicalTimeStamp_lock;
extern shared unsigned long     *logicalTimeStamp_counter;

extern private int current_proc;
extern cthread_t current_thread_array;
extern private locked_queue_t local_thread_queue[MAX_PROC];
extern struct local_info *processor;
extern short threads_started;
extern shared int *num_of_procs;
extern atom_int *total_threads;
extern atom_int *total_active;
typedef long *process_table_t;
extern process_table_t	processes;	/* An array of the processes 	*/
extern private struct sched		*local_scheduler[MAX_PROC];

shared volatile __align128 int   real_num_of_procs;
shared atom_int  real_total_threads;
shared atom_int  real_total_active;
shared atom_int real_sync_init;
shared volatile __align128 short real_monitor_status;
shared LOCK real_print_sema;
extern LOCK *print_sema;

atom_int *sync_init;
processor_name_t	PElist[MAX_PROC];
pthread_t		Thread_id[MAX_PROC];
mutex_t print_lock;
typedef	struct	child_init_str {
    int	node;
    int	(*func)();
};

/* 
*this #define and the following two arrays are for calculating
* clock deltas and determining start times on each processor.  Se docs at
* get_start_times_master().
*/
#define NUM_DELTAS 7
static __align128 long start_time[MAX_PROC];
static __align128 long child_start_time[MAX_PROC];

extern void stack_overflow_hndlr();
static int total_procs;

int
current_processor()
{
    return private_current_proc;
}

void *
child_init(arg)
void	*arg;
{
    struct	child_init_str	*args = (struct	child_init_str *) arg;
    int i;
    int proc = virtual_processor();

    procinit(args->node);
    DBG("Per processor structures inited\n");

    setup_child_sighandlers(stack_overflow_hndlr, stack_overflow_hndlr, stack_overflow_hndlr);

    proc = args->node;
    local_thread_queue[proc] = *(processor[proc].scheduler->lqueue);
    local_scheduler[proc] = processor[proc].scheduler;
    current_thread[proc] = NULL;

    setup_scheduler_sighandlers();

    if (local_scheduler[proc]->sched_vproc_init != 0) {
        (local_scheduler[proc]->sched_vproc_init)(proc);
    }

    /* child participating in start time determination.  See below.*/
    for(i=0; i<NUM_DELTAS; i++) {
	while (child_start_time[proc] != -1) {
	    pthread_yield();
	}
	atomadd(sync_init, -1);
	_pstsp(sync_init);	/* post store subpage */
	child_start_time[proc] = x_all_timer();
    }

    DBG1("proc %d waiting for release\n", proc);
    while (*sync_init != proc + 1) {
	pthread_yield();
    }
    atomadd(sync_init, 1);

    if (proc == total_procs-1) {
	DBG1("proc %d initiating for 2nd release\n", proc);
	atomadd(sync_init, -1);
    }
    DBG1("proc %d waiting for 2nd release\n", proc);
    threads_started = 1;
    while (*sync_init != proc + 1) {
	pthread_yield();
    }

    start_proc_initial_thread(args->func);

    atomadd(sync_init, -1);
    while (*sync_init != 0) {
	pthread_yield();
    }
    DBG("Children initialized. Starting our queue\n");
    local_scheduler[proc]->schedule();
} /* child init */

RESULT
init_convert(num_processors)
int	num_processors;
{
    int	i;

    for(i=0; i<num_processors; i++) {
	processor[i].logical_to_phys = PElist[i];
    }
    return(T_SUCCEED);
}

RESULT
doinit(start_func, num_nodes)
int		(*start_func)();
int		num_nodes;
{
   struct	child_init_str	pthread_fork_args[MAX_PROC];
   int			num_avail_proc; 
   processor_name_t	this_proc;
   int			i, node, size;
   int			stat;
   int			cur_proc_index;
   processor_name_t	current_proc;

   total_procs = num_nodes;

   init_sighandler_struct();


   if (monitoring_enabled) {
       total_procs += number_of_lms;
       if (steering_enabled) total_procs++;
       ksr_monitorinit(total_procs);
   }

   size = MAX_PROC*sizeof(processor_name_t);

   num_of_procs		= &real_num_of_procs;
   *num_of_procs        = num_nodes;
   total_threads	= &real_total_threads;
   *total_threads	= 0;
   total_active 	= &real_total_active;
   *total_active	= 0;
   sync_init 		= &real_sync_init;
   *sync_init		= total_procs;
   monitor_status       = &real_monitor_status;
   *monitor_status      = MONITOR_NOT_INITIALIZED;
   print_sema		= &real_print_sema;
   LOCK_INIT(*print_sema);

   init_shared_data(total_procs);

   if ((num_avail_proc = pthread_procsetinfo(PElist, MAX_PROC)) < *num_of_procs) {
       printf("Not enough PEs available\n");
       exit(1);
   }

   /* bind this thread to where it is so it doesn't (or at least is 
      less likely to) move on us.... */
   pthread_getproc(pthread_self(), &current_proc);
   cur_proc_index = 0;
   for(i=0; i< num_avail_proc; i++) {
       if (current_proc == PElist[i]) {
	   DBG1(" KSR_INIT Currently executing on processor %d\n", i);
	   cur_proc_index = i;
       }
   }
   if (pthread_move(pthread_self(), PElist[cur_proc_index], 
		    TH_BOUND_PROCESSOR) != 0) {
       printf("Pthread_move returns error, node %d\n", node);
       exit(1);
   }


   /* prepare to fork and bind all the Pthread execution engines */
   for(node = 0; node< total_procs; node++){
       pthread_fork_args[node].node = node;
       pthread_fork_args[node].func = start_func;
   }

   for(node = 0; node< total_procs; node++){
       child_start_time[node] = 0;
       if((pthread_create(&(Thread_id[node]), pthread_attr_default, child_init,
			  (void *) (&(pthread_fork_args[node])))) != 0)  {
	   printf("Pthread_create returns error, node %d\n", node);
	   exit(1);
       }
       if (pthread_move(Thread_id[node], PElist[node], TH_BOUND_PROCESSOR) != 0) {
	   printf("Pthread_move returns error, node %d\n", node);
	   exit(1);
       }
   }

   DBG("Processes spawned\n");

   get_start_times_master(total_procs, sync_init, cur_proc_index);

   /* release the processors */
   DBG1("done with timings, sync_init is %d\n", *sync_init);
   atomadd(sync_init, 1);
   for(i=0; i<total_procs; i++) {
       pthread_join(Thread_id[i], (void**)&stat);
       pthread_detach(&Thread_id[i]);
   }
   exit(0);
}

static void
get_start_times_master(total_procs, sync_init, current_proc)
int 		total_procs;
atom_int 	*sync_init;
int		current_proc;
{
   /* Try to figure out reasonable start_time values......................
    *    This is complicated because the real-time clock, x_all_timer() isn't
    *    synchronized across KSR rings.  Here, we try to figure out roughly
    *    what the delta is between this processor's clock and those of the
    *    child threads.  We do this NUM_DELTAS times and later use the middle
    *    value.  The basic technique is that the child N is waiting for the
    *    value of child_start_time[N] to be set to -1.  When that happens, the
    *    child decrements *sync_init, post-stores the subpage containing it
    *    and grabs the value of x_all_timer() and puts it in
    *    child_start_time[N].  Meanwhile, this thread is watching for the
    *    value of *sync_init to change.  As soon as it sees the new value, it
    *    grabs x_all_timer() and stores it in start_time[N].  The difference
    *    between these two is the clock delta value for that iteration.
    *    We do multiple iterations to get multiple guestimates for the delta.
    *    These multiple guestimates are then bubble sorted and the middle
    *    value choosen for the clock delta.  We can then calculate processor
    *    start times by adding this delta to a local clock value.
    */    
    int 	i, j;
    long	delta[NUM_DELTAS][MAX_PROC];

    for (i = 0; i< NUM_DELTAS; i++) { 
	int last;
	*sync_init = total_procs;
	last = *sync_init;
	child_start_time[*sync_init-1] = -1;   /* trigger first proc */
	while (*sync_init != 0) {
	    int proc = *sync_init;
	    if (proc != last) {
		start_time[proc] = x_all_timer();
		child_start_time[proc-1] = -1;
		last = proc;
	    }
	}
	start_time[0] = x_all_timer();
       /* clock info in place */
       for(j=total_procs-1; j>=0; j--) {  /* loop in reverse */
	   /* doing this in reverse averts race conditions 
	      on child_start_time[0] */
	   delta[i][j] = child_start_time[j] - start_time[j];
	   child_start_time[j] = 0;
       }
   }

   for (j=0 ; j < total_procs; j++) {
       /* bubble sort the deltas */
       for (i=0; i<NUM_DELTAS; i++) {
	   int k;
	   for(k = i+1; k<NUM_DELTAS; k++) {
	       if (labs(delta[i][j]) > labs(delta[k][j])) {
		   long tmp = delta[k][j];
		   delta[k][j] = delta[i][j];
		   delta[i][j] = tmp;
	       }
	   }
       }
   }
   {  /* set the final start_time values using current time and middle delta */
       long time_now = x_all_timer();
       int mid_delta = NUM_DELTAS/2 -1;

       delta[mid_delta][current_proc] = 0;
       for (j=0 ; j < total_procs; j++) {
	   if (labs(delta[mid_delta][j]) < 50) delta[mid_delta][j] = 0;
	   start_time[j] = time_now + delta[mid_delta][j];
       }
   }
   if ((debug_table_size > 0)&&debug_query(__FILE__)) {
       for (i=0; i< NUM_DELTAS; i++) {
	   int j;
	   printf("delta[%d] is: ", i);
	   for (j=0; j<total_procs-1; j++) {
	       printf("%lx, ", delta[i][j]);
	   }
	   printf("%lx\n", delta[i][total_procs -1]);
       }
       printf("      start times are: ");
       for (i=0; i<total_procs-1; i++) {
	   printf("%lx, ", start_time[i]);
       }
       printf("%lx\n", start_time[total_procs -1]);
   }

}
check_num_procs(procs)
int procs;
{
    int total_procs = procs;
    int count = pthread_procsetinfo(NULL, 0);
    if (monitoring_enabled) {
	total_procs += number_of_lms;   /* procs reserved for monitoring */
	if (steering_enabled) total_procs++;
    }
    if(count < total_procs)  {
	printf("Value of number of processors is greater than available\n");
	exit(1);
    }
}

int
check_stack_size(size)
int size;
{
    return(size);
}


any_t
allocate_and_share(size, node)
unsigned	size;
int		node;
{
    any_t	address;

    address = (any_t) malloc(size);
    if(! address) {
	printf("Can't allocate memory\n"); 
    }
    return(address);
}


any_t
allocate_and_copy(size)
unsigned	size;
{
    vm_address_t	address;

    address = 0;
    address = (vm_address_t) malloc(size);
    if(! address) {
	printf("Can't allocate memory\n"); 
    }
    return(address);
}

void *
allocate_monmem(size,node)
int	size;
int	node;
{

    vm_address_t	address;

    address = 0;
    address = (vm_address_t) malloc(size);
    if(! address) {
	printf("Can't allocate memory\n"); 
    }
    return((void *)address);
}

static void 
ksr_monitorinit(procs)
int procs;
{
    monitorinit(procs);
    print_lock = (mutex_t)allocate_monmem(sizeof(struct cth_mutex), N_CURRENT);
}

void
relinquish_processor()
{
    pthread_yield();
}

void
stack_overflow_hndlr( signo, sip)
int signo;
siginfo_t *sip;
{
    fflush(stdout);
    fflush(stderr);
    fprintf(stderr, "signal %d on processor %d.\n", signo,
	    virtual_processor());
/*    if (sip->si_code == SEGV_ACCERR*/
    fprintf(stderr, "This is a possible stack overflow sip = %x\n", sip);

    fprintf(stderr, "code is %x, Thread ", sip->si_code); 
    fprint_thread(stderr, current_thread_array[virtual_processor()]);
    fprintf(stderr, " was active.\nTerminating program.\n");

    exit(1);
}

child_exit(status)
int status;
{
    pthread_exit((void *)status);
}

unsigned long
cthread_timestamp()
{
    unsigned long  timestamp;

    if (logicalTimeStamp) {
	internal_mutex_lock(&logicalTimeStamp_lock->mlock);
	timestamp = (*logicalTimeStamp_counter)++;
	internal_mutex_unlock(&logicalTimeStamp_lock->mlock);
    } else {
	timestamp = (unsigned long) x_all_timer();
	timestamp -= start_time[current_proc];
    }
    return timestamp;
}

void
get_start_timestamp_and_resolution(start_time_p, resolution_p)
unsigned long *start_time_p;
unsigned long *resolution_p;
{
    if (logicalTimeStamp) {
	*resolution_p = 1;
    } else {
	long a = 1.0/secs(8);
	/* timestamp_resolution = 5000000; */
	*resolution_p = (a + 10) / 10000 * 10000;
    }
    *start_time_p = 0;
}

char *
get_stack_memory(size)
int size;
{
    int page_size = getpagesize();
    char *tmp_ptr;
    long tmp = 0;

    /* do weird shit on the KSR.  Try to use the stack of the current 
    ** Pthread as stacks for the child threads.
    ** This avoids a problem in KSROS 1.2.2 where we cannot be executing
    ** on a stack unknown to the OS.
    */
    char *stack_limit = (char *) &stack_limit;

    size += 2000;

    /*  round down to nearest page size multiple */
    stack_limit = (char *) ((((long)stack_limit-size)/ page_size) * page_size);

    for(tmp_ptr = ((char *) &stack_limit) - 16;
	tmp_ptr >= stack_limit;
	tmp_ptr -= page_size) {
	*tmp_ptr = 0;
    }

    return stack_limit;
}

    
