/* socket_sunos4.c
 *
 *      open
 *      close
 *      socket
 *      dup2, dup
 *      fcntl
 *      bind
 *      listen
 *      accept
 *      connect
 *      recv, recvfrom, readv
 *      send, sendto, writev
 */
#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#ifdef HAVE_WINDOWS_H
#ifndef _MT
#define _MT
#endif
#define FD_SETSIZE 1024
#include <windows.h>
#include <winsock.h>
struct iovec {
    void *iov_base;
    int iov_len;
};
int writev(int fd, struct iovec *iov, int iovcnt);
int readv(int fd, struct iovec *iov, int iovcnt);
/* 
 * the following three should be declared by $INCLUDE/io.h, but that filename
 * conflicts with the PBIO include file, so we just include the declarations
 * here.
 */
int open(const char *, int,...);
int dup(int);
int dup2(int, int);
#define NOBLOCK	1
#define BLOCK	0
#define O_NONBLOCK	0x4000
#define O_NDELAY	0x0004
#define F_DUPFD         0	/* Duplicate fildes */
#define F_GETFD         1	/* Get fildes flags (close on exec) */
#define F_SETFD         2	/* Set fildes flags (close on exec) */
#define F_GETFL         3	/* Get file flags */
#define F_SETFL         4	/* Set file flags */
#ifndef _POSIX_SOURCE
#define F_GETOWN        5	/* Get owner - for ASYNC */
#define F_SETOWN        6	/* Set owner - for ASYNC */
#endif				/* !_POSIX_SOURCE */
#define F_GETLK         7	/* Get record-locking information */
#define F_SETLK         8	/* Set or Clear a record-lock * *
				 * (Non-Blocking) */
#define F_SETLKW        9	/* Set or Clear a record-lock (Blocking) */
#ifndef _POSIX_SOURCE
#define F_RGETLK        10	/* Test a remote lock to see if it is * *
				 * blocked */
#define F_RSETLK        11	/* Set or unlock a remote lock */
#define F_CNVT          12	/* Convert a fhandle to an open fd */
#endif
#else
#include <unistd.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#ifdef HAVE_LINUX_UIO_H
/* amateurs. */
#include <linux/uio.h>
int readv __P((int __fd, __const struct iovec * __vector, size_t __count));
int writev __P((int __fd, __const struct iovec * __vector, size_t __count));
#else
#include <sys/uio.h>
#endif

#endif
#include <fcntl.h>
#include <memory.h>
#include <signal.h>

#ifdef STDC_HEADERS
#include <stdarg.h>
#else
#include <varargs.h>
#endif
#include <errno.h>


#include "results.h"
#include "general.h"
#include "sched_utils.h"
#include "internal.h"
#include "sched_io.h"
#include "sched_io_local.h"
#include "init.h"
#include "io_interface.h"
#include "arch_init.h"

int default_syscall(int,...);

#ifndef HAVE_WINDOWS_H
/* 
 * UNIX calls not defined on some systems.  If these definitions conflict 
 * on any systems, add a check for the definition of the function to 
 * configure.in (perhaps must add to chaos_base/acsite.m4 and ./acconfig.h 
 * also).  Then rerun configure and autoheader.
 */
#ifndef RECV_DEFINED
extern int recv ARGS((int s, char *buf, int len, int flags));
#endif
#if !defined(RECVFROM_DEFINED) && !defined(recvfrom)
extern int recvfrom ARGS((int s, char *buf, int len, int flags,
			  struct sockaddr * from, int *fromlen));
#endif
#ifndef SEND_DEFINED
extern int send ARGS((int s, const char *buf, int len, int flags));
#endif
#ifndef SENDTO_DEFINED
extern int sendto ARGS((int s, const char *msg, int len, int flags,
			struct sockaddr * to, int tolen));
#endif
#ifndef ACCEPT_DEFINED
extern int accept ARGS((int s, struct sockaddr * addr, int *addrlen));
#endif
#ifndef CONNECT_DEFINED
extern int connect ARGS((int s, struct sockaddr * name, int namelen));
#endif
#ifndef BIND_DEFINED
extern int bind ARGS((int s, struct sockaddr * name, int namelen));
#endif
#ifndef READV_DEFINED
extern int readv ARGS((int s, struct iovec * iov, int iovcnt));
#endif
#ifndef WRITEV_DEFINED
extern int writev ARGS((int s, const struct iovec * iov, int iovcnt));
#endif
extern int socket ARGS((int domain, int type, int protocol));
#ifndef LISTEN_DEFINED
extern int listen ARGS((int s, int backlog));
#endif

#endif
#ifndef EINPROGRESS
#define EINPROGRESS WSAEINPROGRESS
#define EWOULDBLOCK WSAEWOULDBLOCK
#define ENOBUFS WSAENOBUFS
#define EPROTONOSUPPORT WSAEPROTONOSUPPORT
#endif

/* >>>>>>>>>>>>>>>> socket interface  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */

/*--------------------------  SYS_open -------------------------------*/

int
cth_open(const char *path, int flags,...)
{
    int fd, ret;
    va_list args;

#ifdef STDC_HEADERS
    va_start(args, flags);
#else
    va_start(args);
#endif

    fd = open(path, flags, va_arg(args, int));
    if (fd > 0) {
	if (flags & O_NDELAY || flags & O_NONBLOCK) {
	    socket_descriptors[fd]->flags |= O_NDELAY;
	} else {
#ifdef HAVE_WINDOWS_H
	    int arg = NOBLOCK;
	    ret = ioctlsocket(fd, FIONBIO, &arg);
	    if (ret == SOCKET_ERROR) {
		closesocket(fd);
		return ret;
	    }
#else
	    ret = fcntl(fd, F_SETFL, O_NDELAY);
	    if (ret < 0) {
		close(fd);
		return ret;
	    }
#endif
	}
	FD_SET(fd, global_socket_list);
	if (fd > FD_SETSIZE) {
	    fprintf(stderr, "Internal Error, stupid WINSOCK large FD bug.\n");
	    fprintf(stderr, "Increase FD_SETSIZE.  Item not added to fdset.\n");
	}
	socket_descriptors[fd] = &socket_descriptor_list[fd];
	socket_descriptors[fd]->duplicates = 1;
	socket_descriptors[fd]->syscall = default_syscall;
#ifdef ATOM
	atomadd(global_socket_count, 1);
#else
	mutex_lock(global_socket_lock);
	(*global_socket_count)++;
	mutex_unlock(global_socket_lock);
#endif
    }
    return fd;
}


/*--------------------------  SYS_close -------------------------------*/


int
cth_close(int s)
{
    int ret;
    if (global_socket_list && FD_ISSET(s, global_socket_list)) {
	ret = socket_descriptors[s]->syscall(SYS_close, s);
	if (ret < 0 && ret == EINTR)
	    return ret;

	FD_CLR((unsigned int) s, global_socket_list);
	socket_descriptors[s]->duplicates--;
	socket_descriptors[s] = NULL;
#ifdef ATOM
	atomadd(global_socket_count, -1);
#else
	mutex_lock(global_socket_lock);
	(*global_socket_count)--;
	mutex_unlock(global_socket_lock);
#endif
	return ret;
    } else {
#ifdef HAVE_WINDOWS_H
	return closesocket(s);
#else
	return close(s);
#endif
    }
}




/*--------------------------  SYS_socket -------------------------------*/
int
cth_socket(int domain, int type, int protocol)
{
    int ret, sckt;

    switch (domain) {
    case PF_LOCAL:
	errno = EPROTONOSUPPORT;
	return -1;
	break;
    default:
	/* PF_UNIX, PF_INET, PF_IMPLINK */
	/* fprintf(stderr,"socket request %d %d %d\n",  domain, type, * *
	 * protocol); */
	ret = socket(domain, type, protocol);

	if ((sckt = ret) >= 0) {
	    /* fprintf(stderr,"socket receive %d %d %d %d\n", sckt, * *
	     * domain, type, protocol); */
#ifdef HAVE_WINDOWS_H
	    int arg = NOBLOCK;
	    ret = ioctlsocket(sckt, FIONBIO, &arg);
#else
	    ret = fcntl(sckt, F_SETFL, O_NDELAY);
#endif

	    if (ret >= 0) {

		FD_SET(sckt, global_socket_list);
		if (sckt > FD_SETSIZE) {
		    fprintf(stderr, "Internal Error, stupid WINSOCK large FD bug.\n");
		    fprintf(stderr, "Increase FD_SETSIZE.  Item not added to fdset.\n");
		}
		socket_descriptors[sckt] = &socket_descriptor_list[sckt];
		socket_descriptors[sckt]->type = type;
		socket_descriptors[sckt]->domain = domain;
		socket_descriptors[sckt]->protocol = protocol;
		socket_descriptors[sckt]->flags = 0;
		socket_descriptors[sckt]->state = 0;
		socket_descriptors[sckt]->duplicates = 1;
		socket_descriptors[sckt]->syscall = default_syscall;

#ifdef ATOM
		atomadd(global_socket_count, 1);
#else
		mutex_lock(global_socket_lock);
		(*global_socket_count)++;
		mutex_unlock(global_socket_lock);
#endif
		return sckt;
	    } else
		return ret;
	} else
	    return ret;
	break;
    }
}

/*--------------------------  SYS_dup2 -------------------------------*/

int
cth_dup2(int s1, int s2)
{
    int ret;

    if (global_socket_list && FD_ISSET(s1, global_socket_list)) {
	ret = socket_descriptors[s1]->syscall(SYS_dup2, s1, s2);
	if (ret == s2) {
	    if (socket_descriptors[s2] != NULL) {
		socket_descriptors[s2]->syscall(SYS_close, s2);
		socket_descriptors[s2]->duplicates--;
	    }
	    socket_descriptors[s2] = socket_descriptors[s1];
	    socket_descriptors[s2]->duplicates++;
	}
	return ret;
    } else {
	return dup2(s1, s2);
    }
}


/*--------------------------  SYS_dup -------------------------------*/
int
cth_dup(int s)
{
    int ret;

    if (global_socket_list && FD_ISSET(s, global_socket_list)) {
	ret = socket_descriptors[s]->syscall(SYS_dup, s);
	if (ret >= 0) {
	    socket_descriptors[ret] = socket_descriptors[s];
	    socket_descriptors[ret]->duplicates++;
	}
	return ret;
    } else {
	return dup(s);
    }
}

/*--------------------------  SYS_fcntl -------------------------------*/

int
cth_fcntl(int s, int cmd,...)
{
    int ret, s2, arg;
    va_list args;

#ifdef STDC_HEADERS
    va_start(args, cmd);
#else
    va_start(args);
#endif

    if (global_socket_list && FD_ISSET(s, global_socket_list)) {
	switch (cmd) {
	case F_DUPFD:
	    for (s2 = s + 1; s2 < sizeof(fd_set); s2++) {
		if (socket_descriptors[s2] == NULL) {
		    return dup2(s, s2);
		}
	    }
	    errno = EMFILE;
	    return -1;
	    break;
	case F_SETFL:
	    arg = va_arg(args, int);

	    if (arg & O_NDELAY) {
		socket_descriptors[s]->flags |= O_NDELAY;
		arg &= ~O_NDELAY;
	    } else if (socket_descriptors[s]->flags & O_NDELAY) {
		socket_descriptors[s]->flags &= ~O_NDELAY;
	    }
	    return socket_descriptors[s]->syscall(SYS_fcntl, s, cmd, arg);

	    break;
	case F_GETFL:
	    arg = va_arg(args, int);

	    ret = socket_descriptors[s]->syscall(SYS_fcntl, s, cmd, arg);
	    if (!(socket_descriptors[s]->flags & O_NDELAY)) {
		ret &= ~O_NDELAY;
	    }
	    return ret;
	    break;
	default:{
#ifdef HAVE_WINDOWS_H
		/* 
		 *  ioctlsocket isn't really a substitute for fcntl, 
		 *  but it's what we have on NT.  We'll drop this in tentatively
		 *  to make the compiler happy.  Having it work is another matter.
		 */
		unsigned int *NT_arg = &arg;
		arg = va_arg(args, int);
		return ioctlsocket(s, cmd, NT_arg);
#else
		return fcntl(s, cmd, args);
#endif
	    }

	    break;
	}
    } else {
#ifdef HAVE_WINDOWS_H
	/* 
	 *  ioctlsocket isn't really a substitute for fcntl, 
	 *  but it's what we have on NT.  We'll drop this in tentatively
	 *  to make the compiler happy.  Having it work is another matter.
	 */
	unsigned int *NT_arg = &arg;
	arg = va_arg(args, int);
	return ioctlsocket(s, cmd, NT_arg);
#else
	return fcntl(s, cmd, args);
#endif
    }
}


/*--------------------------  SYS_accept -------------------------------*/

int
cth_accept(int s, struct sockaddr *addr, int *addrlen)
{
    int ret, socket;

    if (global_socket_list && FD_ISSET(s, global_socket_list)) {

	if (!(socket_descriptors[s]->flags & O_NDELAY)) {
	    IO_YIELD(s, PENDING_ACCEPT);

	    if (sched_io_has_error(s)) {
		fprintf(stderr, "accept error %d\n", errno);
		return -1;
	    }
	}
	ret = socket_descriptors[s]->syscall(SYS_accept, s, addr,
					     addrlen);
	if ((socket = ret) >= 0 &&
#ifdef HAVE_WINDOWS_H
	    (ioctlsocket(socket, FIONREAD, &ret)) != SOCKET_ERROR) {
#else
	    (ret = fcntl(socket, F_SETFL, O_NDELAY)) >= 0) {
#endif
	    FD_SET(socket, global_socket_list);
	    if (socket > FD_SETSIZE) {
		fprintf(stderr, "Internal Error, stupid WINSOCK large FD bug.\n");
		fprintf(stderr, "Increase FD_SETSIZE.  Item not added to fdset.\n");
	    }
	    socket_descriptors[socket] = &socket_descriptor_list[socket];
	    socket_descriptors[socket]->type = socket_descriptors[s]->type;
	    socket_descriptors[socket]->domain =
		socket_descriptors[s]->domain;
	    socket_descriptors[socket]->protocol =
		socket_descriptors[s]->protocol;
	    socket_descriptors[socket]->flags =
		socket_descriptors[s]->flags;
	    socket_descriptors[socket]->state = 0;
	    socket_descriptors[socket]->duplicates = 1;
	    socket_descriptors[socket]->syscall =
		socket_descriptors[s]->syscall;
#ifdef ATOM
	    atomadd(global_socket_count, 1);
#else
	    mutex_lock(global_socket_lock);
	    (*global_socket_count)++;
	    mutex_unlock(global_socket_lock);
#endif
	    return socket;
	}
	else {
	    return ret;
	}
    } else {
	return accept(s, addr, addrlen);
    }
}




/*--------------------------  SYS_connect -------------------------------*/

int
cth_connect(int s, struct sockaddr *name, int namelen)
{
    int ret;

    if (global_socket_list && FD_ISSET(s, global_socket_list)) {

	ret = socket_descriptors[s]->syscall(SYS_connect, s, name, namelen);
	if (ret == 0 || errno != EINPROGRESS ||
	    (socket_descriptors[s]->flags & O_NDELAY))
	    return ret;

	/* EINPROGRESS and blocking I/O  only */

	IO_YIELD(s, PENDING_CONNECT);
	if (sched_io_has_error(s)) {
	    fprintf(stderr, "connect error %d\n", errno);
	    return -1;
	}
	return 0;
    } else {
	return connect(s, name, namelen);
    }
}



/*--------------------------  SYS_bind -------------------------------*/
int
cth_bind(int s, struct sockaddr *name, int namelen)
{
    if (global_socket_list && FD_ISSET(s, global_socket_list)) {
	return socket_descriptors[s]->syscall(SYS_bind, s, name, namelen);
    } else {
	return bind(s, name, namelen);
    }
}


/*--------------------------  SYS_listen -------------------------------*/
int
cth_listen(int s, int backlog)
{
    if (global_socket_list && FD_ISSET(s, global_socket_list)) {
	return socket_descriptors[s]->syscall(SYS_listen, s, backlog);
    } else {
	return listen(s, backlog);
    }
}




/*--------------------------  SYS_recv -------------------------------*/

/* !!ATTENTION:  out of band  events not considered  */


int
cth_recv(int s, char *buf, int len, int flags)
{
    int ret;

    if (global_socket_list && FD_ISSET(s, global_socket_list)) {

	do {
	    ret = socket_descriptors[s]->syscall(SYS_recv, s, buf, len, flags);
	    if (ret >= 0 || errno != EWOULDBLOCK ||
		socket_descriptors[s]->flags & O_NDELAY) {
		return ret;
	    }
	    IO_YIELD(s, PENDING_RECV);
	    if (sched_io_has_error(s)) {
		fprintf(stderr, "recv error %d\n", errno);
		return -1;
	    }
	} while (1);
    } else {
	return recv(s, buf, len, flags);
    }
}


/*--------------------------  SYS_recvfrom -------------------------------*/

int
cth_recvfrom(int s, char *buf, int len, int flags,
	     struct sockaddr *from, int *fromlen)
{
    int ret;

    if (global_socket_list && FD_ISSET(s, global_socket_list)) {

	do {
	    ret = socket_descriptors[s]->syscall(SYS_recvfrom, s, buf, len,
						 from, fromlen);
/* fprintf(stderr,"recvfrom ret (%d) %d %d\n",s, ret, errno);
 * inet_ntoa(((struct  sockaddr_in *)from)->sin_addr)); */
	    if (ret >= 0 || errno != EWOULDBLOCK ||
		socket_descriptors[s]->flags & O_NDELAY) {
		return ret;
	    }
	    IO_YIELD(s, PENDING_RECVFROM);

	    if (sched_io_has_error(s)) {
		fprintf(stderr, "recvfrom error %d\n", errno);
		return -1;
	    }
	} while (1);
    } else {
	return recvfrom(s, buf, len, flags, from, fromlen);
    }
}

/*--------------------------  SYS_readv -------------------------------*/

int
cth_readv(int s, struct iovec *iov, int iovcnt)
{
    int ret;

    if (global_socket_list && FD_ISSET(s, global_socket_list)) {
	do {
	    ret = socket_descriptors[s]->syscall(SYS_readv, s, iov, iovcnt);
	    if (ret >= 0 || errno != EWOULDBLOCK ||
		socket_descriptors[s]->flags & O_NDELAY) {
		return ret;
	    }
	    IO_YIELD(s, PENDING_READV);
	    if (sched_io_has_error(s)) {
		fprintf(stderr, "readv error %d\n", errno);
		return -1;
	    }
	} while (1);
    } else {
	return readv(s, iov, iovcnt);
    }
}


/*--------------------------  SYS_send -------------------------------*/
int
cth_send(int s, char *buf, int len, int flags)
{
    int res;

    if (global_socket_list && FD_ISSET(s, global_socket_list)) {
	do {
	    if (sched_io_has_error(s)) {
		fprintf(stderr, "send error %d\n", errno);
		return -1;
	    }
	    res = socket_descriptors[s]->syscall(SYS_send, s, buf, len,
						 flags);

	    if (res >= 0 || errno == EINTR ||
		(socket_descriptors[s]->flags & O_NDELAY) ||
		(errno != ENOBUFS && errno != EWOULDBLOCK))
		return res;

	    IO_YIELD(s, PENDING_SEND);
	} while (1);
    } else {
	return send(s, buf, len, flags);
    }
}


/*--------------------------  SYS_sendto -------------------------------*/
int
cth_sendto(int s, char *buf, int len, int flags, struct sockaddr *to,
	   int tolen)
{
    int ret;

    if (global_socket_list && FD_ISSET(s, global_socket_list)) {
	do {
	    if (sched_io_has_error(s)) {
		fprintf(stderr, "sendto error %d\n", errno);
		return -1;
	    }
	    ret = socket_descriptors[s]->syscall(SYS_sendto, s, buf, len,
						 flags, to, tolen);
/* 
 * fprintf(stderr, "send to (%d) ret %d %d %s\n", s, ret, errno,
 * inet_ntoa(((struct  sockaddr_in *)to)->sin_addr)); */
	    if (ret >= 0 || errno == EINTR ||
		(socket_descriptors[s]->flags & O_NDELAY) ||
		(errno != ENOBUFS && errno != EWOULDBLOCK))
		return ret;

	    IO_YIELD(s, PENDING_SENDTO);
	} while (1);
    } else {
	return sendto(s, buf, len, flags, to, tolen);
    }
}

/*--------------------------  SYS_writev -------------------------------*/
int
cth_writev(int s, struct iovec *iov, int iovcnt)
{
    int res;

    if (global_socket_list && FD_ISSET(s, global_socket_list)) {

	do {
	    if (sched_io_has_error(s)) {
		fprintf(stderr, "writev error %d\n", errno);
		return -1;
	    }
	    res = socket_descriptors[s]->syscall(SYS_writev, s, iov, iovcnt);

	    if (res >= 0 || errno == EINTR ||
		(socket_descriptors[s]->flags & O_NDELAY) ||
		(errno != ENOBUFS && errno != EWOULDBLOCK))
		return res;
	    IO_YIELD(s, PENDING_SEND);
	} while (1);
    } else {
	return writev(s, iov, iovcnt);
    }
}





int
default_syscall(int func,...)
{
    va_list args;

#ifdef STDC_HEADERS
    va_start(args, func);
#else
    va_start(args);
#endif
    switch (func) {
    case SYS_close:
#ifdef HAVE_WINDOWS_H
	return closesocket(va_arg(args, int));
#else
	return close(va_arg(args, int));
#endif
	break;
    case SYS_dup2:
	{
	    int s1;
	    int s2;
	    s1 = va_arg(args, int);
	    s2 = va_arg(args, int);
	    return dup2(s1, s2);
	}
	break;
    case SYS_dup:
	return dup(va_arg(args, int));
	break;
    case SYS_fcntl:
	{
	    int s;
	    int cmd;
	    int arg;
	    s = va_arg(args, int);
	    cmd = va_arg(args, int);
	    arg = va_arg(args, int);
#ifdef HAVE_WINDOWS_H
	    /* 
	     *  ioctlsocket isn't really a substitute for fcntl, 
	     *  but it's what we have on NT.  We'll drop this in tentatively
	     *  to make the compiler happy.  Having it work is another matter.
	     */
	    {
		unsigned int *NT_arg = &arg;
		return ioctlsocket(s, cmd, NT_arg);
	    }
#else
	    return fcntl(s, cmd, arg);
#endif
	}
	break;
    case SYS_accept:
	{
	    int s;
	    void *addr;
	    int *addrlen;
	    s = va_arg(args, int);
	    addr = va_arg(args, void *);
	    addrlen = va_arg(args, int *);
	    return accept(s, addr, addrlen);
	}
	break;
    case SYS_connect:
	{
	    int s;
	    void *addr;
	    int addrlen;
	    s = va_arg(args, int);
	    addr = va_arg(args, void *);
	    addrlen = va_arg(args, int);
	    return connect(s, addr, addrlen);
	}
	break;
    case SYS_bind:
	{
	    int s;
	    void *addr;
	    int addrlen;
	    s = va_arg(args, int);
	    addr = va_arg(args, void *);
	    addrlen = va_arg(args, int);
	    return bind(s, addr, addrlen);
	}
	break;
    case SYS_listen:
	{
	    int s;
	    int backlog;
	    s = va_arg(args, int);
	    backlog = va_arg(args, int);
	    return listen(s, backlog);
	}
	break;

    case SYS_recv:
	{
	    int s;
	    char *buf;
	    int len;
	    int flags;
	    s = va_arg(args, int);
	    buf = va_arg(args, char *);
	    len = va_arg(args, int);
	    flags = va_arg(args, int);
	    return recv(s, buf, len, flags);
	}
	break;
    case SYS_recvfrom:
	{
	    int s;
	    char *buf;
	    int len;
	    int flags;
	    void *from;
	    int *from_len;
	    s = va_arg(args, int);
	    buf = va_arg(args, char *);
	    len = va_arg(args, int);
	    flags = va_arg(args, int);
	    from = va_arg(args, void *);
	    from_len = va_arg(args, int *);
	    return recvfrom(s, buf, len, flags, from, from_len);
	}
	break;
    case SYS_readv:
	{
	    int s;
	    void *iov;
	    int iovcnt;
	    s = va_arg(args, int);
	    iov = va_arg(args, void *);
	    iovcnt = va_arg(args, int);
	    return readv(s, iov, iovcnt);
	}
	break;

    case SYS_send:
	{
	    int s;
	    char *buf;
	    int len;
	    int flags;
	    s = va_arg(args, int);
	    buf = va_arg(args, char *);
	    len = va_arg(args, int);
	    flags = va_arg(args, int);
	    return send(s, buf, len, flags);
	}
	break;
    case SYS_sendto:
	{
	    int s;
	    char *buf;
	    int len;
	    int flags;
	    void *to;
	    int to_len;
	    s = va_arg(args, int);
	    buf = va_arg(args, char *);
	    len = va_arg(args, int);
	    flags = va_arg(args, int);
	    to = va_arg(args, void *);
	    to_len = va_arg(args, int);
	    return sendto(s, buf, len, flags, to, to_len);
	}
	break;
    case SYS_writev:
	{
	    int s;
	    void *iov;
	    int iovcnt;
	    s = va_arg(args, int);
	    iov = va_arg(args, void *);
	    iovcnt = va_arg(args, int);
	    return writev(s, iov, iovcnt);
	}
	break;
    default:
	fprintf(stderr, "Unknown function %d\n", func);
	return -1;
    }
}

#ifdef HAVE_WINDOWS_H
int
writev(int fd, struct iovec *iov, int iovcnt)
{
    /* write a set (iovcnt) of buffers in iovec     */
    int total_write = 0;
    int i;

    for (i = 0; i < iovcnt; i++) {
	total_write += send(fd, iov[i].iov_base, iov[i].iov_len, 0);
    }

    return total_write;
}

int
readv(int fd, struct iovec *iov, int iovcnt)
{
    int total_read = 0;
    int i;

    for (i = 0; i < iovcnt; i++) {
	total_read += recv(fd, iov[i].iov_base, iov[i].iov_len, 0);
    }

    return total_read;

}
#endif
