/*
 * idebug.c
 *	Internal debugging routines.
 */

/* Standard Includes */
#include "config.h"
#include <stdio.h>
#include "assert.h"
#ifdef HAVE_STRING_H
#include <string.h>
#else
#include <strings.h>
#endif
#include <malloc.h>

#include "general.h"
#include "lock.h"
#include "queue.h"
#include "sync.h"
#include "internal.h"

#ifndef STDC_HEADERS
extern void *bsearch ARGS((const void *key, const void *base, size_t nel, 
			   size_t size, 
			   int(*func)(const void *, const void *)));
extern int qsort ARGS((char *base, int nel, int size, int(*func)()));
#endif


char **debug_string_table = NULL;
exported shared int debug_table_size = 0;
extern short threads_started;


int some_debugging = 1;

static
int compar(p1, p2)
const void* p1;
const void* p2;
{
    char **s1 = (char **)p1;
    char **s2 = (char **)p2;
    return strcmp(*s1, *s2);
}

int debug_query(name)
char *name;
{
	char **entry;
	entry = (char **) bsearch((char *)&name, (char *) debug_string_table, 
				  debug_table_size, sizeof(char *), 
				  compar);
	return (entry != 0);
}

void
add_to_debug_string_table(str)
char *str;
{
    /* functions performed prior to cthread_init */
    assert(threads_started == 0);

    if (debug_table_size == 0) {
	debug_string_table = (char **) malloc(sizeof(char *));
    } else {
	debug_string_table = 
	  (char **) realloc((char *)debug_string_table,
			    sizeof(char *) * (debug_table_size + 1));
    }

    debug_string_table[debug_table_size++] = strdup(str);
    qsort((char *) debug_string_table, debug_table_size, sizeof(char *),
	  compar);
}

static void
fprint_obj(f, ptr, name)
FILE *f;
void *ptr;
char *name;
{
    if (name == NULL) {
	name = "";
    }
    fprintf(f, "\"%s\" (0x%lx)", name, (long) ptr);
}

void
fprint_mutex(f, ptr)
FILE *f;
mutex_t ptr;
{
    if (ptr == NULL) {
	fprintf(f, "\"\", (invalid pointer)");
    } else {
	fprint_obj(f, (void *) ptr, ptr->name);
    }
}

void
fprint_condition(f, ptr)
FILE *f;
condition_t ptr;
{
    if (ptr == NULL) {
	fprintf(f, "\"\", (invalid pointer)");
    } else {
	fprint_obj(f, (void *) ptr, ptr->name);
    }
}

void
fprint_thread(f, ptr)
FILE *f;
cthread_t ptr;
{
    if (ptr == NULL) {
	fprintf(f, "\"\", (invalid pointer)");
    } else {
	fprint_obj(f, (void *) ptr, ptr->name);
    }
}

void fprint_pos(f, line, file)
FILE *f;
int line;
char *file;
{
    fprintf(f, "%s(%d)", file, line);
}

void print_mutex (m)
mutex_t m;
{
    fprint_mutex(stdout, m);
}

void print_condition (c)
condition_t c;
{
    fprint_condition(stdout, c);
}

void print_thread (t)
cthread_t t;
{
    fprint_thread(stdout, t);
}

void print_pos (line, file)
int line;
char *file;
{
    fprint_pos(stdout, line, file);
}

