#include <sys/types.h>
#include <signal.h>
#include <malloc.h>

#include "general.h"
#include "sched_utils.h"
#include "config.h"
#include "init.h"
#include "arch_init.h"

struct sigvec sig_handlers[MAX_PROC][NO_OF_SIGNALS];
static struct sigvec  newvec, oldvec;
static struct sigvec  *old_sigpipe_hndlr, *old_sigalrm_hndlr;

extern int sigstack ARGS((struct sigstack *ss, struct sigstack *oss));
extern int sigvec  ARGS((int sig, struct sigvec *vec, struct sigvec *ovec));

/* unix externs */
extern void perror();

void
setup_child_sighandlers(ill_hndlr, seg_hndlr, bus_hndlr)
HANDLERRETTYPE (*ill_hndlr)();
HANDLERRETTYPE (*seg_hndlr)();
HANDLERRETTYPE (*bus_hndlr)();
{
   struct sigstack	sigst;

   if (allow_core_dumps) return;

   sigst.ss_sp = (char *)malloc(10240) + 10240;
   sigst.ss_onstack = 0;
   if (sigstack(&sigst, NULL) != 0) {
       perror("sigstack");
   }
   if (ill_hndlr) {
       newvec.sv_handler = ill_hndlr;
       newvec.sv_mask = sigmask( SIGILL );
       newvec.sv_flags = SV_ONSTACK;
       if (sigvec( SIGILL, &newvec, &oldvec ) != 0) perror("sigvec");
   }
   if (seg_hndlr) {
       newvec.sv_handler = seg_hndlr;
       newvec.sv_mask = sigmask( SIGSEGV );
       newvec.sv_flags = SV_ONSTACK;
       if (sigvec( SIGSEGV, &newvec, &oldvec ) != 0) perror("sigvec"); 
   }
   if (bus_hndlr) {
       newvec.sv_handler = bus_hndlr;
       newvec.sv_mask = sigmask( SIGBUS );
       newvec.sv_flags = SV_ONSTACK;
       if (sigvec( SIGBUS, &newvec, &oldvec ) != 0) perror("sigvec"); 
   }
}

void  setup_term_sighandler(term_hndlr)
HANDLERRETTYPE(*term_hndlr)();
{
   if (term_hndlr) {
       newvec.sv_handler = term_hndlr;
       newvec.sv_mask = sigmask( SIGTERM );
       newvec.sv_flags = 0;
       if (sigvec( SIGTERM, &newvec, &oldvec ) != 0) perror("sigvec"); 
   }
 }

void  setup_chld_sighandler(chld_hndlr)
HANDLERRETTYPE(*chld_hndlr)();
{
   if (chld_hndlr) {
       newvec.sv_handler = chld_hndlr;
       newvec.sv_mask = sigmask( SIGCHLD );
       newvec.sv_flags = 0;
       if (sigvec( SIGCHLD, &newvec, &oldvec ) != 0) perror("sigvec"); 
   }
 }

void
setup_scheduler_sighandlers()
{
    int sig_no;
    int node = virtual_processor();

    /* install the signal handlers */
    for (sig_no = 0; sig_no < NO_OF_SIGNALS; sig_no++) {
	if (sig_handlers[node][sig_no].sv_handler != 0) {
	    DBG1("Overwriting handler for signal %d\n", sig_no);
	    if (sigvec( sig_no, &sig_handlers[node][sig_no], &oldvec) == -1) {
		perror("Signal not installed");
	    }
	}
    }
}

void
init_sighandler_struct()
{
    int i;
    int j;
    for (i = 0; i < MAX_PROC; i++) {
	for (j = 0; j < NO_OF_SIGNALS; j++) {
	    sig_handlers[i][j].sv_flags = 0;
	}
    }
    for (j = 0; j < NO_OF_SIGNALS; j++) {
	sig_handlers[0][j].sv_mask = sigmask(j);
    }
    for (i = 1; i < MAX_PROC; i++) {
	for (j = 0; j < NO_OF_SIGNALS; j++) {
	    sig_handlers[i][j].sv_mask = sig_handlers[0][j].sv_mask;
	}
    }
}

void
fill_sighandler_struct(node, alert_hndlrs)
int node;
HANDLERRETTYPE (*alert_hndlrs[NO_OF_SIGNALS])();
{
    int j;
    for (j = 0; j < NO_OF_SIGNALS; j++) {
	sig_handlers[node][j].sv_handler = alert_hndlrs[j];
    }
}
    

void
setup_master_sighandlers(intr_hndlr, quit_hndlr)
HANDLERRETTYPE (*intr_hndlr)();
HANDLERRETTYPE (*quit_hndlr)();
{
    if (intr_hndlr) {
	newvec.sv_handler = intr_hndlr;
	newvec.sv_mask = sigmask( SIGINT );
	newvec.sv_flags = 0;
	sigvec( SIGINT, &newvec, &oldvec );

        /* add handlers for all the signals that might crash the
         * master and leave the children hanging around
         */
	newvec.sv_mask = 0;
	sigvec( SIGEMT, &newvec, &oldvec );
	sigvec( SIGFPE, &newvec, &oldvec );
	sigvec( SIGHUP, &newvec, &oldvec );
	sigvec( SIGIOT, &newvec, &oldvec );
	sigvec( SIGPIPE, &newvec, &oldvec );
        if( oldvec.sv_handler != SIG_IGN && 
             oldvec.sv_handler != SIG_DFL) {
         	sigvec( SIGPIPE, &oldvec, (struct sigvec *)0);
	}
	sigvec( SIGEMT, &newvec, &oldvec );
	sigvec( SIGUSR1, &newvec, &oldvec );
	sigvec( SIGUSR2, &newvec, &oldvec );
	sigvec( SIGXCPU, &newvec, &oldvec );
	sigvec( SIGXFSZ, &newvec, &oldvec );
    }
    if (quit_hndlr) {
	newvec.sv_handler = quit_hndlr;
	newvec.sv_mask = sigmask( SIGQUIT );
	newvec.sv_flags = 0;
	sigvec( SIGQUIT, &newvec, &oldvec );
    }

}



void
setup_sigpipe_handler(hndlr)
HANDLERRETTYPE (*hndlr)();

{
    if (hndlr != NULL) {
	newvec.sv_handler = hndlr;
	newvec.sv_mask = sigmask( SIGPIPE);
	newvec.sv_flags = 0;
	sigvec( SIGPIPE, &newvec, old_sigpipe_hndlr);

      }
   else 
   if( old_sigalrm_hndlr != NULL) {
	 newvec.sv_handler = (HANDLERRETTYPE (*)())old_sigpipe_hndlr;
	 newvec.sv_mask = sigmask( SIGPIPE);
	 newvec.sv_flags = 0;
	 sigvec( SIGPIPE, &newvec, NULL);
   }
  }


void
setup_sigalrm_handler(hndlr)
HANDLERRETTYPE (*hndlr)();

{
    if (hndlr != NULL) {
	newvec.sv_handler = hndlr;
	newvec.sv_mask = sigmask( SIGALRM);
	newvec.sv_flags = 0;
	sigvec( SIGALRM, &newvec, old_sigalrm_hndlr);

      }
   else 
   if( old_sigalrm_hndlr != NULL) {
	 newvec.sv_handler = (HANDLERRETTYPE (*)())old_sigalrm_hndlr;
	 newvec.sv_mask = sigmask( SIGALRM);
	 newvec.sv_flags = 0;
	 sigvec( SIGALRM, &newvec, NULL);
   }
  }

