#include <stdio.h>
#include <string.h>
#include <sys/types.h>

#include "general.h"
#include "lock.h"
#include "queue.h"
#include "sync.h"
#include "internal.h"
#include "idebug.h"
#include "threads.h"
#include "sched_utils.h"
#include "init.h"
#ifdef WITH_MONITORING
#include "monitor.h"
#endif
#include "mon.sensorextern.h"
#include "memalloc.h"

extern void child_exit ARGS((int));
extern cthread_t current_thread_array[MAX_PROC];
#include "arch_init.h"

/* cthreads functions used but not defined here */
extern void internal_cthread_yield();

extern RESULT app_mutex_alloc(mptr, node, name)
mutex_t *mptr;
int node;
char *name;
{
    RESULT res;

    res = internal_mutex_alloc(mptr, node);
    if (res != T_SUCCEED) {
	return res;
    } else {
	if (name[0] == '&') name++;       /* skip beginning '&' */
	mutex_set_name(*mptr, name);
	if (((*mptr)->name == NULL) && (name != NULL)) {
	    /* name alloc failed, fail whole alloc */
	    memory_free((memory_t) *mptr);
	    return T_NOMEMORY;
	}
	DM_Sensor_mutex_alloc(*mptr, (*mptr)->name);
	return res;
    }
}

void
app_mutex_lock(m, file, line)
mutex_t m;
char	*file;
int	line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];
    DM_Sensor_mutex_begin_lock(m, m->sequence_number);

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" attempting lock on mutex "); print_mutex(m);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    internal_mutex_lock(&(m->mlock));
    m->sequence_number++;
    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" granted lock on mutex "); print_mutex(m);
	printf(" at "); print_pos(line, file); printf("\n");
    }

    DM_Sensor_mutex_end_lock(m, m->sequence_number);
}

void
app_mutex_unlock(m, file, line)
mutex_t m;
char	*file;
int	line;
{
    int seq = m->sequence_number;  /* sequence number is only valid 
				      *before* the unlock */
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" unlocking mutex "); print_mutex(m);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    internal_mutex_unlock(&(m->mlock));

    DM_Sensor_mutex_unlock(m, seq);
}

void
app_mutex_clear(m, file, line)
mutex_t m;
char	*file;
int	line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" clearing mutex "); print_mutex(m);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    internal_mutex_clear(m);

    DM_Sensor_mutex_clear(m);
}

void
app_mutex_init(m, file, line)
mutex_t m;
char	*file;
int	line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" initializing mutex "); print_mutex(m);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    internal_mutex_init(m);

    DM_Sensor_mutex_init(m);
}

void
app_mutex_free(m, file, line)
mutex_t m;
char	*file;
int	line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" freeing mutex "); print_mutex(m);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    internal_mutex_free(m);

    DM_Sensor_mutex_free(m);
}

extern RESULT app_condition_alloc(cptr, node, name)
condition_t *cptr;
int node;
char *name;
{
    RESULT res;

    res = internal_condition_alloc(cptr, node); 
  
    if (res != T_SUCCEED) {
	return res;
    } else {
	if (name[0] == '&') name++;       /* skip beginning '&' */
	condition_set_name(*cptr, name);
	if (((*cptr)->name == NULL) && (name != NULL)) {
	    /* name alloc failed, fail whole alloc */
	    memory_free((memory_t) *cptr);
	    return T_NOMEMORY;
	}
	DM_Sensor_condition_alloc(*cptr, (*cptr)->name);
	return res;
    }
}

void 
app_condition_wait(c, m, file, line)
condition_t c;
mutex_t m;
char *file;
int line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];
    int our_sequence_num;

    queue_self_on_condition(c);
    /* at this point condition sequence number is valid and condition
    ** is locked 
    */
    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" waiting for condition "); print_condition(c);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    our_sequence_num = c->sequence_number;
    DM_Sensor_condition_begin_wait(c, our_sequence_num, m);

    /* this next call waits for the condition */
    unlock_and_yield(c, &(m->mlock), m);
    DM_Sensor_condition_end_wait(c, our_sequence_num, m);
    /* execution continues here after condition is signalled */
    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" continuing from condition "); print_condition(c);
	printf(" at "); print_pos(line, file); printf("\n");
    }
}    

RESULT
app_condition_signal(c, file, line)
condition_t c;
char *file;
int line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];
    RESULT res;
    unsigned int waiter_seq_woken;

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" signalling on condition "); print_condition(c);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    res = internal_condition_signal(c, &waiter_seq_woken);
    DM_Sensor_condition_signal(c, waiter_seq_woken);

    return res;
}    

void 
app_condition_broadcast(c, file, line)
condition_t c;
char *file;
int line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];
    int begin_cond_seq_num, end_cond_seq_num;
    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" broadcasting condition "); print_condition(c);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    internal_condition_broadcast(c, &begin_cond_seq_num, &end_cond_seq_num);
    DM_Sensor_condition_broadcast(c, begin_cond_seq_num, end_cond_seq_num);
}    

void 
app_condition_init(c, file, line)
condition_t c;
char *file;
int line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" initing condition "); print_condition(c);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    internal_condition_init(c);
    DM_Sensor_condition_init(c);
}    

void 
app_condition_clear(c, file, line)
condition_t c;
char *file;
int line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" clearing condition "); print_condition(c);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    internal_condition_clear(c);
    DM_Sensor_condition_clear(c);
}    

void 
app_condition_free(c, file, line)
condition_t c;
char *file;
int line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" freeing condition "); print_condition(c);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    internal_condition_free(c);
    DM_Sensor_condition_free(c);
}    

void
app_cthread_yield(file, line)
char *file;
int line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" yielding at "); print_pos(line, file); printf("\n");
    }
    DM_Sensor_thread_yield();
    internal_cthread_yield();
}

void
app_cthread_exit(res, file, line)
any_t res;
char *file;
int line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" explicit exit at "); print_pos(line, file); printf("\n");
    }
    internal_cthread_exit(res);
}

void 
app_cthread_exit_notification()
{
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	if(current_thread->status & S_DETACHED) {
	    printf(" terminating.  Thread returns to free pool.\n");
	} else {
	    printf(" terminating.  Thread awaits join.\n");
	}
    }
}

RESULT
app_cthread_detach(t, file, line)
cthread_t t;
char *file;
int line;
{
    RESULT res;
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" detaching thread "); print_thread(t);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    if (t == NULL) {
	fprintf(stderr, "Cthread detach called with NULL thread at line %d in file %s.\nExiting....\n", line, file);
	child_exit(1);
    }
    res = internal_cthread_detach(t);
    DM_Sensor_thread_detach(t);
    return res;
}

RESULT
app_cthread_join(t, resptr, file, line)
cthread_t t;
any_t *resptr;
char *file;
int line;
{
    RESULT res;
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" waiting to join thread "); print_thread(t);
	printf(" at "); print_pos(line, file); printf("\n");
    }
    DM_Sensor_thread_begin_join(t);
    res = internal_cthread_join(t, resptr);
    DM_Sensor_thread_end_join(t);
    return res;
}

void
app_cthread_publish(p, name, file, line)
any_t *p;
char *name;
char *file;
int line;
{
    cthread_t current_thread = current_thread_array[virtual_processor()];

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" publishing value %s (%lx)", name, (long)p); 
	printf(" at "); print_pos(line, file); printf("\n");
    }
    internal_cthread_publish(p);
}


cthread_t
app_cthread_fork(func, arg, node, name)
any_t	(*func)();
any_t	arg;
int	node;
char	*name;
{
    cthread_t  newthread;
    cthread_t current_thread = current_thread_array[virtual_processor()];

    
    if (name[0] == '(') {  /* eliminate initial cast from default name */
	name = strrchr(name, ')') + 1;
    }

    if ((debug_table_size > 0) && debug_query("calls")) {
	printf("Thread "); print_thread(current_thread);
	printf(" forking thread %s to node %d\n", name, node); 
    }
    newthread = (cthread_t)cthread_thread_alloc(func, arg, node);

    if (newthread != NULL) {
	newthread->name = name;
	DM_Sensor_thread_fork(newthread->name, newthread, 
			      newthread->fork_count,
			      node);
	cthread_thread_schedule(newthread);
    } else {
	DBG("Thread_alloc returns NULL");
    }
    
   return(newthread);
}

void
app_barrier(file, line)
char *file;
int line;
{
    if ((debug_table_size > 0) && debug_query("calls")) {
	cthread_t current_thread = current_thread_array[virtual_processor()];

	printf("Thread "); print_thread(current_thread);
	printf(" blocking on barrier at");
	print_pos(line, file); printf("\n");
    }
    internal_barrier();
}
