%{/* file: interface.y  */

#include "config.h"
#include "unix_defs.h"
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "code_generation.h"
#include "assert.h"

#define UNDEFINED 2
  /* the definition above has to be different than TRUE and FALSE*/

typedef struct {
  TypeCode      tcode;
  Complex_Type *parameters;
} Typeinfo;

typedef struct {
  int count;
  int *list;
} Number_list;

void check_malloc_result(void *mem);
void adjust_filename(char *name);
void dump(Interface_list_cell *);
char *create_string_from_yytext();
void insertSubtype (Type_Description *t, Type_Description *subt);
char set_has_variable_size(Type_Description *type);
void build_type_data_for_pre_defined_types(Type_Description ***typelist_array,
					   int *typelist_size);

Interface_list_cell    *itfc_list_head, *itfc_list_tail;
Type_Description       **typelist_array;
int                   typelist_size;
char                  idlfilename[255];
char                  has_INOUT_arguments;
char                  has_OUT_arguments;
char                  has_IN_arguments;

/* The following definitions are related to generating list of
** subtypes (which will be used in the invocation code for dealing
** with nested types).
**                    (dilma, Dec 3 1997)
*/
 
typedef enum {
  typedef_info,
  subtype_info
} Pending_kind;

typedef struct {
  Type_Description *type;
  char             *pending_name;
  Pending_kind     pending_kind;
} Undefined_Subtype;

struct {
  int size;                     /* current size of the list */
  int max;                      /* space allocated */
  Undefined_Subtype *array;     /* the list */
} list_undef;

void insertUndefined (Type_Description *t, char *subtype, Pending_kind kind);
void resolveUndefined ();

extern void yyerror(char*);
extern int yylex(void);

%}

%union{
  char		        nothing;
  char 	        *name;
  int    	        number;
  Number_list		*number_list;
  Interface             *interface;
  Interface_list_cell   *interface_list;
  Attribute             *attribute;
  Attribute_list_cell   *attribute_list;
  Complex_Type          *complex_type;
  TypeCode              typecode;
  Typeinfo              typeinfo;
  Operation_list_cell   *operation_list;
  Operation             *operation;
  Argument              *argument;
  Argument_list_cell    *argument_list;
  Argument_direction    direction;
  Context_Info          *context;
  char                  readonly;
  char                  mirror;
  Type_Description      *type_description;
  type_struct           typestruct;
}   

/* tokens */

%token TYPECODE
%token TYPENAME
%token BASETYPENAME
%token BASETYPECODE 
%token ATTRIBUTE
%token INTERFACE
%token MAXLEN
%token NDIMS
%token ID
%token DIGITS
%token NT_ENUM
%token NT_PRE_DEFINED
%token NT_STRUCT
%token NT_STRING
%token NT_SEQUENCE
%token NT_ARRAY
%token NT_INTERFACE
%token OPERATION
%token ARGUMENT
%token IN
%token OUT
%token INOUT
%token READONLY
%token MIRROR
%token CONTEXT
%token TYPEINFO
%token TYPEDEF_OF
%token NB_FIELDS
%token FIELDTYPE
%token FIELDNAME

/* types for non-terminals */

%type	<nothing>	    _description;
%type	<name>		    _name;
%type	<number>	    _number;
%type   <interface>         _interface;
%type	<name>		    _maybe_mirror;
%type   <interface_list>    _interface_list;
%type   <attribute>         _attribute_def;
%type   <attribute_list>    _attribute_list;
%type   <typeinfo>          _typeinfo;
%type   <typeinfo>          _onetype;
%type   <typecode>          _basetcode;
%type   <operation_list>    _operation_list;
%type   <operation>         _operation;
%type   <argument>          _argument;
%type   <argument_list>     _argument_list;
%type   <number_list>       _number_list;
%type   <direction>         _direction;
%type   <readonly>          _readonly;
%type   <mirror>            _mirror;
%type   <context>           _context;
%type   <context>           _context_list;
%type   <nothing>           _typelist;
%type   <type_description>  _typedesc;
%type   <typestruct>        _field_list;
%%

/* nothing */
_description :
		{
		  /* initializating list of interfaces */
		  itfc_list_head = NULL;
		  itfc_list_tail = NULL;
		  typelist_size = 0;
		  typelist_array = NULL;
		  has_OUT_arguments = FALSE;
		  has_INOUT_arguments = FALSE;
		  has_IN_arguments = FALSE;
		  list_undef.max = 100;    /* arbitrary */
		  list_undef.size = 0;
		  list_undef.array = (Undefined_Subtype*) 
		    malloc(sizeof(Undefined_Subtype)*list_undef.max);
		  assert(list_undef.array != NULL);
		  build_type_data_for_pre_defined_types(&typelist_array,
							&typelist_size);
		}

                _interface_list _typelist {
		}
	        ;

/* interface_list */
_interface_list:
                {
		  /*empty, it does not make much sense, but it is useful
		    for testing files from idl_specs */
		}
        |
	_interface_list _interface
                {
		  /* add interface to the end of the interface list
		     being kept */  
		  Interface_list_cell *cell = (Interface_list_cell *)
		       malloc(sizeof(Interface_list_cell));
		  check_malloc_result((void*) cell);
		  cell->itfc = $2;
		  cell->next = NULL;

		  /* The following if statement is superfluous, since we
		     know the way yacc/bison work, and that when this
		     pattern is recognized the itfc_list *is* empty */
		  if (itfc_list_head == NULL) {
		    /* list empty */
		    itfc_list_head = itfc_list_tail = cell;
		  }
		  else {
		    /* update tail cell and tail pointer */
		    itfc_list_tail->next = cell;
		    itfc_list_tail       = cell;
		  }
		}
	;

/* interface */
_interface:
	INTERFACE _name _maybe_mirror _attribute_list _operation_list
        {
	  Interface *itfc = (Interface *) malloc (sizeof(Interface));

	  check_malloc_result((void*) itfc);

	  itfc->name = $2;
	  itfc->att_list = $4;
	  itfc->op_list = $5;
	  itfc->mirror_name = $3;
	  $$ = itfc;
	}
	;


_maybe_mirror:
	/* empty */
        {
	    $$ = NULL;
	} |
	MIRROR _name
	{
	    $$ = $2;
	}
	;

/* attribute_list */
_attribute_list :
	/* empty */
	{
	  $$ = NULL;
	}
	|
	_attribute_def _attribute_list 
        {
	  Attribute_list_cell *cell = (Attribute_list_cell *)
	      malloc(sizeof(Attribute_list_cell));
	  check_malloc_result((void*) cell);
	  cell->att  = $1;
	  cell->next = $2;
	  $$ = cell;
	}
	;

/* attribute_def */
_attribute_def:
	ATTRIBUTE _name  TYPENAME _mirror _name _typeinfo _readonly
        {
	  Attribute *att = (Attribute *) malloc(sizeof(Attribute));
	  Typeinfo         tinfo = $6;
	  check_malloc_result((void*) att);

	  att->name        = $2;
	  att->typename    = $5;
	  att->typecode    = tinfo.tcode;
	  att->parameters  = tinfo.parameters;
	  att->readonly    = $7;
	  att->mirror      = $4;
	  $$ = att;
	}
	;

/* readonly */
_readonly:
        {
	  /* empty */
	  $$ = 0;
	}
        |
	READONLY
        {
	  $$ = 1;
	}
        ;

/* mirror */
_mirror:
        {
	  /* empty */
	  $$ = 0;
	}
        |
	MIRROR
        {
	  $$ = 1;
	}
        ;

/* operation_list */
_operation_list :
	/* empty */
	{
	  $$ = NULL;
	}
	|
	_operation _operation_list 
        {
	  Operation_list_cell *cell = (Operation_list_cell *)
	      malloc(sizeof(Attribute_list_cell));
	  check_malloc_result((void*) cell);
	  cell->op   = $1;
	  cell->next = $2;
	  $$ = cell;
	}
	;

/* attribute_def */
_operation:
	OPERATION _name  TYPENAME _mirror _name _typeinfo _argument_list _context
        {
	  Operation *op = (Operation *) malloc(sizeof(Operation));
	  Typeinfo         tinfo = $6;
	  check_malloc_result((void*) op);

	  op->name                = $2;
	  op->rettypename         = $5;
	  op->rettypecode         = tinfo.tcode;
	  op->rettype_parameters  = tinfo.parameters;
	  op->arg_list            = $7;
	  op->context             = $8;
	  op->mirror              = $4;
	  op->hasOUTarguments     = has_OUT_arguments;
	  op->hasINOUTarguments   = has_INOUT_arguments;
	  op->hasINarguments      = has_IN_arguments;
	  /* getting flags ready for next operation */
	  has_OUT_arguments       = FALSE;
	  has_INOUT_arguments     = FALSE;
	  has_IN_arguments        = FALSE;
	  $$ = op;
	}
	;

/*context */
_context: 
        {
	  /* empty */
	  $$ = NULL;
	}
        |
	CONTEXT _context_list
        {
	  $$ = $2;
	}
        ;

/* context */
_context_list:
	_name 
        {
	  Context_Info *c = (Context_Info*) malloc(sizeof(Context_Info));
	  c->nb_elems = 1;
	  c->list = (char **) malloc(sizeof(char *));
	  c->list[0] = $1;
	  $$ = c;
	}
        |
        _context_list  _name {
	  Context_Info *c = $1;
	  c->nb_elems++;
	  c->list = (char **) realloc(c->list, c->nb_elems*sizeof(char*));
	  c->list[c->nb_elems-1] = $2;
	  $$ = c;
	  }
        ;

	  

/* argument_list */
_argument_list:
	/* empty */
        {
	  $$ = NULL;
	}
        |
        _argument _argument_list
        {
	  /* argument has to be added to the beginning of _argument_list */
	  Argument_list_cell *cell = (Argument_list_cell *) malloc(
					    sizeof(Argument_list_cell));
	  Argument_list_cell *current_list = $2;
	  
	  check_malloc_result((void*) cell);
	  cell->arg = $1;
	  cell->next = NULL;
	  if (current_list == NULL)
	    $$ = cell;
	  else {
	    cell->next = current_list;
	    $$ = cell;
	  }
	}
        ;

/* argument */
_argument:
 	ARGUMENT _name _direction TYPENAME _name _typeinfo
        {
	  Argument *a = (Argument *) malloc(sizeof(Argument));
	  Typeinfo  tinfo = $6;
	  check_malloc_result((void*) a);

	  a->name = $2;
	  a->direction = $3;
	  a->typename = $5;
	  a->typecode = tinfo.tcode;
	  a->type_parameters = tinfo.parameters;
	  $$ = a;
	}
        ;

/* direction */
_direction:
	IN
        {
	  has_IN_arguments = TRUE;
	  $$ = D_IN;
	}
        |
	OUT
        {
	  has_OUT_arguments = TRUE;
	  $$ = D_OUT;
	}
        |
	INOUT
        {
	  has_INOUT_arguments = TRUE;
	  $$ = D_INOUT;
	}
        ;

/* typeinfo */
_typeinfo:
	TYPECODE _onetype
        {
	  $$ = $2;
	}
	;

/* argument_list */
_number_list:
	/* empty */
        {
	    Number_list *list = malloc(sizeof(Number_list));
	    list->count = 0;
	    list->list = malloc(sizeof(int));
	    $$ = list;
	}
        |
        _number_list _number 
        {
	  /* number has to be added to the end of _number_list */
	  Number_list *list = $1;
	  list->list = realloc(list->list, sizeof(int) * (list->count+1));
	  list->list[list->count] = $2;
	  list->count++;
	  $$ = list;
	}
        ;

/* typeinfo */
_onetype:
	NT_ENUM
        {
	  Typeinfo t;
	  t.tcode      = NT_enum;
	  t.parameters = NULL;
	  $$ = t;
	}
	|
	NT_INTERFACE
        {
	  Typeinfo t;
	  t.tcode      = NT_interface;
	  t.parameters = NULL;
	  $$ = t;
	}
	|
	NT_PRE_DEFINED
        {
	  Typeinfo t;
	  t.tcode      = NT_pre_defined;
	  t.parameters = NULL;
	  $$ = t;
	}
	|
	NT_STRUCT
        {
	  Typeinfo t;
	  t.tcode      = NT_struct;
	  t.parameters = NULL;
	  $$ = t;
	}
	|
	NT_STRING MAXLEN _number
        {
	  Typeinfo t;
	  t.tcode      = NT_string;
	  t.parameters = (Complex_Type *) malloc(sizeof(Complex_Type));
	  check_malloc_result((void*) t.parameters);
	  t.parameters->tstring.maximum = $3;
	  $$ = t;
	}
	|
	NT_SEQUENCE MAXLEN _number BASETYPENAME _name BASETYPECODE _basetcode
        {
	  Typeinfo t;
	  t.tcode      = NT_sequence;
	  t.parameters = (Complex_Type *) malloc(sizeof(Complex_Type));
	  check_malloc_result((void*) t.parameters);		  
	  t.parameters->tsequence.maximum      = $3;
	  t.parameters->tsequence.basetypename = $5;
	  t.parameters->tsequence.basetypecode = $7;
	  $$ = t;
	  
	}
	|
	NT_ARRAY NDIMS _number_list BASETYPENAME _name BASETYPECODE _basetcode
        {
	  Typeinfo t;
	  Number_list *list = $3;

	  t.tcode      = NT_array;
	  t.parameters = (Complex_Type *) malloc(sizeof(Complex_Type));
	  check_malloc_result((void*) t.parameters);		  
	  t.parameters->tarray.ndims      = list->count;
	  t.parameters->tarray.dims	  = list->list;
	  t.parameters->tarray.basetypename = $5;
	  t.parameters->tarray.basetypecode = $7;
	  $$ = t;
	}
	;

/* typecode */
_basetcode:
	NT_ENUM
        {
	  $$ = NT_enum;
	}
	|
	NT_PRE_DEFINED
        {
	  $$ = NT_pre_defined;
	}
	|
	NT_INTERFACE
        {
	  $$ = NT_interface;
	}
	|
	NT_STRUCT
        {
	  $$ = NT_struct;
	}
	|
	NT_STRING
        {
	  $$ = NT_string;
	}
	|
	NT_SEQUENCE 
        {
	  $$ = NT_sequence;
	  
	}
	|
	NT_ARRAY
        {
	  /* not implemented yet*/
	  $$ = NT_array;
	}
	;

/* nothing */	
_typelist:
	/* empty */
        {
	}
        |
        _typelist TYPEINFO _typedesc {
	  if (typelist_size) {
	    /* list not empty */
	    typelist_array = (Type_Description**)realloc(typelist_array,
						    sizeof(Type_Description*)*(typelist_size+1));
	    assert(typelist_array!=NULL);
	    typelist_array[typelist_size] = $3;
	    typelist_size++;
	  }
	  else {
	    /* first element in the list */
	    typelist_array = (Type_Description**)
	      malloc(sizeof(Type_Description));
	    assert(typelist_array!=NULL);
	    typelist_array[0] = $3;
	    typelist_size++;
	  }
	}
        ;

/* Complex_Type* */
_typedesc:
	_name NT_STRUCT  NB_FIELDS _number  _field_list
        {
	  int k;

	  Type_Description *t =  (Type_Description *) malloc(sizeof(Type_Description));
	  Complex_Type     *c =  (Complex_Type *) malloc(sizeof(Complex_Type));
	  assert(t!=NULL);
	  assert(c!=NULL);
	  t->name = $1;
	  t->typedef_of = NULL;
	  t->code = NT_struct;
	  assert($4 == $5.nb_fields);
	  c->tstruct = $5;
	  t->type = c;
	  t->nb_subtypes = 0;
	  /* building the list of direct subtypes: array_field_type 
	     is traversed */
	  for (k=0; k < c->tstruct.nb_fields; k++) {
	    switch (c->tstruct.array_field_typecode[k]) {
	    case NT_enum:
	    case NT_pre_defined:
	    case NT_string:
	    case NT_interface:
	      /* no subtype associate, nothing to do */
	      break;
	    case NT_sequence:
	    case NT_array:
	    case NT_struct: {
	      /* add type description for this struct field, which is either
	      ** structure or a sequence. */
	      Type_Description *aux;
	      aux = searchTypeDescription(c->tstruct.array_field_type[k],
					  typelist_array, typelist_size);
	      if ( aux!= NULL) {
		if (aux->code != NT_pre_defined)
		  insertSubtype(t, aux);
	      }
	      else {
		/* the definition for the field didn't appear yet */
		/* The information will be updated after the parsing */
		insertUndefined(t, c->tstruct.array_field_type[k],
				subtype_info);  
	      }
	      break;
	    }
	    case NT_not_known: {
	      /* unexpected situation */
	      printf("Unexpected: Struct fields with unknown typecode\n");
	      assert(1==0);
	    }
	    default: 
	      printf("Unexpected val in c->tstruct.array_field_typecode[%d]:%d(%s)\n", 
		     k, c->tstruct.array_field_typecode[k],
		     c->tstruct.array_field_type[k]);
	      assert(1==0);
	    }
	  }
	  $$ = t;
	}
	|
	_name NT_STRING MAXLEN _number 
        {
	  Type_Description *t =  (Type_Description *) malloc(sizeof(Type_Description));
	  Complex_Type     *c =  (Complex_Type *) malloc(sizeof(Complex_Type));
	  assert(t!=NULL);
	  assert(c!=NULL);
	  t->name = $1;
	  t->typedef_of = NULL;
	  t->code = NT_string;
	  c->tstring.maximum = $4;
	  t->type = c;
	  t->list_subtypes = NULL;
	  t->nb_subtypes = 0;
	  $$ = t;
	}
	|
	_name NT_SEQUENCE MAXLEN _number BASETYPENAME _name BASETYPECODE
	_basetcode 
        {
	  Type_Description *t =  (Type_Description *) malloc(sizeof(Type_Description));
	  Complex_Type     *c =  (Complex_Type *) malloc(sizeof(Complex_Type));
	  assert(t!=NULL);
	  assert(c!=NULL);
	  t->name = $1;
	  t->typedef_of = NULL;
	  t->code = NT_sequence;
	  c->tsequence.maximum = $4;
	  c->tsequence.basetypename = $6;
	  c->tsequence.basetypecode = $8;
	  t->type = c;
	  t->nb_subtypes=0;
	  /* information about the sequence base type may have to be included,
	  ** so that the code generation for invocations can deal with nested types.
	  ** This inclusion is unnecessary if the base type is either a
	  ** primitive type or a string.
	  */
	  switch (c->tsequence.basetypecode) {
	    case NT_enum:
	    case NT_pre_defined:
	    case NT_string:
	    case NT_interface:
	      /* no subtype associate, nothing to do */
	      break;
	    case NT_sequence:
	    case NT_array: 
	    case NT_struct: {
	      /* add type description for this type, which is either
	      ** structure or a sequence. */
	      Type_Description *aux;
	      aux = searchTypeDescription(c->tsequence.basetypename,
					  typelist_array, typelist_size);
	      if ( aux!= NULL) {
		insertSubtype(t, aux);
	      }
	      else {
		insertUndefined(t, c->tsequence.basetypename, subtype_info);
	      }
	      break;
	    }
	    case NT_not_known: {
	      /* unexpected situation */
	      printf("Unexpected: sequence base type with unknown typecode\n");
	      assert(1==0);
	    }
	    default: 
	      printf("Unexpected val in c->tsequence.basetypecode:%d(%s)\n", 
		     c->tsequence.basetypecode, c->tsequence.basetypename);
	      assert(1==0);
	    }
	  	    
	    
	  $$ = t;
	}
	|
	_name NT_ARRAY NDIMS _number_list BASETYPENAME _name BASETYPECODE
	_basetcode 
        {
	  Type_Description *t =  (Type_Description *) malloc(sizeof(Type_Description));
	  Complex_Type     *c =  (Complex_Type *) malloc(sizeof(Complex_Type));
	  Number_list *list = $4;
	  assert(t!=NULL);
	  assert(c!=NULL);
	  t->name = $1;
	  t->typedef_of = NULL;
	  t->code = NT_array;
	  c->tarray.ndims      = list->count;
	  c->tarray.dims	  = list->list;
	  c->tarray.basetypename = $6;
	  c->tarray.basetypecode = $8;
	  t->type = c;
	  t->nb_subtypes=0;
	  /* information about the sequence base type may have to be included,
	  ** so that the code generation for invocations can deal with nested types.
	  ** This inclusion is unnecessary if the base type is either a
	  ** primitive type or a string.
	  */
	  switch (c->tarray.basetypecode) {
	    case NT_enum:
	    case NT_pre_defined:
	    case NT_string:
	    case NT_interface:
	      /* no subtype associate, nothing to do */
	      break;
	    case NT_sequence:
	    case NT_array: 
	    case NT_struct: {
	      /* add type description for this type, which is either
	      ** structure or a sequence. */
	      Type_Description *aux;
	      aux = searchTypeDescription(c->tarray.basetypename,
					  typelist_array, typelist_size);
	      if ( aux!= NULL) {
		if (aux->code != NT_pre_defined)
		insertSubtype(t, aux);
	      }
	      else {
		insertUndefined(t, c->tarray.basetypename, subtype_info);
	      }
	      break;
	    }
	    case NT_not_known: {
	      /* unexpected situation */
	      printf("Unexpected: sequence base type with unknown typecode\n");
	      assert(1==0);
	    }
	    default: 
	      printf("Unexpected val in c->tarray.basetypecode:%d(%s)\n", 
		     c->tarray.basetypecode, c->tarray.basetypename);
	      assert(1==0);
	    }
	  	    
	  $$ = t;
	}
        |
	_name NT_ENUM 
        {
	  Type_Description *t = (Type_Description *) malloc(sizeof(Type_Description));
	  assert(t!=NULL);
	  t->name = $1;
	  t->code = NT_enum;
	  t->type = NULL;
	  t->typedef_of = NULL;
	  t->list_subtypes = NULL;
	  t->nb_subtypes = 0;
	  $$ = t;
	}
        |
        _name TYPEDEF_OF _name
        {
	  Type_Description *aux;
	  Type_Description *t =  
	    (Type_Description *) malloc(sizeof(Type_Description));
	  assert(t!=NULL);
	  t->name = $1;
	  insertUndefined(t, $3, typedef_info);
	  t->type = NULL;
	  /* The list of subtypes will be updated in the routine
	  ** resolveUndefined(), where the information about the
	  ** basic type is certainly completely available. This could
	  ** be done here, but then we would have to guarantee that IDLtoC
	  ** always generates type information in the appropriate order.
	  ** It actually does now, but I dont' see any point on having to
	  ** rely on it.
	  */
	  aux = searchTypeDescription($3, typelist_array, typelist_size);
	  if (aux != NULL) {
	    t->typedef_of = aux;
	    t->code = aux->code;
	  }
	  t->list_subtypes = NULL;
	  t->nb_subtypes = 0;
	  $$ = t;
	}
        ;

/* type_struct */
_field_list:
 	FIELDTYPE _name BASETYPECODE _basetcode FIELDNAME _name 
        {
	  type_struct *stru = (type_struct*) malloc(sizeof(type_struct));
	  stru->nb_fields = 1;
	  stru->array_field_name = (char**)malloc(sizeof(char*));
	  stru->array_field_type = (char**)malloc(sizeof(char*));
	  stru->array_field_typecode = (TypeCode*)malloc(sizeof(TypeCode));
	  stru->array_field_type[0] = $2;
	  stru->array_field_typecode[0] = $4;
	  stru->array_field_name[0] = $6;
	  $$ = *stru;
	}
        |
 	 _field_list FIELDTYPE _name BASETYPECODE _basetcode FIELDNAME _name
        {
	  type_struct stru = $1;
	  stru.array_field_name = (char**)realloc(stru.array_field_name,
				       sizeof(char*)*(stru.nb_fields+1));
	  stru.array_field_type = (char**)realloc(stru.array_field_type,
				       sizeof(char*)*(stru.nb_fields+1));
	  stru.array_field_typecode = (TypeCode*)realloc(stru.array_field_typecode,
				       sizeof(TypeCode)*(stru.nb_fields+1));
	  
	  stru.array_field_type[stru.nb_fields] = $3;
	  stru.array_field_typecode[stru.nb_fields] = $5;
	  stru.array_field_name[stru.nb_fields] = $7;
	  stru.nb_fields++;
	  $$ = stru;
	}
        ;
	
/* char* */
_name :
	ID
		{
		  char *name = create_string_from_yytext();
		  $$ = name;
		}
	;

/* integer */
_number :
        DIGITS 
                {	
		  char *name = create_string_from_yytext();
		   $$ = atoi(name);
		}
	;

%%
		
/*
 * code_generation
 */

/* include lex output */
#include "scanner.c"

int mirror_objects = 0;

int main(int argc, char *argv[]) 
{
  FILE         *idlfile = NULL;
  FILE         *cout = NULL;
  char         coutfilename[255];
  char         *l;
  int	       i, verbose = 0;

  for (i = 1; i< argc; i++) {
      if (argv[i][0] == '-') {
	  /* must be argument */
	  if (strncmp(argv[i], "-v", 2) == 0) {
	      verbose++;
	  } else if (strncmp(argv[i], "-mirror",7) == 0) {
	      mirror_objects++;
	  } else {
	      printf("Unknown argument \"%s\"\n", argv[i]);
	  }
      } else {
	  if (idlfile == NULL) {
	      /* must be input file */
	      strcpy(idlfilename, argv[i]);
	      adjust_filename(idlfilename);
	      strcpy(coutfilename, idlfilename);
	      l = strrchr(coutfilename, '.'); /* has to find it */
	      strcpy(l, "_COBS_.c");
	      /* open the input and output files */
	      if ((idlfile = freopen(idlfilename, "r", stdin)) == NULL) {
		  printf("Can't open file %s for reading\n", idlfilename);
		  exit(1);
	      }
	  } else if (cout == NULL) {
	      /* must be output file */
	      strcpy(coutfilename, argv[i]);
	      if ((cout    = fopen(coutfilename, "w")) == NULL) {
		  printf("Can't open file %s for writing\n", coutfilename);
		  exit(1);
	      }
	  } else {
	      printf("Usage is code_gen [<file_in>] [file_out]\n");
	      exit(1);
	  }
      }
  }
  if (cout == NULL) {
      if (idlfile == NULL) {
	  /* no arguments, the input is stdin, output is stdout */
	  strcpy(idlfilename, "stdin");
	  strcpy(coutfilename, "stdout");
	  idlfile = stdin;
	  cout    = stdout;
      } else {
	  if ((cout    = fopen(coutfilename, "w")) == NULL) {
	      printf("Can't open file %s for writing\n", coutfilename);
	      exit(1);
	  }
      }
  }
  
  if (verbose)
      printf("File to be processed is %s, Output code will be in %s\n", 
	 idlfilename, coutfilename);
  yyparse();
  
  resolveUndefined();
  /* finish initializing useful information for type */
  for (i=0; i < typelist_size-1; i++) {
    if ((typelist_array[i]->typedef_of != NULL)  
	&& (typelist_array[i]->typedef_of == typelist_array[i+1])) {
	/* swap the two */
	Type_Description *tmp = typelist_array[i];
	typelist_array[i] = typelist_array[i+1];
	typelist_array[i+1] = tmp;
    }

    typelist_array[i]->node_info = undefined;
    typelist_array[i]->has_variable_size = UNDEFINED;
  }
  
  /* fill up information about variable sized types */

  for (i=0; i < typelist_size; i++) 
    typelist_array[i]->has_variable_size = 
      set_has_variable_size(typelist_array[i]);

  if (verbose) dump(itfc_list_head);

  /* code_generate needs the name of the .c file in a pure format, ie,
     without the _COBS_.c, so it can have original filename*/
  coutfilename[strlen(coutfilename)-strlen("_COBS_.c")] = '\0';
  code_generate(itfc_list_head, typelist_array, 
		typelist_size, cout, coutfilename);
  
  fclose(idlfile);
  fclose(cout);
  exit(0);
}

void yyerror(str)
char *str;
{
  printf("## Error while parsing %s; near %s , offset = %d, line = %d ####\n",
	 idlfilename,yytext,lex_offset,line_count);
  exit(1);
}

void dump_type(TypeCode t, Complex_Type *c)
{
  switch (t) {
  case NT_enum:
    printf("NT_enum ");
    break;
  case NT_pre_defined:
    printf("NT_pre_defined ");
    break;
  case NT_struct:
    printf("NT_struct ");
    break;
  case NT_string:
    printf("NT_string; Maximum = %d ", 
	   c->tstring.maximum);
    break;
  case NT_sequence:
    printf("NT_sequence;Maximum = %d,Basetypename %s, Basetypecode %d\n",
	   c->tsequence.maximum, 
	   c->tsequence.basetypename, 
	   c->tsequence.basetypecode); 
    break;
  case NT_array: {
    int i;
    printf("NT_array; ndims=%d ( ", c->tarray.ndims);
    for (i=0; i < c->tarray.ndims; i++)
      printf("%d ", c->tarray.dims[i]);
    printf("), Basetypename %s\n", c->tarray.basetypename);
    break;
  }
  default:
    printf("Unknown type in dump_type: %d !!\n", t);
    exit(1);
  }	  
}

void dump (Interface_list_cell *ilist)
{
  Interface_list_cell *ptr = ilist;
  Interface           *itfc;
  Operation_list_cell *olist;
  Operation           *op;
  Argument_list_cell  *arglist;
  Argument            *arg;
  int                i;

  printf("Dumping data structure built from %s\n", idlfilename);
  
  while (ptr != NULL) {
    itfc = ptr->itfc;
    printf("Interface %s\n", itfc->name);
    if (itfc->att_list == NULL)
      printf("No attributes.\n");
    else {
      Attribute_list_cell *alist = itfc->att_list;
      Attribute           *att;
      while (alist != NULL) {
	att = alist->att;
	printf("Attribute %s: typename %s ", att->name, att->typename);
	
	dump_type(att->typecode, att->parameters);
	if (att->readonly)
	  printf(" Readonly");
	printf("\n");
	alist=alist->next;
      }
    }
    /* now dump the operations */
    olist = itfc->op_list;
    if (olist == NULL)
      printf("No operations\n");
    while (olist != NULL) {
      op = olist->op;
      printf("Operation %s: return typename %s ", op->name,
	     op->rettypename);
      dump_type(op->rettypecode, op->rettype_parameters);
      arglist  = op->arg_list;
      while (arglist != NULL) {
	arg = arglist->arg;
	printf("\nArgument %s Typename %s", arg->name, arg->typename);
	switch (arg->direction) {
	case D_IN:
	  printf(" D_IN ");
	  break;
	case D_OUT:
	  printf(" D_OUT ");
	  break;
	case D_INOUT:
	  printf(" D_INOUT ");
	  break;
	default:
	  printf("\n ###### ERROR: ######\n");
	  printf("Unexpected direction for argument %s in dump()\n",
		 arg->name);
	}
	dump_type(arg->typecode, arg->type_parameters);
	printf("\n");
	arglist = arglist->next;
      }
      printf("\n");
      if (op->context != NULL) {
	int i;
	printf("Context ");
	for (i=0; i< op->context->nb_elems; i++)
	  printf("%s ", op->context->list[i]);
	printf("\n");
      }
      olist = olist->next;
    }
    ptr = ptr->next;
  }

  /* dumping information about types */
  for (i=0; i < typelist_size; i++) {
    printf("TYPEINFO %s ", typelist_array[i]->name);
    if (typelist_array[i]->typedef_of)
      printf("TYPEDEF_OF %s", typelist_array[i]->typedef_of->name);
    else {
      int j;
      dump_type(typelist_array[i]->code, typelist_array[i]->type);
      printf("\n");
      /* dumping the list of subtypes for this type also */
      for (j=0; j < typelist_array[i]->nb_subtypes; j++) {
	assert(typelist_array[i]->list_subtypes!=NULL);
	assert(typelist_array[i]->list_subtypes[j]!=NULL);
	printf("\t\tsubtype %s\n", 
	       typelist_array[i]->list_subtypes[j]->name);
      }
    }
    printf("\n");
  }
	   
}


void check_malloc_result(void* mem)
{
  if (mem == NULL) {
    printf("No memory for allocation\n");
    exit(1);
  }
}

void adjust_filename(char *name)
{
  char *l;
  /* The name can be something like "file.desc", or just "file" (in which
     case we add ".desc" to the name before trying to opening it */
  if ( (l = strrchr(name, '.')) == NULL) {
    strcat(name, ".desc");
  }
  else {
    /* it was found '.', so it has to be desc */
    if (strcmp(l,".desc") != 0) {
      printf("File name not valid for interface description.");
      printf("Expected *.desc\n");
      exit(1);
    }
  }
}

char *create_string_from_yytext()
{
  char *st = (char *) malloc((strlen(yytext)+1)*sizeof(char));
  strcpy (st, yytext);
  return(st);
}


Type_Description* searchTypeDescription(char *name,
					Type_Description **typelist_array,
					int typelist_size)
{
  int i;
  for (i=0; i < typelist_size; i++) {
    if (strcmp(name, typelist_array[i]->name) == 0)
      return(typelist_array[i]);
  }
  return(NULL);
}

void
dumpTypeList(Type_Description **typelist_array, int typelist_size)
{
  int i;
  printf("Type list is :\n");
  for (i=0; i < typelist_size; i++) {
      printf("Type \"%s\"  ", typelist_array[i]->name);
      dump_type(typelist_array[i]->code, typelist_array[i]->type);
      printf("\n");
  }
}

void insertSubtype (Type_Description *t, Type_Description *subt)
{
  /* deal with typedefs */
  while (subt->typedef_of != NULL)
    subt = subt->typedef_of;
  if (t->nb_subtypes == 0) {
    /* first subtype */
    t->list_subtypes = (Type_Description **) malloc(sizeof(Type_Description*));
    assert(t->list_subtypes != NULL);
    t->list_subtypes[0] = subt;
  }
  else {
    t->list_subtypes = (Type_Description **) realloc(t->list_subtypes,
						     sizeof(Type_Description*)
						     *(t->nb_subtypes+1));
    assert (t->list_subtypes != NULL);
    t->list_subtypes[t->nb_subtypes] = subt;
  }
  t->nb_subtypes++;
}
    
/* insertUndefined 
**
** keeps a list of subtypes that weren't stored in typelist_array yet
**
**/
void insertUndefined (Type_Description *t, char *subtype, Pending_kind kind)
{
  /* checking if there is room for one more element in the list */
  if (list_undef.size >= list_undef.max)  /* should never be > ... */ {
    list_undef.max += 100;    /* arbitrary ammount */
    list_undef.array = realloc (list_undef.array, 
				sizeof(Undefined_Subtype)*list_undef.max);
    assert(list_undef.array != NULL);
  }
  /* insertion */
  list_undef.array[list_undef.size].type = t;
  list_undef.array[list_undef.size].pending_kind = kind;
  list_undef.array[list_undef.size++].pending_name = subtype;
}

/* resolveUndefined
**
** This routine will be called after all the parsing has been done. 
** It will fill up lacunas related to the list of subtypes 
*/
void resolveUndefined ()
{
  int i;
  Type_Description *aux = NULL;
  for (i=0; i < list_undef.size; i++) {
    switch (list_undef.array[i].pending_kind) {
    case subtype_info:
      aux = searchTypeDescription(list_undef.array[i].pending_name,
				  typelist_array, typelist_size);
      if (aux!=NULL) 
	insertSubtype (list_undef.array[i].type, aux);
      else {
	/* The subtype definition didn't appear anywhere 
	** This shouldn't happen (the IDL to C compiler would not 
	** consider the IDL specification valid
	*/
	printf("Unexpected scenario in interface.y:\n");
	printf("\t subtype %s (associated with %s) did not appear\n",
	       list_undef.array[i].pending_name,
	       list_undef.array[i].type->name);	     
	assert(1==0);
      }      
      break;
    case typedef_info:
      aux = searchTypeDescription(list_undef.array[i].pending_name,
				  typelist_array, typelist_size);
      assert(aux!=NULL);
      list_undef.array[i].type->typedef_of = aux;
      list_undef.array[i].type->code = aux->code;
      /* subtype information */
      if (aux->code != NT_enum && aux->code != NT_pre_defined)
	insertSubtype(list_undef.array[i].type, aux);
      break;
    default:
      printf("Unexpected value for pending_kind (%d)\n",
	     list_undef.array[i].pending_kind);
      assert(1==0);
    }
  }
}


/*  type_has_variable_size : given a type, returns TRUE if the
**                           type has variable size, FALSE otherwise.
**                           (To have variable size means that a variable
**                           of this type can consume a variable amount
**                           of memory when being serialized.
**                           IMPORTANT: besides returning the value,
**                           it sets the information in the type.
*/
char set_has_variable_size(Type_Description *type)
{
  int i;
  Type_Description *aux = NULL;

  if (type->has_variable_size != UNDEFINED)
    return(type->has_variable_size);
  
  if (type->typedef_of != NULL) {
    type->has_variable_size = set_has_variable_size(type->typedef_of);
    return(type->has_variable_size);
  }

  switch(type->code) {
  case NT_enum:
  case NT_pre_defined:
  case NT_interface:
    type->has_variable_size = FALSE;
    break;
  case NT_string:
    type->has_variable_size = TRUE;    
    /* actually this (above) is a simplification.
    ** IDL strings can
    ** be declared to have a maximum size. We will refine
    ** this later. (field "maximum" in the data structure for 
    ** the type has this info). [dilma]
    */
    break;
  case NT_sequence:
    type->has_variable_size = TRUE; /* same simplification as above */
    break;
  case NT_array: {
    /* we need to check if the base type has variable size */
    Type_Description *t =searchTypeDescription(type->type->tarray.basetypename,
					       typelist_array, typelist_size);
    if (t!=NULL) 
      type->has_variable_size = set_has_variable_size(t);
    else {
      /* The subtype definition didn't appear anywhere 
      ** This shouldn't happen (the IDL to C compiler would not 
      ** consider the IDL specification valid
      */
      printf("Unexpected scenario in interface.y:\n");
      printf("\t subtype %s (associated with array type %s) did not appear\n",
	     type->type->tarray.basetypename, type->name);
      assert(1==0);
    }
    break;
  }      
  case NT_struct: {
    /*int i;
      Type_Description *aux;*/
    for (i=0; i < type->type->tstruct.nb_fields ; i++) {
      switch (type->type->tstruct.array_field_typecode[i]) {
      case NT_enum:
      case NT_pre_defined:
      case NT_interface:
	break;
      case NT_string:
      case NT_struct:
      case NT_sequence:
      case NT_array:
	aux = searchTypeDescription(type->type->tstruct.array_field_type[i],
				    typelist_array, typelist_size);
	assert(aux!=NULL);
	switch (aux->has_variable_size) {
	case TRUE: /* one field, is variable,
		   ** therefore the struct is*/
	  type->has_variable_size = TRUE;    
	  return(TRUE);
	case FALSE:
	  break;
	case UNDEFINED:
	  if (set_has_variable_size(aux)) {
	    type->has_variable_size = TRUE;    
	    return(TRUE); /* break the loop and leave */
	  }
	  break;
	default:
	  assert(1==0);
	}
	break;
      case NT_not_known:
      default: 
	printf("Unexpected value in typecode for type %s: %d\n",
	       aux->name, aux->code);
	assert(1==0);
      } /* end of switch(aux->code) */
    }
    /* if didn't return with TRUE so far, we have that all fields
    ** have fixed size */
    type->has_variable_size = FALSE;    
    break;
  }
  case NT_not_known:
    break;
  default: 
      printf("Unexpected value in type->code for type %s: %d\n",
	     type->name, type->code);
      assert(1==0);
  }
  return(type->has_variable_size);
}

void build_type_data_for_pre_defined_types(Type_Description ***typelist_array,
					   int *typelist_size)
{
  int i;
  char *types[12] = {"short", "unsigned_short", "long", "unsigned_long",
		     "float", "double", "boolean", "char", "octet", "enum",
                     "any", "Object"};
  Type_Description *t;
  *typelist_array = (Type_Description**)
    malloc(sizeof(Type_Description*)*(*typelist_size+12));
  assert(typelist_array!=NULL);

  for (i=0; i<12; i++) {
    t = (Type_Description*) malloc(sizeof(Type_Description));
    assert(t!=NULL);
    t->name = (char *) malloc((strlen(types[i])+strlen("CORBA_")+1)
			      *sizeof(char));
    assert(t->name!=NULL);
    strcpy(t->name, "CORBA_");
    strcpy(&(t->name[6]), types[i]);
    t->code = NT_pre_defined;
    t->type = NULL;
    t->typedef_of = NULL;
    t->list_subtypes = NULL;
    t->nb_subtypes = 0;
    /* insert in the list of types */
    (*typelist_array)[*typelist_size+i] = t;
  }
    (*typelist_size) += 12;
}  
