/* File code_generation.c
 *
 * This file contains code_generate and code_for_attributes
 * (The function code_for_operations went out from this file so
 *  compilation during changes could be quickier. And
 *  code_for_obj_creation is in the file with same name).  */

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#if defined(__STDC__)
void *malloc(size_t);
void free(void*);
void *realloc(void*, size_t);
#else
void *malloc();
void free();
void *realloc();
#endif	/* __STDC__ */
#include "code_generation.h"
#include "config.h"
#include "unix_defs.h"
#include "assert.h"

static
void code_for_type_routines(FILE *cout, FILE *hout,
			    Type_Description **typelist_array,
			    int typelist_size);

static char *output_filename = NULL;
/* 
 * code_generate
 */

void
code_generate(Interface_list_cell *itfc_list_head, 
	      Type_Description **typelist_array,
	      int typelist_size,
	      FILE *cout, char *fname)
{
  FILE                     *hout;
#ifdef OTL_STUFF
  FILE                     *otl_out;
  char                     otl_cname[255];
#endif
  char                     hname[255];
  Interface_list_cell      *ptr = itfc_list_head;
  Interface                *itfc;
  int                      define_start;

  fprintf(cout, "/*    File  %s.c    */\n\n", fname);
  fprintf(cout, "#include <malloc.h>\n");
  fprintf(cout, "#include <stdlib.h>\n");
  fprintf(cout, "#include <string.h>\n");
  fprintf(cout, "#include <memory.h>\n");
  fprintf(cout, "#include <gen_thread.h>\n");
  fprintf(cout, "#include <io.h>\n");
  fprintf(cout, "#include <DE.h>\n");
  fprintf(cout, "#include <otl.h>\n");
  fprintf(cout, "#include \"%s_COBS_.h\"\n\n", fname);
  fprintf(cout, "extern object_ref _COBS_otl_initial_ref;\n\n");
  /* The attribute structure definition has to be output in the .h file */
  strcpy(hname, fname);
  strcat(hname, "_COBS_.h");
  if ((hout = fopen(hname, "w")) == NULL) {
    printf("Can't open file %s to ouput attribute definition\n",
	   hname);
    exit(1);
  }
  /* fname can't be used directly to generate 
   *         #define file_COBS__H
   *  because fname can be something like ../../file
   */
  define_start = strlen(fname)-1;  /* last character in fname */
  while (define_start>=0 && fname[define_start] != '/')
    define_start--;
  define_start++;
  fprintf(hout, "/* File %s_COBS_.h  */\n\n", fname);
  fprintf(hout, "#ifndef %s_COBS__H\n", &fname[define_start]);
  fprintf(hout, "#define %s_COBS__H\n", &fname[define_start]);
  output_filename = strdup(&fname[define_start]);
  fprintf(hout, "#include <io.h>\n");
  fprintf(hout, "#include <otl.h>\n");
  fprintf(hout, "#include \"%s.h\"\n\n", fname);
  

#ifdef OTL_STUFF
  /* Code for dealing with OTL object refs (for example, how to
     create virtual functions for the given interfaces) is output
     in a separated file, in order to facilitate other people using
     only this part of the code generation */
  strcpy(otl_cname, fname);
  strcat(otl_cname, "_COBS_OTL_CODE_.c");
  if ((otl_out = fopen(otl_cname, "w")) == NULL) {
    printf("Can't open file %s to ouput attribute definition\n",
	   otl_cname);
    exit(1);
  }

  code_for_otl_preambles(otl_out, fname);
#endif

  /* generating code associated with types, i.e., from the list of
     complex types defined in the idl specification  we generate
     functions to marshall and unmarshall these types. 
     */
  code_for_io_formats(typelist_array, typelist_size, cout, hout);
  
  code_for_type_routines(cout, hout, typelist_array, typelist_size);

  /* generating code for interfaces */
  while (ptr != NULL) {
    itfc = ptr->itfc;
    fprintf(cout, "/* Code for interface %s */\n\n", itfc->name);
#ifdef STATE_STUFF
    code_for_object_state(itfc, cout, fname, &fname[define_start]);
#endif
    code_for_early_decls(itfc, cout);
    if (itfc->att_list != NULL) {
	generate_io_formats_for_state(itfc, cout);
    }
    code_for_attributes(typelist_array, typelist_size, itfc, 
			cout, hout, fname);
    if (itfc->op_list != NULL) {
	early_code_for_operations(typelist_array, typelist_size,
			    itfc, cout, hout, fname);
    }
    code_for_obj_creation(typelist_array, typelist_size, itfc, 
			  cout, hout, fname);
    if (itfc->op_list != NULL) {
	code_for_operations(typelist_array, typelist_size,
			    itfc, cout, hout, fname);
    }
    code_for_object_storage(typelist_array, typelist_size, itfc, cout);
#ifdef THERE
    code_for_otl_ref_use(itfc, otl_out, hout, fname);
#endif
    ptr=ptr->next;
  }

  gen_mirror_class_init(hout, cout, fname);
  fprintf(hout, "\n#endif\n");
  fclose(hout);
#ifdef OTL_STUFF
  fprintf(otl_out, "/* end of %s_COBS_OTL_CODE_.c */\n", fname);
  fclose(otl_out);
#endif
}

/*
static void
free_dummy_op(Operation *op)
{
    free(op->name);
    if (op->rettypename != NULL) free(op->rettypename);
    if (op->rettype_parameters != NULL) free(op->rettype_parameters);
    if (op->arg_list != NULL) {
	Argument_list_cell *alist = op->arg_list, *next;
	while(alist != NULL) {
	    Argument *arg = alist->arg;
	    free(arg->name);
	    free(arg->typename);
	    if (arg->type_parameters != NULL) free(arg->type_parameters);
	    free(arg);
	    next = alist->next;
	    free(alist);
	    alist = next;
	}
    }
    free(op);
}
*/
static void add_op_to_itfc(Interface *itfc, Operation *op)
{
    Operation_list_cell *cell = malloc(sizeof(Operation_list_cell));
    cell->next = NULL;
    cell->op = op;

    if (itfc->op_list == NULL) {
	itfc->op_list = cell;
    } else {
	Operation_list_cell *op_list = itfc->op_list;
	while (op_list->next != NULL) {
	    op_list = op_list->next;
	}
	op_list->next = cell;
    }
}

static Operation *
generate_dummy_get_op(Attribute *att, int attr_count)
{
    Operation *op = malloc(sizeof(Operation));
    op->name = malloc(strlen(att->name) + strlen("_get_") + 1);
    sprintf(op->name, "_get_%s", att->name);
    op->id = attr_count * 2;
    op->rettypename = strdup(att->typename);
    if (att->parameters != NULL) {
	op->rettype_parameters = malloc(sizeof(Complex_Type));
	*op->rettype_parameters = *att->parameters;
    } else {
	op->rettype_parameters = NULL;
    }
    op->rettypecode = att->typecode;
    op->arg_list = NULL;
    op->mirror = 0;
    op->nb_context_elems = 0;
    op->context = NULL;
    op->hasINarguments = FALSE;
    op->hasOUTarguments = FALSE;
    op->hasINOUTarguments = FALSE;
    return op;
}

static Operation *
generate_dummy_set_op(Attribute *att, int attr_count)
{
    Operation *op = malloc(sizeof(Operation));
    op->name = malloc(strlen(att->name) + strlen("_get_") + 1);
    sprintf(op->name, "_set_%s", att->name);
    op->id = attr_count * 2 + 1;
    op->rettypename = strdup("CORBA_void");
    op->rettype_parameters = NULL;
    op->arg_list = malloc(sizeof(Argument_list_cell));
    op->arg_list->next = NULL;
    op->arg_list->arg = malloc(sizeof(Argument));
    op->arg_list->arg->name = strdup("value");
    op->arg_list->arg->direction = D_IN;
    op->arg_list->arg->typename = strdup(att->typename);
    op->arg_list->arg->typecode = att->typecode;
    if (att->parameters != NULL) {
	op->arg_list->arg->type_parameters = malloc(sizeof(Complex_Type));
	*op->arg_list->arg->type_parameters = *att->parameters;
    } else {
	op->arg_list->arg->type_parameters = NULL;
    }
    op->nb_context_elems = 0;
    op->context = NULL;
    op->hasINarguments = TRUE;
    op->hasOUTarguments = FALSE;
    op->hasINOUTarguments = FALSE;
    return op;
}

void
code_for_attributes(Type_Description **typelist_array,
		    int typelist_size,
		    Interface *itfc, FILE *cout, FILE *hout,char *fname)
{
  Attribute_list_cell    *ptr = itfc->att_list;
  Attribute              *att;
  int			 attr_count = 0;

  fprintf(hout, "/* Structure for attributes */\n");
  fprintf(hout, "typedef struct {\n");
  fprintf(hout, "\totl_obj_info *object_info;\n");
  fprintf(hout, "\tattr_list base_reference;\n");
  while (ptr != NULL) {
    att = ptr->att;
    fprintf(hout, "\t%s %s;\n",  att->typename, att->name);
    ptr = ptr->next;
  }
  fprintf(hout, "\tvoid *instanceData;\n");
  fprintf(hout, "} _IDLtoC_%s_attrib_struct;\n\n", itfc->name);

  /* generating the body for set/get functions. Notice that prototypes
     for them were generated by IDLtoC translation (into a .h file) */
  
  ptr = itfc->att_list;
  while (ptr != NULL) {
    Operation *op = generate_dummy_get_op(ptr->att, attr_count);
    add_op_to_itfc(itfc, op);
    att = ptr->att;

    fprintf(cout, "\n/* \"get\" function for attribute %s */\n\n",
	    att->name);
    fprintf(cout, "%s %simpl_%s__get_%s (%s self, CORBA_Environment *ev)\n",
	    att->typename, 
	    ((att->typecode == NT_sequence) || (att->typecode == NT_array)) ? "*" : "",
	    itfc->name, att->name, itfc->name);
    fprintf(cout, "{\n");
    fprintf(cout, "   _IDLtoC_%s_attrib_struct *state;\n", itfc->name);
    fprintf(cout, "   static atom_t otl_self_atom = 0;\n");
    fprintf(cout, "   if (otl_self_atom == 0) {\n");
    fprintf(cout, "   	otl_self_atom = attr_atom_from_string(\"OTL:OBJECT_SELF\");\n");
    fprintf(cout, "   }\n");
    fprintf(cout, "   query_attr(self->obj_name, otl_self_atom,\n");
    fprintf(cout, "              NULL, (void**)&state);\n\n");

    /* which code to generate for returning the attribute value depends
       on the attribute type */
    switch (att->typecode) {
    case NT_enum:
    case NT_pre_defined:
    case NT_interface:
    case NT_struct:    
      /* in these three cases we just have to return the value */
      fprintf(cout,
	      "    return(state->%s);\n", att->name);
      break;
    case NT_string:
      /* in this case we have to follow the recommendations by CORBA,
	 there is memory to be allocated, etc */
      fprintf(cout,"    return((%s) COBS_return_string(\n", att->typename);
      fprintf(cout,"\t(char*)state->%s, %d ));\n",
	      att->name, att->parameters->tstring.maximum);
      break;
    case NT_sequence:
      /* in this case we have to follow the recommendations by CORBA,
	 there is memory to be allocated, etc */
      fprintf(cout, "    {\n");
      fprintf(cout,"\t%s *seq = malloc(sizeof(%s));\n", att->typename,
	      att->typename);
      fprintf(cout,"\t%s *attv = ", att->typename);
      fprintf(cout,"&state->%s;\n", att->name);
      fprintf(cout,"\tseq->_maximum = %d;\n", 
	      att->parameters->tsequence.maximum);
      fprintf(cout,"\tseq->_length = attv->_length;\n");
      fprintf(cout,"\tseq->_buffer = malloc(%d*sizeof(%s));\n",
	      att->parameters->tsequence.maximum, 
	      att->parameters->tsequence.basetypename);
      fprintf(cout,"\tmemcpy(seq->_buffer, attv->_buffer, attv->_length * sizeof(%s));\n",
	      att->parameters->tsequence.basetypename);
      
      fprintf(cout,"\treturn(seq);\n");
      fprintf(cout, "    }\n");
      break;
    case NT_array:
      /* in this case we have to follow the recommendations by CORBA,
	 there is memory to be allocated, etc */
      fprintf(cout, "    {\n");
      fprintf(cout,"\t%s *attv = ", att->typename);
      fprintf(cout,"&state->%s;\n", att->name);
      fprintf(cout,"\t%s *array = malloc(sizeof(%s));\n", att->typename,
	      att->typename);

      fprintf(cout,"\tmemcpy(array, attv, sizeof(%s));\n",
	      att->typename);
      fprintf(cout,"\treturn(array);\n");
      fprintf(cout, "    }\n");
      break;
    default:
      printf("In code_for_attributes: unexpected type for get function.\n");
      exit(1);
    }
    fprintf(cout, "}\n\n");

    /* generating "set" function if attribute is not read only */
    if (!att->readonly && !ptr->att->mirror) {

      Operation *op = generate_dummy_set_op(att, attr_count);
      add_op_to_itfc(itfc, op);
      fprintf(cout, "\n/* \"set\" function for attribute %s */\n\n",
	      att->name);
      fprintf(cout, "void impl_%s__set_%s(%s self, %s ",
	      itfc->name, att->name, itfc->name, att->typename);
      switch(att->typecode) {
      case NT_sequence:
      case NT_struct:
      case NT_array:
	  fprintf(cout, "*value,CORBA_Environment *ev)\n");
	  break;
      default:
	  fprintf(cout, "value,CORBA_Environment *ev)\n");
	  break;
      }
      fprintf(cout, "{\n");
      fprintf(cout, "   _IDLtoC_%s_attrib_struct *state;\n", itfc->name);
      fprintf(cout, "   static atom_t otl_self_atom = 0;\n");
      if (mirror_objects) {
	  fprintf(cout, "   static atom_t otl_chan_handle_atom = 0;\n");
	  fprintf(cout, "   ECSourceHandle event_handle = NULL;\n");
      }
      fprintf(cout, "   if (otl_self_atom == 0) {\n");
      fprintf(cout, "   	otl_self_atom = attr_atom_from_string(\"OTL:OBJECT_SELF\");\n");
      if (mirror_objects) {
	  fprintf(cout, "        otl_chan_handle_atom =\n");
	  fprintf(cout, "            attr_atom_from_string(\"OTL:MON_CHANNEL_HANDLE\");\n");
      }
      fprintf(cout, "   }\n");
      fprintf(cout, "   query_attr(self->obj_name, otl_self_atom,\n");
      fprintf(cout, "              NULL, (void**)&state);\n\n");
      if (mirror_objects) {
	  fprintf(cout, "   query_attr(self->obj_name, otl_chan_handle_atom,\n");
	  fprintf(cout, "              NULL, (void**)&event_handle);\n\n");
      }

      switch (att->typecode) {
      case NT_enum:
      case NT_pre_defined:
      case NT_interface:
	/* In these three cases it is just a mere assignment */
	fprintf(cout, "    state->%s = value;\n", att->name);
	break;
      case NT_struct:   
	/* In these three cases it is just a mere assignment */
	fprintf(cout, "    state->%s = *value;\n", att->name);
	break;
      case NT_string:
	/* there may be allocation related to this ... it has to follow
	   IDL specification defined semantics */ 
	fprintf(cout,"    COBS_assign_string( (char **)\n");
	fprintf(cout,"\t&state->%s,", att->name);
	fprintf(cout,"\n\t\t(char *) value, %d);\n",
		att->parameters->tstring.maximum);
	break;
      case NT_sequence:
	fprintf(cout, "    {\n");
	fprintf(cout,"\t%s *attv = ", att->typename);
	fprintf(cout,"&state->%s;\n", att->name);
	fprintf(cout,"\tCOBS_assign_sequence(\n");
	fprintf(cout,"\t    (COBS_generic_sequence_struct *)attv,\n");
	fprintf(cout,"\t    (COBS_generic_sequence_struct *)value,\n");
	fprintf(cout,"\t\tsizeof(%s));\n", 
		att->parameters->tsequence.basetypename);
	fprintf(cout, "    }\n");
	break;
      case NT_array:
	fprintf(cout, "    {\n");
	fprintf(cout,"\t%s *attv = ", att->typename);
	fprintf(cout,"&state->%s;\n", att->name);
	fprintf(cout,"\tmemcpy(attv, value, sizeof(%s));\n", att->typename);
	fprintf(cout, "    }\n");
	break;
      default:
	printf("In code_for_attributes: unexpected type for get function.\n");
	exit(1);
      }
      if (mirror_objects) {
	  fprintf(cout, "    if (EChas_sinks(event_handle)) {\n");
	  fprintf(cout, "        %s_submit_state(event_handle, state);\n",
		  itfc->name);
	  fprintf(cout, "    }\n");
      }
      fprintf(cout, "}\n\n");
    }
    
    ptr = ptr->next;
    attr_count++;
  }
    
}

static
void code_for_sequence_routines(FILE *cout, FILE *hout, 
				char *name, type_sequence seq,
				Type_Description **typelist_array,
				int typelist_size)
{
    int indent = 0;
    char *basetypename = seq.basetypename;
    char *unique_name;
    Type_Description *aux = searchTypeDescription(basetypename, 
						  typelist_array, 
						  typelist_size);
    assert(aux);
    if (aux->typedef_of != NULL) {
	basetypename = aux->typedef_of->name;
    }
    if (strncmp(name, "CORBA_sequence_", strlen("CORBA_sequence_")) == 0) {
	int len = strlen(name) + strlen(output_filename) + 3;
	unique_name = malloc(len);
	memset(unique_name, 0, len);
	sprintf(unique_name, "%s__%s", output_filename, name);
    } else {
	unique_name = name;
    }

    OUT(indent, hout, "#ifndef %s_Create\n", name);
    OUT(indent, hout, "#define %s_Create %s_Create\n", name, unique_name);
    OUT(indent, hout, "#endif\n");

    OUT(indent, hout, "extern %s *\n%s_Create(int length, %s *init_values);\n",
	name, unique_name, seq.basetypename);
    OUT(indent, cout, "extern %s *\n%s_Create(length, init_values)\n",
	name, unique_name);
    OUT(indent, cout, "int length;\n%s *init_values;\n", seq.basetypename);
    OUT(indent, cout, "{\n");
    indent++;
    OUT(indent, cout, "%s *tmp_seq = malloc(sizeof(*tmp_seq));\n", name);
    OUT(indent, cout, "%s_Init(tmp_seq, length, init_values);\n", name);
    OUT(indent, cout, "return tmp_seq;\n");
    indent--;
    OUT(indent, cout, "}\n\n");

    OUT(indent, hout, "#ifndef %s_Init\n", name);
    OUT(indent, hout, "#define %s_Init %s_Init\n", name, unique_name);
    OUT(indent, hout, "#endif\n");
    OUT(indent, hout, "extern void %s_Init(%s *s, int length, %s *init_values);\n",
	unique_name, name, seq.basetypename);
    OUT(indent, cout, "extern void %s_Init(s, length, init_values)\n",
	unique_name);
    OUT(indent, cout, "%s *s;\n", name);
    OUT(indent, cout, "int length;\n%s *init_values;\n", seq.basetypename);
    OUT(indent, cout, "{\n");
    indent++;
    OUT(indent, cout, "s->_maximum = %d;\n", seq.maximum);
    OUT(indent, cout, "s->_length = length;\n");
    OUT(indent, cout, "if (init_values != NULL) {\n");
    indent++;
    OUT(indent, cout, "s->_buffer = init_values;\n");
    indent--;
    OUT(indent, cout, "} else {\n");
    indent++;
    OUT(indent, cout, "if (length > 0) {\n");
    indent++;
    OUT(indent, cout, "s->_buffer = malloc(sizeof(%s) * length);\n", 
	seq.basetypename);
    OUT(indent, cout, "memset(s->_buffer, 0, sizeof(%s) * length);\n",
	seq.basetypename);
    indent--;
    OUT(indent, cout, "} else {\n");
    indent++;
    OUT(indent, cout, "s->_length = 0;\n");
    OUT(indent, cout, "s->_buffer = NULL;\n");
    indent--;
    OUT(indent, cout, "}\n"); /* if length */
    indent--;
    OUT(indent, cout, "}\n"); /* if init_values */
    indent--;
    OUT(indent, cout, "}\n\n"); /* subroutine start */

    OUT(indent, hout, "#ifndef %s_Free\n", name);
    OUT(indent, hout, "#define %s_Free %s_Free\n", name, unique_name);
    OUT(indent, hout, "#endif\n");
    OUT(indent, hout, "extern void %s_Free(%s *s);\n",
	unique_name, name);

    OUT(indent, cout, "extern void %s_Free(s)\n", unique_name);
    OUT(indent, cout, "%s *s;\n", name);
    OUT(indent, cout, "{\n");
    indent++;
    if (has_dynamic_substructure(seq.basetypename, typelist_array, 
				 typelist_size)) {
	OUT(indent, cout, "int i;\n");
	OUT(indent, cout, "for (i=0; i < s->_length; i++) {\n");
	indent++;
	if (is_string(seq.basetypename, typelist_array, typelist_size)) {
	    OUT(indent, cout, "if (s->_buffer[i]) free(s->_buffer[i]);\n");
	} else {
	    OUT(indent, cout, "%s_Free(&s->_buffer[i]);\n", basetypename);
	}
	indent--;
	OUT(indent, cout, "}\n");
    }
    OUT(indent, cout, "free(s->_buffer);\n");
    OUT(indent, cout, "s->_length = 0;\n");
    OUT(indent, cout, "s->_buffer = 0;\n");
    indent--;
    OUT(indent, cout, "}\n\n");


    OUT(indent, hout, "#ifndef %s_Copy\n", name);
    OUT(indent, hout, "#define %s_Copy %s_Copy\n", name, unique_name);
    OUT(indent, hout, "#endif\n");
    OUT(indent, hout, "extern void %s_Copy(%s *dest, %s *src);\n",
	unique_name, name, name);

    OUT(indent, cout, "extern void %s_Copy(dest, src)\n", unique_name);
    OUT(indent, cout, "%s *dest;\n", name);
    OUT(indent, cout, "%s *src;\n", name);
    OUT(indent, cout, "{\n");
    indent++;
    if (has_dynamic_substructure(seq.basetypename, typelist_array, 
				 typelist_size)) {
	OUT(indent, cout, "int i;\n");
	OUT(indent, cout, "%s_Free(dest);\n", name);
	OUT(indent, cout, "%s_Init(dest, src->_length, NULL);\n", name);
	OUT(indent, cout, "for (i=0; i < src->_length; i++) {\n");
	indent++;
	if (is_string(seq.basetypename, typelist_array, typelist_size)) {
	    OUT(indent++, cout, "if (src->_buffer[i] != NULL) {\n");
	    OUT(indent  , cout,"dest->_buffer[i] = malloc(strlen(src->_buffer[i])+1);\n");
	    OUT(indent--, cout,"strcpy(dest->_buffer[i], src->_buffer[i]);\n");
	    OUT(indent++, cout, "}\n");
	} else {
	    OUT(indent, cout,"%s_Copy(&dest->_buffer[i], &src->_buffer[i]);\n",
		basetypename);
	}
	indent--;
	OUT(indent, cout, "}\n");
    } else {
	OUT(indent, cout, "%s_Free(dest);\n", name);
	OUT(indent, cout, "%s_Init(dest, src->_length, NULL);\n", name);
	OUT(indent, cout, "memcpy(&dest->_buffer[0], &src->_buffer[0], src->_length * sizeof(%s));\n",
	    basetypename);
    }	
    indent--;
    OUT(indent, cout, "}\n\n");
}

static
void code_for_struct_routines(FILE *cout, FILE *hout, 
			      char *name, type_struct str,
			      Type_Description **typelist_array,
			      int typelist_size)
{
    int indent = 0;

    if (has_dynamic_substructure(name, typelist_array, typelist_size)) {
	int i;
	OUT(indent, hout, "extern void %s_Free(%s *s);\n",
	    name, name);

	OUT(indent, cout, "extern void %s_Free(s)\n", name);
	OUT(indent, cout, "%s *s;\n", name);
	OUT(indent, cout, "{\n");
	indent++;
	for (i=0; i<str.nb_fields; i++) {
	    if (has_dynamic_substructure(str.array_field_type[i], 
					 typelist_array, typelist_size)) {
		char *basetypename = str.array_field_type[i];
		Type_Description *aux = searchTypeDescription(basetypename, 
							      typelist_array, 
							      typelist_size);
		assert(aux);
		if (aux->typedef_of != NULL) {
		    basetypename = aux->typedef_of->name;
		}
		if (is_string(str.array_field_type[i], typelist_array, 
			      typelist_size)) {
		    OUT(indent, cout, "if (s->%s) free(s->%s);\n",
			str.array_field_name[i], str.array_field_name[i]);
		} else {
		    OUT(indent, cout, "%s_Free(&s->%s);\n", 
			basetypename, str.array_field_name[i]);
		}
	    }
	}
	indent--;
	OUT(indent, cout, "}\n\n");


	OUT(indent, hout, "extern void %s_Copy(%s *dest, %s *src);\n",
	    name, name, name);

	OUT(indent, cout, "extern void %s_Copy(dest, src)\n", name);
	OUT(indent, cout, "%s *dest;\n", name);
	OUT(indent, cout, "%s *src;\n", name);
	OUT(indent, cout, "{\n");
	indent++;
	for (i=0; i<str.nb_fields; i++) {
	    char *basetypename = str.array_field_type[i];
	    Type_Description *aux = searchTypeDescription(basetypename, 
							  typelist_array, 
							  typelist_size);
	    if (has_dynamic_substructure(str.array_field_type[i], 
					 typelist_array, typelist_size)) {
		assert(aux);
		if (aux->typedef_of != NULL) {
		    basetypename = aux->typedef_of->name;
		}
		if (is_string(str.array_field_type[i], typelist_array, 
			      typelist_size)) {
		    OUT(indent++, cout, "if (src->%s != NULL) {\n",
			str.array_field_name[i]);
		    OUT(indent, cout, "dest->%s = malloc(strlen(src->%s)+1);\n",
			str.array_field_name[i], str.array_field_name[i]);
		    OUT(indent--, cout, "strcpy(dest->%s, src->%s);\n",
			str.array_field_name[i], str.array_field_name[i]);
		    OUT(indent, cout, "}\n");
		} else {
		    OUT(indent, cout, "%s_Copy(&dest->%s, &src->%s);\n", 
			basetypename, str.array_field_name[i],
			str.array_field_name[i]);
		}
	    } else {
		if (aux->code == NT_pre_defined) {
		    OUT(indent, cout, "dest->%s = src->%s;\n",
			str.array_field_name[i], str.array_field_name[i]);
		} else {
		    OUT(indent, cout, "memcpy(&dest->%s, &src->%s, sizeof(src->%s));\n",
			str.array_field_name[i], str.array_field_name[i], 
			str.array_field_name[i]);
		}
	    }
	}
	indent--;
	OUT(indent, cout, "}\n\n");
    }
}

static
void code_for_typedef_routines(FILE *cout, FILE *hout, int i,
			       Type_Description **typelist_array,
			       int typelist_size)
{
    int indent = 0;
    char *name = typelist_array[i]->name;

    if (is_string(name, typelist_array, typelist_size)) return;

    if (has_dynamic_substructure(name, typelist_array, typelist_size)) {
	OUT(indent, hout, "extern void %s_Free(%s *s);\n",
	    name, name);

	OUT(indent, cout, "extern void %s_Free(s)\n", name);
	OUT(indent, cout, "%s *s;\n", name);
	OUT(indent, cout, "{\n");
	indent++;
	OUT(indent, cout, "%s_Free((%s*)s);\n", 
	    typelist_array[i]->typedef_of->name,
	    typelist_array[i]->typedef_of->name);
	indent--;
	OUT(indent, cout, "}\n\n");


	OUT(indent, hout, "extern void %s_Copy(%s *dest, %s *src);\n",
	    name, name, name);

	OUT(indent, cout, "extern void %s_Copy(dest, src)\n", name);
	OUT(indent, cout, "%s *dest;\n", name);
	OUT(indent, cout, "%s *src;\n", name);
	OUT(indent, cout, "{\n");
	indent++;
	OUT(indent, cout, "%s_Copy((%s*)dest, (%s*)src);\n", 
	    typelist_array[i]->typedef_of->name,
	    typelist_array[i]->typedef_of->name,
	    typelist_array[i]->typedef_of->name);
	indent--;
	OUT(indent, cout, "}\n\n");

    }
    if (typelist_array[i]->code == NT_sequence) {
	
	type_sequence *seq = &typelist_array[i]->typedef_of->type->tsequence;
	OUT(indent, hout, "extern void %s_Init(%s *s, int length, %s *init_values);\n",
	    name, name, seq->basetypename);

	OUT(indent, cout, "extern void %s_Init(s, length, init_values)\n", name);
	OUT(indent, cout, "%s *s;\n", name);
	OUT(indent, cout, "int length;\n");
	OUT(indent, cout, "%s *init_values;\n", seq->basetypename);
	OUT(indent, cout, "{\n");
	indent++;
	OUT(indent, cout, "%s_Init((%s*)s, length, init_values);\n", 
	    typelist_array[i]->typedef_of->name,
	    typelist_array[i]->typedef_of->name);
	indent--;
	OUT(indent, cout, "}\n\n");

	OUT(indent, hout, "extern %s * %s_Create(int length, %s *init_values);\n",
	    name, name, seq->basetypename);

	OUT(indent, cout, "extern %s *%s_Create(length, init_values)\n", 
	    name, name);
	OUT(indent, cout, "int length;\n");
	OUT(indent, cout, "%s *init_values;\n", seq->basetypename);
	OUT(indent, cout, "{\n");
	indent++;
	OUT(indent, cout, "return %s_Create(length, init_values);\n", 
	    typelist_array[i]->typedef_of->name);
	indent--;
	OUT(indent, cout, "}\n\n");
    }
}

static
void code_for_array_routines(FILE *cout, FILE *hout, 
			     char *name, type_array array,
			     Type_Description **typelist_array,
			     int typelist_size)
{
    int indent = 0;

    if (has_dynamic_substructure(array.basetypename, typelist_array, 
				 typelist_size)) {
	int i;
	char *basetypename = array.basetypename;
	Type_Description *aux = searchTypeDescription(array.basetypename, 
						      typelist_array, 
						      typelist_size);
	char *index_expr = malloc(3*array.ndims + 1);
	assert(aux);
	if (aux->typedef_of != NULL) {
	    basetypename = aux->typedef_of->name;
	}
	OUT(indent, hout, "extern void %s_Free(%s *s);\n", name, name);
	OUT(indent, cout, "extern void %s_Free(s)\n", name);
	OUT(indent, cout, "%s *s;\n", name);
	OUT(indent, cout, "{\n");
	indent++;
	for (i=0; i<array.ndims; i++) {
	    sprintf(index_expr + 3*i, "[%c]", 'i'+i);
	}
	for (i=0; i<array.ndims; i++) {
	    OUT(indent, cout, "int %c;\n", 'i' + i);
	}
	for (i=0; i<array.ndims; i++) {
	    OUT(indent, cout, "for (%c=0; %c < %d; %c++) {\n", 'i'+i, 'i'+i,
		array.dims[i], 'i'+i);
	    indent++;
	}
	if (is_string(array.basetypename, typelist_array, typelist_size)) {
	    OUT(indent, cout, "if (s%s) free(s%s);\n", index_expr, index_expr);
	} else {
	    OUT(indent, cout, "%s_Free(&(*s%s));\n", basetypename, index_expr);
	}
	for (i=0; i<array.ndims; i++) {
	    indent--;
	    OUT(indent, cout, "}\n");
	}
	indent--;
	OUT(indent, cout, "}\n");

	/* generate copy routine */
	OUT(indent, hout, "extern void %s_Copy(%s *dest, %s *src);\n", name, name,
	    name);
	OUT(indent, cout, "extern void %s_Copy(dest, src)\n", name);
	OUT(indent, cout, "%s *dest;\n", name);
	OUT(indent, cout, "%s *src;\n", name);
	OUT(indent, cout, "{\n");
	indent++;
	for (i=0; i<array.ndims; i++) {
	    OUT(indent, cout, "int %c;\n", 'i' + i);
	}
	for (i=0; i<array.ndims; i++) {
	    OUT(indent, cout, "for (%c=0; %c < %d; %c++) {\n", 'i'+i, 'i'+i,
		array.dims[i], 'i'+i);
	    indent++;
	}
	if (is_string(array.basetypename, typelist_array, typelist_size)) {
	    OUT(indent++, cout, "if ((*src)%s != NULL) {\n", index_expr);
	    OUT(indent, cout, "(*dest)%s = (char*)malloc(\nstrlen((*src)%s)+1);\n", 
		index_expr, index_expr);
	    OUT(indent--, cout, "strcpy((*dest)%s, (*src)%s);\n", index_expr, 
		index_expr);
	    OUT(indent, cout, "}\n");
	    indent--;
	} else {
	    OUT(indent, cout, "%s_Copy(&((*dest)%s), &((*src)%s));\n", 
		basetypename, index_expr, index_expr);
	}
	for (i=0; i<array.ndims; i++) {
	    OUT(indent, cout, "}\n");
	    indent--;
	}
	OUT(indent, cout, "}\n");
	free(index_expr);
    
    }
}

static
void code_for_type_routines(cout, hout, typelist_array, typelist_size)
FILE *cout;
FILE *hout;
Type_Description **typelist_array;
int typelist_size;
{
    int i;
  
    for (i=0; i<typelist_size; i++) {
	if (typelist_array[i]->typedef_of == NULL) {
	    /* It is not a typedef, therefore is a complex type */
	    switch(typelist_array[i]->code) {
	    case NT_string:    /* STRINGS */
		break;
	    case NT_struct:    /* STRUCTURES */
		code_for_struct_routines(cout, hout, 
					   typelist_array[i]->name,
					   typelist_array[i]->type->tstruct,
					   typelist_array, typelist_size);
		break;
	    case NT_sequence:  /* SEQUENCES */
		code_for_sequence_routines(cout, hout, 
					   typelist_array[i]->name,
					   typelist_array[i]->type->tsequence,
					   typelist_array, typelist_size);
		break;
	    case NT_enum: 
	    case NT_pre_defined:
		break;
	    case NT_array:
		code_for_array_routines(cout, hout, 
					typelist_array[i]->name,
					typelist_array[i]->type->tarray,
					typelist_array, typelist_size);
		break;
	    case NT_interface:
		break;
	    case NT_not_known:
	    default:
		printf("In code_generate: we aren't dealing with typecode %d\n",
		       typelist_array[i]->code);
		exit(1);
	    }
	} else {
	    code_for_typedef_routines(cout, hout, i, 
				      typelist_array, typelist_size);
	}
    }
    fprintf(hout, "\n");
}

int
is_string(typename, typelist_array, typelist_size)
char *typename;
Type_Description **typelist_array;
int typelist_size;
{
    Type_Description *aux = searchTypeDescription(typename, 
						  typelist_array, 
						  typelist_size);
    assert(aux);

    if (aux->typedef_of != NULL) {
	return is_string(aux->typedef_of->name, typelist_array, typelist_size);
    }
    return (aux->code == NT_string);
}

int
has_dynamic_substructure(typename, typelist_array, typelist_size)
char *typename;
Type_Description **typelist_array;
int typelist_size;
{
    int i;
    Type_Description *aux = searchTypeDescription(typename, 
						  typelist_array, 
						  typelist_size);
    Complex_Type *type;
    assert(aux);

    if (is_object(aux->code, aux->name)) return 1;

    switch(aux->code) {
    case NT_enum:
    case NT_pre_defined:
	return 0;
    case NT_struct:
	type = aux->type;
	if (aux->typedef_of != NULL) {
	    type = aux->typedef_of->type;
	}
	for (i=0; i < type->tstruct.nb_fields; i++) {
	    if (has_dynamic_substructure(type->tstruct.array_field_type[i], 
					 typelist_array, typelist_size)) {
		return 1;
	    }
	}
	return 0;
    case NT_string:
	return 1;
    case NT_sequence:
	return 1;
    case NT_array:
	if (aux->list_subtypes != NULL) {
	    return has_dynamic_substructure(aux->list_subtypes[0]->name, 
					    typelist_array,	typelist_size);
	} else {
	    return 0;
	}
    default:
	assert(0);
    }
    return 0;
}
	  
typedef struct {
  Type_Description **array;
  int max;
  int top;
}Stack,  *Stack_t;


void initStack(Stack_t stack)
{
  stack->max = 100;  /* arbitrary, just large enough for most
		     ** of our applications. If we need more,
		     ** max will be increased by the "insertStack" 
		     ** operation
		     */
  stack->array = (Type_Description**) 
     malloc(sizeof(Type_Description*)*stack->max);
  assert(stack->array!=NULL);
  stack->top = -1;   /* indicates the index of the stack top */
}
  
int searchStack(Stack_t stack, Type_Description *t)
{
  int i;
  for (i=0; i<=stack->top; i++)   /* "<=" is what we need, no accidental
				 ** mistake here */
    if (t == stack->array[i])
      return(1);
  return(0);
} 

void insertStack(Stack_t stack, Type_Description *t)
{
  if (stack->top == stack->max -1) {
    /* we need more space */
    stack->max += 100;
    stack->array = (Type_Description**) 
     realloc(stack->array, sizeof(Type_Description*)*stack->max);
    assert(stack->array!=NULL);
  }
  
  /* now we insert, but we have to make sure there are no repetitions */
  if (searchStack(stack, t) == 0 && t->node_info == undefined) {
   /* is not there yet */
    stack->array[++stack->top] = t;
    /* mark this type as visited */
    t->node_info = visited;
  }
  else {
    /* it has been inserted before. Here we have to watch
    ** for circular definitions in the nested subtypes
    */
    switch (t->node_info) {
    case visited:
      /* putting it twice won't cause any harm. Actually
      ** it is necessary, to guarantee that the subtype tree is
      ** traversed in in-order
      */
    stack->array[++stack->top] = t;
    case marked:
      /* this type is being used more than once, we don't need
      ** to insert it again here in the stack
      */
      break;
    case expanded:
      /* circular definition, since we are expanding nodes in
      ** a deapth first approach. 
      ** This won't happen if the input idl file is properly compiled,
      ** since idl2c will complain about using a type not defined yet.
      */
      printf("There is a circular type definition involving type %s\n",
	     t->name);
      assert(1==0);
      break;
    default:
      printf("Unexpected val. for t->node_info:%d\n",
	     t->node_info);
      printf(" in code_for_nested_subtypes.c:insertStack\n");
      assert(1==0);
    }
  }
}

subtype_list_t
process_stack(stack)
Stack_t            stack;
{
  Type_Description   *t;
  subtype_list_t     list = NULL, tail = NULL, aux;
  int                i;

  while (stack->top >=0) {
    t = stack->array[stack->top];
    switch (t->node_info) {
    case marked:
      /* somewhere between its insertion and we dealing with it now,
      ** it has been included in the list of subtypes.
      ** Nothing to do.
      **/
      stack->top--;
      break;
    case visited:
      /* time to expand this type */
      if (t->nb_subtypes == 0) {
	/* no subtypes, it is done */
	t->node_info = expanded;
	/* don't decrement.  Just mark and hit it again. */
      } else {
	/* inserting t's subtypes */
	for (i=0; i<t->nb_subtypes; i++) 
	  insertStack(stack, t->list_subtypes[i]);
	t->node_info = expanded;
      }
      break;
    case expanded: {
      /* we already have dealt with t's subtypes */
      /* The following is just an assertion to help during
      ** the initial debuging phase. I will eliminate this
      ** soon
      */
      for (i=0; i<t->nb_subtypes; i++)
	assert(t->list_subtypes[i]->node_info==marked);
      t->node_info = marked;

      /* insert in the end of the list of subtypes */
      if (tail==NULL) {
	tail = list = (subtype_list_t) malloc(sizeof(subtype_list));
      }  else {
	tail->next = (subtype_list_t) malloc(sizeof(subtype_list));
	tail = tail->next;
      }
      tail->type = t;
      tail->next = NULL;

      stack->top--;
      break;
    }
    default:
      printf("Unexpected val. for t->node_info:%d\n",
	     t->node_info);
      printf(" in code_for_nested_subtypes.c:get_list_subtypes\n");
      assert(1==0);
    }
  }
  /* clean up node_info before returning the list */
  aux = list;
  while (aux != NULL) {
    aux->type->node_info = undefined;
    aux  = aux->next;
  }
  return(list);
}


subtype_list_t get_subtype_list_op(Type_Description **typelist_array,
				int typelist_size,
				Operation *op)
{
  Argument_list_cell *alist = op->arg_list;
  Type_Description   *t;
  Stack_t            stack;
  subtype_list_t     list;

  if (typelist_array == NULL) return NULL;
  stack = (Stack_t) malloc(sizeof(Stack));
  assert(stack!=NULL);
  initStack(stack);

  while (alist != NULL) {
    /* find the type for this argument */
    t = searchTypeDescription(alist->arg->typename, 
			      typelist_array, typelist_size);
    if (t != NULL) {
      if ((t->code != NT_pre_defined) &&
	  (t->code != NT_enum)) {
	  insertStack(stack, t);
      }
    }
    alist = alist->next;
  }

  list = process_stack(stack);
  free(stack->array);
  free(stack);
  return list;
}


subtype_list_t get_subtype_list_return(Type_Description **typelist_array,
				int typelist_size,
				Operation *op)
{
  Argument_list_cell *alist = op->arg_list;
  Type_Description   *t;
  Stack_t            stack;
  subtype_list_t     list;

  if (typelist_array == NULL) return NULL;
  stack = (Stack_t) malloc(sizeof(Stack));
  assert(stack!=NULL);
  initStack(stack);

  t = searchTypeDescription(op->rettypename, 
			    typelist_array, typelist_size);
  if (t != NULL) {
      if ((t->code != NT_pre_defined) &&
	  (t->code != NT_enum)) {
	  insertStack(stack, t);
      }
  }
  while (alist != NULL) {
    /* find the type for this argument */
      if (alist->arg->direction != D_IN) {
	  t = searchTypeDescription(alist->arg->typename, 
				    typelist_array, typelist_size);
	  if (t != NULL) {
	      if ((t->code != NT_pre_defined) &&
		  (t->code != NT_enum)) {
		  insertStack(stack, t);
	      }
	  }
      }
      alist = alist->next;
  }

  list = process_stack(stack);
  free(stack->array);
  free(stack);
  return list;
}


subtype_list_t get_subtype_list_all_ops(Type_Description **typelist_array,
				int typelist_size,
				Operation_list_cell *op_list)
{
  Argument_list_cell *alist;
  Type_Description   *t;
  Stack_t            stack;
  subtype_list_t     list;

  if (typelist_array == NULL) return NULL;
  stack = (Stack_t) malloc(sizeof(Stack));
  assert(stack!=NULL);
  initStack(stack);

  while (op_list !=NULL) {
      Operation *op = op_list->op;
      t = searchTypeDescription(op->rettypename, 
			    typelist_array, typelist_size);
      if (t != NULL) {
	  if ((t->code != NT_pre_defined) &&
	      (t->code != NT_enum)) {
	      insertStack(stack, t);
	  }
      }
      alist = op->arg_list;
      while (alist != NULL) {
	  /* find the type for this argument */
	  if (alist->arg->direction != D_IN) {
	      t = searchTypeDescription(alist->arg->typename, 
					typelist_array, typelist_size);
	      if (t != NULL) {
		  if ((t->code != NT_pre_defined) &&
		      (t->code != NT_enum)) {
		      insertStack(stack, t);
		  }
	      }
	  }
	  alist = alist->next;
      }
      op_list = op_list->next;
  }

  list = process_stack(stack);
  free(stack->array);
  free(stack);
  return list;
}

subtype_list_t get_subtype_list_state(Type_Description **typelist_array,
				      int typelist_size,
				      Interface *itfc)
{
  Attribute_list_cell *alist = itfc->att_list;
  Type_Description   *t;
  Stack_t            stack;
  subtype_list_t     list;

  if (typelist_array == NULL) return NULL;
  stack = (Stack_t) malloc(sizeof(Stack));
  assert(stack!=NULL);
  initStack(stack);

  while (alist != NULL) {
    /* find the type for this argument */
    t = searchTypeDescription(alist->att->typename, 
			      typelist_array, typelist_size);
    if (t != NULL) {
	if ((t->code != NT_pre_defined) &&
	    (t->code != NT_enum)) {
	    insertStack(stack, t);
	}
    }
    alist = alist->next;
  }

  list = process_stack(stack);
  free(stack->array);
  free(stack);
  return list;
}

void free_subtype_list(subtype_list_t list)
{
    subtype_list_t next;

    while (list != NULL) {
	next = list->next;
	free(list);
	list = next;
    }
}

int count_subtype_list(subtype_list_t list)
{
    int count = 0;
    while (list != NULL) {
	list = list->next;
	count++;
    }
    return count;
}
