/*
 * code_for_complex_types:
 *
 *      generates code for marshalling/unmarshalling complex types
 */

#include <stdio.h>
#include <string.h>
#include "assert.h"
#include "config.h"
#include "unix_defs.h"
#include "code_generation.h"

/* making the variables typelist_array and typelist_size global,
** so we can use it in generate_io_type without having to change
** all the places that already call this function. We only need
** them  now that we are dealing with "typedefs"
*/

extern Type_Description **typelist_array;
extern int typelist_size;

/* auxiliary routines */
void code_for_strings(FILE *cout, FILE *hout, char *name, type_struct tstru);
void code_for_structs(FILE *cout, FILE *hout, char *name,
		      type_struct tstru);
void code_for_sequences(FILE *cout, FILE *hout, char *name, type_sequence seq);
void code_for_arrays(FILE *cout, FILE *hout, char *name, type_array arr);
void code_for_typedefs(FILE *cout, FILE *hout, Type_Description *type);
void generate_io_type(FILE *cout, TypeCode tp, char *type_name);


void code_for_io_formats(Type_Description **typelist_array,
			 int typelist_size,
			 FILE *cout, FILE *hout)
{
  int i;
  
  if (typelist_size) {
    for (i=0; i<typelist_size; i++) {
      if (typelist_array[i]->typedef_of == NULL) {
	/* It is not a typedef, therefore is a complex type */
	switch(typelist_array[i]->code) {
	case NT_string:    /* STRINGS */
	  break;
	case NT_struct:    /* STRUCTURES */
	  code_for_structs(cout, hout, typelist_array[i]->name,
			   typelist_array[i]->type->tstruct);
	  break;
	case NT_sequence:  /* SEQUENCES */
	  code_for_sequences(cout, hout, typelist_array[i]->name,
			     typelist_array[i]->type->tsequence);
	  break;
	case NT_enum: 
	case NT_pre_defined:
	  /* for this we don't need to generate special io formats*/
	  break;
	case NT_array:
	    code_for_arrays(cout, hout, typelist_array[i]->name,
			     typelist_array[i]->type->tarray);
	  break;
	case NT_interface:
	case  NT_not_known:
	default:
	  printf("In code_generate: we aren't dealing with typecode %d\n",
		 typelist_array[i]->code);
	  exit(1);
	}
      } else {
	  if (strcmp(typelist_array[i]->typedef_of->name, "CORBA_Object") != 0) {
	      /* if it's not an object */
	      code_for_typedefs(cout, hout, typelist_array[i]);
	  }
      }
    }
    fprintf(hout, "\n");
  }
  else
    fprintf(hout, "/* No complex types defined, therefore no IOFormats.*/\n");
  
}

enum CORBA_TCKind {
    tk_null		= 0,
    tk_void		= 1,
    tk_short		= 2,
    tk_long		= 3,
    tk_ushort		= 4,
    tk_ulong		= 5,
    tk_float		= 6,
    tk_double		= 7,
    tk_boolean		= 8,
    tk_char		= 9,
    tk_octet		= 10,
    tk_any		= 11,
    tk_TypeCode		= 12,
    tk_Principal	= 13,
    tk_objref		= 14,
    tk_struct		= 15,
    tk_union		= 16,
    tk_enum		= 17,
    tk_string		= 18,
    tk_sequence		= 19,
    tk_array		= 20,
    tk_alias		= 21,
    tk_except		= 22,

    /* these five are OMG-IDL data type extensions */
    tk_longlong		= 23,		/* 94-9-32 Appendix A (+ 2)*/
    tk_ulonglong	= 24,		/* 94-9-32 Appendix A (+ 2)*/
    tk_longdouble	= 25,		/* 94-9-32 Appendix A (+ 2)*/
    tk_wchar		= 26,		/* 94-9-32 Appendix A (+ 2)*/
    tk_wstring		= 27,		/* 94-9-32 Appendix A (+ 2)*/

    /*
    * This symbol is not defined by CORBA 2.0.  It's used to speed up
    * dispatch based on TCKind values, and lets many important ones
    * just be table lookups.  It must always be the last enum value!!
    */
    TC_KIND_COUNT
};

static int
CORBA_tc_of(int idl2otl_tk, char *type_name)
{
    switch(idl2otl_tk) {
    case NT_enum:
	return tk_enum;
    case NT_pre_defined:
    {
	Type_Description *aux;
	char *type = &type_name[6];    /* 6 because we want to skip
				       ** the inicial "CORBA_" part 
				       */
	if (strcmp(type, "null") == 0) return tk_null;
	if (strcmp(type, "void") == 0) return tk_void;
	if (strcmp(type, "short") == 0) return tk_short;
	if (strcmp(type, "long") == 0) return tk_long;
	if (strcmp(type, "unsigned_short") == 0) return tk_ushort;
	if (strcmp(type, "unsigned_long") == 0) return tk_ulong;
	if (strcmp(type, "float") == 0) return tk_float;
	if (strcmp(type, "double") == 0) return tk_double;
	if (strcmp(type, "boolean") == 0) return tk_boolean;
	if (strcmp(type, "char") == 0) return tk_char;
	if (strcmp(type, "octet") == 0) return tk_octet;
	if (strcmp(type, "any") == 0) return tk_any;
	if (strcmp(type, "object") == 0) return tk_objref;
	if (strcmp(type, "Object") == 0) return tk_objref;
	aux = searchTypeDescription(type_name, typelist_array, 
				    typelist_size);
	if (aux != NULL) {
	    assert(aux->typedef_of!=NULL);
	    return CORBA_tc_of (NT_pre_defined, aux->typedef_of->name);
	} else {
	    printf("error in generate_io_type: name of pre_defined type");
	    printf(" (%s)invalid (not even a typedef ...) !!\n", type_name);
	    printf(" (file code_for_io_formats.c)\n");
	    return 0;
	}
    }
    case NT_struct:
	return tk_struct;
    case NT_string:
	return tk_string;
    case NT_sequence:
	return tk_sequence;
    case NT_array:
	return tk_array;
    case NT_interface:
	return tk_objref;
    case NT_not_known:
	return tk_alias;
    default:
	printf("Unknown typecode\n");
	return 0;
    }
}    

void code_for_structs(FILE *cout, FILE *hout, char *name, type_struct tstru)
{
  int i;

  fprintf(cout, "/* Definitions related to type %s (struct).*/\n", name);
  /* defining the IOFormat/ for the string */
  fprintf(cout, "/* type %s (struct)*/\n", name);
  fprintf(cout, "static IOFormat %s_format = NULL;\n", name);
  fprintf(cout, "static IOField %s_fields[] = {\n", name);
  for (i=0; i < tstru.nb_fields; i++) {
    fprintf(cout, "   {\"%s\", \"", tstru.array_field_name[i]);
    generate_io_type(cout, tstru.array_field_typecode[i],
		     tstru.array_field_type[i]);
    fprintf(cout, "\", sizeof(%s),\n     IOOffset(%s*, %s) },\n",
	    tstru.array_field_type[i], name, 
	    tstru.array_field_name[i]);	
  }
  fprintf(cout, "   { NULL, NULL, 0, 0}\n");
  fprintf(cout, "};\n\n"); 
  fprintf(cout, "static long %s_iiop[] = {\n", name);
  for (i=0; i < tstru.nb_fields; i++) {
      if (is_object(tstru.array_field_typecode[i],
		    tstru.array_field_type[i])) {
	  fprintf(cout, "    %d\n", tk_objref);
      } else if (tstru.array_field_typecode[i] == NT_array) {
	  fprintf(cout, "    (long)&");
	  generate_io_type(cout, tstru.array_field_typecode[i],
		    tstru.array_field_type[i]);
	  fprintf(cout, "_array_iiop,\n");
      } else {
	  fprintf(cout, "   %d,\n",CORBA_tc_of(tstru.array_field_typecode[i],
					       tstru.array_field_type[i]));
      }
  }
  fprintf(cout, "0};\n\n"); 
}

void code_for_strings(FILE *cout, FILE *hout, char *name, type_struct tstru)
{
  fprintf(cout, "/* Definitions related to type %s (string).*/\n", name);
  fprintf(cout, "/* type %s (string)*/\n", name);
  fprintf(cout, "static long %s_iiop[] = {\n", name);
  fprintf(cout, "   %d,\n", tk_string);
  fprintf(cout, "0};\n\n"); 
}


void code_for_sequences(FILE *cout, FILE *hout, char *name, type_sequence seq)
{
  fprintf(cout, "/* Definitions related to type %s (seq).*/\n", name);
  /* defining the IOFormat/ for the sequence */
  fprintf(cout, "/* type %s (sequence)*/\n", name);
  fprintf(cout, "static IOFormat %s_format = NULL;\n", name);
  fprintf(cout, "static IOField %s_fields[] = {\n", name);
  fprintf(cout, "   {\"_maximum\", \"integer\", sizeof(CORBA_unsigned_long),\n");
  fprintf(cout, "	IOOffset(%s*, _maximum) },\n", name);
  fprintf(cout, "   {\"_length\", \"integer\", sizeof(CORBA_unsigned_long),\n");
  fprintf(cout, "	IOOffset(%s*, _length) },\n", name);
  fprintf(cout, "   {\"_buffer\", \"");
  generate_io_type(cout, seq.basetypecode, seq.basetypename);
  fprintf(cout, "[_length]\", sizeof(%s),\n", seq.basetypename);
  fprintf(cout, "	IOOffset(%s*, _buffer) },\n", name);
  fprintf(cout, "   { NULL, NULL, 0, 0}\n");
  fprintf(cout, "};\n\n"); 
  fprintf(cout, "static long %s_iiop[] = {-1};\n", name);
}

void code_for_arrays(FILE *cout, FILE *hout, char *name, type_array arr)
{
  int i, element_count = 1;
  fprintf(cout, "/* Definitions related to type %s (array).*/\n", name);
  /* defining the IOFormat/ for the array */
  fprintf(cout, "/* type %s (array)*/\n", name);
  fprintf(cout, "static IOFormat %s_format = NULL;\n", name);
  fprintf(cout, "static IOField %s_fields[] = {\n", name);
  fprintf(cout, "   {\"_buffer\", \"");
  generate_io_type(cout, arr.basetypecode, arr.basetypename);
  for(i=0; i<arr.ndims; i++) {
      fprintf(cout, "[%d]", arr.dims[i]);
      element_count = element_count *arr.dims[i];
  }
  fprintf(cout, "\", sizeof(%s), 0},\n", arr.basetypename);
  fprintf(cout, "   { NULL, NULL, 0, 0}\n");
  fprintf(cout, "};\n\n"); 
  fprintf(cout, "static long %s_iiop[] = {-1};\n", name);
  fprintf(cout, "static struct array_info_s %s_array_iiop = {\n", name);
  fprintf(cout, "	%d, \"%s\", 0/*static */, %d, %d};\n", tk_array,
	  name, CORBA_tc_of(arr.basetypecode, arr.basetypename),
	  element_count);
  fprintf(cout, "static struct array_info_s %s_darray_iiop = {\n", name);
  fprintf(cout, "	%d, \"%s\", 1/* dynamic */, %d, %d};\n", tk_array,
	  name, CORBA_tc_of(arr.basetypecode, arr.basetypename),
	  element_count);
}

int is_object(TypeCode tp, char *type_name) 
{
    switch(tp) {
    case NT_interface:
	return 1;
    case NT_pre_defined: {
	char *type = &type_name[6];    /* 6 because we want to skip
				       ** the inicial "CORBA_" part 
				       */
	if ((strcmp(type, "object") == 0) || (strcmp(type, "Object") == 0)) {
	    return 1;
	} else {
	    return 0;
	}
    }
    default:
	return 0;
    }
}

void generate_io_type(FILE *cout, TypeCode tp, char *type_name)
{
  
  switch(tp) {
  case NT_enum:
    fprintf(cout, "enumeration");
    break;
  case NT_pre_defined: 
    /* We are going to isolate mainly two cases: integer and floats */
    {
	char *type = &type_name[6];    /* 6 because we want to skip
				       ** the inicial "CORBA_" part 
				       */
	if ( (strcmp(type, "short") == 0)
	     || (strcmp(type, "unsigned_short") == 0) 
	     || (strcmp(type, "long") == 0) 
	     || (strcmp(type, "unsigned_long") == 0) 
	     || (strcmp(type, "boolean") == 0) 
	     || (strcmp(type, "octet") == 0) 
	    ) {
	    fprintf(cout, "integer");
	} else {
	    if ( (strcmp(type, "float") == 0) 
		 || (strcmp(type, "double") == 0)) {
		fprintf(cout, "float");
	    } else if (strcmp(type, "char") == 0) {
		fprintf(cout, "char");
	    } else if (strcmp(type, "object") == 0) {
		fprintf(cout, "COBS_object_ref");
	    } else if (strcmp(type, "Object") == 0) {
		fprintf(cout, "COBS_object_ref");
	    } else if (strcmp(type, "any") == 0) {
		fprintf(cout, "any");
	    } else {
		/* all possible pre_defined values have been analysed, therefore
		** we know that this is a "typedef" of a pre defined type.
		** We have to find information about which basic type it refers
		** to */
		Type_Description *aux = searchTypeDescription(type_name, 
							      typelist_array, 
							      typelist_size);
		if (aux != NULL) {
		    assert(aux->typedef_of!=NULL);
		    generate_io_type (cout, tp, aux->typedef_of->name);
		} else {
		    printf("error in generate_io_type: name of pre_defined type");
		    printf(" (%s)invalid (not even a typedef ...) !!\n", type);
		    printf(" (file code_for_io_formats.c)\n");
		    assert(2==1); 
		}
	    }
	}
    }
    break;
  case NT_string:
    fprintf(cout, "string");
    break;
  case NT_array:
  case NT_sequence:
  case NT_struct:
  {
      Type_Description *aux = searchTypeDescription(type_name, 
						    typelist_array, 
						    typelist_size);
      if ((aux != NULL) && (aux->typedef_of != NULL)) {

	  generate_io_type (cout, tp, aux->typedef_of->name);
      } else {
	  fprintf(cout, "%s", type_name);
      }
      break;
  }
  case NT_interface:
      fprintf(cout, "string");
    break;
  case NT_not_known:
  default:
    printf("tstru.array_field_typecode[i] is %d\n", tp);
    assert(FALSE);
  }
}
  
void generate_io_formats_for_operations(char *itfcname, Operation *op,
					FILE *cout, FILE *hout)
{
    Argument_list_cell *alist;
    char               name[255];

    sprintf(name, "%s_%s", itfcname, op->name);

    if (op->hasINarguments || op->hasINOUTarguments) {
	fprintf(cout, "/* arg type for operation %s */\n", name);
	/* generating structure for operation arguments */
	fprintf(cout, "typedef struct %s_params {\n", name);
	alist = op->arg_list;
	while (alist != NULL) {
	    if (alist->arg->direction != D_OUT) {
		if (is_object(alist->arg->typecode, alist->arg->typename)) {
		    fprintf(cout, "    char *%s;\n", alist->arg->name);
		} else if (alist->arg->typecode == NT_array) {
		    fprintf(cout, "    int _%s_size;\n", alist->arg->name);
		    fprintf(cout, "    %s *%s;\n", 
			    alist->arg->type_parameters->tarray.basetypename,
			    alist->arg->name);
		} else {
		    fprintf(cout, "    %s %s;\n", alist->arg->typename,
			    alist->arg->name);
		}
	    }
	    alist = alist->next;

	}
	fprintf(cout, "} *%s_params_p;\n", name);
    }
    if (op_has_internal_return(op)) {
	fprintf(cout, "/* internal return type for operation %s */\n", name);
	/* generating structure for operation arguments */
	fprintf(cout, "typedef struct %s__return {\n", name);
	if (strcmp(op->rettypename, "CORBA_void") != 0) {
	    if (is_object(op->rettypecode, op->rettypename)) {
		fprintf(cout, "    char *return_value;\n");
	    } else if (op->rettypecode == NT_array) {
		fprintf(cout, "    int _return_size;\n");
		fprintf(cout, "    %s *return_value;\n", 
			op->rettype_parameters->tarray.basetypename);
	    } else {
		fprintf(cout, "    %s return_value;\n",
			op->rettypename);
	    }
	}
	alist = op->arg_list;
	while (alist != NULL) {
	    if (alist->arg->direction != D_IN) {
		if (is_object(alist->arg->typecode, alist->arg->typename)) {
		    fprintf(cout, "    char *%s;\n", alist->arg->name);
		} else if (alist->arg->typecode == NT_array) {
		    fprintf(cout, "    int _%s_size;\n", alist->arg->name);
		    fprintf(cout, "    %s *%s;\n", 
			    alist->arg->type_parameters->tarray.basetypename,
			    alist->arg->name);
		} else {
		    fprintf(cout, "    %s %s;\n", alist->arg->typename,
			    alist->arg->name);
		}
	    }
	    alist = alist->next;
	}
	fprintf(cout, "} *%s__return_p;\n", name);
    }
    if (op->hasINarguments || op->hasINOUTarguments) {
	fprintf(cout, "/* IOFormat stuff for operation %s */\n", name);
	fprintf(cout, "static IOFormat %s_format = NULL;\n", name);
	fprintf(cout, "static IOField %s_fields[] = {\n", name);
	alist = op->arg_list;
	while (alist != NULL) {
	    Argument  *arg = alist->arg;
	    if (arg->direction != D_OUT) {
		if (alist->arg->typecode == NT_array) {
		    fprintf(cout, "   { \"_%s_size\", \"integer\", sizeof(int), IOOffset(%s_params_p, _%s_size)},\n",
			    arg->name, name, arg->name);
		    
		    fprintf(cout, "   { \"%s\", \"", arg->name);
		    generate_io_type(cout,
				     arg->type_parameters->tarray.basetypecode,
				     arg->type_parameters->tarray.basetypename);
		    fprintf(cout, "[_%s_size]\", sizeof(%s), IOOffset(%s_params_p, %s) },\n",
			    arg->name,
			    arg->type_parameters->tarray.basetypename, 
			    name, arg->name);
		} else {
		    fprintf(cout, "   { \"%s\", \"", arg->name);
		    if (is_object(alist->arg->typecode, alist->arg->typename)) {
			fprintf(cout, "string\", sizeof(char *), IOOffset(%s_params_p, %s) },\n",
				name, arg->name);
		    } else {
			generate_io_type (cout, arg->typecode, arg->typename);
			fprintf(cout, "\", sizeof(%s), IOOffset(%s_params_p, %s) },\n",
				arg->typename, name, arg->name);
		    }
		}
	    }
	    alist = alist->next;
	}
	fprintf(cout, "   { NULL, NULL, 0, 0}\n");
	fprintf(cout, "};\n\n");
	fprintf(cout, "static long %s_iiop[] = {\n", name);
	alist = op->arg_list;
	while (alist != NULL) {
	    Argument  *arg = alist->arg;
	    if (arg->direction != D_OUT) {
		if (arg->typecode == NT_array) {
		    /* extra entry for index */
		    fprintf(cout, "    %d, (long)&%s_darray_iiop,\n", tk_array, arg->typename);
		} else if (is_object(arg->typecode, arg->typename)) {
		    fprintf(cout, "    %d,\n", tk_objref);
		} else {
		    fprintf(cout, "     %d,\n", CORBA_tc_of(arg->typecode, 
							    arg->typename));
		}
	    }
	    alist = alist->next;
	}
	fprintf(cout, "0};\n\n");
    } 
    
    if (op_has_internal_return(op)) {
	/* operation has a return value or out parameters */
	fprintf(cout, "/* IOFormat for operation %s, internal return */\n",
		name);
	fprintf(cout, "static IOFormat %s_return_format = NULL;\n", name);
	fprintf(cout, "static IOField %s_return_fields[] = {\n", name);
	if (strcmp(op->rettypename, "CORBA_void") != 0) {
	    switch(op->rettypecode) {
	    case NT_array:
		fprintf(cout, "   { \"_return_size\", \"integer\", sizeof(int), 0},\n");
		fprintf(cout, "   { \"return_value\", \"");
		generate_io_type(cout, 
				 op->rettype_parameters->tarray.basetypecode,
				 op->rettype_parameters->tarray.basetypename);
		fprintf(cout, "[_return_size]\", sizeof(%s",
			op->rettype_parameters->tarray.basetypename);
		fprintf(cout, "), IOOffset(%s_%s__return_p, return_value) },\n", 
			itfcname, op->name);
		break;
	    case NT_interface:
	    case NT_pre_defined:
		if (is_object(op->rettypecode, op->rettypename)) {
		    fprintf(cout, "   { \"return_value\", \"");
		    fprintf(cout, "string");
		    fprintf(cout, "\", sizeof(char*), 0 },\n");
		    break;
		}
		/* falling through */
	    default:
		fprintf(cout, "   { \"return_value\", \"");
		generate_io_type (cout, op->rettypecode, op->rettypename);
		fprintf(cout, "\", sizeof(%s), 0 },\n", op->rettypename);
		break;
	    }
	}
	alist = op->arg_list;
	while (alist != NULL) {
	    Argument  *arg = alist->arg;
	    if (arg->direction != D_IN) {
		if (alist->arg->typecode == NT_array) {
		    fprintf(cout, "   { \"_%s_size\", \"integer\", sizeof(int), IOOffset(%s__return_p, _%s_size)},\n",
			    arg->name, name, arg->name);
		    
		    fprintf(cout, "   { \"%s\", \"", arg->name);
		    generate_io_type(cout,
				     arg->type_parameters->tarray.basetypecode,
				     arg->type_parameters->tarray.basetypename);
		    fprintf(cout, "[_%s_size]\", sizeof(%s), IOOffset(%s__return_p, %s) },\n",
			    arg->name,
			    arg->type_parameters->tarray.basetypename, 
			    name, arg->name);
		} else {
		    fprintf(cout, "   { \"%s\", \"", arg->name);
		    generate_io_type (cout, arg->typecode, arg->typename);
		    if (is_object(alist->arg->typecode, alist->arg->typename)) {
			fprintf(cout, "\", sizeof(char *), IOOffset(%s__return_p, %s) },\n",
				name, arg->name);
		    } else {
			fprintf(cout, "\", sizeof(%s), IOOffset(%s__return_p, %s) },\n",
				arg->typename, name, arg->name);
		    }
		}
	    }
	    alist = alist->next;
	}
	fprintf(cout, "   { NULL, NULL, 0, 0}\n");
	fprintf(cout, "};\n\n");
	fprintf(cout, "static long %s_return_iiop[] = {\n", name);
	if (strcmp(op->rettypename, "CORBA_void") != 0) {
	    if (op->rettypecode == NT_array) {
		/* extra entry for index */
		fprintf(cout, "    %d, (long)&", tk_array);
		generate_io_type(cout, op->rettypecode, op->rettypename);
		fprintf(cout, "_darray_iiop,\n");
	    } else if (is_object(op->rettypecode, op->rettypename)) {
		fprintf(cout, "    %d,\n", tk_objref);
	    } else {
		fprintf(cout, "    %d,\n", CORBA_tc_of(op->rettypecode, op->rettypename));
	    }
	}
	alist = op->arg_list;
	while (alist != NULL) {
	    Argument  *arg = alist->arg;
	    if (arg->direction != D_IN) {
		if (is_object(alist->arg->typecode, alist->arg->typename)) {
		    fprintf(cout, "    %d,\n", tk_objref);
		} else {
		    fprintf(cout, "    %d,\n", CORBA_tc_of(alist->arg->typecode, alist->arg->typename));
		}
	    }
	    alist = alist->next;
	}
	fprintf(cout, "0};\n\n");
    }
}


void generate_io_formats_for_state(Interface *itfc, FILE *cout)
{
    Attribute_list_cell *alist = itfc->att_list;

    fprintf(cout, "/* IOFormat stuff for state %s */\n", itfc->name);
    if (mirror_objects && (itfc->att_list != NULL)) 
	fprintf(cout, "static IOFormat %s_state_format = NULL;\n", itfc->name);
    fprintf(cout, "static IOField %s_state_fields[] = {\n", itfc->name);
    while (alist != NULL) {
	Attribute  *att = alist->att;
	fprintf(cout, "   { \"%s\", \"", att->name);
	generate_io_type (cout, att->typecode, att->typename);
	fprintf(cout, "\", sizeof(%s), IOOffset(_IDLtoC_%s_attrib_struct*, %s) },\n",
		att->typename, itfc->name, att->name);
	alist = alist->next;
    }
    fprintf(cout, "   { NULL, NULL, 0, 0}\n");
    fprintf(cout, "};\n\n");

    if (itfc->mirror_name != NULL) {
	fprintf(cout, "IOField %s_mirror_state_fields[] = {\n", itfc->mirror_name);
	alist = itfc->att_list;
	while (alist != NULL) {
	    Attribute  *att = alist->att;
	    if (att->mirror) {
		fprintf(cout, "   { \"%s\", \"", att->name);
		generate_io_type (cout, att->typecode, att->typename);
		fprintf(cout, "\", sizeof(%s), IOOffset(_IDLtoC_%s_attrib_struct*, %s) },\n",
			att->typename, itfc->name, att->name);
	    }
	    alist = alist->next;
	}
	fprintf(cout, "   { NULL, NULL, 0, 0}\n");
	fprintf(cout, "};\n\n");
    }
}

void
code_for_typedefs(FILE* cout, FILE *hout, Type_Description *type)
{
  assert(type->typedef_of!=NULL);
  fprintf(hout, "/* Definitions related to type %s (typedef).*/\n", 
	  type->name);
  /* defining a structure for this data structure */
  fprintf(hout, "typedef struct _%s_rec {\n", type->name);	
  fprintf(hout, "   %s val;\n", type->typedef_of->name);
  fprintf(hout, "} %s_rec, *%s_rec_ptr;\n", type->name, type->name);

  /* declaring IOFormat and IOField for the typedef */
  fprintf(hout, "extern IOFormat %s_format;\n", type->name);
  fprintf(hout, "extern IOField %s_fields[];\n\n", type->name);

  fprintf(cout, "/* Definitions related to type %s (typedef).*/\n", 
	  type->name);
  /* defining the IOFormat/ for the string */
  fprintf(cout, "/* type %s */\n", type->name);
  fprintf(cout, "IOFormat %s_format = NULL;\n", type->name);
  fprintf(cout, "IOField %s_fields[] = {\n", type->name);
  fprintf(cout, "   { \"val\", \"%s\", sizeof(%s),\n", 
	  type->typedef_of->name, type->typedef_of->name);
  fprintf(cout, "     IOOffset(%s_rec_ptr, val) },\n", type->name);
  fprintf(cout, "   { NULL, NULL, 0, 0}\n");
  fprintf(cout, "};\n\n");
/*  if (type->typedef_of->code != NT_string) {
      fprintf(cout, "static int *%s_iiop = &%s_iiop[0];\n", type->name,
	      type->typedef_of->name);
  } else {
      fprintf(cout, "static int %s_iiop[] = {%d};\n", type->name,
	      type->typedef_of->code);
	      }*/
/*  fprintf(cout, "static int %s_iiop[] = {%d};\n", type->name,
    type->typedef_of->code);*/
}
