

#include "config.h"
#ifdef HAVE_WINDOWS_H
#include <windows.h>
#else
#include <sys/time.h>
#endif
#include "io.h"
#include "DE.h"
#include "gen_thread.h"

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/types.h>
#include <signal.h>
#ifdef HAVE_WINDOWS_H
#include <windows.h>
#include <process.h>
#define srand48(s) srand(s)
#define drand48() (((double)rand())/((double)RAND_MAX))
#define lrand48() rand()
#define kill(x,y) 
#else
extern double drand48();
extern long lrand48();
void srand48 ARGS((long seedval));
#include <sys/wait.h>
#endif
#include "useful.h"

static int quiet = 0;

static int do_regression_master_test();

typedef struct _complex_rec {
    double r;
    double i;
} complex, *complex_ptr;

static IOField complex_field_list[] =
{
    {"r", "double", sizeof(double), IOOffset(complex_ptr, r)},
    {"i", "double", sizeof(double), IOOffset(complex_ptr, i)},
    {NULL, NULL, 0, 0}
};

typedef struct _simple_rec {
    int integer_field;
    short short_field;
    long long_field;
    char *string;
    complex complex_field;
    double double_field;
    char char_field;
} simple_rec, *simple_rec_ptr;

static IOField simple_field_list[] =
{
    {"integer_field", "integer",
     sizeof(int), IOOffset(simple_rec_ptr, integer_field)},
    {"short_field", "integer",
     sizeof(short), IOOffset(simple_rec_ptr, short_field)},
    {"long_field", "integer",
     sizeof(long), IOOffset(simple_rec_ptr, long_field)},
    {"string_field", "string",
     sizeof(char *), IOOffset(simple_rec_ptr, string)},
    {"complex_field", "complex",
     sizeof(complex), IOOffset(simple_rec_ptr, complex_field)},
    {"double_field", "float",
     sizeof(double), IOOffset(simple_rec_ptr, double_field)},
    {"char_field", "char",
     sizeof(char), IOOffset(simple_rec_ptr, char_field)},
    {NULL, NULL, 0, 0}
};

static DEFormat simple_format_list[] =
{
    {"complex", complex_field_list},
    {NULL, NULL},
};

typedef struct _big_complex_rec {
    int junk;
    double r;
    double i;
} big_complex, *big_complex_ptr;

static IOField big_complex_field_list[] =
{
    {"junk", "integer", sizeof(int), IOOffset(big_complex_ptr, junk)},
    {"r", "double", sizeof(double), IOOffset(big_complex_ptr, r)},
    {"i", "double", sizeof(double), IOOffset(big_complex_ptr, i)},
    {NULL, NULL, 0, 0}
};

typedef struct _big_simple_rec {
    int junk_field;
    int integer_field;
    short short_field;
    long long_field;
    char *string;
    big_complex complex_field;
    double double_field;
    char char_field;
} big_simple_rec, *big_simple_rec_ptr;

static IOField big_simple_field_list[] =
{
    {"junk_field", "integer",
     sizeof(int), IOOffset(big_simple_rec_ptr, junk_field)},
    {"integer_field", "integer",
     sizeof(int), IOOffset(big_simple_rec_ptr, integer_field)},
    {"short_field", "integer",
     sizeof(short), IOOffset(big_simple_rec_ptr, short_field)},
    {"long_field", "integer",
     sizeof(long), IOOffset(big_simple_rec_ptr, long_field)},
    {"string_field", "string",
     sizeof(char *), IOOffset(big_simple_rec_ptr, string)},
    {"complex_field", "complex",
     sizeof(big_complex), IOOffset(big_simple_rec_ptr, complex_field)},
    {"double_field", "float",
     sizeof(double), IOOffset(big_simple_rec_ptr, double_field)},
    {"char_field", "char",
     sizeof(char), IOOffset(big_simple_rec_ptr, char_field)},
    {NULL, NULL, 0, 0}
};

static DEFormat big_format_list[] =
{
    {"complex", big_complex_field_list},
    {NULL, NULL},
};

#define REPEAT_COUNT 10

static
void
typed_handler(vevent, client_data)
void *vevent;
void *client_data;
{
    simple_rec_ptr event = vevent;
    long sum = 0, scan_sum = 0;
    sum += event->integer_field % 100;
    sum += event->short_field % 100;
    sum += event->long_field % 100;
    sum += ((int)(event->complex_field.r * 100.0)) % 100;
    sum += ((int)(event->complex_field.i * 100.0)) % 100;
    sum += ((int)(event->double_field * 100.0)) % 100;
    sum += event->char_field;
    sum = sum % 100;
    if (event->string) 
	sscanf(event->string, "testing, sum is %ld", &scan_sum);
    if (sum != scan_sum) {
	printf("Received record checksum does not match, expected %ld, got %ld.\n",
	       scan_sum, sum);
    }
    if ((quiet == 0) || (sum != scan_sum)) {
	printf("In %d the handler, event data is :\n", quiet);
	printf("	integer_field = %d\n", event->integer_field);
	printf("	short_field = %d\n", event->short_field);
	printf("	long_field = %ld\n", event->long_field);
	printf("	string = %s\n", event->string ? event->string : "NULL");
	printf("	complex_field.r = %g\n", event->complex_field.r);
	printf("	complex_field.i = %g\n", event->complex_field.i);
	printf("	double_field = %g\n", event->double_field);
	printf("	char_field = %c\n", event->char_field);
    }
    sum = 0;
    if (client_data != NULL) {
	int tmp = *((int*)client_data);
	if (tmp > 0) {
	    *((int*)client_data) = tmp - 1;
	}
    }
}

static char tmp_string[128];

static
void generate_record(event)
simple_rec_ptr event;
{
    long sum = 0;
    event->integer_field = (int) lrand48() % 100;
    sum += event->integer_field;
    event->short_field = (short) lrand48();
    sum += event->short_field % 100;
    event->long_field = (long) lrand48();
    sum += event->long_field % 100;

    event->complex_field.r = drand48();
    sum += ((int)(event->complex_field.r * 100.0)) % 100;
    event->complex_field.i = drand48();
    sum += ((int)(event->complex_field.i * 100.0)) % 100;

    event->double_field = drand48();
    sum += ((int)(event->double_field * 100.0)) % 100;
    event->char_field = lrand48() % 128;
    sum += event->char_field;
    sum = sum % 100;
    sprintf(tmp_string, "testing, sum is %ld\n", sum);
    event->string = tmp_string;
}

static
void generate_big_record(event)
big_simple_rec_ptr event;
{
    long sum = 0;
    event->integer_field = (int) lrand48();
    sum += event->integer_field % 100;
    event->short_field = (short) lrand48();
    sum += event->short_field % 100;
    event->long_field = (long) lrand48();
    sum += event->long_field % 100;

    event->complex_field.r = drand48();
    sum += ((int)(event->complex_field.r * 100.0)) % 100;
    event->complex_field.i = drand48();
    sum += ((int)(event->complex_field.i * 100.0)) % 100;

    event->double_field = drand48();
    sum += ((int)(event->double_field * 100.0)) % 100;
    event->char_field = lrand48() % 128;
    sum += event->char_field;
    sum = sum % 100;
    sprintf(tmp_string, "testing, sum is %ld\n", sum);
    event->string = tmp_string;
}

static int regression = 0;
static int poll_thread_running = 0;

static void
do_source_side(de, chan, done_flag, count, use_big)
DExchange de;
EChannel chan;
int *done_flag;
int count;
int use_big;
{
    ECSourceHandle handle;
    int ret = 0;

    if (quiet == 0) printf("chan is %lx\n", (long)chan);
    if (use_big) {
	handle = ECsource_typed_subscribe(chan, big_simple_field_list, 
					  big_format_list);
    } else {
	handle = ECsource_typed_subscribe(chan, simple_field_list, 
					  simple_format_list);
    }
    while (!ret) {
/*	if (drand48() < 0.1) {*/
	    if (use_big) {
		big_simple_rec a;
		generate_big_record(&a);
		if (quiet == 0)
		    printf("I'm submitting %ld\n", a.long_field);
		ECsubmit_typed_event(handle, &a);
	    } else {
		simple_rec a;
		generate_record(&a);
		if (quiet == 0)
		    printf("I'm submitting %ld\n", a.long_field);
		ECsubmit_typed_event(handle, &a);
	    }
	    if (count != -1) {
		count--;
	    }
/*	}*/
	if (! poll_thread_running) {
	    if (regression) {
		DExchange_poll_and_handle_timeout(de, 0, 100000);
	    } else {
		DExchange_poll_and_handle_timeout(de, 0, 1000000);
	    }
	} else {
	    thr_thread_yield();
	}
	if (done_flag && *done_flag) ret++;
	if (count != -1) {
	    if (count == 0) ret++;
	}	    
    }
    ECcancel_source_subscribe(handle);
}

static pid_t sink_proc = 0;

static void
fail_and_die(signal)
int signal;
{
    fprintf(stderr, "Channel test failed to complete in reasonable time\n");
    if (sink_proc != 0) {
	kill(sink_proc, 9);
    }
    exit(1);
}

int
main(argc, argv)
int argc;
char **argv;
{
    EChannel chan;
    double r = drand48();
    int consumer = 0;
    int both = 0;
    int regression_master = 1;
    DExchange de;

    srand48(1);
    if (argc <= 1) {
	/* 
	 * we're the first guy, create a couple of event channel and 
	 * print out the IDs. 
	 */
	EChannel chan;

	de = DExchange_create();
	DExchange_listen(de, 0);
	chan = EChannel_typed_create(de, simple_field_list, 
				     simple_format_list);
	if (quiet == 0) printf("  channel ID is %s\n", ECglobal_id(chan));

	while (1) {
	    DExchange_poll_and_handle_timeout(de, 0, 100000);
	}
    }
    while (argv[1] && (argv[1][0] == '-')) {
	if (argv[1][1] == 'c') {
	    r = .9;
	    regression_master = 0;
	} else if (argv[1][1] == 's') {
	    r = 0;
	    regression_master = 0;
	} else if (argv[1][1] == 'b') {
	    both++;
	    regression_master = 0;
	} else if (argv[1][1] == 'q') {
	    quiet++;
	} else if (argv[1][1] == 'v') {
	    quiet--;
	} else if (argv[1][1] == 'r') {
	    regression++;
	    quiet++;
	}
	argv++;
    }
    if (regression && regression_master) {
	return do_regression_master_test();
    }
    de = DExchange_create();
    DExchange_listen(de, 0);
    consumer = (r > 0.5);
    if (quiet <= 0) printf("r is %g\n", r);
    chan = EChannel_open(de, argv[1]);
    if (chan == NULL) {
	printf("Failed to contact channel %s\n", argv[1]);
	exit(1);
    }
    if (both) {
	EChannel chan2;
	chan = EChannel_open(de, argv[1]);
	chan2 = EChannel_open(de, argv[1]);
	(void) ECsink_typed_subscribe(chan, simple_field_list, 
				      simple_format_list,
				      typed_handler, NULL);
	if (quiet <= 0) printf("chan is %lx, chan2 is %lx\n", (long)chan, (long)chan2);
	/* do source side forever */
	do_source_side(de, chan, NULL, -1, 0);
    } else if (consumer) {
	int repeat_count = REPEAT_COUNT;
	if (regression) {
#ifdef HAVE_WINDOWS_H
	  SetTimer(NULL, 5, 1000, (TIMERPROC) fail_and_die);
#else
	  struct sigaction sigact;
	  sigact.sa_flags = 0;
	  sigact.sa_handler = fail_and_die;
	  sigemptyset(&sigact.sa_mask);
	  sigaddset(&sigact.sa_mask, SIGALRM);
	  sigaction(SIGALRM, &sigact, NULL);
	  alarm(300);
#endif
	}
	if (quiet <= 0) printf("I'm a consumer\n");
	(void) ECsink_typed_subscribe(chan, simple_field_list,
				      simple_format_list,
				      typed_handler, 
				      regression ? &repeat_count : NULL);
	while (repeat_count != 0) {
	    DExchange_poll_and_handle(de, 1);
	}
    } else {
	if (quiet <= 0) printf("I'm a supplier\n");
	do_source_side(de, chan, NULL, -1, 0);
    }
    return 0;
}

static int stop_poll_thread = 0;
#ifdef USE_PTHREADS
static int
poll_thread(de)
void *de;
{
    while (!stop_poll_thread) {
	DExchange_poll_and_handle((DExchange)de, 1);
    }
    return 0;
}
#endif

static
pid_t
run_subprocess(args, channel_id)
char **args;
{
#ifdef HAVE_WINDOWS_H
    int child;
    child = _spawnv(_P_NOWAIT, "./master_test.exe", args);
    if (child == -1) {
	perror("spawnv");
    }
    return child;
#else
    pid_t child = fork();
    if (child == 0) {
	/* I'm the child */
	execv("./master_test", args);
    }
    return child;
#endif
}

static int
do_regression_master_test()
{
    DExchange de;
    EChannel chan;

#ifdef HAVE_WINDOWS_H
    SetTimer(NULL, 5, 1000, (TIMERPROC) fail_and_die);
#else
    struct sigaction sigact;
    sigact.sa_flags = 0;
    sigact.sa_handler = fail_and_die;
    sigemptyset(&sigact.sa_mask);
    sigaddset(&sigact.sa_mask, SIGALRM);
    sigaction(SIGALRM, &sigact, NULL);
    alarm(300);
#endif

#ifdef USE_PTHREADS
    gen_pthread_init();
#endif

    de = DExchange_create();
    DExchange_listen(de, 0);

#ifdef USE_PTHREADS
    poll_thread_running++;
    thr_fork(poll_thread, de);
#endif
    srand48(1);

    chan = EChannel_typed_create(de, simple_field_list, simple_format_list);
    
    if (quiet <= 0) printf("  channel ID is %s\n", ECglobal_id(chan));

    /* local tests */
    {
	int repeat_count = REPEAT_COUNT;
	int repeat_count1 = REPEAT_COUNT / 2;
	int repeat_count2 = REPEAT_COUNT - repeat_count1;
	ECSinkHandle handle;

	handle = ECsink_typed_subscribe(chan, simple_field_list,
					simple_format_list, typed_handler,
					&repeat_count);
	do_source_side(de, chan, NULL, repeat_count1, 0);
	do_source_side(de, chan, NULL, repeat_count2, 1);
	if (repeat_count != 0) {
	    fprintf(stderr, "Not all in-process events delivered, test 1\n");
	} else {
	    if (quiet < 2) 
		fprintf(stderr, "Passed in-process delivery tests\n");
	}	
	ECcancel_sink_subscribe(handle);
    }

    /* multiple process tests */
    {
	int repeat_count1 = REPEAT_COUNT / 2;
	int repeat_count2 = REPEAT_COUNT - repeat_count1;
	char *args[] = {"master_test", "-r", "-c", NULL, NULL};
	int i;
	int exit_state;

	args[sizeof(args)/sizeof(char*) -2] = ECglobal_id(chan);
	sink_proc = run_subprocess(args);

	/* give him time to start */
	for(i=0; i< 4; i++) {
#ifndef USE_PTHREADS
	    int j;
	    for (j=0; j< 25 ; j++)
		DExchange_poll_and_handle_timeout(de, 0, 100000);
#else
	    sleep(1);
#endif
	}

	do_source_side(de, chan, NULL, repeat_count1, 0);
	do_source_side(de, chan, NULL, repeat_count2, 1);
#ifdef HAVE_WINDOWS_H
	if (_cwait(&exit_state, sink_proc, 0) == -1) {
	    perror("cwait");
	}
	if (exit_state == 0) {
	    if (quiet < 2) 
		printf("Passed single remote sink test\n");
	} else {
	    printf("Single remote sink exit with status %d\n",
		   exit_state);
	}
#else
	if (waitpid(sink_proc, &exit_state, 0) == -1) {
	    perror("waitpid");
	}
	if (WIFEXITED(exit_state)) {
	    if (WEXITSTATUS(exit_state) == 0) {
		if (quiet < 2) 
		    printf("Passed single remote sink test\n");
	    } else {
		printf("Single remote sink exit with status %d\n",
		       WEXITSTATUS(exit_state));
	    }
	} else if (WIFSIGNALED(exit_state)) {
	    printf("Single remote sink died with signal %d\n",
		   WTERMSIG(exit_state));
	}
#endif
    }
    stop_poll_thread++;
    EChannel_destroy(chan);
    DExchange_close(de);
    DExchange_free(de);
    return 0;
}
