
#include "config.h"

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <sys/types.h>
#ifndef HAVE_WINDOWS_H
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <sys/socket.h>
#ifdef HAVE_SYS_RESOURCE_H
#include <sys/resource.h>
#endif
#include <netinet/in.h>
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#include <unistd.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#include <signal.h>
#else
#include <windows.h>
#include <winsock.h>
#endif
#include "io.h"
#include "DE.h"
#ifdef HAVE_GEN_THREAD_H
#include "gen_thread.h"
#endif
#include "de_internal.h"
#include "group_formats.h"
#include "comm_group.h"
#include "useful.h"

#ifndef PRINTF_DEFINED
extern int printf ARGS((const char *,...));
#endif
#ifndef FPRINTF_DEFINED
extern int fprintf ARGS((FILE *, const char *,...));
#endif
#ifndef STRDUP_DEFINED
extern char *strdup();
#endif
#if !defined(GETPEERNAME_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int getpeername ARGS((int, struct sockaddr *, int *));
#endif
extern void free ARGS((void *));
extern time_t time();
#if !defined(GETHOSTNAME_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int gethostname ARGS((char *name, int namelen));
#endif
#if defined(RLIMIT_CPU) && !defined(SETRLIMIT_DEFINED)
extern int setrlimit ARGS((int resource, struct rlimit * rlp));
#endif

/* global variables */
static DExchange de = NULL;
static int debug_flag = 0;

int return_format_id = -1;
int return_matching_format_id = -1;
int provisional_use_format_id = -1;

typedef struct group {
    char *user_name;
    char *application_name;
    char *group_type;
    int expiration;		/* in seconds, 0=never */
    time_t create_time;
    char *group_id;
    char *host;
    int port;
    void *data;
    int data_len;
} *group_ptr;

typedef struct group_list {
    group_ptr group;
    struct group_list *next;
} *group_list;

group_list global_group_list = NULL;

time_t current_time = 0;

static void assign_unique_group_id();

void
free_group(grp)
group_ptr grp;
{
    free(grp->user_name);
    free(grp->application_name);
    free(grp->group_type);
    free(grp->group_id);
    free(grp->host);
    free(grp);
}

void
free_group_list(list)
group_list list;
{
    while (list != NULL) {
	group_list next = list->next;
	free(list);
	list = next;
    }
}

group_list
append_group_list(list, new)
group_list list;
group_list new;
{
    group_list orig = list;
    if (list == NULL)
	return new;

    while (list->next != NULL) {
	list = list->next;
    }
    list->next = new;
    return orig;
}

void
dump_group_list(list)
group_list list;
{
    printf("Group list includes: ");
    while (list != NULL) {
	printf(" \"%s\",", list->group->group_id);
	list = list->next;
    }
    printf("\n");
}

int
count_group_list(list)
group_list list;
{
    int count = 0;
    while (list != NULL) {
	list = list->next;
	count++;
    }
    return count;
}

group_list
copy_group_list(list)
group_list list;
{
    group_list copy_base = NULL;
    group_list last = NULL;
    while (list != NULL) {
	group_list next = list->next;
	group_list tmp = (group_list) malloc(sizeof(struct group_list));
	tmp->next = NULL;
	tmp->group = list->group;
	if (last != NULL) {
	    last->next = tmp;
	} else {
	    copy_base = tmp;
	}
	last = tmp;
	list = next;
    }
    return copy_base;
}

group_list
search_group_list(list, compar_func, data)
group_list list;
int (*compar_func) ();
void *data;
{
    group_list return_list = NULL;
    group_list last = NULL;
    while (list != NULL) {
	if ((compar_func) (list->group, data)) {
	    group_list tmp = (group_list) malloc(sizeof(struct group_list));

	    tmp->group = list->group;
	    if (last != NULL) {
		last->next = tmp;
		last = tmp;
	    } else {
		return_list = tmp;
		last = tmp;
	    }
	    last->next = NULL;
	}
	list = list->next;
    }
    return return_list;
}

void
return_contact(grp, de, dep)
group_ptr grp;
DExchange de;
DEPort dep;
{
    return_contact_msg msg;

    if (grp == NULL) {
	msg.host = NULL;
	msg.port = -1;
	msg.group_id = NULL;
	if (debug_flag) {
	    printf("No valid group selected\n");
	}
    } else {
	msg.host = grp->host;
	msg.port = grp->port;
	msg.group_id = grp->group_id;
	if (debug_flag) {
	    printf("Returning group\n");
	    printf("\tgroup_id = %s\n", grp->group_id);
	    printf("\tcontact point = host %s - port %d\n", grp->host,
		   grp->port);
	}
    }
    DEport_write_data(dep, return_format_id, &msg);
}

void
eliminate_expired_groups()
{
    group_list tmp_list = global_group_list;
    current_time = time(NULL);
    if (tmp_list == NULL)
	return;

    while (tmp_list->next != NULL) {
	group_ptr grp = tmp_list->next->group;
	if (current_time >= grp->create_time + grp->expiration) {
	    group_list next = tmp_list->next->next;
	    free_group(grp);
	    free(tmp_list->next);
	    tmp_list->next = next;
	} else {
	    tmp_list = tmp_list->next;
	}
    }

    tmp_list = global_group_list;
    if (tmp_list != NULL) {
	if (current_time >= tmp_list->group->create_time +
	    tmp_list->group->expiration) {
	    global_group_list = tmp_list->next;
	    free(tmp_list);
	    return;
	}
    }
}

int
group_id_eq(grp, name)
group_ptr grp;
char *name;
{
    if (grp->group_id != NULL) {
	return (strcmp(grp->group_id, name) == 0);
    } else {
	return 0;
    }
}

int
user_name_eq(grp, name)
group_ptr grp;
char *name;
{
    if (grp->user_name != NULL) {
	return (strcmp(grp->user_name, name) == 0);
    } else {
	return 0;
    }
}

int
application_name_eq(grp, name)
group_ptr grp;
char *name;
{
    if (grp->application_name != NULL) {
	return (strcmp(grp->application_name, name) == 0);
    } else {
	return 0;
    }
}

int
application_name_neq(grp, name)
group_ptr grp;
char *name;
{
    if (grp->application_name != NULL) {
	return (strcmp(grp->application_name, name) != 0);
    } else {
	return 1;
    }
}

int
group_type_eq(grp, name)
group_ptr grp;
char *name;
{
    if (grp->group_type != NULL) {
	return (strcmp(grp->group_type, name) == 0);
    } else {
	return 0;
    }
}

int
group_type_neq(grp, name)
group_ptr grp;
char *name;
{
    if (grp->group_type != NULL) {
	return (strcmp(grp->group_type, name) != 0);
    } else {
	return 1;
    }
}

int
all_eq(grp, match)
group_ptr grp;
group_ptr match;
{
    if (match->user_name != NULL) {
	if (grp->user_name == NULL)
	    return 0;
	if (strcmp(grp->user_name, match->user_name) != 0) {
	    return 0;
	}
    } else if (grp->user_name != NULL) {
	return 0;
    }
    if (match->application_name != NULL) {
	if (grp->application_name == NULL)
	    return 0;
	if (strcmp(grp->application_name, match->application_name) != 0) {
	    return 0;
	}
    } else if (grp->application_name != NULL) {
	return 0;
    }
    if (match->group_type != NULL) {
	if (grp->group_type == NULL)
	    return 0;
	if (strcmp(grp->group_type, match->group_type) != 0) {
	    return 0;
	}
    } else if (grp->group_type != NULL) {
	return 0;
    }
    return 1;
}

static group_list
search_for_match(group_id, user_name, application_name, group_type)
char *group_id;
char *user_name;
char *application_name;
char *group_type;
{
    group_list current_list, tmp_list;
    struct group tmp_grp;

    if (debug_flag) {
	printf("\nBefore Expiration ");
	dump_group_list(global_group_list);
    }
    eliminate_expired_groups();

    if (debug_flag) {
	printf("Requesting contact point for group:\n");
	if (group_id != NULL) {
	    printf("\tGroup id = %s\n", group_id);
	}
	if (user_name != NULL) {
	    printf("\tUser name = %s\n", user_name);
	}
	if (application_name != NULL) {
	    printf("\tApplication name = %s\n", application_name);
	}
	if (group_type != NULL) {
	    printf("\tGroup type = %s\n", group_type);
	}
    }
    if (group_id != NULL) {
	/* if group_id is specified, *MUST* match exactly */
	current_list = search_group_list(global_group_list, group_id_eq,
					 (void *) group_id);
	return current_list;
    }
    current_list = copy_group_list(global_group_list);

    if (debug_flag) {
	printf("Initial ");
	dump_group_list(current_list);
    }
    tmp_list = current_list;
    tmp_grp.user_name = user_name;
    tmp_grp.application_name = application_name;
    tmp_grp.group_type = group_type;
    current_list = search_group_list(global_group_list, all_eq,
				     (void *) &tmp_grp);
    free_group_list(tmp_list);

    if (debug_flag) {
	printf("Final match ");
	dump_group_list(current_list);
    }
    return current_list;
}

static int
handle_setup_group(de, dep, format_id, data, record_len, param)
DExchange de;
DEPort dep;
int format_id;
void *data;
int record_len;
void *param;
{
    setup_group_msg_ptr msg = (setup_group_msg_ptr) data;
    group_ptr new_grp;
    group_list tmp, match_list;

    if (msg->host == NULL) {
	return_contact(NULL, de, dep);
	return 1;
    }
    /* 
     *  As a step to eliminate race conditions in group inquiries and setup,
     *  a repeated setup within 10 seconds for the same group identifiers 
     *  will not be accepted.
     */
    match_list = search_for_match(NULL, msg->user_name, msg->application_name,
				  msg->group_type);
    if (match_list != NULL) {
	if (match_list->next != NULL) {
	    /* take most recent (should be first) */
	    if ((time(NULL) - match_list->group->create_time) < 10) {
		return_contact(NULL, de, dep);
		return 1;
	    }
	} else {
	    if ((time(NULL) - match_list->group->create_time) < 10) {
		return_contact(NULL, de, dep);
		return 1;
	    }
	}
    }
    new_grp = (group_ptr) malloc(sizeof(struct group));
    if (msg->user_name != NULL) {
	new_grp->user_name = strdup(msg->user_name);
    } else {
	new_grp->user_name = NULL;
    }
    if (msg->application_name != NULL) {
	new_grp->application_name = strdup(msg->application_name);
    } else {
	new_grp->application_name = NULL;
    }
    if (msg->group_type != NULL) {
	new_grp->group_type = strdup(msg->group_type);
    } else {
	new_grp->group_type = NULL;
    }
    new_grp->expiration = msg->expiration;
    new_grp->create_time = time(NULL);
    new_grp->host = strdup(msg->host);
    new_grp->port = msg->port;
    new_grp->data = NULL;
    new_grp->data_len = 0;

    if ((new_grp->expiration > 604800) ||
	(new_grp->expiration < 0)) {
	/* 
	 * if they ask for negative time or for more than a week,
	 * give them a day, arbitrarily.
	 */
	new_grp->expiration = 86400;	/* = 60*60*24 */
    }
    assign_unique_group_id(new_grp, global_group_list);

    tmp = (group_list) malloc(sizeof(struct group_list));
    tmp->group = new_grp;
    tmp->next = global_group_list;
    global_group_list = tmp;
    if (debug_flag) {
	printf("\nCreating group:\n");
	if (new_grp->user_name != NULL) {
	    printf("\tUser name = %s\n", new_grp->user_name);
	} else {
	    printf("\tUser name = <NULL>\n");
	}
	if (new_grp->application_name != NULL) {
	    printf("\tApplication name = %s\n", new_grp->application_name);
	} else {
	    printf("\tApplication name = <NULL>\n");
	}
	if (new_grp->group_type != NULL) {
	    printf("\tGroup type = %s\n", new_grp->group_type);
	} else {
	    printf("\tGroup type = NULL\n");
	}
	printf("\tvalidity duration = %d (seconds)\n", new_grp->expiration);
	printf("\tcreate time = %s", ctime(&new_grp->create_time));
	printf("\tassigned group_id = %s\n", new_grp->group_id);
	printf("\tcontact point = host %s - port %d\n", new_grp->host,
	       new_grp->port);

	printf("\nAfter creation ");
	dump_group_list(global_group_list);

    }
    return_contact(new_grp, de, dep);
    return 1;
}

static int
handle_remove_group(de, dep, format_id, data, record_len, param)
DExchange de;
DEPort dep;
int format_id;
void *data;
int record_len;
void *param;
{
    remove_group_msg_ptr msg = (remove_group_msg_ptr) data;
    group_list match_list;

    match_list = search_for_match(msg->group_id, NULL, NULL, NULL);

    if (match_list != NULL) {
	match_list->group->expiration = 0;
    }
    eliminate_expired_groups();

    return 1;
}

void
gen_unique_group_id(grp, base, list)
group_ptr grp;
char *base;
group_list list;
{
    char *tmp_area = malloc(strlen(base) + 10);
    int counter = 1;
    while (1) {			/* forever (will return below) */
	group_list tmp;
	sprintf(tmp_area, "%s_%d", base, counter++);
	tmp = search_group_list(list, group_id_eq, (void *) tmp_area);
	if (tmp == NULL) {
	    grp->group_id = strdup(tmp_area);
	    return;
	} else {
	    free_group_list(tmp);
	}
    }
}

static void
assign_unique_group_id(grp, list)
group_ptr grp;
group_list list;
{
    if (grp->application_name != NULL) {
	gen_unique_group_id(grp, grp->application_name, list);
    } else if (grp->user_name != NULL) {
	gen_unique_group_id(grp, grp->user_name, list);
    } else if (grp->group_type != NULL) {
	gen_unique_group_id(grp, grp->group_type, list);
    } else {
	gen_unique_group_id(grp, "anon", list);
    }
}

static int
handle_init_contact(de, dep, format_id, data, record_len, param)
DExchange de;
DEPort dep;
int format_id;
void *data;
int record_len;
void *param;
{
    init_contact_msg_ptr msg = (init_contact_msg_ptr) data;
    group_list match_list;

    match_list = search_for_match(msg->group_id, msg->user_name,
				  msg->application_name, msg->group_type);

    if (match_list == NULL) {
	return_contact(NULL, de, dep);
    } else {
	if (match_list->next != NULL) {
	    /* take most recent (should be first) */
	    return_contact(match_list->group, de, dep);
	} else {
	    return_contact(match_list->group, de, dep);
	}
    }
    free_group_list(match_list);
    return 1;
}

static int
handle_assoc_group_info(de, dep, format_id, data, record_len, param)
DExchange de;
DEPort dep;
int format_id;
void *data;
int record_len;
void *param;
{
    assoc_group_info_msg_ptr msg = (assoc_group_info_msg_ptr) data;
    group_list match_list;

    match_list = search_for_match(msg->group_name, NULL, NULL, NULL);

    if (match_list == NULL) {
	return_contact(NULL, de, dep);
    } else {
	if (match_list->group->data != NULL) {
	    free(match_list->group->data);
	}
	match_list->group->data = malloc(msg->data_len);
	memcpy(match_list->group->data, msg->data, msg->data_len);
	match_list->group->data_len = msg->data_len;
	return_contact(match_list->group, de, dep);
    }
    free_group_list(match_list);
    return 1;
}

static char **
parse_list(string)
char *string;
{
    int count = 0;
    char *end;
    char *tmp;
    char **result = (char **) malloc(sizeof(char *));

    if (string == NULL) {
	return NULL;
    }
    end = string + strlen(string);
    result[count] = NULL;
    for (tmp = string; tmp < end; tmp++) {
	if (*tmp == ',') {
	    *(tmp--) = 0;
	}
	while (((*tmp == ' ') || (*tmp == '	')) && (tmp >= string)) {
	    /* swallow preceeding white */
	    *(tmp--) = 0;
	}
    }
    tmp = string;
    while (tmp < end) {
	if ((*tmp == ' ') || (*tmp == '	') || (*tmp == 0)) {
	    /* skip white */
	    tmp++;
	} else {
	    result = (char **) realloc(result, (count + 2) * sizeof(char *));
	    result[count] = tmp;
	    tmp += strlen(tmp);
	    count++;
	}
    }
    result[count] = NULL;
    return result;
}

static int
handle_find_matching(de, dep, format_id, data, record_len, param)
DExchange de;
DEPort dep;
int format_id;
void *data;
int record_len;
void *param;
{
    find_matching_msg_ptr msg = (find_matching_msg_ptr) data;
    ret_group_struct return_msg;
    group_list match_result = NULL;
    group_list tmp_list;
    char **app_list;
    char **type_list;
    group_list user_list_result;
    int app, type, count;

    if (debug_flag) {
	printf("\nBefore Expiration ");
	dump_group_list(global_group_list);
    }
    eliminate_expired_groups();

    if (debug_flag) {
	printf("Got find_matching request:\n");
	if (msg->user_name != NULL) {
	    printf("\tUser name = %s\n", msg->user_name);
	}
	if (msg->app_list != NULL) {
	    printf("\tApplication list = %s\n", msg->app_list);
	}
	if (msg->type_list != NULL) {
	    printf("\tGroup type list = %s\n", msg->type_list);
	}
    }
    if (debug_flag) {
	printf("Initial ");
	dump_group_list(global_group_list);
    }
    app_list = parse_list(msg->app_list);
    type_list = parse_list(msg->type_list);

    user_list_result = search_group_list(global_group_list, user_name_eq,
					 (void *) msg->user_name);
    for (app = 0; (app_list != NULL) && (app_list[app] != NULL); app++) {
	group_list app_list_result;
	if (strcmp(app_list[app], "*") != 0) {
	    app_list_result = search_group_list(user_list_result,
						application_name_eq,
						(void *) app_list[app]);
	} else {
	    /* wildcard */
	    int prev_app;
	    app_list_result = copy_group_list(user_list_result);
	    for (prev_app = 0; prev_app < app; prev_app++) {

		group_list tmp = app_list_result;
		app_list_result = search_group_list(app_list_result,
						    application_name_neq,
					    (void *) app_list[prev_app]);
		free_group_list(tmp);
	    }
	}
	for (type = 0; (type_list != NULL) && (type_list[type] != NULL); type++) {
	    group_list result;
	    if (strcmp(type_list[type], "*") != 0) {
		result = search_group_list(app_list_result, group_type_eq,
					   (void *) type_list[type]);
	    } else {
		/* wildcard */
		int prev_type;
		result = copy_group_list(app_list_result);
		for (prev_type = 0; prev_type < type; prev_type++) {
		    group_list tmp = result;
		    result = search_group_list(result, group_type_neq,
					  (void *) type_list[prev_type]);
		    free_group_list(tmp);
		}
	    }
	    match_result = append_group_list(match_result, result);
	}
	free_group_list(app_list_result);
    }
    free_group_list(user_list_result);

    if (debug_flag) {
	printf("Final match ");
	dump_group_list(match_result);
    }
    return_msg.count = count_group_list(match_result);
    return_msg.list = (comm_group_list) malloc(return_msg.count * sizeof(comm_group_struct));
    tmp_list = match_result;
    count = 0;
    while (tmp_list != NULL) {
	return_msg.list[count].user_name = tmp_list->group->user_name;
	return_msg.list[count].application_name = tmp_list->group->application_name;
	return_msg.list[count].group_type = tmp_list->group->group_type;
	return_msg.list[count].group_id = tmp_list->group->group_id;
	return_msg.list[count].host = tmp_list->group->host;
	return_msg.list[count].port = tmp_list->group->port;
	return_msg.list[count].info_len = tmp_list->group->data_len;
	return_msg.list[count].info = tmp_list->group->data;
	if (debug_flag) {
	    printf("\tapp - \"%s\", type - \"%s\",  ID=%s on %s,%d\n",
		   return_msg.list[count].application_name,
		   return_msg.list[count].group_type,
		   return_msg.list[count].group_id,
		   return_msg.list[count].host,
		   return_msg.list[count].port);
	}
	count++;
	tmp_list = tmp_list->next;
    }
    DEport_write_data(dep, return_matching_format_id, &return_msg);
    free_group_list(match_result);
    return 1;
}

#ifdef HAS_STRUCT_HOSTENT
typedef struct host_info {
    struct in_addr ip_addr;
    time_t intro_time;
} *host_info_p;

static host_info_p hostlist = NULL;
static int host_count = 0;

static time_t
get_time_for_host(ip_addr, hostname)
struct in_addr ip_addr;
char *hostname;
{
    int i;
    if (hostlist == NULL) {
	hostlist = malloc(sizeof(struct host_info));
    }
    for (i = 0; i < host_count; i++) {
	if (memcmp(&ip_addr, &hostlist[i].ip_addr, sizeof(struct in_addr)) == 0) {
	    return hostlist[i].intro_time;
	}
    }
    hostlist = realloc(hostlist, sizeof(struct host_info) * (host_count + 1));
    memcpy(&hostlist[host_count].ip_addr, &ip_addr, sizeof(struct in_addr));
    hostlist[host_count].intro_time = time(NULL);
    host_count++;
    return hostlist[host_count - 1].intro_time;
}

static char *postfix = GROUP_SERVICE_DOMAIN;
#endif

/* 5 days of free use of our daemons for every host */
#define GRACE_PERIOD_SEC 60 * 60 * 24 * 5

static void
out_domain_rejection(de, dep)
DExchange de;
DEPort dep;
{
    struct sockaddr sock_addr;
    int sock_len = sizeof(sock_addr);
    IOFile from_port = DEport_from_port(dep);

    if (from_port == NULL) {
	DEport_close(dep);
	return;
    }
    if (getpeername(file_id_IOfile(from_port), &sock_addr,
		    &sock_len) == 0) {
	if (sock_addr.sa_family == AF_INET) {
#ifdef HAS_STRUCT_HOSTENT
	    struct hostent *host;
	    struct sockaddr_in *in_sock = (struct sockaddr_in *) &sock_addr;
	    host = gethostbyaddr((char *) &in_sock->sin_addr,
				 sizeof(struct in_addr), AF_INET);
	    if (host != NULL) {
		char **alias;
		time_t intro_time;
		int postlen = strlen(postfix);
		int hostlen = strlen(host->h_name);
		/* 
		 * Check to see if our GROUP_SERVICE_DOMAIN is a postfix
		 * of the hostname.  If not, we'll reject the host as being
		 * outside of our service domain.
		 */
		if (hostlen > postlen) {
		    if (strcmp(&host->h_name[hostlen - postlen], postfix) == 0) {
			/* good host */
			return;
		    }
		} else {
		    if (strcmp(host->h_name, "localhost") == 0) {
			return;
		    }
		}
		for (alias = host->h_aliases; *alias != 0; alias++) {
		    if (hostlen > postlen) {
			if (strcmp(alias[hostlen - postlen], postfix) == 0) {
			    /* good host */
			    return;
			}
		    }
		}

		intro_time = get_time_for_host(in_sock->sin_addr, host->h_name);
		if ((time(NULL) - intro_time) < GRACE_PERIOD_SEC) {
		    provisional_use_msg msg;
		    msg.domain = postfix;
		    msg.time = intro_time + GRACE_PERIOD_SEC - time(NULL);
		    DEport_write_data(dep, provisional_use_format_id, &msg);
		    return;
		}
		printf("Rejecting out-domain host %s, aliases: ", host->h_name);
		for (alias = host->h_aliases; *alias != 0; alias++) {
		    printf(" %s", *alias);
		}
		printf("\n");
	    }
#else
	    if (debug_flag) printf("Out domain rejection impossible\n");
	    return;
#endif
	}
    }
    if (debug_flag) {
	printf("out domain rejection rejecting host \n");
    }
    /* must not be a good host, close it down ... */
    DEport_close(dep);
}

static void
die_with_error(sig)
int sig;
{
    fprintf(stderr, "Group server not responding.  Timeout.\n");
    exit(1);
}

extern int
main(argc, argv)
int argc;
char **argv;
{
    char env[128];
    int i, no_fork = 0;

    de = DExchange_create();
    de->enable_debug = FALSE;

    for (i = 1; i < argc; i++) {
	if (strcmp(argv[i], "-no_fork") == 0) {
	    no_fork++;
	}
    }
    /* substitute this host name for the group server host */
    strcpy(env, "GROUP_SERVER_HOST=");
    gethostname(&env[strlen(env)], sizeof(env) - strlen(env));
    putenv(env);

#ifndef HAVE_WINDOWS_H
    signal(SIGALRM, die_with_error);
    alarm(30);
#endif
    if (group_server_test()) {
	/* already running */
	exit(0);
#ifndef HAVE_WINDOWS_H
    } else {
	/* make really, really certain there's no group_server running */
	FILE *group_server_pid_file = fopen("/tmp/group_server_pid", "r");
	alarm(0);
	if (group_server_pid_file != NULL) {
	    long server_pid;
	    if (fscanf(group_server_pid_file, "%lx", &server_pid) == 1) {
		if (kill(server_pid, 0) == 0) {
		    fprintf(stderr, "Group Server %lx not responding, but still running\n",
			    server_pid);
		    exit(0);
		}
	    }
	    fclose(group_server_pid_file);
	}
	if (!no_fork) {
	    if (fork() != 0) {
		/* I'm the parent, return now */
		exit(0);
	    }
	}
	group_server_pid_file = fopen("/tmp/group_server_pid", "w");
	if (group_server_pid_file) {
	    fprintf(group_server_pid_file, "%lx\n", (long) getpid());
	    fclose(group_server_pid_file);
	}
#endif
    }
#ifdef RLIMIT_CPU
    {
	struct rlimit limit;
	limit.rlim_cur = limit.rlim_max = 300;
	if (setrlimit(RLIMIT_CPU, &limit) != 0)
	    perror("rlimit cpu");
	limit.rlim_cur = limit.rlim_max = 1024 * 1024 * 10;
	if (setrlimit(RLIMIT_DATA, &limit) != 0)
	    perror("rlimit data");
	limit.rlim_cur = limit.rlim_max = 1024 * 1024 * 10;
	if (setrlimit(RLIMIT_STACK, &limit) != 0)
	    perror("rlimit stack");
#ifdef RLIMIT_VMEM
	limit.rlim_cur = limit.rlim_max = 1024 * 1024 * 10;
	if (setrlimit(RLIMIT_VMEM, &limit) != 0)
	    perror("rlimit vmem");
#endif
    }
#endif
    DExchange_set_forward(de, DENever_Forward);
    DExchange_set_block_default(de, FALSE);
    if (argc > 1) {
	debug_flag = 1;
    }
    if (DExchange_listen(de, PORT) != 0) {
	fprintf(stderr, "DExchange listen failed\n");
	exit(1);
    }
    DExchange_register_format(de, "setup group", setup_group_msg_flds);
    DExchange_register_function(de, "setup group", handle_setup_group, NULL);

    DExchange_register_format(de, "remove group", remove_group_msg_flds);
    DExchange_register_function(de, "remove group", handle_remove_group, NULL);

    DExchange_register_format(de, "init contact", init_contact_msg_flds);
    DExchange_register_function(de, "init contact", handle_init_contact, NULL);

    DExchange_register_format(de, "assoc group info",
			      assoc_group_info_msg_flds);
    DExchange_register_function(de, "assoc group info",
				handle_assoc_group_info, NULL);

    DExchange_register_format(de, "provisional use",
			      provisional_use_msg_flds);
    provisional_use_format_id = DEget_format_id(de, "provisional use");

    DExchange_register_format(de, "return contact", return_contact_msg_flds);
    return_format_id = DEget_format_id(de, "return contact");

    DExchange_register_format(de, "comm_group_struct", comm_group_struct_flds);
    DExchange_register_format(de, "return_matching", return_matching_msg_flds);
    return_matching_format_id = DEget_format_id(de, "return_matching");

    DExchange_register_format(de, "find_matching", find_matching_msg_flds);
    DExchange_register_function(de, "find_matching", handle_find_matching, NULL);

    if (strcmp(GROUP_SERVICE_DOMAIN, "") != 0) {
	DExchange_register_open_handler(de, out_domain_rejection);
    }
    DExchange_register_close_handler(de, NULL);

    while (1) {
	DExchange_poll_and_handle(de, 1);
    }
}
