
#include "config.h"
#include <stdio.h>
#ifdef HAVE_WINDOWS_H
#include <windows.h>
#include <winsock.h>
#else
#include <unistd.h>
#ifdef HAVE_PWD_H
#include "pwd.h"
#endif
#endif
#include <string.h>
#include "useful.h"

#include "io.h"
#include "DE.h"
#include "comm_group.h"
#include "group_formats.h"

int DEgroup_debug = 0;

typedef struct _group_server {
    DExchange de;
    DEPort dep;
} group_server_struct, *group_server;

static char *group_server_host = NULL;

static char *get_comm_user();
static char *get_comm_app_name();
static char *get_comm_comm_type();
static void setup_on_exit_handler ARGS((char *group_id, char *host, int port));
static void handle_prov_msg ARGS((group_server gs,
				  provisional_use_msg_ptr msg));
extern void *(*DEmalloc) ARGS((int size));
extern void *(*DErealloc) ARGS((void *block, int size));
extern void (*DEfree) ARGS((void *block));


static void
default_error_routine()
{
    fprintf(stderr, "Failed to contact Group server on host %s, port %d\n",
	    group_server_host, PORT);
    fprintf(stderr, " For more info on servers see:\n");
    fprintf(stderr, "  http://www.cc.gatech.edu/systems/projects/MOSS/servers.html\n");
    exit(1);
}

static void (*group_error_routine) () = default_error_routine;


extern void
set_comm_group_error_handler(error_routine)
group_error_routine_type error_routine;
{
    group_error_routine = error_routine;
}

static group_server
connect_comm_group_server(do_err, do_fallback)
int do_err;
int do_fallback;
{
    group_server gs = (group_server) DEmalloc(sizeof(group_server_struct));
    gs->de = NULL;
    gs->dep = NULL;

    if (group_server_host == NULL) {	/* environment override */
	group_server_host = getenv("GROUP_SERVER_HOST");
    }
    if (group_server_host == NULL) {
	group_server_host = GROUP_SERVER_HOST;	/* from configure */
    }
    if (DEgroup_debug)
	printf("Contacting group server\n");
    gs->de = DExchange_create();
    DEset_debug_flags(gs->de, DENo_Debug, 1);
    DEControlList_set_control_style(DExchange_get_control_list(gs->de),
				    DESingleThreaded);
    gs->dep = DExchange_initiate_conn(gs->de, group_server_host,
				      PORT, TRUE);
    if ((gs->dep == NULL) && do_fallback) {
	gs->dep = DExchange_initiate_conn(gs->de, "marquesas.cc.gatech.edu",
					  PORT, TRUE);
    }
    if (gs->dep == NULL) {
	if (do_err && (group_error_routine != NULL)) {
	    group_error_routine();
	}
	return NULL;
    }
    return gs;
}

static void
shutdown_comm_group_server(gs)
group_server gs;
{
    DExchange_close(gs->de);
    DExchange_free(gs->de);
    DEfree(gs);
}

static void
handle_prov_msg(gs, msg)
group_server gs;
provisional_use_msg_ptr msg;
{
    static int warned = 0;

    if (warned)
	return;

    warned++;
    fprintf(stderr, "The group_server daemon on %s serves the domain \"%s\"\n",
	    DEport_host_name(gs->dep), msg->domain);
    fprintf(stderr, " See http://www.cc.gatech.edu/systems/projects/MOSS/servers.html for more info.\n");
    fprintf(stderr, "  Temporary use allowed for ");
    if (msg->time > 60 * 60 * 48) {	/* 2 days */
	int days = msg->time / (60 * 60 * 24);
	fprintf(stderr, "%d more days.\n", days);
    } else if (msg->time > 60 * 60 * 2) {
	int hours = msg->time / (60 * 60);
	fprintf(stderr, "%d more hours.\n", hours);
    } else {
	int mins = msg->time / 60;
	fprintf(stderr, "%d more minutes.\n", mins);
    }
}

extern char *
setup_comm_group(user_name, application_name, group_type, expiration_time,
		 host, port)
char *user_name;
char *application_name;
char *group_type;
int expiration_time;
char *host;
int port;
{
    setup_group_msg msg;

    msg.application_name = get_comm_app_name();
    if (msg.application_name == NULL) {
	msg.application_name = application_name;
    }
    msg.group_type = get_comm_comm_type();
    if (msg.group_type == NULL) {
	msg.group_type = group_type;
    }
    return setup_specific_comm_group(msg.application_name,
				     msg.group_type, expiration_time, host,
				     port);
}

extern char *
setup_specific_comm_group(application_name, group_type,
			  expiration_time, host, port)
char *application_name;
char *group_type;
int expiration_time;
char *host;
int port;
{
    setup_group_msg msg;
    return_contact_msg_ptr return_msg;
    int setup_format_id;
    int return_format_id;
    int prov_use_format_id;
    group_server gs;
    char *group_id = NULL;

    msg.user_name = get_comm_user();
    msg.application_name = application_name;
    msg.group_type = group_type;
    msg.expiration = expiration_time;
    msg.host = host;
    msg.port = port;
    if (host == NULL) {
	return NULL;
    }
    if (DEgroup_debug)
	printf("Setup comm group %s, %s, %s\n",
	       msg.user_name, msg.application_name,
	       msg.group_type);
    gs = connect_comm_group_server(1, 1);
    DExchange_register_format(gs->de, "setup group", setup_group_msg_flds);
    DExchange_register_format(gs->de, "return contact", return_contact_msg_flds);
    DExchange_register_format(gs->de, "provisional use",
			      provisional_use_msg_flds);
    prov_use_format_id = DEget_format_id(gs->de, "provisional use");

    DEport_set_format_block(gs->dep, "provisional use", FALSE);
    DEport_set_format_block(gs->dep, "return contact", FALSE);
    if (gs == NULL) {
	return NULL;
    }
    setup_format_id = DEget_format_id(gs->de, "setup group");
    DEport_write_data(gs->dep, setup_format_id,
		      &msg);

    return_msg =
	(return_contact_msg_ptr) DEport_read_data(gs->dep, &return_format_id);

    if (return_format_id == prov_use_format_id) {
	handle_prov_msg(gs, (provisional_use_msg_ptr) return_msg);
	return_msg = (return_contact_msg_ptr)
	    DEport_read_data(gs->dep, &return_format_id);
    }
    if ((!return_msg) || (return_msg->group_id == NULL)) {
	if ((!return_msg) && (group_error_routine != NULL)) {
	    group_error_routine();
	}
	shutdown_comm_group_server(gs);
	return NULL;
    } else {
	group_id = strdup(return_msg->group_id);
	shutdown_comm_group_server(gs);
	setup_on_exit_handler(group_id, host, port);
	return group_id;
    }
}

extern void
assoc_group_info(group_name, data, data_len)
char *group_name;
void *data;
int data_len;
{
    assoc_group_info_msg msg;
    return_contact_msg_ptr return_msg;
    int assoc_format_id;
    int return_format_id;
    int prov_use_format_id;
    group_server gs;

    msg.group_name = group_name;
    msg.data = data;
    msg.data_len = data_len;

    if (DEgroup_debug)
	printf("Assoc group info - group \"%s\"\n",
	       msg.group_name);
    gs = connect_comm_group_server(1, 1);
    DExchange_register_format(gs->de, "assoc group info",
			      assoc_group_info_msg_flds);
    DExchange_register_format(gs->de, "return contact",
			      return_contact_msg_flds);
    DExchange_register_format(gs->de, "provisional use",
			      provisional_use_msg_flds);
    prov_use_format_id = DEget_format_id(gs->de, "provisional use");

    DEport_set_format_block(gs->dep, "provisional use", FALSE);
    DEport_set_format_block(gs->dep, "return contact", FALSE);
    if (gs == NULL) {
	return;
    }
    assoc_format_id = DEget_format_id(gs->de, "assoc group info");
    DEport_write_data(gs->dep, assoc_format_id,
		      &msg);

    return_msg =
	(return_contact_msg_ptr) DEport_read_data(gs->dep, &return_format_id);

    if (return_format_id == prov_use_format_id) {
	handle_prov_msg(gs, (provisional_use_msg_ptr) return_msg);
	return_msg = (return_contact_msg_ptr)
	    DEport_read_data(gs->dep, &return_format_id);
    }
    if ((!return_msg) || (return_msg->group_id == NULL)) {
	if ((!return_msg) && (group_error_routine != NULL)) {
	    group_error_routine();
	}
	shutdown_comm_group_server(gs);
    } else {
	shutdown_comm_group_server(gs);
    }
}

extern int
group_server_present()
{
    group_server gs = connect_comm_group_server(0, 1);
    if (gs == NULL)
	return 0;
    shutdown_comm_group_server(gs);
    return 1;
}

extern int
group_server_test()
{
    /* no fallback for testing locality */
    group_server gs = connect_comm_group_server(0, 0);
    if (gs == NULL)
	return 0;
    shutdown_comm_group_server(gs);
    return 1;
}

extern int
init_comm_group_contact(user_name, application_name, group_type, host_p, port_p)
char *user_name;
char *application_name;
char *group_type;
char **host_p;
int *port_p;
{
    init_contact_msg msg;
    return_contact_msg_ptr return_msg;
    int init_format_id;
    int return_format_id;
    int prov_use_format_id;
    group_server gs;
    int return_value;

    msg.group_id = NULL;
    if (user_name && (user_name[0] == 0)) {
	msg.user_name = NULL;
    } else {
	msg.user_name = user_name;
    }
    if (application_name && (application_name[0] == 0)) {
	msg.application_name = NULL;
    } else {
	msg.application_name = application_name;
    }
    if (group_type && (group_type[0] == 0)) {
	msg.group_type = NULL;
    } else {
	msg.group_type = group_type;
    }

    if (DEgroup_debug)
	printf("Init comm group contact %s, %s, %s\n",
	       msg.user_name, msg.application_name,
	       msg.group_type);
    gs = connect_comm_group_server(1, 1);

    if (gs == NULL) {
	return 0;
    }
    DExchange_register_format(gs->de, "init contact", init_contact_msg_flds);
    DExchange_register_format(gs->de, "return contact", return_contact_msg_flds);
    DExchange_register_format(gs->de, "provisional use",
			      provisional_use_msg_flds);
    prov_use_format_id = DEget_format_id(gs->de, "provisional use");

    DEport_set_format_block(gs->dep, "provisional use", FALSE);
    DEport_set_format_block(gs->dep, "return contact", FALSE);
    init_format_id = DEget_format_id(gs->de, "init contact");
    DEport_write_data(gs->dep, init_format_id, &msg);

    return_msg =
	(return_contact_msg_ptr) DEport_read_data(gs->dep, &return_format_id);

    if (return_format_id == prov_use_format_id) {
	handle_prov_msg(gs, (provisional_use_msg_ptr) return_msg);
	return_msg = (return_contact_msg_ptr)
	    DEport_read_data(gs->dep, &return_format_id);
    }
    if (return_msg == NULL) {
	if (group_error_routine != NULL) {
	    group_error_routine();
	}
	return -1;
    }
    if (host_p != NULL) {
	if (return_msg->host != NULL) {
	    *host_p = strdup(return_msg->host);
	} else {
	    *host_p = NULL;
	}
    }
    if (port_p != NULL) {
	*port_p = return_msg->port;
    }
    return_value = ((return_msg->host != NULL) && (return_msg->port != -1));
    shutdown_comm_group_server(gs);

    return return_value;
}

extern int
init_specific_comm_group_contact(group_id, host_p, port_p)
char *group_id;
char **host_p;
int *port_p;
{
    init_contact_msg msg;
    return_contact_msg_ptr return_msg;
    int init_format_id;
    int return_format_id;
    int prov_use_format_id;
    group_server gs;
    int return_value;

    if (group_id && (group_id[0] == 0)) {
	msg.group_id = NULL;
    } else {
	msg.group_id = group_id;
    }
    msg.user_name = NULL;
    msg.application_name = NULL;
    msg.group_type = NULL;

    if (DEgroup_debug)
	printf("init specific group contact %s\n",
	       group_id);
    gs = connect_comm_group_server(1, 1);
    if (gs == NULL) {
	return 0;
    }
    DExchange_register_format(gs->de, "init contact", init_contact_msg_flds);
    DExchange_register_format(gs->de, "return contact", return_contact_msg_flds);
    DExchange_register_format(gs->de, "provisional use",
			      provisional_use_msg_flds);
    prov_use_format_id = DEget_format_id(gs->de, "provisional use");

    DEport_set_format_block(gs->dep, "provisional use", FALSE);
    DEport_set_format_block(gs->dep, "return contact", FALSE);
    init_format_id = DEget_format_id(gs->de, "init contact");
    DEport_write_data(gs->dep, init_format_id, &msg);

    return_msg =
	(return_contact_msg_ptr) DEport_read_data(gs->dep, &return_format_id);

    if (return_format_id == prov_use_format_id) {
	handle_prov_msg(gs, (provisional_use_msg_ptr) return_msg);
	return_msg = (return_contact_msg_ptr)
	    DEport_read_data(gs->dep, &return_format_id);
    }
    if (return_msg == NULL) {
	if (group_error_routine != NULL) {
	    group_error_routine();
	}
	return -1;
    }
    if (host_p != NULL) {
	if (return_msg->host != NULL) {
	    *host_p = strdup(return_msg->host);
	} else {
	    *host_p = NULL;
	}
    }
    if (port_p != NULL) {
	*port_p = return_msg->port;
    }
    return_value = ((return_msg->host != NULL) && (return_msg->port != -1));
    shutdown_comm_group_server(gs);

    return return_value;
}


static char *comm_group_user = NULL;

#ifdef HAVE_WINDOWS_H
#define MAX_USER_ID_SIZE	40
static char buffer[MAX_USER_ID_SIZE];
#endif

static
char *
get_comm_user()
{
    if (comm_group_user == NULL) {
	comm_group_user = getenv("COMM_GROUP_USER");

	/* if not, try getlogin() */
	if (comm_group_user == NULL) {
#ifdef HAVE_WINDOWS_H
	    DWORD len = MAX_USER_ID_SIZE;
	    if (GetUserName(buffer, &len)) {
		comm_group_user = buffer;
	    }
#elif defined(HAVE_GETLOGIN)
	    /* should use getlogin_r() if available */
	    if (getlogin() != NULL) {
		comm_group_user = strdup(getlogin());
	    }
	    /* if not, try getpwuid(getuid()) */
	    if (comm_group_user == NULL) {
		struct passwd *pw = getpwuid(getuid());
		if ((pw != NULL) && (pw->pw_name != NULL)) {
		    comm_group_user = strdup(pw->pw_name);
		}
	    }
#endif
	}
    }
    return comm_group_user;
}

static char *comm_group_app_name = NULL;

extern
void
comm_group_set_app_name(app_name)
char *app_name;
{
    char *last_slash = strrchr(app_name, '/');
    if (last_slash != NULL) {
	comm_group_app_name = strdup(++last_slash);
    } else {
	comm_group_app_name = app_name;
    }
}

static
char *
get_comm_app_name()
{
    if (comm_group_app_name == NULL) {
	comm_group_app_name = getenv("COMM_GROUP_APP_NAME");
    }
    return comm_group_app_name;
}

static char *comm_group_app_list = NULL;

extern void
comm_group_set_app_list(list)
char *list;
{
    comm_group_app_list = list;
}

static
char *
get_comm_group_app_list()
{
    if (comm_group_app_list == NULL) {
	comm_group_app_list = getenv("COMM_GROUP_APP_LIST");
    }
    if (comm_group_app_list == NULL) {
	comm_group_app_list = get_comm_app_name();
    }
    return comm_group_app_list;
}


static char *comm_group_comm_type = NULL;

extern
void
comm_group_set_comm_type(comm_type)
char *comm_type;
{
    comm_group_comm_type = comm_type;
}

static
char *
get_comm_comm_type()
{
    if (comm_group_comm_type == NULL) {
	comm_group_comm_type = getenv("COMM_GROUP_COMM_TYPE");
    }
    return comm_group_comm_type;
}

static char *comm_group_type_list = NULL;

extern
void
comm_group_set_type_list(list)
char *list;
{
    comm_group_type_list = list;
}

static
char *
get_comm_group_type_list()
{
    if (comm_group_type_list == NULL) {
	comm_group_type_list = getenv("COMM_GROUP_TYPE_LIST");
    }
    if (comm_group_type_list == NULL) {
	comm_group_type_list = get_comm_comm_type();
    }
    return comm_group_type_list;
}

comm_group_return
matching_comm_groups(app_list, type_list)
char *app_list;
char *type_list;
{
    char *comm_user = get_comm_user();
    find_matching_msg msg;
    comm_group_return return_msg;
    int find_format_id;
    int return_format_id;
    int prov_use_format_id;
    group_server gs;

    if (app_list == NULL) {
	app_list = get_comm_group_app_list();
    }
    if (type_list == NULL) {
	get_comm_group_type_list();
    }
    if (DEgroup_debug)
	printf("matching comm groups %s, %s\n",
	       app_list, type_list);
    gs = connect_comm_group_server(1, 1);
    if (gs == NULL) {
	return NULL;
    }
    DExchange_register_format(gs->de, "return contact", return_contact_msg_flds);
    DExchange_register_format(gs->de, "comm_group_struct", comm_group_struct_flds);
    DExchange_register_format(gs->de, "return_matching", return_matching_msg_flds);
    DExchange_register_format(gs->de, "find_matching", find_matching_msg_flds);

    DExchange_register_format(gs->de, "provisional use",
			      provisional_use_msg_flds);
    prov_use_format_id = DEget_format_id(gs->de, "provisional use");

    DEport_set_format_block(gs->dep, "provisional use", FALSE);
    DEport_set_format_block(gs->dep, "return contact", FALSE);
    DEport_set_format_block(gs->dep, "return_matching", FALSE);
    msg.user_name = comm_user;
    msg.app_list = app_list;
    msg.type_list = type_list;
    find_format_id = DEget_format_id(gs->de, "find_matching");
    DEport_write_data(gs->dep, find_format_id, &msg);

    return_msg =
	(comm_group_return) DEport_read_data(gs->dep, &return_format_id);

    if (return_format_id == prov_use_format_id) {
	handle_prov_msg(gs, (provisional_use_msg_ptr) return_msg);
	return_msg = (comm_group_return)
	    DEport_read_data(gs->dep, &return_format_id);
    }
    if (return_msg == NULL) {
	if (group_error_routine != NULL) {
	    group_error_routine();
	}
	return NULL;
    }
    DEtake_buffer(gs->de, return_msg);

    shutdown_comm_group_server(gs);

    return return_msg;
}

extern DEPort
DExchange_initiate_first(de, groups, default_block_flag)
DExchange de;
comm_group_return groups;
int default_block_flag;
{
    DEPort dep = NULL;
    int i;
    for (i = 0; i < groups->count; i++) {
	dep = DExchange_initiate_conn(de, groups->list[i].host,
				      groups->list[i].port,
				      default_block_flag);
	if (dep != NULL) {
	    return dep;
	}
    }
    return NULL;
}

#ifdef HAVE_ON_EXIT
extern int on_exit();

static void
group_on_exit_handler(status, arg)
int status;
void *arg;
{
    remove_group_msg_ptr tmp_msg = (remove_group_msg_ptr) arg;
    int remove_format_id;
    group_server gs;

    gs = connect_comm_group_server(0, 1);
    if (gs == NULL) {
	return;
    }
    DExchange_register_format(gs->de, "remove group", remove_group_msg_flds);

    remove_format_id = DEget_format_id(gs->de, "remove group");
    DEport_write_data(gs->dep, remove_format_id, tmp_msg);

    shutdown_comm_group_server(gs);

    DEfree(tmp_msg->group_id);
    DEfree(tmp_msg->host);
    DEfree(tmp_msg);
}

static void
setup_on_exit_handler(group_id, host, port)
char *group_id;
char *host;
int port;
{
    remove_group_msg_ptr tmp_msg =
    (remove_group_msg_ptr) DEmalloc(sizeof(remove_group_msg));
    tmp_msg->group_id = strdup(group_id);
    tmp_msg->host = strdup(host);
    tmp_msg->port = port;
    if (on_exit(group_on_exit_handler, tmp_msg) != 0) {
	fprintf(stderr, "On Exit failed\n");
    }
}
#else
#ifdef HAVE_ATEXIT
typedef struct _linked_list {
    remove_group_msg_ptr msg;
    struct _linked_list *next;
} *linked_list_ptr;

static linked_list_ptr list_of_msgs = NULL;

static void
group_atexit_handler()
{
    int remove_format_id;
    group_server gs;

    gs = connect_comm_group_server(0, 1);
    if (gs == NULL) {
	return;
    }
    DExchange_register_format(gs->de, "remove group", remove_group_msg_flds);
    remove_format_id = DEget_format_id(gs->de, "remove group");
    while (list_of_msgs != NULL) {
	linked_list_ptr tmp_next = list_of_msgs->next;
	DEport_write_data(gs->dep, remove_format_id, list_of_msgs->msg);

	DEfree(list_of_msgs->msg->group_id);
	DEfree(list_of_msgs->msg->host);
	DEfree(list_of_msgs->msg);
	DEfree(list_of_msgs);

	list_of_msgs = tmp_next;
    }

    shutdown_comm_group_server(gs);

}

static void
setup_on_exit_handler(group_id, host, port)
char *group_id;
char *host;
int port;
{
    static int already_registered_handler = 0;

    remove_group_msg_ptr tmp_msg = DEmalloc(sizeof(remove_group_msg));
    linked_list_ptr list_elem = DEmalloc(sizeof(struct _linked_list));

    tmp_msg->group_id = strdup(group_id);
    tmp_msg->host = strdup(host);
    tmp_msg->port = port;
    list_elem->msg = tmp_msg;

    /* 
     * CRITICAL SECTION -- UNPROTECTED ACCESS TO SHARED DATA
     * Potential problem if threaded.
     * Danger is slight and failure is tolerable (a group doesn't get removed).
     */
    list_elem->next = list_of_msgs;
    list_of_msgs = list_elem;
    /* End of Critical Section */

    if (already_registered_handler == 0) {
	if (atexit((void (*)(void)) group_atexit_handler) != 0) {
	    fprintf(stderr, "Atexit failed\n");
	}
	already_registered_handler = 1;
    }
}

#else
 /* 
  * no on_exit and no atexit.  Nothing we can do to remove registered
  * groups 
  */
static void
setup_on_exit_handler(group_id, host, port)
char *group_id;
char *host;
int port;
{
    /* nothing to work with */
}

#endif
#endif
