
#include "config.h"

#include <malloc.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>

#include "io.h"
#include "DE.h"
#include "comm_group.h"

static int connect_format_id = -1;
static int init_format_id = -1;

typedef struct conn_msg_s {
    char *server_name;
    char *hostname;
    int port;
} connect_msg, *connect_msg_p;

IOField COBS_connect_msg_flds[] =
{
    {"server name", "string",
     sizeof(char *), IOOffset(connect_msg_p, server_name)},
    {"host name", "string",
     sizeof(char *), IOOffset(connect_msg_p, hostname)},
    {"port number", "integer",
     sizeof(int), IOOffset(connect_msg_p, port)},
    {NULL, NULL, 0, 0}
};

typedef struct init_msg_s {
    int rank;
} init_msg, *init_msg_p;

IOField COBS_init_msg_flds[] =
{
    {"rank", "integer",
     sizeof(int), IOOffset(init_msg_p, rank)},
    {NULL, NULL, 0, 0}
};

static int handle_connect_msg ARGS((DExchange de, DEPort dep,
				    int format_id, void *data,
				    int record_len, void *client_data));

typedef struct server_struct {
    char *server_name;
    char *in_app_name;
    int connection_fd;
    DEPort dep;
} server_s, *server_p;


static server_p server_list = NULL;

static
void
add_server_to_list(dep, name)
DEPort dep;
char *name;
{
    int server_count = 0;
    while ((server_list[server_count].server_name != NULL) &&
	   (strcmp(server_list[server_count].server_name, name) != 0)) {
	server_count++;
    }

    if (server_list[server_count].server_name == NULL) {
	/* reached the end of the list */
	server_list = (server_p) realloc(server_list,
				  sizeof(server_s) * (server_count + 2));
	server_list[server_count + 1].server_name = NULL;
	server_list[server_count + 1].in_app_name = NULL;
	server_list[server_count + 1].connection_fd = -1;
	server_list[server_count + 1].dep = NULL;
    }
    if (server_list[server_count].server_name == NULL) {
	server_list[server_count].server_name = strdup(name);
    }
    if (dep != NULL) {
	server_list[server_count].connection_fd =
	    file_id_IOfile(DEport_from_port(dep));
    }
    server_list[server_count].dep = dep;

}

static
 server_p
create_initial_server_list(in_app_name, de)
char *in_app_name;
DExchange de;
{
    server_p list = (server_p) malloc(sizeof(server_s) * 2);
    char server_name[256];

    sprintf(server_name, "%s__%d", DExchange_host_name(de),
	    DExchange_inet_port(de));
    list[0].server_name = strdup(server_name);
    list[0].in_app_name = strdup(in_app_name);
    list[0].connection_fd = -1;
    list[0].dep = NULL;
    list[1].server_name = NULL;
    list[1].in_app_name = NULL;
    list[1].connection_fd = 0;
    list[1].dep = NULL;
    return list;
}

static
int
handle_init_msg(de, dep, format_id, data, record_len, client_data)
DExchange de;
DEPort dep;
int format_id;
void *data;
int record_len;
void *client_data;
{
    init_msg_p msg = (init_msg_p) data;
    printf("We are rank %d\n", msg->rank);
    return 0;
}


static
int
handle_connect_msg(de, dep, format_id, data, record_len, client_data)
DExchange de;
DEPort dep;
int format_id;
void *data;
int record_len;
void *client_data;
{
    connect_msg_p msg = (connect_msg_p) data;
    char *hub_name = msg->server_name;
    char *contact_host = msg->hostname;
    int contact_port = msg->port;
    int i;

    printf("Got a message to connect to %s,%d (%s)\n", contact_host,
	   contact_port, hub_name);
    i = 0;
    while (server_list[i].server_name != NULL) {
	if (strcmp(hub_name, server_list[i].server_name) == 0) {
	    printf("Already talking to him\n");
	    return 0;
	}
	i++;
    }
    if (DExchange_initiate_conn(de, contact_host, contact_port, 1) == NULL) {
	printf("Failed to get in touch with %s\n", hub_name);
    }
    return 0;
}

int connection_count = 0;
int our_rank = -1;

static void
handle_new_connection(de, dep)
DExchange de;
DEPort dep;
{
    int i = 0;
    connect_msg conn;
    char server_name[256];
    init_msg contact_msg;
    init_msg_p return_msg;
    int format_id;

    sprintf(server_name, "%s__%d", DEport_host_name(dep),
	    DEport_port_number(dep));
    printf("Got a connection from %s\n", server_name);
    i = 0;
    while (server_list[i].server_name != NULL) {
	if (strcmp(server_name, server_list[i].server_name) == 0) {
	    printf("Whoa, got a duplicate connection! %s\n", server_name);
	}
	i++;
    }
    connection_count++;
    add_server_to_list(dep, server_name);

    if (our_rank != 0) {
	/* we're not the first guy */
	if (connection_count == 1) {
	    /* we're talking to the first guy */
	    return_msg = (init_msg_p) DEport_read_data(dep, &format_id);
	    if (return_msg == NULL) {
		IOperror(DEport_from_port(dep), "On initial message");
		exit(1);
	    }
	    assert(format_id == init_format_id);
	}
    } else {
	/* we're the first guy */
	int i;
	contact_msg.rank = connection_count;
	DEport_write_data(dep, init_format_id, &contact_msg);
	i = 1;
	while (server_list[i + 1].dep != NULL) {
	    /* tell him to connect the others too... */
	    DEPort other_dep = server_list[i].dep;
	    conn.server_name = server_list[i].server_name;
	    conn.hostname = DEport_host_name(other_dep);
	    conn.port = DEport_port_number(other_dep);
	    printf("telling him to connect to %s, %d\n", conn.hostname,
		   conn.port);
	    DEport_write_data(dep, connect_format_id, &conn);
	    i++;
	}
    }

    printf("Server list is now :");
    i = 0;
    while (server_list[i].server_name != NULL) {
	printf("%s, ", server_list[i++].server_name);
    }
    printf("\n");

}

static void
handle_end_connection(de, dep)
DExchange de;
DEPort dep;
{
    int i = 0;

    i = 0;
    while (server_list[i].server_name != NULL) {
	if (server_list[i].dep == dep) {
	    break;
	}
	i++;
    }
/* printf("Killing connection from %s\n", server_list[i].server_name); */
    free(server_list[i].server_name);
    while (server_list[i].server_name != NULL) {
	server_list[i] = server_list[i + 1];
	i++;
    }
}

extern DExchange
init_group(name, target_count)
char *name;
int target_count;
{
    char *group_name = NULL;
    DExchange de = NULL;
    DEPort dep = NULL;

    de = DExchange_create();
    DExchange_set_forward(de, DENever_Forward);

    if (DExchange_listen(de, 0) != 0) {
	fprintf(stderr, "DExchange listen failed\n");
	exit(1);
    }
    server_list = create_initial_server_list(name, de);

    DExchange_register_open_handler(de, handle_new_connection);
    DExchange_register_close_handler(de, handle_end_connection);

    DExchange_register_format(de, "Connect", COBS_connect_msg_flds);
    DExchange_register_filter(de, "Connect", handle_connect_msg, NULL);
    connect_format_id = DEget_format_id(de, "Connect");

    DExchange_register_format(de, "initial message",
			      COBS_init_msg_flds);
    init_format_id = DEget_format_id(de, "initial message");
    DExchange_register_function(de, "initial message",
				handle_init_msg, NULL);

    connection_count = 0;
    while ((group_name == NULL) && (dep == NULL)) {
	comm_group_return groups;
	int i;
	groups = matching_comm_groups(name, "cobs");
	for (i = 0; i < groups->count; i++) {
	    printf("Matching group : %s, %s, %s, %s, %s, %d\n",
	     groups->list[i].user_name, groups->list[i].application_name,
		   groups->list[i].group_type, groups->list[i].group_id,
		   groups->list[i].host, groups->list[i].port);
	}
	dep = DExchange_initiate_first(de, groups, 1);
	free(groups);
	if (dep == NULL) {
	    printf("No relevant Comm groups found, starting one\n");
	    group_name = setup_comm_group(NULL, name, "cobs", 60,
					  DExchange_host_name(de),
					  DExchange_inet_port(de));
	}
    }
    if (group_name != NULL) {
	/* we started the group, our rank is zero */
	our_rank = 0;
    } else {
	/* our_rank filled in in init_conn */
    }

    while (connection_count < (target_count - 1)) {
	/* wait until we're talking to target_count-1 other folks. */
	DExchange_poll_and_handle(de, 1);
    }
    return de;
}

void
main(argc, argv)
int argc;
char **argv;
{
    if (argc == 2) {
	int size;
	if (sscanf(argv[1], "%d", &size) == 1) {
	    init_group("testing", size);
	    exit(0);
	}
    }
    /* wrong number of args or couldn't sscanf number */
    printf("Usage:  test_group <group_size>\n");
    exit(1);
}
