

/***** Includes *****/
#include "config.h"
#include <sys/types.h>

#ifdef HAVE_WINDOWS_H
#include <windows.h>
#include <winsock.h>
#define getpid()	_getpid()
#else
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_TIMES_H
#include <sys/times.h>
#endif
#include <sys/socket.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#ifdef HAVE_SYS_UN_H
#include <sys/un.h>
#endif
#ifdef HAVE_HOSTLIB_H
#include "hostLib.h"
#endif
#ifdef HAVE_STREAMS_UN_H
#include <streams/un.h>
#endif
#include <netinet/in.h>
#include <arpa/inet.h>
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#endif
#include <stdio.h>
#include <fcntl.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <errno.h>
#include <signal.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#include "useful.h"
#include "io.h"
#include "io_interface.h"

#include "DE.h"

#if !defined(SELECT_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int select ARGS((int width, fd_set * readfds, fd_set * writefds,
			fd_set * exceptfds, struct timeval * timeout));
#endif
#if !defined(BIND_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int bind ARGS((int s, struct sockaddr * name, int namelen));
#endif
#if !defined(ACCEPT_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int accept ARGS((int s, struct sockaddr * addr, int *addrlen));
#endif
#if !defined(CONNECT_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int connect ARGS((int s, struct sockaddr * name, int namelen));
#endif
#if !defined(SETSOCKOPT_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int setsockopt ARGS((int s, int level, int optname, const char *optval, int optlen));
#endif
#if !defined(GETSOCKNAME_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int getsockname ARGS((int s, struct sockaddr * name, int *namelen));
#endif
#if defined(HAVE_GETDOMAINNAME) && !defined(GETDOMAINNAME_DEFINED)
extern int getdomainname ARGS((char *name, int namelen));
#endif
#if !defined(LISTEN_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int listen ARGS((int s, int backlog));
#endif
#ifndef HAVE_WINDOWS_H
extern int socket ARGS((int domain, int type, int protocol));
#endif
#if !defined(GETHOSTNAME_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int gethostname ARGS((char *name, int namelen));
#endif

extern void *(*DEmalloc) ARGS((int size));
extern void *(*DErealloc) ARGS((void *block, int size));
extern void (*DEfree) ARGS((void *block));

#ifndef SOCKET_ERROR
#define SOCKET_ERROR -1
#endif

static int
get_self_ip_addr()
{
    struct hostent *host;
    char buf[256];
    char **p;
#ifdef SIOCGIFCONF
    char *ifreqs;
    struct ifreq* ifr;
    struct sockaddr_in *sai;
    struct ifconf ifaces;
    int ss;
    int ifrn;
#endif
    int rv = 0;
    gethostname(buf, sizeof(buf));
    host = gethostbyname(buf);
    if (host == NULL) return 0;
    for (p=host->h_addr_list; *p !=0; p++) {
	struct in_addr *in = *(struct in_addr **)p;
	if (ntohl(in->s_addr) != INADDR_LOOPBACK) return ntohl(in->s_addr);
    }
    /*
     *  Since we couldn't find an IP address in some logical way, we'll open
     *  a DGRAM socket and ask it first for the list of its interfaces, and
     *  then checking for an interface we can use, and then finally asking that
     *  interface what its address is.
     */
#ifdef SIOCGIFCONF
    ss = socket (AF_INET, SOCK_DGRAM, 0);
    ifaces.ifc_len = 64 * sizeof (struct ifreq);
    ifaces.ifc_buf = ifreqs = malloc (ifaces.ifc_len);
    /*
     *  if we can't SIOCGIFCONF we're kind of dead anyway, bail.
     */
    if (ioctl (ss, SIOCGIFCONF, &ifaces) >= 0) {
        ifr = ifaces.ifc_req;
        ifrn = ifaces.ifc_len / sizeof (struct ifreq);
        for (; ifrn--; ifr++) {
            /*
             *  Basically we'll take the first interface satisfying the following:
             *  up, running, not loopback, address family is INET.
             */
            ioctl (ss, SIOCGIFFLAGS, ifr);
            if (ifr->ifr_flags & IFF_LOOPBACK) continue;
            if (!(ifr->ifr_flags & IFF_UP)) continue;
            if (!(ifr->ifr_flags & IFF_RUNNING)) continue;
            sai = (struct sockaddr_in *) &(ifr->ifr_addr);
            /*
               sure would be nice to test for AF_INET here but it doesn't
               cooperate and I've done enough for now ...
               if (sai->sin_addr.s.addr != AF_INET) continue;
            */
            if (sai->sin_addr.s_addr == INADDR_ANY) continue;
            if (sai->sin_addr.s_addr == INADDR_LOOPBACK) continue;
            rv = sai->sin_addr.s_addr;
            break;
          }
      }
    close (ss);
    free (ifreqs);
#endif
    /*
     *  Absolute last resort.  If we can't figure out anything else, look
     *  for the DE_LAST_RESORT_IP_ADDR environment variable.
     */
    if (rv == 0) {
        char *c = getenv ("DE_LAST_RESORT_IP_ADDR");
        if (c != NULL) {
	    rv = inet_addr(c);
/*     struct in_addr inp; inet_aton (c, &inp);   rv = inp.s_addr; */
          }
      }
    /*
     *  hopefully by now we've set rv to something useful.  If not,
     *  GET A BETTER NETWORK CONFIGURATION.
     */
    return rv;
}

extern void
DEget_qual_hostname(char *buf, int len)
{
    struct hostent *host;

    char *network_string = getenv("DEXCHANGE_NETWORK");
    gethostname(buf, len);
    buf[len - 1] = '\0';
    if (memchr(buf, '.', strlen(buf)) == NULL) {
	/* no dots, probably not fully qualified */
#ifdef HAVE_GETDOMAINNAME
	int end = strlen(buf);
	buf[end] = '.';
	getdomainname((&buf[end]) + 1, len - strlen(buf));
	if (buf[end+1] == 0) {
	    char *tmp_name;
	    buf[end] = 0;
	    /* getdomainname was useless, hope that gethostbyname helps */
	    tmp_name = (gethostbyname(buf))->h_name;
	    strncpy(buf, tmp_name, len);
	}
#else
	/* no getdomainname, hope that gethostbyname will help */
	char *tmp_name = (gethostbyname(buf))->h_name;
	strncpy(buf, tmp_name, len);
#endif
	buf[len - 1] = '\0';
    }
    if (((host = gethostbyname(buf)) == NULL) ||
	(memchr(buf, '.', strlen(buf)) == NULL)){
	/* bloody hell, what do you have to do? */
	int IP = get_self_ip_addr();
	host = gethostbyaddr((char*)&IP, sizeof(IP), AF_INET);
	if (host != NULL) {
	  strncpy(buf, host->h_name, len);
	}
    }
    
    if (network_string != NULL) {
	int name_len = strlen(buf) + 2 + strlen(network_string);
	char *new_name_str = DEmalloc(name_len);
	char *first_dot = strchr(buf, '.');

	/* stick the DEXCHANGE_NETWORK value in there */
	memset(new_name_str, 0, name_len);
	*first_dot = 0;
	first_dot++;
	sprintf(new_name_str, "%s-%s.%s", buf, network_string, first_dot);
	if (gethostbyname(new_name_str) != NULL) {
	    /* host has no NETWORK interface */
	    strcpy(buf, new_name_str);
	}
	DEfree(new_name_str);
    }

    if (((host = gethostbyname(buf)) == NULL) ||
	(memchr(buf, '.', strlen(buf)) == NULL)){
	/* just use the bloody IP address */
	gethostname(buf, len);
	host  = gethostbyname(buf);
	if (host != NULL) {
	    struct in_addr ip;
	    char *tmp;
	    ip.s_addr = get_self_ip_addr();
	    tmp = inet_ntoa(ip);
	    strncpy(buf, tmp, len);
	} else {
	    static int warn_once = 0;
	    if (warn_once == 0) {
		warn_once++;
		printf("Attempts to establish your fully qualified hostname, or indeed any\nuseful network name, have failed horribly.  Sorry.\n");
	    }
	    strncpy(buf, "Unknown", len);
	}
    }
}

/* 
 * Accept socket connection from other data exchs.
 */
extern DEPort
DExchange_accept_conn_sock(de, conn_sock)
DExchange de;
int conn_sock;
{
    int sock;
    DEPort dep;

    struct linger linger_val;
    IOFile from_port, to_port;

    linger_val.l_onoff = 1;
    linger_val.l_linger = 60;
    if (de == NULL)
	return NULL;
    if ((sock = accept(conn_sock, (struct sockaddr *) 0, (int *) 0)) == SOCKET_ERROR) {
	fprintf(stderr, "Cannot accept socket connection\n");
	return NULL;
    }
    if (setsockopt(sock, SOL_SOCKET, SO_LINGER, (char *) &linger_val,
		   sizeof(struct linger)) != 0) {
	perror("set SO_LINGER");
	return NULL;
    }
#ifdef TCP_NODELAY
    {
	/* added by Arno Schoedl schoedl@usa.net */
	int delay_value = 1;
	setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char *) &delay_value,
		   sizeof(delay_value));
    }
#endif
    /* open PBIO descriptor for receiving information */
    from_port = create_IOfile();
    set_conn_IOfile(from_port, (void *) sock);
    set_socket_interface_IOfile(from_port);

    /* open PBIO descriptor for sending information */
    to_port = create_IOfile();
    set_conn_IOfile(to_port, (void *) sock);
    set_socket_interface_IOfile(to_port);

    dep = DEPort_create(de);

    /* try the accept-connection handshake */
    if (!DExchange_accept_port_init(dep, from_port, to_port)) {
	DEport_close(dep);
	fprintf(stderr, "BAD!\n");
	return NULL;
    }
    /* provide for general PBIO-level event handling */
    DEPort_add_general_select(dep, sock);

    return dep;
}

/* 
 * Initiate a socket connection with another data exchange.  If port_num is -1,
 * establish a unix socket connection (name_str stores the file name of
 * the waiting socket).  Otherwise, establish an INET socket connection
 * (name_str stores the machine name).
 */
static int
check_host(hostname, sin_addr)
char *hostname;
void *sin_addr;
{
#ifdef HAS_STRUCT_HOSTENT
    struct hostent *host_addr;
    host_addr = gethostbyname(hostname);
    if (host_addr == NULL) {
	unsigned long addr = inet_addr(hostname);
	if (addr == -1) {
	    /* 
	     *  not translatable as a hostname or 
	     * as a dot-style string IP address
	     */
	    return 0;
	}
	assert(sizeof(int) == sizeof(struct in_addr));
	*((int*)sin_addr) = (int)addr;
    } else {
	memcpy(sin_addr, host_addr->h_addr, host_addr->h_length);
    }
    return 1;
#else
    /* VxWorks ? */
    *((int *)sin_addr) = hostGetByName(hostname);
    return (*(int*)sin_addr) != -1;
#endif
}

DEPort
DExchange_inet_initiate_conn(de, name_str, int_port_num, block_by_default)
DExchange de;
char *name_str;
int int_port_num;
int block_by_default;
{
    int sock;
    DEPort dep;
    IOFile from_port, to_port = NULL;

    struct linger linger_val;
    int sock_opt_val = 1;
    u_short port_num = int_port_num;

    linger_val.l_onoff = 1;
    linger_val.l_linger = 60;
    if (int_port_num == -1) {
#if defined(AF_UNIX) && !defined(HAVE_WINDOWS_H)
	/* unix socket connection, name_str is the file name */
	struct sockaddr_un sock_addr;
	if ((sock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0) {
	    return NULL;
	}
	sock_addr.sun_family = AF_UNIX;
	strcpy(sock_addr.sun_path, name_str);
	if (connect(sock, (struct sockaddr *) &sock_addr,
		    sizeof sock_addr) < 0) {
	    return NULL;
	}
#else
	fprintf(stderr, "DExchange_initiate_conn port_num parameter == -1 and unix sockets not available.\n");
	return NULL;
#endif
    } else {
	/* INET socket connection, name_str is the machine name */
	struct sockaddr_in sock_addr;
	char *network_string;
	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) == SOCKET_ERROR) {
	    return NULL;
	}
	sock_addr.sin_family = AF_INET;
	if ((network_string = getenv("DEXCHANGE_NETWORK")) != NULL) {
	    int name_len = strlen(name_str) + 2 + strlen(network_string);
	    char *new_name_str = DEmalloc(name_len);
	    char *first_dot = strchr(name_str, '.');
	    memset(new_name_str, 0, name_len);
	    if (first_dot == NULL) {
		strcpy(new_name_str, name_str);
		strcat(new_name_str, "-");
		strcat(new_name_str, network_string);
	    } else {
		strncpy(new_name_str, name_str, first_dot - name_str);
		strcat(new_name_str, "-");
		strcat(new_name_str, network_string);
		strcat(new_name_str, first_dot);
	    }
	    if (check_host(new_name_str, (void*)&sock_addr.sin_addr) == 0)  {
		/* host has no NETWORK interface */
		if (check_host(name_str, (void*)&sock_addr.sin_addr) == 0) {
		    return NULL;
		}
	    } else {
		if (DExchange_debugging_on(de)) {
		    fprintf(stderr, "--> Using non default network interface with hostname %s\n",
			    new_name_str);
		}
	    }
	    DEfree(new_name_str);
	} else {
	    if (check_host(name_str, (void*)&sock_addr.sin_addr) == 0) {
		return NULL;
	    }
	}
	sock_addr.sin_port = htons(port_num);
	if (connect(sock, (struct sockaddr *) &sock_addr,
		    sizeof sock_addr) == SOCKET_ERROR) {
#ifdef WSAEWOULDBLOCK
	    int err = WSAGetLastError();
	    if (err != WSAEWOULDBLOCK || err != WSAEINPROGRESS)
#endif
		return NULL;
	}
    }
    sock_opt_val = 1;
    setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (char *) &sock_opt_val,
	       sizeof(sock_opt_val));
    setsockopt(sock, SOL_SOCKET, SO_LINGER, (char *) &linger_val,
	       sizeof(struct linger));

#ifdef TCP_NODELAY
    {
	/* added by Arno Schoedl schoedl@usa.net */
	int delay_value = 1;

	setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char *) &delay_value,
		   sizeof(delay_value));
    }
#endif
    /* open PBIO file descriptor for sending information */
    to_port = create_IOfile();
    set_conn_IOfile(to_port, (void *) sock);
    set_socket_interface_IOfile(to_port);

    /* open PBIO file descriptor for receiving information */
    from_port = create_IOfile();
    set_conn_IOfile(from_port, (void *) sock);
    set_socket_interface_IOfile(to_port);

    dep = DEPort_create(de);

    /* try the initiate-port handshake */
    if (!DExchange_initiate_port_init(dep, from_port, to_port,
				      block_by_default)) {
	DEport_close(dep);
	return NULL;
    }
    /* provide for general PBIO-level event handling */
    DEPort_add_general_select(dep, sock);

    return dep;
}

/* 
 * Create an IP socket for connection from other data exchanges.
 */
extern int
DExchange_inet_listen(de, int_port_num)
DExchange de;
int int_port_num;
{
    int length;
    struct sockaddr_in sock_addr;
    int sock_opt_val = 1;
    int conn_sock;
    u_short port_num = int_port_num;

    conn_sock = socket(AF_INET, SOCK_STREAM, 0);
    if (conn_sock == SOCKET_ERROR) {
	fprintf(stderr, "Cannot open INET socket\n");
	return -1;
    }
    sock_addr.sin_family = AF_INET;
    sock_addr.sin_addr.s_addr = INADDR_ANY;
    sock_addr.sin_port = htons(port_num);
    if (setsockopt(conn_sock, SOL_SOCKET, SO_REUSEADDR, (char *) &sock_opt_val,
		   sizeof(sock_opt_val)) != 0) {
	fprintf(stderr, "Failed to set 1REUSEADDR on INET socket\n");
	return -1;
    }
    if (bind(conn_sock, (struct sockaddr *) &sock_addr,
	     sizeof sock_addr) == SOCKET_ERROR) {
	fprintf(stderr, "Cannot bind INET socket\n");
	return -1;
    }
    sock_opt_val = 1;
    if (setsockopt(conn_sock, SOL_SOCKET, SO_REUSEADDR, (char *) &sock_opt_val,
		   sizeof(sock_opt_val)) != 0) {
	perror("Failed to set 2REUSEADDR on INET socket");
	return -1;
    }
    length = sizeof sock_addr;
    if (getsockname(conn_sock, (struct sockaddr *) &sock_addr, &length) < 0) {
	fprintf(stderr, "Cannot get socket name\n");
	return -1;
    }
    /* begin listening for conns and set the backlog */
    if (listen(conn_sock, FD_SETSIZE)) {
	fprintf(stderr, "listen failed\n");
	return -1;
    }
    /* set the port num as one we can be contacted at */
    DExchange_set_port_number(de, ntohs(sock_addr.sin_port));

    /* No general PBIO handling, just call our handler if read possible */
    DExchange_add_select(de, conn_sock,
			 (GenericHandlerFunc) DExchange_accept_conn_sock,
			 (void *) de, (void *) conn_sock);

    /* in the event the DE is shut down, close the socket */
    /* 
     *  -- Don't do this...  Close() seems to hang on sockets after 
     *  listen() for some reason.  I haven't found anywhere that defines 
     *  this behavior, but it seems relatively uniform. 
     */
    /* DExchange_add_close(de, close_socket_fd, (void*)conn_sock, NULL); */

    return 0;
}


#ifndef _WINSOCKAPI_
static void
unlink_and_close(name_as_addr, fd_as_addr)
void *name_as_addr;
void *fd_as_addr;
{
    unlink((char *) name_as_addr);
    DEfree(name_as_addr);
    close((int) (long) fd_as_addr);
}

/* 
 * Create an UNIX socket for connection from other data exchanges.
 */
extern int
DExchange_open_unix(de, usock_name)
DExchange de;
char *usock_name;
{
    /* NT doesn't support this type socket */
    struct sockaddr_un sock_addr;
    int conn_sock;
    char name_buf[32];
    char *socket_name = usock_name;

    conn_sock = socket(AF_UNIX, SOCK_STREAM, 0);
    if (conn_sock < 0) {
	fprintf(stderr, "Cannot open UNIX socket\n");
	return -1;
    }
    if (socket_name == NULL) {
	sprintf(&name_buf[0], "/tmp/dataexch_%lx", (long) getpid());
	socket_name = strdup(&name_buf[0]);
    }
    sock_addr.sun_family = AF_UNIX;
    strcpy(sock_addr.sun_path, socket_name);
    if (bind(conn_sock, (struct sockaddr *) &sock_addr,
	     sizeof sock_addr) < 0) {
	fprintf(stderr, "Cannot bind UNIX socket to >%s<\n", socket_name);
	return -1;
    }
    /* begin listening for conns and set the backlog */
    if (listen(conn_sock, FD_SETSIZE)) {
	fprintf(stderr, "listen failed\n");
	return -1;
    }
    /* No general PBIO handling, just call our handler if read possible */
    DExchange_add_select(de, conn_sock,
			 (GenericHandlerFunc) DExchange_accept_conn_sock,
			 (void *) de, (void *) conn_sock);

    /* in the event the DE is shut down, close the socket, unlink file */
    DExchange_add_close(de, unlink_and_close, (void *) socket_name,
			(void *) conn_sock);

    return 0;
}
#endif
