/***** Includes *****/
#include "config.h"
#include <sys/types.h>

#ifdef HAVE_WINDOWS_H
#include <windows.h>
#include <winsock.h>
#define getpid()	_getpid()
#else
#include <sys/socket.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#endif
#include <stdio.h>
#include <fcntl.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <errno.h>
#include <signal.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#include <string.h>

#include "useful.h"
#include "io.h"
#include "io_interface.h"

#include "DE.h"
#ifdef HAVE_GEN_THREAD_H
#include "gen_thread.h"
#else
#define thr_mutex_alloc() NULL
#endif
#include "de_internal.h"
#include "de_lock.h"
#include "de_control.h"
#include "comm_group.h"
#include "de_monitor.h"

#if defined(HAVE_GETDOMAINNAME) && !defined(GETDOMAINNAME_DEFINED)
extern int getdomainname ARGS((char *name, int namelen));
#endif
#if !defined(GETHOSTNAME_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int gethostname ARGS((char *name, int namelen));
#endif
extern pid_t getpid();

void
DEMon_data_format_name_sensor ARGS((DExchange de, char *format_name));

void
DEMon_dexchange_create_sensor ARGS((DExchange de));

void
DEMon_deport_create_sensor ARGS((DExchange de, DEPort dep));

void
DEMon_signon_msg_sensor ARGS((DExchange de, DEPort dep, char *name));

void
DEMon_signoff_msg_sensor ARGS((DExchange de, DEPort dep, char *name));

void
DEMon_forward_record_sensor ARGS((DExchange de, char *format_name, int port));

void
DEMon_register_func_sensor ARGS((DExchange de, char *format_name, int which_func));

void
DEMon_register_filter_sensor ARGS((DExchange de, char *format_name, int which_filter));


#define Max(i,j) ((i<j) ? j : i)

int de_debug_flag = FALSE;
int de_data_debug_flag = FALSE;

typedef enum {
    write_record_state, write_format_state, not_blocked
} block_type;

#ifndef HAVE_WINDOWS_H
extern unsigned alarm ARGS((unsigned));
#endif

static void record_blocking_state ARGS((block_type state,
					DEPort port));
static int block_count = 0;
static DEPort block_port = NULL;
static block_type block_state = not_blocked;

static void
clear_blocking_state()
{
    block_port = NULL;
    block_state = not_blocked;
    block_count++;
}

static void
record_blocking_state(state, port)
block_type state;
DEPort port;
{
    block_port = port;
    block_state = state;
}

static int last_block_count = 0;

static void
check_blocking_state(signal)
int signal;
{
    if ((block_count == last_block_count) && (block_state != not_blocked)
	&& (block_port != NULL)) {
	fprintf(stderr, "DataExchange Warning!  No forward progress!!\n");
	fprintf(stderr, " Writing to port [%s, %d] \"%s\".\n",
		block_port->hostname, block_port->port,
		block_port->exchange_name);
    }
    last_block_count = block_count;
#ifdef HAVE_WINDOWS_H
    SetTimer(NULL, 0, 1000, (TIMERPROC) check_blocking_state);
#else
    alarm(1);
#endif

}


void
DEinit_block_check()
{
#ifdef HAVE_WINDOWS_H
    SetTimer(NULL, 0, 1000, (TIMERPROC) check_blocking_state);
#else
    struct sigaction sigact;
    sigact.sa_flags = 0;
    sigact.sa_handler = check_blocking_state;
    sigemptyset(&sigact.sa_mask);
    sigaddset(&sigact.sa_mask, SIGALRM);
    sigaction(SIGALRM, &sigact, NULL);
    alarm(1);
#endif
}

static void reset_input_conversions ARGS((DExchange de, int format_id));
static void register_format_to_all ARGS((DExchange de, int format_id));
static void register_all_formats ARGS((DEPort dep));
static IOFormat register_format_to_port ARGS((DEPort dep, int format_id));
static void DExchange_handle_write_error ARGS((DExchange de, DEPort dep));
static int get_formatID ARGS((DExchange de, char *name));
static void DExchange_general_handler ARGS((DEPort dep, void *unused));
static void DExchange_general_poll_handler ARGS((DEPort dep, void *unused));
static void create_debugging_exchange ARGS((DExchange de));
static void remove_port_from_DE ARGS((DExchange de, DEPort dep));
static void add_to_pending_conn_list ARGS((DExchange de, char *host,
					   int port));
static int is_pending_conn ARGS((DExchange de, char *host, int port));
static void add_to_pending_wait_list ARGS((DExchange de, char *host, int port,
					   int condition));
static void remove_from_pending_conn_list ARGS((DExchange de, char *host,
						int port, DEPort dep));


void *(*DEmalloc) ARGS((int size)) = (DEMallocFunction) malloc;
void *(*DErealloc) ARGS((void *block, int size)) = (DEReallocFunction) realloc;
void (*DEfree) ARGS((void *block)) = (DEFreeFunction) free;

extern void
DE_set_allocators(malloc_func, realloc_func, free_func)
DEMallocFunction malloc_func;
DEReallocFunction realloc_func;
DEFreeFunction free_func;
{
    DEmalloc = malloc_func;
    DErealloc = realloc_func;
    DEfree = free_func;
}

int
DEget_format_id(de, name)
DExchange de;
char *name;
{
    int id;
    DExchange_lock(de);
    id = get_formatID(de, name);
    DExchange_unlock(de);
    return id;
}

char *
DEget_format_name(de, format_id)
DExchange de;
int format_id;
{
    if (format_id > de->max_formatID) {
	return NULL;
    }
    return de->format_list[format_id].format_name;
}

static void
InitFormatNode(fname, iof_rec)
char *fname;
ioformat_data *iof_rec;
{
    iof_rec->format_name = strdup(fname);
    iof_rec->forward_policy = DENever_Forward;
    iof_rec->system_format = FALSE;
    iof_rec->fixed_format = FALSE;
    iof_rec->internal_field_list = NULL;
    iof_rec->filter_count = 0;
    iof_rec->funct_count = 0;
    iof_rec->funct = (DataHandlingFunc *) DEmalloc(sizeof(iof_rec->funct[0]));
    iof_rec->funct[0] = NULL;
    iof_rec->funct_data = (void **) DEmalloc(sizeof(iof_rec->funct_data[0]));
    iof_rec->funct_data[0] = NULL;
    iof_rec->thr_funct_count = 0;
    iof_rec->thr_funct = (DataHandlingFunc *) DEmalloc(sizeof(iof_rec->thr_funct[0]));
    iof_rec->thr_funct[0] = NULL;
    iof_rec->thr_funct_data = (void **) DEmalloc(sizeof(iof_rec->thr_funct_data[0]));
    iof_rec->thr_funct_data[0] = NULL;
    iof_rec->filter = (DataHandlingFunc *) DEmalloc(sizeof(iof_rec->filter[0]));
    iof_rec->filter[0] = NULL;
    iof_rec->filter_data = (void **) DEmalloc(sizeof(iof_rec->filter_data[0]));
    iof_rec->filter_data[0] = NULL;
}

static int
process_signon_message(de, dep, signon_format_id, data, data_length,
		       client_data)
DExchange de;
DEPort dep;
int signon_format_id;
void *data;
int data_length;
void *client_data;
{
    int format_id;
    IOFieldList field_list = de->format_list[signon_format_id].internal_field_list;
    IOFieldPtr name_field = get_IOfieldPtrFromList(field_list,
						   "format_name");

    char *name;

    assert(name_field != NULL);
    name = get_IOstring(name_field, data);

    DBG2(de, "handle sign-on message [%s, %s]\n", dep->exchange_name, name);
    DEMon_signon_msg_sensor(de, dep, name);

    format_id = get_formatID(de, name);
    dep->formats[format_id].forward_data = TRUE;

    DEfree(name_field);
    return 0;
}


static int
process_signoff_message(de, dep, signoff_format_id, data, data_length,
			client_data)
DExchange de;
DEPort dep;
int signoff_format_id;
void *data;
int data_length;
void *client_data;
{
    int format_id;
    IOFieldList field_list = de->format_list[signoff_format_id].internal_field_list;
    IOFieldPtr name_field = get_IOfieldPtrFromList(field_list,
						   "format_name");
    char *name;

    assert(name_field != NULL);
    name = get_IOstring(name_field, data);

    DBG2(de, "handle sign-off message [%s, %s]\n", dep->exchange_name, name);
    DEMon_signoff_msg_sensor(de, dep, name);

    format_id = get_formatID(de, name);
    dep->formats[format_id].forward_data = FALSE;

    DEfree(name_field);
    return 0;
}


static int
process_shutdown_message(de, dep, shutdown_format_id, data, data_length,
			 client_data)
DExchange de;
DEPort dep;			/* child port if connectionless */
int shutdown_format_id;
void *data;
int data_length;
void *client_data;
{
    int i;
    IOFieldList field_list;
    IOFieldPtr whom_field, msg_field;
    int whom;
    char *message;

    assert(dep != NULL);
    field_list = de->format_list[shutdown_format_id].internal_field_list;
    whom_field = get_IOfieldPtrFromList(field_list, "whom");
    field_list = de->format_list[shutdown_format_id].internal_field_list;
    msg_field = get_IOfieldPtrFromList(field_list, "message");
    assert(whom_field != NULL);
    assert(msg_field != NULL);

    whom = get_IOint(whom_field, data);
    message = get_IOstring(msg_field, data);
    DEfree(whom_field);
    DEfree(msg_field);

    switch (whom) {
    case SHUTDOWN_CONNECTION:
    default:
	if (dep->parentPort != NULL) {
	    /* close on parent if connectionless */
	    dep = dep->parentPort;
	}
	DEport_close(dep);
	break;

    case SHUTDOWN_CONNECTION_INPUT:
	if (dep->parentPort == NULL && dep->from_port != NULL) {
	    DEPort_remove_from_control(dep);
	    close_IOfile(dep->from_port);
	    free_IOfile(dep->from_port);
	    dep->from_port = NULL;
	} else if (dep->parentPort != NULL) {	/* connectionless */
	    /* 
	     * for connectionless, SHUTDOWN...INPUT has same effect as
	     * SHUTDOWN...OUTPUT -- it closes the port 
	     */
	    dep = dep->parentPort;
	    DEPort_remove_from_control(dep);
	    DEport_close(dep);
	}
	break;

    case SHUTDOWN_CONNECTION_OUTPUT:
	if (dep->parentPort == NULL && dep->to_port != NULL) {
	    close_IOfile(dep->to_port);
	    free_IOfile(dep->to_port);
	    dep->to_port = NULL;
	} else if (dep->parentPort != NULL) {	/* connectionless */
	    /* 
	     * for connectionless, SHUTDOWN...INPUT has same effect as
	     * SHUTDOWN...OUTPUT -- it closes the port 
	     */
	    dep = dep->parentPort;
	    DEPort_remove_from_control(dep);
	    DEport_close(dep);
	}
	break;

    case SHUTDOWN_ALL_CONNECTION:
	for (i = 0; i < de->portCount; i++) {
	    DEPort tmp_dep = de->ports[i];
	    if (tmp_dep != NULL && tmp_dep != dep && tmp_dep->parentPort == NULL) {
		DExchange_shutdown_connection(de, tmp_dep, SHUTDOWN_CONNECTION,
					      message);
		DEport_close(tmp_dep);
	    } else if (tmp_dep != NULL && tmp_dep->parentPort != NULL) {
		/* 
		 *  connectionless.  For each child in childPort list, send
		 *  shutdown message 
		 */
		int j;
		for (j = 0; j < dep->childPortCount; j++) {
		    DEPort child_dep = dep->childPorts[j];
		    DExchange_shutdown_connection(de, child_dep, SHUTDOWN_CONNECTION,
						  message);
		}
		dep->dg_shutdown(dep->client_data);
	    }
	}
	DEport_close(dep);	/* closes parent port if connectionless */
	break;

    case SHUTDOWN_ALL_INPUT:
	for (i = 0; i < de->portCount; i++) {
	    DEPort tmp_dep = de->ports[i];
	    if (tmp_dep != NULL && dep->from_port != NULL && dep->parentPort == NULL) {
		DEPort_remove_from_control(tmp_dep);
		close_IOfile(tmp_dep->from_port);
		free_IOfile(tmp_dep->from_port);
		tmp_dep->from_port = NULL;
	    } else if (dep->parentPort != NULL) {
		/* connectionless - close port */
		tmp_dep->dg_shutdown(tmp_dep->client_data);
		DEport_close(tmp_dep);
	    }
	}
	break;

    case SHUTDOWN_ALL_OUTPUT:
	for (i = 0; i < de->portCount; i++) {
	    DEPort tmp_dep = de->ports[i];
	    if (tmp_dep != NULL && dep->to_port != NULL && dep->parentPort == NULL) {
		close_IOfile(tmp_dep->to_port);
		free_IOfile(tmp_dep->to_port);
		tmp_dep->to_port = NULL;
	    } else if (dep->parentPort != NULL) {
		/* connectionless - close port */
		tmp_dep->dg_shutdown(tmp_dep->client_data);
		DEport_close(tmp_dep);
	    }
	}
	break;

    case SHUTDOWN_SERVER:
	for (i = 0; i < de->portCount; i++) {
	    DEPort tmp_dep = de->ports[i];
	    if (tmp_dep != NULL && tmp_dep != dep && tmp_dep->parentPort == NULL) {
		DExchange_shutdown_connection(de, tmp_dep, SHUTDOWN_CONNECTION,
					      message);
		DEport_close(tmp_dep);
	    } else if (tmp_dep != NULL && dep->parentPort != NULL) {
		/* connectionless - close port */
		int j;
		for (j = 0; j < dep->childPortCount; j++) {
		    DEPort child_dep = dep->childPorts[j];
		    DExchange_shutdown_connection(de, child_dep, SHUTDOWN_CONNECTION,
						  message);
		}
		dep->dg_shutdown(dep->client_data);
	    }
	}
	DEport_close(dep);
	DExchange_close(de);
	DExchange_free(de);
	break;

    case SHUTDOWN_SYSTEM:
	for (i = 0; i < de->portCount; i++) {
	    DEPort tmp_dep = de->ports[i];
	    if (tmp_dep != NULL && tmp_dep != dep) {
		DExchange_shutdown_connection(de, tmp_dep, SHUTDOWN_SYSTEM,
					      message);
		DEport_close(tmp_dep);
	    } else if (tmp_dep != NULL && tmp_dep->parentPort != NULL) {
		/* 
		 *  connectionless.  For each child in childPort list, send
		 *  shutdown message 
		 */
		int j;
		for (j = 0; j < dep->childPortCount; j++) {
		    DEPort child_dep = dep->childPorts[j];
		    DExchange_shutdown_connection(de, child_dep, SHUTDOWN_CONNECTION,
						  message);
		}
		dep->dg_shutdown(dep->client_data);
	    }
	}

	DEport_close(dep);
	DExchange_close(de);
	DExchange_free(de);
	exit(0);
	break;
    }
    return 0;
}

extern void
DExchange_forward_data(de, dep, formatID, data)
DExchange de;
DEPort dep;
int formatID;
void *data;
{
    int port_index;
    /* route messages .. */
    DEMon_data_format_name_sensor(de, de->format_list[formatID].format_name);
    DDBG(de, "\tForwarding data, ");
    /* connectionless  */
    if (dep != NULL && dep->dg_write != NULL) {
	int buf_size;
	char *xfer_buffer;
	IOFormat ioformat = NULL;
	DEPort parent_dep;
	if (dep->parentPort) {
	    /* get parent to find children */
	    parent_dep = dep->parentPort;
	} else {
	    parent_dep = dep;	/* no children so I am the parent */
	}
	if ((de->format_list[formatID].forward_policy != DENever_Forward) &&
	    de->format_list[formatID].system_format == FALSE) {
	    DDBG(de, "routing to ");
	    for (port_index = 0; port_index < parent_dep->childPortCount;
		 port_index++) {
		DEPort tmp_dep = parent_dep->childPorts[port_index];
		if ((tmp_dep != NULL) && tmp_dep->formats[formatID].forward_data) {
		    if ((de->format_list[formatID].forward_policy == DEAll_But_Sender) &&
			(dep->dg_addrCmp(dep->client_data, tmp_dep->client_data) == 0))
			continue;
		    DDBG1(de, " %d\n", port_index);
		    if (tmp_dep->formats[formatID].output_format == NULL) {
			register_format_to_port(tmp_dep, formatID);
		    }
		    /* forward messages here... */
		    ioformat = tmp_dep->formats[formatID].output_format;
		    /* 
		     *  returns pointer to temporary buffer in 'pbio space' 
		     */
		    DExchange_context_lock(de);
		    xfer_buffer = encode_IOcontext_buffer(de->context,
							  ioformat,
							data, &buf_size);
		    DExchange_context_unlock(de);
		    record_blocking_state(write_record_state, tmp_dep);
		    DEPort_write_lock(de, tmp_dep);
		    tmp_dep->dg_write(xfer_buffer, buf_size, &tmp_dep->client_data);
		    DEPort_write_unlock(de, tmp_dep);
		    clear_blocking_state();
		}
	    }
	} else {
	    DDBG(de, "Never forward");
	}
    } else if ((de->format_list[formatID].forward_policy != DENever_Forward)
	       && de->format_list[formatID].system_format == FALSE) {
	DDBG(de, "routing to ");
	for (port_index = 0; port_index < de->portCount; port_index++) {
	    DEPort tmp_dep = de->ports[port_index];
	    if ((tmp_dep != NULL) &&
		tmp_dep->formats[formatID].forward_data &&
		tmp_dep->to_port) {

		if ((de->format_list[formatID].forward_policy == DEAll_But_Sender) &&
		    (tmp_dep == dep))
		    continue;

		if (tmp_dep->formats[formatID].output_format == NULL) {
		    register_format_to_port(tmp_dep, formatID);
		}
		record_blocking_state(write_record_state, tmp_dep);
		DDBG1(de, " %d", port_index);
		/* forward messages here... */
		DEPort_write_lock(de, tmp_dep);
		if (write_IOfile(tmp_dep->to_port,
				 tmp_dep->formats[formatID].output_format,
				 data) != 1) {
		    IOperror(tmp_dep->to_port, "Forward write error");
		    fprintf(stderr, " on connection %s from...\n",
			    tmp_dep->exchange_name);

		    DExchange_handle_write_error(de, tmp_dep);
		}
		DEPort_write_unlock(de, tmp_dep);
		clear_blocking_state();
		DEMon_forward_record_sensor(de,
				   de->format_list[formatID].format_name,
					    port_index);

	    } else if (tmp_dep != NULL) {
		DBG3(de, "{-%d, port=%d, of=%lx} ", port_index,
		     tmp_dep->formats[formatID].forward_data,
		     (long) tmp_dep->formats[formatID].output_format);
	    }
	}
    } else {
	DDBG(de, "Never forward");
    }

    DDBG(de, "\n");
}

typedef struct _func_record {
    thr_mutex_t lock;
    int thr_count;
} *thr_func_record;

typedef struct _fork_record {
    DExchange de;
    DEPort dep;
    int formatID;
    void *data;
    int length;
    DataHandlingFunc funct;
    void *funct_data;
    thr_func_record func_record;
} *thr_fork_record;

static void
thr_handler_wrapper(fork_rec)
thr_fork_record fork_rec;
{
    fork_rec->funct(fork_rec->de, fork_rec->dep, fork_rec->formatID,
		 fork_rec->data, fork_rec->length, fork_rec->funct_data);
    thr_mutex_lock(fork_rec->func_record->lock);
    fork_rec->func_record->thr_count--;
    if (fork_rec->func_record->thr_count == 0) {
	thr_mutex_unlock(fork_rec->func_record->lock);
	DEfree(fork_rec->data);
	thr_mutex_free(fork_rec->func_record->lock);
	DEfree(fork_rec->func_record);
    } else {
	thr_mutex_unlock(fork_rec->func_record->lock);
    }
    DEfree(fork_rec);
}

static int
ProcessData(de, dep, formatID, data, length)
DExchange de;
DEPort dep;
int formatID;
char *data;
int length;
{
    int i, port_index;
    int forward_data = TRUE;
    DEPort tmp_dep = NULL;

    BUF_TAKEN(de) = 0;

    /* 
     *  if connectionless (childPortCount > 0), locate child port and 
     *  make it the current port 
     */
    for (port_index = 0; port_index < dep->childPortCount;
	 port_index++) {
	tmp_dep = dep->childPorts[port_index];
	if (dep->dg_addrCmp(dep->client_data, tmp_dep->client_data) == 0) {
	    dep = tmp_dep;
	    break;
	}
    }

    DDBG2(de, "Handle data of format [%s] from exchange = %s\n",
	  de->format_list[formatID].format_name, dep->exchange_name);

    /* Execute any registered filters */
    for (i = 0; (i < de->format_list[formatID].filter_count) && forward_data;
	 i++) {
	if (de->format_list[formatID].filter[i] != NULL) {
	    int result = de->format_list[formatID].filter[i] (de, dep,
						  formatID, data, length,
			       de->format_list[formatID].filter_data[i]);
	    if (!result)
		forward_data = FALSE;
	    DDBG2(de, "filter #%d result code was %d.\n", i, result);
	}
	if (BUF_TAKEN(de)) {
	    DDBG(de, "Buffer was taken, processing terminated\n");
	    return 0;
	}
    }

    if (!forward_data) {
	DDBG(de, "\nFilter canceled forward.. returning\n");
	return 0;
    }
    DExchange_forward_data(de, dep, formatID, data);
    /* Then, execute any registered functions */
    for (i = 0; i < de->format_list[formatID].funct_count; i++) {
	if (de->format_list[formatID].funct[i] != NULL) {
	    DDBG2(de, "Executing funct #%d for [%s].\n", i,
		  de->format_list[formatID].format_name);
	    de->format_list[formatID].funct[i] (de, dep, formatID, data,
			length, de->format_list[formatID].funct_data[i]);
	}
	if (BUF_TAKEN(de))
	    return 0;
    }
    /* Then, execute any registered threaded functions */
    if (de->format_list[formatID].thr_funct_count != 0) {
	thr_func_record rec = DEmalloc(sizeof(*rec));
	rec->lock = thr_mutex_alloc();
	rec->thr_count = 0;
	thr_mutex_lock(rec->lock);
	DEtake_buffer(de, data);
	for (i = 0; i < de->format_list[formatID].thr_funct_count; i++) {
	    if (de->format_list[formatID].thr_funct[i] != NULL) {
		thr_fork_record sub_rec = DEmalloc(sizeof(*sub_rec));
		thr_thread_t thr;
		rec->thr_count++;
		sub_rec->de = de;
		sub_rec->dep = dep;
		sub_rec->formatID = formatID;
		sub_rec->data = data;
		sub_rec->length = length;
		sub_rec->funct = de->format_list[formatID].thr_funct[i];
		sub_rec->funct_data = de->format_list[formatID].thr_funct_data[i];
		sub_rec->func_record = rec;
		DDBG2(de, "Forking funct #%d for [%s].\n", i,
		      de->format_list[formatID].format_name);
		thr = thr_fork((void_arg_func) thr_handler_wrapper, sub_rec);
		if (thr == NULL) {
		    printf("Ran out of threads in ProcessData!\n");
		} else {
		    thr_thread_detach(thr);
		}
	    }
	}
	if (rec->thr_count == 0) {
	    thr_mutex_unlock(rec->lock);
	    thr_mutex_free(rec->lock);
	    DEfree(rec);
	} else {
	    thr_mutex_unlock(rec->lock);
	}
    }
    return SUCCESS;

}

/* 
 * Create and initialize a connection data structure.
 */
extern DEPort
DEPort_create(de)
DExchange de;
{
    int i;
    DEPort dep = (DEPort) DEmalloc(sizeof(struct _DEPort));
    dep->exchange_name = dep->hostname = NULL;
    dep->exch_pid = dep->port = -1;
    dep->to_port = dep->from_port = NULL;
    dep->read_lock = thr_mutex_alloc();
    dep->write_lock = thr_mutex_alloc();
    dep->notify_on_shutdown = TRUE;
    dep->closed = 0;
    dep->forward_comments = TRUE;
    dep->block_by_default = FALSE;
    dep->num_local_formats = 0;
    dep->local_to_global_formatID = (int *) DEmalloc(sizeof(int) *
						 (de->max_formatID + 1));
    dep->formats = (port_format_info *)
	DEmalloc(sizeof(port_format_info) * de->max_formatID);
    memset(dep->formats, 0, sizeof(port_format_info) * de->max_formatID);
    for (i = 0; i < de->max_formatID; i++) {
	dep->formats[i].forward_data = !dep->block_by_default;
	dep->formats[i].input_format = dep->formats[i].output_format = NULL;
    }
    dep->port_close_handler = NULL;
    dep->dg_read = NULL;
    dep->dg_write = NULL;
    dep->dg_shutdown = NULL;
    dep->childPortCount = 0;
    dep->childPorts = NULL;
    dep->parentPort = NULL;
    dep->de = de;
    return dep;
}


static void
DEPort_set_first_block(dep, block)
DEPort dep;
int block;
{
    int i;
    dep->block_by_default = block;
    for (i = 0; i < dep->de->max_formatID; i++) {
	dep->formats[i].forward_data = !block;
    }
}

extern void
DExchange_set_format_forward(de, format_name, policy)
DExchange de;
char *format_name;
DEForwardStyle policy;
{
    int format_id;
    DExchange_lock(de);
    format_id = get_formatID(de, format_name);
    de->format_list[format_id].forward_policy = policy;
    DExchange_unlock(de);
}

static void
DExchange_set_format_system(de, format_name, system_fmt)
DExchange de;
char *format_name;
int system_fmt;
{
    int format_id;
    DExchange_lock(de);
    format_id = get_formatID(de, format_name);
    de->format_list[format_id].system_format = system_fmt;
    DExchange_unlock(de);
}

/* 
 * tag a format as being fixed and inflexible in its internal representation.
 */
static void
internal_set_format_fixed(de, format_id, value)
DExchange de;
int format_id;
int value;
{
    int i;
    de->format_list[format_id].fixed_format = value;
    for (i = 0; i < de->portCount; i++) {
	DEPort tmp_dep = de->ports[i];
	if (tmp_dep != NULL) {
	    if (tmp_dep->formats[format_id].input_format != NULL) {
		/* make sure we're told if the format changes */
		set_notify_of_format_change(tmp_dep->from_port,
				  de->format_list[format_id].format_name,
					    value);
	    }
	}
    }
}

extern void
DExchange_set_format_fixed(de, format_name, value)
DExchange de;
char *format_name;
int value;
{
    int id;
    DExchange_lock(de);
    id = get_formatID(de, format_name);
    internal_set_format_fixed(de, id, value);
    DExchange_unlock(de);
}


static
void
DExchange_handle_write_error(de, dep)
DExchange de;
DEPort dep;
{
    if (poll_IOfile(dep->from_port)) {
	/* there's still stuff to be read from this port */
	if (dep->to_port)
	    /* 
	     * This port will die soon.  Set to_port to NULL so we don't
	     * write anymore.  However, this leaves a memory leak because the
	     * IOfile is never deallocated.  Fix this sometime.
	     */
	    dep->to_port = NULL;
    } else {
	/* nothing left to read, shut the guy down... */
	DEport_close(dep);
    }
}

/* 
 * Close an on-line connection.
 */
extern void
DEport_close(dep)
DEPort dep;			/* parent dep if connectionless */
{
    DExchange de = dep->de;

    if (dep->closed) return;
    DEport_fail_conditions(dep);

    DExchange_lock(de);
    if (dep->port_close_handler) {
	dep->port_close_handler(de, dep);
    } else if (de->port_close_handler) {
	de->port_close_handler(de, dep);
    }
    if ((dep != NULL) && (dep->childPortCount == 0)) {
	if (dep->from_port) {
	    DEPort_remove_from_control(dep);
	    close_IOfile(dep->from_port);
	}
	if (dep->to_port) {
	    close_IOfile(dep->to_port);
	}
	if (dep->to_port) {
	    free_IOfile(dep->to_port);
	    dep->to_port = NULL;
	}
	if (dep->from_port) {
	    free_IOfile(dep->from_port);
	    dep->from_port = NULL;
	}
	if (dep->read_lock) {
	    thr_mutex_free(dep->read_lock);
	    dep->read_lock = NULL;
	    thr_mutex_free(dep->write_lock);
	    dep->write_lock = NULL;
	}
    } else if (dep != NULL) {	/* connectionless */
	int i;
	for (i = 0; i < dep->childPortCount; i++) {
	    DEPort tmp_dep;
	    tmp_dep = dep->childPorts[i];
	    DEfree(tmp_dep->client_data);
	    DEfree(tmp_dep->local_to_global_formatID);
	    DEfree(tmp_dep->formats);
	    DEfree(tmp_dep->exchange_name);
	    DEfree(tmp_dep->hostname);
	    DEfree(tmp_dep);
	}
	dep->dg_shutdown(dep->client_data);
	/* free(dep->client_data);  could be statically allocated */
    }

    DEfree(dep->local_to_global_formatID);
    DEfree(dep->formats);
    DEfree(dep->exchange_name);
    DEfree(dep->hostname);
    dep->closed = 1;
    DExchange_unlock(de);
}

extern void
DEport_free(dep)
DEPort dep;
{
    remove_port_from_DE(dep->de, dep);
    DEfree(dep);
}

static void
default_close_handler(de, dep)
DExchange de;
DEPort dep;
{
    if (dep) {
	if (dep->to_port) {
	    if (IOhas_error(dep->to_port)) {
		IOperror(dep->to_port, "To port error");
	    }
	}
	if (dep->from_port) {
	    if (IOhas_error(dep->from_port)) {
		IOperror(dep->from_port, "From port error");
	    }
	}
    }
}

#ifdef HAVE_WINDOWS_H
/* Winsock init stuff  */
/* ask for ver 1.1 */
static WORD wVersionRequested = MAKEWORD(1, 1);
static WSADATA wsaData;
int nErrorStatus;
#endif


/* initial handshaking message */
typedef struct _ExchFirstMsg {
    char *exchange_name;
    int exch_pid;
    char *mach_name;
    int port, pid;
    char *usock_name;
    int default_accept;
} ExchFirstMsg, *ExchFirstMsgPtr;

static IOField exch_first_msg_flds[] =
{
    {"hub_name", "string", sizeof(char *), IOOffset(ExchFirstMsgPtr, exchange_name)},
 {"hub_id", "integer", sizeof(int), IOOffset(ExchFirstMsgPtr, exch_pid)},
    {"mach_name", "string", sizeof(char *), IOOffset(ExchFirstMsgPtr, mach_name)},
    {"port", "integer", sizeof(int), IOOffset(ExchFirstMsgPtr, port)},
    {"pid", "integer", sizeof(int), IOOffset(ExchFirstMsgPtr, pid)},
    {"usock_name", "string", sizeof(char *), IOOffset(ExchFirstMsgPtr, usock_name)},
    {"default_forward", "integer", sizeof(int),
     IOOffset(ExchFirstMsgPtr, default_accept)},
    {(char *) 0, (char *) 0, 0, 0}
};


/* format sign on or off msg */
typedef struct _FormatSignMsg {
    int requester_id;		/* exch_pid of requester */
    char *format_name;
} FormatSignMsg, *FormatSignMsgPtr;

static IOField format_sign_msg_flds[] =
{
    {"requester_id", "integer", sizeof(int),
     IOOffset(FormatSignMsgPtr, requester_id)},
    {"format_name", "string", sizeof(char *),
     IOOffset(FormatSignMsgPtr, format_name)},
    {(char *) 0, (char *) 0, 0, 0}
};


/* shut down system or just close a connection link */
typedef struct _ShutDownMsg {
    int whom;			/* shut down whom */
    char *message;
    char *exchange_name;
    int exch_pid;
} ShutDownMsg, *ShutDownMsgPtr;

static IOField shut_down_msg_flds[] =
{
    {"whom", "integer", sizeof(int), IOOffset(ShutDownMsgPtr, whom)},
{"message", "string", sizeof(char *), IOOffset(ShutDownMsgPtr, message)},
    {"hub name", "string", sizeof(char *), IOOffset(ShutDownMsgPtr, exchange_name)},
  {"hub id", "integer", sizeof(int), IOOffset(ShutDownMsgPtr, exch_pid)},
    {(char *) 0, (char *) 0, 0, 0}
};

extern IOField channel_attend_msg_flds[];
extern IOField sink_subscribe_msg_flds[];
extern IOField source_subscribe_msg_flds[];
extern IOField request_event_msg_flds[];
extern IOField AttendRec_flds[];
extern IOField channel_attend_response_msg_flds[];
extern IOField channel_exists_attend_response_msg_flds[];
extern IOField member_subscribe_msg_flds[];
extern IOField event_msg_flds[];
extern IOField event_vec_elem_flds[];
extern IOField eventv_msg_flds[];

extern IOField field_list_flds[];
extern IOField format_list_flds[];
extern IOField channel_derive_msg_flds[];
extern IOField src_derive_msg_flds[];
extern IOField src_derive_resp_msg_flds[];
extern IOField proto_derive_msg_flds[];
extern IOField data_update_msg_flds[];

struct sys_format {
    char *name;
    IOField *field_list;
    DataHandlingFunc func;
    int *format_num_pointer;
};

struct sys_format sys_format_list[] =
{
    {"format-signon-message", format_sign_msg_flds, process_signon_message,
     &DE_format_signon_message_format_id},
    {"format-signoff-message", format_sign_msg_flds, process_signoff_message,
     &DE_format_signoff_message_format_id},
    {"shut-down-message", shut_down_msg_flds, process_shutdown_message,
     &DE_shut_down_message_format_id},
    {"Initialize Datahub Connection", exch_first_msg_flds, NULL, NULL},
    {"Channel Attend", channel_attend_msg_flds, DEChannel_Attend_handler, 
     &DE_Channel_Attend_format_id},
    {"Sink Subscribe", sink_subscribe_msg_flds,
     DEChannel_SinkSubscribe_handler, &DE_Sink_Subscribe_format_id},
    {"Sink Unsubscribe", sink_subscribe_msg_flds,
     DEChannel_SinkUnsubscribe_handler, &DE_Sink_Unsubscribe_format_id},
    {"Source Subscribe", source_subscribe_msg_flds,
     DEChannel_SourceSubscribe_handler, &DE_Source_Subscribe_format_id},
    {"Source Unsubscribe", source_subscribe_msg_flds,
     DEChannel_SourceUnsubscribe_handler, &DE_Source_Unsubscribe_format_id},
    {"Request Event", request_event_msg_flds,
     DEChannel_RequestEvent_handler, &DE_Request_Event_format_id},
    {"Member Subscribe", member_subscribe_msg_flds,
     DEChannel_MemberSubscribe_handler, &DE_Member_Subscribe_format_id},
    {"AttendRec", AttendRec_flds, NULL, NULL},
    {"Channel Attend Response", channel_attend_response_msg_flds,
     DEChannel_Attend_Response_handler, &DE_Channel_Attend_Response_format_id},
    {"Channel Exists Attend Response", channel_exists_attend_response_msg_flds,
     DEChannel_Exists_Attend_Response_handler, 
     &DE_Channel_Exists_Attend_Response_format_id},
    {"Event Message", event_msg_flds, DEChannel_Event_handler, 
     &DE_Event_Message_format_id},
    {"EventVecElem", event_vec_elem_flds, NULL, NULL},
    {"EventV Message", eventv_msg_flds, DEChannel_EventV_handler, 
     &DE_EventV_Message_format_id},

    /* support for derived event channels */
    {"IOfield_list", field_list_flds, NULL, NULL},
    {"DEFormatList", format_list_flds, NULL, NULL},
    {"Channel Derive", channel_derive_msg_flds,
     DEChannel_Derive_handler, &DE_Channel_Derive_format_id},
    {"Channel Source Derive", src_derive_msg_flds,
     DEChannel_Source_Derive_handler, &DE_Channel_Source_Derive_format_id},
    {"Channel Source Derive Resp", src_derive_resp_msg_flds,
     DEChannel_Source_Derive_Resp_handler, 
     &DE_Channel_Source_Derive_Resp_format_id},
    {"Proto Derive", proto_derive_msg_flds, DEProto_Derive_handler, 
     &DE_Proto_Derive_format_id},
    {"Data Update Message", data_update_msg_flds, DEData_Update_handler, 
     &DE_Data_Update_Message_format_id},

    {NULL, NULL, NULL, NULL}};

static int sys_format_id_max = 0;

/* 
 * Create and initialize a data exchange.
 */
extern DExchange
DExchange_create()
{
    int len;
    int i;
    char name_buf[256];
    DExchange de = (DExchange) DEmalloc(sizeof(DExchange_s));

    if (de == NULL)
	return NULL;
    /* initialize the winsock package */
#ifdef HAVE_WINDOWS_H
    nErrorStatus = WSAStartup(wVersionRequested, &wsaData);
    if (nErrorStatus != 0) {
	fprintf(stderr, "Could not initialize windows socket library!");
	WSACleanup();
	exit(-1);
    }
#endif

    /* initialize data structs */
    DEget_qual_hostname(name_buf, 128);
    len = strlen(name_buf);
    name_buf[len] = '_';
    sprintf(&name_buf[len + 1], "%u", (unsigned int) getpid());
    de->exchange_name = strdup(name_buf);
    de->exch_pid = getpid();
    de->port = -1;
    DEget_qual_hostname(name_buf, 128);
    de->hostname = strdup(name_buf);
    de->default_block = FALSE;
    de->forward_default = DEAll_But_Sender;
    de->enable_debug = /*TRUE*/FALSE; /* no debugging */
    de->debug_hub = NULL;
    de->debug_flag = 0;
    de->data_debug_flag = 0;
    de->no_debug_flag = 0;

    de->portCount = 0;
    de->ports = (DEPort *) DEmalloc(sizeof(DEPort));
    memset((char *) de->ports, 0, sizeof(DEPort));
    de->pending_conns = DEmalloc(sizeof(struct PendingConnElement));
    memset((char *) de->pending_conns, 0, sizeof(struct PendingConnElement));
    de->format_list = (ioformat_data *) DEmalloc(sizeof(ioformat_data));
    memset((char *) de->format_list, 0, sizeof(ioformat_data));
    de->max_formatID = 0;

    de->control_list = DEControlList_create();
    de->buffer_data = malloc(2*sizeof(de->buffer_data[0]));
    de->max_handler_level = 1;
    de->handler_level = 1;
    DATA_BUF(de) = NULL;
    BUF_SIZE(de) = 0;
    BUF_TAKEN(de) = 0;
    de->handler_level = 0;
    DATA_BUF(de) = DEmalloc(1);
    BUF_SIZE(de) = 1;
    BUF_TAKEN(de) = 0;
    de->format_handler = NULL;
    de->comment_handler = NULL;
    de->open_handler = NULL;
    de->port_close_handler = default_close_handler;
    de->close_callbacks = NULL;
    de->exchange_lock = thr_mutex_alloc();
    de->locked = 0;
    de->closed = 0;
    de->queued_records = NULL;
    de->active_fields = NULL;

    if (getenv("PBIO_BIG_FORMAT_IDS") == NULL) {
	putenv("PBIO_FIXED_FORMAT_IDS=1");
    }
    de->context = NULL;
    de->context_lock = thr_mutex_alloc();

    /* defaults for network interface */
    de->init_conn_func = DExchange_inet_initiate_conn;
    de->listen_func = DExchange_inet_listen;

    de->channel_count = 0;
    de->channel_list = NULL;

    de->monitor_info = NULL;
    /* 
     * ignore SIGPIPE's  (these pop up when ports die.  we catch the 
     * failed writes) 
     */
#ifdef SIGPIPE
    signal(SIGPIPE, SIG_IGN);
#endif

    /* register system formats */
    i = 0;
    while (sys_format_list[i].name != NULL) {
	DExchange_register_format(de, sys_format_list[i].name,
				  sys_format_list[i].field_list);
	if (sys_format_list[i].func != NULL) {
	    DExchange_register_function(de, sys_format_list[i].name,
					sys_format_list[i].func, NULL);
	}
	DExchange_set_format_system(de, sys_format_list[i].name, TRUE);

	/*
	 * warning.  In the name of efficiency, violating a bit of 
	 * compartmentalization in the assignments of format ids.
	 */
	if (sys_format_list[i].format_num_pointer != NULL) {
	    *sys_format_list[i].format_num_pointer = (de->max_formatID -1);
	}
	i++;
    }
    sys_format_id_max = de->max_formatID;
    DEMon_dexchange_create_sensor(de);
    return de;
}

static void
add_port_to_DE(de, dep)
DExchange de;
DEPort dep;
{
    de->ports = (DEPort *) DErealloc(de->ports,
				   sizeof(DEPort) * (de->portCount + 2));
    de->ports[de->portCount++] = dep;
}

static void
add_childPort_to_parentPort(dep, child_dep)
DEPort dep;
DEPort child_dep;
{
    if (dep->de != child_dep->de) {
	assert(FALSE);
    } else {
	dep->childPorts = (DEPort *) DErealloc(dep->childPorts,
					       sizeof(DEPort) *
					       (dep->childPortCount + 2));
	dep->childPorts[dep->childPortCount++] = child_dep;
    }
}

static void
remove_port_from_DE(de, dep)
DExchange de;
DEPort dep;
{
    int i = 0;
    while ((i < de->portCount) && (de->ports[i] != dep)) {
	i++;
    }
    if (i == de->portCount) {
	/* assert(FALSE); */
	/* port may not be there yet if the file is not truely open */
    } else {
	while (i < (de->portCount - 1)) {
	    de->ports[i] = de->ports[i + 1];
	    i++;
	}
	de->portCount--;
    }
}


/* 
 * Destroy a data exchange.
 */
void
DExchange_close(de)
DExchange de;
{
    int i;
    if (de != NULL) {
	for(i=0; i < de->portCount; i++) {
	    DEport_close(de->ports[i]);
	}

	if (de->close_callbacks != NULL) {
	    FunctionListElement *callbacks = de->close_callbacks;
	    while (de->close_callbacks->func != NULL) {
		de->close_callbacks->func(de->close_callbacks->dep,
					  de->close_callbacks->arg);
		callbacks++;
	    }
	    DEfree(de->close_callbacks);
	}
	/* destroy data structs */
	for(de->handler_level=0; de->handler_level<de->max_handler_level; de->handler_level++) {
	    if (DATA_BUF(de)) {
		DEfree(DATA_BUF(de));
	    }
	}
	DEfree(de->buffer_data);
	for (i = 0; i < de->max_formatID; i++) {
	    if (de->format_list[i].funct != 0)
		DEfree(de->format_list[i].funct);
	    if (de->format_list[i].funct_data != 0)
		DEfree(de->format_list[i].funct_data);
	    if (de->format_list[i].thr_funct != 0)
		DEfree(de->format_list[i].thr_funct);
	    if (de->format_list[i].thr_funct_data != 0)
		DEfree(de->format_list[i].thr_funct_data);
	    if (de->format_list[i].filter != 0)
		DEfree(de->format_list[i].filter);
	    if (de->format_list[i].filter_data != 0)
		DEfree(de->format_list[i].filter_data);
	    if (de->format_list[i].format_name != 0)
		DEfree(de->format_list[i].format_name);
	    if (de->format_list[i].internal_field_list != 0)
		free_field_list(de->format_list[i].internal_field_list);
	}
	DEControlList_close(de->control_list);
	if (de->debug_hub) {
	    DExchange_close(de->debug_hub);
	    DExchange_free(de->debug_hub);
	}
	DEfree(de->format_list);
	DEfree(de->hostname);
	DEfree(de->exchange_name);
	DEfree(de->pending_conns);
	if (de->channel_list) {
	    DEfree(de->channel_list);
	}
	thr_mutex_free(de->exchange_lock);
	if (de->monitor_info) {
	    DEfree(de->monitor_info);
	}
	if (de->context) {
	    free_IOcontext(de->context);
	}
	thr_mutex_free(de->context_lock);
    }
}

extern void
DExchange_free(de)
DExchange de;
{
    while(de->portCount != 0) {
	DEport_free(de->ports[0]);
    }
    DEfree(de->ports);
    DEControlList_free(de->control_list);
    DEfree(de);
}

/* 
 * Create an IP socket for connection from other data exchanges.
 */
extern int
DExchange_open_inet(de)
DExchange de;
{
    return DExchange_listen(de, 0);
}


extern DEPort
DExchange_get_conn(de, host, port, block_by_default)
DExchange de;
char *host;
int port;
int block_by_default;
{
    int i;
    DEPort dep = NULL;
    DExchange_lock(de);
    for (i = 0; i < de->portCount; i++) {
	if ((DEport_port_number(de->ports[i]) == port) &&
	    (strcmp(DEport_host_name(de->ports[i]), host) == 0)) {
	    DExchange_unlock(de);
	    return de->ports[i];
	}
    }
    if (is_pending_conn(de, host, port)) {
	int condition = DECondition_get(de, NULL);
	/* dep will be set before condition is released */
	DECondition_set_client_data(de, condition, &dep);
	add_to_pending_wait_list(de, host, port, condition);
	DExchange_unlock(de);
	DECondition_wait(de, condition);
	return dep;
    } else {
	add_to_pending_conn_list(de, host, port);
    }
    DExchange_unlock(de);
    dep = DExchange_initiate_conn(de, host, port, block_by_default);
    DExchange_lock(de);
    remove_from_pending_conn_list(de, host, port, dep);
    DExchange_unlock(de);
    return dep;
}

#ifndef SOCKET_ERROR
#define SOCKET_ERROR -1
#endif

/* 
 * DExchange_accept_port_init() handles the handshaking from the side of the
 * initiating port.  The DataExchange initiation handshake is asymmetric, so
 * one side should call this after the net connection is setup (typically by
 * DExchange_accept_conn_sock() or a similar routine) and the other side
 * should call DExchange_initiate_port_init() (Typically by
 * DExchange_initiate_conn() or a similar routine). 
 */
extern int
DExchange_accept_port_init(dep, from_port, to_port)
DEPort dep;
IOFile from_port;
IOFile to_port;
{
    /* 
     *  Presumably no locking necessary.  This dep is only known to the 
     *  guy who called us, so there should be no interference. 
     */
    DExchange de = dep->de;
    char name_buf[256];
    ExchFirstMsg exch_first_msg, port_first_msg;
    IOFormat exch_first_msg_fmt;

    dep->from_port = from_port;
    dep->to_port = to_port;

    if (!open_created_IOfile(from_port, "r")) {
	return FALSE;
    }
    set_IOconversion(from_port, "Initialize Datahub Connection",
		     exch_first_msg_flds,
		     sizeof(port_first_msg));

    if (read_IOfile(from_port, &port_first_msg) == 0) {
	return FALSE;
    }
    DEPort_set_first_block(dep, !port_first_msg.default_accept);

    dep->exchange_name =
	(port_first_msg.exchange_name) ? strdup(port_first_msg.exchange_name) : strdup("");
    dep->exch_pid = port_first_msg.exch_pid;
    dep->port = port_first_msg.port;
    dep->hostname =
	(port_first_msg.mach_name) ? strdup(port_first_msg.mach_name) : strdup("");
    dep->block_by_default = !port_first_msg.default_accept;

    if (!open_created_IOfile(to_port, "w")) {
	return FALSE;
    }
    add_port_to_DE(de, dep);

    record_blocking_state(write_format_state, dep);
    exch_first_msg_fmt =
	register_IOrecord_format("Initialize Datahub Connection",
				 exch_first_msg_flds,
				 dep->to_port);
    clear_blocking_state();
    if (exch_first_msg_fmt == NULL) {
	return FALSE;
    }
    exch_first_msg.exchange_name = de->exchange_name;
    exch_first_msg.exch_pid = de->exch_pid;
    DEget_qual_hostname(name_buf, 128);
    exch_first_msg.mach_name = &name_buf[0];
    exch_first_msg.port = de->port;
    exch_first_msg.pid = getpid();
    exch_first_msg.usock_name = NULL;
    exch_first_msg.default_accept = !de->default_block;
    record_blocking_state(write_record_state, dep);
    if (write_IOfile(dep->to_port, exch_first_msg_fmt, &exch_first_msg) == 0) {
	return FALSE;
    }
    clear_blocking_state();

    DBG1(de, "Accepting Connection from \"%s\"\n", dep->exchange_name);

    DBG4(de, "\tConnecting exchange_pid %lx, listen port = %d\n\thostname = %s, block_by_default=%d\n",
	 (long) dep->exch_pid, dep->port,
	 dep->hostname, dep->block_by_default);
    if (!port_first_msg.mach_name)
	dep->hostname = NULL;
    if (de->open_handler) {
	de->open_handler(de, dep);
    }
    DEMon_deport_create_sensor(de, dep);
    return TRUE;
}

/* 
 * DExchange_initiate_port_init() handles the handshaking from the side of the
 * initiating port.  The DataExchange initiation handshake is asymmetric, so
 * one side should call this after the net connection is setup (typically by
 * DExchange_initiate_conn() or a similar routine) and the other side should
 * call DExchange_accept_port_init() (Typically by
 * DExchange_accept_conn_sock() or a similar routine). 
 */
extern int
DExchange_initiate_port_init(dep, from_port, to_port, block_by_default)
DEPort dep;
IOFile from_port;
IOFile to_port;
int block_by_default;
{
    /* 
     *  Presumably no locking necessary.  This dep is only known to the 
     *  guy who called us, so there should be no interference. 
     */
    DExchange de = dep->de;
    char name_buf[256];
    ExchFirstMsg exch_first_msg, port_first_msg;
    IOFormat exch_first_msg_fmt;

    dep->from_port = from_port;
    dep->to_port = to_port;

    if (!open_created_IOfile(to_port, "w")) {
	return FALSE;
    }
    exch_first_msg_fmt =
	register_IOrecord_format("Initialize Datahub Connection",
				 exch_first_msg_flds,
				 to_port);
    if (exch_first_msg_fmt == NULL) {
	return FALSE;
    }
    exch_first_msg.exchange_name = de->exchange_name;
    exch_first_msg.exch_pid = de->exch_pid;
    DEget_qual_hostname(name_buf, 128);
    exch_first_msg.mach_name = &name_buf[0];
    exch_first_msg.port = de->port;
    exch_first_msg.pid = getpid();
    exch_first_msg.usock_name = NULL;
    exch_first_msg.default_accept = !block_by_default;
    if (write_IOfile(to_port, exch_first_msg_fmt, &exch_first_msg) == 0) {
	return FALSE;
    }
    if (!open_created_IOfile(from_port, "r")) {
	return FALSE;
    }
    set_IOconversion(from_port, "Initialize Datahub Connection",
		     exch_first_msg_flds,
		     sizeof(port_first_msg));
    read_IOfile(from_port, &port_first_msg);

    DEPort_set_first_block(dep, !port_first_msg.default_accept);
    dep->exchange_name =
	(port_first_msg.exchange_name) ? strdup(port_first_msg.exchange_name) : strdup("");
    dep->exch_pid = port_first_msg.exch_pid;
    dep->port = port_first_msg.port;
    dep->hostname =
	(port_first_msg.mach_name) ? strdup(port_first_msg.mach_name) : strdup("");
    dep->block_by_default = !port_first_msg.default_accept;
    add_port_to_DE(de, dep);

    DBG2(de, "%s: DExchange_initiate_conn successful, exchange_name is %s\n",
	 de->exchange_name, dep->exchange_name);

    DBG5(de, "dep->exch_name = %s, exch_pid %lx, port = %d, hostname = %s, block_by_default=%d\n",
	 dep->exchange_name, (long) dep->exch_pid, dep->port,
	 dep->hostname, dep->block_by_default);
    if (!port_first_msg.mach_name)
	dep->hostname = NULL;

    if (de->open_handler) {
	de->open_handler(de, dep);
    }
    return TRUE;
}

/* 
 * Setup an INET datagram socket.
 */
DEPort
DExchange_dg_init_conn(de, block_by_default, startup, client_data)
DExchange de;
int block_by_default;
StartUpFunc startup;
void *client_data;
{
    DEPort dep;
    char name_buf[256];
    int buf_size = 0;
    int required_size;
    char *xfer_buffer;
    char read_buf[MAX_DG_BUF_SIZ], *read_buf_p = &read_buf[0];
    int read_buf_siz = MAX_DG_BUF_SIZ;
    ExchFirstMsg exch_first_msg, *port_first_msg_p;
    IOFormat exch_first_msg_fmt;

    if (de->context == NULL) {
	/* create IO context if it does not already exist */
	de->context = create_IOcontext(NULL);
    }
    dep = DEPort_create(de);
    dep->childPortCount = 0;
    dep->childPorts = (DEPort *) DEmalloc(sizeof(DEPort));
    memset((char *) dep->childPorts, 0, sizeof(DEPort));
    dep->client_data = client_data;
    add_port_to_DE(de, dep);
    register_format_to_port(dep, DE_format_signon_message_format_id);
    register_format_to_port(dep, DE_format_signoff_message_format_id);
    register_format_to_port(dep, DE_shut_down_message_format_id);

    startup(de, dep, &client_data);
    if ((exch_first_msg_fmt =
	 register_format_to_port(dep, get_formatID(de, "Initialize Datahub Connection")))
	== NULL) {
	return NULL;
    }
    exch_first_msg.exchange_name = strdup(de->exchange_name);
    exch_first_msg.exch_pid = de->exch_pid;
    DEget_qual_hostname(name_buf, 128);
    exch_first_msg.mach_name = &name_buf[0];
    exch_first_msg.port = de->port;
    exch_first_msg.pid = getpid();
    exch_first_msg.usock_name = NULL;
    exch_first_msg.default_accept = !block_by_default;
    DExchange_context_lock(de);
    xfer_buffer = encode_IOcontext_buffer(de->context,
					  exch_first_msg_fmt,
					  (void *) &exch_first_msg,
					  &buf_size);
    DExchange_context_unlock(de);
    if (dep->dg_write(xfer_buffer, buf_size, &client_data) != 0) {
	printf("DExchange_dg_initiate_conn:  failed writing first message\n");
	return NULL;
    }
    if (dep->dg_read(&read_buf_p, &read_buf_siz, (void **) &client_data) <= 0) {
	printf("\t\tDExchange_dg_init_conn receive error\n");
	exit(1);
    }
    required_size = this_IOrecord_length(de->context, read_buf, 
					 read_buf_siz);
    if (BUF_SIZE(de) < required_size) {
	DATA_BUF(de) = DErealloc(DATA_BUF(de), required_size);
	BUF_SIZE(de) = required_size;
    }
    DExchange_context_lock(de);
    if (!decode_to_buffer_IOcontext(de->context, read_buf,
				    (void *) DATA_BUF(de))) {
	printf("\t\tDExchange_dg_init_conn - could not decode message\n");
	exit(1);
    }
    DExchange_context_unlock(de);
    port_first_msg_p = (ExchFirstMsg *) DATA_BUF(de);

    dep->exchange_name =
	(port_first_msg_p->exchange_name) ? strdup(port_first_msg_p->exchange_name) : strdup("");
    dep->exch_pid = port_first_msg_p->exch_pid;
    dep->hostname =
	(port_first_msg_p->mach_name) ? strdup(port_first_msg_p->mach_name) :
	strdup("");
    dep->block_by_default = !port_first_msg_p->default_accept;

    DBG1(de, "%s: DExchange_dg_init_conn successful\n", de->exchange_name);
    DBG4(de, "exch_pid %lx, port = %d, hostname = %s, block_by_default=%d\n",
	 (long) dep->exch_pid, dep->port,
	 dep->hostname, dep->block_by_default);
    if (!port_first_msg_p->mach_name)
	dep->hostname = NULL;
    if (de->open_handler) {
	de->open_handler(de, dep);
    }
    return dep;
}

extern void
DExchange_dg_assignFuncs(dep, read_c, write_c, shutdown_c,
			 get_new_cli_c, compareAddr_c)
DEPort dep;
ReadFunc read_c;
WriteFunc write_c;
ShutdownFunc shutdown_c;
GetNewCliFunc get_new_cli_c;
AddrCmpFunc compareAddr_c;
{
    dep->dg_read = read_c;
    dep->dg_write = write_c;
    dep->dg_shutdown = shutdown_c;
    dep->dg_get_cli = get_new_cli_c;
    dep->dg_addrCmp = compareAddr_c;
}


/* 
 * Setup an INET datagram socket.
 * Called once by user needing to accept connections from others.  
 * Creates parent DEPort for receiving messages.  
 * Subsequent connection requests by clients will cause a child port
 * to be created for this parent port.   The child port contains
 * information about a receiving client and is used when sending messages.
 * A single DEPort with its input port and output port is no longer sufficient
 * since a single open socket may receive from multiple data sources and
 * and send to multiple recipients. 
 */
DEPort
DExchange_dg_accept_conn_setup(de, conn_init, client_data)
DExchange de;
StartUpFunc conn_init;
void *client_data;
{
    DEPort dep;

    if (de->context == NULL) {
	/* create IO context if none already exists */
	de->context = create_IOcontext(NULL);
    }
    dep = DEPort_create(de);
    dep->client_data = client_data;
    dep->childPortCount = 0;
    dep->childPorts = (DEPort *) DEmalloc(sizeof(DEPort));
    memset((char *) dep->childPorts, 0, sizeof(DEPort));
    add_port_to_DE(de, dep);
    if (conn_init(de, dep, &client_data) != 0)
	return NULL;

    record_blocking_state(write_format_state, dep);
    DExchange_context_lock(de);
    if (register_IOcontext_format("Initialize Datahub Connection",
			    exch_first_msg_flds, de->context) == NULL) {
	DExchange_context_unlock(de);
	return NULL;
    }
    clear_blocking_state();
    set_IOconversion((IOFile) de->context, "Initialize Datahub Connection",
		     exch_first_msg_flds, sizeof(ExchFirstMsg));
    DExchange_context_unlock(de);

    if (de->open_handler) {
	de->open_handler(de, dep);
    }
    return dep;
}

/* 
 * Called when initial handshake message arrives from client.
 * Creates child DEport structure for client.  Returns message
 * to client to complete handshake.
 */
void
DExchange_dg_accept_conn(dep, ioformat, buf)
DEPort dep;
IOFormat ioformat;
char *buf;
{
    DExchange de = dep->de;
    DEPort new_dep;
    int buf_siz, cli_siz;
    ExchFirstMsg exch_first_msg;
    char name_buf[256];
    char *xfer_buffer;

    ExchFirstMsg *port_first_msg_p = (ExchFirstMsg *) buf;
    new_dep = DEPort_create(de);	/* create child dep */
    new_dep->exchange_name = (port_first_msg_p->exchange_name) ? strdup(port_first_msg_p->exchange_name) : strdup("");
    new_dep->exch_pid = port_first_msg_p->exch_pid;
    new_dep->hostname =
	(port_first_msg_p->mach_name) ? strdup(port_first_msg_p->mach_name) : strdup("");
    new_dep->block_by_default = !port_first_msg_p->default_accept;

    new_dep->dg_read = dep->dg_read;
    new_dep->dg_write = dep->dg_write;
    new_dep->dg_shutdown = dep->dg_shutdown;
    new_dep->dg_get_cli = dep->dg_get_cli;
    new_dep->dg_addrCmp = dep->dg_addrCmp;
    new_dep->childPortCount = 0;
    new_dep->childPorts = NULL;
    new_dep->parentPort = dep;	/* pointer back to parent */
    cli_siz = dep->dg_get_cli((void **) &new_dep->client_data);
    memcpy(new_dep->client_data, dep->client_data, cli_siz);
    add_childPort_to_parentPort(dep, new_dep);

    exch_first_msg.exchange_name = strdup(de->exchange_name);
    exch_first_msg.exch_pid = de->exch_pid;
    DEget_qual_hostname(name_buf, 128);
    exch_first_msg.mach_name = &name_buf[0];
    /* UDP doesn't know anything about port nums */
    exch_first_msg.port = 0;
    exch_first_msg.pid = getpid();
    exch_first_msg.usock_name = NULL;
    exch_first_msg.default_accept = !de->default_block;

    DExchange_context_lock(de);
    xfer_buffer = encode_IOcontext_buffer(de->context,
					  ioformat,
					  (void *) &exch_first_msg,
					  &buf_siz);
    DExchange_context_unlock(de);
    record_blocking_state(write_record_state, dep);
    if (dep->dg_write(xfer_buffer, buf_siz, &dep->client_data) != 0) {
	printf("DExchange_dg_accept_conn:  failed writing first message\n");
	return;
    }
    clear_blocking_state();

    DBG2(de, "%s: DExchange_dg_accept_conn_sock from \"%s\" successful\n",
	 de->exchange_name, new_dep->exchange_name);
    DBG4(de, "exch_pid %lx, port = %d, hostname = %s, block_by_default=%d\n",
	 (long) new_dep->exch_pid, new_dep->port, new_dep->hostname,
	 new_dep->block_by_default);

    if (!port_first_msg_p->mach_name)
	dep->hostname = NULL;
    if (de->open_handler) {
	de->open_handler(de, dep);
    }
    DEMon_deport_create_sensor(de, dep);
}

/* 
 * Start a connection with a file.  If flag is "r", then this is a
 * file to be used as input.  If flag is "w", this file is used
 * for output.
 */
extern DEPort
DExchange_accept_conn_file(de, file_name, flag)
DExchange de;
char *file_name;
char *flag;
{
    DEPort dep;

    if (flag[0] != 'r' && flag[0] != 'w')
	return NULL;
    if (flag[0] == 'r') {
	dep = DEPort_create(de);
	DEPort_set_first_block(dep, TRUE);
	dep->from_port = open_IOfile(file_name, "r");
	if (dep->from_port == NULL) {
	    fprintf(stderr, "Cannot open PBIO file >%s< for read\n",
		    file_name);
	    DEport_close(dep);
	    return NULL;
	}
	dep->to_port = NULL;
    } else {
	dep = DEPort_create(de);
	DEPort_set_first_block(dep, FALSE);
	dep->from_port = NULL;
	dep->to_port = open_IOfile(file_name, "w");
	if (dep->to_port == NULL) {
	    fprintf(stderr, "Cannot open PBIO file >%s< for write\n",
		    file_name);
	    DEport_close(dep);
	    return NULL;
	}
	dep->from_port = NULL;
    }
    dep->exchange_name = DEmalloc(strlen(file_name) + 20);
    sprintf(dep->exchange_name, "%s_file_%s",
	    (flag[0] == 'r') ? "read_from" : "write_to", file_name);
    dep->exch_pid = -1;
    dep->port = -1;
    dep->hostname = NULL;
    dep->block_by_default = (flag[0] != 'w');
    DExchange_lock(de);
    add_port_to_DE(de, dep);

    if (flag[0] == 'w') {
	register_all_formats(dep);
    }
    DExchange_unlock(de);
    DBG2(de, "%s: DExchange_accept_conn_file for \"%s\" successful\n",
	 de->exchange_name, file_name);

    if (de->open_handler) {
	de->open_handler(de, dep);
    }
    if (dep->from_port) {
	/* 
	 * If this is an input file (I.E. it has a from port) add to the poll
	 * list, not select list, because NT doesn't let us do select() on
	 * files (and it returns TRUE all the time anyway, so why not poll()?
	 */
	DEPort_add_general_poll(dep);
    }
    return dep;
}

extern DExchange
DExchange_of_port(dep)
DEPort dep;
{
    return dep->de;
}

extern int
DEport_write_data(dep, format_id, data)
DEPort dep;
int format_id;
void *data;
{
    DExchange de = dep->de;
    if (dep->closed) return 0;
    if (dep != NULL && dep->dg_write != NULL) {		/* connectionless */
	int buf_size = 0;
	char *xfer_buffer = NULL;
	IOFormat ioformat;
	if (dep->formats[format_id].output_format == NULL) {
	    if (de->format_list[format_id].internal_field_list == NULL) {
		fprintf(stderr, "Format %s not registered\n",
			de->format_list[format_id].format_name);
		return 0;
	    }
	    register_format_to_port(dep, format_id);
	}
	ioformat = dep->formats[format_id].output_format;

	DExchange_context_lock(de);
	/* returns pointer to temporary buffer in 'pbio space' */
	xfer_buffer = encode_IOcontext_buffer(de->context,
					      ioformat,
					      data, &buf_size);
	DEPort_write_lock(de, dep);
	dep->dg_write(xfer_buffer, buf_size, &dep->client_data);
	DEPort_write_unlock(de, dep);
	DExchange_context_unlock(de);
    } else if (dep->to_port) {
	if (dep->formats[format_id].output_format == NULL) {
	    if (de->format_list[format_id].internal_field_list == NULL) {
		fprintf(stderr, "Format %s not registered\n",
			de->format_list[format_id].format_name);
		return 0;
	    }
	    register_format_to_port(dep, format_id);
	}
	if (format_id > sys_format_id_max) {
	    DDBG2(de, "send data of format [%s] to %s\n",
	     de->format_list[format_id].format_name, dep->exchange_name);
	}
	record_blocking_state(write_record_state, dep);
	DEPort_write_lock(de, dep);
	if (write_IOfile(dep->to_port,
			 dep->formats[format_id].output_format,
			 data) != 1) {
	    IOperror(dep->to_port, "DEport_write_data error");
	    fprintf(stderr, " on connection to %s...\n", dep->exchange_name);
	    DExchange_handle_write_error(de, dep);
	}
	DEPort_write_unlock(de, dep);
	clear_blocking_state();
    } else {
	DDBG2(de, "Data not sent to %s, forward_data = %d\n",
	      dep->exchange_name, dep->formats[format_id].forward_data);
    }
    return 1;
}

extern int
DEport_write_data_by_name(dep, format_name, data)
DEPort dep;
char *format_name;
void *data;
{
    DExchange de = dep->de;
    int format_id;
    DExchange_lock(de);
    format_id = get_formatID(de, format_name);
    DExchange_unlock(de);
    return DEport_write_data(dep, format_id, data);
}


extern int
DEport_set_format_block(dep, format_name, block)
DEPort dep;
char *format_name;
int block;
{
    FormatSignMsg fmt_msg;
    DExchange de = dep->de;
    fmt_msg.requester_id = de->exch_pid;
    fmt_msg.format_name = format_name;
    if (block) {
	return DEport_write_data(dep, DE_format_signoff_message_format_id,
				 &fmt_msg);
    } else {
	return DEport_write_data(dep, DE_format_signon_message_format_id,
				 &fmt_msg);
    }
}

extern int
DExchange_shutdown_connection(de, dep, conn_flag, shutdown_msg)
DExchange de;
DEPort dep;
ShutDownFlag conn_flag;
char *shutdown_msg;
{
    ShutDownMsg sd_msg;
    sd_msg.whom = conn_flag;
    sd_msg.message = shutdown_msg;
    sd_msg.exchange_name = de->exchange_name;
    sd_msg.exch_pid = de->exch_pid;
    return DEport_write_data(dep, DE_shut_down_message_format_id, &sd_msg);
}

static int
DExchange_handle_incoming_format ARGS((DExchange de, DEPort dep));
static void
DExchange_handle_incoming_comment ARGS((DExchange de, DEPort dep));

static int
DExchange_handle_incoming_data(de, dep)
DExchange de;
DEPort dep;
{
    /* 
     * dep is read locked coming in.
     */
    IOFormat ioRecordFormat = next_IOrecord_format(dep->from_port);
    int formatID = 0;
    int rec_length = next_IOrecord_length(dep->from_port);

    DExchange_lock(de);
    while (dep->formats[formatID].input_format != ioRecordFormat) {
	formatID++;
	assert(formatID < de->max_formatID);
    }
    DExchange_unlock(de);
    de->handler_level++;
    if (de->handler_level >= de->max_handler_level) {
	de->max_handler_level++;
	de->buffer_data = realloc(de->buffer_data,
				  (de->max_handler_level+1)*sizeof(de->buffer_data[0]));
	de->handler_level++;
	DATA_BUF(de) = NULL;
	BUF_SIZE(de) = 0;
	de->handler_level--;
	DATA_BUF(de) = DEmalloc(1);
	BUF_SIZE(de) = 1;

    }
    if ((BUF_SIZE(de) < rec_length) ||
	BUF_SIZE(de) < next_raw_IOrecord_length(dep->from_port)) {
	BUF_SIZE(de) = rec_length;
	if (BUF_SIZE(de) < next_raw_IOrecord_length(dep->from_port))
	    BUF_SIZE(de) = next_raw_IOrecord_length(dep->from_port);
	DATA_BUF(de) = DErealloc(DATA_BUF(de), BUF_SIZE(de));
	if (DATA_BUF(de) == NULL) {
	    fprintf(stderr, "DataExchange out of memory to buffer incoming record.\n");
	    fprintf(stderr, "Format name is %s.  Raw incoming record size %d.  Converted size is %d.\n",
		    name_of_IOformat(ioRecordFormat),
		    next_raw_IOrecord_length(dep->from_port),
		    next_IOrecord_length(dep->from_port));
	    fprintf(stderr, "On connection to %s...\n", dep->exchange_name);
	    read_to_buffer_IOfile(dep->from_port, DATA_BUF(de),
				  BUF_SIZE(de));
	    BUF_SIZE(de) = 1024;
	    DATA_BUF(de) = DEmalloc(BUF_SIZE(de));
	    DEPort_read_unlock(de, dep);
	    return -1;
	}
    }
    /* use pbio to read the connections */
    if (read_to_buffer_IOfile(dep->from_port, DATA_BUF(de),
			      BUF_SIZE(de)) != 1) {
	IOperror(dep->from_port, "read error");
	fprintf(stderr, "On connection to %s...\n", dep->exchange_name);
	DEPort_read_unlock(de, dep);
	DEport_close(dep);
	return -1;
    }
    DEPort_read_unlock(de, dep);
    ProcessData(de, dep, formatID, DATA_BUF(de), rec_length);
    de->handler_level--;
    return formatID;
}


extern char *
DEport_read_data(dep, ret_format_id)
DEPort dep;
int *ret_format_id;
{
    DExchange de = dep->de;
    int done = 0;

    /* 
     * This is an external user entry point, so read lock the dep now.
     */
    if (dep->closed) return NULL;

    DEPort_read_lock(de, dep);
    if (dep != NULL && dep->dg_write != NULL) {		/* connectionless */
	*ret_format_id = DExchange_dg_general_handler((void *) dep, NULL);
    } else {
	while (!done) {
	    switch (next_IOrecord_type(dep->from_port)) {
	    case IOerror:
		DBG(de, "Got a direct IOerror on read_data\n");
		DEPort_read_unlock(de, dep);
		DEport_close(dep);
		done = TRUE;
		break;
	    case IOend:
		DBG(de, "Got a direct IOend on read_data\n");
		DEPort_read_unlock(de, dep);
		DEport_close(dep);
		done = TRUE;
		break;
	    case IOdata:
		*ret_format_id = DExchange_handle_incoming_data(de, dep);
		/* 
		 * unlock done in handle_incoming_data
		 */
		done = TRUE;
		break;
	    case IOformat:
		DExchange_handle_incoming_format(de, dep);
		/* don't unlock because we're not done */
		break;
	    case IOcomment:
		DExchange_handle_incoming_comment(de, dep);
		/* don't unlock because we're not done */
		break;
	    }
	}

    }
    /* unlock should have been performed when done was set to true */
    if ((dep->from_port == NULL && dep->dg_write == NULL) ||
	(dep->dg_write == NULL && IOhas_error(dep->from_port))) {
	return NULL;
    } else {
	/* actual buffer used is current handler_level + 1 */
	void *buf;
	de->handler_level++;
	buf = DATA_BUF(de);
	de->handler_level--;
	return buf;
    }
}

static void
internal_register_format ARGS((DExchange de, char *format_name,
			       IOFieldList field_list, int pointer_size));

static int
DExchange_handle_incoming_format(de, dep)
DExchange de;
DEPort dep;
{
    IOFormat rec_fmt;
    char *format_name;
    IOFieldList field_list;
    ioformat_data *iof_rec;
    int format_id;

    /* use pbio to read the connections */
    rec_fmt = read_format_IOfile(dep->from_port);

    if (rec_fmt == NULL) {
	IOperror(dep->from_port, "read format error");
	fprintf(stderr, "On connection to %s...\n",
		dep->exchange_name);
	DEport_close(dep);
	return -1;
    }
    format_name = name_of_IOformat(rec_fmt);
    field_list = field_list_of_IOformat(rec_fmt);

    DExchange_lock(de);
    format_id = get_formatID(de, format_name);
    iof_rec = &(de->format_list[format_id]);
    internal_register_format(de, format_name, field_list,
			     pointer_size_of_IOformat(rec_fmt));
    DExchange_unlock(de);

    /* set conversion for the current port */
    set_IOconversion(dep->from_port, format_name,
		     iof_rec->internal_field_list,
		     struct_size_IOfield(dep->from_port,
					 iof_rec->internal_field_list));

    if (is_fixed_format(de, format_id)) {
	/* Don't want to know if file format changes */
	set_notify_of_format_change(dep->from_port, format_name, TRUE);
    } else {
	/* make sure we're told if the file format changes */
	set_notify_of_format_change(dep->from_port, format_name, TRUE);
    }

    if (format_id > sys_format_id_max) {
	DBG2(de, "set conversion %s for %s\n", format_name,
	     dep->exchange_name);
    }
    if (index_of_IOformat(rec_fmt) >= dep->num_local_formats) {
	/* extend local mapping array */
	int count = index_of_IOformat(rec_fmt) + 1;
	dep->num_local_formats = count;
    }
    dep->local_to_global_formatID[index_of_IOformat(rec_fmt)] = format_id;
    dep->formats[format_id].input_format = rec_fmt;

    if ((de->format_handler) && (!de->format_list[format_id].system_format)) {
	de->format_handler(de, dep, format_name);
    }
    return 0;
}


extern void
DExchange_register_format(de, format_name, field_list)
DExchange de;
char *format_name;
IOFieldList field_list;
{
    int format_id;
    ioformat_data *iof_rec;
    DExchange_lock(de);
    format_id = get_formatID(de, format_name);
    iof_rec = &(de->format_list[format_id]);

    if (iof_rec->internal_field_list != NULL) {
	free_field_list(iof_rec->internal_field_list);
	iof_rec->internal_field_list = NULL;
    }
    internal_set_format_fixed(de, format_id, TRUE);
    internal_register_format(de, format_name, field_list, sizeof(char *));
    DExchange_unlock(de);
}

static void
internal_register_format(de, format_name, field_list, pointer_size)
DExchange de;
char *format_name;
IOFieldList field_list;
int pointer_size;
{
    int format_id = get_formatID(de, format_name);
    ioformat_data *iof_rec = &(de->format_list[format_id]);
    IOFieldList max_field_list = NULL;
    int max_pointer_size = pointer_size;

    assert(pointer_size != 0);

    if (iof_rec->internal_field_list == NULL) {
	/* new format */
	max_field_list = copy_field_list(field_list);
    } else {
	/* gotten one of these before */
	if ((is_fixed_format(de, format_id) ||
	     (compare_field_lists(field_list,
				  iof_rec->internal_field_list) == 0))) {

	    /* 
	     * either field lists are the same or or format stays fixed.
	     * In any case, no new max_field_list to use.
	     */
	    max_field_list = NULL;
	} else {
	    /* 
	     * ordering in the max_field_list() call is important.    
	     * max_field_lists() prefers the first list, we want it to
	     * prefer the established one... 
	     */
	    max_field_list = max_field_lists(iof_rec->internal_field_list,
					     field_list);
	    if (compare_field_lists(max_field_list,
				    iof_rec->internal_field_list) == 0) {
		/* old internal format is OK */
		free_field_list(max_field_list);
		max_field_list = NULL;
	    } else {
		max_pointer_size = Max(max_pointer_size,
				       iof_rec->internal_pointer_size);
		force_align_field_list(max_field_list, max_pointer_size);
	    }
	}
    }

    if (max_field_list != NULL) {
	/* 
	 * register new max format and establish new conversions where
	 * format was known before
	 */
	int struct_size = struct_size_field_list(max_field_list,
						 max_pointer_size);

	if (iof_rec->internal_field_list != NULL) {
	    /* 
	     * reformat any stored data records 'cause our internal  
	     * format is changing
	     */
	    int old_struct_size =
	    struct_size_field_list(iof_rec->internal_field_list,
				   iof_rec->internal_pointer_size);
	    DErewrite_queued_data(de, format_id,
				  iof_rec->internal_field_list,
				  old_struct_size,
				  iof_rec->internal_pointer_size,
				  max_field_list, struct_size,
				  max_pointer_size);
	}
	if (iof_rec->internal_field_list != NULL) {
	    free_field_list(iof_rec->internal_field_list);
	}
	iof_rec->internal_field_list = max_field_list;
	iof_rec->internal_pointer_size = max_pointer_size;
	/* 
	 * if forwarding format, re-set conversion, set NULL to ouput 
	 * format pointer for reregistration. 
	 */
	reset_input_conversions(de, format_id);

	/* register the format to all outgoing links */
	register_format_to_all(de, format_id);
    }
}

static void
register_subformats_to_port(dep, format_id)
DEPort dep;
int format_id;
{
    ioformat_data *iof_rec = &(dep->de->format_list[format_id]);
    char **name_list = get_subformat_names(iof_rec->internal_field_list);
    if (name_list != NULL) {
	int i = 0;
	while (name_list[i] != NULL) {
	    int subformat_id = get_formatID(dep->de, name_list[i]);
	    register_subformats_to_port(dep, subformat_id);
	    if (dep->to_port == NULL)
		return;		/* dep went away */
	    register_format_to_port(dep, subformat_id);
	    DEfree(name_list[i]);
	    name_list[i] = NULL;
	    i++;
	}
	DEfree(name_list);
    }
}

static IOFormat
register_format_to_port(dep, format_id)
DEPort dep;
int format_id;
{
    ioformat_data *iof_rec = &(dep->de->format_list[format_id]);
    char *format_name = iof_rec->format_name;

    if (dep != NULL && dep->dg_write != NULL) {		/* connectionless */
	DExchange de = dep->de;
	IOFormat ioformat;
	int ss =
	struct_size_field_list(iof_rec->internal_field_list,
			       iof_rec->internal_pointer_size);
	/* 
	 *  register prior formats to port first to cover data dependencies 
	 */
	DExchange_context_lock(de);
	register_subformats_to_port(dep, format_id);

	/* register this format to context */
	record_blocking_state(write_format_state, dep);
	ioformat = register_IOcontext_format(de->format_list[format_id].format_name,
			  de->format_list[format_id].internal_field_list,
					     de->context);
	clear_blocking_state();
	set_IOconversion((IOFile) de->context,
			 de->format_list[format_id].format_name,
		     de->format_list[format_id].internal_field_list, ss);
	DExchange_context_unlock(de);
	dep->formats[format_id].output_format = ioformat;
	DBG1(de, "register format %s to context\n", format_name);
	if (dep->formats[format_id].output_format == NULL) {
	    fprintf(stderr, "Write format error while sending format to %s\n",
		    dep->exchange_name);
	    return NULL;
	}
	return ioformat;

    } else {

	if (dep->to_port == NULL)
	    return NULL;	/* dep went away */

	if ((dep->formats[format_id].output_format != NULL) &&
	    is_fixed_format(dep->de, format_id)) {
	    return NULL;
	}
	/* 
	 *  register prior formats to port first to cover data dependencies 
	 */
	register_subformats_to_port(dep, format_id);

	if (dep->to_port == NULL)
	    return NULL;	/* dep went away */

	record_blocking_state(write_format_state, dep);
	DEPort_write_lock(dep->de, dep);
	dep->formats[format_id].output_format =
	    register_general_IOrecord_format(format_name,
					     iof_rec->internal_field_list,
					     dep->to_port,
					     iof_rec->internal_pointer_size, 
					     NULL);
	DEPort_write_unlock(dep->de, dep);
	clear_blocking_state();
	if (format_id > sys_format_id_max) {
	    DBG2(dep->de, "register format %s to %s\n", format_name, dep->exchange_name);
	}
	if (dep->formats[format_id].output_format == NULL) {
	    IOperror(dep->to_port, "write format error");
	    fprintf(stderr, "while sending format to %s\n", dep->exchange_name);
	    DExchange_handle_write_error(dep->de, dep);
	}
	return NULL;
    }
}

static void
register_format_to_all(de, format_id)
DExchange de;
int format_id;
{
    int i;
    ioformat_data *iof_rec = &(de->format_list[format_id]);

    if (iof_rec->forward_policy != DENever_Forward) {
	/* register output format */
	for (i = 0; i < de->portCount; i++) {
	    DEPort tmp_dep = de->ports[i];
	    if (tmp_dep != NULL && tmp_dep->to_port) {
		if (tmp_dep->formats[format_id].forward_data == 0) {
		    /* don't bother */
		    continue;
		}
		register_format_to_port(tmp_dep, format_id);
	    }
	}
    }
}

/* 
 * Use for initial registration only!  Sets forward_data field to
 * default. 
 */
static void
register_all_formats(dep)
DEPort dep;
{
    int i;
    for (i = 0; i < dep->de->max_formatID; i++) {
	ioformat_data *iof_rec = &(dep->de->format_list[i]);

	if (iof_rec->forward_policy != DENever_Forward) {
	    /* register output format */
	    if (dep->to_port && (iof_rec->internal_field_list != NULL)) {
		register_format_to_port(dep, i);
		DBG2(dep->de, "Registering format to port %d, bbd = %d\n",
		     i, dep->block_by_default);
	    }
	    dep->formats[i].forward_data = !dep->block_by_default;
	} else {
	    dep->formats[i].forward_data = FALSE;
	}

    }
}

static void
reset_input_conversions(de, format_id)
DExchange de;
int format_id;
{
    int i;
    int ss;
    ioformat_data *iof_rec;
    char *format_name;

    iof_rec = &(de->format_list[format_id]);
    format_name = iof_rec->format_name;
    ss = struct_size_field_list(iof_rec->internal_field_list,
				iof_rec->internal_pointer_size);
    if (de->context) {
	DExchange_context_lock(de);
	register_IOcontext_format(de->format_list[format_id].format_name,
			  de->format_list[format_id].internal_field_list,
				  de->context);
	set_IOconversion((IOFile) de->context,
			 de->format_list[format_id].format_name,
		     de->format_list[format_id].internal_field_list, ss);
	DExchange_context_unlock(de);
    }
    for (i = 0; i < de->portCount; i++) {
	DEPort tmp_dep = de->ports[i];
	if ((tmp_dep != NULL) && (tmp_dep->dg_write == NULL)) {
	    if (tmp_dep->formats[format_id].input_format != NULL) {
		DEPort_read_lock(de, tmp_dep);

		set_general_IOconversion(tmp_dep->from_port, format_name,
					 iof_rec->internal_field_list,
				     ss, iof_rec->internal_pointer_size);
		/* make sure we're told if the format changes */
		set_notify_of_format_change(tmp_dep->from_port,
					    format_name, TRUE);
		DBG2(de, "re-set conversion %s for %s\n", format_name,
		     tmp_dep->exchange_name);
		DEPort_read_unlock(de, tmp_dep);
	    }
	}
    }
}

static void
DExchange_handle_incoming_comment(de, dep)
DExchange de;
DEPort dep;
{
    char *comment;
    int i;

    /* use pbio to read the connections */
    comment = read_comment_IOfile(dep->from_port);
    if (de->comment_handler)
	de->comment_handler(de, dep, comment);

    for (i = 0; i < de->portCount; i++) {
	DEPort tmp_dep = de->ports[i];
	if ((tmp_dep != NULL) && (tmp_dep != dep) &&
	    tmp_dep->forward_comments && tmp_dep->to_port) {
	    record_blocking_state(write_record_state, tmp_dep);
	    write_comment_IOfile(tmp_dep->to_port, comment);
	    clear_blocking_state();
	}
    }
}


static
int
get_formatID(de, name)
DExchange de;
char *name;
{
    int id = -1, i;

    /* do we recognize the new format ?? */
    for (i = 0; i < de->max_formatID; i++) {
	if (strcmp(name, de->format_list[i].format_name) == 0) {
	    id = i;
	}
    }
    if (id < 0) {
	/* Insert the format name into the table and initialize it. */
	ioformat_data *iof_rec;
	id = de->max_formatID++;
	de->format_list =
	    (ioformat_data *) DErealloc(de->format_list,
					de->max_formatID *
					sizeof(struct _ioformat_data));
	iof_rec = &(de->format_list[id]);
	InitFormatNode(name, iof_rec);
	iof_rec->forward_policy = de->forward_default;
	for (i = 0; i < de->portCount; i++) {
	    DEPort tmp_dep = de->ports[i];
	    if (tmp_dep != NULL && tmp_dep->childPortCount == 0) {
		/* connection-oriented */
		tmp_dep->formats = (port_format_info *)
		    DErealloc(tmp_dep->formats,
			    de->max_formatID * sizeof(port_format_info));
		tmp_dep->formats[id].forward_data = !tmp_dep->block_by_default;
		tmp_dep->formats[id].output_format = NULL;
		tmp_dep->formats[id].input_format = NULL;
		tmp_dep->local_to_global_formatID =
		    (int *) DErealloc(tmp_dep->local_to_global_formatID,
				   sizeof(int) * (de->max_formatID + 1));
	    } else if (tmp_dep != NULL) {
		/* connectionless */
		for (i = 0; i < tmp_dep->childPortCount; i++) {
		    DEPort child_dep = tmp_dep->childPorts[i];
		    child_dep->formats = (port_format_info *)
			DErealloc(child_dep->formats,
			    de->max_formatID * sizeof(port_format_info));
		    child_dep->formats[id].forward_data = !child_dep->block_by_default;
		    child_dep->formats[id].output_format = NULL;
		    child_dep->formats[id].input_format = NULL;
		    child_dep->local_to_global_formatID =
			(int *) DErealloc(child_dep->local_to_global_formatID,
				   sizeof(int) * (de->max_formatID + 1));
		}
	    }
	}
    }
    return id;
}

extern void
DEPort_add_general_select(dep, fd)
DEPort dep;
int fd;
{
    DEControlList_add_select(dep->de->control_list, fd,
			  (GenericHandlerFunc) DExchange_general_handler,
			     dep, NULL);
}

extern void
DEPort_add_general_poll(dep)
DEPort dep;
{
    DEControlList_add_poll(dep->de->control_list,
		     (GenericHandlerFunc) DExchange_general_poll_handler,
			   dep, NULL);
}


extern DEtask_handle
DExchange_add_periodic(de, period, func, arg1, arg2)
DExchange de;
long period;
GenericHandlerFunc func;
void *arg1;
void *arg2;
{
    return DEControlList_add_periodic(de->control_list, period, func, 
				      arg1, arg2);
}

extern void
DExchange_add_close(de, func, dep, arg)
DExchange de;
GenericHandlerFunc func;
void *dep;
void *arg;
{
    int count = 0;
    DExchange_lock(de);
    if (de->close_callbacks == NULL) {
	de->close_callbacks = DEmalloc(sizeof(FunctionListElement) * 2);
	de->close_callbacks->func = NULL;
    } else {
	while (de->close_callbacks[count].func != NULL)
	    count++;
	de->close_callbacks = DErealloc(de->close_callbacks,
			      sizeof(FunctionListElement) * (count + 2));
    }
    de->close_callbacks[count].func = func;
    de->close_callbacks[count].dep = dep;
    de->close_callbacks[count].arg = arg;
    de->close_callbacks[count + 1].func = NULL;
    DExchange_unlock(de);
}

static void
DExchange_general_poll_handler(dep, unused)
DEPort dep;
void *unused;
{
    if (poll_IOfile(dep->from_port)) {
	DExchange_general_handler(dep, unused);
    }
}

static void
DExchange_general_handler(dep, unused)
DEPort dep;
void *unused;
{
    DExchange de = dep->de;
    if (dep->from_port == NULL)
	return;

    DEPort_read_lock(de, dep);
    if (dep->from_port == NULL) {
	DEPort_read_unlock(de, dep);
	return;
    }
    switch (next_IOrecord_type(dep->from_port)) {
    case IOerror:
    case IOend:
	DEport_close(dep);
/*	DEPort_read_unlock(de, dep);*/
	break;
    case IOdata:
	DExchange_handle_incoming_data(de, dep);
	/* 
	 * unlock done in handle_incoming_data
	 */
	break;
    case IOformat:
	DExchange_handle_incoming_format(de, dep);
	DEPort_read_unlock(de, dep);
	break;
    case IOcomment:
	DExchange_handle_incoming_comment(de, dep);
	DEPort_read_unlock(de, dep);
	break;
    }
}



/* 
 * connectionless can't look ahead to determine the format type of the
 * next message before the message is retrieved so any branching must
 * occur after the record is read.  Also, all receives are on a single
 * socket so we are forced to assign the socket a single handler for all
 * stages of processing. 
 */
extern int
DExchange_dg_general_handler(dep_param, buffer)
void *dep_param;
void *buffer;
{
    int format_id = 0;
    char *actual_buffer;
    char read_buf[MAX_DG_BUF_SIZ], *read_buf_p = &read_buf[0];
    int read_buf_siz = MAX_DG_BUF_SIZ;
    IOFormat ioformat = NULL;
    DEPort dep = (DEPort) dep_param;
    IOFormat *ioformats = NULL;	/* list of ioformats to register */

    DExchange de = dep->de;

    /* receive from wire */
    if (dep->dg_read(&read_buf_p, &read_buf_siz,
		     &dep->client_data) <= 0) {
	printf("\t\tDExchange_dg_general_handler receive error\n");
	exit(0);
    }
    DEPort_read_unlock(de, dep);
    /* 
     * get list of formats potentially applicable to this message 
     * (perhaps as a nested format) 
     */
    DExchange_context_lock(de);
    if ((ioformats = get_subformats_IOcontext(de->context,
					   (void *) read_buf)) == NULL) {
	printf("\t\tDExchange_dg_general_handler: ioformat list is null\n");
	exit(0);
    }
    DExchange_context_unlock(de);
    /* 
     * compare format list received to list of formats known to this DE.  
     * For each format not known to this DE, make it known.
     */
    DExchange_lock(de);
    while (*ioformats != NULL) {
	for (format_id = 0; format_id < de->max_formatID; format_id++) {
	    if (strcmp(de->format_list[format_id].format_name,
		       name_of_IOformat(*ioformats)) == 0) {
		break;
	    }
	}
	if (format_id >= de->max_formatID ||
	    de->format_list[format_id].internal_field_list == NULL) {
	    internal_register_format(de, name_of_IOformat(*ioformats),
				     field_list_of_IOformat(*ioformats),
				     sizeof(char *));
	}
	ioformats++;
    }
    DExchange_unlock(de);

    if (BUF_SIZE(de) < MAX_DG_BUF_SIZ) {
	DATA_BUF(de) = DErealloc(DATA_BUF(de), MAX_DG_BUF_SIZ);
	BUF_SIZE(de) = MAX_DG_BUF_SIZ;
	if (DATA_BUF(de) == NULL) {
	    fprintf(stderr, "DataExchange out of memory to buffer incoming datagram record.\n");
	    fprintf(stderr, "MAX_DG_BUF_SIZ is %d.\n", MAX_DG_BUF_SIZ);
	    fprintf(stderr, "On connection to %s...\n", dep->exchange_name);
	    BUF_SIZE(de) = 1;
	    DATA_BUF(de) = DEmalloc(BUF_SIZE(de));
	    DEPort_read_unlock(de, dep);
	    return -1;
	}
    }
    actual_buffer = (buffer) ? (buffer) : (DATA_BUF(de));

    DExchange_context_lock(de);
    if (!decode_to_buffer_IOcontext(de->context, read_buf,
				    (void *) actual_buffer)) {
	printf("\t\tDExchange_dg_general_handler: could not decode message\n");
	exit(1);
    }
    ioformat = get_format_IOcontext(de->context, read_buf);
    DExchange_context_unlock(de);

    if (!strcmp(name_of_IOformat(ioformat), "Initialize Datahub Connection")) {
	DExchange_dg_accept_conn(dep, ioformat, actual_buffer);
    } else {
	ProcessData(de, dep, format_id, actual_buffer, BUF_SIZE(de));
    }
    return format_id;
}

extern void
DExchange_register_function(de, fname, funct, user_data)
DExchange de;
char *fname;
DataHandlingFunc funct;
void *user_data;
{
    int formatID;
    ioformat_data *iof_rec;
    int i = 0;

    DExchange_lock(de);
    formatID = get_formatID(de, fname);
    iof_rec = &(de->format_list[formatID]);

    while (i < iof_rec->funct_count && iof_rec->funct[i] != NULL &&
	   iof_rec->funct[i] != funct)
	i++;
    if (i >= iof_rec->funct_count) {
	iof_rec->funct_count++;
	iof_rec->funct =
	    (DataHandlingFunc *) DErealloc(iof_rec->funct, sizeof(DataHandlingFunc)
					   * iof_rec->funct_count);
	iof_rec->funct_data =
	    (void **) DErealloc(iof_rec->funct_data,
				sizeof(void *) * iof_rec->funct_count);
    }
    if (iof_rec->funct[i] == funct) {
	DBG2(de, "Function #%i for format [%s] exists.\n", i, fname);
    } else {
	iof_rec->funct[i] = funct;
	iof_rec->funct_data[i] = user_data;
	if (formatID > sys_format_id_max) {
	    DBG2(de, "Registered new function for format [%s] as function #%d.\n",
		 fname, i);
	    DEMon_register_func_sensor(de, fname, i);
	}
    }
    DExchange_unlock(de);
}

extern void
DExchange_register_threaded_function(de, fname, funct, user_data)
DExchange de;
char *fname;
DataHandlingFunc funct;
void *user_data;
{
    int formatID;
    ioformat_data *iof_rec;
    int i = 0;

    DExchange_lock(de);
    formatID = get_formatID(de, fname);
    iof_rec = &(de->format_list[formatID]);

    while (i < iof_rec->thr_funct_count && iof_rec->thr_funct[i] != NULL &&
	   iof_rec->thr_funct[i] != funct)
	i++;
    if (i >= iof_rec->thr_funct_count) {
	iof_rec->thr_funct_count++;
	iof_rec->thr_funct =
	    (DataHandlingFunc *) DErealloc(iof_rec->thr_funct, sizeof(DataHandlingFunc)
					   * iof_rec->thr_funct_count);
	iof_rec->thr_funct_data =
	    (void **) DErealloc(iof_rec->thr_funct_data,
			      sizeof(void *) * iof_rec->thr_funct_count);
    }
    if (iof_rec->thr_funct[i] == funct) {
	DBG2(de, "Function #%i for format [%s] exists.\n", i, fname);
    } else {
	iof_rec->thr_funct[i] = funct;
	iof_rec->thr_funct_data[i] = user_data;
	if (formatID > sys_format_id_max) {
	    DBG2(de, "Registered new function for format [%s] as function #%d.\n",
		 fname, i);
	    DEMon_register_func_sensor(de, fname, i);
	}
    }
    DExchange_unlock(de);
}

/* if funct == NULL, unregister all */
extern int
DExchange_unregister_function(de, fname, funct, user_data)
DExchange de;
char *fname;
DataHandlingFunc funct;
void *user_data;
{
    int formatID;
    ioformat_data *iof_rec;
    int i = 0;

    DExchange_lock(de);

    formatID = get_formatID(de, fname);
    iof_rec = &(de->format_list[formatID]);

    if (funct == NULL) {
	for (i = 0; i < iof_rec->funct_count; i++) {
	    iof_rec->funct[i] = NULL;
	    iof_rec->funct_data[i] = NULL;
	}
	DExchange_unlock(de);
	return iof_rec->funct_count;
    } else {
	while (i < iof_rec->funct_count && iof_rec->funct[i] != funct)
	    i++;
	if (iof_rec->funct[i] == funct) {
	    iof_rec->funct[i] = NULL;
	    iof_rec->funct_data[i] = NULL;
	    DExchange_unlock(de);
	    return i;
	} else {
	    DExchange_unlock(de);
	    return -1;
	}
    }
}

extern void
DExchange_register_filter(de, fname, funct, user_data)
DExchange de;
char *fname;
DataHandlingFunc funct;
void *user_data;
{
    int formatID;
    ioformat_data *iof_rec;
    int i = 0;

    DExchange_lock(de);

    formatID = get_formatID(de, fname);
    iof_rec = &(de->format_list[formatID]);

    while (i < iof_rec->filter_count && iof_rec->filter[i] != NULL &&
	   iof_rec->filter[i] != funct)
	i++;
    if (i >= iof_rec->filter_count) {
	iof_rec->filter_count++;
	iof_rec->filter =
	    (DataHandlingFunc *) DErealloc(iof_rec->filter,
					   sizeof(DataHandlingFunc)
					   * iof_rec->filter_count);
	iof_rec->filter_data =
	    (void **) DErealloc(iof_rec->filter_data, sizeof(void *)
				* iof_rec->filter_count);
    }
    if (iof_rec->filter[i] == funct) {
	DBG2(de, "Function #%i for format [%s] exists.\n", i, fname);
    } else {
	iof_rec->filter[i] = funct;
	iof_rec->filter_data[i] = user_data;
	if (formatID > sys_format_id_max) {
	    DBG2(de, "Registered new filter for format [%s] as filter #%d.\n",
		 fname, i);
	    DEMon_register_filter_sensor(de, fname, i);
	}
    }
    DExchange_unlock(de);
    return;
}

extern int
DExchange_unregister_filter(de, fname, funct, user_data)
DExchange de;
char *fname;
DataHandlingFunc funct;
void *user_data;
{
    int formatID;
    ioformat_data *iof_rec;
    int i = 0;

    DExchange_lock(de);
    formatID = get_formatID(de, fname);
    iof_rec = &(de->format_list[formatID]);

    if (funct == NULL) {
	for (i = 0; i < iof_rec->filter_count; i++) {
	    iof_rec->filter[i] = NULL;
	}
	DExchange_unlock(de);
	return iof_rec->filter_count;
    } else {
	while (i < iof_rec->filter_count && iof_rec->filter[i] != funct) {
	    i++;
	}
	if (iof_rec->filter[i] == funct) {
	    DExchange_unlock(de);
	    return i;
	} else {
	    DExchange_unlock(de);
	    return -1;
	}
    }
}

IOFile
DEport_from_port(dep)
DEPort dep;
{
    return dep->from_port;
}

IOFile
DEport_to_port(dep)
DEPort dep;
{
    return dep->to_port;
}

char *
DExchange_host_name(de)
DExchange de;
{
    return de->hostname;
}

char *
DExchange_name(de)
DExchange de;
{
    return de->exchange_name;
}

int
DExchange_inet_port(de)
DExchange de;
{
    return de->port;
}

extern char *
DEport_name(dep)
DEPort dep;
{
    return dep->exchange_name;
}

extern char *
DEport_host_name(dep)
DEPort dep;
{
    return dep->hostname;
}

extern DEControlList
DExchange_get_control_list(de)
DExchange de;
{
    return de->control_list;
}

extern int
DEport_port_number(dep)
DEPort dep;
{
    return dep->port;
}


extern void
DExchange_set_port_number(de, port_num)
DExchange de;
int port_num;
{
    de->port = port_num;
    if (de->enable_debug) {
	create_debugging_exchange(de);
    }
}

extern int
DExchange_register_format_handler(de, funct)
DExchange de;
FormatHandlingFunc funct;
{
    de->format_handler = funct;
    return 0;
}

extern int
DExchange_register_comment_handler(de, funct)
DExchange de;
CommentHandlingFunc funct;
{
    de->comment_handler = funct;
    return 0;
}


extern int
DExchange_register_close_handler(de, funct)
DExchange de;
OpenCloseHandlingFunc funct;
{
    de->port_close_handler = funct;
    return 0;
}

extern int
DEport_register_close_handler(dep, funct)
DEPort dep;
OpenCloseHandlingFunc funct;
{
    dep->port_close_handler = funct;
    return 0;
}

extern int
DExchange_register_open_handler(de, funct)
DExchange de;
OpenCloseHandlingFunc funct;
{
    de->open_handler = funct;
    return 0;
}

extern void
DExchange_set_forward(de, style)
DExchange de;
DEForwardStyle style;
{
    de->forward_default = style;
}

extern void
DExchange_set_block_default(de, value)
DExchange de;
int value;
{
    de->default_block = value;
}

static void
create_debugging_exchange(de)
DExchange de;
{
    DExchange debug_de = DExchange_create();
    de->debug_hub = debug_de;
    DEset_debug_flags(debug_de, DENo_Debug, 1);

    debug_de->enable_debug = 0;
    debug_de->monitor_info = DEMon_private_info_create();
    DEControlList_associate(de->control_list, debug_de);
    if (DExchange_listen(debug_de, 0) == -1) {
	fprintf(stderr, "Unable to listen debug exchange");
	exit(-1);
    }
    if (group_server_present()) {
	char *group_id = NULL;
	int i = 0;
	while (group_id == NULL) {
	    char name[256];
	    memset(&name[0], 0, sizeof(name));
	    sprintf(name, "proc_%lx_%d", (long) getpid(), i);
	    group_id =
		setup_specific_comm_group(name, "exchange_debug",
					  60 * 60,
					  DExchange_host_name(debug_de),
					  DExchange_inet_port(debug_de));
	    i++;
	}
	free(group_id);
    }
}

DEPort
DExchange_initiate_conn(de, name_str, int_port_num, block_by_default)
DExchange de;
char *name_str;
int int_port_num;
int block_by_default;
{
    return de->init_conn_func(de, name_str, int_port_num, block_by_default);
}

extern int
DExchange_listen(de, int_port_num)
DExchange de;
int int_port_num;
{
    return de->listen_func(de, int_port_num);
}

extern void
DExchange_set_interface(de, init_conn_func, listen_func)
DExchange de;
InitiateConnFunc init_conn_func;
ListenFunc listen_func;
{
    de->init_conn_func = init_conn_func;
    de->listen_func = listen_func;
}

extern void
DEset_debug_flags(de, flag, value)
DExchange de;
DEDebugFlag flag;
int value;
{
    switch (flag) {
    case DEData_Debug:
	de->data_debug_flag = value;
	break;
    case DEOther_Debug:
	de->debug_flag = value;
	break;
    case DENo_Debug:
	de->no_debug_flag = value;
	break;
    }
}

extern int
DExchange_debugging_on(de)
DExchange de;
{
    /* expression mirrors those in useful.h */
    return ((de_debug_flag || de->debug_flag) && !de->no_debug_flag);
}

static void
add_to_pending_conn_list(de, host, port)
DExchange de;
char *host;
int port;
{
    /* assume exchange is locked */
    int pending_count = 0;
    DBG2(de, "Adding host %s, port %d as pending connection\n", host, port);
    while (de->pending_conns[pending_count].host != NULL)
	pending_count++;

    de->pending_conns = DErealloc(de->pending_conns,
				  (pending_count + 2) *
				  sizeof(struct PendingConnElement));
    de->pending_conns[pending_count].host = host;
    de->pending_conns[pending_count].port = port;
    de->pending_conns[pending_count].notify_list = DEmalloc(sizeof(int));
    de->pending_conns[pending_count].notify_list[0] = -1;
    de->pending_conns[pending_count + 1].host = NULL;
}

static int
is_pending_conn(de, host, port)
DExchange de;
char *host;
int port;
{
    /* assume de is locked */
    PendingConnList list = de->pending_conns;
    while (list->host != NULL) {
	if ((strcmp(host, list->host) == 0) && (port == list->port)) {
	    DBG2(de, "Host %s, port %d IS pending\n", host, port);

	    return 1;
	}
	list++;
    }
    return 0;
}

static void
add_to_pending_wait_list(de, host, port, condition)
DExchange de;
char *host;
int port;
int condition;
{
    PendingConnList list = de->pending_conns;
    while (list->host != NULL) {
	if ((strcmp(host, list->host) == 0) && (port == list->port)) {
	    int notify_count = 0;
	    int *notify_list = list->notify_list;
	    while (*(notify_list++) != -1)
		notify_count++;
	    list->notify_list = DErealloc(list->notify_list,
				       (notify_count + 2) * sizeof(int));
	    list->notify_list[notify_count] = condition;
	    list->notify_list[notify_count + 1] = -1;
	    DBG3(de, "Waiting for host %s, port %d, condition %d\n", host, port, condition);

	    return;
	}
	list++;
    }
    assert(FALSE);
}

static void
remove_from_pending_conn_list(de, host, port, dep)
DExchange de;
char *host;
int port;
DEPort dep;
{
    PendingConnList list = de->pending_conns;
    while (list->host != NULL) {
	if ((strcmp(host, list->host) == 0) && (port == list->port)) {
	    int *notify_list = list->notify_list;
	    while (*notify_list != -1) {
		DEPort *depptr = DECondition_get_client_data(de, *notify_list);
		*depptr = dep;
		DECondition_signal(de, *notify_list);
		DBG3(de, "Notify host %s, port %d, condition %d\n", host, port,
		     *notify_list);
		notify_list++;
	    }
	    DEfree(list->notify_list);
	    while (list->host != NULL) {
		*list = *(list + 1);
		list++;
	    }
	    DBG2(de, "Removed host %s, port %d as pending\n", host, port)
		return;
	}
	list++;
    }
    assert(FALSE);
}

/* 
 * these are initialized (hopefully to the same value) everytime 
 * a DataExchange is created 
 */
int DE_Channel_Attend_Response_format_id = -1;
int DE_Channel_Attend_format_id = -1;
int DE_Channel_Derive_format_id = -1;
int DE_Channel_Exists_Attend_Response_format_id = -1;
int DE_Channel_Source_Derive_Resp_format_id = -1;
int DE_Channel_Source_Derive_format_id = -1;
int DE_Data_Update_Message_format_id = -1;
int DE_EventV_Message_format_id = -1;
int DE_Event_Message_format_id = -1;
int DE_Member_Subscribe_format_id = -1;
int DE_Proto_Derive_format_id = -1;
int DE_Request_Event_format_id = -1;
int DE_Sink_Subscribe_format_id = -1;
int DE_Sink_Unsubscribe_format_id = -1;
int DE_Source_Subscribe_format_id = -1;
int DE_Source_Unsubscribe_format_id = -1;
int DE_format_signoff_message_format_id = -1;
int DE_format_signon_message_format_id = -1;
int DE_shut_down_message_format_id = -1;
