#include "config.h"
#include "io.h"
#ifdef HAVE_GEN_THREAD_H
#include "gen_thread.h"
#else
#define gen_thr_initialized() 0
#define thr_mutex_lock(m)
#define thr_mutex_unlock(m)
#endif
#include "DE.h"
#include "de_internal.h"
#include "de_lock.h"
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#include <stdio.h>
#include <string.h>
#include "useful.h"
#ifndef NO_DERIVED
#include "ecl.h"
#endif
#include "channel.h"

IOField field_list_flds[] =
{
    {"field_name", "string", sizeof(char *), IOOffset(IOFieldList, field_name)},
    {"field_type", "string", sizeof(char *), IOOffset(IOFieldList, field_type)},
    {"field_size", "integer", sizeof(int), IOOffset(IOFieldList, field_size)},
    {"field_offset", "integer", sizeof(int), IOOffset(IOFieldList, field_offset)},
    {(char *) 0, (char *) 0, 0, 0}
};

IOField format_list_flds[] =
{
    {"format_name", "string", sizeof(char *), 
     IOOffset(format_list_element*, format_name)},
    {"field_list_len", "integer", sizeof(int), 
     IOOffset(format_list_element*, field_list_len)},
    {"field_list", "IOfield_list[field_list_len]", sizeof(IOField),
     IOOffset(format_list_element*, field_list)},
    {(char *) 0, (char *) 0, 0, 0}
};

/* message to send to arbitrator to derive a new the channel */
typedef struct _DeriveMsg {
    char *chan_str;
    int cond;
    char client_channel[8];
    char *filter;
    int  field_list_len;
    IOFieldList field_list;
    int  format_list_len;
    format_list_element *format_list;

    char *init_data_block;
    int  init_data_len;
} DeriveMsg, *DeriveMsgPtr;

IOField channel_derive_msg_flds[] =
{
    {"channel", "string", sizeof(char *), IOOffset(DeriveMsgPtr, chan_str)},
    {"condition", "integer", sizeof(int), IOOffset(DeriveMsgPtr, cond)},
    {"client channel", "char[8]", 1, IOOffset(DeriveMsgPtr, client_channel)},
    {"filter", "string", sizeof(char*), IOOffset(DeriveMsgPtr, filter)},
    {"field_list_len", "integer", sizeof(int),
     IOOffset(DeriveMsgPtr, field_list_len)},
    {"field_list", "IOfield_list[field_list_len]", sizeof(IOField),
     IOOffset(DeriveMsgPtr, field_list)},
    {"format_list_len", "integer", sizeof(int),
     IOOffset(DeriveMsgPtr, format_list_len)},
    {"format_list", "DEFormatList[format_list_len]", 
     sizeof(format_list_element), IOOffset(DeriveMsgPtr, format_list)},
    {"init_data_block", "char[init_data_len]", 
     1, IOOffset(DeriveMsgPtr, init_data_block)},
    {"init_data_len", "integer", 
     sizeof(int), IOOffset(DeriveMsgPtr, init_data_len)},
    {(char *) 0, (char *) 0, 0, 0}
};

/* message to send to members to tell them about a derived channel */
typedef struct _SrcDeriveMsg {
    char *der_chan_id;
    int cond;
    char *filter;
    int typed_output;
    char parent_channel[8];
    char *init_data_block;
    char init_data_len;
} SrcDeriveMsg, *SrcDeriveMsgPtr;

IOField src_derive_msg_flds[] =
{
    {"der_channel", "string", sizeof(char *),
     IOOffset(SrcDeriveMsgPtr, der_chan_id)},
    {"condition", "integer", sizeof(int), IOOffset(SrcDeriveMsgPtr, cond)},
    {"filter", "string", sizeof(char*), IOOffset(SrcDeriveMsgPtr, filter)},
    {"typed_output", "integer", sizeof(int), 
     IOOffset(SrcDeriveMsgPtr, typed_output)},
    {"parent_channel", "char[8]", 1, IOOffset(SrcDeriveMsgPtr, parent_channel)},
    {"init_data_block", "char[init_data_len]", 
     1, IOOffset(SrcDeriveMsgPtr, init_data_block)},
    {"init_data_len", "integer", 
     sizeof(int), IOOffset(SrcDeriveMsgPtr, init_data_len)},
    {(char *) 0, (char *) 0, 0, 0}
};

/* message to send to members to tell them about a derived channel */
typedef struct _SrcDeriveRespMsg {
    int cond;
} SrcDeriveRespMsg, *SrcDeriveRespMsgPtr;

IOField src_derive_resp_msg_flds[] =
{
    {"condition", "integer", sizeof(int), IOOffset(SrcDeriveRespMsgPtr, cond)},
    {(char *) 0, (char *) 0, 0, 0}
};

typedef struct _DataUpdateMsg {
    char channel[8];
    char global_chan_id[16];
    int data_len;
    char *data;
} DataUpdateMsg, *DataUpdateMsgPtr;

IOField data_update_msg_flds[] =
{
    {"channel", "char[8]", sizeof(char),
     IOOffset(DataUpdateMsgPtr, channel)},
    {"global_chan_id", "char[16]", sizeof(char),
     IOOffset(DataUpdateMsgPtr, global_chan_id)},
    {"data_len", "integer", sizeof(int),
     IOOffset(DataUpdateMsgPtr, data_len)},
    {"data", "char[data_len]", sizeof(char),
     IOOffset(DataUpdateMsgPtr, data)},
    {(char *) 0, (char *) 0, 0, 0}
};

static DEFormatList get_format_list ARGS((IOFormat ioformat));
static void free_format_list ARGS((DEFormatList list));
static IOFieldList simplify_field_list ARGS((IOFieldList list));
static ECDataHandle new_data_handle();

struct _ECDataHandle {
    IOContext iocontext;
    IOFormat  type_format;
    EChannel  channel;
    IOFieldList data_field_list;
    DEFormatList data_subformat_list;
};

static void
encode_data_block ARGS((ECdata_spec filter_data, char **data_block, 
			int *data_len, ECDataHandle handle, DExchange de));

static EChannel
do_channel_derive ARGS((DExchange de, EChannel orig_channel, char *filter, 
			IOFieldList output_field_list, 
			DEFormatList output_format_list,
			char *init_data, int init_data_len));
static void
do_chan_derive ARGS((EChannel parent_chan, EChannel der_chan, char *filter,
		     int has_output_param));

extern EChannel
EChannel_derive(de, chan_id, event_filter)
DExchange de;
char *chan_id;
char *event_filter;
{
    return EChannel_typed_derive_data(de, chan_id, event_filter, NULL, 
				      NULL, NULL);
}

extern EChannel
EChannel_typed_derive(de, chan_id, event_filter, field_list, format_list)
DExchange de;
char *chan_id;
char *event_filter;
IOFieldList field_list;
DEFormatList format_list;
{
    return EChannel_typed_derive_data(de, chan_id, event_filter, field_list,
				      format_list, NULL);
}

extern EChannel
EChannel_typed_derive_data(de, chan_id, event_filter, field_list,
			   format_list, filter_data)
DExchange de;
char *chan_id;
char *event_filter;
IOFieldList field_list;
DEFormatList format_list;
ECdata_spec filter_data;
{
#ifndef NO_DERIVED
    char *host_str;
    char *chan_str;
    int port_id;
    DEPort dep;
    EChannel chan, parent_chan;
    DeriveMsg derive_msg;
    int cond, i;

    ECtrace_out("ECchannel_derive", 1);

    if (field_list == NULL) {
	ECtrace_out("This is an untyped derivation without an output parameter", 3);
    } else {
	ECtrace_out("This is an typed derivation with an output parameter", 3);
    }
    if (EC_break_up_triple(chan_id, &host_str, &port_id, &chan_str,
			   (void**)&parent_chan) != 0) {
	ECtrace_out("ECchannel_derive", 0);
	return NULL;
    }	

    if ((strcmp(host_str, DExchange_host_name(de)) == 0) &&
	(port_id == DExchange_inet_port(de))) {
	/* this channel is really local */
	char *filter_data_block = NULL;
	int filter_data_len = 0;
	if (filter_data != NULL) {
	    encode_data_block(filter_data, &filter_data_block, 
			      &filter_data_len, NULL, de);
	}
	chan = do_channel_derive(de, parent_chan, event_filter, field_list,
				 format_list, filter_data_block,
				 filter_data_len);
	if (filter_data_block) {
	    DEfree(filter_data_block);
	}
	DEfree(host_str);
	ECtrace_out("ECchannel_derive returning local %lx", 0, (long) chan);
	return chan;
    }

    if (field_list == NULL) {
	chan = EChannel_create(de);
    } else {	
	chan = EChannel_typed_create(de, field_list, format_list);
    }

    ECtrace_out("getting a connection, registering close handler", 3);
    dep = DExchange_get_conn(de, host_str, port_id, 0);
    if (dep == NULL) {
	printf("Failed to contact host of channel \"%s\"\n", chan_id);
	EChannel_destroy(chan);
	DEfree(host_str);
	return NULL;
    }
    DEport_register_close_handler(dep, DEChannel_close_handler);

    ECtrace_out("getting condition and setting local chan condition", 3);
    cond = DECondition_get(de, dep);
    DECondition_set_client_data(de, cond, &chan);

    /* give the filter to the arbitrator for him to propogate */
    /* returns through regular attend response like an open */
    ECtrace_out("creating attend_msg and writing to connection", 3);
    memset(&derive_msg, 0, sizeof(derive_msg));
    derive_msg.chan_str = chan_str;
    derive_msg.cond = cond;
    memcpy(&derive_msg.client_channel[0], &chan, sizeof(chan));
    derive_msg.filter = event_filter;

    /* send output field and format lists */
    derive_msg.field_list = field_list;
    derive_msg.field_list_len = 0;
    while(field_list && 
	  (field_list[derive_msg.field_list_len].field_name != NULL)) {
	derive_msg.field_list_len++;
    }
    if (field_list)
	derive_msg.field_list_len++;  /* send terminator as well */
    i = 0;
    while(format_list && format_list[i].format_name != NULL) i++;
    if (format_list) {
	derive_msg.format_list = DEmalloc(i * sizeof(format_list_element));  
	derive_msg.format_list_len = i;  /* +1 because we send terminator */
    } else {
	derive_msg.format_list = NULL;
	derive_msg.format_list_len = 0;
    }
    for (i=0; i<derive_msg.format_list_len; i++) {
	int j = 0;
	derive_msg.format_list[i].format_name = format_list[i].format_name;
	derive_msg.format_list[i].field_list = format_list[i].field_list;
	while (format_list[i].field_list[j].field_name != NULL) {
	    j++;
	}
	j++; /* send terminator as well */
	derive_msg.format_list[i].field_list_len = j;
    }
    ECtrace_out("Encoding derive message\n", 3);

    encode_data_block(filter_data, &derive_msg.init_data_block, 
		      &derive_msg.init_data_len, NULL, chan->de);
    DEport_write_data(dep, DE_Channel_Derive_format_id, &derive_msg);
    DEfree(host_str);  /* this frees host_str and chan_str */
    DEfree(derive_msg.format_list);
    if (derive_msg.init_data_block != NULL) {
	DEfree(derive_msg.init_data_block);
    }
    ECtrace_out("Sending derive msg, channel %s cond %d filter %d\n", 3,
		derive_msg.chan_str, derive_msg.cond, derive_msg.filter);
    if (DECondition_wait(de, cond)) {
	chan->ready++;
	ECtrace_out("ECchannel_Derive", 0);
	return chan;
    } else {
	/* wait failed, dep is dead */
	EChannel_destroy(chan);
	ECtrace_out("ECchannel_Derive", 0);
	return NULL;
    }
#else
    printf("No derived event channels in this version of DataExchange\n");
    return NULL;
#endif
}

#ifndef NO_DERIVED
static void
add_standard_routines(context)
ecl_parse_context context;
{
    static char extern_string[] = "int printf(string format, ...);";

    static ecl_extern_entry externs[] = 
    {
	{"printf", (void*)(long)printf},
	{(void*)0, (void*)0}
    };
    ecl_assoc_externs(context, externs);
    ecl_parse_for_context(extern_string, context);
}

static ecl_code
generate_filter_code(filter, src, input_field_list, input_format_list, 
		     output_field_list, output_format_list,
		     data_field_list, data_format_list)
char *filter;
ECSourceHandle src;
IOFieldList input_field_list;
DEFormatList input_format_list;
IOFieldList output_field_list;
DEFormatList output_format_list;
IOFieldList data_field_list;
DEFormatList data_format_list;
{
    sm_ref typ, output_type, input_type, input_param, output_param;
    ecl_parse_context parse_context = new_ecl_parse_context();
    ecl_code code;

    add_standard_routines(parse_context);
    while ((input_format_list != NULL) && 
	   (input_format_list->format_name != NULL)) {
	/* step through input formats */
	typ = ecl_build_type_node(input_format_list->format_name,
			      input_format_list->field_list);
	ecl_add_decl_to_parse_context(input_format_list->format_name, typ, 
				  parse_context);
	input_format_list++;
    }

    /* handle nested input types???? */
    input_type =  ecl_build_type_node("input_type", input_field_list);
    ecl_add_decl_to_parse_context("input_type", input_type, parse_context);

    input_param = ecl_build_param_node("input", input_type, 0);

    ecl_add_decl_to_parse_context("input", input_param, parse_context);

    while ((output_format_list != NULL) && 
	   (output_format_list->format_name != NULL)) {
	/* step through output formats */
	typ = ecl_build_type_node(output_format_list->format_name,
			      output_format_list->field_list);
	ecl_add_decl_to_parse_context(output_format_list->format_name, typ, 
				  parse_context);
	output_format_list++;
    }

    if (output_field_list != NULL) {
	output_type = ecl_build_type_node("output_type", output_field_list);
	ecl_add_decl_to_parse_context("output_type", output_type, parse_context);
	output_param = ecl_build_param_node("output", output_type, 1);

	ecl_add_decl_to_parse_context("output", output_param, parse_context);
    } else {
	output_param = ecl_build_param_node("output", input_type, 1);
	ecl_add_decl_to_parse_context("output", output_param, parse_context);
    }

    if (data_field_list != NULL) {
	sm_ref filter_data_type, filter_data_param;
	while ((data_format_list != NULL) && 
	       (data_format_list->format_name != NULL)) {
	    /* step through data formats */
	    typ = ecl_build_type_node(data_format_list->format_name,
				      data_format_list->field_list);
	    ecl_add_decl_to_parse_context(data_format_list->format_name, typ, 
					  parse_context);
	    data_format_list++;
	}
	filter_data_type = ecl_build_type_node("filter_data_type",
					       data_field_list);
	ecl_add_decl_to_parse_context("filter_data_type", filter_data_type, 
				      parse_context);
	filter_data_param = ecl_build_param_node("filter_data", 
						 filter_data_type, 2);

	ecl_add_decl_to_parse_context("filter_data", filter_data_param, 
				      parse_context);

    }
    code = ecl_code_gen(filter, parse_context);
    ecl_free_parse_context(parse_context);
    return code;
}
#endif

static void
setup_conversions(context, ioformat, field_list, format_list)
IOContext context;
IOFormat ioformat;
IOFieldList field_list;
DEFormatList format_list;
{
    IOFormat *ioformats;
    IOFormat *format;
    int native_struct_size;

    if ((ioformats = get_subformats_IOformat(ioformat)) == NULL) {
	fprintf(stderr, "IOFormat list is null\n");
	return;
    }
    format = ioformats;
    while (format_list && (*format != NULL)) {
	int i = 0;
	if (!has_conversion_IOformat(*format)) {
	    while(format_list[i].format_name != NULL) {
		if (strcmp(format_list[i].format_name,
			   name_of_IOformat(*format)) == 0) {
		    break;
		}
		i++;
	    }
	    if (format_list[i].field_list != NULL) {
		native_struct_size = 
		    struct_size_IOfield((IOFile)context, 
					format_list[i].field_list);
		set_conversion_IOcontext(context, *format,
					 format_list[i].field_list,
					 native_struct_size);
	    }
	}
	format++;
    }
    free(ioformats);
    native_struct_size = struct_size_IOfield((IOFile)context, 
					     field_list);
    set_conversion_IOcontext(context, ioformat,
			     field_list, native_struct_size);
}

/* 
 * do channel_derive does most of the work involved in deriving an event
 * channel.  If output_field_list and output_format_list are NULL, this is
 * this is an untyped derivation and the output types are the same as the
 * input types.  (No "output" parameter to handler.)  If init_data is
 * non-NULL, there is data associated with the derived channel and this
 * gives the (pbio-encoded) initial value.
 */
static EChannel
do_channel_derive(de, parent_chan, filter, output_field_list,
		  output_format_list, init_data, init_data_len)
DExchange de;
EChannel parent_chan;
char *filter;
IOFieldList output_field_list;
DEFormatList output_format_list;
char *init_data;
int init_data_len;
{
    EChannel chan;
    char *der_chan_str;
    int i;
    IOFormat data_format = NULL;
    IOFieldList input_field_list;
    DEFormatList input_format_list;

    input_field_list = EChannel_get_field_list(parent_chan);
    input_format_list = EChannel_get_format_list(parent_chan);
    if (output_field_list == NULL) {
	/* use field lists for parent channel when doing untyped derive */
	chan = EChannel_typed_create(de, input_field_list, 
				     input_format_list);
    } else {
	chan = EChannel_typed_create(de, output_field_list, 
				     output_format_list);
    }
    der_chan_str = ECglobal_id(chan);

    if (init_data != NULL) {
	IOFieldList data_field_list = NULL;
	DEFormatList tmp_list, data_format_list = NULL;
	void *data;
	DExchange_context_lock(de);
	data_format = get_format_IOcontext(chan->subcontext, init_data);
	data_field_list = field_list_of_IOformat(data_format);
	data_format_list = get_format_list(data_format);
	data = DEmalloc(this_IOrecord_length(chan->subcontext, init_data, 
					     init_data_len));
	chan->filter_data = data;
	chan->data_field_list = simplify_field_list(data_field_list);
	tmp_list = data_format_list;
	while ((tmp_list != NULL) && (tmp_list->format_name != NULL)) {
	    tmp_list->field_list = simplify_field_list(tmp_list->field_list);
	    tmp_list++;
	}
	chan->data_format_list = data_format_list;
	if (!has_conversion_IOformat(data_format)) {
	    setup_conversions(chan->subcontext, data_format, 
			      chan->data_field_list, chan->data_format_list);
	}
	decode_to_buffer_IOcontext(chan->subcontext, init_data, data);
	DExchange_context_unlock(de);
    }

    do_chan_derive(parent_chan, chan, filter, 
		   (output_field_list != NULL) /* has_output_param */);

    for (i=0; i< parent_chan->local_source_list_len; i++) {
	int ret;
	ECSourceHandle src = parent_chan->local_source_list[i].source_handle;
	
	ret = do_internal_source_derive(src, chan, filter,
					input_field_list, input_format_list,
					output_field_list,output_format_list,
					chan->data_field_list,
					chan->data_format_list);
	if (ret != 1) {
	    EChannel_destroy(chan);
	    return NULL;
	}
    }
    if (input_format_list) free_format_list(input_format_list);
    free_field_list(input_field_list);
    ECtrace_out("send derive request to all members", 3);
    for (i = 0; i < parent_chan->rem_member_list_len; i++) {
	int cond;
	SrcDeriveMsg src_derive_msg;

	cond = DECondition_get(de, parent_chan->rem_member_list[i].dep);
	DECondition_set_client_data(de, cond, &chan);
	memset(&src_derive_msg, 0, sizeof(src_derive_msg));
	src_derive_msg.der_chan_id = der_chan_str;
	src_derive_msg.filter = filter;
	src_derive_msg.cond = cond;
	src_derive_msg.init_data_block = init_data;
	src_derive_msg.init_data_len = init_data_len;
	src_derive_msg.typed_output = ( output_field_list != NULL);
	memcpy(src_derive_msg.parent_channel,
	       &parent_chan->rem_member_list[i].channel[0], 8);
	DEport_write_data(parent_chan->rem_member_list[i].dep,
			  DE_Channel_Source_Derive_format_id, 
			  &src_derive_msg);
	/* wait here to make sure that each source */
	/* subscribes, need a barrier */
	if (DECondition_wait(de, cond))
	    chan->ready++;
    }
    DEfree(der_chan_str);
    return chan;
}

extern int
do_internal_source_derive(src, der_chan, filter, 
			  input_field_list, input_format_list,
			  output_field_list, output_format_list,
			  data_field_list, data_format_list)
ECSourceHandle src;
EChannel der_chan;
char *filter;
IOFieldList input_field_list;
DEFormatList input_format_list;
IOFieldList output_field_list;
DEFormatList output_format_list;
IOFieldList data_field_list;
DEFormatList data_format_list;
{
#ifndef NO_DERIVED
    FilterFunc filterfunc;
    ecl_code code_struct;
    if (src->der_source_list == NULL) {
        src->der_source_list = malloc(sizeof(struct _derived_source));
    } else {
        src->der_source_list = DErealloc(src->der_source_list,
					 (src->der_list_len + 1) * 
					 sizeof(struct _derived_source));
    }
    src->der_source_list[src->der_list_len].derived_chan = der_chan;
    
    code_struct = generate_filter_code(filter, src, input_field_list,
				       input_format_list, 
				       output_field_list, 
				       output_format_list,
				       data_field_list,
				       data_format_list);
    src->der_source_list[src->der_list_len].has_output_param = 
	(output_field_list != NULL);
    if (output_field_list == NULL) {
	/* output list is the same as input list */
	output_field_list = EChannel_get_field_list(der_chan);
	output_format_list = EChannel_get_format_list(der_chan);
    }
    src->der_source_list[src->der_list_len].source_handle = 
	ECsource_typed_subscribe(der_chan, output_field_list, 
				 output_format_list);
    if (code_struct == NULL) return 0;
    filterfunc = (FilterFunc) code_struct->func;
    src->der_source_list[src->der_list_len].filter = filterfunc;
    src->der_source_list[src->der_list_len].code_struct = code_struct;
    src->der_source_list[src->der_list_len].target_type_len =
	struct_size_field_list(output_field_list, sizeof(char*));
    
    src->der_list_len++;
#endif
    return 1;
}

static void
do_chan_derive(parent_chan, der_chan, filter, has_output_param)
EChannel parent_chan;
EChannel der_chan;
char *filter;
int has_output_param;
{
    /* add derived chan to list in parent channel */
    parent_chan->der_chan_list = DErealloc(parent_chan->der_chan_list,
					   (parent_chan->der_list_len + 1) * 
					   sizeof(struct _derived_chan));
    parent_chan->der_chan_list[parent_chan->der_list_len].derived_chan = der_chan;
    parent_chan->der_chan_list[parent_chan->der_list_len].filter = 
	strdup(filter);
    parent_chan->der_chan_list[parent_chan->der_list_len].has_output_param = 
	has_output_param;
    parent_chan->der_list_len++;

    /* add parent to list in derived channel */
    der_chan->parent_chan_list = DErealloc(der_chan->parent_chan_list,
					   (der_chan->parent_list_len + 1) * 
					   sizeof(EChannel));
    der_chan->parent_chan_list[der_chan->parent_list_len] = parent_chan;
    der_chan->parent_list_len++;

}

extern void
remove_derived_channel(parent_chan, der_chan)
EChannel parent_chan;
EChannel der_chan;
{
    int i;
    /* walk derived list */
    for (i=0; i< parent_chan->der_list_len; i++) {
	if (parent_chan->der_chan_list[i].derived_chan == der_chan) {
	    DEfree(parent_chan->der_chan_list[i].filter);

	    /* copy down higher derived channels */
	    for ( ; i < parent_chan->der_list_len -1; i++) {
		parent_chan->der_chan_list[i] =
		    parent_chan->der_chan_list[i+1];
	    }
	    parent_chan->der_list_len--;
	    break;
	}
    }
    /* walk sources */
    for (i=0; i< parent_chan->local_source_list_len; i++) {
	ECSourceHandle src_entry = parent_chan->local_source_list[i].source_handle;
	int j;
	for (j=0; j < src_entry->der_list_len; j++) {
	    derived_source src = &src_entry->der_source_list[j];
	    if (src->derived_chan == der_chan) {
		DEfree((void*)(long)src->filter);
		/* src_handle will be freed when der_channel is destroyed */

		/* copy down higher derived sources */
		for ( ; j < src_entry->der_list_len -1; j++) {
		    src_entry->der_source_list[j] = src_entry->der_source_list[j+1];
		}
		src_entry->der_list_len--;
		break;  /*break out of inner for loop */
	    }
	}
    }
}

/* create a derived event channel and reply to sender */
/* send derive request to all sources of the parent channel */
extern int
DEChannel_Derive_handler(de, dep, attend_format_id,
			 data, data_length, client_data)
DExchange de;
DEPort dep;
int attend_format_id;
void *data;
int data_length;
void *client_data;
{
    DeriveMsgPtr derive_msg = (DeriveMsgPtr) data;
    AttendResponseMsg ret_msg;
    EChannel chan, parent_chan;
    int i, derive_cond, modifier_index;
    DEFormatList output_format_list;
    IOFieldList field_list = NULL;
    char *filter;

    ECtrace_out("DEChannel_Derive_handler", 1);
    DEport_register_close_handler(dep, DEChannel_close_handler);

    parent_chan = string_to_channel_ptr(derive_msg->chan_str);

    /* check to see that parent channel is valid */

    /* check to see if filter already exists */

    derive_cond = derive_msg->cond;

    output_format_list = DEmalloc((derive_msg->format_list_len+1) * 
				  sizeof(DEFormat));
    for (i=0; i< derive_msg->format_list_len; i++) {
	output_format_list[i].format_name = 
	    derive_msg->format_list[i].format_name;
	output_format_list[i].field_list =
	    copy_field_list(derive_msg->format_list[i].field_list);
    }
    if (derive_msg->field_list != NULL) {
	field_list = copy_field_list(derive_msg->field_list);
    }
    filter = malloc(strlen(derive_msg->filter) + 2);
    strcpy(filter, derive_msg->filter);
    filter[strlen(filter) + 1] = 0; /* add extra NULL termination */

    output_format_list[derive_msg->format_list_len].format_name = NULL;
    output_format_list[derive_msg->format_list_len].field_list = NULL;
    ECtrace_out("Deriving channel with filter %s\n", 3, filter);
    chan = do_channel_derive(de, parent_chan, filter, 
			     field_list, output_format_list, 
			     derive_msg->init_data_block, 
			     derive_msg->init_data_len);
    DEfree(output_format_list);
    ECtrace_out("adding modifier to  remote member list", 3);
    if (chan->rem_member_list == NULL) {
	chan->rem_member_list = DEmalloc(sizeof(struct _rem_attendr));
    }
    chan->rem_member_list = DErealloc(chan->rem_member_list,
	  sizeof(struct _rem_attendr) * (chan->rem_member_list_len + 1));
    chan->rem_member_list[chan->rem_member_list_len].dep = dep;
    memcpy(chan->rem_member_list[chan->rem_member_list_len].channel,
	   &derive_msg->client_channel[0], 8);
    modifier_index = chan->rem_member_list_len;
    chan->rem_member_list_len++;

    ECtrace_out("send sink list, local and remote to modifier", 3);
    memset(&ret_msg, 0, sizeof(ret_msg));
    ret_msg.condition = derive_cond;
    ret_msg.sink_count = 1;
    ret_msg.sinks = DEmalloc(sizeof(AttendRec));
    ret_msg.sinks[0].host = chan->global_id.home_host;
    ret_msg.sinks[0].IPport = chan->global_id.IPport;
    ret_msg.sinks[0].filter = NULL;
    if (chan->type_format) {
	ret_msg.type_name = global_name_of_IOformat(chan->type_format);
	ECtrace_out("type of format is %s\n", 3, ret_msg.type_name);
    }
    memset(ret_msg.sinks[0].channel, 0, sizeof(ret_msg.sinks[0].channel));
    memcpy(ret_msg.sinks[0].channel, &chan, sizeof(chan));
    for (i = 0; i < chan->rem_sink_list_len; i++) {
	if (chan->rem_sink_list[i].dep != NULL) {
	    ret_msg.sinks = DErealloc(ret_msg.sinks,
			   (ret_msg.sink_count + 1) * sizeof(AttendRec));
	    ret_msg.sinks[ret_msg.sink_count].host =
		DEport_host_name(chan->rem_sink_list[i].dep);
	    ret_msg.sinks[ret_msg.sink_count].IPport =
		DEport_port_number(chan->rem_sink_list[i].dep);
	    memset(ret_msg.sinks[ret_msg.sink_count].channel, 0,
		   sizeof(ret_msg.sinks[0].channel));
	    ret_msg.sinks[ret_msg.sink_count].filter = NULL;
	    memcpy(ret_msg.sinks[ret_msg.sink_count].channel,
		   chan->rem_sink_list[i].channel,
		   sizeof(chan->rem_sink_list[0].channel));
	    ret_msg.sink_count++;
	}
    }
    ret_msg.member_count = 1;
    ret_msg.members = DEmalloc(sizeof(AttendRec));
    ret_msg.members[0].host = chan->global_id.home_host;
    ret_msg.members[0].IPport = chan->global_id.IPport;
    ret_msg.members[0].filter = NULL;
    memset(ret_msg.members[0].channel, 0,
	   sizeof(ret_msg.members[0].channel));
    memcpy(ret_msg.members[0].channel, &chan, sizeof(chan));
    for (i = 0; i < chan->rem_member_list_len; i++) {
	if (chan->rem_member_list[i].dep != NULL &&
	    i != modifier_index) {
	    ret_msg.members = DErealloc(ret_msg.members,
			 (ret_msg.member_count + 1) * sizeof(AttendRec));
	    ret_msg.members[ret_msg.member_count].host =
		DEport_host_name(chan->rem_member_list[i].dep);
	    ret_msg.members[ret_msg.member_count].IPport =
		DEport_port_number(chan->rem_member_list[i].dep);
	    ret_msg.members[ret_msg.member_count].filter = NULL;
	    memset(ret_msg.members[ret_msg.member_count].channel, 0,
		   sizeof(ret_msg.members[0].channel));
	    memcpy(ret_msg.members[ret_msg.member_count].channel,
		   chan->rem_member_list[i].channel, sizeof(chan));
	    ret_msg.member_count++;
	}
    }
    DEport_write_data(dep, DE_Channel_Attend_Response_format_id, &ret_msg);
    DEfree(ret_msg.type_name);
    DEfree(ret_msg.sinks);
    DEfree(ret_msg.members);

    if (channel_verbose) {
	printf("\n\nDEChannel_Derive_handler\n");
	dump_EChannel(parent_chan);
    }
    ECtrace_out("DEChannel_Derive_handler", 0);
    return 0;
}

ECDataHandle
EChannel_data_open(channel, data_field_list, data_subformat_list)
EChannel channel;
IOFieldList data_field_list;
DEFormatList data_subformat_list;
{
    ECDataHandle handle = new_data_handle();
    ECdata_struct data_spec;
    data_spec.data_field_list = data_field_list;
    data_spec.data_subformat_list = data_subformat_list;
    data_spec.initial_value = NULL;

    /* this doesn't really encode, just initializes the handle */
    handle->channel = channel;
    handle->data_field_list = data_field_list;
    handle->data_subformat_list = data_subformat_list;
    encode_data_block(&data_spec, NULL, NULL, handle, channel->de);
    return handle;
}

static void
send_remote_data_update(chan, i, encoded_data, data_len)
EChannel chan;
int i;
void *encoded_data;
int data_len;
{
    DataUpdateMsg msg;
    memset(&msg, 0, sizeof(msg));
    msg.data = encoded_data;
    msg.data_len = data_len;
    memcpy(msg.channel, chan->rem_member_list[i].channel, sizeof(msg.channel));
    memcpy(msg.global_chan_id, chan->global_id.channel,
	   sizeof(msg.global_chan_id));
    ECtrace_out("send_remote_data_update, chan %lx, event %lx", 1, chan, encoded_data);
    DEport_write_data(chan->rem_member_list[i].dep, 
		      DE_Data_Update_Message_format_id,
		      &msg);
    ECtrace_out("send_remote_data_update", 0);
}

extern int
DEData_Update_handler(de, dep, data_update_format_id, data, data_length,
		      client_data)
DExchange de;
DEPort dep;
int data_update_format_id;
void *data;
int data_length;
void *client_data;
{
    EChannel chan;
    DataUpdateMsg *msg = (DataUpdateMsg *) data;
    void *local_data;
    IOFormat data_format;

    memcpy(&chan, &msg->channel, sizeof(chan));
    ECtrace_out("In Data_Update_Handler, chan %lx", 1, (long) chan);
		
    EChannel_lock(chan);
    local_data = DEmalloc(this_IOrecord_length(chan->subcontext,
					       msg->data, msg->data_len));
    data_format = get_format_IOcontext(chan->subcontext, msg->data);
    if (!has_conversion_IOformat(data_format)) {
	setup_conversions(chan->subcontext, data_format, 
			  chan->data_field_list, chan->data_format_list);
    }
    DExchange_context_lock(de);
    decode_to_buffer_IOcontext(chan->subcontext, msg->data, local_data);
    DExchange_context_unlock(de);
    DEfree(chan->filter_data);
    chan->filter_data = local_data;
    EChannel_unlock(chan);
    ECtrace_out("Exitting Data_Update_Handler, chan %lx", 0, (long) chan);
    return 0;
}

void
EChannel_data_update(handle, data)
ECDataHandle handle;
void *data;
{
    void *encoded_block, *local_data;
    IOFormat data_format;
    EChannel chan = handle->channel;
    int block_len, i;
    ECdata_struct data_spec;
    data_spec.data_field_list = handle->data_field_list;
    data_spec.data_subformat_list = handle->data_subformat_list;
    data_spec.initial_value = data;
    encode_data_block(&data_spec, (char**)&encoded_block, &block_len, handle,
		      chan->de);
    EChannel_lock(chan);
    ECtrace_out("In echannel_data_update, handle %lx, data %lx, members %d",
		1, handle, data, chan->rem_member_list_len);
    for (i = 0; i < chan->rem_member_list_len; i++) {
	if (chan->rem_member_list[i].dep != NULL) {
	    EChannel_unlock(chan);
	    send_remote_data_update(chan, i, encoded_block,
				    block_len);
	    EChannel_lock(chan);
	}
    }
    DExchange_context_lock(chan->de);
    local_data = DEmalloc(this_IOrecord_length(handle->iocontext,
					       encoded_block, block_len));
    data_format = get_format_IOcontext(handle->iocontext, encoded_block);
    if (!has_conversion_IOformat(data_format)) {
	setup_conversions(handle->iocontext, data_format, 
			  handle->data_field_list,
			  handle->data_subformat_list);
    }
    decode_to_buffer_IOcontext(handle->iocontext, encoded_block,
			       local_data);
    DExchange_context_unlock(chan->de);
    DEfree(chan->filter_data);
    chan->filter_data = local_data;
    DEfree(encoded_block);
    EChannel_unlock(chan);
    ECtrace_out("echannel_data_update", 0);
}

static
IOFieldList
simplify_field_list(list)
IOFieldList list;
{
    int i = 0;
    while (list[i].field_name != NULL) {
	char *colon = strchr(list[i].field_type, ':');
	if (colon != NULL) {
	    /* make field type be simple, not format server style */
	    *colon = 0;	
	}
	i++;
    }
    return list;
}

extern IOFieldList
EChannel_get_field_list(chan)
EChannel chan;
{

    IOFieldList list = copy_field_list(field_list_of_IOformat(chan->type_format));
    return simplify_field_list(list);
}

static DEFormatList
get_format_list(ioformat)
IOFormat ioformat;
{
    IOFormat *formats = get_subformats_IOformat(ioformat);
    DEFormatList format_list = malloc(sizeof(format_list[0]));
    int format_count = 0;
    IOFormat *orig_formats = formats;
    while(formats[1] != NULL) {  /* last format is top-level */
	IOFieldList list = copy_field_list(field_list_of_IOformat(*formats));
	format_list = realloc(format_list, 
			      sizeof(format_list[0]) * (format_count + 2));
	format_list[format_count].format_name = 
	    strdup(name_of_IOformat(*formats));
	format_list[format_count].field_list = simplify_field_list(list);
	formats++;
	format_count++;
    }
    free(orig_formats);
    format_list[format_count].format_name = NULL;
    format_list[format_count].field_list = NULL;
    if (format_count == 0) {
	free(format_list);
	return NULL;
    }
    return format_list;
}
    
static void
free_format_list(list)
DEFormatList list;
{
    int i= 0;
    while (list[i].format_name != NULL) {
	DEfree(list[i].format_name);
	free_field_list(list[0].field_list);
	i++;
    }
    DEfree(list);
}

extern DEFormatList
EChannel_get_format_list(chan)
EChannel chan;
{
    return get_format_list(chan->type_format);
}

/* open and subscribe to a derived event channel */
extern int
DEChannel_Source_Derive_handler(de, dep, attend_format_id,
				data, data_length, client_data)
DExchange de;
DEPort dep;
int attend_format_id;
void *data;
int data_length;
void *client_data;
{
    SrcDeriveMsgPtr msg = (SrcDeriveMsgPtr) data;
    EChannel parent_chan, der_chan;
    SrcDeriveRespMsg ret_msg;
    IOFieldList output_field_list = NULL, input_field_list;
    DEFormatList output_format_list = NULL, input_format_list;
    int cond;
    int i;

    ECtrace_out("DEChannel_Source_Derive_handler", 1);

    DEport_register_close_handler(dep, DEChannel_close_handler);

    memcpy(&parent_chan, &msg->parent_channel[0], sizeof(parent_chan));

    cond = msg->cond;

    /* open the derivative */
    der_chan = EChannel_open(de, msg->der_chan_id);

    do_chan_derive(parent_chan, der_chan, msg->filter, msg->typed_output);

    if (msg->typed_output) {
	output_field_list = EChannel_get_field_list(der_chan);
	output_format_list = EChannel_get_format_list(der_chan);
    }
    input_field_list = EChannel_get_field_list(parent_chan);
    input_format_list = EChannel_get_format_list(parent_chan);

    for (i=0; i< parent_chan->local_source_list_len; i++) {
	int ret;
	ECSourceHandle src = parent_chan->local_source_list[i].source_handle;
	ret = do_internal_source_derive(src, der_chan, msg->filter, 
					input_field_list, input_format_list,
					output_field_list,output_format_list,
					der_chan->data_field_list, 
					der_chan->data_format_list);
	if (ret != 1) {
	    EChannel_destroy(der_chan);
	    return 0;
	}
    }

    i=0;
    while (output_format_list && 
	   (output_format_list[i].format_name != NULL)) { 
	free_field_list(output_format_list[i].field_list);
	free(output_format_list[i].format_name);
	i++;
    }
    if (output_format_list) free(output_format_list);

    ECtrace_out("create return message and send back to arbitrator", 3);
    memset(&ret_msg, 0, sizeof(ret_msg));
    ret_msg.cond = cond;

    DEport_write_data(dep, DE_Channel_Source_Derive_Resp_format_id, 
		      &ret_msg);

    if (channel_verbose) {
	printf("\n\nDEChannel_Source_Derive_handler\n");
	dump_EChannel(parent_chan);
    }
    ECtrace_out("DEChannel_Source_Derive_handler", 0);
    return 0;
}

/* return message from source to arbitrator signaling that */
/* the derived channel was created on that source */
extern int
DEChannel_Source_Derive_Resp_handler(de, dep, attend_response_format_id,
				     data, data_length, client_data)
DExchange de;
DEPort dep;
int attend_response_format_id;
void *data;
int data_length;
void *client_data;
{
    SrcDeriveRespMsgPtr msg = (SrcDeriveRespMsgPtr)data;

    ECtrace_out("DEChannel_Source_Derive_Resp_handler", 1);

    DEport_register_close_handler(dep, DEChannel_close_handler);

    DECondition_signal(de, msg->cond);

    ECtrace_out("DEChannel_Source_Derive_Resp_handler", 0);
    return 0;
}

static ECDataHandle
new_data_handle()
{
    ECDataHandle tmp = malloc(sizeof(*tmp));
    tmp->iocontext = NULL;
    tmp->type_format = NULL;
    tmp->channel = NULL;
    tmp->data_field_list = NULL;
    tmp->data_subformat_list = NULL;
    return tmp;
}

extern void
EChannel_data_close(handle)
ECDataHandle handle;
{
    free_IOsubcontext(handle->iocontext);
    free(handle);
}

static void
encode_data_block(filter_data, data_block, data_len, handle, de)
ECdata_spec filter_data;
char **data_block;
int *data_len;
ECDataHandle handle;
DExchange de;
{
    ECDataHandle tmp_handle = handle;
    void *tmp_data;
    void *return_data;

    if (filter_data == NULL) {
	*data_block = NULL;
	*data_len = 0;
	return;
    }
    if (handle == NULL) {
	tmp_handle = new_data_handle();
    } else {
	de = handle->channel->de;
    }
    DExchange_context_lock(de);
    if (tmp_handle->iocontext == NULL) {
	tmp_handle->iocontext = create_IOsubcontext(de->context);
    }
    if (tmp_handle->type_format == NULL) {
	EChannel_register_subformats(tmp_handle->iocontext,
				     filter_data->data_field_list,
				     filter_data->data_subformat_list);
	tmp_handle->type_format = 
	    register_IOcontext_format("Tmp channel data",
				      filter_data->data_field_list, 
				      tmp_handle->iocontext);
    }
    if (data_block == NULL) {
	DExchange_context_unlock(de);
	return;
    }
    tmp_data = encode_IOcontext_buffer(tmp_handle->iocontext,
				       tmp_handle->type_format,
				       filter_data->initial_value, 
				       data_len);
    /* must preserve data before unlocking */
    return_data = DEmalloc(*data_len);
    memcpy(return_data, tmp_data, *data_len);
    *data_block = return_data;
    DExchange_context_unlock(de);
    if (handle == NULL) {
	EChannel_data_close(tmp_handle);
    }
}

extern int
EC_break_up_triple(chan_id, host_ptr, port_ptr, chan_str_ptr, addr_ptr)
char *chan_id;
char **host_ptr;
int *port_ptr;
char **chan_str_ptr;
void **addr_ptr;
{
    char *chan_id_copy = DEmalloc(strlen(chan_id) + 1);
    char *port_str;
    char *chan_str;
    char *tmp;

    strcpy(chan_id_copy, chan_id);
    *host_ptr = chan_id_copy;
    tmp = strchr(chan_id_copy, '@');
    if (tmp == NULL) {
	free(chan_id_copy);
	return 1;
    }
    *tmp = 0;			/* kill first @ */
    port_str = ++tmp;
    tmp = strchr(tmp, '@');
    if (tmp == NULL) {
	free(chan_id_copy);
	return 1;
    }
    *tmp = 0;			/* kill second @ */
    chan_str = ++tmp;

    sscanf(port_str, "%d", port_ptr);

    *addr_ptr = (void*) string_to_channel_ptr(chan_str);
    *chan_str_ptr = chan_str;
    return 0;
}
