#include "config.h"
#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include <ctype.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#include "useful.h"
#include "io.h"

#include "DE.h"
#include "de_internal.h"

#ifndef tolower
extern int tolower ARGS((int ch));
#endif

extern void
DEtake_buffer(de, data)
DExchange de;
void *data;
{
    if (data != DATA_BUF(de)) {
	/* try handler_level +1 */
	de->handler_level++;
	if (data != DATA_BUF(de)) {
	    fprintf(stderr, "Error: DEtake_buffer called with record not associated with de\n");
	} else {
	    DATA_BUF(de) = DEmalloc(1);
	    BUF_SIZE(de) = 1;
	    BUF_TAKEN(de) = 1;
	}
	de->handler_level--;
	return;
    }
    DATA_BUF(de) = DEmalloc(1);
    BUF_SIZE(de) = 1;
    BUF_TAKEN(de) = 1;
}

extern void
DEreturn_buffer(de, data)
DExchange de;
void *data;
{
    /* we can imagine doing something fancier, but for the moment... */
    DEfree(data);
}

extern DESavedRecPtr
DEqueue_data(de, dep, data, format_id, record_length)
DExchange de;
DEPort dep;
void *data;
int format_id;
int record_length;
{
    DESavedRecPtr rec_ptr;
    if (data != DATA_BUF(de)) {
	fprintf(stderr, "Error: DEqueue_data called with record not associated with de\n");
	return NULL;
    }
    rec_ptr = (DESavedRecPtr) DEmalloc(sizeof(DESavedRec));
    rec_ptr->format_id = format_id;
    rec_ptr->data = data;
    rec_ptr->length = record_length;
    rec_ptr->dep = dep;
    rec_ptr->prev = NULL;	/* we're the first record */
    rec_ptr->next = de->queued_records;		/* others follow us */
    de->queued_records = rec_ptr;
    DATA_BUF(de) = DEmalloc(1);
    BUF_SIZE(de) = 1;
    BUF_TAKEN(de) = 1;
    return rec_ptr;
}

extern
void
DErelease_data(rec_ptr)
DESavedRecPtr rec_ptr;
{
    if (rec_ptr->prev == NULL) {
	/* we were first */
	rec_ptr->dep->de->queued_records = rec_ptr->next;
	if (rec_ptr->next) {
	    rec_ptr->next->prev = NULL;
	}
    } else {
	rec_ptr->prev->next = rec_ptr->next;
	if (rec_ptr->next) {
	    rec_ptr->next->prev = rec_ptr->prev;
	}
    }
    DDBG1(rec_ptr->dep->de, "Releasing record of type %s\n",
	  rec_ptr->dep->de->format_list[rec_ptr->format_id].format_name);
    DExchange_forward_data(rec_ptr->dep->de, rec_ptr->dep,
			   rec_ptr->format_id, rec_ptr->data);

    DEfree(rec_ptr->data);
    DEfree(rec_ptr);
}

/* not really part of the pbio exported functions */
extern void IOconvert_record ARGS((IOConversionPtr conv, void *src, void *dest,
			      void *final_string_base, void *src_string_base));


extern void
DErewrite_queued_data(de, format_id, old_field_list, old_struct_size,
		      old_pointer_size, new_field_list, new_struct_size,
		      new_pointer_size)
DExchange de;
int format_id;
IOFieldList old_field_list;
int old_struct_size;
int old_pointer_size;
IOFieldList new_field_list;
int new_struct_size;
int new_pointer_size;
{
    IOConversionPtr conv;
    DESavedRecPtr rec_ptr = de->queued_records;
    int delta = new_struct_size - old_struct_size;

    if (rec_ptr == NULL)
	return;

    conv = IOcreate_mem_conv(old_field_list, old_struct_size, old_pointer_size,
			     new_field_list, new_struct_size,
			     new_pointer_size);

    for (; rec_ptr != NULL; rec_ptr = rec_ptr->next) {
	if (rec_ptr->format_id == format_id) {
	    char *tmp = (char *) DEmalloc(rec_ptr->length + delta);
	    IOconvert_record(conv, rec_ptr->data, tmp, NULL, NULL);
	    DEfree(rec_ptr->data);
	    rec_ptr->data = tmp;
	}
    }
}

extern void *
DESavedRec_get_data_ptr(rec)
DESavedRecPtr rec;
{
    return rec->data;
}

extern int
DESavedRec_format_id(rec)
DESavedRecPtr rec;
{
    return rec->format_id;
}

extern void
DEget_field_info(de, format_name, field_name, size_p, field_type_p)
DExchange de;
char *format_name;
char *field_name;
int *size_p;
const char **field_type_p;
{
    int format_id = DEget_format_id(de, format_name);
    int i = 0;
    int not_found = 1;

    i = 0;
    while (not_found) {
	const char *list_field_name =
	    de->format_list[format_id].internal_field_list[i].field_name;
	if (strcmp(list_field_name, field_name) == 0) {
	    if (size_p)
		*size_p =
		    de->format_list[format_id].internal_field_list[i].field_size;
	    if (field_type_p)
		*field_type_p =
		    de->format_list[format_id].internal_field_list[i].field_type;
	    not_found = 0;
	}
	i++;
    }
    if (not_found) {
	if (size_p)
	    *size_p = 0;
	if (field_type_p)
	    *field_type_p = NULL;
    }
}

static
 IOdata_type
str_to_data_type(str)
char *str;
{
    char *tmp = DEmalloc(strlen(str) + 1);
    char *free_later = tmp;
    strcpy(tmp, str);
    str = tmp;			/* make a copy of str parameter */

    while (isspace((int)*str)) {	/* skip preceeding space */
	str++;
    }
    tmp = str + strlen(str) - 1;
    while (isspace((int)*tmp)) {	/* kill trailing space */
	*tmp = 0;
	tmp--;
    }
    tmp = str;
    while (*tmp) {		/* map to lower case */
	*tmp = tolower(*tmp);
	tmp++;
    }
    if (strcmp(str, "integer") == 0) {
	DEfree(free_later);
	return integer_type;
    } else if (strcmp(str, "unsigned integer") == 0) {
	DEfree(free_later);
	return unsigned_type;
    } else if (strcmp(str, "float") == 0) {
	DEfree(free_later);
	return float_type;
    } else if (strcmp(str, "double") == 0) {
	DEfree(free_later);
	return float_type;
    } else if (strcmp(str, "char") == 0) {
	DEfree(free_later);
	return char_type;
    } else if (strcmp(str, "string") == 0) {
	DEfree(free_later);
	return string_type;
    } else if (strcmp(str, "enumeration") == 0) {
	DEfree(free_later);
	return enumeration_type;
    } else {
	DEfree(free_later);
	return unknown_type;
    }
}

extern DERecFieldPtr
DEmake_field_ptr(de, format_name, field_name, get_size, get_type)
DExchange de;
char *format_name;
char *field_name;
int get_size;
char *get_type;
{
    IOFieldPtr io_field_ptr;
    IOdata_type data_type = str_to_data_type(get_type);
    DERecFieldPtr tmp_field_ptr;
    int format_id = DEget_format_id(de, format_name);

    io_field_ptr =
	get_IOfieldPtrFromList(de->format_list[format_id].internal_field_list,
			       field_name);
    if (io_field_ptr == NULL)
	return NULL;
    if (data_type == unknown_type)
	return NULL;

    tmp_field_ptr = (DERecFieldPtr) DEmalloc(sizeof(DERecField));
    tmp_field_ptr->io_field_ptr = io_field_ptr;
    tmp_field_ptr->de = de;
    tmp_field_ptr->get_size = get_size;
    tmp_field_ptr->get_type = data_type;
    tmp_field_ptr->prev = NULL;	/* we're the first record */
    tmp_field_ptr->next = de->active_fields;	/* others follow us */
    de->active_fields = tmp_field_ptr;
    return tmp_field_ptr;
}

extern void
DEfree_field_info(field_p)
DERecFieldPtr field_p;
{
    DExchange de = field_p->de;
    if (field_p->prev == NULL) {
	de->active_fields = field_p->next;
	if (de->active_fields != NULL) {
	    de->active_fields->prev = NULL;
	}
    } else {
	field_p->prev->next = field_p->next;
	if (field_p->next != NULL) {
	    field_p->next->prev = field_p->prev;
	}
    }
    DEfree(field_p);
}

extern
 DEDataResult
DEget_data(de, data, record, field)
DExchange de;
void *data;
void *record;
DERecFieldPtr field;
{
    switch (field->get_type) {
    case unknown_type:
    case integer_type:
	if (field->get_size == sizeof(short)) {
	    *((short *) data) = get_IOshort(field->io_field_ptr, record);
	} else if (field->get_size == sizeof(int)) {
	    *((int *) data) = get_IOint(field->io_field_ptr, record);
	} else if (field->get_size == sizeof(long)) {
	    *((long *) data) = get_IOlong(field->io_field_ptr, record);
	} else if (field->get_size == 8) {
	    /* long must be 4...  try get_IOlong8 */
	    void *far_end = (char *) data + 4;
#ifdef WORDS_BIGENDIAN
	    get_IOlong8(field->io_field_ptr, record, (unsigned long *) far_end,
			(long *) data);
#else
	    get_IOlong8(field->io_field_ptr, record, (unsigned long *) data,
			(long *) far_end);
#endif
	} else {
	    assert(FALSE);
	}
	break;
    case unsigned_type:
	if (field->get_size == sizeof(unsigned short)) {
	    *((unsigned short *) data) = get_IOushort(field->io_field_ptr, record);
	} else if (field->get_size == sizeof(unsigned int)) {
	    *((unsigned int *) data) = get_IOuint(field->io_field_ptr, record);
	} else if (field->get_size == sizeof(unsigned long)) {
	    *((unsigned long *) data) = get_IOulong(field->io_field_ptr, record);
	} else if (field->get_size == 8) {
	    /* long must be 4...  try get_IOlong8 */
	    void *far_end = (char *) data + 4;
#ifdef WORDS_BIGENDIAN
	    get_IOulong8(field->io_field_ptr, record,
			 (unsigned long *) far_end,
			 (unsigned long *) data);
#else
	    get_IOulong8(field->io_field_ptr, record, (unsigned long *) data,
			 (unsigned long *) far_end);
#endif
	} else {
	    assert(FALSE);
	}
	break;
    case float_type:
	break;
    case char_type:
	break;
    case string_type:
	break;
    case enumeration_type:
	break;
    default:
	assert(FALSE);
    }
    return DEData_OK;
}
