
/***** Includes *****/
#include "config.h"
#include <sys/types.h>

#ifdef HAVE_WINDOWS_H
#define FD_SETSIZE 1024
#include <windows.h>
#include <winsock.h>
#include <io.h>
#define getpid()	_getpid()
static char* WSAerror_str ARGS((int err));
#else
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_TIMES_H
#include <sys/times.h>
#endif
#include <sys/socket.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#endif
#include <stdio.h>
#include <fcntl.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <errno.h>
#include <signal.h>
#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif

#include "useful.h"

#include "DE.h"
#ifdef HAVE_GEN_THREAD_H
#include "gen_thread.h"
#else
#define thr_mutex_alloc() NULL
#endif
#include "de_internal.h"
#include "de_lock.h"

#if !defined(SELECT_DEFINED) & !defined(HAVE_WINDOWS_H)
extern int select ARGS((int width, fd_set * readfds, fd_set * writefds,
			fd_set * exceptfds, struct timeval * timeout));
#endif
#ifndef GETTIMEOFDAY_DEFINED
int gettimeofday ARGS((struct timeval *, void *));
#endif

static void wake_server_thread ARGS((DEControlList cl));
static void shutdown_wake_mechanism ARGS((DEControlList cl));

typedef struct _DEtask_handle {
    DEControlList cl;
    long period;
    thr_thread_t executing;
    struct timeval next_time;
    GenericHandlerFunc func;
    void *arg1;
    void *arg2;
    DEtask_handle next;
} DEtask_handle_s;

#undef timercmp
#define	timercmp(tvp, uvp, cmp) \
	/* CSTYLED */ \
	(((tvp)->tv_sec cmp (uvp)->tv_sec) || \
	((((tvp)->tv_sec == (uvp)->tv_sec) && \
	/* CSTYLED */ \
	((tvp)->tv_usec cmp (uvp)->tv_usec))))

static void
set_soonest_timeout(timeout, task_list, now)
struct timeval *timeout;
DEtask_handle task_list;
struct timeval now;
{
    struct timeval this_delay;
    if (task_list == NULL) return;
    this_delay.tv_sec = task_list->next_time.tv_sec - now.tv_sec;
    this_delay.tv_usec = task_list->next_time.tv_usec - now.tv_usec;
    if (this_delay.tv_usec < 0) {
	this_delay.tv_sec--;
	this_delay.tv_usec += 1000000;
    }
    if (this_delay.tv_sec < 0) {
	this_delay.tv_sec = this_delay.tv_usec = 0;
    }
    if ((timeout->tv_sec == -1) || (timercmp(&this_delay, timeout, <))) {
	*timeout = this_delay;
    }
    set_soonest_timeout(timeout, task_list->next, now);
}

static void
increment_time(time, increment)
struct timeval *time;
long increment;
{
    time->tv_usec += increment;
    if (time->tv_usec >= 1000000) {
	time->tv_sec += (time->tv_usec / 1000000);
	time->tv_usec = (time->tv_usec % 1000000);
    }
}

extern int
DEControlList_poll_and_handle_timeout(cl, timeout_sec, timeout_usec)
DEControlList cl;
int timeout_sec;
int timeout_usec;
{
    int i, res;
    fd_set rd_set;
    struct timeval timeout;
    int tmp_select_consistency_number = cl->select_consistency_number;

    if (cl->closed) return 0;

    if ((cl->control_style != DESingleThreaded) &&
	(cl->server_thread == NULL)) {
	/* no server thread set, must be this one */
	cl->server_thread = thr_thread_self();
    }
    if ((cl->control_style != DESingleThreaded) &&
	(cl->server_thread != thr_thread_self())) {
	/* What?  We're polling, but we're not the server thread? */
	fprintf(stderr, "Warning:  Multiple threads calling DExchange_poll_and_handle on the \n          same DExchange.\n");
	fprintf(stderr, "          This situation may result in unexpected I/O blocking.\n");
	fprintf(stderr, "          Server thread set to %lx.\n", (long) thr_thread_self());
	cl->server_thread = thr_thread_self();
    }
    rd_set = *(fd_set *) cl->fdset;
    if ((timeout_sec >= 0) || (cl->periodic_task_list != NULL)) {
	struct timeval now;
#ifndef HAVE_WINDOWS_H
	/* GSE...  No gettimeofday on windows.  
	 * Must use _ftime, get millisec time, convert to usec.  Bleh.
	 * Will fix this another day, when we have derivation on NT.
	 */
	gettimeofday(&now, NULL);
#endif
	if (timeout_usec >= 1000000) {
	    timeout_sec += (timeout_usec / 1000000);
	    timeout_usec = timeout_usec % 1000000;
	}
	timeout.tv_sec = timeout_sec;
	timeout.tv_usec = timeout_usec;
	
	set_soonest_timeout(&timeout, cl->periodic_task_list, now);
	res = select(FD_SETSIZE, &rd_set, (fd_set *) NULL, (fd_set *) NULL,
		     &timeout);
    } else {
	res = select(FD_SETSIZE, &rd_set, (fd_set *) NULL, (fd_set *) NULL,
		     NULL);
    }
    if (cl->closed) return 0;
#ifndef HAVE_WINDOWS_H
    if (res == -1) {
	if (errno == EINTR) {
	    return 0;
	}
	/* 
	 * if upon returning from select the consistency number
	 * has changed, the only safe thing to do is return so 
	 * that a new select can happen with consistent info.
	 */
	if (cl->select_consistency_number != 
	    tmp_select_consistency_number) return 1;
	if (errno == 0) {
	    /* 
	     * odd, but x86 Solaris seems to return -1 from select but not set
	     * the errno in some circumstances.  Just return when this happens.
	     */
	    return 0;
	}
	if (errno == EBADF) {
	    int i;
	    int found_one = 0;
	    for (i = 0; i < FD_SETSIZE; i++) {
		if (FD_ISSET(i, &rd_set)) {
		    fd_set test_set;
		    timeout.tv_usec = 0;
		    timeout.tv_sec = 0;
		    FD_ZERO(&test_set);
		    FD_SET(i, &test_set);
		    errno = 0;
		    select(FD_SETSIZE, &test_set, (fd_set *) NULL,
			   (fd_set *) NULL, &timeout);
		    if (errno == EBADF) {
			fprintf(stderr, "Select failed, fd %d is bad.  Removing from select list.\n",
				i);
			FD_CLR(i, (fd_set *) cl->fdset);
			found_one++;
			FD_CLR(i, &rd_set);
		    }
		}
	    }
	    if (cl->closed) return 0;
/* if (found_one == 0) { fprintf(stderr, "Bad file descriptor in select
 * Warning.  Failed to localize.\n"); } */
	} else {

	    exit(1);
	    return -1;
	}
    }
#else
    /* 
     * Have to ignore the bad invalue for select on NT because it can't
     * do selects on files.  Otherwise you will get a bunch of
     * irritating select errors that aren't really errors 
     */
    if (res == SOCKET_ERROR) {
	int errno_val;
	errno_val = WSAGetLastError();
	if (errno_val == WSAEINTR || errno_val == WSAEINVAL) {
	    return 0;
	} else {
	    fprintf(stderr, "select failed, errno %d\n", 
		    WSAerror_str(errno_val));
	}
	return -1;
    }
#endif

    /* 
     * if upon returning from select the consistency number
     * has changed, the only safe thing to do is return so 
     * that a new select can happen with consistent info.
     */
    if (cl->select_consistency_number != 
	tmp_select_consistency_number) return 1;
    /* 
     * Careful!  We're reading the control list here without locking!
     * Something bad *might* happen, it's just unlikely.
     */
    if (res != 0) {
	for (i = 0; i <= cl->sel_item_max; i++) {
	    if (cl->closed) return 0;
	    if (FD_ISSET(i, &rd_set)) {
		if (cl->select_items[i].func != NULL) {
		    cl->select_items[i].func(cl->select_items[i].dep,
					     cl->select_items[i].arg);
		} else {
		    assert(!FD_ISSET(i, (fd_set *)cl->fdset));
		}
		if (cl->select_consistency_number != 
		    tmp_select_consistency_number) return 1;
	    }
	}
    }
    if (cl->periodic_task_list != NULL) {
	/* handle periodic tasks */
	DEtask_handle this_periodic_task = cl->periodic_task_list;
	struct timeval now;

#ifndef HAVE_WINDOWS_H
	/* GSE...  No gettimeofday on windows.  
	 * Must use _ftime, get millisec time, convert to usec.  Bleh.
	 * Will fix this another day, when we have derivation on NT.
	 */
	gettimeofday(&now, NULL);
#endif
	while (this_periodic_task != NULL ) {
	    
	    if (timercmp(&now, &this_periodic_task->next_time, >)) {
		increment_time(&this_periodic_task->next_time,
			       this_periodic_task->period);
		this_periodic_task->executing = thr_thread_self();
		this_periodic_task->func(this_periodic_task->arg1,
					 this_periodic_task->arg2);
		this_periodic_task->executing = (thr_thread_t) -1;
	    }
	    /* 
	     * if upon returning from a handler the consistency number
	     * has changed, the only safe thing to do is return so 
	     * that a new select can happen with consistent info.
	     */
	    if (cl->select_consistency_number != 
		tmp_select_consistency_number) return 1;
	    this_periodic_task = this_periodic_task->next;
	}
    }
    for (i = 0; i < cl->poll_item_count; i++) {
	cl->poll_items[i].func(cl->poll_items[i].dep,
			       cl->poll_items[i].arg);
	/* 
	 * if upon returning from a handler the consistency number
	 * has changed, the only safe thing to do is return so 
	 * that a new select can happen with consistent info.
	 */
	if (cl->select_consistency_number != 
	    tmp_select_consistency_number) return 1;
    }
    DEControlList_event_dispatch(cl);
    cl->select_consistency_number++;
    return res + cl->poll_item_count;
}

extern void
DEControlList_add_select(cl, fd, func, dep, arg)
DEControlList cl;
int fd;
GenericHandlerFunc func;
DEPort dep;
void *arg;
{
    DEControlList_lock(cl);
    cl->select_consistency_number++;
    if (fd > cl->sel_item_max) {
	int i;
	cl->select_items = (FunctionListElement *) DErealloc(cl->select_items, (fd + 1) * sizeof(DEControlList_s));
	for (i = cl->sel_item_max + 1; i < fd; i++) {
	    cl->select_items[i].func = NULL;
	    cl->select_items[i].dep = NULL;
	    cl->select_items[i].arg = NULL;
	}
	cl->sel_item_max = fd;
    }
    FD_SET(fd, (fd_set *) cl->fdset);
    if (fd > FD_SETSIZE) {
	fprintf(stderr, "Internal Error, stupid WINSOCK large FD bug.\n");
	fprintf(stderr, "Increase FD_SETSIZE.  Item not added to fdset.\n");
    }
    cl->select_items[fd].func = func;
    cl->select_items[fd].dep = dep;
    cl->select_items[fd].arg = arg;
    if (cl->control_style != DESingleThreaded) {
	wake_server_thread(cl);
    }
    DEControlList_unlock(cl);
}

extern void
DEControlList_remove_select(cl, fd)
DEControlList cl;
int fd;
{
    DEControlList_lock(cl);
    cl->select_consistency_number++;
    if (fd > cl->sel_item_max) {
	printf("Tried to remove fd larger than select list size\n");
    } else {
	cl->select_items[fd].func = NULL;
	cl->select_items[fd].dep = NULL;
	cl->select_items[fd].arg = NULL;
    }
    FD_CLR(fd, (fd_set *) cl->fdset);
    DEControlList_unlock(cl);
}

extern DEControlList
DEControlList_create()
{
    DEControlList new_list = (DEControlList) DEmalloc(sizeof(DEControlList_s));
    new_list->fdset = DEmalloc(sizeof(fd_set));
    FD_ZERO((fd_set *) new_list->fdset);
    new_list->reference_count = 1;
    new_list->free_reference_count = 1;
    new_list->sel_item_max = 0;
    new_list->select_items = (FunctionListElement *) DEmalloc(sizeof(FunctionListElement));
    new_list->select_items[0].func = NULL;
    new_list->select_items[0].dep = NULL;
    new_list->select_items[0].arg = NULL;

    new_list->poll_item_count = 0;
    new_list->poll_items = (FunctionListElement *) DEmalloc(sizeof(FunctionListElement));
    new_list->poll_items[0].func = NULL;
    new_list->poll_items[0].dep = NULL;
    new_list->poll_items[0].arg = NULL;
    new_list->select_consistency_number = 0;
    new_list->control_list_lock = thr_mutex_alloc();
    new_list->condition_list = NULL;
    new_list->next_condition_num = 1;
    new_list->wake_read_fd = -1;
    new_list->wake_write_fd = -1;
    new_list->event_queue_head = NULL;
    new_list->event_queue_tail = NULL;
    new_list->periodic_task_list = NULL;
    new_list->server_thread = NULL;
    new_list->closed = 0;
    if (gen_thr_initialized() && !gen_null_initialized()) {
	DEControlList_set_control_style(new_list, DEDedicatedServerThread);
    } else {
	DEControlList_set_control_style(new_list, DESingleThreaded);
    }
    return new_list;
}

extern void
DEControlList_close(cl)
DEControlList cl;
{
    cl->reference_count--;
    if (cl->reference_count == 0) {
	DEControlList_event_flush(cl);
	shutdown_wake_mechanism(cl);
	DEfree(cl->fdset);
	thr_mutex_free(cl->control_list_lock);
	DEfree(cl->select_items);
	DEfree(cl->poll_items);
	cl->closed = 1;
    }
}

extern void
DEControlList_free(cl)
DEControlList cl;
{
    cl->free_reference_count--;
    if(cl->free_reference_count == 0) {
	DEfree(cl);
    }
}

extern void
DEControlList_add_poll(cl, func, dep, arg)
DEControlList cl;
GenericHandlerFunc func;
DEPort dep;
void *arg;
{
    int item_num;
    DEControlList_lock(cl);
    item_num = cl->poll_item_count;
    cl->select_consistency_number++;
    cl->poll_items = (FunctionListElement *)
	DErealloc(cl->poll_items, (item_num + 2) * sizeof(DEControlList_s));
    cl->poll_items[item_num].func = func;
    cl->poll_items[item_num].dep = dep;
    cl->poll_items[item_num].arg = arg;
    cl->poll_item_count++;
    DEControlList_unlock(cl);
}

extern DEtask_handle
DEControlList_add_periodic(cl, period, func, arg1, arg2)
DEControlList cl;
long period;
GenericHandlerFunc func;
void *arg1;
void *arg2;
{
    DEtask_handle handle = malloc(sizeof(struct _DEtask_handle));

    handle->cl = cl;
    handle->period = period;
    handle->executing = (thr_thread_t) -1;
#ifndef HAVE_WINDOWS_H
    /* GSE...  No gettimeofday on windows.  
     * Must use _ftime, get millisec time, convert to usec.  Bleh.
     * Will fix this another day, when we have derivation on NT.
     */
    gettimeofday(&handle->next_time, NULL);
#endif
    increment_time(&handle->next_time, period);
    handle->func = func;
    handle->arg1 = arg1;
    handle->arg2 = arg2;
    handle->next = NULL;

    DEControlList_lock(cl);
    if (cl->periodic_task_list == NULL) {
	cl->periodic_task_list = handle;
    } else {
	handle->next = cl->periodic_task_list;
	cl->periodic_task_list = handle;
    }
    DEControlList_unlock(cl);
    if (cl->control_style != DESingleThreaded) {
	wake_server_thread(cl);
    }
    return handle;
}

extern void
DEtask_remove_periodic(handle)
DEtask_handle handle;
{
    DEControlList cl = handle->cl;
    DEtask_handle list, last = NULL;
    DEControlList_lock(cl);
    list = cl->periodic_task_list;
    
    while(list != handle) {
	last = list;
	list = list->next;
	if (list == NULL) {
	    fprintf(stderr, "Periodic task not found for removal\n");
	    DEControlList_unlock(cl);
	    return;
	}
    }
    /* unlink task */
    if (last == NULL) {
	cl->periodic_task_list = list->next;
    } else {
	last->next = list->next;
    }
    if (handle->executing != thr_thread_self()) {
	/* someone besides us executing this ? */
        int i = 0;
	while (handle->executing != (thr_thread_t)-1) {
	    /* wait until they're done */
	    DEControlList_unlock(cl);
	    thr_thread_yield();
	    DEControlList_lock(cl);
	    i++;
	    if (i > 1000) {
	        /* give up */
	        continue;
	    }
	}
    }
    DEfree(handle);
    cl->select_consistency_number++;
    DEControlList_unlock(cl);
}

extern void
DEPort_remove_from_control(dep)
DEPort dep;
{
    int i = 0;
    DEControlList cl = dep->de->control_list;
    DEControlList_lock(cl);
    cl->select_consistency_number++;
    while ((i <= cl->sel_item_max) && (cl->select_items[i].dep != dep)) {
	i++;
    }
    if (i <= cl->sel_item_max) {
	FD_CLR((unsigned) i, (fd_set *) cl->fdset);
	cl->select_items[i].func = NULL;
	cl->select_items[i].dep = NULL;
	cl->select_items[i].arg = NULL;
	DEControlList_unlock(cl);
	return;
    }
    i = 0;
    while ((cl->poll_items[i].dep != dep) && (i < cl->poll_item_count)) {
	i++;
    }
    if (i < cl->poll_item_count) {
	while (i < cl->poll_item_count - 1) {
	    cl->poll_items[i] = cl->poll_items[i + 1];
	    i++;
	}
	cl->poll_items[i].func = NULL;
	cl->poll_items[i].dep = NULL;
	cl->poll_items[i].arg = NULL;
	cl->poll_item_count--;
	DEControlList_unlock(cl);
	return;
    } else {
	DBG(dep->de, "Warning! dep not in control list!\n");
    }
    DEControlList_unlock(cl);
}

extern void
DEControlList_associate(cl, de)
DEControlList cl;
DExchange de;
{
    if (de->control_list->sel_item_max != 0) {
	if (de->control_list->sel_item_max > de->control_list->wake_read_fd) {
	    fprintf(stderr, "Only newly initialized DExchange values can be used in DEControlList_associate()\n");
	    return;
	}
    }
    DEControlList_close(de->control_list);
    DEControlList_free(de->control_list);
    cl->reference_count++;
    cl->free_reference_count++;
    de->control_list = cl;
}

extern int
DEControlList_poll_and_handle(cl, block)
DEControlList cl;
int block;
{
    if (block) {
	return DEControlList_poll_and_handle_timeout(cl, -1, 0);
    } else {
	return DEControlList_poll_and_handle_timeout(cl, 0, 0);
    }
}

/* 
 * Poll all connections and handle new connection forward formats,
 * comments, and data.
 */
extern int
DExchange_poll_and_handle(de, block)
DExchange de;
int block;
{
    return DEControlList_poll_and_handle(de->control_list, block);
}

extern int
DExchange_poll_and_handle_timeout(de, timeout_secs, timeout_usecs)
DExchange de;
int timeout_secs;
int timeout_usecs;
{
    return DEControlList_poll_and_handle_timeout(de->control_list, timeout_secs,
						 timeout_usecs);
}

extern void
DExchange_add_select(de, fd, func, arg1, arg2)
DExchange de;
int fd;
GenericHandlerFunc func;
void *arg1;
void *arg2;
{
    DEControlList_add_select(de->control_list, fd, func, (DEPort) arg1, arg2);
}

extern void
DExchange_remove_select(de, fd)
DExchange de;
int fd;
{
   DEControlList_remove_select(de->control_list, fd);
}


extern void
DExchange_add_poll(de, func, arg1, arg2)
DExchange de;
GenericHandlerFunc func;
void *arg1;
void *arg2;
{
    DEControlList_add_poll(de->control_list, func, (DEPort) arg1, arg2);
}


void
wake_DE_server(de)
DExchange de;
{
    wake_server_thread(de->control_list);
}

static void
shutdown_wake_mechanism(cl)
DEControlList cl;
{
    if (cl->wake_read_fd == -1) return;
    wake_server_thread(cl);
    close(cl->wake_read_fd);
    close(cl->wake_write_fd);
    cl->wake_read_fd = cl->wake_write_fd = -1;
}

static void read_wake_fd(fd_as_ptr, junk)
void *fd_as_ptr;
void *junk;
{
    char buffer;
    int fd = (int) (long)fd_as_ptr;
#ifdef HAVE_WINDOWS_H
    recv(fd, &buffer, 1, 0);
#else
    read(fd, &buffer, 1);
#endif
}

#ifdef HAVE_WINDOWS_H
static char*
WSAerror_str(err)
int err;
{
    switch(err) {
    case WSAEINTR: return "WSAEINTR";
    case WSAEBADF: return "WSAEBADF";
    case WSAEACCES: return "WSAEACCES";
    case WSAEFAULT: return "WSAEFAULT";
    case WSAEINVAL: return "WSAEINVAL";
    case WSAEMFILE: return "WSAEMFILE";
    case WSAEWOULDBLOCK: return "WSAEWOULDBLOCK";
    case WSAEINPROGRESS: return "WSAEINPROGRESS";
    case WSAEALREADY: return "WSAEALREADY";
    case WSAENOTSOCK: return "WSAENOTSOCK";
    case WSAEDESTADDRREQ: return "WSAEDESTADDRREQ";
    case WSAEMSGSIZE: return "WSAEMSGSIZE";
    case WSAEPROTOTYPE: return "WSAEPROTOTYPE";
    case WSAENOPROTOOPT: return "WSAENOPROTOOPT";
    case WSAEPROTONOSUPPORT: return "WSAEPROTONOSUPPORT";
    case WSAESOCKTNOSUPPORT: return "WSAESOCKTNOSUPPORT";
    case WSAEOPNOTSUPP: return "WSAEOPNOTSUPP";
    case WSAEPFNOSUPPORT: return "WSAEPFNOSUPPORT";
    case WSAEAFNOSUPPORT: return "WSAEAFNOSUPPORT";
    case WSAEADDRINUSE: return "WSAEADDRINUSE";
    case WSAEADDRNOTAVAIL: return "WSAEADDRNOTAVAIL";
    case WSAENETDOWN: return "WSAENETDOWN";
    case WSAENETUNREACH: return "WSAENETUNREACH";
    case WSAENETRESET: return "WSAENETRESET";
    case WSAECONNABORTED: return "WSAECONNABORTED";
    case WSAECONNRESET: return "WSAECONNRESET";
    case WSAENOBUFS: return "WSAENOBUFS";
    case WSAEISCONN: return "WSAEISCONN";
    case WSAENOTCONN: return "WSAENOTCONN";
    case WSAESHUTDOWN: return "WSAESHUTDOWN";
    case WSAETOOMANYREFS: return "WSAETOOMANYREFS";
    case WSAETIMEDOUT: return "WSAETIMEDOUT";
    case WSAECONNREFUSED: return "WSAECONNREFUSED";
    case WSAELOOP: return "WSAELOOP";
    case WSAENAMETOOLONG: return "WSAENAMETOOLONG";
    case WSAEHOSTDOWN: return "WSAEHOSTDOWN";
    case WSAEHOSTUNREACH: return "WSAEHOSTUNREACH";
    case WSAENOTEMPTY: return "WSAENOTEMPTY";
    case WSAEPROCLIM: return "WSAEPROCLIM";
    case WSAEUSERS: return "WSAEUSERS";
    case WSAEDQUOT: return "WSAEDQUOT";
    case WSAESTALE: return "WSAESTALE";
    case WSAEREMOTE: return "WSAEREMOTE";
    case WSAEDISCON: return "WSAEDISCON";
    case WSASYSNOTREADY: return "WSASYSNOTREADY";
    case WSAVERNOTSUPPORTED: return "WSAVERNOTSUPPORTED";
    case WSANOTINITIALISED: return "WSANOTINITIALISED";
    default: return "Unknown Winsock error";
    }
}
/*
 *  Note.  Unfortunately, the _pipe() function on WinNT 
 *  produces FDs that you can't use in select().  This ruins what we want
 *  this pipe for, which is to wake up a thread sleeping in select().
 *  So, we need to introduce a pipe function that returns two socket FDs.
 *  NT Sux.
 */

int
pipe(filedes)
int filedes[2];
{
    
    int length;
    struct sockaddr_in sock_addr;
    int sock_opt_val = 1;
    int sock1, sock2, conn_sock;
    unsigned long block = TRUE;
    int delay_value = 1;
   
    conn_sock = socket(AF_INET, SOCK_STREAM, 0);
    if (conn_sock == SOCKET_ERROR) {
	fprintf(stderr, "Cannot open INET socket\n");
	return -1;
    }
    sock_addr.sin_family = PF_INET;
    sock_addr.sin_addr.s_addr = INADDR_ANY;
    sock_addr.sin_port = 0;
    if (bind(conn_sock, (struct sockaddr *) &sock_addr,
	     sizeof sock_addr) == SOCKET_ERROR) {
	fprintf(stderr, "Cannot bind INET socket\n");
	return -1;
    }
    length = sizeof sock_addr;
    if (getsockname(conn_sock, (struct sockaddr *) &sock_addr, &length) < 0) {
	fprintf(stderr, "Cannot get socket name\n");
	return -1;
    }
    /* begin listening for conns */
    if (listen(conn_sock, FD_SETSIZE)) {
	fprintf(stderr, "listen failed\n");
	return -1;
    }

/* send sock */
    if ((sock1 = socket(AF_INET, SOCK_STREAM, 0)) == SOCKET_ERROR) {
	return -1;
    }
    sock_addr.sin_addr.s_addr = 0x0100007f;  /* loopback */
    sock_addr.sin_family = PF_INET;
    if (ioctlsocket(sock1, FIONBIO, &block) != 0) {
	printf("ioctl failed\n");
    }
    if (connect(sock1, (struct sockaddr *) &sock_addr,
		sizeof sock_addr) == SOCKET_ERROR) {
	int err = WSAGetLastError();
	if (err != WSAEWOULDBLOCK) {
	    printf("unexpected error from connect, %s\n", WSAerror_str(err));
	}
    }

    if ((sock2 = accept(conn_sock, (struct sockaddr *) 0, (int *) 0)) == SOCKET_ERROR) {
	    int err = WSAGetLastError();
	    printf("err was %s\n", WSAerror_str(err));
    }
    
    setsockopt(sock2, IPPROTO_TCP, TCP_NODELAY, (char *) &delay_value,
	       sizeof(delay_value));
    {
	fd_set stXcptFDS,stWriteFDS;
	struct timeval stTimeOut;	/* for select() timeout (none) */
	int wRet;

	FD_ZERO((fd_set FAR*)&(stXcptFDS));
	FD_ZERO((fd_set FAR*)&(stWriteFDS));
	FD_SET(sock1, (fd_set FAR*)&(stWriteFDS));
	FD_SET(sock1, (fd_set FAR*)&(stXcptFDS));
	stTimeOut.tv_sec  = 10;
	stTimeOut.tv_usec = 0;
	wRet = select(-1, NULL, 
		      (fd_set FAR*)&(stWriteFDS),
		      (fd_set FAR*)&(stXcptFDS), 
		      NULL);
	if (wRet == SOCKET_ERROR) {
	    int err = WSAGetLastError();
	    printf("err was %s\n", WSAerror_str(err));
	}
    }
    setsockopt(sock1, IPPROTO_TCP, TCP_NODELAY, (char *) &delay_value,
	       sizeof(delay_value));

    filedes[0] = sock1;
    filedes[1] = sock2;
    return 0;
}
#endif

static void
setup_wake_mechanism(cl)
DEControlList cl;
{
    int filedes[2];

    if (cl->wake_read_fd != -1) return;
    if (pipe(filedes) != 0) {
	perror("Pipe for wake not created.  Wake mechanism inoperative.");
	return;
    }
    cl->wake_read_fd = filedes[0];
    cl->wake_write_fd = filedes[1];
    DEControlList_unlock(cl);
    DEControlList_add_select(cl, cl->wake_read_fd, read_wake_fd, 
			     (DEPort)cl->wake_read_fd, NULL);
    DEControlList_lock(cl);
    
}

static void
wake_server_thread(cl)
DEControlList cl;
{
    static char buffer = 'W';  /* doesn't matter what we write */
    if (cl->wake_write_fd != -1) {
#ifdef HAVE_WINDOWS_H
	send(cl->wake_write_fd, &buffer, 1, 0);
#else
	write(cl->wake_write_fd, &buffer, 1);
#endif
    }
}

extern void
DEControlList_set_control_style(cl, style)
DEControlList cl;
DEControlStyle style;
{
    DEControlList_lock(cl);
    cl->control_style = style;
    switch (style) {
    case DESingleThreaded:
	shutdown_wake_mechanism(cl);
	break;
    case DEDedicatedServerThread:
	setup_wake_mechanism(cl);
	break;
    default:
	assert(FALSE);
    }
    DEControlList_unlock(cl);
}

extern DEControlStyle
DEControlList_get_control_style(cl)
DEControlList cl;
{
    return cl->control_style;
}

typedef struct _DECondition {
    DECondition next;
    int condition_num;
    int waiting;
    int signaled;
    int failed;
    thr_condition_t cond_condition;
    DEPort dep;
    void *client_data;
} DECondition_s;

static DECondition
DECondition_find(cl, condition)
DEControlList cl;
int condition;
{
    DECondition next = cl->condition_list;
    while (next != NULL) {
	if (next->condition_num == condition) {
	    return next;
	}
	next = next->next;
    }
    fprintf(stderr, "Serious internal error.  Use of condition %d, no longer in control list\n", condition);
    return NULL;
}

extern int
DECondition_get(de, dep)
DExchange de;
DEPort dep;
{
    DEControlList cl = de->control_list;
    DECondition cond = DEmalloc(sizeof(DECondition_s));
    DEControlList_lock(cl);
    cond->next = cl->condition_list;
    cl->condition_list = cond;
    cond->condition_num = cl->next_condition_num++;
    cond->dep = dep;
    if (cl->next_condition_num >= 0xffffff) {
	/* recycle at  (16 M - 1) [ Caution on number reuse ] */
	cl->next_condition_num = 0;
    }
    cond->waiting = 0;
    cond->signaled = 0;
    cond->failed = 0;

    switch (cl->control_style) {
    case DESingleThreaded:
	break;
    case DEDedicatedServerThread:
    case DEOccasionalPolling:
	cond->cond_condition = thr_condition_alloc();
	break;
    }
    DEControlList_unlock(cl);
    return cond->condition_num;
}

static int de_control_debug_flag = 0;

static void
DECondition_trigger(cond, cl) 
DECondition cond;
DEControlList cl;
{
    if (de_control_debug_flag) {
	printf("Triggering DEcondition %d\n", cond->condition_num);
    }
    if (cond->waiting) {
	switch (cl->control_style) {
	case DESingleThreaded:
	    break;
	case DEDedicatedServerThread:
	case DEOccasionalPolling:
	    if (de_control_debug_flag) {
		printf("Triggering DEcondition %d, thr_cond %lx\n", 
		       cond->condition_num, (long)cond->cond_condition);
	    }
	    thr_condition_signal(cond->cond_condition);
	    break;
	}
    }
    if (de_control_debug_flag) {
	printf("After trigger for DEcondition %d\n", 
	       cond->condition_num);
    }
}

extern void
DEport_fail_conditions(dep)
DEPort dep;
{
    DEControlList cl = dep->de->control_list;
    DECondition cond_list;
    DEControlList_lock(cl);
    cond_list = cl->condition_list;
    while(cond_list != NULL) {
	if (cond_list->dep == dep) {
	    cond_list->failed = 1;
	    DECondition_trigger(cond_list, cl);
	}
	cond_list = cond_list->next;
    }
    DEControlList_unlock(cl);
}

void
DECondition_destroy(cl, condition)
DEControlList cl;
int condition;
{
    DECondition cond = NULL, prev = NULL;
    DECondition next = NULL;

    if (cl->condition_list->condition_num == condition) {
	cond = cl->condition_list;
	cl->condition_list = cl->condition_list->next;
    } else {
	prev = cl->condition_list;
	next = cl->condition_list->next;
	while (next != NULL) {
	    if (next->condition_num == condition) {
		cond = next;
		prev->next = next->next;
		break;
	    }
	    prev = next;
	    next = next->next;
	}
    }
    if (cond == NULL) {
	fprintf(stderr, "Serious internal error.  Use of condition %d, no longer in control list\n", condition);
    } else {
	/* free internal elements */
	switch (cl->control_style) {
	case DESingleThreaded:
	    break;
	case DEDedicatedServerThread:
	case DEOccasionalPolling:
	    thr_condition_free(cond->cond_condition);
	    break;
	}
	DEfree(cond);
    }
}

extern int
DECondition_wait(de, condition)
DExchange de;
int condition;
{
    DECondition cond;
    DEControlList cl = de->control_list;
    int result;

    if (de_control_debug_flag) {
	printf("Waiting for DEcondition %d\n", condition);
    }
    DEControlList_lock(cl);
    if (de_control_debug_flag) {
	printf("1Waiting for DEcondition %d\n", condition);
    }
    cond = DECondition_find(cl, condition);

    if (cond->signaled) {
	if (de_control_debug_flag) {
	    printf("DEcondition %d already signalled\n", condition);
	}
	DEControlList_unlock(cl);
	return 1;
    }
    if (cond->failed) {
	if (de_control_debug_flag) {
	    printf("DEcondition %d already failed\n", condition);
	}
	DEControlList_unlock(cl);
	return 0;
    }
    cond->waiting++;
    if (de_control_debug_flag) {
	printf("In condition wait, server thread = %lx\n", 
	       (long)cl->server_thread);
    }
    switch (cl->control_style) {
    case DESingleThreaded:
    case DEDedicatedServerThread:
    case DEOccasionalPolling:
	if (cl->server_thread == NULL) {
	    while (!(cond->signaled || cond->failed)) {
		if (de_control_debug_flag) {
		    printf("Polling for DEcondition %d\n", condition);
		}
		DEControlList_unlock(cl);
		DEControlList_poll_and_handle_timeout(cl, 0, 100000);
		DEControlList_lock(cl);
	    }
	    if (de_control_debug_flag) {
		printf("after Polling for DEcondition %d\n", condition);
	    }
	    /* the poll and handle will set cl->server_thread, restore it */
	    cl->server_thread = NULL;
	    if (de_control_debug_flag) {
		printf("In condition wait, reset server thread = %lx\n", 
		       (long)cl->server_thread);
	    }
	} else if (thr_thread_self() == cl->server_thread) {
	    /* we're the server thread */
	    while (!(cond->signaled || cond->failed)) {
		if (de_control_debug_flag) {
		    printf("polling for DEcondition %d\n", condition);
		}
		DEControlList_unlock(cl);
		DEControlList_poll_and_handle_timeout(cl, 0, 100000);
		DEControlList_lock(cl);
	    }
	} else {
	    /* some other thread is the server thread */
	    if (!gen_thr_initialized()) {
		fprintf(stderr, "Gen_Thread library not initialized.\n");
		break;
	    }
	    if (de_control_debug_flag) {
		printf("Waiting for DEcondition %d, thr_cond %lx\n", 
		       condition, (long)cond->cond_condition);
	    }
	    thr_condition_wait(cond->cond_condition, cl->control_list_lock);
	    if (de_control_debug_flag) {
		printf("After wait for DEcondition %d, thr_cond %lx\n", 
		       condition, (long)cond->cond_condition);
	    }
	}
	break;
    }
    result = cond->signaled;
    DECondition_destroy(cl, condition);
    DEControlList_unlock(cl);
    if (de_control_debug_flag) {
	printf("Return from wait DEcondition %d\n", condition);
    }
    return result;
}

extern void
DECondition_signal(de, condition)
DExchange de;
int condition;
{
    DECondition cond;
    DEControlList cl = de->control_list;
    DEControlList_lock(cl);
    cond = DECondition_find(cl, condition);
    cond->signaled = 1;
    DECondition_trigger(cond, cl);
    DEControlList_unlock(cl);
}

extern void
DECondition_set_client_data(de, condition, client_data)
DExchange de;
int condition;
void *client_data;
{
    DECondition cond;
    DEControlList cl = de->control_list;
    DEControlList_lock(cl);
    cond = DECondition_find(cl, condition);
    cond->client_data = client_data;
    DEControlList_unlock(cl);
}

extern void *
DECondition_get_client_data(de, condition)
DExchange de;
int condition;
{
    DECondition cond;
    void *client_data;
    DEControlList cl = de->control_list;
    DEControlList_lock(cl);
    cond = DECondition_find(cl, condition);
    client_data = cond->client_data;
    DEControlList_unlock(cl);
    return client_data;
}
