#include "config.h"
#define FUNCPROTO
#include "io.h"
#ifdef HAVE_GEN_THREAD_H
#include "gen_thread.h"
#else
#define gen_thr_initialized() 0
#define thr_mutex_lock(m)
#define thr_mutex_unlock(m)
#endif
#include "DE.h"
#include "de_internal.h"
#include "de_lock.h"
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#include <stdio.h>
#include <string.h>
#include "useful.h"
#include "comm_group.h"
#ifdef STDC_HEADERS
#include <stdarg.h>
#else
#include <varargs.h>
#endif
extern int vfprintf();

#ifndef NO_DERIVED
#include "ecl.h"
#endif
#include "channel.h"

IOField channel_attend_msg_flds[] =
{
 {"channel", "string", sizeof(char *), IOOffset(AttendMsgPtr, chan_str)},
    {"condition", "integer", sizeof(int), IOOffset(AttendMsgPtr, cond)},
{"client channel", "char[8]", 1, IOOffset(AttendMsgPtr, client_channel)},
    {(char *) 0, (char *) 0, 0, 0}
};

IOField sink_subscribe_msg_flds[] =
{
    {"server_channel", "char[8]", 1,
     IOOffset(SinkSubscribeMsgPtr, server_channel)},
    {"client_channel", "char[8]", 1,
     IOOffset(SinkSubscribeMsgPtr, client_channel)},
    {(char *) 0, (char *) 0, 0, 0}
};

/* basically the same but still ... */

IOField source_subscribe_msg_flds[] =
{
    {"server_channel", "char[8]", 1,
     IOOffset(SourceSubscribeMsgPtr, server_channel)},
    {"client_channel", "char[8]", 1,
     IOOffset(SourceSubscribeMsgPtr, client_channel)},
    {(char *) 0, (char *) 0, 0, 0}
};

IOField channel_exists_attend_response_msg_flds[] =
{
    {"condition", "integer", sizeof(int),
     IOOffset(ExistsAttendResponseMsgPtr, condition)},
    {"channel", "char[8]", 1, IOOffset(ExistsAttendResponseMsgPtr, channel)},
    {(char *) 0, (char *) 0, 0, 0}
};

IOField AttendRec_flds[] =
{
    {"host", "string", sizeof(char *), IOOffset(AttendRecPtr, host)},
    {"IPport", "integer", sizeof(int), IOOffset(AttendRecPtr, IPport)},
    {"channel", "char[8]", 1, IOOffset(AttendRecPtr, channel)},
    {"filter", "string", sizeof(char*), IOOffset(AttendRecPtr, filter)},
    {"filter_has_output_param", "integer", sizeof(int), IOOffset(AttendRecPtr, filter_has_output_param)},
    {(char *) 0, (char *) 0, 0, 0}
};

IOField channel_attend_response_msg_flds[] =
{
    {"condition", "integer", sizeof(int),
     IOOffset(AttendResponseMsgPtr, condition)},
    {"type_name", "string", sizeof(char *),
     IOOffset(AttendResponseMsgPtr, type_name)},
    {"sink_count", "integer", sizeof(int),
     IOOffset(AttendResponseMsgPtr, sink_count)},
    {"sinks", "AttendRec[sink_count]", sizeof(AttendRec),
     IOOffset(AttendResponseMsgPtr, sinks)},
    {"source_count", "integer", sizeof(int),
     IOOffset(AttendResponseMsgPtr, source_count)},
    {"sources", "AttendRec[source_count]", sizeof(AttendRec),
     IOOffset(AttendResponseMsgPtr, sources)},
    {"member_count", "integer", sizeof(int),
     IOOffset(AttendResponseMsgPtr, member_count)},
    {"members", "AttendRec[member_count]", sizeof(AttendRec),
     IOOffset(AttendResponseMsgPtr, members)},
    {"der_count", "integer", sizeof(int),
     IOOffset(AttendResponseMsgPtr, der_count)},
    {"derivatives", "AttendRec[der_count]", sizeof(AttendRec),
     IOOffset(AttendResponseMsgPtr, derivatives)},
    {(char *) 0, (char *) 0, 0, 0}
};

IOField member_subscribe_msg_flds[] =
{
    {"server_channel", "char[8]", 1,
     IOOffset(MemberSubscribeMsgPtr, server_channel)},
    {"client_channel", "char[8]", 1,
     IOOffset(MemberSubscribeMsgPtr, client_channel)},
    {(char *) 0, (char *) 0, 0, 0}
};


IOField event_msg_flds[] =
{
    {"channel", "char[8]", sizeof(char),
     IOOffset(EventMsgPtr, channel)},
    {"global_chan_id", "char[16]", sizeof(char),
     IOOffset(EventMsgPtr, global_chan_id)},
    {"event_len", "integer", sizeof(int),
     IOOffset(EventMsgPtr, event_len)},
    {"event", "char[event_len]", sizeof(char),
     IOOffset(EventMsgPtr, event)},
    {(char *) 0, (char *) 0, 0, 0}
};

IOField request_event_msg_flds[] =
{
    {"channel", "char[8]", sizeof(char),
     IOOffset(RequestEventMsgPtr, channel)},
    {(char *) 0, (char *) 0, 0, 0}
};

IOField event_vec_elem_flds[] =
{
    {"len", "integer", sizeof(((IOEncodeVector)0)[0].iov_len), 
     IOOffset(IOEncodeVector, iov_len)},
    {"elem", "char[len]", sizeof(char), IOOffset(IOEncodeVector, iov_base)},
    {(char *) 0, (char *) 0, 0, 0}
};

IOField eventv_msg_flds[] =
{
    {"channel", "char[8]", sizeof(char),
     IOOffset(EventVMsgPtr, channel)},
    {"global_chan_id", "char[16]", sizeof(char),
     IOOffset(EventVMsgPtr, global_chan_id)},
    {"eventv_len", "integer", sizeof(int),
     IOOffset(EventVMsgPtr, eventv_len)},
    {"eventv", "EventVecElem[eventv_len]", sizeof(struct _io_encode_vec),
     IOOffset(EventVMsgPtr, eventv)},
    {(char *) 0, (char *) 0, 0, 0}
};

static event_queue dequeue_event ARGS((DEControlList control_list));
static void internal_cancel_source_subscribe ARGS((ECSourceHandle handle));

int channel_verbose = -1;

/* trace_out - for debuging and tracing */
/* prints the value of function string, formats the output by */
/* indenting first if direction is a 2, and un-indenting if */
/* direction is a 0, if direction is a 1, indenting is increased */
/* for the traces that follow. */
#ifdef STDC_HEADERS
extern void
ECtrace_out(char *format, int direction, ...)
#else
extern void
ECtrace_out(format, direction, va_alist)
char *format;
int direction;
va_dcl
#endif
{
    va_list ap;
    static int tabstop = 0;
    int i;

    if (channel_verbose == -1) {
	char *verb = getenv("CHANNEL_VERBOSE");
	channel_verbose = 0;
	if (verb) {
	    if (sscanf(verb, "%d", &channel_verbose) != 1) {
		/* enable at lowest level */
		channel_verbose = 1;
	    }
	}
    }

    if (channel_verbose) {
#ifdef STDC_HEADERS
	va_start(ap, direction);
#else
	va_start(ap);
#endif
	if (direction == 0)
	    tabstop--;
	if (direction == 2)
	    tabstop++;
	for (i = 0; i < tabstop; i++)
	    printf("  ");
	vfprintf(stdout, format, ap);
	va_end(ap);
	switch (direction) {
	case 0:
	    printf(" stop");
	    break;
	case 1:
	    printf(" start");
	    tabstop++;
	    break;
	case 2:
	    tabstop--;
	    break;
	}
	printf("\n");
    }
}

extern void
dump_EChannel(chan)
EChannel chan;
{
    char *ID = ECglobal_id(chan);
    int i;

    printf("Local Channel %016lx\n", (long) chan);
    printf("    Sink count = %d, Source count = %d\n", chan->sink_count,
	   chan->source_count);
    printf("    Global ID = %s\n", ID);
    DEfree(ID);
    if (chan->type_format) {
	char *format_name = global_name_of_IOformat(chan->type_format);
	printf("	type = %s\n", format_name);
	free(format_name);
    }
    printf("    %d remote sinks\n", chan->rem_sink_list_len);
    for (i = 0; i < chan->rem_sink_list_len; i++) {
	printf("	%d) - channel %02x%02x%02x%02x%02x%02x%02x%02x, %lx\n",
	       i, chan->rem_sink_list[i].channel[0],
	       chan->rem_sink_list[i].channel[1],
	       chan->rem_sink_list[i].channel[2],
	       chan->rem_sink_list[i].channel[3],
	       chan->rem_sink_list[i].channel[4],
	       chan->rem_sink_list[i].channel[5],
	       chan->rem_sink_list[i].channel[6],
	       chan->rem_sink_list[i].channel[7],
	       (long) chan->rem_sink_list[i].dep);
    }

    printf("    %d remote sources\n", chan->rem_source_list_len);
    for (i = 0; i < chan->rem_source_list_len; i++) {
	printf("	%d) - channel %02x%02x%02x%02x%02x%02x%02x%02x, %lx\n",
	       i, chan->rem_source_list[i].channel[0],
	       chan->rem_source_list[i].channel[1],
	       chan->rem_source_list[i].channel[2],
	       chan->rem_source_list[i].channel[3],
	       chan->rem_source_list[i].channel[4],
	       chan->rem_source_list[i].channel[5],
	       chan->rem_source_list[i].channel[6],
	       chan->rem_source_list[i].channel[7],
	       (long) chan->rem_source_list[i].dep);
    }

    printf("    %d remote members\n", chan->rem_member_list_len);
    for (i = 0; i < chan->rem_member_list_len; i++) {
	printf("	%d) - channel %02x%02x%02x%02x%02x%02x%02x%02x, %lx\n",
	       i, chan->rem_member_list[i].channel[0],
	       chan->rem_member_list[i].channel[1],
	       chan->rem_member_list[i].channel[2],
	       chan->rem_member_list[i].channel[3],
	       chan->rem_member_list[i].channel[4],
	       chan->rem_member_list[i].channel[5],
	       chan->rem_member_list[i].channel[6],
	       chan->rem_member_list[i].channel[7],
	       (long) chan->rem_member_list[i].dep);
    }
    printf("    %d local sinks\n", chan->local_sink_list_len);
    for (i = 0; i < chan->local_sink_list_len; i++) {
	printf("	%d) - func %lx, data %lx, de %lx, context %lx, condition %d\n", i,
	       (long) chan->local_sink_list[i].func,
	       (long) chan->local_sink_list[i].client_data,
	       (long) chan->local_sink_list[i].de,
	       (long) chan->local_sink_list[i].context,
	       (int)  chan->local_sink_list[i].cond);
    }
    printf("    %d local sources\n", chan->local_source_list_len);
    for (i = 0; i < chan->local_source_list_len; i++) {
	printf("	%d) -  func %lx, data %lx, ioformat %lx", i,
	       (long) chan->local_source_list[i].func,
	       (long) chan->local_source_list[i].client_data,
	       (long) chan->local_source_list[i].source_handle);
    }
    printf("    %d derived channels\n", chan->der_list_len);
    for (i = 0; i < chan->der_list_len; i++) {
	printf("    derived event channel %d\n", i);
	printf("    ========================\n");
	printf("	filter = %s\n", chan->der_chan_list[i].filter);
	printf("	has_output_param = %d\n", 
	       chan->der_chan_list[i].has_output_param);
	dump_EChannel(chan->der_chan_list[i].derived_chan);
    }
}

static void
deref_block(block)
free_block_rec_p block;
{
    DExchange locking_de = block->locking_de;
    block->ref_count--;
    DExchange_lock(locking_de);
    if (block->ref_count == 0) {
	if (block->de) {
	    DEreturn_buffer(block->de, block->block);
	} else if (block->free_func != (EventFreeFunction) NULL) {
	    block->free_func(block->block);
	} else if (block->block != NULL) {
	    DEfree(block->block);
	} else {
	    assert(FALSE);
	}
	DEfree(block);
    }
    DExchange_unlock(locking_de);
}

static ECSourceHandle
new_source_handle()
{
    ECSourceHandle handle = DEmalloc(sizeof(*handle));
    handle->chan = NULL;
    handle->ioformat = NULL;
    handle->source_num = 0;
    handle->der_list_len = 0;
    handle->der_source_list = NULL;
    return handle;
}

extern void
EChannel_lock(chan)
EChannel chan;
{
    if (!gen_thr_initialized())
	return;

    if (chan->lock) {
	thr_mutex_lock(chan->lock);
    }
}

extern void
EChannel_unlock(chan)
EChannel chan;
{
    if (!gen_thr_initialized())
	return;

    if (chan->lock) {
	thr_mutex_unlock(chan->lock);
    }
}

extern EChannel
string_to_channel_ptr(char *chan_id)
{
    unsigned char chan_char[8] = {0, 0, 0, 0, 0, 0, 0, 0};
    unsigned int chan_int[8] =	{0, 0, 0, 0, 0, 0, 0, 0};
    EChannel chan;

    sscanf(chan_id, "%02x%02x%02x%02x%02x%02x%02x%02x",
	   &chan_int[0], &chan_int[1], &chan_int[2], &chan_int[3],
	   &chan_int[4], &chan_int[5], &chan_int[6], &chan_int[7]);

    chan_char[0] = (unsigned char) chan_int[0];
    chan_char[1] = (unsigned char) chan_int[1];
    chan_char[2] = (unsigned char) chan_int[2];
    chan_char[3] = (unsigned char) chan_int[3];
    chan_char[4] = (unsigned char) chan_int[4];
    chan_char[5] = (unsigned char) chan_int[5];
    chan_char[6] = (unsigned char) chan_int[6];
    chan_char[7] = (unsigned char) chan_int[7];
    memcpy(&chan, chan_char, sizeof(chan));
    return chan;
}

extern void
EChannel_register_subformats(context, field_list, format_list)
IOContext context;
IOFieldList field_list;
DEFormatList format_list;
{ 
    char **subformat_names;
    int i = 0;
    subformat_names = get_subformat_names(field_list);
    while(subformat_names && (subformat_names[i] != NULL)) {
	IOFormat subformat;
	subformat = get_IOformat_by_name((IOFile)context, 
					 subformat_names[i]);
	if ((subformat == NULL) && format_list){
	    int j = 0;
	    while (format_list[j].format_name != NULL) {
		if (strcmp(format_list[j].format_name, subformat_names[i])==0){
		    EChannel_register_subformats(context, format_list[j].field_list, 
						 format_list);
		    subformat = 
			register_IOcontext_format(subformat_names[i],
						  format_list[j].field_list, 
						  context);
		    break;
		}
		j++;
	    }
	}
	if (subformat == NULL) {
	    if (IOhas_error((IOFile)context)) {
		IOperror((IOFile)context, "subformat registration");
	    }
	    fprintf(stderr, "Subformat name \"%s\" registration failure\n",
		    subformat_names[i]);
	}
	free(subformat_names[i]);
	i++;
    }
    if (subformat_names) free(subformat_names);
}

extern
EChannel
EChannel_typed_create(de, field_list, format_list)
DExchange de;
IOFieldList field_list;
DEFormatList format_list;
{
    EChannel chan = EChannel_create(de);

    DExchange_context_lock(de);
    if (de->context == NULL) {
	/* create IO context if it does not already exist */
	de->context = create_IOcontext(NULL);
    }
    chan->subcontext = create_IOsubcontext(de->context);
    EChannel_register_subformats(chan->subcontext, field_list, format_list);

    chan->type_format = 
	register_IOcontext_format("anonymous", field_list, 
				  chan->subcontext);

    DExchange_context_unlock(de);
    
    return chan;
}


extern
 EChannel
EChannel_create(de)
DExchange de;
{
    EChannel chan = DEmalloc(sizeof(EChannel_struct));

    ECtrace_out("EChannel_create", 1);

    chan->sink_count = 0;
    chan->source_count = 0;

    chan->rem_sink_list_len = 0;
    chan->rem_sink_list = DEmalloc(sizeof(struct _rem_attendr));

    chan->rem_source_list_len = 0;
    chan->rem_source_list = DEmalloc(sizeof(struct _rem_attendr));

    chan->rem_member_list_len = 0;
    chan->rem_member_list = DEmalloc(sizeof(struct _rem_attendr));
    chan->local_sink_list_len = 0;
    chan->local_sink_list = DEmalloc(sizeof(struct _local_sink));
    chan->local_source_list_len = 0;
    chan->local_source_list = DEmalloc(sizeof(struct _local_source));

    chan->der_list_len = 0;
    chan->der_chan_list = DEmalloc(sizeof(struct _derived_chan));
    chan->parent_list_len = 0;
    chan->parent_chan_list = DEmalloc(sizeof(EChannel));

    chan->global_id.home_host = DExchange_host_name(de);
    chan->global_id.IPport = DExchange_inet_port(de);
    memset(&chan->global_id.channel, 0, sizeof(chan->global_id.channel));
    if (chan->global_id.IPport == -1) {
	printf("Warning, IPport = -1.  Must do DExchange_listen() before using channels.\n");
	DEfree(chan);
	return NULL;
    }
    chan->lock = thr_mutex_alloc();
    chan->ready = 1;
    chan->subscribe_func = (ECSubscribeFunction) NULL;

    DExchange_lock(de);
    if (de->channel_list == NULL) {
	de->channel_list = DEmalloc(sizeof(EChannel));
    } else {
	de->channel_list =
	    (EChannel *) DErealloc(de->channel_list,
			     sizeof(EChannel) * (de->channel_count + 1));
    }
    de->channel_list[de->channel_count++] = chan;
    DExchange_unlock(de);
    ECtrace_out("EChannel_create returning channel %lx", 0, chan);
    chan->subcontext = NULL;
    chan->filter_data = NULL;
    chan->data_field_list = NULL;
    chan->data_format_list = NULL;

    chan->de = de;
    chan->type_format = NULL;
    chan->periodic_info = NULL;
    return chan;
}

extern
void
EChannel_destroy(chan)
EChannel chan;
{
    int i;
/* event_queue item; */
    DExchange de = chan->de;

    ECtrace_out("EChannel_destroy of channel %lx", 1, chan);
    if (chan->periodic_info != NULL) {
	DEperiodic_info_free(chan->periodic_info);
    }
    EChannel_lock(chan);
    DEfree(chan->rem_sink_list);
    DEfree(chan->rem_source_list);
    DEfree(chan->rem_member_list);
    for(i = 0; i< chan->local_sink_list_len; i++) {
	if (chan->local_sink_list[i].context != NULL) {
	    free_IOcontext(chan->local_sink_list[i].context);
	    chan->local_sink_list[i].context = NULL;
	}
    }
    DEfree(chan->local_sink_list);
    if (chan->subcontext) free_IOsubcontext(chan->subcontext);

    /* destroy and free the derived event channels */
    for (i = 0; i < chan->der_list_len; i++) {
	EChannel_destroy(chan->der_chan_list[i].derived_chan);
    }
    for (i=0; i< chan->parent_list_len; i++) {
	remove_derived_channel(chan->parent_chan_list[i], chan);
    }
    DEfree(chan->parent_chan_list);

    while(chan->local_source_list_len != 0) {
	int last = chan->local_source_list_len -1;
	internal_cancel_source_subscribe(chan->local_source_list[last].source_handle);
    }
    DEfree(chan->local_source_list);

    DEfree(chan->der_chan_list);
    thr_mutex_free(chan->lock);

    chan->rem_member_list_len = 0;
    chan->ready = 0;

    DExchange_lock(de);
    for (i = 0; i < de->channel_count; i++) {
	if (de->channel_list[i] == chan) {
	    de->channel_count--;
	    for (; i < (de->channel_count); i++) {
		de->channel_list[i] = de->channel_list[i + 1];
	    }
	    break;
	}
    }
    DEfree(chan);
    DExchange_unlock(de);
    ECtrace_out("EChannel_destroy", 0);
}

static void send_request_event ARGS((EChannel chan, int i));

static void send_remote_event ARGS((EChannel chan, int i, void *event,
				    int event_len));
static void sendv_remote_event ARGS((EChannel chan, int i,
				     IOEncodeVector eventV));
static void queue_event_for_dispatch ARGS((EChannel chan, int sink_index,
					   void *event, int event_len,
					   free_block_rec_p block_rec));
static void queue_eventv_for_dispatch ARGS((EChannel chan, int sink_index,
					    IOEncodeVector eventv,
					    free_block_rec_p block_rec));

static void send_sink_subscribe ARGS((EChannel chan, int i, int subscribe));

static void send_source_subscribe ARGS((EChannel chan, int i, int subscribe));

static void do_event_dispatch ARGS((DEControlList control_list));

static void
internal_submit_general_event ARGS((ECSourceHandle handle, void *event,
				    int event_len,
				    free_block_rec_p block_rec));

static void
internal_submit_general_eventV ARGS((ECSourceHandle handle, 
				     IOEncodeVector eventV,
				     free_block_rec_p block_rec));


extern void
ECrequest_event(handle, req_try)
ECSinkHandle handle;
int req_try;
{
    int i = 0;
    EChannel chan = handle->chan;
    int cond = 0;

    EChannel_lock(chan);
    ECtrace_out("In ecrequest_event, handle %lx", 1, handle);
    for (i = 0; i < chan->rem_source_list_len; i++) {
	if (chan->rem_source_list[i].dep != NULL) {
	    EChannel_unlock(chan);
	    send_request_event(chan, i);
	    EChannel_lock(chan);
	}
    }

    DEControlList_lock(chan->de->control_list);
    for (i = 0; i < chan->local_source_list_len; i++) {
	if (chan->local_source_list[i].func) {
	    chan->local_source_list[i].func(chan->local_source_list[i].source_handle,
					    chan->local_source_list[i].client_data);
	}
    }
    DEControlList_unlock(chan->de->control_list);

    /* get condition for req try */
    if (req_try) {
	cond = DECondition_get(chan->de, NULL);
	chan->local_sink_list[handle->sink_num].cond = cond;
    }
  
    EChannel_unlock(chan);

    /* wait on it */
    if (req_try) {
	if (DECondition_wait(chan->de, cond)) {
	    ECtrace_out("ECrequest_event sink is ready %lx", 3, chan);
	} else {
	    /* wait failed, dep is dead - is that so ? */
	    EChannel_destroy(chan);
	    ECtrace_out("ECrequest_event, wait failed", 3);
	}
    }

    ECtrace_out("exit ecrequest_event, handle %lx", 0, handle);
}

extern void
ECsubmit_event(handle, event, event_len)
ECSourceHandle handle;
void *event;
int event_len;
{
    internal_submit_general_event(handle, event, event_len, NULL);
    do_event_dispatch(handle->chan->de->control_list);
}

extern void
ECsubmit_typed_event(handle, event)
ECSourceHandle handle;
void *event;
{
    internal_submit_general_event(handle, event, 0, NULL);
    do_event_dispatch(handle->chan->de->control_list);
}

static void
int_submit_general_typed_event(handle, event, free_func)
ECSourceHandle handle;
void *event;
EventFreeFunction free_func;
{
    free_block_rec_p block_rec = DEmalloc(sizeof(struct free_block_rec));
    block_rec->ref_count = 1;
    block_rec->de = NULL;
    block_rec->free_func = free_func;
    block_rec->block = event;
    block_rec->locking_de = handle->chan->de;
    internal_submit_general_event(handle, event, 0, block_rec);
    deref_block(block_rec);
}

extern void
ECsubmit_general_typed_event(handle, event, free_func)
ECSourceHandle handle;
void *event;
EventFreeFunction free_func;
{
    int_submit_general_typed_event(handle, event, free_func);
    do_event_dispatch(handle->chan->de->control_list);
}

extern void
ECsubmit_general_event(handle, event, event_len, free_func)
ECSourceHandle handle;
void *event;
int event_len;
EventFreeFunction free_func;
{
    free_block_rec_p block_rec = DEmalloc(sizeof(struct free_block_rec));
    block_rec->ref_count = 1;
    block_rec->de = NULL;
    block_rec->free_func = free_func;
    block_rec->block = NULL;
    block_rec->locking_de = handle->chan->de;
    internal_submit_general_event(handle, event, event_len, block_rec);
    deref_block(block_rec);
    do_event_dispatch(handle->chan->de->control_list);
}

extern void
ECsubmit_general_eventV(handle, eventV, free_func)
ECSourceHandle handle;
IOEncodeVector eventV;
EventFreeFunction free_func;
{
    free_block_rec_p block_rec = DEmalloc(sizeof(struct free_block_rec));
    block_rec->ref_count = 1;
    block_rec->de = NULL;
    block_rec->block = eventV;
    block_rec->free_func = (EventFreeFunction) free_func;
    block_rec->locking_de = handle->chan->de;
    internal_submit_general_eventV(handle, eventV, block_rec);
    deref_block(block_rec);
    do_event_dispatch(handle->chan->de->control_list);
}

extern void
ECsubmit_eventV(handle, eventV)
ECSourceHandle handle;
IOEncodeVector eventV;
{
    internal_submit_general_eventV(handle, eventV, NULL);
    do_event_dispatch(handle->chan->de->control_list);
}

extern void
ECsubmit_eventV_block(handle, eventV, free_block)
ECSourceHandle handle;
IOEncodeVector eventV;
void *free_block;
{
    free_block_rec_p block_rec = DEmalloc(sizeof(struct free_block_rec));
    block_rec->ref_count = 1;
    block_rec->de = NULL;
    block_rec->block = free_block;
    block_rec->free_func = (EventFreeFunction) NULL;
    block_rec->locking_de = handle->chan->de;

    internal_submit_general_eventV(handle, eventV, block_rec);
    deref_block(block_rec);
    do_event_dispatch(handle->chan->de->control_list);
}

static void
internal_submit_general_event(handle, event, event_len, block_rec)
ECSourceHandle handle;
void *event;
int event_len;
free_block_rec_p block_rec;
{
    int i = 0;
    IOEncodeVector encodeV = NULL;
    EChannel_lock(handle->chan);
    ECtrace_out("In ecsubmit_general_event, handle %lx, event %lx", 1, handle,
	      event);
    for (i = 0; i < handle->chan->rem_sink_list_len; i++) {
	if (handle->ioformat == NULL) {
	    /* untyped event */
	    if (handle->chan->rem_sink_list[i].dep != NULL) {
		EChannel_unlock(handle->chan);
		send_remote_event(handle->chan, i, event, event_len);
		EChannel_lock(handle->chan);
	    }
	} else {
	    /* typed event */
	    int locked = 0;
	    if (encodeV == NULL) {
		DExchange_context_lock(handle->chan->de);
		locked++;
		encodeV = encode_IOcontext_to_vector((IOContext)iofile_of_IOformat(handle->ioformat),
						     handle->ioformat,
						     event);
		if (encodeV == NULL) {
		    IOperror((IOFile)handle->chan->subcontext, 
			     "encode_IOcontext");
		}
	    }
	    if (handle->chan->rem_sink_list[i].dep != NULL) {
		EChannel_unlock(handle->chan);
		sendv_remote_event(handle->chan, i, encodeV);
		EChannel_lock(handle->chan);
	    }
	    if (locked) {
		DExchange_context_unlock(handle->chan->de);
	    }
	}
    }
    DEControlList_lock(handle->chan->de->control_list);
    for (i = 0; i < handle->chan->local_sink_list_len; i++) {
	if (handle->chan->local_sink_list[i].func != (ECHandlerFunction) NULL) {
	    IOFormat sink_format = handle->chan->local_sink_list[i].format;
	    if (!handle->ioformat || !sink_format ||
		global_name_eq(handle->ioformat, sink_format)) {
		queue_event_for_dispatch(handle->chan, i, event, event_len,
					 block_rec);
	    } else {
		int len;
		char *buffer;
		IOContext context;
		DExchange_context_lock(handle->chan->de);
		context = (IOContext)
		    iofile_of_IOformat(handle->ioformat);
		buffer = encode_IOcontext_buffer(context,
						 handle->ioformat, event,
						 &len);
		DExchange_context_unlock(handle->chan->de);
		queue_event_for_dispatch(handle->chan, i, buffer, len, NULL);
	    }
	}
    }
    DEControlList_unlock(handle->chan->de->control_list);
    for (i = 0; i < handle->der_list_len; i++) {
	int type_len = handle->der_source_list[i].target_type_len;
	FilterFunc func = handle->der_source_list[i].filter;
	EChannel derived_chan = handle->der_source_list[i].derived_chan;
	ECSourceHandle der_handle = handle->der_source_list[i].source_handle;

	ECtrace_out("working on derived chan %lx", 3, derived_chan);
	ECtrace_out("filter is %lx, has output %d\n", 3, (long) func,
		    handle->der_source_list[i].has_output_param);
	if (!EChas_sinks(der_handle)) {
	    ECtrace_out("Output channel has no sinks", 3);
	    continue;
	}
	if (handle->der_source_list[i].has_output_param) {
	    int tmp;
	    void *event2 = DEmalloc(type_len);
	    memset(event2, 0, type_len);
	    tmp = (func)(event, event2, derived_chan->filter_data);
	    if (tmp != 0) {
		ECtrace_out("Derivation function returned %d, submitting event", 
			    3, tmp);
		int_submit_general_typed_event(der_handle, event2, free);
	    } else {
		ECtrace_out("Derivation function returned 0, no event submitted", 3);
	    }
	} else {
	    int tmp = (func)(event, NULL, derived_chan->filter_data);
	    if (tmp != 0) {
		ECtrace_out("Derivation function returned %d, submitting event", 
			    3, tmp);
		internal_submit_general_event(der_handle, event, 0, 
					      block_rec);
	    } else {
		ECtrace_out("Derivation function returned 0, no event submitted", 3);
	    }
	}
    }
    ECtrace_out("exit ecsubmit_general_event, handle %lx", 0, handle);
    EChannel_unlock(handle->chan);
}

static void
internal_submit_general_eventV(handle, eventV, block_rec)
ECSourceHandle handle;
IOEncodeVector eventV;
free_block_rec_p block_rec;
{
    int i = 0;
    EChannel_lock(handle->chan);
    ECtrace_out("In ecsubmit_general_eventV, handle %lx, eventV %lx", 1, handle, eventV);
    for (i = 0; i < handle->chan->rem_sink_list_len; i++) {
	if (handle->chan->rem_sink_list[i].dep != NULL) {
	    EChannel_unlock(handle->chan);
	    sendv_remote_event(handle->chan, i, eventV);
	    EChannel_lock(handle->chan);
	}
    }
    DEControlList_lock(handle->chan->de->control_list);
    for (i = 0; i < handle->chan->local_sink_list_len; i++) {
	if (handle->chan->local_sink_list[i].func != (ECHandlerFunction) NULL) {
	    queue_eventv_for_dispatch(handle->chan, i, eventV, block_rec);
	}
    }
    DEControlList_unlock(handle->chan->de->control_list);
    for (i = 0; i < handle->chan->der_list_len; i++) {
	ECSourceHandle der_handle;
	ECtrace_out("working on derivitive", 3);
	/* GSE really derived chans here? */
	der_handle = handle->der_source_list[i].source_handle;
	ECsubmit_eventV(der_handle, eventV);
    }
    ECtrace_out("exit ecsubmit_general_eventV, handle %lx", 0, handle);
    EChannel_unlock(handle->chan);
}

extern int
EChas_sinks(handle)
ECSourceHandle handle;
{
    int i;
    ECSourceHandle der_handle;
    if ((handle->chan->rem_sink_list_len != 0) ||
	(handle->chan->local_sink_list_len != 0)) {
	return 1;
    }
    if (handle->chan->der_list_len == 0) {
	return 0;
    }
    der_handle = new_source_handle();
    /* check derived event channels */
    for (i = 0; i < handle->chan->der_list_len; i++) {
	der_handle->chan = handle->chan->der_chan_list[i].derived_chan;
	if (EChas_sinks(der_handle)) {
	    DEfree(der_handle);
	    return 1;
	}
    }
    DEfree(der_handle);
    return 0;
}

static int
num_local_sinks(chan)
EChannel chan;
{
    int i;
    int count = 0;
    for (i = 0; i< chan->local_sink_list_len; i++) {
	if (chan->local_sink_list[i].func != (ECHandlerFunction) NULL) {
	    count++;
	}
    }
    if (chan->der_list_len != 0) {
	for (i = 0; i < chan->der_list_len; i++) {
	    count += num_local_sinks(chan->der_chan_list[i].derived_chan);
	}
    }
    return count;
}

static ECSinkHandle
internal_sink_subscribe ARGS((EChannel chan, DExchange de, 
			      ECHandlerFunction func, 
			      void *client_data));

static ECSinkHandle
internal_typed_sink_subscribe(chan, de, field_list, format_list, func, client_data)
EChannel chan;
DExchange de;
IOFieldList field_list;
DEFormatList format_list;
ECTypedHandlerFunction func;
void *client_data;
{
    IOFieldList type_field_list;
    ECSinkHandle handle;
    IOContext context;

    if (chan->type_format == NULL) {
	fprintf(stderr, "Typed subscribe to an untyped channel not allowed\n");
	return NULL;
    }

    DExchange_context_lock(chan->de);
    if (has_conversion_IOformat(chan->type_format)) {
	type_field_list = field_list_of_IOformat(chan->type_format);

	if (!compare_field_lists(type_field_list, field_list)) {
	    fprintf(stderr, "Incompatible field lists for the same incoming format in different ECtyped_sink_subscribe() calls.\n");
	    DExchange_context_unlock(chan->de);
	    return NULL;
	}
    }

    handle = internal_sink_subscribe(chan, de, (ECHandlerFunction) func,
				     client_data);
    chan->local_sink_list[handle->sink_num].context = context =
	create_IOsubcontext(de->context);
    EChannel_register_subformats(context, field_list, format_list);
    chan->local_sink_list[handle->sink_num].format = 
	register_IOcontext_format("anonymous", field_list, context);
    DExchange_context_unlock(chan->de);
    chan->local_sink_list[handle->sink_num].field_list = field_list;
    chan->local_sink_list[handle->sink_num].format_list = format_list;
    return handle;
    
}


extern ECSinkHandle
ECsink_typed_subscribe(chan, field_list, format_list, func, client_data)
EChannel chan;
IOFieldList field_list;
DEFormatList format_list;
ECTypedHandlerFunction func;
void *client_data;
{
    return internal_typed_sink_subscribe(chan, chan->de, field_list, 
					 format_list, func, client_data);
}

extern ECSinkHandle
ECsink_typed_subscribe_context(chan, field_list, format_list, func, 
			       client_data, de)
EChannel chan;
IOFieldList field_list;
DEFormatList format_list;
ECTypedHandlerFunction func;
void *client_data;
DExchange de;
{
    return internal_typed_sink_subscribe(chan, de, field_list, format_list, 
					   func, client_data);
}

extern ECSinkHandle
ECsink_subscribe(chan, func, client_data)
EChannel chan;
ECHandlerFunction func;
void *client_data;
{
    return internal_sink_subscribe(chan, chan->de, func, client_data);
}

extern ECSinkHandle
ECsink_subscribe_context(chan, func, client_data, de)
EChannel chan;
ECHandlerFunction func;
void *client_data;
DExchange de;
{
    return internal_sink_subscribe(chan, de, func, client_data);
}

extern void
ECcancel_sink_subscribe(handle)
ECSinkHandle handle;
{
    EChannel chan = handle->chan;
    EChannel_lock(handle->chan);

    ECtrace_out("ECcancel_sink_subscribe, chan %lx", 1, chan);

    if (num_local_sinks(chan) == 1) {
	int i;
	ECtrace_out("working on remote members", 3);
	for (i = 0; i < chan->rem_member_list_len; i++) {
	    if (chan->rem_member_list[i].dep != NULL) {
		EChannel_unlock(chan);
		send_sink_subscribe(chan, i, 0); /* false means unsubscribe */
		EChannel_lock(chan);
	    }
	}
    }
    if (chan->local_sink_list[handle->sink_num].context != NULL) {
	free_IOcontext(chan->local_sink_list[handle->sink_num].context);
	chan->local_sink_list[handle->sink_num].context = NULL;
    }
    chan->local_sink_list[handle->sink_num].func = (ECHandlerFunction)NULL;
    ECtrace_out("ECcancel_sink_subscribe, chan %lx", 0, chan);
    EChannel_unlock(chan);
    DEfree(handle);
}
    
static ECSinkHandle
internal_sink_subscribe(chan, de, func, client_data)
EChannel chan;
DExchange de;
ECHandlerFunction func;
void *client_data;
{
    int i;
    ECSinkHandle handle = DEmalloc(sizeof(*handle));

    ECtrace_out("ECsink_subscribe, chan %lx", 1, chan);
    handle->chan = chan;
    EChannel_lock(handle->chan);
    ECtrace_out("working on remote members", 3);
    for (i = 0; i < chan->rem_member_list_len; i++) {
	if (chan->rem_member_list[i].dep != NULL) {
	    EChannel_unlock(chan);
	    send_sink_subscribe(chan, i, 1); /* true means "subscribe" */
	    EChannel_lock(chan);
	}
    }
    chan->local_sink_list =
	DErealloc(chan->local_sink_list,
	   sizeof(struct _local_sink) * (chan->local_sink_list_len + 1));
    chan->local_sink_list[chan->local_sink_list_len].func = func;
    chan->local_sink_list[chan->local_sink_list_len].client_data = client_data;
    chan->local_sink_list[chan->local_sink_list_len].de = de;
    chan->local_sink_list[chan->local_sink_list_len].context = NULL;
    chan->local_sink_list[chan->local_sink_list_len].cond = 0;
    handle->sink_num = chan->local_sink_list_len;

    (chan->local_sink_list_len)++;
    EChannel_unlock(chan);
    if (chan->subscribe_func) {
	chan->subscribe_func(1, chan->rem_sink_list_len,
			     chan->subscribe_client_data);
    }
    if (channel_verbose > 0) {
	printf("After Subscribe\n");
	dump_EChannel(chan);
    }
    ECtrace_out("ECsink_subscribe chan %lx, return handle %lx", 0, chan, handle);
    return handle;
}

/*
 *   Extended with (func, client_data) to support a pull model. The idea is 
 *   that the handler would be executed when the source get a request and 
 *   it would submit an event as part of its execution - fabianb
 *
 */

extern ECSourceHandle
ECsource_subscribe(chan)
EChannel chan;
{
    return ECPsource_subscribe(chan, (ECRequestHandlerFunction)NULL, NULL);
}

extern ECSourceHandle
ECPsource_subscribe(chan, func, client_data)
EChannel chan;
ECRequestHandlerFunction func;
void *client_data;
{
    int i;
  
    ECSourceHandle handle = new_source_handle();

    if (chan->type_format != NULL) {
	fprintf(stderr, 
		"ECsource_subscribe() failed.  No untyped sources on typed channels\n");
	return NULL;
    }

    ECtrace_out("ECsource_subscribe, chan %lx", 1, chan);
    handle->chan = chan;
    EChannel_lock(chan);
    chan->local_source_list =
	DErealloc(chan->local_source_list,
		  sizeof(struct _local_source) * (chan->local_source_list_len + 1));
    chan->local_source_list[chan->local_source_list_len].func = func;
    chan->local_source_list[chan->local_source_list_len].client_data = client_data;
    chan->local_source_list[chan->local_source_list_len].source_handle = handle;
    handle->source_num = chan->local_source_list_len;

    (chan->local_source_list_len)++;

    ECtrace_out("working on remote members", 3);
    for (i = 0; i < chan->rem_member_list_len; i++) {
	if (chan->rem_member_list[i].dep != NULL) {
	    EChannel_unlock(chan);
	    send_source_subscribe(chan, i, 1); /* true means "subscribe" */
	    EChannel_lock(chan);
	}
    }

    EChannel_unlock(chan);

    ECtrace_out("Member counts: \n Local sources %d Local sinks %d \n Remote Sources %d Remote Sinks %d Remote Members %d", 3, 
		chan->local_source_list_len,
		chan->local_sink_list_len,
		chan->rem_source_list_len, 
		chan->rem_sink_list_len, 
		chan->rem_member_list_len);

    ECtrace_out("ECsource_subscribe, chan %lx, returning handle %lx", 0, chan,
		handle);
    return handle;
}

extern void
ECcancel_source_subscribe(handle)
ECSourceHandle handle;
{
    EChannel chan = handle->chan;
    EChannel_lock(chan);
    internal_cancel_source_subscribe(handle);
    EChannel_unlock(chan);
}

static void
internal_cancel_source_subscribe(handle)
ECSourceHandle handle;
{
    int i;  
    EChannel chan = handle->chan;
    if (handle->ioformat != NULL) 
	free_IOcontext((IOContext)iofile_of_IOformat(handle->ioformat));

    /* remove source from channel's list */
    for (i=0; i< handle->chan->local_source_list_len; i++) {
	if (handle->chan->local_source_list[i].source_handle == handle) {
	    for ( ; i < handle->chan->local_source_list_len -1 ; i++) {
		handle->chan->local_source_list[i] =
		    handle->chan->local_source_list[i+1];
	    }
	    handle->chan->local_source_list_len--;
	    break;
	}
    }

    /* 
     *  only if it is the last one in this side it will send a msg 
     *  we need this for remote sources now 
     *
     */
  
    if (chan->local_source_list_len == 1) {
	int i;
	ECtrace_out("working on remote members", 3);
	for (i = 0; i < chan->rem_member_list_len; i++) {
	    if (chan->rem_member_list[i].dep != NULL) {
		EChannel_unlock(chan);
		send_source_subscribe(chan, i, 0); /* false means unsubscribe*/
		EChannel_lock(chan);
	    }
	}
    }
#ifndef NO_DERIVED
    /* deal with derived sources */
    for (i=0; i< handle->der_list_len; i++) {
	ecl_code_free((void*)(long)handle->der_source_list[i].code_struct);
	ECcancel_source_subscribe(handle->der_source_list[i].source_handle);
    }
    
#endif
    DEfree(handle);
}

/*
 *   Extended with (func, client_data) to support a pull model. The idea is 
 *   that the handler would be executed when the source get a request and 
 *   it would submit an event as part of its execution - fabianb
 */ 


extern ECSourceHandle 
ECsource_typed_subscribe (chan, field_list, format_list)
EChannel chan;
IOFieldList field_list;
DEFormatList format_list;
{
    return ECPsource_typed_subscribe(chan, field_list, format_list, 
				     (ECRequestHandlerFunction) NULL, 
				     NULL);
}


extern ECSourceHandle
ECPsource_typed_subscribe(chan, field_list, format_list, func, client_data)
EChannel chan;
IOFieldList field_list;
DEFormatList format_list;
ECRequestHandlerFunction func;
void *client_data;
{
    ECSourceHandle handle = new_source_handle();
    IOContext	source_context = create_IOsubcontext(chan->de->context);
    int i = 0;

    if (chan->type_format == NULL) {
	fprintf(stderr, "ECsource_subscribe() failed.  No typed sources on untyped channels");
	return NULL;
    }
    ECtrace_out("ECsource_subscribe, chan %lx", 1, chan);
    handle->chan = chan;
    EChannel_lock(chan);

    /* assert that the formats are compatible -GSE- */
    
    DExchange_context_lock(chan->de);
    while (format_list && (format_list[i].format_name != NULL)) {
	if (register_IOcontext_format(format_list[i].format_name,
				      format_list[i].field_list,
				      source_context) == NULL) {
	    fprintf(stderr, "Registration of format \"%s\" failed\n",
		    format_list[i].format_name);
	}
	i++;
    }
    handle->ioformat = register_IOcontext_format("anonymous", field_list,
						 source_context);
    DExchange_context_unlock(chan->de);

    /* add to our channels list of sources */
    chan->local_source_list =
	DErealloc(chan->local_source_list,
		  sizeof(struct _local_source) * (chan->local_source_list_len + 1));
    chan->local_source_list[chan->local_source_list_len].func = func;
    chan->local_source_list[chan->local_source_list_len].client_data = client_data;
    chan->local_source_list[chan->local_source_list_len].source_handle = handle;
    handle->source_num = chan->local_source_list_len;
    (chan->local_source_list_len)++;

    ECtrace_out("working on remote members", 3);
    for (i = 0; i < chan->rem_member_list_len; i++) {
	if (chan->rem_member_list[i].dep != NULL) {
	    EChannel_unlock(chan);
	    send_source_subscribe(chan, i, 1); /* true means "subscribe" */
	    EChannel_lock(chan);
	}
    }

    /* for each channel derived for ours, derive a source */
    for (i=0; i< chan->der_list_len; i++) {
	EChannel der_chan = chan->der_chan_list[i].derived_chan;
	IOFieldList output_field_list = NULL;
	DEFormatList output_format_list = NULL;
	int j, ret;
	
	if (chan->der_chan_list[i].has_output_param) {
	    output_field_list = EChannel_get_field_list(der_chan);
	    output_format_list = EChannel_get_format_list(der_chan);
	}
	ret = do_internal_source_derive(handle, der_chan,
					chan->der_chan_list[i].filter, 
					field_list, format_list,
					output_field_list, output_format_list,
					der_chan->data_field_list, 
					der_chan->data_format_list);
	if (ret != 1) {
	    EChannel_destroy(der_chan);
	    return NULL;
	}
	j=0;
	while (output_format_list && 
	       (output_format_list[j].format_name != NULL)) { 
	    free_field_list(output_format_list[j].field_list);
	    free(output_format_list[j].format_name);
	    j++;
	}
	if (output_format_list)	free(output_format_list);
	if (output_field_list) free_field_list(output_field_list);
    }

    EChannel_unlock(chan);

    ECtrace_out("Member counts: \n Local sources %d Local sinks %d \n Remote Sources %d Remote Sinks %d Remote Members %d", 3,
		chan->local_source_list_len,
		chan->local_sink_list_len, 
		chan->rem_source_list_len,  
		chan->rem_sink_list_len,
		chan->rem_member_list_len);

    ECtrace_out("ECsource_subscribe, chan %lx, returning handle %lx", 0, chan,
		handle);

    return handle;
}

extern char *
ECglobal_id(chan)
EChannel chan;
{
    int len = strlen(chan->global_id.home_host);
    char port_str[20] =
    {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    char chan_str[17] =
    {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    unsigned char chan_char[8] =
    {0, 0, 0, 0, 0, 0, 0, 0};
    char *global_id_str;

    sprintf(port_str, "%d", chan->global_id.IPport);
    memset(chan_char, 0, sizeof(chan_char));
    memcpy(chan_char, &chan, sizeof(chan));
    sprintf(chan_str, "%02x%02x%02x%02x%02x%02x%02x%02x",
	    (unsigned int) chan_char[0],
	    (unsigned int) chan_char[1],
	    (unsigned int) chan_char[2],
	    (unsigned int) chan_char[3],
	    (unsigned int) chan_char[4],
	    (unsigned int) chan_char[5],
	    (unsigned int) chan_char[6],
	    (unsigned int) chan_char[7]);
    len += strlen(port_str);
    len += 16 /* chan_str */  + 2 /* separators */ ;
    global_id_str = DEmalloc(len + 1);
    memset(global_id_str, 0, len + 1);
    sprintf(global_id_str, "%s@%s@%s", chan->global_id.home_host, port_str,
	    chan_str);
    return global_id_str;
}

extern EChannel
EChannel_open(de, chan_id)
DExchange de;
char *chan_id;
{
    char *chan_id_copy = DEmalloc(strlen(chan_id) + 1);
    char *host_str;
    char *port_str;
    char *chan_str;
    int port_id;
    char *tmp;
    DEPort dep;
    EChannel chan;
    AttendMsg attend_msg;
    int cond;

    ECtrace_out("ECchannel_open, ID=%s", 1, chan_id);
    strcpy(chan_id_copy, chan_id);
    host_str = chan_id_copy;
    tmp = strchr(chan_id_copy, '@');
    if (tmp == NULL) {
	ECtrace_out("ECchannel_open, bad id", 0);
	return NULL;
    }
    *tmp = 0;			/* kill first @ */
    port_str = ++tmp;
    tmp = strchr(tmp, '@');
    if (tmp == NULL) {
	ECtrace_out("ECchannel_open, bad id2", 0);
	return NULL;
    }
    *tmp = 0;			/* kill second @ */
    chan_str = ++tmp;

    sscanf(port_str, "%d", &port_id);
    if ((strcmp(host_str, DExchange_host_name(de)) == 0) &&
	(port_id == DExchange_inet_port(de))) {
	/* this channel is really local */
	return string_to_channel_ptr(chan_str);
    }
    chan = EChannel_create(de);
    ECtrace_out("getting a connection to %s, registering close handler", 3, 
	      host_str);
    dep = DExchange_get_conn(de, host_str, port_id, 0);
    if (dep == NULL) {
	return NULL;
    }
    DEport_register_close_handler(dep, DEChannel_close_handler);
    cond = DECondition_get(de, dep);
    DECondition_set_client_data(de, cond, &chan);
    memset(&attend_msg, 0, sizeof(attend_msg));
    attend_msg.chan_str = chan_str;
    attend_msg.cond = cond;
    memcpy(&attend_msg.client_channel[0], &chan, sizeof(chan));
    ECtrace_out("creating attend_msg and writing to connection", 3);
    DEport_write_data(dep, DE_Channel_Attend_format_id, &attend_msg);
    if (channel_verbose > 0) {
	printf("Sending attend msg, channel %s cond %d\n",
	       attend_msg.chan_str, attend_msg.cond);
    }
    if (DECondition_wait(de, cond)) {
	ECtrace_out("ECchannel_open channel is ready %lx", 0, chan);
	return chan;
    } else {
	/* wait failed, dep is dead */
	EChannel_destroy(chan);
	ECtrace_out("ECchannel_open, wait failed, channel stillborn", 0);
	return NULL;
    }
}

extern void
ECcancel_sink_subscribe ARGS((ECSinkHandle handle));

static void
send_request_event(chan, i)
EChannel chan;
int i;
{
    RequestEventMsg msg;

    ECtrace_out("send_request_event", 1);
    ECtrace_out("create msg & write to rem_source_list port", 3);

    memset(&msg, 0, sizeof(msg));
    memcpy(msg.channel, chan->rem_member_list[i].channel,
	   sizeof(chan->rem_member_list[i].channel));

    DEport_write_data(chan->rem_source_list[i].dep,
		      DE_Request_Event_format_id, &msg);
    ECtrace_out("send_request_event", 0);
}

static void
send_remote_event(chan, i, event, event_len)
EChannel chan;
int i;
void *event;
int event_len;
{
    EventMsg msg;
    memset(&msg, 0, sizeof(msg));
    msg.event = event;
    msg.event_len = event_len;
    memcpy(msg.channel, chan->rem_sink_list[i].channel, sizeof(msg.channel));
    memcpy(msg.global_chan_id, chan->global_id.channel,
	   sizeof(msg.global_chan_id));
    ECtrace_out("send_remote_event, chan %lx, event %lx", 1, chan, event);
    ECtrace_out("create event msg and write to remote member list port", 3);
    DEport_write_data(chan->rem_sink_list[i].dep, DE_Event_Message_format_id,
		      &msg);
    ECtrace_out("send_remote_event", 0);
}

static void
sendv_remote_event(chan, i, eventV)
EChannel chan;
int i;
IOEncodeVector eventV;
{
    int vec_count = 0;
    EventVMsg msg;
    memset(&msg, 0, sizeof(msg));
    while (eventV[vec_count].iov_base != NULL) {
	vec_count++;
    }
    ECtrace_out("sendv_remote_event, chan %lx, eventv %lx", 1, chan, eventV);
    ECtrace_out("create event msg and write to remote member list port", 3);
    msg.eventv = eventV;
    msg.eventv_len = vec_count;
    memcpy(msg.channel, chan->rem_sink_list[i].channel, sizeof(msg.channel));
    memcpy(msg.global_chan_id, chan->global_id.channel,
	   sizeof(msg.global_chan_id));
    DEport_write_data(chan->rem_sink_list[i].dep, DE_EventV_Message_format_id,
		      &msg);
    ECtrace_out("sendv_remote_event", 0);
}

static void
send_sink_subscribe(chan, i, subscribe)
EChannel chan;
int i;
int subscribe;
{
    SinkSubscribeMsg msg;

    memset(&msg, 0, sizeof(msg));
    memcpy(msg.server_channel, chan->rem_member_list[i].channel,
	   sizeof(chan->rem_member_list[i].channel));
    memcpy(msg.client_channel, &chan, sizeof(chan));
    ECtrace_out("send_sink_subscription", 1);
    ECtrace_out("create msg and write to rem_member_list port", 3);
    if (subscribe) {
	DEport_write_data(chan->rem_member_list[i].dep, 
			  DE_Sink_Subscribe_format_id, &msg);
    } else {
	DEport_write_data(chan->rem_member_list[i].dep, 
			  DE_Sink_Unsubscribe_format_id, &msg);
    }
    ECtrace_out("send_sink_subscription, subscribe = %d", 0, subscribe);
}

static void
send_source_subscribe(chan, i, subscribe)
EChannel chan;
int i;
int subscribe;
{
    SourceSubscribeMsg msg;

    memset(&msg, 0, sizeof(msg));
    memcpy(msg.server_channel, chan->rem_member_list[i].channel,
	   sizeof(chan->rem_member_list[i].channel));
    memcpy(msg.client_channel, &chan, sizeof(chan));
    ECtrace_out("send_source_subscription", 1);
    ECtrace_out("create msg and write to rem_member_list port", 3);
    if (subscribe) {
	DEport_write_data(chan->rem_member_list[i].dep, 
			  DE_Source_Subscribe_format_id, &msg);
    } else {
	DEport_write_data(chan->rem_member_list[i].dep, 
			  DE_Source_Unsubscribe_format_id, &msg);
    }
    ECtrace_out("send_source_subscription, subscribe = %d", 0, subscribe);
}

static void
send_member_subscribe(chan, i)
EChannel chan;
int i;
{
    MemberSubscribeMsg msg;

    ECtrace_out("send_member_subscription", 1);
    ECtrace_out("create msg & write to rem_member_list port", 3);
    memset(&msg, 0, sizeof(msg));
    memcpy(msg.server_channel, chan->rem_member_list[i].channel,
	   sizeof(chan->rem_member_list[i].channel));
    memcpy(msg.client_channel, &chan, sizeof(chan));
    DEport_write_data(chan->rem_member_list[i].dep,
		      DE_Member_Subscribe_format_id, &msg);
    ECtrace_out("send_member_subscription", 0);
}


static void
queue_event_for_dispatch(chan, sink_index, event, event_len, block_rec)
EChannel chan;
int sink_index;
void *event;
int event_len;
free_block_rec_p block_rec;
{
    event_item *item = DEmalloc(sizeof(event_item));
    DExchange de = chan->local_sink_list[sink_index].de;
    IOContext context = chan->local_sink_list[sink_index].context;
    DEFormatList format_list = chan->local_sink_list[sink_index].format_list;
    IOFieldList field_list = chan->local_sink_list[sink_index].field_list;
    int native_struct_size;

    if ((context != NULL) && (event_len != 0)) {
	IOFormat ioformat;
	void *decode_buffer;
	/* decode typed event */
	DExchange_context_lock(de);
	ioformat = get_format_IOcontext(context, event);
	if (ioformat == NULL) {
	    fprintf(stderr, "Unknown IOFormat in encoded block\n");
	}
	ECtrace_out("Queue event got typed event, format \"%s\"", 
		  3, name_of_IOformat(ioformat));
	if (!has_conversion_IOformat(ioformat)) {
	    IOFormat *ioformats;
	    IOFormat *format;
	    if ((ioformats = get_subformats_IOcontext(context,
						      event)) == NULL) {
		fprintf(stderr, "IOFormat list is null\n");
		return;
	    }
	    format = ioformats;
	    while (format_list && (*format != NULL)) {
		int i = 0;
		if (!has_conversion_IOformat(*format)) {
		    while(format_list[i].format_name != NULL) {
			if (strcmp(format_list[i].format_name,
				   name_of_IOformat(*format)) == 0) {
			    break;
			}
			i++;
		    }
		    if (format_list[i].field_list != NULL) {
			native_struct_size = 
			    struct_size_IOfield((IOFile)context, 
						format_list[i].field_list);
			set_conversion_IOcontext(context, *format,
						 format_list[i].field_list,
						 native_struct_size);
		    }
		}
		format++;
	    }
	    free(ioformats);
	    native_struct_size = struct_size_IOfield((IOFile)context, 
						     field_list);
	    set_conversion_IOcontext(context, ioformat,
				     field_list, native_struct_size);
	}
	decode_buffer = DEmalloc(this_IOrecord_length(context, event,event_len));
	decode_to_buffer_IOcontext(context, event, decode_buffer);
	DExchange_context_unlock(de);
	block_rec = DEmalloc(sizeof(*block_rec));
	block_rec->ref_count = 0;
	block_rec->de = NULL;
	block_rec->block = decode_buffer;
	block_rec->free_func = (EventFreeFunction) NULL;
	block_rec->locking_de = chan->de;
	event = decode_buffer;
	event_len = 0;
    }
	
    item->event = event;
    item->event_len = event_len;
    item->eventv = NULL;
    item->sink_index = sink_index;
    item->chan = chan;
    if (block_rec != NULL)
	block_rec->ref_count++;
    item->block_rec = block_rec;
    ECtrace_out("Queue event for local dispatch, chan %lx, event %lx", 3, chan,
	      event);

    item->next = NULL;
    item->prev = de->control_list->event_queue_tail;
    if (item->prev) {
	item->prev->next = item;
    }
    de->control_list->event_queue_tail = item;
    if (de->control_list->event_queue_head == NULL) {
	de->control_list->event_queue_head = item;
    }
    if (chan->de != de) {
	/* if we're putting this in some other DE's queue, wake any server */
	wake_DE_server(de);
    }
}


static void
queue_eventv_for_dispatch(chan, sink_index, eventv, block_rec)
EChannel chan;
int sink_index;
IOEncodeVector eventv;
free_block_rec_p block_rec;
{
    event_item *item = DEmalloc(sizeof(event_item));
    DExchange de = chan->local_sink_list[sink_index].de;
    item->event = NULL;
    item->event_len = 0;
    item->eventv = eventv;
    item->sink_index = sink_index;
    item->chan = chan;
    block_rec->ref_count++;
    item->block_rec = block_rec;

    item->next = NULL;
    item->prev = de->control_list->event_queue_tail;
    de->control_list->event_queue_tail = item;
    if (de->control_list->event_queue_head == NULL) {
	de->control_list->event_queue_head = item;
    }
    if (channel_verbose > 0)
	printf("queuev for dispatch\n");
}

static event_queue
dequeue_event(control_list)
DEControlList control_list;
{
    event_queue item;

    if (control_list->event_queue_head == NULL) {
	return NULL;
    } else {
	item = control_list->event_queue_head;
	control_list->event_queue_head = item->next;
    }
    if (item == control_list->event_queue_tail) {
	control_list->event_queue_tail = NULL;
    }
    item->prev = NULL;
    item->next = NULL;
    return item;
}

static void
dispatch_event(func, event, event_len, client_data)
ECHandlerFunction func;
void *event;
int event_len;
void *client_data;
{
    func(event, event_len, client_data);
}

static void
dispatch_typed_event(func, event, client_data)
ECHandlerFunction func;
void *event;
void *client_data;
{
    ((ECTypedHandlerFunction)func)(event, client_data);
}

static void
do_event_dispatch(control_list)
DEControlList control_list;
{
    event_queue item;
    DEControlList_lock(control_list);
    while ((item = dequeue_event(control_list)) != NULL) {
	EChannel chan = item->chan;
	ECtrace_out("dequeued event", 3);
	if (chan->local_sink_list[item->sink_index].func !=
	    (ECHandlerFunction) NULL) {
	    int sink = item->sink_index;
	    DEControlList_unlock(control_list);
	    if (item->eventv != NULL) {
		int i = 0;
		char *tmp;
		char *event;
		int event_len = 0;

		IOEncodeVector eventV = item->eventv;
		while (eventV[i].iov_base != NULL) {
		    event_len += eventV[i++].iov_len;
		}
		tmp = event = DEmalloc(event_len);
		i = 0;
		while (eventV[i].iov_base != NULL) {
		    memcpy(tmp, eventV[i].iov_base, eventV[i].iov_len);
		    tmp += eventV[i++].iov_len;
		}
		dispatch_event(chan->local_sink_list[sink].func,
			       event, event_len,
			       chan->local_sink_list[sink].client_data);

		EChannel_lock(chan);
		if (chan->local_sink_list[sink].cond) {
		    DECondition_signal(chan->de,
				       chan->local_sink_list[sink].cond);
		}
		chan->local_sink_list[sink].cond = 0;
		EChannel_unlock(chan);
		DEfree(event);
	    } else {
		if (chan->local_sink_list[item->sink_index].context == NULL) {
		    /* untyped event */
		    dispatch_event(chan->local_sink_list[sink].func,
				   item->event, item->event_len,
				   chan->local_sink_list[sink].client_data);
		} else {
		    /* typed event */
		    dispatch_typed_event(chan->local_sink_list[sink].func,
					 item->event, 
					 chan->local_sink_list[sink].client_data);
		}

		EChannel_lock(chan);
		if (chan->local_sink_list[sink].cond) {
		    DECondition_signal(chan->de, 
				       chan->local_sink_list[sink].cond);
		}
		chan->local_sink_list[sink].cond = 0;
		EChannel_unlock(chan);
	    }
	    DEControlList_lock(control_list);
	}
	if (item->block_rec) {
	    deref_block(item->block_rec);
	}
	DEfree(item);
    }
    DEControlList_unlock(control_list);
}

static void
do_event_flush(control_list)
DEControlList control_list;
{
    event_queue item;
    DEControlList_lock(control_list);
    while ((item = dequeue_event(control_list)) != NULL) {
	if (channel_verbose > 0)
	    printf("flushing event\n");
	if (item->block_rec) {
	    deref_block(item->block_rec);
	}
	DEfree(item);
    }
    DEControlList_unlock(control_list);
}

extern void
EChannel_subscribe_handler(chan, func, client_data)
EChannel chan;
ECSubscribeFunction func;
void *client_data;
{
    ECtrace_out("EChannel_subscribe_handler", 1);
    chan->subscribe_func = func;
    chan->subscribe_client_data = client_data;
    ECtrace_out("EChannel_subscribe_handler", 0);
}

extern int
DEChannel_RequestEvent_handler(de, dep, event_format_id, data, data_length,
			      client_data)
DExchange de;
DEPort dep;
int event_format_id;
void *data;
int data_length;
void *client_data;
{
    int i = 0;
    EChannel chan;
    RequestEventMsg *msg = (RequestEventMsg *) data;
    DEtake_buffer(de, data);

    if (channel_verbose > 0)
	printf("In request event handler\n");
    memcpy(&chan, &msg->channel, sizeof(chan));
    EChannel_lock(chan);

    for (i = 0; i < chan->local_source_list_len; i++) {
	if (chan->local_source_list[i].func != 
	    (ECRequestHandlerFunction) NULL) {
	    chan->local_source_list[i].func(chan->local_source_list[i].source_handle,
					    chan->local_source_list[i].client_data);
	}
    }

    EChannel_unlock(chan);
    return 0;
}

extern int
DEChannel_Event_handler(de, dep, event_format_id, data, data_length,
			client_data)
DExchange de;
DEPort dep;
int event_format_id;
void *data;
int data_length;
void *client_data;
{
    int i = 0;
    EChannel chan;
    EventMsg *msg = (EventMsg *) data;
    free_block_rec_p block_rec = DEmalloc(sizeof(struct free_block_rec));
    block_rec->ref_count = 1;
    block_rec->de = de;
    block_rec->block = data;
    block_rec->free_func = (EventFreeFunction) NULL;
    DEtake_buffer(de, data);

    if (channel_verbose > 0)
	printf("In event handler\n");
    memcpy(&chan, &msg->channel, sizeof(chan));
    block_rec->locking_de = chan->de;
    EChannel_lock(chan);
    for (i = 0; i < chan->local_sink_list_len; i++) {
	if (chan->local_sink_list[i].func != (ECHandlerFunction) NULL) {
	    queue_event_for_dispatch(chan, i, msg->event, msg->event_len,
				     block_rec);
	}
    }
    deref_block(block_rec);
    EChannel_unlock(chan);
    return 0;
}

extern int
DEChannel_EventV_handler(de, dep, event_format_id, data, data_length,
			 client_data)
DExchange de;
DEPort dep;
int event_format_id;
void *data;
int data_length;
void *client_data;
{
    EventVMsg *msg = (EventVMsg *) data;
    int i, event_len = 0;
    EChannel chan;
    free_block_rec_p block_rec = DEmalloc(sizeof(struct free_block_rec));
    block_rec->ref_count = 1;
    block_rec->de = de;
    block_rec->block = data;
    block_rec->free_func = (EventFreeFunction) NULL;
    DEtake_buffer(de, data);

    if (channel_verbose > 0)
	printf("In eventv handler\n");
    for (i = 0; i < msg->eventv_len - 1; i++) {
	/* assure ourselves that the message is contiguous */
	assert((char *) msg->eventv[i].iov_base + msg->eventv[i].iov_len ==
	       (char *) msg->eventv[i + 1].iov_base);
	event_len += msg->eventv[i].iov_len;
    }
    event_len += msg->eventv[msg->eventv_len - 1].iov_len;
    memcpy(&chan, &msg->channel, sizeof(chan));
    block_rec->locking_de = chan->de;
    EChannel_lock(chan);
    for (i = 0; i < chan->local_sink_list_len; i++) {
	if (chan->local_sink_list[i].func != (ECHandlerFunction) NULL) {
	    queue_event_for_dispatch(chan, i, msg->eventv[0].iov_base,
				     event_len, block_rec);
	}
    }
    deref_block(block_rec);
    EChannel_unlock(chan);
    return 0;
}

extern int
DEChannel_Attend_handler(de, dep, attend_format_id,
			 data, data_length, client_data)
DExchange de;
DEPort dep;
int attend_format_id;
void *data;
int data_length;
void *client_data;
{
    AttendMsgPtr msg = (AttendMsgPtr) data;
    AttendResponseMsg ret_msg;
    ExistsAttendResponseMsg exists_msg;
    EChannel chan, der_chan;
    int i;

    ECtrace_out("DEChannel_Attend_handler", 1);
    DEport_register_close_handler(dep, DEChannel_close_handler);

    chan = string_to_channel_ptr(msg->chan_str);

    if (channel_verbose > 0) {
	printf("In attend handler, channel %s (%lx) cond %d\n",
	       msg->chan_str, (long) chan, msg->cond);
    }
    /* check to see that channel is valid */

    ECtrace_out("Checking for already existing remote channel chan %lx", 3, 
	      chan);
    for (i = 0; i < chan->rem_member_list_len; i++) {
	if (chan->rem_member_list[i].dep != NULL) {
	    if (chan->rem_member_list[i].dep == dep) {
		ECtrace_out("Found existing channel, notify attender", 3);
		exists_msg.condition = msg->cond;
		memcpy(&exists_msg.channel[0], 
		       &chan->rem_member_list[i].channel[0],
		       sizeof(exists_msg.channel));
		ECtrace_out("writing exists response name", 3);
		DEport_write_data(dep, DE_Channel_Exists_Attend_Response_format_id, &exists_msg);
		ECtrace_out("DEChannel_Attend_handler", 0);
		return 0;
	    }
	}
    }

    /* send him the sink list (add local) */
    ECtrace_out("send him sink list, local and remote", 3);
    memset(&ret_msg, 0, sizeof(ret_msg));
    ret_msg.condition = msg->cond;
    ret_msg.type_name = NULL;
    if (chan->type_format) {
	ret_msg.type_name = global_name_of_IOformat(chan->type_format);
	ECtrace_out("type of format is %s", 3, ret_msg.type_name);
    }
    ret_msg.sink_count = 0;
    ret_msg.sinks = DEmalloc(sizeof(AttendRec));
    if (chan->local_sink_list_len != 0) {
	ret_msg.sinks[ret_msg.sink_count].host = chan->global_id.home_host;
	ret_msg.sinks[ret_msg.sink_count].IPport = chan->global_id.IPport;
	ret_msg.sinks[ret_msg.sink_count].filter = NULL;
	memset(ret_msg.sinks[ret_msg.sink_count].channel, 0,
	       sizeof(ret_msg.sinks[0].channel));
	memcpy(ret_msg.sinks[ret_msg.sink_count].channel, &chan, sizeof(chan));
	ret_msg.sink_count++;
    }



    for (i = 0; i < chan->rem_sink_list_len; i++) {
	if (chan->rem_sink_list[i].dep != NULL) {
	    ret_msg.sinks = DErealloc(ret_msg.sinks,
			   (ret_msg.sink_count + 1) * sizeof(AttendRec));

	    ret_msg.sinks[ret_msg.sink_count].host =
		DEport_host_name(chan->rem_sink_list[i].dep);
	    ret_msg.sinks[ret_msg.sink_count].IPport =
		DEport_port_number(chan->rem_sink_list[i].dep);
	    ret_msg.sinks[ret_msg.sink_count].filter = NULL;
	    memset(ret_msg.sinks[ret_msg.sink_count].channel, 0,
		   sizeof(ret_msg.sinks[0].channel));
	    memcpy(ret_msg.sinks[ret_msg.sink_count].channel,
		   chan->rem_sink_list[i].channel,
		   sizeof(chan->rem_sink_list[0].channel));
	    ret_msg.sink_count++;
	}
    }

    ECtrace_out("send him source list, local and remote", 3);

    ret_msg.source_count = 0;
    ret_msg.sources = DEmalloc(sizeof(AttendRec));
    if (chan->local_source_list_len != 0) {
	ret_msg.sources[ret_msg.source_count].host = chan->global_id.home_host;
	ret_msg.sources[ret_msg.source_count].IPport = chan->global_id.IPport;
	ret_msg.sources[ret_msg.source_count].filter = NULL;
	memset(ret_msg.sources[ret_msg.source_count].channel, 0,
	       sizeof(ret_msg.sources[0].channel));
	memcpy(ret_msg.sources[ret_msg.source_count].channel, &chan, 
	       sizeof(chan));
	ret_msg.source_count++;
    }
    for (i = 0; i < chan->rem_source_list_len; i++) {
	if (chan->rem_source_list[i].dep != NULL) {
	    ret_msg.sources = DErealloc(ret_msg.sources,
					(ret_msg.source_count + 1) * sizeof(AttendRec));

	    ret_msg.sources[ret_msg.source_count].host =
		DEport_host_name(chan->rem_source_list[i].dep);
	    ret_msg.sources[ret_msg.source_count].IPport =
		DEport_port_number(chan->rem_source_list[i].dep);
	    ret_msg.sources[ret_msg.source_count].filter = NULL;
	    memset(ret_msg.sources[ret_msg.source_count].channel, 0,
		   sizeof(ret_msg.sources[0].channel));
	    memcpy(ret_msg.sources[ret_msg.source_count].channel,
		   chan->rem_source_list[i].channel,
		   sizeof(chan->rem_source_list[0].channel));
	    ret_msg.source_count++;
	}
    }

    ECtrace_out("send him the remote list", 3);

    ret_msg.member_count = 1;
    ret_msg.members = DEmalloc(sizeof(AttendRec));
    ret_msg.members[0].host = chan->global_id.home_host;
    ret_msg.members[0].IPport = chan->global_id.IPport;
    memset(ret_msg.members[0].channel, 0,
	   sizeof(ret_msg.members[0].channel));
    ret_msg.members[0].filter = NULL;
    memcpy(ret_msg.members[0].channel, &chan, sizeof(chan));
    for (i = 0; i < chan->rem_member_list_len; i++) {
	if (chan->rem_member_list[i].dep != NULL) {
	    ret_msg.members = DErealloc(ret_msg.members,
			 (ret_msg.member_count + 1) * sizeof(AttendRec));
	    ret_msg.members[ret_msg.member_count].host =
		DEport_host_name(chan->rem_member_list[i].dep);
	    ret_msg.members[ret_msg.member_count].IPport =
		DEport_port_number(chan->rem_member_list[i].dep);
	    memset(ret_msg.members[ret_msg.member_count].channel, 0,
		   sizeof(ret_msg.members[0].channel));
	    ret_msg.members[ret_msg.member_count].filter = NULL;
	    memcpy(ret_msg.members[ret_msg.member_count].channel,
		   chan->rem_member_list[i].channel,
		   sizeof(chan));
	    ret_msg.member_count++;
	}
    }

    ECtrace_out("Sending list: \n Remote sink count %d Remote source count %d Remote member count %d",
		3, ret_msg.sink_count, ret_msg.source_count,
		ret_msg.member_count);
    
    ECtrace_out("sending derivitive channel IDs", 3);
    ret_msg.der_count = 0;
    for (i = 0; i < chan->der_list_len; i++) {
	der_chan = chan->der_chan_list[i].derived_chan;
	if (ret_msg.der_count == 0)
	    ret_msg.derivatives = DEmalloc(sizeof(AttendRec));
	else
	    ret_msg.derivatives = DErealloc(ret_msg.derivatives,
			    (ret_msg.der_count + 1) * sizeof(AttendRec));
	ret_msg.derivatives[ret_msg.der_count].host = chan->global_id.home_host;
	ret_msg.derivatives[ret_msg.der_count].IPport = chan->global_id.IPport;
	memset(ret_msg.derivatives[ret_msg.der_count].channel, 0,
	       sizeof(ret_msg.derivatives[ret_msg.der_count].channel));
	memcpy(ret_msg.derivatives[ret_msg.der_count].channel,
	       &der_chan, sizeof(der_chan));
	ret_msg.derivatives[ret_msg.der_count].filter = 
	    chan->der_chan_list[i].filter;
	ret_msg.derivatives[ret_msg.der_count].filter_has_output_param =
	    chan->der_chan_list[i].has_output_param;
	ret_msg.der_count++;
    }

    ECtrace_out("writing data", 3);
    DEport_write_data(dep, DE_Channel_Attend_Response_format_id, &ret_msg);
    DEfree(ret_msg.sinks);
    DEfree(ret_msg.sources);

    DEfree(ret_msg.members);
    free(ret_msg.type_name);
    ECtrace_out("adding new member to remote member list", 3);
    if (chan->rem_member_list == NULL) {
	chan->rem_member_list = DEmalloc(sizeof(struct _rem_attendr));
    }
    chan->rem_member_list =
	DErealloc(chan->rem_member_list,
	  sizeof(struct _rem_attendr) * (chan->rem_member_list_len + 1));
    chan->rem_member_list[chan->rem_member_list_len].dep = dep;
    memcpy(chan->rem_member_list[chan->rem_member_list_len].channel,
	   &msg->client_channel[0], 8);

    chan->rem_member_list_len++;
    if (channel_verbose > 0) {
	printf("\n\nDEChannel_Attend_handler\n");
	dump_EChannel(chan);
    }
    ECtrace_out("DEChannel_Attend_handler", 0);
    return 0;
}

extern int
DEChannel_SinkSubscribe_handler(de, dep, sink_subscribe_format_id,
				data, data_length, client_data)
DExchange de;
DEPort dep;
int sink_subscribe_format_id;
void *data;
int data_length;
void *client_data;
{
    SinkSubscribeMsgPtr msg = (SinkSubscribeMsgPtr) data;
    EChannel chan;

    memcpy(&chan, &msg->server_channel[0], sizeof(chan));
    ECtrace_out("DEChannel_SinkSubscribe_handler, chan %lx", 1, chan);

    /* check to see that channel is valid */

    ECtrace_out("add him to the sink list", 3);
    if (channel_verbose > 0)
	printf("adding sink to chan %lx\n", (long) chan);

    chan->rem_sink_list =
	DErealloc(chan->rem_sink_list,
	    sizeof(struct _rem_attendr) * (chan->rem_sink_list_len + 1));
    chan->rem_sink_list[chan->rem_sink_list_len].dep = dep;
    memcpy(chan->rem_sink_list[chan->rem_sink_list_len].channel,
	   msg->client_channel, 8);
    chan->rem_sink_list_len++;
    if (chan->subscribe_func) {
	chan->subscribe_func(1, chan->rem_sink_list_len,
			     chan->subscribe_client_data);
    }
    if (channel_verbose) {
	printf("\n\nDEChannel_SinkSubscribe_handler\n");
	dump_EChannel(chan);
    }
    ECtrace_out("DEChannel_SinkSubscribe_handler", 0);

    return 0;
}

extern int
DEChannel_SinkUnsubscribe_handler(de, dep, sink_subscribe_format_id,
				data, data_length, client_data)
DExchange de;
DEPort dep;
int sink_subscribe_format_id;
void *data;
int data_length;
void *client_data;
{
    SinkSubscribeMsgPtr msg = (SinkSubscribeMsgPtr) data;
    EChannel chan;
    int i;

    memcpy(&chan, &msg->server_channel[0], sizeof(chan));
    ECtrace_out("DEChannel_SinkUnsubscribe_handler, chan %lx", 1, chan);

    /* check to see that channel is valid */

    ECtrace_out("remove him from the sink list", 3);
    if (channel_verbose > 0)
	printf("removing sink from chan %lx\n", (long) chan);

    for (i=0; i< chan->rem_sink_list_len; i++) {
	if (memcmp(chan->rem_sink_list[i].channel,
		   msg->client_channel, 8) == 0) {
	    chan->rem_sink_list[i].dep = NULL;
	    if (i == (chan->rem_sink_list_len -1)) {
		chan->rem_sink_list_len--;
	    }
	}
    }
    if (channel_verbose) {
	printf("\n\nDEChannel_SinkUnsubscribe_handler\n");
	dump_EChannel(chan);
    }
    ECtrace_out("DEChannel_SinkUnsubscribe_handler", 0);

    return 0;
}

extern int
DEChannel_SourceSubscribe_handler(de, dep, source_subscribe_format_id,
				data, data_length, client_data)
DExchange de;
DEPort dep;
int source_subscribe_format_id;
void *data;
int data_length;
void *client_data;
{
    SourceSubscribeMsgPtr msg = (SourceSubscribeMsgPtr) data;
    EChannel chan;

    memcpy(&chan, &msg->server_channel[0], sizeof(chan));
    ECtrace_out("DEChannel_SourceSubscribe_handler, chan %lx", 1, chan);

    /* check to see that channel is valid */

    ECtrace_out("add source to chan %lx source list", 3, (long) chan);

    chan->rem_source_list =
	DErealloc(chan->rem_source_list,
		  sizeof(struct _rem_attendr) * (chan->rem_source_list_len + 1));
    chan->rem_source_list[chan->rem_source_list_len].dep = dep;
    memcpy(chan->rem_source_list[chan->rem_source_list_len].channel,
	   msg->client_channel, 8);
    chan->rem_source_list_len++;

    if (channel_verbose) {
	printf("\n\nDEChannel_SourceSubscribe_handler\n");
	dump_EChannel(chan);
    }
    ECtrace_out("DEChannel_SourceSubscribe_handler", 0);

    return 0;
}

extern int
DEChannel_SourceUnsubscribe_handler(de, dep, source_subscribe_format_id,
				  data, data_length, client_data)
DExchange de;
DEPort dep;
int source_subscribe_format_id;
void *data;
int data_length;
void *client_data;
{
    SourceSubscribeMsgPtr msg = (SourceSubscribeMsgPtr) data;
    EChannel chan;
    int i;

    memcpy(&chan, &msg->server_channel[0], sizeof(chan));
    ECtrace_out("DEChannel_SourceUnsubscribe_handler, chan %lx", 1, chan);

    /* check to see that channel is valid */

    ECtrace_out("remove him from the source list", 3);
    if (channel_verbose > 0)
	printf("removing source from chan %lx\n", (long) chan);

    for (i=0; i< chan->rem_source_list_len; i++) {
	if (memcmp(chan->rem_source_list[i].channel,
		   msg->client_channel, 8) == 0) {
	    chan->rem_source_list[i].dep = NULL;
	    if (i == (chan->rem_source_list_len -1)) {
		chan->rem_source_list_len--;
	    }
	}
    }
    if (channel_verbose) {
	printf("\n\nDEChannel_SourceUnsubscribe_handler\n");
	dump_EChannel(chan);
    }
    ECtrace_out("DEChannel_SourceUnsubscribe_handler", 0);

    return 0;
}

extern int
DEChannel_MemberSubscribe_handler(de, dep, member_subscribe_format_id,
				  data, data_length, client_data)
DExchange de;
DEPort dep;
int member_subscribe_format_id;
void *data;
int data_length;
void *client_data;
{
    MemberSubscribeMsgPtr msg = (MemberSubscribeMsgPtr) data;
    EChannel chan;


    ECtrace_out("DEChannel_MemberSubscribe_handler", 1);
    DEport_register_close_handler(dep, DEChannel_close_handler);
    /* check to see that channel is valid */
    memcpy(&chan, &msg->server_channel[0], sizeof(chan));

    /* add him to the the member list */
    ECtrace_out("add him to the the member list", 3);
    chan->rem_member_list =
	DErealloc(chan->rem_member_list,
	  sizeof(struct _rem_attendr) * (chan->rem_member_list_len + 1));
    chan->rem_member_list[chan->rem_member_list_len].dep = dep;
    memcpy(chan->rem_member_list[chan->rem_member_list_len].channel,
	   msg->client_channel, 8);
    chan->rem_member_list_len++;

    if (channel_verbose) {
	printf("\n\nDEChannel_MemberSubscribe_handler\n");
	dump_EChannel(chan);
    }
    ECtrace_out("DEChannel_MemberSubscribe_handler", 0);
    return 0;
}

extern
void
DEChannel_close_handler(de, dep)
DExchange de;
DEPort dep;
{
    int i;

    ECtrace_out("DEChannel_close_handler, de->channel_count = %d, dep %lx", 1, de->channel_count, dep);
    for (i = 0; i < de->channel_count; i++) {
	EChannel chan = de->channel_list[i];
	int j;
	int deleted = 0;
	ECtrace_out("removing him from remote member list", 3);
	for (j = 0; j < chan->rem_member_list_len; j++) {
	    if (chan->rem_member_list[j].dep == dep) {
		chan->rem_member_list[j].dep = NULL;
	    }
	}
	for (j = 0; (j + deleted) < chan->rem_member_list_len; j++) {
	    if (chan->rem_member_list[j].dep == NULL) {
		deleted++;
	    }
	    if ((deleted != 0) &&
		(j + deleted != chan->rem_member_list_len)) {
		chan->rem_member_list[j] =
		    chan->rem_member_list[j + deleted];
	    }
	}
	chan->rem_member_list_len -= deleted;

	deleted = 0;
	ECtrace_out("removing him from remote sink list", 3);
	for (j = 0; j < chan->rem_sink_list_len; j++) {
	    if (chan->rem_sink_list[j].dep == dep) {
		chan->rem_sink_list[j].dep = NULL;
	    }
	}
	for (j = 0; (j + deleted) < chan->rem_sink_list_len; j++) {
	    if (chan->rem_sink_list[j].dep == NULL) {
		deleted++;
	    }
	    if ((deleted != 0) &&
		(j + deleted != chan->rem_sink_list_len)) {
		chan->rem_sink_list[j] = chan->rem_sink_list[j + deleted];
	    }
	}
	chan->rem_sink_list_len -= deleted;
	ECtrace_out("Remote sinks deleted %d", 3, deleted);

	deleted = 0;
	ECtrace_out("removing him from remote sources list", 3);
	for (j = 0; j < chan->rem_source_list_len; j++) {
	    if (chan->rem_source_list[j].dep == dep) {
		chan->rem_source_list[j].dep = NULL;
	    }
	}
	for (j = 0; (j + deleted) < chan->rem_source_list_len; j++) {
	    if (chan->rem_source_list[j].dep == NULL) {
		deleted++;
	    }
	    if ((deleted != 0) &&
		(j + deleted != chan->rem_source_list_len)) {
		chan->rem_source_list[j] =
		    chan->rem_source_list[j + deleted];
	    }
	}
	chan->rem_source_list_len -= deleted;

	ECtrace_out("Remote sources deleted %d", 3, deleted);

    }
    ECtrace_out("DEChannel_close_handler", 0);
}


extern int
DEChannel_Exists_Attend_Response_handler(de, dep, 
					 attend_exists_response_format_id,
					 data, data_length, client_data)
DExchange de;
DEPort dep;
int attend_exists_response_format_id;
void *data;
int data_length;
void *client_data;
{
    ExistsAttendResponseMsgPtr msg = (ExistsAttendResponseMsgPtr) data;
    EChannel chan, old_chan, *chan_ptr;

    ECtrace_out("DEChannel_Exists_Attend_Response_handler", 1);
    if (channel_verbose > 0) {
	printf("In exists attend handler response, cond %d\n", msg->condition);
    }

    DEport_register_close_handler(dep, DEChannel_close_handler);
    chan_ptr = DECondition_get_client_data(de, msg->condition);
    old_chan = *chan_ptr;

    ECtrace_out("DEChannel_exists_Attend_Response_handler, destroying old chan %lx", 
	      3, old_chan);
    EChannel_destroy(old_chan);

    /* 
     * change the channel variable in the waiting EChannel_open() to return
     * the already-existing channel instead.
     */
    memcpy(&chan, msg->channel, sizeof(chan));
    *chan_ptr = chan;
    DECondition_signal(de, msg->condition);
    ECtrace_out("DEChannel_Exists_Attend_Response_handler", 0);
    return 0;
}


extern int
DEChannel_Attend_Response_handler(de, dep, attend_response_format_id,
				  data, data_length, client_data)
DExchange de;
DEPort dep;
int attend_response_format_id;
void *data;
int data_length;
void *client_data;
{
    AttendResponseMsgPtr msg = (AttendResponseMsgPtr) data;
    EChannel chan, der_chan, *chan_ptr;
    int i;
    char port_str[20] =
    {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    char chan_str[17] =
    {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    char *global_id_str;

    ECtrace_out("DEChannel_Attend_Response_handler", 1);
    if (channel_verbose > 0) {
	printf("In attend handler response, cond %d\n", msg->condition);
    }
    /* preserve the message... */
    DEtake_buffer(de, data);

    DEport_register_close_handler(dep, DEChannel_close_handler);
    chan_ptr = DECondition_get_client_data(de, msg->condition);
    chan = *chan_ptr;

    ECtrace_out("DEChannel_Attend_Response_handler, chan %lx, type %x", 3, chan,
	      msg->type_name);
    if (msg->type_name != NULL) {
	DExchange_context_lock(chan->de);
	if (chan->de->context == NULL) {
	    /* create IO context if it does not already exist */
	    de->context = create_IOcontext(NULL);
	}
	chan->type_format = get_IOformat_by_name_IOcontext(chan->de->context,
							   msg->type_name);
	DExchange_context_unlock(chan->de);
    }

    ECtrace_out("Building lists: \n Remote sink count %d Remote source count %d Remote member count %d\n",
		3, msg->sink_count, msg->source_count, msg->member_count);
    
    /* build remote sink list */

    chan->rem_sink_list =
	DErealloc(chan->rem_sink_list,
		  sizeof(struct _rem_attendr) * msg->sink_count);
    chan->rem_sink_list_len = 0;

    ECtrace_out("connect and register close handler for remote sinks", 3);
    for (i = 0; i < msg->sink_count; i++) {
	int j = chan->rem_sink_list_len;
	if ((strcmp(msg->sinks[i].host, DExchange_host_name(de)) == 0) &&
	    (msg->sinks[i].IPport == DExchange_inet_port(de))) {
	    continue;
	}
	chan->rem_sink_list[j].dep =
	    DExchange_get_conn(de, msg->sinks[i].host,
			       msg->sinks[i].IPport, FALSE);
	if (chan->rem_sink_list[j].dep == NULL) {
	    printf("Failed to contact %s, %d\n", msg->sinks[i].host,
		   msg->sinks[i].IPport);
	    continue;
	}
	DEport_register_close_handler(chan->rem_sink_list[j].dep,
				      DEChannel_close_handler);
	memcpy(chan->rem_sink_list[j].channel, msg->sinks[i].channel, 8);
	chan->rem_sink_list_len++;
    }

    /* build remote source list */

    chan->rem_source_list =
	DErealloc(chan->rem_source_list,
		  sizeof(struct _rem_attendr) * msg->source_count);
    chan->rem_source_list_len = 0;

    ECtrace_out("connect and register close handler for remote sources", 3);
    for (i = 0; i < msg->source_count; i++) {
	int j = chan->rem_source_list_len;
	if ((strcmp(msg->sources[i].host, DExchange_host_name(de)) == 0) &&
	    (msg->sources[i].IPport == DExchange_inet_port(de))) {
	    continue;
	}
	chan->rem_source_list[j].dep =
	    DExchange_get_conn(de, msg->sources[i].host,
			       msg->sources[i].IPport, FALSE);
	if (chan->rem_source_list[j].dep == NULL) {
	    printf("Failed to contact %s, %d\n", msg->sources[i].host,
		   msg->sources[i].IPport);
	    continue;
	}
	DEport_register_close_handler(chan->rem_source_list[j].dep,
				      DEChannel_close_handler);
	memcpy(chan->rem_source_list[j].channel, msg->sources[i].channel, 8);
	chan->rem_source_list_len++;
    }

    /* build remote member list */

    chan->rem_member_list =
	DErealloc(chan->rem_member_list,
		  sizeof(struct _rem_attendr) * msg->member_count);
    chan->rem_member_list_len = 0;
    for (i = 0; i < msg->member_count; i++) {
	int j = chan->rem_member_list_len;
	if ((strcmp(msg->members[i].host, DExchange_host_name(de)) == 0) &&
	    (msg->members[i].IPport == DExchange_inet_port(de))) {
	    continue;
	}
	ECtrace_out("connect and register close handler for remote members", 3);
	chan->rem_member_list[j].dep =
	    DExchange_get_conn(de, msg->members[i].host,
			       msg->members[i].IPport, FALSE);
	if (chan->rem_member_list[j].dep == NULL) {
	    printf("Failed to contact %s, %d\n", msg->members[i].host,
		   msg->members[i].IPport);
	    continue;
	}
	DEport_register_close_handler(chan->rem_member_list[j].dep,
				      DEChannel_close_handler);
	memcpy(chan->rem_member_list[i].channel, msg->members[i].channel, 8);
	if (i != 0) {
	    /* first entry is channel we got this from */
	    send_member_subscribe(chan, j);
	}
	chan->rem_member_list_len++;
    }
    for (i = 0; i < msg->der_count; i++) {
	char *filter;
	int len = strlen(msg->derivatives[i].host);
	ECtrace_out("open, subscribe, and add derived channel to der_list", 3);

	sprintf(port_str, "%d", msg->derivatives[i].IPport);
	sprintf(chan_str, "%02x%02x%02x%02x%02x%02x%02x%02x",
		msg->derivatives[i].channel[0],
		msg->derivatives[i].channel[1],
		msg->derivatives[i].channel[2],
		msg->derivatives[i].channel[3],
		msg->derivatives[i].channel[4],
		msg->derivatives[i].channel[5],
		msg->derivatives[i].channel[6],
		msg->derivatives[i].channel[7]);

	len += strlen(port_str);
	len += 16 /* chan_str */  + 2 /* separators */ ;
	global_id_str = DEmalloc(len + 1);
	sprintf(global_id_str, "%s@%d@%s", msg->derivatives[i].host,
		msg->derivatives[i].IPport, chan_str);

	der_chan = EChannel_open(de, global_id_str);

	chan->der_chan_list = DErealloc(chan->der_chan_list,
			    (chan->der_list_len + 1) * sizeof(struct _derived_chan));
	chan->der_chan_list[chan->der_list_len].derived_chan = der_chan;
	filter = DEmalloc(strlen(msg->derivatives[i].filter) + 2);
	strcpy(filter, msg->derivatives[i].filter);
	filter[strlen(filter) + 1] = 0; /* add extra NULL termination */
	chan->der_chan_list[chan->der_list_len].filter = filter;
	chan->der_chan_list[chan->der_list_len].has_output_param =
	    msg->derivatives[i].filter_has_output_param;
	chan->der_list_len++;
    }

    if (channel_verbose > 0) {
	printf("\n\nDEChannel_Attend_Response_handler\n");
	dump_EChannel(chan);
    }
    ECtrace_out("DEChannel_Attend_Response_handler", 0);
    chan->ready++;
    DECondition_signal(de, msg->condition);
    DEreturn_buffer(de, data);
    return 0;
}

extern
void
DEControlList_event_dispatch(control_list)
DEControlList control_list;
{
    if (control_list->event_queue_head != NULL) {
	do_event_dispatch(control_list);
    }
}

extern
void
DEControlList_event_flush(control_list)
DEControlList control_list;
{
    if (control_list->event_queue_head != NULL) {
	do_event_flush(control_list);
    }
}

extern EChannel
DEchannel_initiate_first(de, groups)
DExchange de;
comm_group_return groups;
{
    EChannel chan = NULL;
    int i;
    for (i = 0; i < groups->count; i++) {
	if (groups->list[i].info_len != 0) {
	    chan = EChannel_open(de, (char *) groups->list[i].info);
	    if (chan != NULL) {
		return chan;
	    }
	}
    }
    return NULL;
}
