
#include "OODE.h"

#ifndef FALSE
#define FALSE (0)
#endif
#ifndef TRUE
#define TRUE (!FALSE)
#endif
#ifndef NULL
#define NULL ((void*)0)
#endif

extern "C"
{
#include "DE.h"
#include "comm_group.h"
}

#include <stdlib.h>

//#############################################################################
//##                                                         ERROR DECLARATIONS

class OODEError
{
public:

  typedef enum { IGNORE, STOP_SIGNAL, ABORT } ACTION;

  static void POST( const char* const error, const char* const cause,
		    const char* const file, const int line,
		    const ACTION action = IGNORE,
		    const char* const next = NULL );
};

//-----------------------------------------------------------------------------

#define OODE_ERROR_CLASS_DECLARATION( name, action )			\
class OODEError ## name							\
{									\
public:									\
  static void POST( const char* const cause,				\
		    const char* const file, const int line,		\
		    const char* const next = NULL ) {			\
    OODEError::POST( #name, cause,					\
		      file, line,					\
		      OODEError:: ## action, next );			\
  }									\
  static int WHEN( const int condition, const char* const cause,	\
		   const char* const file, const int line,		\
		   const char* const next = NULL ) {			\
    if ( condition )							\
      { POST( cause, file, line, next ); return TRUE; }			\
    else return FALSE;							\
  }									\
  static int UNLESS( const int condition, const char* const cause,	\
		     const char* const file, const int line,		\
		     const char* const next = NULL ) {			\
    if ( ! condition )							\
      { POST( cause, file, line, next ); return TRUE; }			\
    else return FALSE;							\
  }									\
}

#define OODE_ERROR_INHIBIT_DECLARATION( name, action )			\
class OODEError ## name							\
{									\
public:									\
  static void POST( const char* const cause,				\
		    const char* const file, const int line,		\
		    const char* const next = NULL )			\
  {}									\
  static int WHEN( const int condition, const char* const cause,	\
		   const char* const file, const int line,		\
		   const char* const next = NULL ) {			\
    return FALSE;							\
  }									\
  static int UNLESS( const int condition, const char* const cause,	\
		     const char* const file, const int line,		\
		     const char* const next = NULL ) {			\
    return FALSE;							\
  }									\
}

//-----------------------------------------------------------------------------

#ifdef OODE_INHIBIT_ALL_ERROR_CHECKING

#define OODE_ERROR_WHEN(         error, condition )
#define OODE_ERROR_WHEN_THEN(    error, condition, recover )
#define OODE_ERROR_UNLESS(       error, condition )
#define OODE_ERROR_UNLESS_THEN(  error, condition, recover )
#define OODE_ERROR_POST(         error, cause )
#define OODE_ERROR_POST_THEN(    error, cause, recover )

#else

#define OODE_ERROR_WHEN( error, condition )			\
{								\
  OODEError ## error ## ::WHEN( (condition), #condition,	\
				 __FILE__, __LINE__ );		\
}

#define OODE_ERROR_WHEN_THEN( error, condition, recover )	\
{								\
  if ( OODEError ## error ## ::WHEN( (condition), #condition,	\
				      __FILE__, __LINE__,	\
				      #recover ) )		\
    { recover ; }						\
}

#define OODE_ERROR_UNLESS( error, condition )			\
{								\
  OODEError ## error ## ::UNLESS( (condition),			\
				   "!( " #condition " )",	\
				   __FILE__, __LINE__ );	\
}

#define OODE_ERROR_UNLESS_THEN( error, condition, recover )	\
{								\
  if (OODEError ## error ## ::UNLESS( (condition),		\
				       "!( " #condition " )",	\
				       __FILE__, __LINE__,	\
				       #recover ) )		\
    { recover ; }						\
}

#define OODE_ERROR_POST( error, cause )			\
{								\
  OODEError ## error ## ::POST( cause, __FILE__, __LINE__ );	\
}

#define OODE_ERROR_POST_THEN( error, cause, recover )		\
{								\
  OODEError ## error ## ::POST( cause, __FILE__, __LINE__,	\
				 #recover );			\
    { recover ; }						\
}

#endif

//=============================================================================
//==                                                              ERROR CLASSES

#ifdef OODE_INHIBIT_WARNINGS
OODE_ERROR_INHIBIT_DECLARATION( WARNING, IGNORE );
#else
OODE_ERROR_CLASS_DECLARATION( WARNING, IGNORE );
#endif

#ifdef OODE_INHIBIT_API_ARGUMENT_CHECKING
OODE_ERROR_INHIBIT_DECLARATION( API_ARGUMENT, STOP_SIGNAL );
#else
OODE_ERROR_CLASS_DECLARATION( API_ARGUMENT, STOP_SIGNAL );
#endif

#ifdef OODE_INHIBIT_LIBRARY_STATE_CHECKING
OODE_ERROR_INHIBIT_DECLARATION( LIBRARY_STATE, STOP_SIGNAL );
#else
OODE_ERROR_CLASS_DECLARATION( LIBRARY_STATE, STOP_SIGNAL );
#endif

#ifdef OODE_INHIBIT_CONFIGURATION_CHECKING
OODE_ERROR_INHIBIT_DECLARATION( CONFIGURATION, STOP_SIGNAL );
#else
OODE_ERROR_CLASS_DECLARATION( CONFIGURATION, STOP_SIGNAL );
#endif

#ifdef OODE_INHIBIT_INTERNAL_DE_CHECKING
OODE_ERROR_INHIBIT_DECLARATION( INTERNAL, STOP_SIGNAL );//ABORT );
#else
OODE_ERROR_CLASS_DECLARATION( INTERNAL, STOP_SIGNAL );//ABORT );
#endif

#ifdef OODE_INHIBIT_MEMORY_ALLOCATION_CHECKING
OODE_ERROR_INHIBIT_DECLARATION( MALLOC, STOP_SIGNAL );//ABORT );
#else
OODE_ERROR_CLASS_DECLARATION( MALLOC, STOP_SIGNAL );//ABORT );
#endif

#ifdef OODE_INHIBIT_SANITY_CHECKING
OODE_ERROR_INHIBIT_DECLARATION( PARANOID, IGNORE );
#else
OODE_ERROR_CLASS_DECLARATION( PARANOID, IGNORE );
#endif

//-----------------------------------------------------------------------------

#ifdef OODE_INHIBIT_DYN_ARRAY_BOUNDS_CHECKING
OODE_ERROR_INHIBIT_DECLARATION( DYN_ARRAY_BOUNDS, STOP_SIGNAL );
#else
OODE_ERROR_CLASS_DECLARATION( DYN_ARRAY_BOUNDS, STOP_SIGNAL );
#endif

#ifdef OODE_INHIBIT_MEMORY_ALLOCATION_CHECKING
OODE_ERROR_INHIBIT_DECLARATION( DYN_ARRAY_MALLOC, STOP_SIGNAL );//ABORT );
#else
OODE_ERROR_CLASS_DECLARATION( DYN_ARRAY_MALLOC, STOP_SIGNAL );//ABORT );
#endif

//#############################################################################
//##                                                             ERROR HANDLING

#ifndef OODE_INHIBIT_ALL_ERROR_CHECKING

#include <iostream.h>
#include <sys/types.h>
#include <signal.h>
#include <unistd.h>

void OODEError::POST( const char* const error, const char* const cause,
		       const char* const file, const int line,
		       const ACTION action, const char* const next )
{
  cerr << '"' << file << "\", line " << line << ": "
       << error << " error:\n\t\t" << cause << endl;
  switch ( action ) {
  case IGNORE:
    break;
  case STOP_SIGNAL:
    if ( next ) {
      cerr << "\t continuing will: " << next << "\n";
    }
    cerr << "\t stopping..." << endl;
    kill( getpid(), SIGSTOP );
    cerr << "\t ... continuing." << endl;
    break;
  case ABORT:
    cerr << "\t impossible to continue, aborting execution." << endl;
    exit(0);
    break;
  }
}

#endif

//#############################################################################
//##                                         SIMPLISTIC DYNAMIC ARRAY TEMPLATES

#include <limits.h>

#define OODE_DYN_ARRAYS__CAN_TRUNCATE_NONEXISTANT
#define OODE_DYN_ARRAYS__POTENTIALLY_AMBIGUOUS_REMOVE_OK

//=============================================================================

template <class ELEMENT>
class OODE_DynArray
{
  // a dynamically growable array of ELEMENTs
  // ELEMENTs are classes or types which must have at least the following:
  //    default constructor: ELEMENT::ELEMENT()
  //    default destructor: ELEMENT::~ELEMENT()
  //    default assignment: ELEMENT& ELEMENT::operator=( const ELEMENT& )
  //    zero assignment: ELEMENT& ELEMENT::operator=( const '0' )

public:

  enum { BAD_INDEX = -1 };

protected:

  ELEMENT* array;
  short used; short avail;

  static void INDEXABLE( const short index )
  {
    OODE_ERROR_WHEN( DYN_ARRAY_BOUNDS, index < 0 );
    OODE_ERROR_WHEN( DYN_ARRAY_BOUNDS, index == SHRT_MAX );
  }
  void EXISTS( const short index ) const
  {
    INDEXABLE( index );
    OODE_ERROR_WHEN( DYN_ARRAY_BOUNDS, index >= used );
  }

public:

  OODE_DynArray()
    : array(0), used(0), avail(0)
  {}

  OODE_DynArray( const short size )
    : used(0)
  {
    INDEXABLE( size );
    array = new ELEMENT[size];
    OODE_ERROR_WHEN( DYN_ARRAY_MALLOC, ! array );
    avail = size;
  }

  ~OODE_DynArray()
  {
    if ( avail > 0 )
      delete[] array;
  }

  void grow( const short size )
  {
    if ( size > avail ) {
      ELEMENT* bigger = new ELEMENT[size];
      OODE_ERROR_WHEN( DYN_ARRAY_MALLOC,  ! bigger );
      for ( short i = 0; i < used; ++i )
	bigger[i] = array[i];
      if ( avail > 0 )
	delete[] array;
      array = bigger;
      avail = size;
    }
  }

  void allocate( const short size )
  {
    if ( size > used ) {
      grow( size );
      for ( ; used < size; ++used )
	array[used] = 0;
    }
  }

  void append( const ELEMENT& element )
  {
    INDEXABLE( used );
    grow( used + 1 );
    array[used] = element;
    ++used;
  }

  void insert( const ELEMENT& element, const short index )
  {
    INDEXABLE( index );
    if ( index < used ) {
      INDEXABLE( used );
      grow( used + 1 );
      for ( short i = used; i > index; --i )
	array[i] = array[i-1];
      array[index] = element;
      ++used;
    } else {
      allocate( index + 1 );
      array[index] = element;
    }
  }

  void remove( const short index )
  {
    EXISTS( index );
    for ( short i = index ; i < used; ++i )
      array[i] = array[i+1];
    --used;
  }

  void replace( const short index, const ELEMENT& element )
  {
    INDEXABLE( index );
    if ( index >= used )
      allocate( index + 1 );
    array[index] = element;
  }

  void truncate( const short from = 0 )
  {
#  ifdef OODE_DYN_ARRAYS__CAN_TRUNCATE_NONEXISTANT
     INDEXABLE( from );
#  else
     EXISTS( from );
#  endif
    used = from;
  }

  const ELEMENT& operator[]( const short index ) const
  {
    EXISTS( index );
    return array[index];
  }

  const ELEMENT* read() const
  {
    return array;
  }

  short length() const
  {
    return used;
  }

  short size() const
  {
    return avail;
  }

};

//=============================================================================

template <class ELEMENT>
class OODE_DynSearchableArray
  : public OODE_DynArray<ELEMENT>
{
  // extends arrays to be searchable.
  // ELEMENTs must meet requirements of OODE_DynArray,
  // and also must have one of these equality operators:
  //    int ELEMENT::operator==( const ELEMENT&) const
  //    int operator==( const ELEMENT&, const ELEMENT& )

public:

  OODE_DynSearchableArray()
  {}

  OODE_DynSearchableArray( const short size )
    : OODE_DynArray<ELEMENT>( size )
  {}

  short find( const ELEMENT& element ) const
  {
    for ( short i = 0; i < used; ++i )
      if ( element == array[i] )
	return i;
    return OODE_DynArray<ELEMENT>::BAD_INDEX;
  }

  void remove( const short index )
  {
    OODE_DynArray<ELEMENT>::remove( index );
  }

#ifdef OODE_DYN_ARRAYS__POTENTIALLY_AMBIGUOUS_REMOVE_OK

  int remove( const ELEMENT& element )
  {
    short index = find( element );
    if ( index != OODE_DynArray<ELEMENT>::BAD_INDEX )
      OODE_DynArray<ELEMENT>::remove( index );
    return index != OODE_DynArray<ELEMENT>::BAD_INDEX;
  }

#endif

};

//=============================================================================

template <class ELEMENT, class KEY>
class OODE_DynSearchableKeyedArray
  : public OODE_DynSearchableArray<ELEMENT>
{
  // extends searchable arrays to have search keys.
  // ELEMENTSs must meet requirements of OODE_DynSearchableArray,
  // and also provide a conversion operator:
  //    ELEMENT::operator KEY()
  // and the KEY class must have one of these equality operators:
  //    int KEY::operator==( const KEY&) const
  //    int operator==( const KEY&, const KEY& )
  // note that ELEMENT and KEY cannot be the same class.

public:

  OODE_DynSearchableKeyedArray()
  {}

  OODE_DynSearchableKeyedArray( const short size )
    : OODE_DynSearchableArray<ELEMENT>( size )
  {}

  short find( const ELEMENT& element ) const
  { return OODE_DynSearchableArray<ELEMENT>::find( element ); }

  typedef const KEY& (*CONVERTER)( const ELEMENT& );

  short find( const KEY& key, CONVERTER convert ) const
  {
    for ( short i = 0; i < used; ++i )
      if ( key == (*convert)(array[i]) )
	return i;
    return OODE_DynArray<ELEMENT>::BAD_INDEX;
  }

  ELEMENT lookup( const KEY& key, CONVERTER convert, const ELEMENT& bad ) const
  { 
   short index = find( key, convert );
    if ( index != OODE_DynArray<ELEMENT>::BAD_INDEX )
      return array[index];
    else
      return bad;
  }

  void remove( const short index )
  {
    OODE_DynArray<ELEMENT>::remove( index );
  }

#ifdef OODE_DYN_ARRAYS__POTENTIALLY_AMBIGUOUS_REMOVE_OK

  int remove( const ELEMENT& element )
  { return OODE_DynSearchableArray<ELEMENT>::remove( element ); }

  int remove( const KEY& key, CONVERTER convert )
  {
    short index = find( key, convert );
    if ( index != OODE_DynArray<ELEMENT>::BAD_INDEX )
      OODE_DynArray<ELEMENT>::remove( index );
    return index != OODE_DynArray<ELEMENT>::BAD_INDEX;
  }

#endif

};

//=============================================================================

template <class ELEMENT>
class OODE_DynSortedArray
  : protected OODE_DynSearchableArray<ELEMENT>
{
  // limits searchable arrays to be sorted.
  // ELEMENTs must meet requirements of OODE_DynSearchableArray,
  // and also must have one of these relational operators:
  //    int ELEMENT::operator<( const ELEMENT& ) const
  //    int operator<( const ELEMENT&, const ELEMENT& )

public:

  enum { BAD_INDEX = OODE_DynSearchableArray<ELEMENT>::BAD_INDEX };

  OODE_DynSortedArray()
  {}

  OODE_DynSortedArray( const short size )
    : OODE_DynSearchableArray<ELEMENT>( size )
  {}

  short find( const ELEMENT& element ) const
  {
    short lower = 0, probe, upper = used;
    while ( lower < upper ) {
      probe = (lower+upper)>>1;
      if ( element == array[probe] ) { return probe;      }
      if ( element <  array[probe] ) { upper = probe;     }
      else                           { lower = probe + 1; }
    }
    return OODE_DynArray<ELEMENT>::BAD_INDEX;
  }

  short insert( const ELEMENT& element )
  {
    short lower = 0, probe, upper = used;
    while ( lower < upper ) {
      probe = (lower+upper)>>1;
      if ( element == array[probe] ) { return probe;      }
      if ( element <  array[probe] ) { upper = probe;     }
      else                           { lower = probe + 1; }
    }
    OODE_DynArray<ELEMENT>::insert( element, lower );
    return lower;
  }

  void remove( const short index )
  {
    OODE_DynArray<ELEMENT>::remove( index );
  }

#ifdef OODE_DYN_ARRAYS__POTENTIALLY_AMBIGUOUS_REMOVE_OK

  int remove( const ELEMENT& element )
  {
    short index = find( element );
    if ( index != OODE_DynArray<ELEMENT>::BAD_INDEX )
      OODE_DynArray<ELEMENT>::remove( index );
    return index != OODE_DynArray<ELEMENT>::BAD_INDEX;
  }

#endif

  void truncate()
  {
    OODE_DynArray<ELEMENT>::truncate(0);
  }

  const ELEMENT& operator[]( const short index ) const
  {
    return OODE_DynArray<ELEMENT>::operator[]( index );
  }

  const ELEMENT* read() const
  {
    return OODE_DynArray<ELEMENT>::read();
  }

  short length() const
  {
    return OODE_DynArray<ELEMENT>::length();
  }

  short size() const
  {
    return OODE_DynArray<ELEMENT>::size();
  }
};

//=============================================================================

template <class ELEMENT, class KEY>
class OODE_DynSortedKeyedArray
  : public OODE_DynSortedArray<ELEMENT>
{
  // extends sorted arrays to have search keys.
  // ELEMENTSs must meet requirements of OODE_DynSortedArray,
  // and also provide a conversion operator:
  //    ELEMENT::operator KEY()
  // and the KEY class must have one of these equality operators:
  //    int KEY::operator==( const KEY&) const
  //    int operator==( const KEY&, const KEY& )
  // note that ELEMENT and KEY cannot be the same class.

public:

  OODE_DynSortedKeyedArray()
  {}

  OODE_DynSortedKeyedArray( const short size )
    : OODE_DynSortedArray<ELEMENT>( size )
  {}

  short find( const ELEMENT& element ) const
  { return OODE_DynSortedArray<ELEMENT>::find( element ); }

  typedef const KEY& (*CONVERTER)( const ELEMENT& );

  short find( const KEY& key, CONVERTER convert ) const
  {
    short lower = 0, probe, upper = used;
    while ( lower < upper ) {
      probe = (lower+upper)>>1;
      if ( key == (*convert)(array[probe]) ) { return probe;      }
      if ( key <  (*convert)(array[probe]) ) { upper = probe;     }
      else                                   { lower = probe + 1; }
    }
    return OODE_DynArray<ELEMENT>::BAD_INDEX;
  }

  ELEMENT lookup( const KEY& key, CONVERTER convert, const ELEMENT& bad ) const
  {
    short index = find( key, convert );
    if ( index != OODE_DynArray<ELEMENT>::BAD_INDEX )
      return array[index];
    else
      return bad;
  }

  void remove( const short index )
  {
    OODE_DynArray<ELEMENT>::remove( index );
  }

#ifdef OODE_DYN_ARRAYS__POTENTIALLY_AMBIGUOUS_REMOVE_OK

  int remove( const ELEMENT& element )
  { return OODE_DynSortedArray<ELEMENT>::remove( element ); }

  int remove( const KEY& key, CONVERTER convert )
  {
    short index = find( key, convert );
    if ( index != OODE_DynArray<ELEMENT>::BAD_INDEX )
      OODE_DynArray<ELEMENT>::remove( index );
    return index != OODE_DynArray<ELEMENT>::BAD_INDEX;
  }

#endif
};

//#############################################################################
//##                                                    TEMPLATE INSTANTIATIONS

#define OODE_DYN_ARRAYS__DECLARE( CHILD, TYPE, ELEMENT )	\
typedef OODE_Dyn ## TYPE ## Array< ELEMENT > CHILD ## _Base;	\
class CHILD : public CHILD ## _Base				\
{								\
public:								\
  CHILD () {}							\
  CHILD ( const short size ) : CHILD ## _Base( size ) {}	\
  void append( const ELEMENT& element )				\
    { CHILD ## _Base::append( element ); }			\
};

#define OODE_DYN_ARRAYS__DECLARE_KEYED( CHILD, TYPE, ELEMENT, KEY, FIELD ) \
typedef OODE_Dyn ## TYPE ## KeyedArray< ELEMENT, KEY > CHILD ## _Base;	   \
class CHILD : public CHILD ## _Base					   \
{									   \
 public:								   \
  CHILD () {}								   \
  CHILD ( const short size ) : CHILD ## _Base( size ) {}		   \
 private:								   \
  static const KEY& __convert( const ELEMENT& element )			   \
    { return element->FIELD; }						   \
 public:								   \
  short find( const ELEMENT& element ) const				   \
    { return CHILD ## _Base::find( element ); }				   \
  short find( const KEY& key ) const					   \
    { return CHILD ## _Base::find( key, (CONVERTER)__convert ); }	   \
  ELEMENT lookup( const KEY& key ) const				   \
    { return CHILD ## _Base::lookup( key, (CONVERTER)__convert,		   \
				     (ELEMENT)NULL ); }			   \
  void remove( const short index )					   \
    { CHILD ## _Base::remove( index ); }				   \
  int remove( const ELEMENT& element )					   \
    { return CHILD ## _Base::remove( element ); }			   \
  int remove( const KEY& key )						   \
    { return CHILD ## _Base::remove( key, (CONVERTER)__convert ); }	   \
};

typedef OODExchange* OODExchange_Ptr;
OODE_DYN_ARRAYS__DECLARE_KEYED( OODExchange_Array,
				Searchable, OODExchange_Ptr,
				DExchange, __dexchange )

typedef OODEPort* OODEPort_Ptr;
OODE_DYN_ARRAYS__DECLARE_KEYED( OODEPort_Array,
				Searchable, OODEPort_Ptr,
				DEPort, __deport )

typedef OODEMessage* OODEMessage_Ptr;
OODE_DYN_ARRAYS__DECLARE( OODEMessage_Array, Sorted, OODEMessage_Ptr )

struct OODE_Enabled_Messages_Struct
{
  OODExchange* __exchange;
  OODEMessage_Array* __enabled_messages;
};

typedef OODE_Enabled_Messages_Struct* OODE_Enabled_Messages_Struct_Ptr;
OODE_DYN_ARRAYS__DECLARE_KEYED( OODE_Enabled_Messages_Array, Searchable,
				OODE_Enabled_Messages_Struct_Ptr,
				OODExchange_Ptr, __exchange )

//#############################################################################
//##                                                                CLASS OODE

OODE* OODE::__the_running_oode = NULL;

//-----------------------------------------------------------------------------

OODE::OODE()
{
#ifndef OODE_INHIBIT_SANITY_CHECKING
  OODE_ERROR_WHEN( INTERNAL, CONNECTION != SHUTDOWN_CONNECTION );
  OODE_ERROR_WHEN( INTERNAL, INPUT != SHUTDOWN_CONNECTION_INPUT );
  OODE_ERROR_WHEN( INTERNAL, OUTPUT != SHUTDOWN_CONNECTION_OUTPUT );
  OODE_ERROR_WHEN( INTERNAL, ALL != SHUTDOWN_ALL_CONNECTION );
  OODE_ERROR_WHEN( INTERNAL, ALL_INPUT != SHUTDOWN_ALL_INPUT );
  OODE_ERROR_WHEN( INTERNAL, ALL_OUTPUT != SHUTDOWN_ALL_OUTPUT );
//OODE_ERROR_WHEN( INTERNAL, SERVER != SHUTDOWN_SERVER );
  OODE_ERROR_WHEN( INTERNAL, SYSTEM != SHUTDOWN_SYSTEM );
  OODE_ERROR_WHEN( INTERNAL, NEVER_FORWARD != DENever_Forward );
  OODE_ERROR_WHEN( INTERNAL, ALL_BUT_SENDER != DEAll_But_Sender );
  OODE_ERROR_WHEN( INTERNAL, FORWARD_TO_ALL != DEForward_To_All );
#endif

  __configuration = NOT_RUNNING;
  __exchange_array = NULL;

#ifndef OODE_INHIBIT_JAVA_IMPLEMENTATION
  __java_enabled = FALSE;
#endif
}

//-----------------------------------------------------------------------------

OODE::~OODE()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __configuration == NOT_RUNNING, close() );
}

//-----------------------------------------------------------------------------

int   OODE::create( int configuration )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __the_running_oode == NULL, return FALSE );

  OODE_ERROR_UNLESS_THEN( PARANOID,
			   __configuration == NOT_RUNNING, return FALSE );

  OODE_ERROR_UNLESS_THEN( PARANOID, ! __exchange_array, return FALSE );
  __exchange_array = new OODExchange_Array( 1 );
  OODE_ERROR_WHEN_THEN( MALLOC, ! __exchange_array, return FALSE );

  switch ( configuration ) {
  case DE_WRAPPERS:
    break;
  case NOT_RUNNING:
    OODE_ERROR_POST( API_ARGUMENT, "invalid configuration" );
    return FALSE;
  default:
    OODE_ERROR_POST( API_ARGUMENT, "unrecognized configuration" );
    return FALSE;
  }

  __configuration = configuration;
  __the_running_oode = this;

  return TRUE;
}

//-----------------------------------------------------------------------------

int       OODE::configuration()
{
  return __configuration;
}

//-----------------------------------------------------------------------------

void      OODE::close()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   this == __the_running_oode, return );

  OODE_ERROR_WHEN_THEN( PARANOID, __configuration == NOT_RUNNING, return );
  OODE_ERROR_WHEN_THEN( PARANOID, ! __exchange_array, return );

  short i;

  for ( i = 0; i < __exchange_array->length(); ++i )
    (*__exchange_array)[i]->close();

  for ( i = 0; i < __exchange_array->length(); ++i )
    delete (*__exchange_array)[i]; //!!! warn about this in the docs

  delete __exchange_array;

  __exchange_array = NULL;
  __configuration = NOT_RUNNING;
  __the_running_oode = NULL;
}

//-----------------------------------------------------------------------------

void  OODE::init_block_check()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   this == __the_running_oode, return );

  OODE_ERROR_WHEN_THEN( PARANOID,
			 __configuration == NOT_RUNNING, return );

  DEinit_block_check();
}

//-----------------------------------------------------------------------------

void  OODE::print_version()
{
  DEprint_version();
}

//-----------------------------------------------------------------------------

short        OODE::num_exchanges()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   this == __the_running_oode,
			   return OODExchange_Array::BAD_INDEX );

  OODE_ERROR_WHEN_THEN( PARANOID,
			 __configuration == NOT_RUNNING,
			 return OODExchange_Array::BAD_INDEX );

  OODE_ERROR_WHEN_THEN( PARANOID,
			 ! __exchange_array,
			 return OODExchange_Array::BAD_INDEX );

  return __exchange_array->length();
}

//-----------------------------------------------------------------------------

OODExchange* OODE::get_exchange( short index )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   this == __the_running_oode, return NULL );

  OODE_ERROR_WHEN_THEN( PARANOID,
			 __configuration == NOT_RUNNING, return NULL );

  OODE_ERROR_WHEN_THEN( PARANOID,
			 ! __exchange_array, return NULL );

  return (*__exchange_array)[index];
}

//-----------------------------------------------------------------------------

OODExchange* OODE::exchange_factory()
{
  return new OODExchange( this );
}

//-----------------------------------------------------------------------------

void OODE::__comment_callback( DExchange de, DEPort dep, char* com )
{
  OODE_ERROR_WHEN_THEN( INTERNAL, ! __the_running_oode, return );

  OODE_ERROR_WHEN_THEN( PARANOID,
			 ! __the_running_oode->__exchange_array, return );

  OODExchange* cxxde = __the_running_oode->__exchange_array->lookup( de );

  OODE_ERROR_WHEN_THEN( INTERNAL, ! cxxde, return );

  OODE_ERROR_WHEN_THEN( PARANOID, ! cxxde->__port_array, return );

  OODEPort* cxxdep = cxxde->__port_array->lookup( dep );

  OODE_ERROR_WHEN_THEN( INTERNAL, ! cxxdep, return );

  cxxdep->comment_callback( com );
}

void OODE::__open_callback( DExchange de, DEPort dep )
{
  OODE_ERROR_WHEN_THEN( INTERNAL, ! __the_running_oode, return );

  OODE_ERROR_WHEN_THEN( PARANOID,
			 ! __the_running_oode->__exchange_array, return );

  OODExchange* cxxde = __the_running_oode->__exchange_array->lookup( de );

  OODE_ERROR_WHEN_THEN( INTERNAL, ! cxxde, return );

  OODE_ERROR_WHEN_THEN( PARANOID, ! cxxde->__port_array, return );

  DEPort null = NULL;
  OODEPort* cxxdep = cxxde->__port_array->lookup( null );

  if ( ! cxxdep ) {

    cxxdep = cxxde->port_factory();

    OODE_ERROR_WHEN_THEN( MALLOC, ! cxxdep, return );

    cxxde->__port_array->append( cxxdep );

    cxxdep->__lib_created = TRUE;

  }

  cxxdep->__deport = dep;

  cxxdep->open_callback( cxxdep->__lib_created );
}

void OODE::__close_callback( DExchange de, DEPort dep )
{
  OODE_ERROR_WHEN_THEN( INTERNAL, ! __the_running_oode, return );

  OODE_ERROR_WHEN_THEN( PARANOID,
			 ! __the_running_oode->__exchange_array, return );

  OODExchange* cxxde = __the_running_oode->__exchange_array->lookup( de );

  OODE_ERROR_WHEN_THEN( INTERNAL, ! cxxde, return );

  OODE_ERROR_WHEN_THEN( PARANOID, ! cxxde->__port_array, return );

  short index = cxxde->__port_array->find( dep );

  OODE_ERROR_WHEN_THEN( INTERNAL,
			 index == OODEPort_Array::BAD_INDEX, return );

  OODEPort* cxxdep = (*cxxde->__port_array)[index];

  cxxde->__port_array->remove( index );

  cxxdep->close_callback( cxxdep->__lib_created );

  cxxdep->__attached = FALSE;

  if ( cxxdep->__lib_created )
    delete cxxdep;
}

//#############################################################################

OODExchange::OODExchange( OODE* cxxde )
{
  __cxxde = cxxde;
  __dexchange = NULL;
  __running = FALSE;
  __default_blocking = FALSE;
  __port_array = NULL;

#ifndef OODE_INHIBIT_JAVA_IMPLEMENTATION
  __java_enabled = FALSE;
#endif
}

//-----------------------------------------------------------------------------

OODExchange::~OODExchange()
{
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, __running, close() );
}

//-----------------------------------------------------------------------------

int OODExchange::create()
{
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE,
			 __cxxde != OODE::__the_running_oode, return FALSE );

  __port_array = new OODEPort_Array( 1 );
  OODE_ERROR_WHEN_THEN( MALLOC, ! __port_array, return FALSE );

  __dexchange = DExchange_create();
  OODE_ERROR_WHEN_THEN( INTERNAL, ! __dexchange, return FALSE );

  DExchange_register_open_handler( __dexchange, OODE::__open_callback );
  DExchange_register_close_handler( __dexchange, OODE::__close_callback );

  __cxxde->__exchange_array->append( this );

  __default_blocking = FALSE;
  __running = TRUE;
  return TRUE;
}

//-----------------------------------------------------------------------------

int OODExchange::running()
{
  return __running;
}

//-----------------------------------------------------------------------------

void    OODExchange::close()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return );

  DExchange_close( __dexchange ); // let DE close the ports
  __dexchange = NULL;

  OODE_ERROR_UNLESS( INTERNAL, __port_array->length() == 0 );

  delete __port_array;
  __port_array = NULL;

  __cxxde->__exchange_array->remove( this );

  __running = FALSE;
}

//-----------------------------------------------------------------------------

void OODExchange::change_default_blocking( int block )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return );
  DExchange_set_block_default( __dexchange, block );
  __default_blocking = block;
}

//-----------------------------------------------------------------------------

void OODExchange::change_default_forwarding( int style )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return );
# ifndef OODE_INHIBIT_API_ARGUMENT_CHECKING
  switch ( style ) {
  case OODE::NEVER_FORWARD:
  case OODE::ALL_BUT_SENDER:
  case OODE::FORWARD_TO_ALL:
    break;
  default:
    OODE_ERROR_POST( API_ARGUMENT, "invalid forwarding style" );
    return;
  }
# endif
  DExchange_set_forward( __dexchange, (DEForwardStyle) style );
}

//-----------------------------------------------------------------------------

void OODExchange::set_format_forwarding( char* format, int style )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return );
# ifndef OODE_INHIBIT_API_ARGUMENT_CHECKING
  switch ( style ) {
  case OODE::NEVER_FORWARD:
  case OODE::ALL_BUT_SENDER:
  case OODE::FORWARD_TO_ALL:
    break;
  default:
    OODE_ERROR_POST( API_ARGUMENT, "invalid forwarding style" );
    return;
  }
# endif
  DExchange_set_format_forward( __dexchange, format, (DEForwardStyle) style );
}

//-----------------------------------------------------------------------------

void OODExchange::set_format_fixed( char* format, int flag )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return );
  DExchange_set_format_fixed( __dexchange, format, flag );
}

//-----------------------------------------------------------------------------

char* OODExchange::listen( char* user, char* app, char* group, int timeout )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return NULL );

  int result = DExchange_open_inet( __dexchange );

  OODE_ERROR_UNLESS_THEN( WARNING, result < 0, return NULL );

  char* group_id
    = setup_comm_group( user, app, group, timeout,
			DExchange_host_name( __dexchange ),
			DExchange_inet_port( __dexchange ) );

  OODE_ERROR_WHEN_THEN( WARNING, ! group_id, return NULL );

  return group_id;
}

int OODExchange::listen( char* unix_socket )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return FALSE );
  return ( DExchange_open_unix( __dexchange, unix_socket ) >= 0 );
}

int OODExchange::listen( int inet_port )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return FALSE );
  return ( DExchange_listen( __dexchange, inet_port ) >= 0 );
}

int OODExchange::listen()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return FALSE );
  return ( DExchange_open_inet( __dexchange ) >= 0 );
}

//-----------------------------------------------------------------------------

void OODExchange::enable_comment_handling()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return );
  DExchange_register_comment_handler( __dexchange, OODE::__comment_callback );
}

//-----------------------------------------------------------------------------

int OODExchange::poll_and_handle( int block )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return -1 );
  return DExchange_poll_and_handle( __dexchange, block );
}

int OODExchange::poll_and_handle( int secs, int usecs )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return -1 );
  return DExchange_poll_and_handle_timeout( __dexchange, secs, usecs );
}

//-----------------------------------------------------------------------------

char* OODExchange::exchange_name()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return NULL );
  return DExchange_name( __dexchange );
}

char* OODExchange::host_name()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return NULL );
  return DExchange_host_name( __dexchange );
}

long   OODExchange::inet_port()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return 0L );
  return (long) DExchange_inet_port( __dexchange );
}

//-----------------------------------------------------------------------------

int     OODExchange::num_ports()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running,
			   return OODEPort_Array::BAD_INDEX );
  return __port_array->length();
}

OODEPort* OODExchange::get_port( int index )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __running, return NULL );
  return (*__port_array)[index];
}

OODEPort* OODExchange::port_factory()
{
  return new OODEPort( this );
}

//#############################################################################

int OODEPort::connect( char* user, char* app, char* group )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return FALSE );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, __attached, return FALSE );

  char* host; int port;
  init_comm_group_contact( user, app, group, &host, &port );

  OODE_ERROR_WHEN_THEN( WARNING, ! host, return FALSE );

  __cxxdexchange->__port_array->append( this );

  __deport = NULL;

  DEPort de_port
    = DExchange_initiate_conn( __cxxdexchange->__dexchange,
			       host, port,
			       __cxxdexchange->__default_blocking );

  if ( de_port ) {

    OODE_ERROR_WHEN_THEN( INTERNAL, __deport != de_port, return FALSE );

  } else {

    __cxxdexchange->__port_array->remove( this );

    OODE_ERROR_POST_THEN( WARNING, "failed to connect", return FALSE );
  }

  __attached = TRUE;
  return TRUE;
}

int OODEPort::connect( char* host_name, long inet_port )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return FALSE );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, __attached, return FALSE );

  __cxxdexchange->__port_array->append( this );

  __deport = NULL;

  DEPort de_port
    = DExchange_initiate_conn( __cxxdexchange->__dexchange,
			       host_name, (int)inet_port,
			       __cxxdexchange->__default_blocking );

  if ( de_port ) {

    OODE_ERROR_WHEN_THEN( INTERNAL, __deport != de_port, return FALSE );

  } else {

    __cxxdexchange->__port_array->remove( this );

    OODE_ERROR_POST_THEN( WARNING, "failed to connect", return FALSE );
  }

  __attached = TRUE;
  return TRUE;
}

int OODEPort::connect( char* unix_socket )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return FALSE );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, __attached, return FALSE );

  __cxxdexchange->__port_array->append( this );

  __deport = NULL;

  DEPort de_port
    = DExchange_initiate_conn( __cxxdexchange->__dexchange,
			       unix_socket, -1,
			       __cxxdexchange->__default_blocking );

  if ( de_port ) {

    OODE_ERROR_WHEN_THEN( INTERNAL, __deport != de_port, return FALSE );

  } else {

    __cxxdexchange->__port_array->remove( this );

    OODE_ERROR_POST_THEN( WARNING, "failed to connect", return FALSE );
  }

  __attached = TRUE;
  return TRUE;
}

//-----------------------------------------------------------------------------

int OODEPort::read( char* file_name )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return FALSE );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, __attached, return FALSE );

  __cxxdexchange->__port_array->append( this );

  __deport = NULL;

  DEPort de_port
    = DExchange_accept_conn_file( __cxxdexchange->__dexchange,
				  file_name, "r" );

  if ( de_port ) {

    OODE_ERROR_WHEN_THEN( INTERNAL, __deport != de_port, return FALSE );

  } else {

    __cxxdexchange->__port_array->remove( this );

    OODE_ERROR_POST_THEN( WARNING, "failed to connect", return FALSE );
  }

  __attached = TRUE;
  return TRUE;
}

int OODEPort::write( char* file_name )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return FALSE );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, __attached, return FALSE );

  __cxxdexchange->__port_array->append( this );

  __deport = NULL;

  DEPort de_port
    = DExchange_accept_conn_file( __cxxdexchange->__dexchange,
				  file_name, "w" );

  if ( de_port ) {

    OODE_ERROR_WHEN_THEN( INTERNAL, __deport != de_port, return FALSE );

  } else {

    __cxxdexchange->__port_array->remove( this );

    OODE_ERROR_POST_THEN( WARNING, "failed to connect", return FALSE );
  }

  __attached = TRUE;
  return TRUE;
}

//-----------------------------------------------------------------------------

void OODEPort::open_callback( int )
{
}

//-----------------------------------------------------------------------------

int OODEPort::attached()
{
  return __attached;
}

//-----------------------------------------------------------------------------

void OODEPort::close()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __attached, return );

  DEport_close( __deport );

  OODE_ERROR_WHEN( INTERNAL, __attached );
}

//-----------------------------------------------------------------------------

void OODEPort::close_callback( int )
{
}

//-----------------------------------------------------------------------------

void OODEPort::comment_callback( char* )
{
}

//-----------------------------------------------------------------------------

int OODEPort::shutdown( int flag, char* message )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return FALSE );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __attached, return FALSE );

  return
    DExchange_shutdown_connection( __cxxdexchange->__dexchange, __deport,
				   (ShutDownFlag) flag, message );
}

//-----------------------------------------------------------------------------

int OODEPort::set_format_blocking( char* format, int block )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return FALSE );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __attached, return FALSE );

  return DEport_set_format_block( __deport, format, block );
}

//-----------------------------------------------------------------------------

void OODEPort::comment( char* msg )
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __attached, return );

  write_comment_IOfile( DEport_to_port( __deport ), msg );
}

//-----------------------------------------------------------------------------

char* OODEPort::port_name()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return NULL );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __attached, return NULL );

  return DEport_name( __deport );
}

char* OODEPort::host_name()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return NULL );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __attached, return NULL );

  return DEport_host_name( __deport );
}

long   OODEPort::port_number()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return -1L );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __attached, return -1L );

  return DEport_port_number( __deport );
}

OODExchange* OODEPort::exchange()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   __cxxdexchange->__running, return NULL );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, __attached, return NULL );

  OODE_ERROR_UNLESS_THEN( PARANOID,
	       __cxxdexchange->__dexchange == DExchange_of_port( __deport ),
			   return NULL );
  return NULL;
}

//-----------------------------------------------------------------------------

OODEPort::OODEPort( OODExchange* cxxde )
{
  __cxxdexchange = cxxde;
  __lib_created = FALSE;
  __attached = FALSE;
  __deport = NULL;

#ifndef OODE_INHIBIT_JAVA_IMPLEMENTATION
  __java_enabled = FALSE;
#endif
}

OODEPort::~OODEPort()
{
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, __attached, close() );
}

//#############################################################################

void* OODEMessage::__tmp_data = NULL;
DExchange OODEMessage::__tmp_de = NULL;

//-----------------------------------------------------------------------------

int OODEMessage::allocate_buffer()
{
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, !__data, return FALSE );
  __data = malloc( __info->__size );
  OODE_ERROR_WHEN_THEN( MALLOC, !__data, return FALSE );
  return __data != NULL;
}

void OODEMessage::free_buffer()
{
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, !__data, return );
  free( __data );
}

void OODEMessage::enable_filtering( OODExchange* exch )
{
  OODE_ERROR_WHEN_THEN( PARANOID, !__info, return );
  if ( ! __info->__filtering_array ) {
    __info->__filtering_array = new OODE_Enabled_Messages_Array( 1 );
    OODE_ERROR_WHEN_THEN( MALLOC, !__info->__filtering_array, return );
  }
  struct OODE_Enabled_Messages_Struct* enabled_messages_struct
    = __info->__filtering_array->lookup( exch );
  if ( ! enabled_messages_struct ) {
    enabled_messages_struct = new OODE_Enabled_Messages_Struct;
    OODE_ERROR_WHEN_THEN( MALLOC, ! enabled_messages_struct, return );
    __info->__filtering_array->append( enabled_messages_struct );
    enabled_messages_struct->__exchange = exch;
    enabled_messages_struct->__enabled_messages = new OODEMessage_Array( 1 );
    OODE_ERROR_WHEN_THEN( MALLOC,
			  ! enabled_messages_struct->__enabled_messages,
			  return );
  }
  short int index = enabled_messages_struct->__enabled_messages->find( this );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE,
			index != OODEMessage_Array::BAD_INDEX, return );
  enabled_messages_struct->__enabled_messages->append( this );
  if ( enabled_messages_struct->__enabled_messages->length() == 1 )
    DExchange_register_filter( exch->__dexchange, __info->__name,
			       __filter_callback, enabled_messages_struct );
}

void OODEMessage::disable_filtering( OODExchange* exch )
{
  OODE_ERROR_WHEN_THEN( PARANOID, !__info, return );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, !__info->__filtering_array, return );
  short int index = __info->__filtering_array->find( exch );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE,
			index == OODEMessage_Array::BAD_INDEX, return );
  struct OODE_Enabled_Messages_Struct* enabled_messages_struct
    = (*__info->__filtering_array)[ index ];
  enabled_messages_struct->__enabled_messages->remove( index );
  if ( enabled_messages_struct->__enabled_messages->length() == 0 )
    DExchange_unregister_filter(  exch->__dexchange, __info->__name,
				  __filter_callback, enabled_messages_struct );
}

int OODEMessage::filter_callback( OODEPort* )
{
  return TRUE;
}

void OODEMessage::enable_processing( OODExchange* exch )
{
  OODE_ERROR_WHEN_THEN( PARANOID, !__info, return );
  if ( ! __info->__processing_array ) {
    __info->__processing_array = new OODE_Enabled_Messages_Array( 1 );
    OODE_ERROR_WHEN_THEN( MALLOC, !__info->__processing_array, return );
  }
  struct OODE_Enabled_Messages_Struct* enabled_messages_struct
    = __info->__processing_array->lookup( exch );
  if ( ! enabled_messages_struct ) {
    enabled_messages_struct = new OODE_Enabled_Messages_Struct;
    OODE_ERROR_WHEN_THEN( MALLOC, ! enabled_messages_struct, return );
    __info->__processing_array->append( enabled_messages_struct );
    enabled_messages_struct->__exchange = exch;
    enabled_messages_struct->__enabled_messages = new OODEMessage_Array( 1 );
    OODE_ERROR_WHEN_THEN( MALLOC,
			  ! enabled_messages_struct->__enabled_messages,
			  return );
  }
  short int index = enabled_messages_struct->__enabled_messages->find( this );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE,
			index != OODEMessage_Array::BAD_INDEX, return );
  enabled_messages_struct->__enabled_messages->append( this );
  if ( enabled_messages_struct->__enabled_messages->length() == 1 )
    DExchange_register_function( exch->__dexchange, __info->__name,
				 __process_callback, enabled_messages_struct );
}

void OODEMessage::disable_processing( OODExchange* exch )
{
  OODE_ERROR_WHEN_THEN( PARANOID, !__info, return );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, !__info->__processing_array, return );
  short int index = __info->__processing_array->find( exch );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE,
			index == OODEMessage_Array::BAD_INDEX, return );
  struct OODE_Enabled_Messages_Struct* enabled_messages_struct
    = (*__info->__processing_array)[ index ];
  enabled_messages_struct->__enabled_messages->remove( index );
  if ( enabled_messages_struct->__enabled_messages->length() == 0 )
    DExchange_unregister_function(  exch->__dexchange, __info->__name,
				    __process_callback,
				    enabled_messages_struct );
}

void OODEMessage::process_callback( OODEPort* )
{
}

int OODEMessage::send( OODEPort* to )
{
  OODE_ERROR_WHEN_THEN( API_ARGUMENT,  ! to, return FALSE );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! to->__cxxdexchange, return FALSE );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! to->__cxxdexchange->__dexchange,
			 return FALSE );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! to->__deport, return FALSE );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! __info, return FALSE );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, __info->__format_ID < 0, return FALSE );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! __data, return FALSE );

  return DEport_write_data( to->__deport, __info->__format_ID, __data );
}

void OODEMessage::forward( OODEPort* from )
{
  OODE_ERROR_WHEN_THEN( API_ARGUMENT,  ! from, return );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! from->__cxxdexchange, return );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! from->__cxxdexchange->__dexchange,
			 return );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! from->__deport, return );
  OODE_ERROR_WHEN_THEN( PARANOID,      ! __info, return );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, __info->__format_ID < 0, return );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! __data, return );

  DExchange_forward_data( from->__cxxdexchange->__dexchange,
			  from->__deport, __info->__format_ID, __data );
}

void OODEMessage::register_format( OODExchange* exch )
{
  OODE_ERROR_WHEN_THEN( API_ARGUMENT,  ! exch, return );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! exch->__dexchange, return );
  OODE_ERROR_WHEN_THEN( PARANOID,      ! __info, return );

  if ( __info->__format_ID >= 0 ) return; // already registered

  DExchange_register_format( exch->__dexchange, __info->__name,
			     __info->__field_list );

  __info->__format_ID = DEget_format_id( exch->__dexchange, __info->__name );
}

void OODEMessage::set_format_fixed( OODExchange* exch, int flag )
{
  OODE_ERROR_WHEN_THEN( API_ARGUMENT,  ! exch, return );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! exch->__dexchange, return );
  OODE_ERROR_WHEN_THEN( PARANOID,      ! __info, return );

  DExchange_set_format_fixed( exch->__dexchange, __info->__name, flag );
}

void OODEMessage::set_format_forwarding( OODExchange* exch, int style )
{
  OODE_ERROR_WHEN_THEN( API_ARGUMENT,  ! exch, return );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! exch->__dexchange, return );
  OODE_ERROR_WHEN_THEN( PARANOID,      ! __info, return );

  DExchange_set_format_forward( exch->__dexchange, __info->__name,
				(DEForwardStyle)style );
}

int OODEMessage::set_format_blocking( OODEPort* port, int block )
{
  OODE_ERROR_WHEN_THEN( API_ARGUMENT,  ! port, return FALSE );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! port->__deport, return FALSE );
  OODE_ERROR_WHEN_THEN( PARANOID,      ! __info, return FALSE );

  return DEport_set_format_block( port->__deport, __info->__name, block );
}

void OODEMessage::take_buffer()
{
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! __tmp_de, return );
  if ( __tmp_data ) free( __tmp_data );
  DEtake_buffer( __tmp_de, __data );
  __tmp_data = __data;
}

OODEMessage::OODEMessage()
{
#ifndef OODE_INHIBIT_JAVA_IMPLEMENTATION
  __java_enabled = FALSE;
#endif
}

OODEMessage::~OODEMessage()
{
#ifndef OODE_INHIBIT_LIBRARY_STATE_CHECKING
  struct OODE_Enabled_Messages_Struct* enabled_messages_struct;
  short int exchange; short int message;
  int __this_message_was_enabled = FALSE;
  if ( __info->__filtering_array )
  for ( exchange = 0;
	exchange < __info->__filtering_array->length();
	++exchange ) {
    enabled_messages_struct = (*__info->__filtering_array)[exchange];
    for ( message = 0;
	  message < enabled_messages_struct->__enabled_messages->length();
	  ++ message ) {
      if ( this == (*enabled_messages_struct->__enabled_messages)[message] ) {
	enabled_messages_struct->__enabled_messages->remove( message );
	__this_message_was_enabled = TRUE;
      }
    }
  }
  if ( __info->__processing_array )
  for ( exchange = 0;
	exchange < __info->__processing_array->length();
	++exchange ) {
    enabled_messages_struct = (*__info->__processing_array)[exchange];
    for ( message = 0;
	  message < enabled_messages_struct->__enabled_messages->length();
	  ++ message ) {
      if ( this == (*enabled_messages_struct->__enabled_messages)[message] ) {
	enabled_messages_struct->__enabled_messages->remove( message );
	__this_message_was_enabled = TRUE;
      }
    }
  }
  OODE_ERROR_WHEN( LIBRARY_STATE,  __this_message_was_enabled );
#endif
}

int OODEMessage::__filter_callback( DExchange de, DEPort dep,
				    int, void* data, int, void* ptr )
{
  struct OODE_Enabled_Messages_Struct* enabled_messages_struct
    = (OODE_Enabled_Messages_Struct*)ptr;
  OODE_ERROR_WHEN_THEN( INTERNAL, ! enabled_messages_struct, return TRUE );
  OODE_ERROR_WHEN_THEN( PARANOID, ! data, return TRUE );
  OODEMessage_Array* enabled_messages
    = enabled_messages_struct->__enabled_messages;
  OODE_ERROR_WHEN_THEN( INTERNAL, ! enabled_messages, return TRUE );
  OODE_ERROR_WHEN_THEN( INTERNAL,
			enabled_messages->length() <= 0, return TRUE );
  OODExchange* exchange = enabled_messages_struct->__exchange;
  OODE_ERROR_WHEN_THEN( PARANOID, exchange->__dexchange != de, return TRUE );
  OODEPort* port = exchange->__port_array->lookup( dep );
  OODE_ERROR_WHEN_THEN( INTERNAL, ! port, return TRUE );
  OODE_ERROR_UNLESS_THEN( INTERNAL, ! __tmp_de, return TRUE );
  __tmp_de = de;
  for ( int i = 0; i < enabled_messages->length(); ++i ) {
    OODEMessage* msg = (*enabled_messages)[i];
    OODE_ERROR_WHEN_THEN( INTERNAL, ! msg, return TRUE );
    __tmp_data = msg->__data;
    msg->__data = data;
    if ( ! msg->filter_callback( port ) ) return FALSE;
    if ( __tmp_data == msg->__data ) { __tmp_data = NULL; return FALSE; }
    msg->__data = __tmp_data;
  }
  __tmp_data = NULL;
  __tmp_de = NULL;
  return TRUE;
}

int OODEMessage::__process_callback( DExchange de, DEPort dep,
				     int, void* data, int, void* ptr )
{
  struct OODE_Enabled_Messages_Struct* enabled_messages_struct
    = (OODE_Enabled_Messages_Struct*)ptr;
  OODE_ERROR_WHEN_THEN( INTERNAL, ! enabled_messages_struct, return TRUE );
  OODE_ERROR_WHEN_THEN( PARANOID, ! data, return TRUE );
  OODEMessage_Array* enabled_messages
    = enabled_messages_struct->__enabled_messages;
  OODE_ERROR_WHEN_THEN( INTERNAL, ! enabled_messages, return TRUE );
  OODE_ERROR_WHEN_THEN( INTERNAL,
			enabled_messages->length() <= 0, return TRUE );
  OODExchange* exchange = enabled_messages_struct->__exchange;
  OODE_ERROR_WHEN_THEN( PARANOID, exchange->__dexchange != de, return TRUE );
  OODEPort* port = exchange->__port_array->lookup( dep );
  OODE_ERROR_WHEN_THEN( INTERNAL, ! port, return TRUE );
  OODE_ERROR_UNLESS_THEN( INTERNAL, ! __tmp_de, return TRUE );
  __tmp_de = de;
  for ( int i = 0; i < enabled_messages->length(); ++i ) {
    OODEMessage* msg = (*enabled_messages)[i];
    OODE_ERROR_WHEN_THEN( INTERNAL, ! msg, return TRUE );
    __tmp_data = msg->__data;
    msg->__data = data;
    msg->process_callback( port );
    if ( __tmp_data == msg->__data ) { __tmp_data = NULL; return FALSE; }
    msg->__data = __tmp_data;
  }
  __tmp_data = NULL;
  __tmp_de = NULL;
  return TRUE;
}

//#############################################################################

#ifndef OODE_INHIBIT_JAVA_IMPLEMENTATION

//#############################################################################

#define OODE_PARANOID_GetStringUTFChars( env, jstr )			\
jsize jstr ## _len = env->GetStringUTFLength( jstr );			\
const char* const jstr ## _utf = env->GetStringUTFChars( jstr, NULL );	\
OODE_ERROR_WHEN( PARANOID, jstr ## _utf[ jstr ## _len ] );

//#############################################################################

static jfieldID  OODE_FID__cxxjde;
static jmethodID OODE_MID__exchange_factory;

inline static void OODE_INIT_IDS
  ( JNIEnv* env, jobject obj )
{
  jclass cls = env->GetObjectClass( obj );
  OODE_FID__cxxjde = env->GetFieldID( cls, "__cxxjde", "J" );
  OODE_ERROR_WHEN( INTERNAL, ! OODE_FID__cxxjde );
  OODE_MID__exchange_factory
    = env->GetMethodID( cls, "exchange_factory",
			"()Ldataexchange/OODExchange;" );
  OODE_ERROR_WHEN( INTERNAL, ! OODE_MID__exchange_factory );
}

inline static OODE_Java_Enabled* OODE_get__cxxjde
  ( JNIEnv* env, jobject obj )
{ return (OODE_Java_Enabled*)(env->GetLongField( obj, OODE_FID__cxxjde )); }

inline static void OODE_set__cxxjde
  ( JNIEnv* env, jobject obj, OODE_Java_Enabled* val )
{ env->SetLongField( obj, OODE_FID__cxxjde, (jlong)val ); }

inline static jobject OODE__exchange_factory
  ( JNIEnv* env, jobject obj )
{ return env->CallObjectMethod( obj, OODE_MID__exchange_factory ); }

//=============================================================================

static jfieldID  OODExchange_FID__jde;
static jfieldID  OODExchange_FID__cxxjdexchange;
static jmethodID OODExchange_MID__port_factory;

inline static void OODExchange_INIT_IDS
  ( JNIEnv* env, jobject obj )
{
  jclass cls = env->GetObjectClass( obj );
  OODExchange_FID__jde
    = env->GetFieldID( cls, "__jde", "Ldataexchange/OODE;" );
  OODE_ERROR_WHEN( INTERNAL, ! OODExchange_FID__jde );
  OODExchange_FID__cxxjdexchange
    = env->GetFieldID( cls, "__cxxjdexchange", "J" );
  OODE_ERROR_WHEN( INTERNAL, ! OODExchange_FID__cxxjdexchange );
  OODExchange_MID__port_factory
    = env->GetMethodID( cls, "port_factory", "()Ldataexchange/OODEPort;" );
  OODE_ERROR_WHEN( INTERNAL, ! OODExchange_MID__port_factory );
}

inline static OODE_Java_Enabled* OODExchange_get__cxxjde
  ( JNIEnv* env, jobject obj )
{
  jobject jde = env->GetObjectField( obj, OODExchange_FID__jde );
  OODE_ERROR_WHEN_THEN( INTERNAL, !jde, return NULL );
  return OODE_get__cxxjde( env, jde );
}
inline static OODExchange_Java_Enabled* OODExchange_get__cxxjdexchange
  ( JNIEnv* env, jobject obj )
{ return (OODExchange_Java_Enabled*)
    (env->GetLongField( obj, OODExchange_FID__cxxjdexchange )); }
inline static void OODExchange_set__cxxjdexchange
  ( JNIEnv* env, jobject obj, OODExchange_Java_Enabled* val )
{ env->SetLongField( obj, OODExchange_FID__cxxjdexchange, (jlong)val ); }

inline static jobject OODExchange__port_factory
  ( JNIEnv* env, jobject obj )
{ return env->CallObjectMethod( obj, OODExchange_MID__port_factory ); }

//=============================================================================

static jfieldID  OODEPort_FID__jdexchange;
static jfieldID  OODEPort_FID__cxxjdeport;
static jmethodID OODEPort_MID__open_callback;
static jmethodID OODEPort_MID__close_callback;
static jmethodID OODEPort_MID__comment_callback;

inline static void OODEPort_INIT_IDS
  ( JNIEnv* env, jobject obj )
{
  jclass cls = env->GetObjectClass( obj );
  OODEPort_FID__jdexchange
    = env->GetFieldID( cls, "__jdexchange", "Ldataexchange/OODExchange;" );
  OODE_ERROR_WHEN( INTERNAL, ! OODEPort_FID__jdexchange );
  OODEPort_FID__cxxjdeport
    = env->GetFieldID( cls, "__cxxjdeport", "J" );
  OODE_ERROR_WHEN( INTERNAL, ! OODEPort_FID__cxxjdeport );
  OODEPort_MID__open_callback
    = env->GetMethodID( cls, "open_callback", "(Z)V" );
  OODE_ERROR_WHEN( INTERNAL, ! OODEPort_MID__open_callback );
  OODEPort_MID__close_callback
    = env->GetMethodID( cls, "close_callback", "(Z)V" );
  OODE_ERROR_WHEN( INTERNAL, ! OODEPort_MID__close_callback );
  OODEPort_MID__comment_callback
    = env->GetMethodID( cls, "comment_callback", "(Ljava/lang/String;)V" );
  OODE_ERROR_WHEN( INTERNAL, ! OODEPort_MID__comment_callback );
}

inline static OODExchange_Java_Enabled* OODEPort_get__cxxjdexchange
  ( JNIEnv* env, jobject obj )
{
  jobject jdexchange = env->GetObjectField( obj, OODEPort_FID__jdexchange );
  OODE_ERROR_WHEN_THEN( INTERNAL, ! jdexchange, return NULL );
  return OODExchange_get__cxxjdexchange( env, jdexchange );
}
inline static OODEPort_Java_Enabled* OODEPort_get__cxxjdeport
  ( JNIEnv* env, jobject obj )
{ return (OODEPort_Java_Enabled*)(env->GetLongField( obj, OODEPort_FID__cxxjdeport )); }
inline static void OODEPort_set__cxxjdeport
  ( JNIEnv* env, jobject obj, OODEPort_Java_Enabled* val )
{ env->SetLongField( obj, OODEPort_FID__cxxjdeport, (jlong)val ); }
inline static void OODEPort__open_callback
  ( JNIEnv* env, jobject obj, jboolean accepted )
{ env->CallVoidMethod( obj, OODEPort_MID__open_callback, accepted ); }
inline static void OODEPort__close_callback
  ( JNIEnv* env, jobject obj, jboolean accepted )
{ env->CallVoidMethod( obj, OODEPort_MID__close_callback, accepted ); }
inline static void OODEPort__comment_callback
  ( JNIEnv* env, jobject obj, jstring comment )
{ env->CallVoidMethod( obj, OODEPort_MID__comment_callback, comment ); }

//=============================================================================

static jfieldID  OODEMessage_FID__cxxjdemessage;
static jmethodID OODEMessage_MID__filter_callback;
static jmethodID OODEMessage_MID__process_callback;

inline static void OODEMessage_INIT_IDS
  ( JNIEnv* env, jobject obj )
{
  jclass cls = env->GetObjectClass( obj );
  OODEMessage_FID__cxxjdemessage
    = env->GetFieldID( cls, "__cxxjdemessage", "J" );
  OODE_ERROR_WHEN( INTERNAL, ! OODEMessage_FID__cxxjdemessage );
  OODEMessage_MID__filter_callback
    = env->GetMethodID( cls, "filter_callback", "(Ldataexchange/OODEPort;)Z");
  OODE_ERROR_WHEN( INTERNAL, ! OODEMessage_MID__filter_callback );
  OODEMessage_MID__process_callback
    = env->GetMethodID( cls, "process_callback", "(Ldataexchange/OODEPort;)V");
  OODE_ERROR_WHEN( INTERNAL, ! OODEMessage_MID__process_callback );
}

inline static OODEMessage_Java_Enabled* OODEMessage_get__cxxjdemessage
  ( JNIEnv* env, jobject obj )
{ return (OODEMessage_Java_Enabled*)(env->GetLongField( obj, OODEMessage_FID__cxxjdemessage )); }

OODEMessage_Java_Enabled* OODEMessage_Java_Enabled::__get__cxxjdemessage
  ( JNIEnv* env, jobject obj, OODEMessage_Static_Info_Struct* jobj_info )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( INTERNAL, !cxxjdemessage, return NULL );
  OODE_ERROR_WHEN_THEN( INTERNAL, jobj_info != cxxjdemessage->__info, return NULL );
  return cxxjdemessage;
}

inline static void OODEMessage_set__cxxjdemessage
  ( JNIEnv* env, jobject obj, OODEMessage_Java_Enabled* val )
{ env->SetLongField( obj, OODEMessage_FID__cxxjdemessage, (jlong)val ); }

void OODEMessage_Java_Enabled::__set__cxxjdemessage
  ( JNIEnv* env, jobject obj, OODEMessage_Java_Enabled* val )
{
  OODE_ERROR_WHEN_THEN( INTERNAL, !val, return );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, !val->__java_this, return );
  val->__java_this = env->NewGlobalRef( obj );
  OODEMessage_set__cxxjdemessage( env, obj, val );
  val->__java_created = TRUE;
}

void OODEMessage_Java_Enabled::__zap__cxxjdemessage
  ( JNIEnv* env, jobject obj, OODEMessage_Java_Enabled* msg )
{
  OODE_ERROR_WHEN_THEN( INTERNAL, !msg, return );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, !msg->__java_this, return );
  OODEMessage_set__cxxjdemessage( env, obj, NULL );
  env->DeleteGlobalRef( msg->__java_this );
  msg->__java_this = NULL;
  msg->__java_created = FALSE;
}

inline static jboolean OODEMessage__filter_callback
  ( JNIEnv* env, jobject obj, jobject from_port )
{ return env->CallBooleanMethod( obj, OODEMessage_MID__filter_callback,
				 from_port ); }
inline static void OODEMessage__process_callback
  ( JNIEnv* env, jobject obj, jobject from_port )
{ env->CallVoidMethod( obj, OODEMessage_MID__process_callback, from_port ); }

//#############################################################################

OODE_Java_Enabled::OODE_Java_Enabled() : OODE()
{
  __java_enabled = TRUE;
  __java_this = NULL;
  __java_created = FALSE;
}

OODE_Java_Enabled::~OODE_Java_Enabled()
{
  OODE_ERROR_UNLESS( LIBRARY_STATE, ! __java_this );
}

OODExchange* OODE_Java_Enabled::exchange_factory()
{
  return new OODExchange_Java_Enabled( this );
}

//=============================================================================

unsigned int OODExchange_Java_Enabled::__java_obj_count = 0;

JNIEnv* OODExchange_Java_Enabled::__polling_env = NULL;
jobject OODExchange_Java_Enabled::__polling_obj = NULL;

OODExchange_Java_Enabled::OODExchange_Java_Enabled( OODE* lib )
  : OODExchange( lib )
{
  __java_enabled = TRUE;
  __java_this = NULL;
  __java_created = FALSE;
}

OODExchange_Java_Enabled::~OODExchange_Java_Enabled()
{
  OODE_ERROR_UNLESS( LIBRARY_STATE, ! __java_this );
}

OODEPort* OODExchange_Java_Enabled::port_factory()
{
  return new OODEPort_Java_Enabled( this );
}

//=============================================================================

unsigned int OODEPort_Java_Enabled::__java_obj_count = 0;

OODEPort_Java_Enabled::OODEPort_Java_Enabled( OODExchange* exch )
  : OODEPort( exch )
{
  __java_enabled = TRUE;
  __java_this = NULL;
  __java_created = FALSE;
}

OODEPort_Java_Enabled::~OODEPort_Java_Enabled()
{
  OODE_ERROR_UNLESS( LIBRARY_STATE, ! __java_this );
}

void OODEPort_Java_Enabled::open_callback( int accepted )
{
  if ( __java_this && OODExchange_Java_Enabled::__polling_env )
    OODEPort__open_callback( OODExchange_Java_Enabled::__polling_env, __java_this,
			       (jboolean)accepted );
}

void OODEPort_Java_Enabled::close_callback( int accepted )
{
  if ( __java_this && OODExchange_Java_Enabled::__polling_env )
    OODEPort__close_callback( OODExchange_Java_Enabled::__polling_env, __java_this,
				(jboolean)accepted );
}

void OODEPort_Java_Enabled::comment_callback( char* comment )
{
  if ( __java_this && OODExchange_Java_Enabled::__polling_env ) {
    jstring str = OODExchange_Java_Enabled::__polling_env
      ->NewStringUTF( comment );
    OODEPort__comment_callback( OODExchange_Java_Enabled::__polling_env,
				__java_this,
				str );
  }
}

//=============================================================================

unsigned int OODEMessage_Java_Enabled::__java_obj_count = 0;

OODEMessage_Java_Enabled::OODEMessage_Java_Enabled()
{
  __java_enabled = TRUE;
  __java_this = NULL;
  __java_created = FALSE;
}

OODEMessage_Java_Enabled::~OODEMessage_Java_Enabled()
{
  OODE_ERROR_UNLESS( LIBRARY_STATE, ! __java_this );
}

int OODEMessage_Java_Enabled::filter_callback( OODEPort* from_port )
{
  int result = TRUE;
  if ( __java_this && OODExchange_Java_Enabled::__polling_env ) {
    OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, from_port->__java_enabled,
			    return TRUE );
    OODEPort_Java_Enabled* port = (OODEPort_Java_Enabled*)from_port;
    if ( ! port->__java_this ) {
      //!!! should use the java OODExchange.port_factory to create one
      OODE_ERROR_POST_THEN( LIBRARY_STATE,
			    "java filter_callback can't find java OODEPort",
			    return TRUE );
    }
    result = OODEMessage__filter_callback( OODExchange_Java_Enabled::__polling_env,
					   __java_this,
					   port->__java_this );
  }
  return result;
}

void OODEMessage_Java_Enabled::process_callback( OODEPort* from_port )
{
  if ( __java_this && OODExchange_Java_Enabled::__polling_env ) {
    OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, from_port->__java_enabled,
			    return );
    OODEPort_Java_Enabled* port = (OODEPort_Java_Enabled*)from_port;
    if ( ! port->__java_this ) {
      //!!! should use the java OODExchange.port_factory to create one
      OODE_ERROR_POST_THEN( LIBRARY_STATE,
			    "java filter_callback can't find java OODEPort",
			    return );
    }
    OODEMessage__process_callback( OODExchange_Java_Enabled::__polling_env,
				   __java_this,
				   port->__java_this );
  }
}

//#############################################################################
extern "C" {
//#############################################################################

JNIEXPORT void JNICALL Java_dataexchange_OODE_OODE_1_1construct
  ( JNIEnv* env, jobject obj )
{
  OODE_INIT_IDS( env, obj );
}

//-----------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_dataexchange_OODE_OODE_1_1destruct
  ( JNIEnv* env, jobject obj )
{
  OODE_Java_Enabled* cxxjde = OODE_get__cxxjde( env, obj );
  if ( cxxjde ) {
    if ( cxxjde->__java_created )
      delete cxxjde;
  }
}

//-----------------------------------------------------------------------------

JNIEXPORT jboolean JNICALL Java_dataexchange_OODE_create
  ( JNIEnv* env, jobject obj, jint config )
{
  OODE_Java_Enabled* cxxjde;
  OODE* cxxde = OODE::__the_running_oode;
  if ( ! cxxde ) {
    cxxjde = new OODE_Java_Enabled();
    OODE_ERROR_WHEN_THEN( MALLOC, !cxxjde, return JNI_FALSE );
    if ( ! cxxjde->create( (int)config ) ) {
      delete cxxjde;
      return JNI_FALSE;
    }
    cxxjde->__java_created = TRUE;
  } else {
    OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			     cxxde->__java_enabled, return JNI_FALSE );
    cxxjde = (OODE_Java_Enabled*)cxxde;
  }
  cxxjde->__java_this = env->NewGlobalRef( obj );
  OODE_set__cxxjde( env, obj, cxxjde );
  return JNI_TRUE;
}

//-----------------------------------------------------------------------------

JNIEXPORT jint JNICALL Java_dataexchange_OODE_configuration
  ( JNIEnv* env, jobject obj )
{
  OODE_Java_Enabled* cxxjde = OODE_get__cxxjde( env, obj );
  if ( ! cxxjde )
    return (jint)OODE::NOT_RUNNING;
  else
    return (jint)cxxjde->configuration();
}

//-----------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_dataexchange_OODE_close
  ( JNIEnv* env, jobject obj )
{
  OODE_Java_Enabled* cxxjde = OODE_get__cxxjde( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjde, return );
  OODE_ERROR_WHEN_THEN( PARANOID,
			 ! env->IsSameObject( obj, cxxjde->__java_this ),
			 return );
  OODE_set__cxxjde( env, obj, NULL );
  env->DeleteGlobalRef( cxxjde->__java_this );
  cxxjde->__java_this = NULL;
  if ( cxxjde->__java_created ) {
    cxxjde->close();
    delete cxxjde;
  }
}

//-----------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_dataexchange_OODE_init_1block_1check
  ( JNIEnv* env, jobject obj )
{
  OODE_Java_Enabled* cxxjde = OODE_get__cxxjde( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjde, return );
  cxxjde->init_block_check();
}

//-----------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_dataexchange_OODE_print_1version
  ( JNIEnv*, jclass )
{
  OODE_Java_Enabled::print_version();
}

//-----------------------------------------------------------------------------

JNIEXPORT jshort JNICALL Java_dataexchange_OODE_num_1exchanges
  ( JNIEnv* env, jobject obj )
{
  OODE_Java_Enabled* cxxjde = OODE_get__cxxjde( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjde,
			 return (jshort)OODExchange_Array::BAD_INDEX );
  return (jshort)cxxjde->num_exchanges();
}

//-----------------------------------------------------------------------------

JNIEXPORT jobject JNICALL Java_dataexchange_OODE_get_1exchange
  ( JNIEnv* env, jobject obj, jshort index )
{
  OODE_Java_Enabled* cxxjde = OODE_get__cxxjde( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjde, return NULL );
  OODExchange* cxxdexchange = cxxjde->get_exchange( (short)index );
  if ( ! cxxdexchange ) return NULL;
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE,
			 ! cxxdexchange->__java_enabled, return NULL );
  OODExchange_Java_Enabled* cxxjdexchange = (OODExchange_Java_Enabled*)cxxdexchange;
  jobject result = cxxjdexchange->__java_this;
  if ( ! result ) {
    result = OODE__exchange_factory( env, obj );
    OODE_ERROR_WHEN_THEN( MALLOC, ! result, return NULL );
    cxxjdexchange->__java_this = env->NewGlobalRef( result );
    OODExchange_set__cxxjdexchange( env, result, cxxjdexchange );
  }
  return result;
}

//=============================================================================
//=============================================================================

JNIEXPORT void JNICALL Java_dataexchange_OODExchange_OODE_1_1construct
  ( JNIEnv* env, jobject obj, jobject /*jde*/ )
{
  if ( OODExchange_Java_Enabled::__java_obj_count == 0 )
    OODExchange_INIT_IDS( env, obj );
  OODE_ERROR_WHEN( INTERNAL, OODExchange_Java_Enabled::__java_obj_count >= UINT_MAX );
  ++ OODExchange_Java_Enabled::__java_obj_count;
}

//-----------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_dataexchange_OODExchange_OODE_1_1destruct
  ( JNIEnv* env, jobject obj )
{
  -- OODExchange_Java_Enabled::__java_obj_count;
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  if ( cxxjdexchange ) {
    if ( cxxjdexchange->__java_created )
      delete cxxjdexchange;
  }
}

//-----------------------------------------------------------------------------

JNIEXPORT jboolean JNICALL Java_dataexchange_OODExchange_create
  ( JNIEnv* env, jobject obj )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, ! cxxjdexchange, return JNI_FALSE );
  OODE_Java_Enabled* cxxjde = OODExchange_get__cxxjde( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjde, return JNI_FALSE );
  OODExchange* cxxdexchange = cxxjde->exchange_factory();
  OODE_ERROR_WHEN_THEN( MALLOC, ! cxxdexchange, return JNI_FALSE );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   cxxdexchange->__java_enabled, return JNI_FALSE );
  cxxjdexchange = (OODExchange_Java_Enabled*)cxxdexchange;
  cxxjdexchange->__java_this = env->NewGlobalRef( obj );
  OODExchange_set__cxxjdexchange( env, obj, cxxjdexchange );
  if ( ! cxxjdexchange->create() ) {
    delete cxxjdexchange;
    return JNI_FALSE;
  }
  cxxjdexchange->__java_created = TRUE;
  return JNI_TRUE;
}

//-----------------------------------------------------------------------------

JNIEXPORT jboolean JNICALL Java_dataexchange_OODExchange_running
  ( JNIEnv* env, jobject obj )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  if ( cxxjdexchange && cxxjdexchange->running() )
    return JNI_TRUE;
  else
    return JNI_FALSE;
}

//-----------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_dataexchange_OODExchange_close
  ( JNIEnv* env, jobject obj )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  OODE_ERROR_UNLESS( WARNING, cxxjdexchange->__java_created );
  OODExchange_set__cxxjdexchange( env, obj, NULL );
  env->DeleteGlobalRef( cxxjdexchange->__java_this );
  cxxjdexchange->__java_this = NULL;
  cxxjdexchange->close();
  if ( cxxjdexchange->__java_created )
    delete cxxjdexchange;
}

//-----------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_dataexchange_OODExchange_change_1default_1blocking
  ( JNIEnv* env, jobject obj, jboolean block )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  cxxjdexchange->change_default_blocking( block );
}

//-----------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_dataexchange_OODExchange_change_1default_1forwarding
  ( JNIEnv* env, jobject obj, jint style )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  cxxjdexchange->change_default_forwarding( style );
}

//-----------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_dataexchange_OODExchange_set_1format_1forwarding
  ( JNIEnv* env, jobject obj, jstring format, jint style )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  OODE_PARANOID_GetStringUTFChars( env, format );
  cxxjdexchange->set_format_forwarding( (char*)format_utf, style );
  env->ReleaseStringUTFChars( format, format_utf );
}

//-----------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_dataexchange_OODExchange_set_1format_1fixed
  ( JNIEnv* env, jobject obj, jstring format, jboolean fixed )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  OODE_PARANOID_GetStringUTFChars( env, format );
  cxxjdexchange->set_format_fixed( (char*)format_utf, fixed );
  env->ReleaseStringUTFChars( format, format_utf );
}

//-----------------------------------------------------------------------------

JNIEXPORT jstring JNICALL Java_dataexchange_OODExchange_listen__Ljava_lang_String_2Ljava_lang_String_2Ljava_lang_String_2I
  ( JNIEnv* env, jobject obj,
    jstring user, jstring app, jstring group, jint timeout )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return NULL );
  OODE_PARANOID_GetStringUTFChars( env, user );
  OODE_PARANOID_GetStringUTFChars( env, app );
  OODE_PARANOID_GetStringUTFChars( env, group );
  char* result
    = cxxjdexchange->listen( (char*)user_utf, (char*)app_utf,
			     (char*)group_utf, (int)timeout );
  env->ReleaseStringUTFChars( user,  user_utf );
  env->ReleaseStringUTFChars( app,   app_utf );
  env->ReleaseStringUTFChars( group, group_utf );
  return env->NewStringUTF( result );
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODExchange_listen__Ljava_lang_String_2
  ( JNIEnv* env, jobject obj, jstring unix_socket )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return JNI_FALSE );
  OODE_PARANOID_GetStringUTFChars( env, unix_socket );
  jboolean result = cxxjdexchange->listen( (char*)unix_socket_utf );
  env->ReleaseStringUTFChars( unix_socket, unix_socket_utf );
  return result;
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODExchange_listen__J
  ( JNIEnv* env, jobject obj, jlong inet_port )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return JNI_FALSE );
  return cxxjdexchange->listen( (int)inet_port );
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODExchange_listen__
  ( JNIEnv* env, jobject obj )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return JNI_FALSE );
  return cxxjdexchange->listen();
}

//-----------------------------------------------------------------------------

JNIEXPORT void JNICALL Java_dataexchange_OODExchange_enable_1comment_1handling
  ( JNIEnv* env, jobject obj )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  cxxjdexchange->enable_comment_handling();
}

//-----------------------------------------------------------------------------

JNIEXPORT jint JNICALL Java_dataexchange_OODExchange_poll_1and_1handle__Z
  ( JNIEnv* env, jobject obj, jboolean block )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return 0 );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   ! OODExchange_Java_Enabled::__polling_env, return 0 );
  OODExchange_Java_Enabled::__polling_env = env; OODExchange_Java_Enabled::__polling_obj = obj;
  jint result = cxxjdexchange->poll_and_handle( block );
  OODExchange_Java_Enabled::__polling_env = NULL;OODExchange_Java_Enabled::__polling_obj = NULL;
  return result;
}

JNIEXPORT jint JNICALL Java_dataexchange_OODExchange_poll_1and_1handle__II
  ( JNIEnv* env, jobject obj, jint secs, jint usecs )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return 0 );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   ! OODExchange_Java_Enabled::__polling_env, return 0 );
  OODExchange_Java_Enabled::__polling_env = env; OODExchange_Java_Enabled::__polling_obj = obj;
  jint result = cxxjdexchange->poll_and_handle( secs, usecs );
  OODExchange_Java_Enabled::__polling_env = NULL; OODExchange_Java_Enabled::__polling_obj = NULL;
  return result;
}

//-----------------------------------------------------------------------------

JNIEXPORT jstring JNICALL Java_dataexchange_OODExchange_exchange_1name
  ( JNIEnv* env, jobject obj )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return NULL );
  char* result = cxxjdexchange->exchange_name();
  return env->NewStringUTF( result );
}

JNIEXPORT jstring JNICALL Java_dataexchange_OODExchange_host_1name
  ( JNIEnv* env, jobject obj )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return NULL );
  char* result = cxxjdexchange->host_name();
  return env->NewStringUTF( result );
}

JNIEXPORT jlong JNICALL Java_dataexchange_OODExchange_inet_1port
  ( JNIEnv* env, jobject obj )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return -1 );
  return cxxjdexchange->inet_port();
}

//-----------------------------------------------------------------------------

JNIEXPORT jint JNICALL Java_dataexchange_OODExchange_num_1ports
  ( JNIEnv* env, jobject obj )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange,
			 return (jshort)OODEPort_Array::BAD_INDEX );
  return cxxjdexchange->num_ports();
}

//-----------------------------------------------------------------------------

JNIEXPORT jobject JNICALL Java_dataexchange_OODExchange_get_1port
  ( JNIEnv* env, jobject obj, jint index )
{
  OODExchange_Java_Enabled* cxxjdexchange = OODExchange_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return NULL );
  OODEPort* cxxdeport = cxxjdexchange->get_port( (short)index );
  if ( ! cxxdeport ) return NULL;
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE,
			 ! cxxdeport->__java_enabled, return NULL );
  OODEPort_Java_Enabled* cxxjdeport = (OODEPort_Java_Enabled*)cxxdeport;
  jobject result = cxxjdeport->__java_this;
  if ( ! result ) {
    result = OODExchange__port_factory( env, obj );
    OODE_ERROR_WHEN_THEN( MALLOC, !result, return NULL );
    cxxjdeport->__java_this = env->NewGlobalRef( result );
    OODEPort_set__cxxjdeport( env, result, cxxjdeport );
  }
  return result;
}

//=============================================================================
//=============================================================================

JNIEXPORT void JNICALL Java_dataexchange_OODEPort_OODE_1_1construct
  ( JNIEnv* env, jobject obj, jobject /*jdexchange*/ )
{
  if ( OODEPort_Java_Enabled::__java_obj_count == 0 )
    OODEPort_INIT_IDS( env, obj );
  OODE_ERROR_WHEN( INTERNAL, OODEPort_Java_Enabled::__java_obj_count >= UINT_MAX );
  ++ OODEPort_Java_Enabled::__java_obj_count;
}

JNIEXPORT void JNICALL Java_dataexchange_OODEPort_OODE_1_1destruct
  ( JNIEnv* env, jobject obj )
{
  -- OODEPort_Java_Enabled::__java_obj_count;
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  if ( cxxjdeport ) {
    if ( cxxjdeport->__java_created )
      delete cxxjdeport;
  }
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODEPort_connect__Ljava_lang_String_2Ljava_lang_String_2Ljava_lang_String_2
  ( JNIEnv* env, jobject obj, jstring user, jstring app, jstring group )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, ! cxxjdeport, return JNI_FALSE );
  OODExchange_Java_Enabled* cxxjdexchange = OODEPort_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return JNI_FALSE );
  OODEPort* cxxdeport = cxxjdexchange->port_factory();
  OODE_ERROR_WHEN_THEN( MALLOC, ! cxxdeport, return JNI_FALSE );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   cxxdeport->__java_enabled, return JNI_FALSE );
  cxxjdeport = (OODEPort_Java_Enabled*)cxxdeport;
  cxxjdeport->__java_this = env->NewGlobalRef( obj );
  OODEPort_set__cxxjdeport( env, obj, cxxjdeport );
  OODE_PARANOID_GetStringUTFChars( env, user );
  OODE_PARANOID_GetStringUTFChars( env, app );
  OODE_PARANOID_GetStringUTFChars( env, group );
  int result
    = cxxjdeport->connect( (char*)user_utf,(char*)app_utf,(char*)group_utf );
  env->ReleaseStringUTFChars( user,  user_utf );
  env->ReleaseStringUTFChars( app,   app_utf );
  env->ReleaseStringUTFChars( group, group_utf );
  if ( ! result ) {
    delete cxxjdeport;
    return JNI_FALSE;
  }
  cxxjdeport->__java_created = TRUE;
  return JNI_TRUE;
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODEPort_connect__Ljava_lang_String_2J
  ( JNIEnv* env, jobject obj, jstring host, jlong port )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, ! cxxjdeport, return JNI_FALSE );
  OODExchange_Java_Enabled* cxxjdexchange = OODEPort_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return JNI_FALSE );
  OODEPort* cxxdeport = cxxjdexchange->port_factory();
  OODE_ERROR_WHEN_THEN( MALLOC, ! cxxdeport, return JNI_FALSE );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   cxxdeport->__java_enabled, return JNI_FALSE );
  cxxjdeport = (OODEPort_Java_Enabled*)cxxdeport;
  cxxjdeport->__java_this = env->NewGlobalRef( obj );
  OODEPort_set__cxxjdeport( env, obj, cxxjdeport );
  OODE_PARANOID_GetStringUTFChars( env, host );
  int result
    = cxxjdeport->connect( (char*)host_utf, (long)port );
  env->ReleaseStringUTFChars( host, host_utf );
  if ( ! result ) {
    delete cxxjdeport;
    return JNI_FALSE;
  }
  cxxjdeport->__java_created = TRUE;
  return JNI_TRUE;
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODEPort_connect__Ljava_lang_String_2
  ( JNIEnv* env, jobject obj, jstring usock )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, ! cxxjdeport, return JNI_FALSE );
  OODExchange_Java_Enabled* cxxjdexchange = OODEPort_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return JNI_FALSE );
  OODEPort* cxxdeport = cxxjdexchange->port_factory();
  OODE_ERROR_WHEN_THEN( MALLOC, ! cxxdeport, return JNI_FALSE );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   cxxdeport->__java_enabled, return JNI_FALSE );
  cxxjdeport = (OODEPort_Java_Enabled*)cxxdeport;
  cxxjdeport->__java_this = env->NewGlobalRef( obj );
  OODEPort_set__cxxjdeport( env, obj, cxxjdeport );
  OODE_PARANOID_GetStringUTFChars( env, usock );
  int result
    = cxxjdeport->connect( (char*)usock_utf );
  env->ReleaseStringUTFChars( usock, usock_utf );
  if ( ! result ) {
    delete cxxjdeport;
    return JNI_FALSE;
  }
  cxxjdeport->__java_created = TRUE;
  return JNI_TRUE;
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODEPort_read
  ( JNIEnv* env, jobject obj, jstring file )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, ! cxxjdeport, return JNI_FALSE );
  OODExchange_Java_Enabled* cxxjdexchange = OODEPort_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return JNI_FALSE );
  OODEPort* cxxdeport = cxxjdexchange->port_factory();
  OODE_ERROR_WHEN_THEN( MALLOC, ! cxxdeport, return JNI_FALSE );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   cxxdeport->__java_enabled, return JNI_FALSE );
  cxxjdeport = (OODEPort_Java_Enabled*)cxxdeport;
  cxxjdeport->__java_this = env->NewGlobalRef( obj );
  OODEPort_set__cxxjdeport( env, obj, cxxjdeport );
  OODE_PARANOID_GetStringUTFChars( env, file );
  int result = cxxjdeport->read( (char*)file_utf );
  env->ReleaseStringUTFChars( file, file_utf );
  if ( ! result ) {
    delete cxxjdeport;
    return JNI_FALSE;
  }
  cxxjdeport->__java_created = TRUE;
  return JNI_TRUE;
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODEPort_write
  ( JNIEnv* env, jobject obj, jstring file )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE, ! cxxjdeport, return JNI_FALSE );
  OODExchange_Java_Enabled* cxxjdexchange = OODEPort_get__cxxjdexchange( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return JNI_FALSE );
  OODEPort* cxxdeport = cxxjdexchange->port_factory();
  OODE_ERROR_WHEN_THEN( MALLOC, ! cxxdeport, return JNI_FALSE );
  OODE_ERROR_UNLESS_THEN( LIBRARY_STATE,
			   cxxdeport->__java_enabled, return JNI_FALSE );
  cxxjdeport = (OODEPort_Java_Enabled*)cxxdeport;
  cxxjdeport->__java_this = env->NewGlobalRef( obj );
  OODEPort_set__cxxjdeport( env, obj, cxxjdeport );
  OODE_PARANOID_GetStringUTFChars( env, file );
  int result = cxxjdeport->write( (char*)file_utf );
  env->ReleaseStringUTFChars( file, file_utf );
  if ( ! result ) {
    delete cxxjdeport;
    return JNI_FALSE;
  }
  cxxjdeport->__java_created = TRUE;
  return JNI_TRUE;
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODEPort_attached
  ( JNIEnv* env, jobject obj )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  if ( cxxjdeport && cxxjdeport->attached() )
    return JNI_TRUE;
  else
    return JNI_FALSE;
}

JNIEXPORT void JNICALL Java_dataexchange_OODEPort_close
  ( JNIEnv* env, jobject obj )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdeport, return );
  OODE_ERROR_UNLESS( WARNING, cxxjdeport->__java_created );
  OODEPort_set__cxxjdeport( env, obj, NULL );
  env->DeleteGlobalRef( cxxjdeport->__java_this );
  cxxjdeport->__java_this = NULL;
  cxxjdeport->close();
  if ( cxxjdeport->__java_created )
    delete cxxjdeport;
}

JNIEXPORT void JNICALL Java_dataexchange_OODEPort_comment
  ( JNIEnv* env, jobject obj, jstring str )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdeport, return );
  OODE_PARANOID_GetStringUTFChars( env, str );
  cxxjdeport->comment( (char*)str_utf );
  env->ReleaseStringUTFChars( str, str_utf );
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODEPort_shutdown
  ( JNIEnv* env, jobject obj, jint flag, jstring str )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdeport, return JNI_FALSE );
  OODE_PARANOID_GetStringUTFChars( env, str );
  int result = cxxjdeport->shutdown( (int)flag, (char*)str_utf );
  env->ReleaseStringUTFChars( str, str_utf );
  return (result) ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODEPort_set_1format_1blocking
  ( JNIEnv* env, jobject obj, jstring str, jboolean flag )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdeport, return JNI_FALSE );
  OODE_PARANOID_GetStringUTFChars( env, str );
  int result = cxxjdeport->set_format_blocking( (char*)str_utf, (int)flag );
  env->ReleaseStringUTFChars( str, str_utf );
  return (result) ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT jstring JNICALL Java_dataexchange_OODEPort_port_1name
  ( JNIEnv* env, jobject obj )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdeport, return NULL );
  char* result = cxxjdeport->port_name();
  return env->NewStringUTF( result );
}

JNIEXPORT jstring JNICALL Java_dataexchange_OODEPort_host_1name
  ( JNIEnv* env, jobject obj )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdeport, return NULL );
  char* result = cxxjdeport->host_name();
  return env->NewStringUTF( result );
}

JNIEXPORT jlong JNICALL Java_dataexchange_OODEPort_port_1number
  ( JNIEnv* env, jobject obj )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdeport, return -1 );
  return (jlong)cxxjdeport->port_number();
}

JNIEXPORT jobject JNICALL Java_dataexchange_OODEPort_exchange
  ( JNIEnv* env, jobject obj )
{
  OODEPort_Java_Enabled* cxxjdeport = OODEPort_get__cxxjdeport( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdeport, return NULL );
  OODExchange* cxxdexchange = cxxjdeport->exchange();
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxdexchange, return NULL );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE,
			 ! cxxdexchange->__java_enabled, return NULL );
  OODExchange_Java_Enabled* cxxjdexchange = (OODExchange_Java_Enabled*)cxxdexchange;
  jobject result = cxxjdexchange->__java_this;
  if ( ! result ) {
    OODE* cxxde = OODE::__the_running_oode;
    OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxde, return NULL );
    OODE_ERROR_UNLESS_THEN( PARANOID, cxxde->__java_enabled, return NULL );
    OODE_Java_Enabled* cxxjde = (OODE_Java_Enabled*)cxxde;
    result = OODE__exchange_factory( env, cxxjde->__java_this );
    OODE_ERROR_WHEN_THEN( MALLOC, ! result, return NULL );
    cxxjdexchange->__java_this = env->NewGlobalRef( result );
    OODExchange_set__cxxjdexchange( env, result, cxxjdexchange );
  }
  return result;
}

//=============================================================================
//=============================================================================

JNIEXPORT void JNICALL Java_dataexchange_OODEMessage_OODE_1_1construct
  ( JNIEnv* env, jobject obj )
{
  if ( OODEMessage_Java_Enabled::__java_obj_count == 0 )
    OODEMessage_INIT_IDS( env, obj );
  OODE_ERROR_WHEN( INTERNAL, OODEMessage_Java_Enabled::__java_obj_count >= UINT_MAX );
  ++ OODEMessage_Java_Enabled::__java_obj_count;
}

JNIEXPORT void JNICALL Java_dataexchange_OODEMessage_OODE_1_1destruct
  ( JNIEnv* env, jobject obj )
{
  -- OODEMessage_Java_Enabled::__java_obj_count;
  OODEMessage_Java_Enabled* cxxjdemessage = OODEMessage_get__cxxjdemessage( env, obj );
  if ( cxxjdemessage ) {
    if ( cxxjdemessage->__java_created )
      delete cxxjdemessage;
  }
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODEMessage_allocate_1buffer
  ( JNIEnv* env, jobject obj )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return JNI_FALSE );
  return (cxxjdemessage->allocate_buffer()) ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT void JNICALL Java_dataexchange_OODEMessage_free_1buffer
  ( JNIEnv* env, jobject obj )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return );
  cxxjdemessage->free_buffer();
}

JNIEXPORT void JNICALL Java_dataexchange_OODEMessage_enable_1filtering
  ( JNIEnv* env, jobject obj, jobject jexch )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return );
  OODExchange_Java_Enabled* cxxjdexchange
    = OODExchange_get__cxxjdexchange( env, jexch );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  cxxjdemessage->enable_filtering( cxxjdexchange );
}

JNIEXPORT void JNICALL Java_dataexchange_OODEMessage_disable_1filtering
  ( JNIEnv* env, jobject obj, jobject jexch )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return );
  OODExchange_Java_Enabled* cxxjdexchange
    = OODExchange_get__cxxjdexchange( env, jexch );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  cxxjdemessage->disable_filtering( cxxjdexchange );
}

JNIEXPORT void JNICALL Java_dataexchange_OODEMessage_enable_1processing
  ( JNIEnv* env, jobject obj, jobject jexch )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return );
  OODExchange_Java_Enabled* cxxjdexchange
    = OODExchange_get__cxxjdexchange( env, jexch );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  cxxjdemessage->enable_processing( cxxjdexchange );
}

JNIEXPORT void JNICALL Java_dataexchange_OODEMessage_disable_1processing
  ( JNIEnv* env, jobject obj, jobject jexch )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return );
  OODExchange_Java_Enabled* cxxjdexchange
    = OODExchange_get__cxxjdexchange( env, jexch );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  cxxjdemessage->disable_processing( cxxjdexchange );
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODEMessage_send
  ( JNIEnv* env, jobject obj, jobject jport )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return JNI_FALSE );
  OODEPort_Java_Enabled* cxxjdeport
    = OODEPort_get__cxxjdeport( env, jport );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdeport, return JNI_FALSE );
  return (cxxjdemessage->send( cxxjdeport )) ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT void JNICALL Java_dataexchange_OODEMessage_forward
  ( JNIEnv* env, jobject obj, jobject jport )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return );
  OODEPort_Java_Enabled* cxxjdeport
    = OODEPort_get__cxxjdeport( env, jport );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdeport, return );
  cxxjdemessage->forward( cxxjdeport );
}

JNIEXPORT void JNICALL Java_dataexchange_OODEMessage_register_1format
  ( JNIEnv* env, jobject obj, jobject jexch )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return );
  OODExchange_Java_Enabled* cxxjdexchange
    = OODExchange_get__cxxjdexchange( env, jexch );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  cxxjdemessage->register_format( cxxjdexchange );
}

JNIEXPORT void JNICALL Java_dataexchange_OODEMessage_set_1format_1fixed
  ( JNIEnv* env, jobject obj, jobject jexch, jboolean flag )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return );
  OODExchange_Java_Enabled* cxxjdexchange
    = OODExchange_get__cxxjdexchange( env, jexch );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  cxxjdemessage->set_format_fixed( cxxjdexchange, flag );
}

JNIEXPORT void JNICALL Java_dataexchange_OODEMessage_set_1format_1forwarding
  ( JNIEnv* env, jobject obj, jobject jexch, jint style )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return );
  OODExchange_Java_Enabled* cxxjdexchange
    = OODExchange_get__cxxjdexchange( env, jexch );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdexchange, return );
  cxxjdemessage->set_format_forwarding( cxxjdexchange, style );
}

JNIEXPORT jboolean JNICALL Java_dataexchange_OODEMessage_set_1format_1blocking
  ( JNIEnv* env, jobject obj, jobject jport, jboolean flag )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return JNI_FALSE );
  OODEPort_Java_Enabled* cxxjdeport
    = OODEPort_get__cxxjdeport( env, jport );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdeport, return JNI_FALSE );
  return (cxxjdemessage->set_format_blocking( cxxjdeport, flag ))
    ? JNI_TRUE : JNI_FALSE;
}

JNIEXPORT void JNICALL Java_dataexchange_OODEMessage_take_1buffer
  ( JNIEnv* env, jobject obj )
{
  OODEMessage_Java_Enabled* cxxjdemessage
    = OODEMessage_get__cxxjdemessage( env, obj );
  OODE_ERROR_WHEN_THEN( LIBRARY_STATE, ! cxxjdemessage, return );
  cxxjdemessage->take_buffer();
}

//#############################################################################
} // ends extern "C"
//#############################################################################

JNINativeMethod OODE_Java_Enabled::native_methods[OODE_Java_Enabled::num_methods] = {
  { "OODE__construct", "()V", Java_dataexchange_OODE_OODE_1_1construct },
  { "OODE__destruct", "()V", Java_dataexchange_OODE_OODE_1_1destruct },
  { "create", "(I)Z", Java_dataexchange_OODE_create },
  { "configuration", "()I", Java_dataexchange_OODE_configuration },
  { "close", "()V", Java_dataexchange_OODE_close },
  { "init_block_check", "()V", Java_dataexchange_OODE_init_1block_1check },
  { "print_version", "()V", Java_dataexchange_OODE_print_1version },
  { "num_exchanges", "()S", Java_dataexchange_OODE_num_1exchanges },
  { "get_exchange", "(S)Ldataexchange/OODExchange;",
    Java_dataexchange_OODE_get_1exchange },
};

JNINativeMethod OODExchange_Java_Enabled::native_methods[OODExchange_Java_Enabled::num_methods] = {
  { "OODE__construct", "(Ldataexchange/OODE;)V",
    Java_dataexchange_OODExchange_OODE_1_1construct },
  { "OODE__destruct", "()V", Java_dataexchange_OODExchange_OODE_1_1destruct },
  { "create", "()Z", Java_dataexchange_OODExchange_create },
  { "running", "()Z", Java_dataexchange_OODExchange_running },
  { "close", "()V", Java_dataexchange_OODExchange_close },
  { "change_default_blocking", "(Z)V",
    Java_dataexchange_OODExchange_change_1default_1blocking },
  { "change_default_forwarding", "(I)V",
    Java_dataexchange_OODExchange_change_1default_1forwarding },
  { "set_format_forwarding", "(Ljava/lang/String;I)V",
    Java_dataexchange_OODExchange_set_1format_1forwarding },
  { "set_format_fixed", "(Ljava/lang/String;Z)V",
    Java_dataexchange_OODExchange_set_1format_1fixed },
  { "listen",
    "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;I)Ljava/lang/String;",
    Java_dataexchange_OODExchange_listen__Ljava_lang_String_2Ljava_lang_String_2Ljava_lang_String_2I },
  { "listen", "(Ljava/lang/String;)Z",
    Java_dataexchange_OODExchange_listen__Ljava_lang_String_2 },
  { "listen", "(J)Z", Java_dataexchange_OODExchange_listen__J },
  { "listen", "()Z", Java_dataexchange_OODExchange_listen__ },
  { "enable_comment_handling", "()V",
    Java_dataexchange_OODExchange_enable_1comment_1handling },
  { "poll_and_handle", "(Z)I",
    Java_dataexchange_OODExchange_poll_1and_1handle__Z },
  { "poll_and_handle", "(II)I",
    Java_dataexchange_OODExchange_poll_1and_1handle__II },
  { "exchange_name", "()Ljava/lang/String;",
    Java_dataexchange_OODExchange_exchange_1name },
  { "host_name", "()Ljava/lang/String;",
    Java_dataexchange_OODExchange_host_1name },
  { "inet_port", "()J", Java_dataexchange_OODExchange_inet_1port },
  { "num_ports", "()I", Java_dataexchange_OODExchange_num_1ports },
  { "get_port", "(I)Ldataexchange/OODEPort;",
    Java_dataexchange_OODExchange_get_1port },
};

JNINativeMethod OODEPort_Java_Enabled::native_methods[OODEPort_Java_Enabled::num_methods] = {
  { "OODE__construct", "(Ldataexchange/OODExchange;)V",
    Java_dataexchange_OODEPort_OODE_1_1construct },
  { "OODE__destruct", "()V", Java_dataexchange_OODEPort_OODE_1_1destruct },
  { "connect", "(Ljava/lang/String;Ljava/lang/String;Ljava/lang/String;)Z",
    Java_dataexchange_OODEPort_connect__Ljava_lang_String_2Ljava_lang_String_2Ljava_lang_String_2 },
  { "connect", "(Ljava/lang/String;J)Z",
    Java_dataexchange_OODEPort_connect__Ljava_lang_String_2J },
  { "connect", "(Ljava/lang/String;)Z",
    Java_dataexchange_OODEPort_connect__Ljava_lang_String_2 },
  { "read", "(Ljava/lang/String;)Z", Java_dataexchange_OODEPort_read },
  { "write", "(Ljava/lang/String;)Z", Java_dataexchange_OODEPort_write },
  { "attached", "()Z", Java_dataexchange_OODEPort_attached },
  { "close", "()V", Java_dataexchange_OODEPort_close },
  { "comment", "(Ljava/lang/String;)V", Java_dataexchange_OODEPort_comment },
  { "shutdown", "(ILjava/lang/String;)Z",
    Java_dataexchange_OODEPort_shutdown },
  { "set_format_blocking", "(Ljava/lang/String;Z)Z", 
    Java_dataexchange_OODEPort_set_1format_1blocking },
  { "port_name", "()Ljava/lang/String;",
    Java_dataexchange_OODEPort_port_1name },
  { "host_name", "()Ljava/lang/String;",
    Java_dataexchange_OODEPort_host_1name },
  { "port_number", "()J", Java_dataexchange_OODEPort_port_1number },
  { "exchange", "()Ldataexchange/OODExchange;",
    Java_dataexchange_OODEPort_exchange },
};

JNINativeMethod OODEMessage_Java_Enabled::native_methods[OODEMessage_Java_Enabled::num_methods] = {
  { "OODE__construct", "()V",
    Java_dataexchange_OODEMessage_OODE_1_1construct },
  { "OODE__destruct", "()V", Java_dataexchange_OODEMessage_OODE_1_1destruct },
  { "allocate_buffer", "()Z", Java_dataexchange_OODEMessage_allocate_1buffer },
  { "free_buffer", "()V", Java_dataexchange_OODEMessage_free_1buffer },
  { "enable_filtering", "(Ldataexchange/OODExchange;)V",
    Java_dataexchange_OODEMessage_enable_1filtering },
  { "disable_filtering", "(Ldataexchange/OODExchange;)V",
    Java_dataexchange_OODEMessage_enable_1processing },
  { "enable_processing", "(Ldataexchange/OODExchange;)V",
    Java_dataexchange_OODEMessage_enable_1processing },
  { "disable_processing", "(Ldataexchange/OODExchange;)V",
    Java_dataexchange_OODEMessage_disable_1processing },
  { "send", "(Ldataexchange/OODEPort;)Z", Java_dataexchange_OODEMessage_send },
  { "forward", "(Ldataexchange/OODEPort;)V",
    Java_dataexchange_OODEMessage_forward },
  { "register_format", "(Ldataexchange/OODExchange;)V",
    Java_dataexchange_OODEMessage_register_1format },
  { "set_format_fixed", "(Ldataexchange/OODExchange;Z)V",
    Java_dataexchange_OODEMessage_set_1format_1fixed },
  { "set_format_forwarding", "(Ldataexchange/OODExchange;I)V",
    Java_dataexchange_OODEMessage_set_1format_1forwarding },
  { "set_format_blocking", "(Ldataexchange/OODEPort;Z)Z",
    Java_dataexchange_OODEMessage_set_1format_1blocking },
  { "take_buffer", "()V", Java_dataexchange_OODEMessage_take_1buffer },
};

#endif
