#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/PushB.h>
#include <Xm/ToggleB.h>
#include <Xm/Label.h>
#include <Xm/RowColumn.h>
#include <Xm/DrawingA.h>
#include <Xm/ScrollBar.h>
#include <Xm/MenuShell.h>
#include <X11/Xaw/SmeBSB.h>


#include "Dial.h"
#include <time.h>
#include <math.h>
#include <assert.h>
#include <string.h>
#include <stdlib.h>
#include <malloc.h>
#include <stdio.h>

#include "../config.h"
#include "gen_thread.h"
#include "io.h"
#include "DE.h"
#include "comm_group.h"
#include "otl.h"
#include "ball.h"
#include "ball_mon.h"
#include "ball_mon_COBS_.h"
#ifndef FPRINTF_DEFINED
extern int fprintf ARGS((FILE *, const char *, ...));
#endif
#ifndef PRINTF_DEFINED
extern int printf ARGS((const char *, ...));
#endif

/*
** typedefs which will be of use...
*/
typedef enum {
    Triangle,
    Square,
    Pentagon,
    Hexagon,
    Octagon,
    Circle,
    Gumby
} shape_type;

typedef struct {
    Boolean	drawn;
    shape_type	shape;
    Boolean	fill;
    unsigned int size;
    float	x, y;
    float	dir_x, dir_y;
} obj, *obj_ptr;

typedef struct {
    char	*name;
    void	(*func)();
    caddr_t	data;
    char	*fg;
    Pixmap	pixmap;
} menu_struct;

#ifndef Max
#define Max(x, y)       (((x) > (y)) ? (x) : (y))
#define Min(x, y)       (((x) > (y)) ? (y) : (x))
#endif

/*
** forward declarations
*/
static void make_commands_form(Widget parent);
static void make_controls_form(Widget parent);
static void fill_toggle();
static void outline_toggle();
static void set_fill_toggles(Widget fill, Widget other);
static void quit_callback();
static void change_speed_CB();
static void change_size();
static void change_size_CB();
static void change_direction(int dial_value);
static void direction_select();
static void expose_obj();
static void resize_canvas();
static void init_obj();
static void init_object_forms();
static void draw_obj_at(shape_type shape, Position x, Position y, 
			Boolean fill, int size, GC gc, Drawable win);
static void redraw_object(Boolean force_redraw);
static void draw_object(obj_ptr obj, GC gc);
static Boolean blt_in_object(obj_ptr last, obj_ptr current, Boolean force_redraw);
static Boolean CvtStringToStringTable(Display *display, XrmValue *args, 
				      Cardinal *num_args, XrmValue *from, 
				      XrmValue *to, XtPointer *conv_data);
extern void set_ball_direction(float angle);
extern void set_ball_size(int size);
extern void set_ball_speed(int speed);


/*
** global variables
*/
static int obj_speed;
static Pixmap shadow_blt = (Pixmap) NULL;
static Boolean fast_mode = True;

static XtAppContext	app_context;
static Boolean bouncing = False;
static Widget canvas = (Widget) NULL, dial, speed, size;
static	Display	*dpy;
static	int	scr;
static Colormap cmap;
obj	current_obj, last_obj;
int dial_min, dial_max;
static int max_object_size;
static int shadow_obj_x, shadow_obj_y;
static Dimension dial_width, dial_height;
static Dimension work_width, work_height;
static GC gc = (GC) NULL;
static GC erase_gc = (GC) NULL;
static double PI = 3.14159265358797323;

/*
**  X application data
*/
typedef struct {
    char **colors;
    char *StopLabel;
    char *StartLabel;
    Boolean blt_control;
} AppData, *AppDataPtr;

char *default_colors[] = {"black", "blue", NULL};
char **defaults = default_colors;

/*
**  the application level resource specification
*/
static XtResource resources[] = {
    {"startLabel", "StartLabel", XtRString, sizeof(char *),
       XtOffset(AppDataPtr, StartLabel), XtRString, "Start"},
    {"stopLabel", "StopLabel", XtRString, sizeof(char *),
       XtOffset(AppDataPtr, StopLabel), XtRString, "Stop"},
    {"bltControl", "BltControl", XtRBoolean, sizeof(Boolean),
       XtOffset(AppDataPtr, blt_control), XtRBoolean, False},
};

/*
**   application command line options:
**    -blt_control False     will result in automatic drawing mode control
*/
XrmOptionDescRec options[] = {
    {"-blt_control", "*bltControl", XrmoptionSepArg, NULL},
};

static AppData	app_data;

char *fallback[] = {
#include "fallback.c"
  NULL,
};

DExchange moss_de = NULL;

static
Boolean
moss_poll()
{
    DExchange_poll_and_handle(moss_de, FALSE);
    return FALSE;
}

void
main(int argc, char **argv)
{
    Widget toplevel, panel;

    Arg	args[5];

    PI = acos(-1.0);
    init_object_forms();

    toplevel = XtAppInitialize(&app_context, "Bounce", 
			       options, XtNumber(options), 
			       &argc, argv,
			       fallback, NULL, 0);

    XtSetTypeConverter(XtRString, XtRStringTable, CvtStringToStringTable,
		       NULL, 0, XtCacheNone, NULL);

    XtGetApplicationResources(toplevel, &app_data, resources,
			      XtNumber(resources), NULL, 0);
    dpy = XtDisplay(toplevel);
    scr = DefaultScreen(dpy);
    cmap = DefaultColormap(dpy, scr);

    panel = XtCreateManagedWidget("panel", xmFormWidgetClass,
				  toplevel, NULL, 0);

    make_commands_form(panel);

    make_controls_form(panel);

    speed = XtCreateManagedWidget("speed", xmScrollBarWidgetClass,
				  panel, NULL, 0);
    XtAddCallback(speed, XmNvalueChangedCallback, change_speed_CB, 0);
    /* get the slider origin in case user changed it */
    XtSetArg(args[0], XmNminimum, &obj_speed);
    XtGetValues(speed, args, 1);

    size = XtCreateManagedWidget("size", xmScrollBarWidgetClass,
				  panel, NULL, 0);
    XtAddCallback(size, XmNvalueChangedCallback, change_size_CB, 0);
    /* get the slider origin in case user changed it */
    XtSetArg(args[0], XmNminimum, &current_obj.size);
    XtSetArg(args[1], XmNmaximum, &max_object_size);
    XtGetValues(size, args, 2);
    /*  max object size is half of slider maximum */
    max_object_size = max_object_size / 2;
    change_size(size, NULL, current_obj.size);

    canvas = XtCreateManagedWidget("canvas", xmDrawingAreaWidgetClass,
				  panel, NULL, 0);
    XtAddCallback(canvas, XmNexposeCallback, expose_obj, 0);
    XtAddCallback(canvas, XmNresizeCallback, resize_canvas, 0);

    resize_canvas(canvas, 0, 0);

    XtRealizeWidget(toplevel);

    init_obj();

    /* initialize a null thread library */
    gen_null_init();

    /* create a dataexchange to use for MOSS and make it single threaded */
    moss_de = DExchange_create();
    DEControlList_set_control_style(DExchange_get_control_list(moss_de),
				    DESingleThreaded);

    /*
     *  Initialize MOSS.  "ball" is used in COBS_init and is the 
     *  application_name for finding a matching group.
     */
    MOSS_Init("ball", moss_de);

    /* 
     * initialize mirror objects defined in ball_mon.idl
     */
    MOSS_ball_mon_Init(moss_de);

    /*
     *  Poll for network events when nothing else to do.
     *  (Really should manage to add network FDs to the XtApplicationContext,
     *  but that requires some extensions to DataExchange.  This approach
     *  is inefficient, but effective.)
     */
    XtAppAddWorkProc(app_context, (XtWorkProc)moss_poll, NULL);

    XtAppMainLoop(app_context);
}

static 
void make_commands_form(Widget parent)
{
    Widget commands, quit;
    Widget fill, outline, fill_box;
    Arg args[4];

    commands = XtCreateManagedWidget("commands", xmRowColumnWidgetClass,
				     parent, NULL, 0);

    quit = XtCreateManagedWidget("quit", xmPushButtonWidgetClass,
				  commands, NULL, 0);
    XtAddCallback(quit, XmNactivateCallback, quit_callback, canvas);

    XtSetArg(args[0], XmNindicatorType, XmONE_OF_MANY);
    fill_box = XtCreateManagedWidget("fill_box", xmRowColumnWidgetClass,
				     commands, args, 1);
    fill = XtCreateManagedWidget("fill", xmToggleButtonWidgetClass,
			 fill_box, args, 0);
    XtSetArg(args[0], XmNset, True);
    outline = XtCreateManagedWidget("outline", xmToggleButtonWidgetClass,
				  fill_box, args, 1);
    XtAddCallback(fill, XmNarmCallback, fill_toggle, outline);
    XtAddCallback(fill, XmNdisarmCallback, fill_toggle, outline);
    XtAddCallback(outline, XmNarmCallback, outline_toggle, fill);
    XtAddCallback(outline, XmNdisarmCallback, outline_toggle, fill);
}

static 
void make_controls_form(Widget parent)
{
    Widget controls, label;
    Arg	args[5];

    controls = XtCreateManagedWidget("controls", xmFormWidgetClass,
				     parent, NULL, 0);
/*    if (app_data.blt_control) {
	Widget blt_box;
	XtSetArg(args[0], XmNindicatorType, XmONE_OF_MANY);
	blt_box = XtCreateManagedWidget("blt_box", xmRowColumnWidgetClass,
				     controls, args, 1);
	XtSetArg(args[0], XmNset, fast_mode);
	fast = XtCreateManagedWidget("blt", xmToggleButtonWidgetClass,
				     blt_box, args, 1);
	XtSetArg(args[0], XmNset, !fast_mode);
	slow = XtCreateManagedWidget("redraw", xmToggleButtonWidgetClass,
				     blt_box, args, 1);
	XtAddCallback(fast, XmNarmCallback, fast_toggle, canvas);
	XtAddCallback(fast, XmNdisarmCallback, fast_toggle, canvas);
	XtAddCallback(slow, XmNarmCallback, slow_toggle, canvas);
	XtAddCallback(slow, XmNdisarmCallback, slow_toggle, canvas);
    }
*/
    dial = XtCreateManagedWidget("direction", dialWidgetClass,
				 controls, NULL, 0);
    XtAddCallback(dial, XtNselect, direction_select, 0);
    /* get the slider origin in case user changed it */
    XtSetArg(args[0], XtNminValue, &dial_min);
    XtSetArg(args[1], XtNmaxValue, &dial_max);
    XtSetArg(args[2], XtNheight, &dial_height);
    XtSetArg(args[3], XtNwidth, &dial_width);
    XtGetValues(dial, args, 4);

    label = XtCreateManagedWidget("dial_label", xmLabelWidgetClass,
				  controls, NULL, 0);
}

/*
**  callback for changing speed.
*/
static 
void change_speed_CB(Widget w, caddr_t client_data, XmScrollBarCallbackStruct *struc)
{
    int slider_val = struc->value;
    obj_speed = slider_val;
    set_ball_speed(slider_val);
}

/*
**  callback for changing size
**  Must ensure that the slider size + slider position doesn't exceed
**   slider maximum.  Max_obj_size is slider_max/2, so just set them both
**   to that if they get too big.
*/
static
void change_size_CB(Widget w, caddr_t client_data, XmScrollBarCallbackStruct *struc)
{
    int slider_val = struc->value;
    change_size(w, client_data, slider_val);
}
		    
static 
void change_size(Widget w, caddr_t client_data, int slider_val)
{
    Arg	args[4];
    int n = 0;
    if (slider_val > max_object_size) {
	XtSetArg(args[n], XmNvalue, max_object_size); n++;
	slider_val = max_object_size;
    }
    XtSetArg(args[n], XmNsliderSize, slider_val); n++;
    XtSetValues(w, args, n);
    current_obj.size = slider_val;

    set_ball_size(current_obj.size);
    if(!bouncing) {
	redraw_object(True);
    }
}

/*
**  callback for direction select.  This uses the X Y coords of the event
**  to determine the apprioriate new dial setting.  It then uses 
**  change_direction() to actually change the object direction and 
**  DialSetValue() to change the value on the dial.
*/
static 
void direction_select(Widget w, caddr_t client_data, XButtonEvent *ev)
{
    double	x_vec, y_vec;
    double angle;
    int dial_value;

    assert((ev->type == ButtonPress) || (ev->type == ButtonRelease));

    if ((ev->x > dial_width) || (ev->y > dial_height)) return;

    x_vec = ev->x - (dial_width / 2);
    y_vec = ev->y - (dial_height / 2);
    angle = atan2(x_vec, y_vec);

    dial_value = (PI - angle) / (2 * PI) * (dial_max -dial_min) + dial_min;

    DialSetValue(w, dial_value);
    change_direction(dial_value);
}

/*
**  change object direction to correspond to a dial value
*/
static void change_direction(int dial_value)
{
    double angle = 360.0*(dial_value - dial_min) / (dial_max - dial_min) - 90.0;

    if (angle < 90.0) angle+=360.0;
    set_ball_direction(angle);
}

/*
**  change object size by monitoring
*/
void outside_change_size(int size_val)
{
    Arg args[4];
    int n = 0;

    if (size_val <= 0) return;
    XtSetArg(args[n], XmNvalue, size_val); n++;
    XtSetValues(size, args, n);
    change_size(size, NULL, size_val);
}

/*
**  change object speed by monitoring
*/
void outside_change_speed(int speed_val)
{
    Arg args[4];
    int n = 0;

    XtSetArg(args[n], XmNvalue, speed_val); n++;
    XtSetValues(speed, args, n);
    obj_speed = speed_val;
}

static
void quit_callback()
{
    exit(0);
}

/*
**	called when fill gets selected or released.
*/
static
void fill_toggle(Widget w, Widget other)
{
    Arg	args[1];
    Boolean fill;

    XtSetArg(args[0], XmNset, &fill);
    XtGetValues(w, args, 1);
    current_obj.fill = fill;
    set_fill_toggles(w, other);
    redraw_object(True);
}

/*
**	called when outline gets selected or released.
*/
static
void outline_toggle(Widget w, Widget other)
{
    Arg	args[1];
    Boolean outline;

    XtSetArg(args[0], XmNset, &outline);
    XtGetValues(w, args, 1);
    current_obj.fill = !outline;
    set_fill_toggles(other, w);
    redraw_object(True);
}

/*
**	set_fill_toggles ensures that fill and outline widgets are
**   set (or unset) as is appropriate for the app state.
**   also switch to appropriate blt mode if not under manual control.
*/
static
void set_fill_toggles(Widget fill, Widget outline)
{
    Arg args[1];

/*    if(!app_data.blt_control) {
	fast_mode = current_obj.fill;
	set_blt_toggles(fast, slow);
    }
    */
    XtSetArg(args[0], XmNset, current_obj.fill);
    XtSetValues(fill, args, 1);
    XtSetArg(args[0], XmNset, !current_obj.fill);
    XtSetValues(outline, args, 1);
}

/*
**	called when blt gets selected or released.
*/
/*
static
void fast_toggle(Widget w)
{
    Arg	args[1];
    Boolean fast_set;
    
    XtSetArg(args[0], XtNset, &fast_set);
    XtGetValues(w, args, 1);

    if (fast_set == fast_mode) return;

    if (fast_mode) {
	draw_obj_at(last_obj.shape, shadow_obj_x, shadow_obj_y,
		    last_obj.fill, last_obj.size, erase_gc, shadow_blt);
    } else {
	draw_obj_at(last_obj.shape, shadow_obj_x, shadow_obj_y,
		    last_obj.fill, last_obj.size, gc, shadow_blt);
    }

    fast_mode = fast_set;

    set_blt_toggles(fast, slow);
    redraw_object(True);
}
*/
/*
**	called when redraw gets selected or released.
*/
/*
static
void slow_toggle(Widget w)
{
    Arg	args[1];
    Boolean slow_set;
    
    XtSetArg(args[0], XtNset, &slow_set);
    XtGetValues(w, args, 1);

    if (slow_set == !fast_mode) return;

    if (fast_mode) {
	draw_obj_at(last_obj.shape, shadow_obj_x, shadow_obj_y,
		    last_obj.fill, last_obj.size, erase_gc, shadow_blt);
    } else {
	draw_obj_at(last_obj.shape, shadow_obj_x, shadow_obj_y,
		    last_obj.fill, last_obj.size, gc, shadow_blt);
    }

    fast_mode = !slow_set;

    set_blt_toggles(fast, slow);
    redraw_object(True);
}
*/
/*
**	set_blt_toggles ensures that blt and redraw widgets are
**   set (or unset) as is appropriate for the app state.
**   (just returns if there are no blt or redraw toggles)
*/
/*
static
void set_blt_toggles(Widget fast, Widget slow)
{
    Arg args[1];

    if (!fast) return;

    XtSetArg(args[0], XtNset, fast_mode);
    XtSetValues(fast, args, 1);
    XtSetArg(args[0], XtNset, !fast_mode);
    XtSetValues(slow, args, 1);
}
*/
static 
void expose_obj()
{
    if (XtIsRealized(canvas)) {
	XClearWindow(XtDisplay(canvas), XtWindow(canvas));
	redraw_object(True);
    }
}

/*
**  When canvas is resized, we must also resize the sliders.
**  Use this opportunity to create the shadow_blt pixmap.  (Here for historical
**  reasons.  This used to depend upon work_width.)
*/
static 
void resize_canvas(Widget w)
{
    Dimension	slider_width, slider_height;
    Arg args[4];

    XtSetArg(args[0], XtNwidth, &work_width);
    XtSetArg(args[1], XtNheight, &work_height);
    XtGetValues(canvas, args, 2);
    if (XtIsRealized(canvas))
	XClearWindow(XtDisplay(canvas), XtWindow(canvas));
    XtSetArg(args[0], XtNwidth, &slider_width);
    XtSetArg(args[1], XtNheight, &slider_height);
    XtGetValues(speed, args, 2);
    if (slider_width != work_width) {
	XtMakeResizeRequest(speed, work_width, slider_height, &slider_width,
			    &slider_height);
    }
    XtSetArg(args[0], XtNwidth, &slider_width);
    XtSetArg(args[1], XtNheight, &slider_height);
    XtGetValues(size, args, 2);
    if (slider_height != work_height) {
	XtMakeResizeRequest(size, slider_width, work_height, &slider_width,
			    &slider_height);
    }
    if (!shadow_blt) {
	shadow_blt = XCreatePixmap(XtDisplay(canvas), DefaultRootWindow(dpy),
				   3*max_object_size, 3*max_object_size, 
				   DefaultDepth(dpy, scr));

	shadow_obj_x = max_object_size;
	shadow_obj_y = max_object_size;
	if (erase_gc) {
	    XFillRectangle(XtDisplay(canvas), shadow_blt, erase_gc, 
			   0, 0, 3*max_object_size, 3*max_object_size);
	}
    }
    redraw_object(True);
}

/*
**  general routine to handle redrawing the object
**  Create GCs if this is the first time through.
**  If fast_mode, try blt_in_object() to see if a single bitblt can
**  do the whole operation.  Else, draw over the old image and redraw
**  the new one.
**  If the object direction has changed, change the dial setting.
*/
static
void redraw_object(Boolean force_redraw)
{

    if (!canvas || !XtIsRealized(canvas)) return;

    if (gc == (GC) NULL) {
	XGCValues	gcv;
	Arg	arg[1];

	XtSetArg(arg[0], XtNforeground, &gcv.foreground);
	XtGetValues(canvas, arg, 1);
	gcv.function = GXcopy;
	/* get a changeable gc for the foreground */
	gc = XCreateGC(XtDisplay(canvas), XtWindow(canvas), 
		       GCForeground|GCFunction, &gcv);

	XtSetArg(arg[0], XtNbackground, &gcv.foreground);
	XtGetValues(canvas, arg, 1);
	gcv.function = GXcopy;
	erase_gc = XtGetGC(canvas, GCForeground|GCFunction, &gcv);
	XFillRectangle(XtDisplay(canvas), shadow_blt, erase_gc, 
		       0, 0, 3*max_object_size, 3*max_object_size);
    }

    if (!fast_mode || !blt_in_object(&last_obj, &current_obj, force_redraw)) {
	if (last_obj.drawn) {
	    draw_object(&last_obj, erase_gc);
	}
	draw_object(&current_obj, gc);
    }
    if((last_obj.dir_x != current_obj.dir_x) ||
       (last_obj.dir_y != current_obj.dir_y)) {
	double angle = atan2(current_obj.dir_x, current_obj.dir_y);
	int dial_value = (PI - angle) / (2 * PI) * (dial_max -dial_min) 
	  + dial_min;
	DialSetValue(dial, dial_value);
    }
    last_obj = current_obj;
    last_obj.drawn = True;
}

/*
**  blt_in_object() will attempt to use a single bitblt to erase and draw anew.
**  If it can do this, it will return True, else False.
**  force_redraw tells it to redraw even if the object hasn't moved since 
**  the last time.
*/
static 
Boolean blt_in_object(obj_ptr last, obj_ptr current, Boolean force_redraw)
{
    Position src_x, src_y, dst_x, dst_y;
    Dimension width, height;
    Position last_x = last->x;
    Position last_y = last->y;
    Position cur_x = current->x;
    Position cur_y = current->y;

    /* if anything has changed, fixup our shadow pixmap */
    if ((last->shape != current->shape) || (last->fill != current->fill) ||
	(last->size != current->size) || force_redraw) {
	XFillRectangle(XtDisplay(canvas), shadow_blt, erase_gc,
		       shadow_obj_x, shadow_obj_y, last->size, last->size);
	draw_obj_at(current_obj.shape, shadow_obj_x, shadow_obj_y,
		    current_obj.fill, current_obj.size, gc, shadow_blt);
    } else {
	/*
	 **  don't really need to redraw.  Say we succeeded.
	 */
	if ((last_x == cur_x) && (last_y == cur_y) && last->drawn) return True;
    }

    /*
    **  The two images, last and current, are too far apart for it to be
    **  efficient to cover them in one blt.  Fail here so that we redraw them.
    */
    if (Max(cur_x, last_x) - Min(cur_x, last_x) > max_object_size) 
      return False;
    if (Max(cur_y, last_y) - Min(cur_y, last_y) > max_object_size) 
      return False;

    if (cur_x < last_x) {
	dst_x = cur_x;
	src_x = shadow_obj_x;
	width = last_x - cur_x + Max(last->size, current->size);
    } else {
	dst_x = last_x;
	src_x = shadow_obj_x - (cur_x - last_x);
	width = cur_x - last_x + Max(last->size, current->size);
    }
    if (cur_y < last_y) {
	dst_y = cur_y;
	src_y = shadow_obj_y;
	height = last_y - cur_y + Max(last->size, current->size);
    } else {
	dst_y = last_y;
	src_y = shadow_obj_y - (cur_y - last_y);
	height = cur_y - last_y + Max(last->size, current->size);
    }

    /*
    **  use a bitblt from the shadow pixmap to cover the old image
    **  and draw the new one in one shot.
    */
    XCopyArea(XtDisplay(canvas), shadow_blt, XtWindow(canvas), gc, 
	      src_x, src_y, width, height, dst_x, dst_y);
    return TRUE;
}


extern void
update_position(x, y)
int x;
int y;
{
    current_obj.x = x;
    current_obj.y = y;
    redraw_object(False);
    XSync(dpy, 0);
}

/*
**  Some ugly shit to follow.  These are structures that define the shapes
**  of the shapes.  gumby is initialized statically.  The rest at init time.
*/
typedef struct {
    float  x,y;
} FPoint;

static FPoint oct[8];
static FPoint hex[6];
static FPoint pent[5];
static FPoint tri[3];
static FPoint gumby[] = {
    {.28, 0},	/* point of head */
    {.46, .05},
    {.46, .21}, /* right shoulder */
    {.69, .25},
    {.74, .46},
    {.665, .50}, /* right finger tip */
    {.665, .43},
    {.64, .305},
    {.47, .315}, /* right armpit */
    {.47, .565},
    {.56, .93},
    {.63, .95},
    {.63, 1.0},  /* right toe */
    {.50, 1.0},
    {.40, .585},
    {.25, 1.0},
    {.15, 1.0},
    {.15, .95},
    {.21, .93},
    {.33, .565},  /* left hip */
    {.33, .32},
    {.11, .34},
    {.04, .10},
    {.01, .055},
    {.065, .015},  /* left finger tip */
    {.065, .065},
    {.15, .26},
    {.32, .22},
    {.275, .04},
};

/*
**  Given an object structure and a number of points, fill the object
**  structure with cornerpoints for a N sided shape.
**  This got complicated when I tried to eliminate artifacts.
*/
#define EPS 0.0000001
static void init_object_struct(FPoint *pts, int num_pts)
{
    int i;
    float	ext_angle = 2*PI/num_pts;
    float	half_angle = ext_angle/2.0;
    float	min_x = 1.0, max_x = -1.0;
    float	min_y = 1.0, max_y = -1.0;
    float	scale_x, scale_y;
    float	trans_x, trans_y;

    for(i=0; i<num_pts; i++) {
	pts[i].x = cos(ext_angle*i+half_angle+PI/2);
	min_x = Min(pts[i].x, min_x); max_x = Max(pts[i].x, max_x);
	pts[i].y = sin(ext_angle*i+half_angle+PI/2);
	min_y = Min(pts[i].y, min_y); max_y = Max(pts[i].y, max_y);
    }
    trans_x = -min_x;  trans_y = -min_y;
    scale_x = 1.0/(max_x - min_x);    scale_y = 1.0/(max_y - min_y);
    for(i=0; i<num_pts; i++) {
	pts[i].x = (pts[i].x + trans_x) * scale_x;
	pts[i].y = (pts[i].y + trans_y) * scale_y;
	if ((pts[i].x < EPS) && (pts[i].x > -EPS)) pts[i].x = 0.0;
	if ((pts[i].y < EPS) && (pts[i].y > -EPS)) pts[i].y = 0.0;
	if ((pts[i].x-1 < EPS) && (pts[i].x-1 > -EPS)) pts[i].x = 1.0;
	if ((pts[i].y-1 < EPS) && (pts[i].y-1 > -EPS)) pts[i].y = 1.0;
    }
}

/*
**  initialize the corner points of the shapes structures
*/
static void init_object_forms()
{
    init_object_struct(oct, 8);
    init_object_struct(hex, 6);
    init_object_struct(pent, 5);
    tri[0].x = 0.5;	tri[0].y = 0;
    tri[1].x = 0; 	tri[1].y = 1;
    tri[2].x = 1; 	tri[2].y = 1;
}

/*
**  Draw a complex object with corner points pts, etc.
*/
static
void draw_comp_obj(FPoint *pts, int num_pts, int x, int y, int size, 
		   Boolean fill, Boolean simple, GC gc, Window win)
{
    int i;
    XPoint	*points;

    points = malloc(sizeof(XPoint) * (num_pts+1));
    for(i=0;i<num_pts; i++) {
	points[i].x = x+size*pts[i].x; points[i].y = y+size*pts[i].y;
    }
    points[num_pts] = points[0];
    if (fill) {
	XFillPolygon(XtDisplay(canvas), win, gc, points, num_pts+1, 
		     simple ? Convex : Nonconvex, CoordModeOrigin);
    } else {
	XDrawLines(XtDisplay(canvas), win, gc, points, num_pts+1, 
		   CoordModeOrigin);
    }
    free(points);
}

/*
**  just a type converting convenience routine.  Draws object in the canvas.
*/
static
void draw_object(obj_ptr obj, GC gc)
{
    int	x = obj->x;
    int y = obj->y;
    draw_obj_at(obj->shape, x, y, obj->fill, obj->size, gc, XtWindow(canvas));
}

/*
**  draws an object with shape, position, etc in a drawable.
*/
static
void draw_obj_at(shape_type shape, Position x, Position y, Boolean fill, 
		 int size, GC gc, Drawable win)
{
    size--;	/* we start at zero.  To use size as multiplier, sub 1 */

    switch (shape) {
    case Circle:
	if (fill) {
	    XFillArc(dpy, win, gc, x, y, size, size,  0, 360*64);
	} else {	    
	    XDrawArc(dpy, win, gc, x, y, size, size,  0, 360*64);
	}
	break;
    case Octagon:
	draw_comp_obj(oct, 8, x, y, size, fill, True, gc, win);
	break;
    case Hexagon:
	draw_comp_obj(hex, 6, x, y, size, fill, True, gc, win);
	break;
    case Pentagon:
	draw_comp_obj(pent, 5, x, y, size, fill, True, gc, win);
	break;
    case Square:
	if (fill) {
	    XFillRectangle(dpy, win, gc, x, y, size, size);
	} else {
	    XDrawRectangle(dpy, win, gc, x, y, size, size);
	}
	break;
    case Triangle:
	draw_comp_obj(tri, 3, x, y, size, fill, True, gc, win);
	break;
    case Gumby:
	draw_comp_obj(gumby, XtNumber(gumby), x, y, size, fill, False, 
		      gc, win);
	break;
    default:
	fprintf(stderr, "unknown shape\n");
    }
}

/*
**      Define a converter for String to StringTable (char **)
**   Delimiters are values in SEP_STR below.  Initial white space
**   in a String is skipped.  (This is, white space following
**   a delimeter in the fromVal string is skipped, as opposed to
**   just initial white space in the fromVal string.)
**      Since I now this is only used by an application, I don't provide
**   a destructor at the moment.
**
**   The done() macro is copied from the Xt doc.  It "does the right thing".
*/
#define SEP_STR ";\n"

#define done(type,value) \
{\
    if(toVal->addr != NULL) {\
	if(toVal->size < sizeof(type)) {\
	    toVal->size = sizeof(type);\
	    return False;\
	}\
	*(type *)(toVal->addr) = (value);\
    } else {\
	static type static_val;\
	static_val = (value);\
	toVal->addr = (XtPointer) &static_val;\
    }\
    toVal->size = sizeof(type);\
    return True;\
}

static Boolean
CvtStringToStringTable(Display *display, XrmValue *args, Cardinal *num_args,
		       XrmValue *fromVal, XrmValue *toVal, XtPointer *conv_data)
{
    int nstrings = 0;
    static char **table;
    char *from_str = strtok((char *) fromVal->addr, SEP_STR);
    
    if (*num_args !=0) {
	XtWarning("String to StringTable conversion needs no arguments");
    }
    table = (char **) malloc(sizeof(char **));
    while (from_str != NULL) {
	int  length = strlen(from_str);
	
	from_str += strspn(from_str, " \t");  /* skip initial blanks */
	table = (char **) realloc(table, (nstrings+2) * sizeof(char**));
	table[nstrings] = (char *) malloc(length+1);
	strcpy(table[nstrings], from_str);
	from_str = strtok(NULL, SEP_STR);
	nstrings++;
    }
    table[nstrings] = NULL;
    done(char **, table);
}

/*
**  stuff below is object interface
*/
static
void init_obj()
{
    last_obj.drawn = False;
    current_obj.shape = Square;
    current_obj.fill = False;
    current_obj.x = 0;
    current_obj.y = 0;
}

void
set_dial(int dial_value)
{
    DialSetValue(dial, dial_value);
}
