/*
 *  this is a sample steerable application.  In this case, this code 
 *  calculates the location of a ball bouncing in a 2D box.  The ball's 
 *  position and direction are calculated and stored as object state.  It's 
 *  speed and size are also stored as object state, but they are not changed 
 *  by the code in this file.  (They may be changed by external steering 
 *  agents however.)
 *  
 *  The classes are defined in ball.idl.  Method bodies are in ball_impl.c
 */
#include <../config.h>
#include <malloc.h>
#include <math.h>
#include <string.h>
#include <stdio.h>
#include <gen_thread.h>
#include <io.h>
#include <DE.h>
#include <otl.h>
#include <time.h>
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_TIMEB_H
#include <sys/timeb.h>
#endif
#include "ball.h"
#include "ball_COBS_.h"
#ifdef HAVE_WINDOWS_H
#include "windows.h"
#include "winsock.h"
#endif

#ifndef M_PI
#define M_PI 3.14159265358797323846
#endif

extern struct timespec start_time, end_time, total_time;

#ifndef GETTIMEOFDAY_DEFINED
    extern void gettimeofday(struct timeval *tp, void *tzp);
#endif

static
void update_position (ball_position bp, ball_direction bd,
		      ball_speed bs, ball_size bz, CORBA_Environment *ev);

int
main(argc, argv)
int argc;
char **argv;
{
    ball_position bp;
    ball_direction bd;
    ball_speed bs;
    ball_size bz;
    struct timeval delay_time;
    CORBA_Environment ev;
    DExchange de;

    delay_time.tv_sec = 0;
    delay_time.tv_usec = 100000;

    /* use a null threads package.  I.E. forks are proc calls. */
    gen_null_init();

    /* setup a singlethreaded DataExchange */
    de = DExchange_create();
    DEControlList_set_control_style(DExchange_get_control_list(de),
				    DESingleThreaded);

    /* initialize the object system with the singlethreaded de */
    init_cobs_DE("ball", NULL, de);

    /* initialize a CORBA environment object to use for invocations. */
    COBS_init_Environment_object(&ev);

    /*  create the objects that will be used during this simulation */
    bp = ball_position__Create(NULL);

    bd = ball_direction__Create(NULL);
    ball_direction__set_direction(bd, 35.0, &ev);

    bs = ball_speed__Create(NULL);
    ball_speed__set_speed(bs, 10, &ev);

    bz = ball_size__Create(NULL);
    ball_size__set_size(bz, 10, &ev);

    while(1) {
	update_position(bp, bd, bs, bz, &ev);
	cobs_poll();
	select(0, NULL, NULL, NULL, &delay_time);
    }
}

static int work_width = 400;
static int work_height = 400;

static
void update_position(bp, bd, bs, bz, ev)
ball_position bp;
ball_direction bd;
ball_speed bs;
ball_size bz;
CORBA_Environment *ev;
{
    long etime;
    float displacement;
    int obj_size = ball_size__get_size(bz, ev);
    int speed = ball_speed__get_speed(bs, ev);

#ifdef HAVE_GETTIMEOFDAY
    static struct timeval last_time = {-1, -1}, current_time;
    gettimeofday(&current_time, NULL);
    if (last_time.tv_sec == -1) {
	last_time = current_time;
	return;
    }

    etime = (current_time.tv_sec - last_time.tv_sec) * 1000000;
    etime += current_time.tv_usec - last_time.tv_usec;
    displacement = speed * etime / 100000.0;
#else
    static struct timeb  last_time = {-1, -1, -1, -1}, current_time;
    ftime(&current_time);
    if (last_time.time == -1) {
	last_time = current_time;
	return;
    }

    etime = (current_time.time - last_time.time) * 1000;
    etime += current_time.millitm - last_time.millitm;
    displacement = speed * etime / 100.0;
#endif

    {
	int width = work_width - obj_size;
	int height = work_height - obj_size;
	float direction = ball_direction__get_direction(bd, ev);
	float radian_dir = direction / 360.0 * 2 * M_PI;
	float dir_x = cos(radian_dir);
	float dir_y = sin(radian_dir);
	position_t new, old = ball_position__get_position(bp, ev);
	int angle_change = 0;

	new.x = old.x + dir_x * displacement;
	new.y = old.y + dir_y * displacement;
	if (new.x < 0) {
	    new.x = -new.x;
	    dir_x = -dir_x;
	    angle_change++;
	}
	if (new.x > width) {
	    new.x = width - (new.x % width);
	    dir_x = -dir_x;
	    angle_change++;
	}	
	if (new.y < 0) {
	    new.y = -new.y;
	    dir_y = -dir_y;
	    angle_change++;
	}
	if (new.y > height) {
	    new.y = height - (new.y % height);
	    dir_y = -dir_y;
	    angle_change++;
	}
	if (angle_change) {
	    float new_dir = atan2(dir_y, dir_x);
	    float new_angle;
	    if (new_dir < 0.0) {
		new_dir += 2 * M_PI;
	    }
	    new_angle = new_dir * 360.0 / (2 * M_PI);
	    ball_direction__set_direction(bd, new_angle, 
					  ev);
	    angle_change = 0;
	}
	ball_position__set_position(bp, &new, ev);
    }


    last_time = current_time;
}

