/*-*- C++ -*-*/
//#############################################################################
//## File: IDLtoC/be_pbio/dumps.cc
//## Author: Jeremy Heiner
//## Organization: Georgia Institute of Technology
//## Copyright: see below
//## Purpose: gathered all the dumps for be_pbio into a single file,
//##      for comprehension and for compilation speed-up.
//#############################################################################
//## History: the front end was developed by SunSoft, see their copyright
//## message below.  Dilma Silva, also here at gatech, developed the code
//## into an idl2c compiler to support the COBS project.  i'm starting with
//## dilma's code.

/*

COPYRIGHT

Copyright 1992, 1993, 1994 Sun Microsystems, Inc.  Printed in the United
States of America.  All Rights Reserved.

This product is protected by copyright and distributed under the following
license restricting its use.

The Interface Definition Language Compiler Front End (CFE) is made
available for your use provided that you include this license and copyright
notice on all media and documentation and the software program in which
this product is incorporated in whole or part. You may copy and extend
functionality (but may not remove functionality) of the Interface
Definition Language CFE without charge, but you are not authorized to
license or distribute it to anyone else except as part of a product or
program developed by you or with the express written consent of Sun
Microsystems, Inc. ("Sun").

The names of Sun Microsystems, Inc. and any of its subsidiaries or
affiliates may not be used in advertising or publicity pertaining to
distribution of Interface Definition Language CFE as permitted herein.

This license is effective until terminated by Sun for failure to comply
with this license.  Upon termination, you shall destroy or return all code
and documentation for the Interface Definition Language CFE.

INTERFACE DEFINITION LANGUAGE CFE IS PROVIDED AS IS WITH NO WARRANTIES OF
ANY KIND INCLUDING THE WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS
FOR A PARTICULAR PURPOSE, NONINFRINGEMENT, OR ARISING FROM A COURSE OF
DEALING, USAGE OR TRADE PRACTICE.

INTERFACE DEFINITION LANGUAGE CFE IS PROVIDED WITH NO SUPPORT AND WITHOUT
ANY OBLIGATION ON THE PART OF Sun OR ANY OF ITS SUBSIDIARIES OR AFFILIATES
TO ASSIST IN ITS USE, CORRECTION, MODIFICATION OR ENHANCEMENT.

SUN OR ANY OF ITS SUBSIDIARIES OR AFFILIATES SHALL HAVE NO LIABILITY WITH
RESPECT TO THE INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY
INTERFACE DEFINITION LANGUAGE CFE OR ANY PART THEREOF.

IN NO EVENT WILL SUN OR ANY OF ITS SUBSIDIARIES OR AFFILIATES BE LIABLE FOR
ANY LOST REVENUE OR PROFITS OR OTHER SPECIAL, INDIRECT AND CONSEQUENTIAL
DAMAGES, EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

Use, duplication, or disclosure by the government is subject to
restrictions as set forth in subparagraph (c)(1)(ii) of the Rights in
Technical Data and Computer Software clause at DFARS 252.227-7013 and FAR
52.227-19.

Sun, Sun Microsystems and the Sun logo are trademarks or registered
trademarks of Sun Microsystems, Inc.

SunSoft, Inc.  
2550 Garcia Avenue 
Mountain View, California  94043

NOTE:

SunOS, SunSoft, Sun, Solaris, Sun Microsystems or the Sun logo are
trademarks or registered trademarks of Sun Microsystems, Inc.

 */

//#############################################################################

#include "../config.h"
#include "../be_generic/be_classes.hh"

#include <fstream.h>

#ifdef HAVE_UNISTD_H
#include	<unistd.h>		// POSIX standard types
#endif
#ifdef _MSC_VER
#define unlink _unlink
#include <stdio.h>
#endif

//#############################################################################

//=============================================================================
/*
 * Prepare an argument for a BE
 */
void
BE_prep_arg( char*, idl_bool )
{
}

//=============================================================================
/*
 * Print out a version string for the BE
 */
void
BE_version()
{
	cerr << GTDEVEL("Generic BE, version 1.0.0") << endl;
}

//=============================================================================
/*
 * Do the work of this BE. 
 */
void
BE_produce()
{
	idl_global->root()->dump( cout );
}

//=============================================================================
/*
 * Abort this run of the BE
 */
void
BE_abort()
{
	cerr << GTDEVEL("Aborting execution.") << endl;
	BE_Root::abort();
	exit(1);
}

//=============================================================================

#define BE_WARN(str) \
	do { \
      idl_global->err()->back_end_warning( line(), file_name() ); \
		cerr << GTDEVEL(str) << endl; \
	} while ( 0 )

#define BE_ERROR(str) \
	do { \
		idl_global->err()->back_end( line(), file_name() ); \
		cerr << GTDEVEL(str) << endl; \
	} while ( 0 )

#define BE_ABORT(str) \
	do { BE_ERROR(str); BE_abort(); } while( 0 )

//#############################################################################

static const char* ATTRIBUTE_NOT_INSIDE_INTERFACE
							= "hmm, attribute not enclosed in an interface???";
static const char* NARROW_TO_INTERFACE_FAILED
							= "couldn't get enclosing interface from scope";
static const char* UNSUPPORTED_PREDEFINED_TYPE
							= "sorry, bools, voids, and anys not supported";
static const char* OPEN_HEADER_FAILED = "couldn't create header file";
static const char* OPEN_SOURCE_FAILED = "couldn't create source file";

static const char* NARROW_TO_ARRAY_FAILED
							= "huh? can't narrow array? shouldn't happen!";
static const char* NARROW_TO_ATTRIBUTE_FAILED
							= "huh? can't narrow attribute? shouldn't happen!";
static const char* NARROW_TO_PREDEFINED_TYPE_FAILED
							= "huh? can't narrow predefined type? shouldn't happen!";
static const char* ZERO_DIMENSIONAL_ARRAY
							= "huh? zero dimensional array? shouldn't happen!";
static const char* UNNAMED_SYMBOL
							= "huh? unnamed symbol? shouldn't happen!";
static const char* LOOKUP_SYMBOL_FAILED
							= "huh? symbol not found? shouldn't happen!";

static const char* DYNAMIC_INDEX_WRONG_SCOPE
							= "index for dynamic array must be in the same interface";
static const char* DYNAMIC_INDEX_WRONG_TYPE
							= "index for dynamic array must be an integer type";
static const char* DYNAMIC_ARRAY_TOO_BIG
							= "only one dimensional dynamic arrays allowed";
static const char* STATIC_ARRAY_TOO_BIG
							= "static arrays cannot have a higher dimension than two";

//#############################################################################
//#############################################################################
//#############################################################################
//####
//####
//####      STATE MACHINE MEMORY
//####
//####
//=============================================================================

static BE_Root* GLOBAL_ROOT;
	// better than idl_global->root() because it's a BE_Root* !

static idl_bool WANT_PBIO_TYPES;
	// I_TRUE means use PBIO's "integer" for all int variants, etc.

static idl_bool DUMPING_INTERFACE;
static idl_bool DUMPING_ATTRIBUTE;
	// I_TRUE means we're currently dumping the similarly named node.

static enum { NOT_DUMPING, HEADER, SOURCE } OUTPUT_FILE;

static void
INIT_STATE_MEMORY()
{
	GLOBAL_ROOT = NULL;
	WANT_PBIO_TYPES = I_FALSE;
	DUMPING_INTERFACE = I_FALSE;
	DUMPING_ATTRIBUTE = I_FALSE;
	OUTPUT_FILE = NOT_DUMPING;
}

//=============================================================================
//#############################################################################
//#############################################################################
//#############################################################################
//####
//####
//####      FILE AND MACRO NAMES, CLEANUP
//####
//####
//=============================================================================

static char* INPUT_FILE_NAME;
static char* INPUT_FILE_BASE; // stripped of dirs & optional ".idl" suffix

static char* HEADER_FILE_NAME;
static char* SOURCE_FILE_NAME;
static char* INCLUDE_MACRO_NAME;

static char* STANDARD_OUTPUT = (char*)"standard_output";
static char* STANDARD_INPUT  = (char*)"standard_input";

static void
CREATE_NAMES()
{

	if ( idl_global->read_from_stdin() ) {

		INPUT_FILE_NAME = STANDARD_INPUT;
		INPUT_FILE_BASE = STANDARD_OUTPUT;

		HEADER_FILE_NAME = NULL;
		SOURCE_FILE_NAME = NULL;
		INCLUDE_MACRO_NAME = NULL;

	} else {

		INPUT_FILE_NAME = idl_global->main_filename()->get_string();

		char*  stripped_name = idl_global->stripped_filename()->get_string();
		size_t stripped_length = strlen( stripped_name );

		char* idl_suffix = strrchr( stripped_name, '.' ); // find extension

		if ( idl_suffix == stripped_name )
			idl_suffix = NULL; // reject "dot" files, as they have no basename

		if ( idl_suffix && 0!=strcmp( idl_suffix, ".idl" ) )
			idl_suffix = NULL; // reject files with other extensions

		if ( ! idl_suffix ) {

			// ok, don't reject, but don't strip it any further
			INPUT_FILE_BASE = stripped_name;

		} else {

			stripped_length -= 4; // base = stripped_name - ".idl"

			INPUT_FILE_BASE = new char[ stripped_length + 1 ];
			strncpy( INPUT_FILE_BASE, stripped_name, stripped_length );
			INPUT_FILE_BASE[ stripped_length ]  = '\0';

		}

		size_t input_base_length = strlen( INPUT_FILE_BASE );

		HEADER_FILE_NAME = new char[ input_base_length + 3 ];
		strncpy( HEADER_FILE_NAME, INPUT_FILE_BASE, input_base_length );
		strcpy( HEADER_FILE_NAME + input_base_length, ".h" );

		SOURCE_FILE_NAME = new char[ input_base_length + 3 ];
		strncpy( SOURCE_FILE_NAME, INPUT_FILE_BASE, input_base_length );
		strcpy( SOURCE_FILE_NAME + input_base_length, ".c" );

		INCLUDE_MACRO_NAME = new char[ input_base_length + 5 ];
		strcpy( INCLUDE_MACRO_NAME, "_" );
		strcat( INCLUDE_MACRO_NAME, INPUT_FILE_BASE );
		strcat( INCLUDE_MACRO_NAME, "_H_" );

		for ( char* ptr = INCLUDE_MACRO_NAME; *ptr; ++ptr )
			if ( isalpha(*ptr) )
				*ptr = toupper(*ptr);   // up-case alphas
			else if ( !isdigit(*ptr) ) // ignore digits
				*ptr = '_';             // and turn everything else to '_'
	}
}

void
BE_Root::abort()
{
	if ( ! idl_global->read_from_stdin() ) {
		unlink( HEADER_FILE_NAME );
		unlink( SOURCE_FILE_NAME );
	}
}


//=============================================================================
//#############################################################################
//#############################################################################
//#############################################################################
//####
//####
//####      OUTPUT HELPERS
//####
//####
//=============================================================================

static void
FILE_PROLOG_DUMP( ostream& o, const char* extension )
{
	o
		<< "/*****************************************************\n"
		<< " ** File: " << INPUT_FILE_BASE << extension << "\n"
		<< " ** \n"
		<< " ** !!! DO NOT EDIT !!!\n"
		<< " ** \n"
		<< " ** Generated from file: " << INPUT_FILE_NAME << "\n"
		<< " ** Using: " << idl_global->prog_name() << "\n"
		<< " ****************************************************/\n"
		<< "\n\n";
}

static void
ONCE_ONLY_DUMP( ostream& o )
{
	o
		<< "#ifndef " << INCLUDE_MACRO_NAME << "\n"
		<< "#define " << INCLUDE_MACRO_NAME << "\n"
		<< "\n\n";
}

static void
END_HEADER_DUMP( ostream& o )
{
	o
		<< "#endif /* " << INCLUDE_MACRO_NAME << "*/\n"
		<< "\n\n";
}

static void
INCLUDE_HEADER_DUMP( ostream &o )
{
	o
		<< "#include \"" << INPUT_FILE_BASE << ".h\"\n"
		<< "\n\n";
}

static void
SCOPED_NAME_DUMP( UTL_ScopedName* n, ostream& o )
{
	UTL_IdListActiveIterator* i = new UTL_IdListActiveIterator( n );

	if ( i->is_done() ) return;
	i->next(); // skip over empty root name ("")
	if ( i->is_done() ) return;
	i->item()->dump( o ); i->next();
	while ( ! i->is_done() ) {
		o << "_"; i->item()->dump( o ); i->next();
	}
}

static void ROOT_SCOPE_DUMP( ostream& );
static void ARRAY_DIMS_DUMP( BE_Array*, ostream& );

static void dump_binary_expr(ostream&,const char*,AST_Expression*,AST_Expression*);
static void dump_unary_expr(ostream&,const char*,AST_Expression*);

//=============================================================================
//#############################################################################
//#############################################################################
//#############################################################################
//####
//####
//####      DUMP()S
//####
//####
//=============================================================================

void
BE_Root::dump( ostream& o )
{
	o << flush;
	INIT_STATE_MEMORY();
	GLOBAL_ROOT = this;

	ostream* ostrptr = &o;

	CREATE_NAMES();

	OUTPUT_FILE = HEADER;
	{
		if ( ! idl_global->read_from_stdin() ) {

			ostrptr = new ofstream( HEADER_FILE_NAME );
			if ( !ostrptr || !*ostrptr )
				BE_ABORT( OPEN_HEADER_FAILED );
		}

		FILE_PROLOG_DUMP( *ostrptr, ".h" );

		if ( !idl_global->read_from_stdin() )
			ONCE_ONLY_DUMP( *ostrptr );

		*ostrptr << "#include \"io.h\"\n\n";

		ROOT_SCOPE_DUMP( *ostrptr );

		if ( !idl_global->read_from_stdin() ) {
			END_HEADER_DUMP( *ostrptr );
			((ofstream*)ostrptr)->close(); delete ostrptr;
		}
	}

	OUTPUT_FILE = SOURCE;
	{
		if ( !idl_global->read_from_stdin() ) {

			ostrptr = new ofstream( SOURCE_FILE_NAME );
			if ( !ostrptr || !*ostrptr )
				BE_ABORT( OPEN_SOURCE_FAILED );
		}

		FILE_PROLOG_DUMP( *ostrptr, ".c" );

		if ( !idl_global->read_from_stdin() )
			INCLUDE_HEADER_DUMP( *ostrptr );

		ROOT_SCOPE_DUMP( *ostrptr );

		if ( !idl_global->read_from_stdin() ) {
			((ofstream*)ostrptr)->close(); delete ostrptr;
		}
	}
	o << flush;
}

static void
ROOT_SCOPE_DUMP( ostream& o )
{
	o << flush;
	UTL_ScopeActiveIterator* i;
	AST_Decl* d;

	i = new UTL_ScopeActiveIterator( GLOBAL_ROOT, UTL_Scope::IK_both );

	for ( ; ! (i->is_done()); i->next() ) {
		d = i->item();
		switch ( d->node_type() ) {

		case AST_Decl::NT_pre_defined:
		case AST_Decl::NT_array:
			break;

		default:
			d->dump( o );
			break;
		}
	}

	delete i;
	o << flush;
}

//=============================================================================

void
BE_Module::dump( ostream& o )
{
	o << flush;
	if ( this != (BE_Module*)GLOBAL_ROOT ) {
		o << "/* BEGIN MODULE "; SCOPED_NAME_DUMP( name(), o ); o << " */\n\n";
	}

	BE_Scope::dump(o);

	if ( this != (BE_Module*)GLOBAL_ROOT ) {
		o << "/* END MODULE "; SCOPED_NAME_DUMP( name(), o ); o << " */\n\n";
	}
	o << flush;
}

//=============================================================================

void
BE_Scope::dump( ostream& o )
{
	o << flush;
	UTL_ScopeActiveIterator* i;
	//	AST_Decl* d;

	i = new UTL_ScopeActiveIterator( this, UTL_Scope::IK_decls );

	for ( ; ! (i->is_done()); i->next() )
		i->item()->dump( o );

	delete i;
	o << flush;
}

//=============================================================================

void
BE_Interface::dump( ostream& o )
{
	o << flush;
	int i;

	if ( DUMPING_ATTRIBUTE ) {

		SCOPED_NAME_DUMP( name(), o );
		if ( ! WANT_PBIO_TYPES )
			o << "_rec";

	} else if ( DUMPING_INTERFACE ) {

		BE_Scope::dump( o ); // we're doing a dump of an inherited interface

	} else {

		DUMPING_INTERFACE = I_TRUE;

		o << "/* INTERFACE "; SCOPED_NAME_DUMP( name(), o ); o << " */\n\n";

		switch ( OUTPUT_FILE ) {

		case HEADER:

			o << "\t" << "typedef struct _"; SCOPED_NAME_DUMP( name(), o );
																		 o << "_rec {\n";

			BE_Scope::dump(o);
			for ( i = 0; i < n_inherits(); ++i )
				inherits()[i]->dump( o );

			o << "\t" << "} "; SCOPED_NAME_DUMP( name(), o ); o << "_rec, "
								"*"; SCOPED_NAME_DUMP( name(), o ); o << "_rec_ptr;\n";

			o << "\t" << "extern IOField "; SCOPED_NAME_DUMP( name(), o );
															  o << "_field_list[];\n";

			break;

		case SOURCE:

			o << "\tIOField "; SCOPED_NAME_DUMP( name(), o );
														  o << "_field_list[] = {\n";

			BE_Scope::dump(o);
			for ( i = 0; i < n_inherits(); ++i )
				inherits()[i]->dump( o );

			o << "\t\t{ (char*)NULL, (char*)NULL, 0, 0 }\n\t};\n";

			break;

		default:
			break;
		}
		o << "\n";
		DUMPING_INTERFACE = I_FALSE;

	}
	o << flush;
}

//=============================================================================

void
BE_Attribute::dump( ostream& o )
{
	o << flush;

	if ( defined_in()->scope_node_type() != AST_Decl::NT_interface )
		BE_ABORT( ATTRIBUTE_NOT_INSIDE_INTERFACE );

	BE_Interface *interface
		= BE_Interface::narrow_from_scope( defined_in() );

	if ( !interface )
		BE_ABORT( NARROW_TO_INTERFACE_FAILED );

	DUMPING_ATTRIBUTE = I_TRUE;

	switch ( OUTPUT_FILE ) {

	case HEADER:

		o << "\t\t"; field_type()->dump(o); o << " ";

		if ( field_type()->node_type() == AST_Decl::NT_array ) {

			BE_Array *array = BE_Array::narrow_from_decl( field_type() );
			if ( ! array )
				BE_ABORT( NARROW_TO_ARRAY_FAILED );

			{
				idl_bool dynamic = I_FALSE;
				AST_Decl* d;

				if ( array->n_dims() < 1 ) // should be impossible
					BE_ABORT( ZERO_DIMENSIONAL_ARRAY );

				AST_Expression* first_dim = array->dims()[0];
				if ( !first_dim )
					BE_ABORT( ZERO_DIMENSIONAL_ARRAY );

				if ( first_dim->ec() == AST_Expression::EC_symbol ) {
					// this could be a dynamic array, so let's check
					if ( first_dim->n() == NULL ) // should be impossible
						BE_ABORT( UNNAMED_SYMBOL );
					d = array->defined_in()->lookup_by_name( first_dim->n(), I_FALSE );
					if ( d == NULL ) // should not happen
						BE_ABORT( LOOKUP_SYMBOL_FAILED );

					dynamic = (d->node_type() == AST_Decl::NT_attr);
				}

				// enforce semantics of array dimensions
				if ( dynamic ) {

					if ( array->n_dims() != 1 )
						BE_ABORT( DYNAMIC_ARRAY_TOO_BIG );

					BE_Attribute *attr = BE_Attribute::narrow_from_decl( d );
					if ( !attr ) BE_ABORT( NARROW_TO_ATTRIBUTE_FAILED );

					if ( array->defined_in() != attr->defined_in() )
						BE_ABORT( DYNAMIC_INDEX_WRONG_SCOPE );

					if ( attr->field_type()->node_type() != AST_Decl::NT_pre_defined )
						BE_ABORT( DYNAMIC_INDEX_WRONG_TYPE );

					BE_PredefinedType* type
						= BE_PredefinedType::narrow_from_decl( attr->field_type() );
					if ( type == NULL ) BE_ABORT( NARROW_TO_PREDEFINED_TYPE_FAILED );

					switch( type->pt() ) {

					case AST_PredefinedType::PT_long:
					case AST_PredefinedType::PT_ulong:
					case AST_PredefinedType::PT_longlong:
					case AST_PredefinedType::PT_ulonglong:
					case AST_PredefinedType::PT_short:
					case AST_PredefinedType::PT_ushort:
						break;

					default:
						BE_ABORT( DYNAMIC_INDEX_WRONG_TYPE );
						break;
					}

				} else { // static array

					if ( array->n_dims() > 2 )
						BE_ABORT( STATIC_ARRAY_TOO_BIG );

					for ( int i = 0; i < array->n_dims(); ++i ) {

						AST_Expression* e = array->dims()[i];

						if ( e->ec() == AST_Expression::EC_symbol ) {
							// evaluation was not performed in "array_dim" in "lex.yy",
							// so do it here...
							e->evaluate( AST_Expression::EK_const );
							if ( ! e->coerce( AST_Expression::EV_ulong ) )
								BE_ABORT ( DYNAMIC_INDEX_WRONG_TYPE );
						}
					}
				}

				// finally, we can dump this declaration

				if ( dynamic ) o << "*";

				local_name()->dump( o );

				if ( ! dynamic ) ARRAY_DIMS_DUMP( array, o );
			}

		} else {

			local_name()->dump(o);
		}

		o << ";\n";
			
		break;

	case SOURCE:


		o << "\t\t" << "{ \""; local_name()->dump(o); o << "\", \"";
		                         WANT_PBIO_TYPES = I_TRUE;
		                         field_type()->dump(o);
		                         WANT_PBIO_TYPES = I_FALSE;
		                         o << "\", sizeof( ";
		                                     field_type()->dump(o); o << " ),\n";
		                         

		o << "\t\t\t" << "IOOffset( "; SCOPED_NAME_DUMP( interface->name(), o );
		                                        o << "_rec_ptr, ";
		                                    local_name()->dump(o);
		                                        o << " ) },\n";
		break;

	default:
		break;
	}

	DUMPING_ATTRIBUTE = I_FALSE;
	o << flush;
}

//=============================================================================

void
BE_Constant::dump( ostream& o )
{
	o << flush;
	if ( OUTPUT_FILE != HEADER )
		return;

	o << "#define "; SCOPED_NAME_DUMP( name(), o ); o << " ";
	constant_value()->dump(o);
	o << "\n\n";
	o << flush;
}

//=============================================================================

void
BE_Array::dump( ostream& o )
{
	o << flush;
	base_type()->dump( o );
	if ( WANT_PBIO_TYPES )
		ARRAY_DIMS_DUMP( this, o );
	o << flush;
}

static void
ARRAY_DIMS_DUMP( BE_Array* a, ostream& o )
{
	for ( long i = 0; i < a->n_dims(); ++i ) {
		o << '['; a->dims()[i]->dump( o ); o << ']';
	}
}

//=============================================================================

void
BE_Enum::dump( ostream& o )
{
	o << flush;
	if ( DUMPING_INTERFACE ) {

		if ( WANT_PBIO_TYPES )
			o << "enumeration";
		else
			SCOPED_NAME_DUMP( name(), o );

	} else if ( OUTPUT_FILE == HEADER ) {

		o << "typedef enum {\n\t";

		UTL_ScopeActiveIterator *i = new UTL_ScopeActiveIterator(this, IK_decls);

		while (!(i->is_done())) {
			SCOPED_NAME_DUMP( i->item()->name(), o );
			i->next();
			if (!(i->is_done()))
				o << ", ";
		}
		delete i;

		o << "\n} "; SCOPED_NAME_DUMP( name(), o ); o << ";\n\n";

	}
	o << flush;
}

//=============================================================================

void
BE_EnumVal::dump( ostream& o )
{
	o << flush;
	// dump nothing here, already taken care of in BE_Enum::dump()
	o << flush;
}

//=============================================================================

void
BE_Expression::dump( ostream& o )
{
	o << flush;

	if ( WANT_PBIO_TYPES )
		AST_Expression::dump(o);
	else switch (ec()) {
		// Binary expressions:
	case EC_add:
		dump_binary_expr(o, "+", v1(), v2());
		break;
	case EC_minus:
		dump_binary_expr(o, "-", v1(), v2());
		break;
	case EC_mul:
		dump_binary_expr(o, "*", v1(), v2());
		break;
	case EC_div:
		dump_binary_expr(o, "/", v1(), v2());
		break;
	case EC_mod:
		dump_binary_expr(o, "%", v1(), v2());
		break;
	case EC_or:
		dump_binary_expr(o, "|", v1(), v2());
		break;
	case EC_xor:
		dump_binary_expr(o, "^", v1(), v2());
		break;
	case EC_and:
		dump_binary_expr(o, "&", v1(), v2());
		break;
	case EC_left:
		dump_binary_expr(o, "<<", v1(), v2());
		break;
	case EC_right:
		dump_binary_expr(o, ">>", v1(), v2());
		break;
		// Unary expressions:
	case EC_u_plus:
		dump_unary_expr(o, "+", v1());
		break;
	case EC_u_minus:
		dump_unary_expr(o, "-", v1());
		break;
	case EC_bit_neg:
		dump_unary_expr(o, "~", v1());
		break;
		// Unevaluated symbol
	case EC_symbol:
		n()->dump(o);
		break;
	case EC_none:
		AST_Expression::dump(o);
		break;
	}
	o << flush;
}

// stupid to have to replicate this code here... oh well...

static void
dump_binary_expr(ostream &o, const char *s, AST_Expression *n1, AST_Expression *n2)
{
  if (n1 != NULL) n1->dump(o);
  o << " " << s << " ";
  if (n2 != NULL) n2->dump(o);
}

static void
dump_unary_expr(ostream &o, const char *s, AST_Expression *e)
{
  o << s;
  e->dump(o);
}

//=============================================================================

void
BE_PredefinedType::dump( ostream& o )
{
	o << flush;
	if ( !WANT_PBIO_TYPES )	{

		if ( pt() == PT_octet )
			o << "unsigned char";
		else
			AST_PredefinedType::dump(o);

	} else switch ( pt() ) {

	case PT_short:
	case PT_long:
	case PT_longlong:
		o << "integer";
		break;

	case PT_ulong:
	case PT_ulonglong:
	case PT_ushort:
	case PT_octet:
		o << "unsigned integer";
		break;

	case PT_float:
	case PT_double:
	case PT_longdouble:
		o << "float";
		break;

	case PT_char:
	case PT_wchar:
		o << "char";
		break;

	case PT_boolean:
	case PT_any:
	case PT_void:
	case PT_pseudo:
		BE_ABORT( UNSUPPORTED_PREDEFINED_TYPE );
		break;

	}
	o << flush;
}

//=============================================================================

void
BE_Sequence::dump( ostream& o )
{
	o << flush;
	//	AST_Sequence::dump(o);
	o << flush;
}

//=============================================================================

void
BE_String::dump( ostream& o )
{
	o << flush;
	if ( ! DUMPING_INTERFACE )
		return;

	if ( WANT_PBIO_TYPES )
		o << "string";
	else
		o << "char*";
	o << flush;
}

//#############################################################################
//#############################################################################
//#############################################################################
//#############################################################################
//#############################################################################
//#############################################################################
//=============================================================================

// void
// BE_Expression::dump( ostream& )
// {
// 	BE_WARN( DUMP_NOT_IMPLEMENTED );
// }

//=============================================================================

void
BE_UnionLabel::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//=============================================================================

void
BE_Decl::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//=============================================================================

// void
// BE_Constant::dump( ostream& )
// {
// 	BE_WARN( DUMP_NOT_IMPLEMENTED );
// }

//=============================================================================

// void
// BE_EnumVal::dump( ostream& )
// {
// 	BE_WARN( DUMP_NOT_IMPLEMENTED );
// }

//=============================================================================

void
BE_Field::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//=============================================================================

void
BE_Argument::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//=============================================================================

// void
// BE_Attribute::dump( ostream& )
// {
// 	BE_WARN( DUMP_NOT_IMPLEMENTED );
// }

//=============================================================================

void
BE_UnionBranch::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//=============================================================================

// void
// BE_Module::dump( ostream& )
// {
// 	BE_WARN( DUMP_NOT_IMPLEMENTED );
// }

//=============================================================================

// void
// BE_Root::dump( ostream& )
// {
// 	BE_WARN( DUMP_NOT_IMPLEMENTED );
// }

//=============================================================================

void
BE_Operation::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//=============================================================================

void
BE_Type::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//=============================================================================

void
BE_ConcreteType::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//=============================================================================

// void
// BE_Array::dump( ostream& )
// {
// 	BE_WARN( DUMP_NOT_IMPLEMENTED );
// }

//=============================================================================

// void
// BE_Enum::dump( ostream& )
// {
// 	BE_WARN( DUMP_NOT_IMPLEMENTED );
// }

//=============================================================================

// void
// BE_PredefinedType::dump( ostream& )
// {
// 	BE_WARN( DUMP_NOT_IMPLEMENTED );
// }

//=============================================================================

// void
// BE_Sequence::dump( ostream& )
// {
// 	BE_WARN( DUMP_NOT_IMPLEMENTED );
// }

//=============================================================================

// void
// BE_String::dump( ostream& )
// {
// 	BE_WARN( DUMP_NOT_IMPLEMENTED );
// }

//=============================================================================

void
BE_Structure::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//=============================================================================

void
BE_Exception::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//=============================================================================

void
BE_Union::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//=============================================================================

// void
// BE_Interface::dump( ostream& )
// {
// 	BE_WARN( DUMP_NOT_IMPLEMENTED );
// }

//=============================================================================

void
BE_InterfaceFwd::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//=============================================================================

void
BE_Typedef::dump( ostream& )
{
	BE_WARN( DUMP_NOT_IMPLEMENTED );
}

//#############################################################################


